#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliModel
from TableOutput import TableFormatter, Headings, Format
import datetime
import time

class DaemonAgent( CliModel.Model ):

   enabled = CliModel.Bool( help='Whether the daemon process is '
                            'configured to run.' )
   running = CliModel.Bool( help='True if this process is actively running' )

   pid = CliModel.Int( help='The PID of the process, if the agent is running',
                       optional=True )
   uptime = CliModel.Float(
         help='The uptime of the process, if the agent is running',
         optional=True )
   starttime = CliModel.Float(
         help='The start time of the process, if the agent is running',
         optional=True )
   isSdkAgent = CliModel.Bool( help='True if this process contains an executable '
                               'that uses EOS SDK to run a third-party agent. '
                               'This field will be omitted if the agent is not '
                               'running.', optional=True )

   option = CliModel.Dict(
      help='A string/string map of options configured for the process',
      valueType=str )
   data = CliModel.Dict(
      help='A string/string map of status data published by the process',
      valueType=str )


class DaemonAgents( CliModel.Model ):

   daemons = CliModel.Dict(
      help='A mapping from the daemon process name to its configuration',
      valueType=DaemonAgent )

   def renderTable( self, tableName, keyHeader, valueHeader, mapping, strDesc ):
      if mapping:
         t = TableFormatter()
         width = t.width_ - 15
         width = width // 2
         # max width of columns are restricted so that table does not
         # split into two
         f = Format( justify='left', wrap=True, maxWidth=width, minWidth=1 )
         f.noPadLeftIs( True )
         headings = ( ( keyHeader, "l" ), ( valueHeader, "l" ) )
         th = Headings( headings )
         th.doApplyHeaders( t )
         for key, value in sorted( mapping.items() ):
            t.newRow( key, value )
            t.formatColumns( *( [ f ] * len( headings ) ) )
         print( '%s:' % tableName ) # pylint: disable=consider-using-f-string
         print( t.output() )
      else:
         print( 'No %s stored.' % strDesc ) # pylint: disable=consider-using-f-string
         print()

   def render( self ):
      if not self.daemons:
         print( "No daemons have been configured." )
      for name, process in sorted( self.daemons.items() ):
         if process.enabled:
            if not process.running:
               processState = 'initializing'
            else:
               processState = 'running'
         else:
            if process.running:
               processState = 'shutting down'
            else:
               processState = 'shutdown'
         # pylint: disable-next=consider-using-f-string
         processState += ' with PID %d' % process.pid if process.pid else ''
         # pylint: disable-next=consider-using-f-string
         print( '{}: {} ({})'.format( 'Agent' if process.isSdkAgent else 'Process',
                                 name, processState ) )
         if process.uptime:
            uptime = datetime.timedelta( seconds=int( process.uptime ) )
            psTime =  "Uptime: "
            psTime += str( uptime )
            # pylint: disable-next=consider-using-f-string
            psTime += " (Start time: %s)" % time.strftime( '%a %b %d %H:%M:%S %Y',
                  time.localtime( process.starttime ) ) if process.starttime else ""
            print( psTime )
         self.renderTable( "Configuration", "Option", "Value", process.option,
                           "configuration options" )
         self.renderTable( "Status", "Data", "Value", process.data,
                           "status data" )
