# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliPlugin
from CliPlugin.DcbxModel import ( DcbxModel,
   InterfaceDescription, ApplicationPriorityEntryData )
from CliPlugin import IntfCli, EthIntfCli
import ConfigMount
import Intf.IntfRange
from DcbxTypes import ( ApplicationPriorityEntryKey, applicationPriorityEntry,
   applicationTable, cliSel, algorithmIds, algorithmDescriptions )
import LazyMount
import ShowCommand
import Tac
import CliGlobal

gv = CliGlobal.CliGlobal( dcbxConfig=None, pfcConfig=None,
                          dcbxStatus=None, lldpConfig=None )

matcherDcbx = CliMatcher.KeywordMatcher( 'dcbx',
      helpdesc='Configure DCBX' )
matcherEts = CliMatcher.KeywordMatcher( 'ets',
      helpdesc='Configure the CEE DCBX priority group' )
matcherRecommendation = CliMatcher.KeywordMatcher( 'recommendation',
      helpdesc='Configure the recommendation instead of configuration TLV')
matcherTrafficClass = CliMatcher.KeywordMatcher( 'traffic-class',
      helpdesc='Configure a Qos map' )
matcherApplication = CliMatcher.KeywordMatcher( 'application',
      helpdesc='Configure the IEEE DCBX application priority table' )
matcherAlgorithm = CliMatcher.EnumMatcher( algorithmDescriptions )

class DcbxIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del gv.dcbxConfig.portEnabled[ self.intf_.name ]

# Straight from LldConfigCli.py
class DcbxModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( mode.intf.name.startswith( 'Ethernet' ) and
               not mode.intf.isSubIntf() )

IntfCli.IntfConfigMode.addModelet( DcbxModelet )
#--------------------------------------------------------------------------------
# [ no | default ] dcbx mode MODE
#--------------------------------------------------------------------------------
class DcbxModeCmd( CliCommand.CliCommandClass ):
   syntax = 'dcbx mode MODE'
   noOrDefaultSyntax = 'dcbx mode ...'
   data = {
      'dcbx' : matcherDcbx,
      'mode' : 'Select DCBX mode',
      'MODE' : CliMatcher.EnumMatcher( {
         'none' : 'Disable DCBX',
         'ieee' : 'Select IEEE DCBX mode',
         'cee' : 'Select CEE DCBX mode',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      newDcbxMode = args.get( 'MODE', 'none' )
      if newDcbxMode == 'none':
         if mode.intf.name in gv.dcbxConfig.portEnabled:
            del gv.dcbxConfig.portEnabled[ mode.intf.name ]
            del gv.dcbxConfig.portDcbxEnabled[ mode.intf.name ]
      else:
         _initETSConfigInfo()
         gv.dcbxConfig.portDcbxEnabled[ mode.intf.name ] = True
         if newDcbxMode == 'ieee':
            gv.dcbxConfig.portEnabled[ mode.intf.name ] = "modeIeee"
         else:
            gv.dcbxConfig.portEnabled[ mode.intf.name ] = "modeCee"

   noOrDefaultHandler = handler

DcbxModelet.addCommandClass( DcbxModeCmd )

def getInfo( recommendation ):
   """
   Return either dcbxConfig.etsRecommendationInfo or dcbxConfig.etsConfigInfo
   """
   return (
      gv.dcbxConfig.etsRecommendationInfo if recommendation
      else gv.dcbxConfig.etsConfigInfo
   )

def setInfo( value, recommendation ):
   """
   Set either dcbxConfig.etsRecommendationInfo or dcbxConfig.etsConfigInfo to value
   """
   if recommendation:
      gv.dcbxConfig.etsRecommendationInfo = value
   else:
      gv.dcbxConfig.etsConfigInfo = value

def _initETSConfigInfo():
   def initInfo( recommendation ):
      info = getInfo( recommendation )

      if not info:
         fieldName = "recommendationInfo" if recommendation else "configInfo"
         setInfo( ( fieldName, ), recommendation )
         info = getInfo( recommendation )

         for c in range(0, 8):
            info.cosToTrafficClass[ c ] = 0
         for t in range(0, 8):
            info.trafficClassBandwidth[ t ] = 0

   initInfo( False )
   initInfo( True )

def _getPGEnabled( recommendation ):
   info = getInfo( recommendation )

   for cos in range( 0, 8 ):
      # If at least one TC has config then mark enable as True
      if ( info.cosToTrafficClass[ cos ] or
           info.trafficClassBandwidth[ cos ] or
           cos in info.trafficClassAlgorithm ):
         return True

   return False

def _setEtsQMap( mode, cos, trafficClass, recommendation ):
   _initETSConfigInfo()
   info = getInfo( recommendation )
   info.cosToTrafficClass[ cos ] = trafficClass
   info.enable = _getPGEnabled( recommendation )

def _setTrafficBandwidth( mode, trafficClass, bandwidth, recommendation  ):
   info = getInfo( recommendation )
   info.trafficClassBandwidth[ trafficClass ] = bandwidth
   info.enable = _getPGEnabled( recommendation )

def _setTrafficAlgorithm( mode, trafficClass, algorithm, recommendation,
                          explicitName=False ):
   info = getInfo( recommendation )
   if algorithm is None:
      del info.trafficClassAlgorithm[ trafficClass ]
   else:
      info.trafficClassAlgorithm[ trafficClass ] = algorithm
   if explicitName:
      info.trafficClassAlgorithmName.add( trafficClass )
   elif trafficClass in info.trafficClassAlgorithmName:
      info.trafficClassAlgorithmName.remove( trafficClass )

   info.enable = _getPGEnabled( recommendation )

#--------------------------------------------------------------------------------
# [ no | default ] dcbx ets qos map cos COS traffic-class TRAFFIC_CLASS
# Example: dcbx ets qos map cos <0-7> traffic-class <0-7>
#--------------------------------------------------------------------------------
class EtsQMapCmd( CliCommand.CliCommandClass ):
   syntax = 'dcbx ets [ recommendation ] qos map cos COS traffic-class TRAFFIC_CLASS'
   noOrDefaultSyntax = syntax
   data = {
      'dcbx' : matcherDcbx,
      'ets' : matcherEts,
      'recommendation' : matcherRecommendation,
      'qos' : 'Configure a Qos map',
      'map' : 'Configure a Qos map',
      'cos' : 'Configure a Qos map',
      'COS' : CliMatcher.IntegerMatcher( 0, 7, helpdesc='Cos' ),
      'traffic-class' : matcherTrafficClass,
      'TRAFFIC_CLASS' : CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='Internal class of traffic' ),
   }

   @staticmethod
   def handler( mode, args ):
      _setEtsQMap( mode, args[ 'COS' ], args[ 'TRAFFIC_CLASS' ],
                   recommendation='recommendation' in args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if not getInfo( 'recommendation' in args ):
         return
      _setEtsQMap( mode, args[ 'COS' ], 0, recommendation='recommendation' in args )

BasicCli.GlobalConfigMode.addCommandClass( EtsQMapCmd )

#--------------------------------------------------------------------------------
# [ no | default ] dcbx ets traffic-class TRAFFIC_CLASS bandwidth BANDWIDTH
# Example: dcbx ets traffic-class <0-7> bandwidth <1-100>
#--------------------------------------------------------------------------------
class EtsTrafficBandwidthCmd( CliCommand.CliCommandClass ):
   syntax = 'dcbx ets [ recommendation ] traffic-class TRAFFIC_CLASS bandwidth ' \
            'BANDWIDTH'
   noOrDefaultSyntax = syntax.replace( 'BANDWIDTH', '...' )
   data = {
      'dcbx' : matcherDcbx,
      'ets' : matcherEts,
      'recommendation': matcherRecommendation,
      'traffic-class' : matcherTrafficClass,
      'TRAFFIC_CLASS' : CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='Internal class of traffic' ),
      'bandwidth' : 'Configure bandwidth',
      'BANDWIDTH' : CliMatcher.IntegerMatcher( 0, 100,
         helpdesc='Bandwidth % - default: 0' ),
   }

   @staticmethod
   def handler( mode, args ):
      trafficClass = args[ 'TRAFFIC_CLASS' ]
      bandwidth = args[ 'BANDWIDTH' ]
      _initETSConfigInfo()
      info = ( gv.dcbxConfig.etsRecommendationInfo
               if 'recommendation' in args else
               gv.dcbxConfig.etsConfigInfo )
      s1 = sum( info.trafficClassBandwidth[ cos ]
                for cos in range( 0, trafficClass ) )
      s2 = sum( info.trafficClassBandwidth[ cos ]
                for cos in range( trafficClass+1, 8 ) )
      total = s1 + s2 + bandwidth
      if total > 100:
         mode.addError( 'Total bandwidth allocation for all Traffic Classes '
                        'cannot exceed 100%' )
         return
      _setTrafficBandwidth( mode, trafficClass, bandwidth,
                            recommendation='recommendation' in args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if not getInfo( 'recommendation' in args ):
         return
      _setTrafficBandwidth( mode, args[ 'TRAFFIC_CLASS' ], 0,
                            recommendation='recommendation' in args )

BasicCli.GlobalConfigMode.addCommandClass( EtsTrafficBandwidthCmd )

#--------------------------------------------------------------------------------
# [ no | default ] dcbx ets traffic-class TRAFFIC_CLASS algorithm ALGORITHM
# Examples: dcbx ets traffic-class <0-7> algorithm <0-255>
#           dcbx ets traffic-class <0-7> algorithm <name>
#--------------------------------------------------------------------------------
etsTrafficAlgorithmSyntax = 'dcbx ets [ recommendation ] traffic-class ' \
                            'TRAFFIC_CLASS algorithm'

class EtsTrafficAlgorithmCmd( CliCommand.CliCommandClass ):
   syntax = etsTrafficAlgorithmSyntax + ' ( ALGORITHM_ID | ALGORITHM_NAME )'
   noOrDefaultSyntax = etsTrafficAlgorithmSyntax + ' ...'
   data = {
      'dcbx' : matcherDcbx,
      'ets' : matcherEts,
      'recommendation': matcherRecommendation,
      'traffic-class' : matcherTrafficClass,
      'TRAFFIC_CLASS' : CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='Internal class of traffic' ),
      'algorithm' : 'Configure algorithm',
      'ALGORITHM_ID' : CliMatcher.IntegerMatcher( 0, 255,
         helpdesc='Algorithm - default: 255' ),
      'ALGORITHM_NAME' : matcherAlgorithm,
   }

   @staticmethod
   def handler( mode, args ):
      trafficClass = args[ 'TRAFFIC_CLASS' ]
      if 'ALGORITHM_NAME' in args:
         explicitName = args['ALGORITHM_NAME']
         algorithm = algorithmIds[ explicitName ]
      else:
         explicitName = None
         algorithm = args[ 'ALGORITHM_ID' ]
      _initETSConfigInfo()
      _setTrafficAlgorithm( mode, trafficClass, algorithm, 'recommendation' in args,
                            explicitName=explicitName is not None )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if not getInfo( 'recommendation' in args ):
         return
      _setTrafficAlgorithm( mode, args[ 'TRAFFIC_CLASS' ], None,
                            recommendation='recommendation' in args )

BasicCli.GlobalConfigMode.addCommandClass( EtsTrafficAlgorithmCmd )

#--------------------------------------------------------------------------------
# [ no | default ] dcbx application
#              ( ( ether ETHER_PORT ) | ( PROTOCOL PORT ) | GROUP ) priority PRIORITY
#--------------------------------------------------------------------------------
class ApplicationPriorityCmd( CliCommand.CliCommandClass ):
   syntax = ( 'dcbx application '
              '( ( ether ETHER_PORT ) | ( PROTOCOL PORT ) | GROUP ) '
              'priority PRIORITY' )
   noOrDefaultSyntax = ( 'dcbx application '
                         '( ( ether ETHER_PORT ) | ( PROTOCOL PORT ) | GROUP ) '
                         'priority ...' )
   data = {
      'dcbx' : matcherDcbx,
      'application' : matcherApplication,
      'ether' : 'Configure an EtherType rule',
      'ETHER_PORT' : CliMatcher.IntegerMatcher( 1536, 65535, helpdesc='EtherType' ),
      'PROTOCOL' : CliMatcher.EnumMatcher( {
         'tcp-sctp' : 'Configure a TCP/SCTP port number rule',
         'tcp-sctp-udp' : 'Configure a TCP/SCTP/UDP port number rule',
         'udp' : 'Configure a UDP port number rule',
      } ),
      'PORT' : CliMatcher.IntegerMatcher( 1, 65535,
         helpdesc='Number of the port to use' ),
      'GROUP' : CliMatcher.EnumMatcher( {
         'iscsi' : 'Configure the iSCSI rule',
      } ),
      'priority' : 'Configure application priority',
      'PRIORITY' : CliMatcher.IntegerMatcher( 0, 7, helpdesc='Priority' ),
   }

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      priority = args.get( 'PRIORITY', -1 )
      if 'GROUP' in args:
         applications = applicationTable[ args[ 'GROUP' ] ]
      elif 'ether' in args:
         applications = ( ( 'ether', args[ 'ETHER_PORT' ] ), )
      else:
         appMap =  { 'tcp-sctp': 'tcpSctp', 'udp': 'udp',
                     'tcp-sctp-udp': 'tcpSctpUdp' }
         applications = ( ( appMap[ args[ 'PROTOCOL' ] ], args[ 'PORT' ] ), )

      if not no and len( gv.dcbxConfig.applicationPriorityEntry ) + \
            len( applications ) > 168:
         mode.addError(
               "You cannot configure more than 168 application priority "
               "table entries." )
         return

      for sel, protocolId in applications:
         if no:
            key = ApplicationPriorityEntryKey( sel, protocolId )
            # If the user specified a priority in the no command, only delete the
            # entry if that priority is actually the one specified in the table.
            if key in gv.dcbxConfig.applicationPriorityEntry and (
               priority < 0 or
               gv.dcbxConfig.applicationPriorityEntry[ key ].priority == priority ):
               del gv.dcbxConfig.applicationPriorityEntry[ key ]
         else:
            entry = applicationPriorityEntry( priority, sel, protocolId )
            gv.dcbxConfig.addApplicationPriorityEntry( entry )

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( ApplicationPriorityCmd )

#--------------------------------------------------------------------------------
# ( no | default ) dcbx application
#--------------------------------------------------------------------------------
class NoDcbxApplicationCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'dcbx application'
   data = {
      'dcbx' : matcherDcbx,
      'application' : matcherApplication,
   }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      """Clear the application priority table."""
      gv.dcbxConfig.applicationPriorityEntry.clear()

BasicCli.GlobalConfigMode.addCommandClass( NoDcbxApplicationCmd )

#--------------------------------------------------------------------------------
# show dcbx [ INTF ] [ TYPE ]
#--------------------------------------------------------------------------------
def parsePriorityFlowControlConfigInfo( interface, pfcci, portName ):
   pfcPortConfig = gv.pfcConfig.portConfig.get( portName, None )

   interface.pfcciPfcCapability = pfcci.pfcCapability
   interface.pfcciPfcEnable = pfcci.pfcEnable
   interface.pfcWilling = pfcci.willing
   interface.macSecBypassCapability = pfcci.macSecBypassCapability

   if pfcPortConfig:
      interface.pfcPortConfigEnabled = pfcPortConfig.enabled
      interface.pfcPortConfigPriorities = pfcPortConfig.priorities

def parseApplicationPriorityConfigInfo( interface, apci ):
   for entry in apci.applicationPriorityEntry.values():
      apciEntry = cliSel( entry.sel )
      priority = entry.priority
      protocolId = entry.protocolId
      # check if entry already exists in map, if it does just append data
      apciEntryObject = interface.apciEntries.get( apciEntry )

      if apciEntryObject:
         interface.apciEntries[ apciEntry ].priority.append( priority )
         interface.apciEntries[ apciEntry ].protocolId.append( protocolId )
      else:
         interface.apciEntries[ apciEntry ] = ApplicationPriorityEntryData(
                                             priority = [ priority ],
                                             protocolId = [ protocolId ] )
   # TODO: print a warning if our config doesn't match what we received

def parseTypeArgument( interface, intfKey, typ ):
   # here we should check the interface provided, then we parse the metadata into
   # the cliModel class for use later
   if typ is None or typ == 'status':
      interfaceObject = gv.dcbxConfig.portEnabled.get( intfKey )
      if interfaceObject:
         interface.dcbxEnabled = True
         lldpConfigObject = gv.lldpConfig.portConfig.get( intfKey )
         if lldpConfigObject:
            interface.txEnabled = 'tx' in lldpConfigObject.adminStatus.lower()
            interface.rxEnabled = 'rx' in lldpConfigObject.adminStatus.lower()

   interface.ieeeTlvReceived = intfKey in gv.dcbxStatus.peerPortStatus

   if not interface.ieeeTlvReceived:
      return

   # if we have received ieee TLV we continue parsing the configuration/metadata
   pps = gv.dcbxStatus.peerPortStatus[ intfKey ]
   interface.lastLldpduTimestamp = pps.update + Tac.utcNow() - Tac.now()

   interface.pfcTlvReceived = pps.priorityFlowControlConfigTlvReceived
   interface.pfcTlvMalformed = pps.priorityFlowControlConfigTlvMalformed
   interface.apcTlvReceived = pps.applicationPriorityConfigTlvReceived
   interface.apcTlvMalformed = pps.applicationPriorityConfigTlvMalformed

   pfcInfo = pps.priorityFlowControlConfigInfo
   apcInfo = pps.applicationPriorityConfigInfo

   if typ is None or typ == 'priority-flow-control-configuration':
      parsePriorityFlowControlConfigInfo(interface, pfcInfo, intfKey )

   if typ is None or typ == 'application-priority-configuration':
      parseApplicationPriorityConfigInfo( interface, apcInfo )

def showDcbxModel( mode, args ):
   # creating the capi compliant model that we will populate
   capiModel = DcbxModel()
   intfs = args.get( 'INTF' )
   if not intfs:
      candidates = set( gv.dcbxStatus.peerPortStatus )
      for intf in gv.dcbxConfig.portEnabled:
         if gv.dcbxConfig.portEnabled[ intf ]:
            candidates.add( intf )
      intfs = Arnet.sortIntf( candidates )

   # populate interfaces
   for key in intfs:
      capiModel.interfaces[ key ] = InterfaceDescription( dcbxEnabled = False,
                                                      txEnabled = False,
                                                      rxEnabled = False,
                                                      ieeeTlvReceived = False,
                                                      lastLldpduTimestamp = 0.00,
                                                      pfcTlvReceived = 0,
                                                      pfcTlvMalformed = 0,
                                                      apcTlvReceived = 0,
                                                      apcTlvMalformed = 0,
                                                      pfcPortConfigEnabled = False,
                                                      pfcPortConfigPriorities = 0,
                                                      pfcciPfcCapability = 0,
                                                      pfcciPfcEnable = 0,
                                                      pfcWilling = False,
                                                   macSecBypassCapability = False )
      parseTypeArgument( capiModel.interfaces[ key ], key, args.get( 'TYPE' ) )

   capiModel.populate( gv.dcbxConfig, args.get( 'TYPE' ) == 'ets-configuration' )

   return capiModel # return class/model interfaces with populated fields

showDcbxTypes =  {
   'application-priority-configuration' : ( 'Show IEEE DCBX peer application '
                                             'priority configuration' ),
   'priority-flow-control-configuration' : ( 'Show IEEE DCBX peer priority '
                                             'flow control configuration' ),
   'status' : 'Show DCBX status',
}

showDcbxTypes[ 'ets-configuration' ] = \
      'Show IEEE DCBX ETS recommendation / configuration'

class ShowDcbx( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dcbx [ INTF ] [ TYPE ]'
   data = {
      'dcbx' : 'Show IEEE DCBX information',
      'INTF' : Intf.IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( EthIntfCli.EthPhyAutoIntfType, ) ),
      'TYPE' : CliMatcher.EnumMatcher( showDcbxTypes )
   }
   cliModel = DcbxModel
   handler = showDcbxModel

BasicCli.addShowCommandClass( ShowDcbx )

#------------------------------------------------------------------------------------
#                       *** End of 'show dcbx' ***
#------------------------------------------------------------------------------------

#-----------------------------------------------------
# Register show dcbx command into "show tech-support".
#-----------------------------------------------------
# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2010-01-01 00:08:30',
   cmds=[ 'show dcbx' ] )

def Plugin( entityManager ):
   gv.dcbxConfig = ConfigMount.mount(
      entityManager, "dcb/dcbx/config", "Dcbx::Config", "w" )
   gv.pfcConfig = LazyMount.mount(
      entityManager, "dcb/pfc/config", "Pfc::Config", "r" )
   gv.dcbxStatus = LazyMount.mount(
      entityManager, "dcb/dcbx/status", "Dcbx::Status", "r" )
   gv.lldpConfig = LazyMount.mount(
      entityManager, "l2discovery/lldp/config", "Lldp::Config", "r" )
   IntfCli.Intf.registerDependentClass( DcbxIntf )
