# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=bad-string-format-type

import Arnet
from IntfModels import Interface
from CliModel import Dict, List, Model, Int, Float, Bool, Submodel
import time
from DcbxLib import (
      applicationPriorityConfigInfoStr,
      priorityFlowControlConfigInfoStr,
      etsConfigInfoStr
)
from DcbxTypes import (
      PriorityFlowControlConfigInfo,
      applicationPriorityEntry,
      selFromCli
)
import Tac

# example with empty model
# {
#   "jsonrpc": "2.0",
#   "id": "EapiExplorer-1",
#   "result": [
#     {
#       "interfaces": {}
#     }
#   ]
# }

# example with basic Ethernet1 on a dut. 
# {
#     "interfaces": {
#         "Ethernet1": {
#             "apcTlvMalformed": 0,
#             "rxEnabled": true,
#             "pfcPortConfigPriorities": 0,
#             "pfcTlvMalformed": 0,
#             "pfcPortConfigEnabled": false,
#             "pfcciPfcCapability": 3,
#             "apcTlvReceived": 1,
#             "pfcWilling": true,
#             "macSecBypassCapability": true,
#             "txEnabled": true,
#             "lastLldpduTimestamp": 1659994313.6726196,
#             "pfcTlvReceived": 1,
#             "dcbxEnabled": true,
#             "ieeeTlvReceived": true,
#             "apciEntries": {
#                 "tcp-sctp": {
#                     "priority": [
#                         6
#                     ],
#                     "protocolId": [
#                         21909
#                     ]
#                 }
#             },
#             "pfcciPfcEnable": 240
#         }
#     }
# }
# {
#     "interfaces": {
#         "Ethernet1": {
#             "apcTlvMalformed": 0,
#             "rxEnabled": true,
#             "pfcPortConfigPriorities": 0,
#             "pfcTlvMalformed": 0,
#             "pfcPortConfigEnabled": false,
#             "pfcciPfcCapability": 3,
#             "apcTlvReceived": 1,
#             "pfcWilling": true,
#             "macSecBypassCapability": true,
#             "txEnabled": true,
#             "lastLldpduTimestamp": 1659994313.67262,
#             "pfcTlvReceived": 1,
#             "dcbxEnabled": true,
#             "ieeeTlvReceived": true,
#             "apciEntries": {
#                 "tcp-sctp": {
#                     "priority": [
#                         0,
#                         4,
#                         0,
#                     ],
#                     "protocolId": [
#                         390,
#                         578,
#                         947,
#                     ]
#                 },
#                 "tcp-sctp-udp": {
#                     "priority": [
#                         7,
#                         1,
#                         7,
#                     ],
#                     "protocolId": [
#                         614,
#                         1110,
#                         1486,
#                     ]
#                 },
#                 "udp": {
#                     "priority": [
#                         2,
#                         4,
#                         2,
#                     ],
#                     "protocolId": [
#                         1571,
#                         5231,
#                         9268,
#                     ]
#                 },
#                 "ether": {
#                     "priority": [
#                         2,
#                         7,
#                         3,
#                     ],
#                     "protocolId": [
#                         1986,
#                         3864,
#                         6989,
#                     ]
#                 }
#             },
#             "pfcciPfcEnable": 240
#         }
#     }
# }

class ApplicationPriorityEntryData( Model ):

   protocolId = List( help="Protocol ID that maps to the protocol's priority",
                      valueType=int )
   priority = List( help="Corresponding protocol ID enabled priority",
                    valueType=int ) 

class InterfaceDescription( Model ):

   dcbxEnabled = Bool( help="Interface exists in the DCBX configuration scheme" ) 
   txEnabled = Bool( help="LLDP(Link Layer Discovery Protocol) adminstatus "
                          "is enabled for transmit" )
   rxEnabled = Bool( help="LLDP adminstatus is enabled for receive" ) 
   ieeeTlvReceived = Bool( help="Ieee TLVs(Type-Lengh-Values) have been received" )
   lastLldpduTimestamp = Float( help="Unix time stamp taken whenever an LLDPDU "
                                    "is received" )
   pfcTlvReceived = Int( help="Number of PFC(priority flow control) TlVs received" ) 
   pfcTlvMalformed =Int( help="Number of PFC(priority flow control) TlVs that"
                         " are malformed" ) 
   pfcPortConfigEnabled = Bool( help="Port PFC configured" )
   pfcPortConfigPriorities = Int( help="Number of PFC configuration priorities" )
   pfcciPfcCapability = Int( help="PFC capabilities" )
   pfcciPfcEnable = Int( help="PFC enabled priorities" )
   pfcWilling = Bool( help="PFC configuration" )
   apcTlvReceived = Int( help="Number of Application priority configuration "
                              "TLVs received" ) 
   apcTlvMalformed = Int( help="Number of Application priority configuration TLVs "
                               "that are malformed" ) 
   macSecBypassCapability = Bool( help="Capable of bypassing MACsec" )
   # Map containing the Apci entries and metadata(protocol id + enabled priorities)
   apciEntries =  Dict( help="Application priority configuration(apci)"
                        " object. This object contains metadata for every apci"
                        " entry. There is a one-to-one mapping between each "
                        " entry protocol ID and the enabled priorities stored "
                        "within each value",
                        valueType=ApplicationPriorityEntryData )

   # returns the number of bits set
   def bitsSet( self , n ):
      return f'{n:b}'.count( '1' )

   def printLines( self, lines, prefix='  ' ):
      dash = "-"
      for line in lines:
         print( prefix + dash, line )
         dash = " "

   # this function renders/prints priority flow control config information
   def renderPfcConfigInfo( self ):
      pfcci = PriorityFlowControlConfigInfo()
      pfcci.willing = self.pfcWilling
      pfcci.macSecBypassCapability = self.macSecBypassCapability
      pfcci.pfcCapability = self.pfcciPfcCapability
      pfcci.pfcEnable = self.pfcciPfcEnable

      self.printLines( priorityFlowControlConfigInfoStr( pfcci ) )

      if self.pfcPortConfigEnabled and \
           self.bitsSet( self.pfcPortConfigPriorities ) > self.pfcciPfcCapability:
         print( '    WARNING: PFC is configured locally on more priorities than'
                ' the peer can support' )
   
      if ( ( self.pfcPortConfigEnabled and self.pfcciPfcEnable != 
               self.pfcPortConfigPriorities ) or ( not self.pfcPortConfigEnabled
               and self.pfcciPfcEnable ) ) or ( not self.pfcPortConfigEnabled and 
               self.pfcciPfcEnable ):
         print( '    WARNING: peer PFC configuration does not match'
                ' the local PFC configuration' )
      
   # this function renders/prints application flow control config information
   def renderAfcConfigInfo( self ):
      apci = Tac.newInstance( "Dcbx::ApplicationPriorityConfigInfo", "tmp" )
      for key, metadata in self.apciEntries.items():
         for index, protocol in enumerate( metadata.protocolId ):
            sel = selFromCli( key )
            priority = metadata.priority[ index ]
            apci.addApplicationPriorityEntry(
                  applicationPriorityEntry( priority, sel, protocol ) )

      self.printLines( applicationPriorityConfigInfoStr( apci ) )
         
   # render function for interface and it's flow control and app control config
   def render( self ):
      metadataString = ''
      if self.dcbxEnabled:
         metadataString += ' IEEE DCBX is enabled '

         if self.txEnabled and self.rxEnabled:
            metadataString += 'and active'
         elif self.txEnabled:
            metadataString += 'and only partially active because LLDP is '\
                              'enabled for transmit only'
         elif self.rxEnabled: 
            metadataString += 'and only partially active because LLDP is '\
                              'enabled for receive only'
         else:
            metadataString += 'but not active because LLDP is disabled'

         print( metadataString )

      if not self.ieeeTlvReceived:
         print( ' No IEEE DCBX TLVs were received.' )
      
      metadataString = ' Last LLDPDU received on %s'% time.ctime(
                                                      self.lastLldpduTimestamp )
      print( metadataString )

      # next we will try to print the flow control configuration using 
      # the metadata we have collected
      metadataString = ''
      humanType = 'priority flow control configuration'
      if self.pfcTlvReceived == 0:
         metadataString += ' No %s TLVs received' % humanType
      elif self.pfcTlvMalformed > 0:
         metadataString += ' ERROR: %d out of %d %s TLVs ' \
                           'received were malformed' % \
                           ( self.pfcTlvMalformed, self.pfcTlvReceived, humanType )
      elif self.pfcTlvReceived > 1:
         metadataString += ' ERROR: %d %s TLVs received' % \
                           ( self.pfcTlvReceived, humanType )
      else:
         self.renderPfcConfigInfo()

      print( metadataString )

      metadataString = ''
      humanType = 'application priority configuration'
      if self.apcTlvReceived == 0:
         metadataString += ' No %s TLVs received' % humanType
      elif self.apcTlvMalformed > 0:
         metadataString += ' ERROR: %d out of %d %s TLVs ' \
                           'received were malformed' % \
                           ( self.apcTlvMalformed, self.apcTlvReceived, humanType )
      elif self.apcTlvReceived > 1:
         metadataString += ' ERROR: %d %s TLVs received' % \
                           ( self.apcTlvReceived, humanType )
      else:
         self.renderAfcConfigInfo()

      print( metadataString )

class TrafficClassModel( Model ):
   priority = Int( help="Traffic class priority" )
   bandwidth = Int( help="Bandwidth value" )
   algorithm = Int( help="Algorithm identifier" )

class EtsTlvModel( Model ):
   """
   Example text output:
   Traffic Class Priority Bandwidth (%) Algorithm
   ------------- -------- ------------- ---------
            0        3        10            0
            1        4        20          255
   ...

   Example json output:
   {
      "trafficClasses": [
         {
            "priority": 3,
            "bandwidth": 10,
            "algorithm": 0
         },
         {
            "priority": 4,
            "bandwidth": 20,
            "algorithm": 255
         },
         ...
      ]
   }
   """
   _enabled = Bool( help="Whether this TLV should be printed" )
   _recommendation = Bool( help="Whether the TLV is the ETS recommendation TLV" )
   trafficClasses = List( help="Traffic classes data", valueType=TrafficClassModel )

   def populate( self, config, isRecommendation ):
      """
      config is either ets config or recommendation
      """

      self._recommendation = isRecommendation

      if config is None:
         return

      # Can still be disabled if set then reset by user
      self._enabled = config.enable

      self.trafficClasses = [
         TrafficClassModel(
            priority=config.cosToTrafficClass.get( i, 0 ),
            bandwidth=config.trafficClassBandwidth.get( i, 0 ),
            algorithm=config.trafficClassAlgorithm.get( i, 255 )
         )
         for i in range( 8 )
      ]

   def render( self ):
      # This TLV has not been enabled, don't display it
      if not self._enabled:
         return

      priorities = [ tc.priority for tc in self.trafficClasses ]
      bandwidths = [ tc.bandwidth for tc in self.trafficClasses ]
      algorithms = [ tc.algorithm for tc in self.trafficClasses ]

      print( '\n'.join( etsConfigInfoStr(
         priorities, bandwidths, algorithms, self._recommendation ) ) )

class DcbxModel( Model ):
   # Map with interface class holding all interface metadata
   interfaces = Dict( help="Map to interface class which holds all of the "
                      "interface metadata", 
                      valueType=InterfaceDescription,
                      keyType=Interface )
   etsConfig = Submodel( valueType=EtsTlvModel,
      help="ETS configuration TLV" )
   etsRecommendation = Submodel( valueType=EtsTlvModel,
      help="ETS recommendation TLV" )

   def populate( self, dcbxConfig, showEts ):
      self.etsConfig = EtsTlvModel()
      self.etsRecommendation = EtsTlvModel()

      self.etsConfig.populate( dcbxConfig.etsConfigInfo
                               if showEts else None, False )
      self.etsRecommendation.populate( dcbxConfig.etsRecommendationInfo
                                       if showEts else None, True )

   # render function which renders the interface correspondong to each intf key
   def render( self ):
      for interface in Arnet.sortIntf( self.interfaces ):
         print( interface + ': ' )
         self.interfaces[ interface ].render()

      if self.etsConfig:
         self.etsConfig.render()

      if self.etsRecommendation:
         self.etsRecommendation.render()
