#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import Arnet
import BasicCli
import CliCommand
import CliMatcher
import LazyMount
import ShowCommand
import Tac
import CliPlugin.VrfCli as VrfCli
import CliPlugin.AclCli as AclCli
import CliPlugin.DhcpSnoopingCli as DhcpSnoopingCli
import CliPlugin.DhcpRelayCommonModel as DhcpRelayCommonModel
from CliPlugin.DhcpRelayHelperCli import (
   drStatus, drCounterConfig, drConfig, optionFormatEncode
)
from CliPlugin.AclCliModel import AllAclList
from CliPlugin.VlanCli import vlanSetMatcher
from CliToken.Dhcp import dhcpMatcherForShow
from CliToken.Ip import ipMatcherForShow
from CliToken.Ipv6 import ipv6MatcherForShow
from DhcpRelayVrfCliLib import ( keyHostnameVrf,
                                 keyIpAddrVrf,
                                 keyIpv6AddrVrf )

matcherCounters = CliMatcher.KeywordMatcher( 
      'counters', helpdesc='Counters' )
matcherDebug = CliMatcher.KeywordMatcher( 
      'debug', helpdesc='Internal counters for debugging' )
matcherRelay = CliMatcher.KeywordMatcher( 
      'relay', helpdesc='DHCP Relay' )
matcherSnoopingV6 = CliMatcher.KeywordMatcher( 
      'snooping', helpdesc='DHCPv6 Snooping information' )
matcherSnooping = CliMatcher.KeywordMatcher(
      'snooping', helpdesc='DHCP Snooping information' )
nodeDetail = CliCommand.guardedKeyword(
      'detail', helpdesc='Access list detail',
      guard=AclCli.countersPerChipEnabledGuard )
nodeSnoopingV6 = CliCommand.guardedKeyword( 
      'snooping', helpdesc='DHCPv6 Snooping information', 
      guard=DhcpSnoopingCli.dhcp6SnoopingSupportedGuard )
nodeSnooping = CliCommand.guardedKeyword( 
      'snooping', helpdesc='DHCP Snooping information', 
      guard=DhcpSnoopingCli.dhcpSnoopingSupportedGuard )
addressFamilyExprForShow = CliMatcher.EnumMatcher( {
   'ip' : ipMatcherForShow.helpdesc_,
   'ipv6' : ipv6MatcherForShow.helpdesc_,
} )

bridgingConfig = None

#--------------------------------------------------------------------------------
# show ( ip | ipv6 ) dhcp relay counters
#--------------------------------------------------------------------------------
def updateDhcpRelayCounters( mode ):
   drCounterConfig().counterUpdateRequestTime = Tac.now()
   def countersUpdated():
      return ( drStatus().counterStatus.counterUpdateTime >=
               drCounterConfig().counterUpdateRequestTime )
   try:
      Tac.waitFor( countersUpdated, description='Counter update',
                   maxDelay=0.1, sleep=True, timeout=5 )
   except Tac.Timeout:
      mode.addWarning( 'Displaying stale counters' )

def showDhcpRelayCounters( mode, args ):
   result = DhcpRelayCommonModel.DhcpRelayCounterModel()
   status = drStatus()
   emptyIntf = Tac.newInstance( "Arnet::IntfId", "" ).stringValue

   if status.runControl is True:
      # Agent is running; Request agent to update counters.
      updateDhcpRelayCounters( mode )

   result.globalCounters = DhcpRelayCommonModel.GlobalCounters()
   result.globalCounters.allRequests = DhcpRelayCommonModel.Counters()
   result.globalCounters.allResponses = DhcpRelayCommonModel.Counters()
   counterStatus = drStatus().counterStatus
   if not counterStatus:
      return result
   
   result.globalCounters.allRequests.received = counterStatus.totalRequestsReceived
   result.globalCounters.allRequests.forwarded = counterStatus.totalRequestsForwarded
   result.globalCounters.allRequests.dropped = counterStatus.totalRequestsDropped
   result.globalCounters.allResponses.received = counterStatus.totalRepliesReceived
   result.globalCounters.allResponses.forwarded = counterStatus.totalRepliesForwarded
   result.globalCounters.allResponses.dropped = counterStatus.totalRepliesDropped
   result.globalCounters.lastResetTime = \
         Ark.switchTimeToUtc( counterStatus.lastResetAllTime )

   def populateInterfaceCounter( counterIntfStatus ):
      request = DhcpRelayCommonModel.Counters()
      reply = DhcpRelayCommonModel.Counters()
      request.received = counterIntfStatus.requestsReceived
      request.forwarded = counterIntfStatus.requestsForwarded
      request.dropped = counterIntfStatus.requestsDropped
      reply.received = counterIntfStatus.repliesReceived
      reply.forwarded = counterIntfStatus.repliesForwarded
      reply.dropped = counterIntfStatus.repliesDropped
      intfCounter = DhcpRelayCommonModel.InterfaceCounter()
      intfCounter.requests = request
      intfCounter.replies = reply
      intfCounter.lastResetTime = \
            Ark.switchTimeToUtc( counterIntfStatus.lastResetTime )
      return intfCounter

   counterIntfs = counterStatus.counterIntfStatus
   if counterIntfs:
      result.interfaceCounters = {}
      for intfname in counterIntfs:
         if intfname != emptyIntf:
            counterIntfStatus = counterIntfs[ intfname ]
            result.interfaceCounters[ intfname ] = \
                  populateInterfaceCounter( counterIntfStatus )

   return result

def getAllV4Helpers( v4Helpers, vrfIp=False ):
   allV4Helpers = []
   for x in v4Helpers:
      helper = DhcpRelayCommonModel.Helper()
      helper.serverAddr = Arnet.IpGenAddr( x.ip.v4Addr )
      helper.vrf = x.vrfName
      if not vrfIp:
         if x.srcIp.stringValue != '0.0.0.0':
            helper.srcAddr = x.srcIp.stringValue
         if x.srcIntf != '':
            helper.srcIntf = x.srcIntf 
      allV4Helpers.append( helper )
   return allV4Helpers

def getAllV6Helpers( v6Helpers, config, globalMode=False, vrfIp=False ):
   allV6Helpers = []
   for x in v6Helpers:
      helper = DhcpRelayCommonModel.Helper()
      helper.serverAddr = Arnet.IpGenAddr( x.ip.v6Addr.stringValue )
      helper.vrf = x.vrfName
      if not vrfIp:
         if globalMode:
            helper.linkAddr = config.serverIp6Global[ x ]
         else:
            helper.linkAddr = config.serverIp6[ x ]
         if x.srcIp.stringValue != '::':
            helper.srcAddr = x.srcIp.stringValue
         if x.srcIntf != '':
            helper.srcIntf = x.srcIntf 
      allV6Helpers.append( helper )
   return allV6Helpers

def getAllHostnameHelpers( hostnameHelpers ):
   allHostnameHelpers = []
   for x in hostnameHelpers:
      helper = DhcpRelayCommonModel.Helper()
      helper.serverHostname = x.hostname
      helper.vrf = x.vrfName
      if x.srcIp.stringValue != '0.0.0.0':
         helper.srcAddr = x.srcIp.stringValue
      if x.srcIntf != '':
         helper.srcIntf = x.srcIntf 
      allHostnameHelpers.append( helper )
   return allHostnameHelpers

class DhcpRelayCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ADDR_FAMILY dhcp relay counters'
   data = {
      'ADDR_FAMILY' : addressFamilyExprForShow,
      'dhcp' : dhcpMatcherForShow,
      'relay' : matcherRelay,
      'counters' : matcherCounters,
   }

   handler = showDhcpRelayCounters
   cliModel = DhcpRelayCommonModel.DhcpRelayCounterModel

BasicCli.addShowCommandClass( DhcpRelayCountersCmd )

#--------------------------------------------------------------------------------
# show ( ip | ipv6 ) helper-address
#--------------------------------------------------------------------------------
def showIpDhcpRelay( mode, args ):
   config = drConfig()
   infoOpt = config.circuitIdOptOn or config.informationOptOn
   circuitIdFormat = config.circuitIdEncodingFormat
   remoteIdFormatV4 = config.remoteIdEncodingFormatV4
   llAddrOpt = config.clientLinkLayerAddrOptOn
   remoteIdEncodingFormat = config.remoteIdEncodingFormat
   result = DhcpRelayCommonModel.IpDhcpRelay()
   # DhcpRelay runnability only depends on alwaysOn and
   # exsitence of config.intfConfig
   # pylint: disable=too-many-nested-blocks
   if not ( config.alwaysOn or config.intfConfig or drStatus().intfStatus ):
      result.activeState = False
   else:
      result.activeState = True
   result.alwaysOn = config.alwaysOn
   result.option82 = infoOpt
   option82SubOpts = DhcpRelayCommonModel.Option82SubOptionsModel()
   option82SubOpts.circuitIdOpt = config.circuitIdOptOn or config.informationOptOn
   option82SubOpts.remoteIdOpt = config.informationOptOn
   option82SubOpts.vendorSpecificSubOpt = config.vendorSpecificSubOptOn
   result.option82SubOpts = option82SubOpts

   result.linkLayerAddrOpt = llAddrOpt
   result.remoteIdEncodingFormat = remoteIdEncodingFormat
   result.allSubnets = config.smartRelayGlobal
   result.allSubnetsV6 = config.allSubnetsV6Global
   result.tunnelReqDisable = drStatus().tunnelReqDisable
   result.mlagPeerLinkReqDisable = drStatus().mlagPeerLinkReqDisable
   result.reqFloodSuppressionVlanConfig = \
         sorted( list( config.reqFloodDisableVlan ) )
   result.reqFloodSuppressionVlanOperational = \
         sorted( list( drStatus().reqFloodDisableVlan ) )
   result.dhcpReqFloodRestrictDisabledDueToSnooping = \
      bool( config.reqFloodDisableVlan ) and drStatus().dhcpSnoopingRunControl
   result.vssControlDisable = config.vssControlDisable
   
   if ( config.serverIpGlobal or config.serverHostnameGlobal or
         config.serverIp6Global ):
      defaultServersShowModel = DhcpRelayCommonModel.DefaultServersShowModel()
      v4Helpers = sorted( config.serverIpGlobal, key=keyIpAddrVrf )
      hostnameHelpers = sorted( config.serverHostnameGlobal, key=keyHostnameVrf )
      v6Helpers = sorted( config.serverIp6Global, key=keyIpv6AddrVrf )

      defaultServersShowModel.helpersV4 += getAllV4Helpers( v4Helpers )
      defaultServersShowModel.helpersV4 += \
                                 getAllHostnameHelpers( hostnameHelpers )
      defaultServersShowModel.helpersV6 += getAllV6Helpers( v6Helpers, config,
                                                            globalMode=True )
      result.defaultServers = defaultServersShowModel
   
   result.configuredInterfaces = {}
   
   if drStatus().vssSubOptUnsupportedServers is None:
      vssSubOptServers = None
      vssV6SubOptServers = None
   else:
      vssSubOptServers = drStatus().vssSubOptUnsupportedServers.serverIp
      vssV6SubOptServers = drStatus().vssSubOptUnsupportedServers.serverIp6
   
   if vssSubOptServers is not None or vssV6SubOptServers is not None :
      vssSubOptServersModel = DhcpRelayCommonModel.VssUnsupportedServersModel()

      if vssSubOptServers:
         v4Helpers = sorted( vssSubOptServers, key=keyIpAddrVrf )
         vssSubOptServersModel.vssHelpersV4 += getAllV4Helpers( v4Helpers,
                                                                  vrfIp=True ) 
      if vssV6SubOptServers:
         v6Helpers = sorted( vssV6SubOptServers, key=keyIpv6AddrVrf )
         vssSubOptServersModel.vssHelpersV6 += getAllV6Helpers( v6Helpers,
                                                                  None,
                                                                  False,
                                                                  vrfIp=True )

      result.vssSubOptUnsupportedServers = vssSubOptServersModel 
 
      # Looping through status.intfStatus in order to differentiate
      # between interfaces configured by dhcp relay mode vs interface mode
   for intfId in Arnet.sortIntf( drStatus().intfStatus ):
      intfConfig = config.intfConfig.get( intfId )
      drIntfStatus = drStatus().intfStatus.get( intfId )
      if not drIntfStatus:
         continue
      intfUsingGlobalModeServersV4 = drIntfStatus.usingGlobalModeServersV4
      intfUsingGlobalModeServersV6 = drIntfStatus.usingGlobalModeServersV6
      if intfConfig:
         # Has interface mode configurations
         # BUT it might still have dhcp relay global mode configurations
         cfgdIntf = DhcpRelayCommonModel.ConfiguredInterface()
         if infoOpt or ( intfConfig.circuitId != intfConfig.name ):
            if intfConfig.circuitId == intfConfig.name:
               cfgdIntf.circuitId = intfConfig.circuitId
            else:
               # if non-default circuitId apply the configured format
               cfgdIntf.circuitId = optionFormatEncode( intfConfig.circuitId,
                     circuitIdFormat )
         if config.informationOptOn:
            if intfConfig.remoteId:
               # remoteId in the desired format
               cfgdIntf.remoteId = optionFormatEncode( intfConfig.remoteId,
                     remoteIdFormatV4 )
            else:
               # default value not affected by format
               cfgdIntf.remoteId = bridgingConfig.bridgeMacAddr
         if intfConfig.smartRelay == "srDefault":
            cfgdIntf.allSubnets = config.smartRelayGlobal
         elif intfConfig.smartRelay == "srOn":
            cfgdIntf.allSubnets = True
         elif intfConfig.smartRelay == "srOff":
            cfgdIntf.allSubnets = False
         if intfConfig.allSubnetsV6 == "srDefault":
            cfgdIntf.allSubnetsV6 = config.allSubnetsV6Global
         elif intfConfig.allSubnetsV6 == "srOn":
            cfgdIntf.allSubnetsV6 = True
         elif intfConfig.allSubnetsV6 == "srOff":
            cfgdIntf.allSubnetsV6 = False
         cfgdIntf.helpers = []
         
         #IPv4
         v4Helpers = []
         hostnameHelpers = []
         if intfConfig.serverIp or intfConfig.serverHostname:
            v4Helpers = sorted( intfConfig.serverIp,
                                 key=keyIpAddrVrf )
            hostnameHelpers = sorted( intfConfig.serverHostname,
                                       key=keyHostnameVrf )
            assert v4Helpers + hostnameHelpers
         # IPv6
         v6Helpers = []
         if intfConfig.serverIp6:
            v6Helpers = sorted( intfConfig.serverIp6,
                                 key=keyIpv6AddrVrf )
            assert v6Helpers
         
         cfgdIntf.helpers += getAllV4Helpers( v4Helpers )
         cfgdIntf.helpers += getAllV6Helpers( v6Helpers, intfConfig )
         cfgdIntf.helpers += getAllHostnameHelpers( hostnameHelpers )
         
         cfgdIntf.usingGlobalModeServersV4 = intfUsingGlobalModeServersV4
         cfgdIntf.usingGlobalModeServersV6 = intfUsingGlobalModeServersV6
         
         cfgdIntf.disabledV4 = intfConfig.disabledV4
         cfgdIntf.disabledV6 = intfConfig.disabledV6
         result.configuredInterfaces[ intfId ] = cfgdIntf
         if intfConfig.disabledV4 or intfConfig.disabledV6:
            # no need to check for warnings
            continue

         # check if the interface does not have any server configured
         if not ( ( intfUsingGlobalModeServersV4 or intfUsingGlobalModeServersV6 )
                  or ( v4Helpers or hostnameHelpers ) or
                  v6Helpers ):
            warning = "No DHCP server configured for interface {}"
            mode.addWarning( warning.format( intfId ) )

      else:
         # This intf only has dhcp relay global mode configurations
         if not (  intfUsingGlobalModeServersV4 or
                  intfUsingGlobalModeServersV6  ):
            continue
         cfgdIntf = DhcpRelayCommonModel.ConfiguredInterface()
         
         # Set attributes
         cfgdIntf.usingGlobalModeServersV4 = intfUsingGlobalModeServersV4
         cfgdIntf.usingGlobalModeServersV6 = intfUsingGlobalModeServersV6

         # Set global command attributes
         if config.circuitIdOptOn:
            cfgdIntf.circuitId = optionFormatEncode( intfId,
                  circuitIdFormat )
         if config.informationOptOn:
            cfgdIntf.remoteId = optionFormatEncode( 
                  bridgingConfig.bridgeMacAddr, remoteIdFormatV4 )
         cfgdIntf.allSubnets = config.smartRelayGlobal
         cfgdIntf.allSubnetsV6 = config.allSubnetsV6Global

         result.configuredInterfaces[ intfId ] = cfgdIntf

   return result

class HelperAddressCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ADDR_FAMILY helper-address'
   data = {
      'ADDR_FAMILY' : addressFamilyExprForShow,
      'helper-address' : 'DHCP server status and configuration',
   }

   handler = showIpDhcpRelay
   cliModel = DhcpRelayCommonModel.IpDhcpRelay

BasicCli.addShowCommandClass( HelperAddressCmd )

#--------------------------------------------------------------------------------
# show ip dhcp relay
#--------------------------------------------------------------------------------
class IpDhcpRelayCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip dhcp relay'
   data = {
      'ip' : ipMatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'relay' : matcherRelay,
   }

   handler = showIpDhcpRelay
   cliModel = DhcpRelayCommonModel.IpDhcpRelay

BasicCli.addShowCommandClass( IpDhcpRelayCmd )

#--------------------------------------------------------------------------------
# show ( ip | ipv6 ) dhcp relay access-list [ ACLNAME ]
#--------------------------------------------------------------------------------
class IpDhcpRelayAccessListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ADDR_FAMILY dhcp relay access-list [ ACLNAME ]'
   data = {
      'ADDR_FAMILY' : addressFamilyExprForShow,
      'dhcp' : dhcpMatcherForShow,
      'relay' : matcherRelay,
      'access-list' : AclCli.accessListMatcher,
      'ACLNAME' : AclCli.ipOrIpv6AclNameExpression,
   }

   handler = "DhcpRelayShowHandler.showAcl"
   cliModel = AllAclList

BasicCli.addShowCommandClass( IpDhcpRelayAccessListCmd )

#--------------------------------------------------------------------------------
# show ip dhcp snooping
#--------------------------------------------------------------------------------
class IpDhcpSnoopingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip dhcp snooping'
   data = {
      'ip' : ipMatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnooping,
   }

   cliModel = "DhcpRelaySnoopingModel.DhcpSnoopingModel"
   handler = "DhcpRelayShowHandler.showDhcpSnooping"

BasicCli.addShowCommandClass( IpDhcpSnoopingCmd )

#--------------------------------------------------------------------------------
# show ip dhcp snooping counters [ vlan VLANSET ] [ detail ]
#--------------------------------------------------------------------------------  
class IpDhcpSnoopingCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip dhcp snooping counters [ vlan VLANSET ] [ detail ]'
   data = {
      'ip' : ipMatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnooping,
      'counters' : matcherCounters,
      'vlan' : 'VLAN specific DHCP snooping counters',
      'VLANSET': vlanSetMatcher,
      'detail' : 'Counter values in detail'
   }

   cliModel = "DhcpRelaySnoopingModel.DhcpSnoopingCounterModel"
   handler = "DhcpRelayShowHandler.showDhcpSnoopingCounters"

BasicCli.addShowCommandClass( IpDhcpSnoopingCountersCmd )

#--------------------------------------------------------------------------------
# show ip dhcp snooping counters debug
#--------------------------------------------------------------------------------
class IpDhcpSnoopingCountersDebugCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip dhcp snooping counters debug'
   data = {
      'ip' : ipMatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnooping,
      'counters' : matcherCounters,
      'debug' : matcherDebug,
   }

   cliModel = "DhcpRelaySnoopingModel.DhcpSnoopingDebugCounterModel"
   handler = "DhcpRelayShowHandler.showDhcpSnoopingDebugCounters"

BasicCli.addShowCommandClass( IpDhcpSnoopingCountersDebugCmd )

#--------------------------------------------------------------------------------
# show ip dhcp snooping hardware
#--------------------------------------------------------------------------------
class IpDhcpSnoopingHardwareCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip dhcp snooping hardware'
   data = {
      'ip' : ipMatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnooping,
      'hardware' : 'DHCP Snooping hardware status',
   }

   cliModel = "DhcpRelaySnoopingModel.DhcpSnoopingHardwareModel"
   handler = "DhcpRelayShowHandler.showDhcpSnoopingHardware"

BasicCli.addShowCommandClass( IpDhcpSnoopingHardwareCmd )

#--------------------------------------------------------------------------------
# show ipv6 dhcp relay installed routes
#--------------------------------------------------------------------------------
class Ipv6DhcpRelayInstalledRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 dhcp relay installed routes [ VRF ]'
   data = {
      'ipv6' : ipv6MatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'relay' : matcherRelay,
      'installed' : 'Installed routes for DHCP server assigned addresses',
      'routes' : 'Install routes for DHCP server assigned addresses',
      'VRF' : VrfCli.VrfExprFactory( helpdesc="VRF name" ),
   }

   cliModel = "DhcpRelayCommonModel.Dhcp6InstalledRoutes"
   handler = "DhcpRelayShowHandler.showInstalledRoutes"

BasicCli.addShowCommandClass( Ipv6DhcpRelayInstalledRoutesCmd )

#--------------------------------------------------------------------------------
# show ipv6 dhcp snooping
#--------------------------------------------------------------------------------
class Ipv6DhcpSnoopingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 dhcp snooping'
   data = {
      'ipv6' : ipv6MatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnoopingV6,
   }

   cliModel = "DhcpRelaySnoopingModel.Dhcp6SnoopingModel"
   handler = "DhcpRelayShowHandler.showDhcp6Snooping"

BasicCli.addShowCommandClass( Ipv6DhcpSnoopingCmd )

#--------------------------------------------------------------------------------
# show ipv6 dhcp snooping counters [ vlan VLANSET ] [ detail ]
#--------------------------------------------------------------------------------
class Ipv6DhcpSnoopingCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 dhcp snooping counters [ vlan VLANSET ] [ detail ]'
   data = {
      'ipv6' : ipv6MatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnoopingV6,
      'counters' : matcherCounters,
      'vlan' : 'VLAN specific DHCP snooping counters',
      'VLANSET': vlanSetMatcher,
      'detail' : 'Counter values in detail'
   }

   cliModel = "DhcpRelaySnoopingModel.Dhcp6SnoopingCounterModel"
   handler = "DhcpRelayShowHandler.showDhcp6SnoopingCounters"

BasicCli.addShowCommandClass( Ipv6DhcpSnoopingCountersCmd )

#--------------------------------------------------------------------------------
# show ipv6 dhcp snooping counters debug
#--------------------------------------------------------------------------------
class Ipv6DhcpSnoopingCountersDebugCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 dhcp snooping counters debug'
   data = {
      'ipv6' : ipv6MatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnoopingV6,
      'counters' : matcherCounters,
      'debug' : matcherDebug,
   }

   cliModel = "DhcpRelaySnoopingModel.Dhcp6SnoopingDebugCounterModel"
   handler = "DhcpRelayShowHandler.showDhcp6SnoopingDebugCounters"

BasicCli.addShowCommandClass( Ipv6DhcpSnoopingCountersDebugCmd )

#--------------------------------------------------------------------------------
# show ipv6 dhcp snooping hardware
#--------------------------------------------------------------------------------
class Ipv6DhcpSnoopingHardwareCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 dhcp snooping hardware'
   data = {
      'ipv6' : ipv6MatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnoopingV6,
      'hardware' : 'DHCPv6 Snooping hardware status',
   }

   cliModel = "DhcpRelaySnoopingModel.Dhcp6SnoopingHardwareModel"
   handler = "DhcpRelayShowHandler.showDhcp6SnoopingHardware"

BasicCli.addShowCommandClass( Ipv6DhcpSnoopingHardwareCmd )

def Plugin( entityManager ):
   global bridgingConfig

   bridgingConfig = LazyMount.mount( entityManager, "bridging/config",
                                     "Bridging::Config", "r" )
