#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: library DhcpServer

import Arnet
import BasicCli
import CliCommand
import CliMatcher
from CliPlugin import IpAddrMatcher
from CliPlugin import Ip6AddrMatcher
from CliPlugin import MacAddr
from CliPlugin import VrfCli
from CliPlugin.IraIpRouteCliLib import isValidPrefixWithError # BUG368920
from CliPlugin.IraIp6RouteCliLib import isValidIpv6PrefixWithError
from CliPlugin.IraIpIntfCli import DhcpIntfConfigModelet # BUG368914
from CliPlugin.DhcpServerCliSubmodeLib import DhcpServerLeaseTimeBase
from CliPlugin.DhcpServerCliSubmodeLib import DhcpServerLeaseTimeExpression
from CliPlugin.DhcpServerCliSubmodeLib import DhcpServerTftpOption66Base
from CliPlugin.DhcpServerCliSubmodeLib import DhcpServerTftpOption150Base
from CliPlugin.DhcpServerCliSubmodeLib import DhcpServerTftpBootFileBase
from CliPlugin.DhcpServerCliSubmodeLib import DhcpServerPrivateOptionExpression
from CliPlugin.DhcpServerCliSubmodeLib import DhcpServerIpAddrPrivateOptionBase
from CliPlugin.DhcpServerCliSubmodeLib import DhcpServerV4DefaultGatewayBase
from CliPlugin.DhcpServerCliSubmodeLib import DhcpServerArbitraryOptionExpression
from CliPlugin.DhcpServerCliSubmodeLib import DhcpServerIpAddrArbitraryOptionBase
from CliPlugin.DhcpServerCliSubmodeLib import InformationOptionExprFactory
from CliPlugin.DhcpServerCliSubmodeLib import RemoteIdExprFactory
from CliPlugin.DhcpServerCliSubmodeLib import RemoteIdHexOrStringExprFactory
from CliPlugin.DhcpServerCliSubmodeLib import CircuitIdHexOrStringFactory
from CliPlugin.DhcpServerCliSubmodeLib import DhcpServerV4DnsServer
from CliPlugin.DhcpServerCliSubmodeLib import DhcpServerV6DnsServer
from CliPlugin.DhcpServerCliSubmodeLib import generateTftpServerExpression
from CliPlugin.DhcpServerCliSubmodeLib import IP_ADDRAdapter
from CliPlugin.DhcpServerCliSubmodeLib import IP6_ADDRAdapter
from CliPlugin.DhcpServerCliSubmodeLib import tftpBootFileAdapter
from CliPlugin.DhcpServerCliSubmodeLib import dhcpOptionExpressionData
from CliPlugin.DhcpServerCliSubmodeLib import AristaSwitchRemoteIdExpr
from CliPlugin.DhcpServerCliSubmodeLib import AristaSwitchInformationOptionExpr
from CliMode.DhcpServer import _maxDnsServers
from CliMode.DhcpServer import DhcpServerMode, DhcpServerSubnetBaseMode
from CliMode.DhcpServer import DhcpServerRangeBaseMode
from CliMode.DhcpServer import DhcpServerReservationsBaseMode
from CliMode.DhcpServer import DhcpServerReservationsMacAddressBaseMode
from CliMode.DhcpServer import DhcpServerVendorOptionBaseMode
from CliMode.DhcpServer import DhcpServerReservationsAristaSwitchBaseMode
from CliMode.DhcpServer import DhcpServerReservationsInfoOptionMode
from CliMode.DhcpServer import DhcpServerSubnetV4Mode

import CliToken
from HostnameCli import (
   HostnameMatcher
)
from EosDhcpServerLib import tacCircuitIdRemoteIdHexOrStr
from EosDhcpServerLib import featureFlexibleMatchingFuture
from EosDhcpServerLib import featureEchoClientId
from EosDhcpServerLib import featureArbitraryOptionV6
import Tac
from Url import UrlMatcher
from CliDynamicSymbol import CliDynamicPlugin

# CliDynamic Plugin
clientDynamicSubmodes = CliDynamicPlugin( "DhcpServerClientClassMode" )
assignDynamicSubmodes = CliDynamicPlugin( "DhcpServerClientAssignMode" )
matchDynamicSubmodes = CliDynamicPlugin( "DhcpServerClientMatchMode" )

dnsHelp = "DHCP DNS configuration"
dnsServerHelp = 'Set the DNS server(s) for %s DHCP clients'
dnsServerIpv4Help = 'Set the DNS server(s) for IPv4 DHCP clients'
dnsServerIpv6Help = 'Set the DNS server(s) for IPv6 DHCP clients'
dhcpHelp = 'DHCP server, client and relay interface configuration'

class EnterDhcpServerVrfMode( CliCommand.CliCommandClass ):
   syntax = '''dhcp server [ VRFNAME ]'''
   noOrDefaultSyntax = syntax

   data = {
      'dhcp': 'DHCP configuration',
      'server': 'DHCP server configuration',
      'VRFNAME': VrfCli.VrfExprFactory( helpdesc="Specify VRF for DHCP server" ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'VRFNAME' not in args:
         # If VRFNAME not specified, look it up from the routing context.
         args[ 'VRFNAME' ] = VrfCli.vrfMap.lookupCliModeVrf( mode, None )

   handler = "DhcpServerCliHandler.doEnterDhcpServerVrfMode"

   noOrDefaultHandler = "DhcpServerCliHandler.noEnterDhcpServerVrfMode"

class DhcpServerVendorOptionV4Mode( DhcpServerVendorOptionBaseMode,
                                    BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv4 Vendor Option"

   def __init__( self, parent, session, vendorId ):
      self.vrf = parent.vrf
      param = ( vendorId, self.vrf, 'ipv4' )
      DhcpServerVendorOptionBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.vendorOption = (
            parent.dhcpServerConfig.vendorOptionIpv4.newMember( vendorId ) )

class EnterDhcpServerVendorOptionMode( CliCommand.CliCommandClass ):
   syntax = '''vendor-option ipv4 ( default | VENDOR_ID )'''
   noOrDefaultSyntax = syntax

   data = {
      'vendor-option': 'Configure vendor specific option',
      'ipv4': 'Set the vendor specific option for IPv4 DHCP clients',
      'VENDOR_ID': CliMatcher.StringMatcher( helpname='VENDOR_ID',
                                             helpdesc='Vendor identifier name' ),
      'default': 'Set as default vendor specific option',
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      # IPv4
      args[ 'af' ] = 'ipv4'
      args[ 'vendorMode' ] = DhcpServerVendorOptionV4Mode
      args[ 'VENDOR_ID' ] = args.get( 'VENDOR_ID', 'default' )

   handler = "DhcpServerCliHandler.doEnterDhcpServerVendorOptionMode"

   noOrDefaultHandler = "DhcpServerCliHandler.noEnterDhcpServerVendorOptionMode"

class DhcpServerVendorOptionSubOption( CliCommand.CliCommandClass ):
   syntax = '''sub-option NUMBER type
               ( string data STRING )        |
               ( ipv4-address data { IP_ADDR } )   |
               ( array ipv4-address data { IP_ADDR } )
            '''
   noOrDefaultSyntax = '''sub-option NUMBER'''
   data = {
         'sub-option': 'Configure a sub-option',
         'NUMBER': CliMatcher.IntegerMatcher( 1, 254, helpdesc="Sub-option code" ),
         'type': 'Configure sub-option type',
         'array': 'Configure type as array',
         'string': 'Configure type as string',
         'ipv4-address': 'Configure type as ipv4-address',
         'data': 'Configure sub-option data',
         'STRING': CliMatcher.QuotedStringMatcher( requireQuote=True,
                                                   helpdesc="A quoted string" ),
         'IP_ADDR': CliCommand.Node( matcher=IpAddrMatcher.ipAddrMatcher,
                                     maxMatches=2**6 - 1 )
         }

   @staticmethod
   def adapter( mode, args, argsList ):
      # Save ipv4 address as list
      ipAddresses = args.get( 'IP_ADDR' )
      if ipAddresses and not isinstance( ipAddresses, list ):
         ipAddresses = [ ipAddresses ]
         args[ 'IP_ADDR' ] = ipAddresses

   handler = "DhcpServerCliHandler.setDhcpServerVendorOptionSubOption"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerVendorOptionSubOption"

class DhcpServerDisabled( CliCommand.CliCommandClass ):
   syntax = '''disabled'''
   noOrDefaultSyntax = '''disabled'''

   data = {
      'disabled': 'Disable DHCP server'
   }

   handler = "DhcpServerCliHandler.setDhcpServerDisabled"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerDisabled"

class DhcpServerGlobalLeaseTime( CliCommand.CliCommandClass ):
   syntax = '''lease time ( ipv4 | ipv6 ) TIME'''
   noOrDefaultSyntax = '''lease time ( ipv4 | ipv6 ) ...'''

   data = {
      'lease': 'DHCP leases configuration',
      'time': 'Set the duration of the lease',
      'ipv4': 'Set the duration for an IPv4 lease',
      'ipv6': 'Set the duration for an IPv6 lease',
      'TIME': DhcpServerLeaseTimeExpression
   }

   handler = "DhcpServerCliHandler.setDhcpServerGlobalLeaseTime"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerGlobalLeaseTime"

class DhcpServerSubnetLeaseTime( DhcpServerLeaseTimeBase ):
   data = {
      'lease': 'DHCP Subnet leases configuration',
      'time': 'Set the duration of the lease',
      'TIME': DhcpServerLeaseTimeExpression
   }

   handler = "DhcpServerCliHandler.setDhcpServerSubnetLeaseTime"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerSubnetLeaseTime"

class DhcpServerDnsDomainName( CliCommand.CliCommandClass ):
   syntax = '''dns domain name ( ipv4 | ipv6 ) NAME'''
   noOrDefaultSyntax = '''dns domain name (ipv4 | ipv6) ...'''

   data = {
      'dns': dnsHelp,
      'domain': 'DHCP DNS domain configuration',
      'name': 'Set the domain name',
      'ipv4': 'Set the domain name for IPv4 DHCP clients',
      'ipv6': 'Set the domain name for IPv6 DHCP clients',
      'NAME': CliMatcher.PatternMatcher( r'.+',
                                         helpname='WORD',
                                         helpdesc='Domain name' )
   }

   handler = "DhcpServerCliHandler.setDhcpServerDnsDomainName"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerDnsDomainName"

class DhcpServerDnsServer( CliCommand.CliCommandClass ):
   syntax = '''dns server ( ipv4 { SERVERS } ) | ( ipv6 { SERVERS6 } )'''
   noOrDefaultSyntax = '''dns server (ipv4 | ipv6) ...'''

   data = {
      'dns': dnsHelp,
      'server': "Set the DNS server(s)",
      'ipv4': dnsServerIpv4Help,
      'ipv6': dnsServerIpv6Help,
      'SERVERS': CliCommand.Node(
         matcher=IpAddrMatcher.ipAddrMatcher,
         maxMatches=_maxDnsServers ),
      'SERVERS6': CliCommand.Node(
         matcher=Ip6AddrMatcher.ip6AddrMatcher,
         maxMatches=_maxDnsServers )
   }

   handler = "DhcpServerCliHandler.setDhcpServerDnsServer"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerDnsServer"

def fsFunc( fs ):
   return fs.scheme == 'file:' and fs.supportsWrite()

class DhcpServerDebugLog( CliCommand.CliCommandClass ):
   syntax = '''debug log FILE'''
   noOrDefaultSyntax = '''debug log ...'''

   data = {
      'debug': "DHCP server debugging configuration",
      'log': "DHCP server debug log configuration",
      'FILE': UrlMatcher( fsFunc, "Location for debug log" ),
   }

   handler = "DhcpServerCliHandler.setDhcpServerDebugLog"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerDebugLog"

class DhcpServerDisableEchoClientId( CliCommand.CliCommandClass ):
   syntax = '''option ipv4 client-id disable'''
   noOrDefaultSyntax = syntax

   data = {
      'option': "DHCP option",
      'ipv4': dhcpOptionExpressionData[ 'ipv4' ],
      'client-id': "DHCP client id option",
      'disable': "Prevent DHCPv4 server from sending back client id"
   }

   handler = "DhcpServerCliHandler.disableEchoClientId"

   noOrDefaultHandler = "DhcpServerCliHandler.noDisableEchoClientId"

class DhcpServerTftpOption66( CliCommand.CliCommandClass ):
   syntax = '''TFTP_EXPR ipv4 NAME'''
   noOrDefaultSyntax = '''TFTP_EXPR ipv4 ...'''

   data = {
      'TFTP_EXPR': generateTftpServerExpression( option="66" ),
      'ipv4': 'Set the TFTP option for IPv4 DHCP clients',
      'NAME': CliMatcher.PatternMatcher( pattern=r'[a-zA-Z0-9\._-]+',
                                             helpname='SERVER',
                                             helpdesc='TFTP server name' ),
   }

   handler = "DhcpServerCliHandler.setDhcpServerTftpOption66"

   noOrDefaultHandler = handler

class DhcpServerTftpOption150( CliCommand.CliCommandClass ):
   syntax = '''TFTP_EXPR ipv4 { SERVER_ADDR }'''
   noOrDefaultSyntax = '''TFTP_EXPR ipv4 ...'''

   data = {
      'TFTP_EXPR': generateTftpServerExpression( option="150" ),
      'ipv4': 'Set the TFTP option for IPv4 DHCP clients',
      'SERVER_ADDR': CliCommand.Node( matcher=IpAddrMatcher.ipAddrMatcher ),
   }

   handler = "DhcpServerCliHandler.setDhcpServerTftpOption150"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerTftpOption150"

class DhcpServerTftpBootFile( CliCommand.CliCommandClass ):
   syntax = '''TFTP_EXPR file ( ipv4 | ipv6 ) FILENAME'''
   noOrDefaultSyntax = '''TFTP_EXPR file ( ipv4 | ipv6 ) ...'''

   data = {
      'TFTP_EXPR': generateTftpServerExpression(),
      'ipv4': 'Set the TFTP option for IPv4 DHCP clients',
      'ipv6': 'Set the TFTP option for IPv6 DHCP clients',
      'file': 'Set the boot file name',
      'FILENAME': CliMatcher.StringMatcher( helpname='FILENAME',
                                            helpdesc='TFTP boot file name' )
   }

   adapter = tftpBootFileAdapter

   handler = "DhcpServerCliHandler.setDhcpServerTftpBootFile"

   noOrDefaultHandler = handler

class DhcpServerSubnetTftpOption66( DhcpServerTftpOption66Base ):
   data = {
      'TFTP_EXPR': generateTftpServerExpression( option="66" ),
      'NAME': CliMatcher.PatternMatcher( pattern=r'[a-zA-Z0-9\._-]+',
                                             helpname='SERVER',
                                             helpdesc='TFTP server name' ),
   }

   handler = "DhcpServerCliHandler.setDhcpServerSubnetTftpOption66"

   noOrDefaultHandler = handler

class DhcpServerSubnetTftpOption150( DhcpServerTftpOption150Base ):
   data = {
      'TFTP_EXPR': generateTftpServerExpression( option="150" ),
      'SERVER_ADDR': CliCommand.Node( matcher=IpAddrMatcher.ipAddrMatcher ),
   }

   handler = "DhcpServerCliHandler.setDhcpServerSubnetTftpOption150"

   noOrDefaultHandler = handler

class DhcpServerSubnetTftpBootFile( DhcpServerTftpBootFileBase ):
   data = {
      'TFTP_EXPR': generateTftpServerExpression(),
      'file': 'Set the boot file name',
      'FILENAME': CliMatcher.StringMatcher( helpname='FILENAME',
                                            helpdesc='TFTP boot file name' ),
   }

   handler = "DhcpServerCliHandler.setDhcpServerSubnetTftpBootFile"

   noOrDefaultHandler = handler

class DhcpServerSubnetV6Mode( DhcpServerSubnetBaseMode, BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv6 Subnet"

   def __init__( self, parent, session, param ):
      prefix = Arnet.Ip6Prefix( param )
      self.vrf = parent.vrf
      self.subnetConfig = (
            parent.dhcpServerConfig.subnetConfigIpv6.newMember( prefix ) )
      param = ( self.subnetConfig, param, self.vrf, 'ipv6' )
      DhcpServerSubnetBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dhcpServerConfig = parent.dhcpServerConfig

class EnterDhcpServerSubnetMode( CliCommand.CliCommandClass ):
   syntax = '''subnet ( SUBNET | SUBNET6 )'''
   noOrDefaultSyntax = syntax

   data = {
      'subnet': 'Configure a subnet',
      'SUBNET': IpAddrMatcher.IpPrefixMatcher( 'IPv4 subnet to configure ' ),
      'SUBNET6': Ip6AddrMatcher.Ip6PrefixMatcher( 'IPv6 subnet to configure ' )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      '''
      Save the necessary attributes according to specific address family
      to 'args' to pass to handler() and noOrDefaultHandler().
      '''
      subnet = args.pop( 'SUBNET', None )
      args[ 'SUBNET' ] = subnet if subnet else args.pop( 'SUBNET6' )
      args[ 'prefixCheck' ] = isValidPrefixWithError if subnet else \
                              isValidIpv6PrefixWithError
      args[ 'subnetMode' ] = DhcpServerSubnetV4Mode if subnet else \
                            DhcpServerSubnetV6Mode
      args[ 'af' ] = 'ipv4' if subnet else 'ipv6'

   handler = "DhcpServerCliHandler.doEnterDhcpServerSubnetMode"

   noOrDefaultHandler = "DhcpServerCliHandler.noEnterDhcpServerSubnetMode"

def generateDnsServerExpression( af ):
   class DhcpServerDnsServerExpression( CliCommand.CliExpression ):
      expression = 'dns server'
      data = {
         'dns': dnsHelp,
         'server': dnsServerHelp % af
      }
   return DhcpServerDnsServerExpression

class DhcpServerSubnetV4DefaultGateway( DhcpServerV4DefaultGatewayBase ):
   data = {
         'default-gateway': 'Configure the subnet\'s default gateway '
                            'sent to DHCP clients'
         }
   data.update( DhcpServerV4DefaultGatewayBase.data )

   handler = "DhcpServerCliHandler.setDhcpServerSubnetV4DefaultGateway"

   noOrDefaultHandler = handler

class DhcpServerClientClassExpression( CliCommand.CliExpression ):
   expression = 'client class'
   data = {
      'client': 'Client options',
      'class': 'Client class options'
   }

class DhcpServerClientClassDefExprFactory( CliCommand.CliExpressionFactory ):
   def __init__( self, modeLevel ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.modeLevel = modeLevel

   def generate( self, name ):
      class DhcpServerClientClassDefinitionExpr( CliCommand.CliExpression ):
         expression = 'definition CLIENT_CLASS'
         data = {
          'definition': f'Define a {self.modeLevel} client class',
          'CLIENT_CLASS': CliMatcher.PatternMatcher( pattern=r'[a-zA-Z0-9\._-]+',
                                                     helpname='CLIENT_CLASS',
                                                     helpdesc='Client class name' )
          }
      return DhcpServerClientClassDefinitionExpr

GlobalClientClassDefExpr = DhcpServerClientClassDefExprFactory( "global" )
SubnetClientClassDefExpr = DhcpServerClientClassDefExprFactory( "subnet" )
RangeClientClassDefExpr = DhcpServerClientClassDefExprFactory( "range" )

class EnterDhcpServerClientClassBaseMode( CliCommand.CliCommandClass ):

   handler = "DhcpServerCliHandler.doEnterDhcpServerClientClassBaseMode"
   noOrDefaultHandler = "DhcpServerCliHandler.noEnterDhcpServerClientClassBaseMode"

class EnterDhcpServerClientClassMode( EnterDhcpServerClientClassBaseMode ):
   syntax = '''CLIENT_CLASS_EXPR ( ipv4 | ipv6 ) DEFINITION_EXPR'''
   noOrDefaultSyntax = syntax

   data = {
      'CLIENT_CLASS_EXPR': DhcpServerClientClassExpression,
      'ipv4': 'IPv4 DHCP client class options',
      'ipv6': 'IPv6 DHCP client class options',
      'DEFINITION_EXPR': GlobalClientClassDefExpr
    }

   @staticmethod
   def adapter( mode, args, argsList ):
      ipv4 = 'ipv4' in args
      if ipv4:
         args[ 'af' ] = 'ipv4'
         args[ 'clientClassAfMode' ] = (
            clientDynamicSubmodes.DhcpServerClientClassV4Mode )
         args[ 'clientClassConfig' ] = mode.dhcpServerConfig.clientClassConfigIpv4
      else:
         args[ 'af' ] = 'ipv6'
         args[ 'clientClassAfMode' ] = (
            clientDynamicSubmodes.DhcpServerClientClassV6Mode )
         args[ 'clientClassConfig' ] = mode.dhcpServerConfig.clientClassConfigIpv6

class DhcpServerGlobalV4PrivateOptionExpression( DhcpServerPrivateOptionExpression ):
   expression = 'private-option ipv4 CODE [ always-send ] type'
   DhcpServerPrivateOptionExpression.data.update(
         { 'ipv4': 'Private option for IPv4 DHCP clients' } )

class DhcpServerGlobalV6PrivateOptionExpression( DhcpServerPrivateOptionExpression ):
   expression = 'private-option ipv6 CODE [ always-send ] type'
   DhcpServerPrivateOptionExpression.data.update(
         { 'ipv6': 'Private option for IPv6 DHCP clients' } )

# Note, many of the following private-option command classes share the
# same noOrDefaultSyntax, so we keep one of them.

class DhcpServerGlobalV4StringPrivateOption( CliCommand.CliCommandClass ):
   syntax = '''PRIVATE-OPTION string data STRING'''
   noOrDefaultSyntax = '''private-option ipv4 CODE ...'''
   data = {
         'PRIVATE-OPTION': DhcpServerGlobalV4PrivateOptionExpression,
         'string': 'Type as string',
         'STRING': dhcpOptionExpressionData[ 'stringMatcherV4' ]
          }

   handler = "DhcpServerCliHandler.setDhcpServerGlobalV4StringPrivateOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerGlobalV4StringPrivateOption" )

class DhcpServerGlobalV6StringPrivateOption( CliCommand.CliCommandClass ):
   syntax = '''PRIVATE-OPTION string data STRING'''
   noOrDefaultSyntax = '''private-option ipv6 CODE ...'''
   data = {
         'PRIVATE-OPTION': DhcpServerGlobalV6PrivateOptionExpression,
         'string': 'Type as string',
         'STRING': dhcpOptionExpressionData[ 'stringMatcherV4' ]
          }

   handler = "DhcpServerCliHandler.setDhcpServerGlobalV6StringPrivateOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerGlobalV6StringPrivateOption" )

class DhcpServerGlobalIpv4AddrPrivateOption(
                           DhcpServerIpAddrPrivateOptionBase ):
   syntax = '''PRIVATE-OPTION ipv4-address data { IP_ADDR }'''
   data = {
         'PRIVATE-OPTION': DhcpServerGlobalV4PrivateOptionExpression,
         'ipv4-address': 'Type as IPv4 address',
         'IP_ADDR': CliCommand.Node( matcher=IpAddrMatcher.ipAddrMatcher ),
          }

   handler = "DhcpServerCliHandler.setDhcpServerGlobalIpv4AddrPrivateOption"

class DhcpServerGlobalIpv6AddrPrivateOption(
                           DhcpServerIpAddrPrivateOptionBase ):
   syntax = '''PRIVATE-OPTION ipv6-address data { IP6_ADDR }'''
   data = {
         'PRIVATE-OPTION': DhcpServerGlobalV6PrivateOptionExpression,
         'ipv6-address': 'Type as IPv6 address',
         'IP6_ADDR': CliCommand.Node( matcher=Ip6AddrMatcher.ip6AddrMatcher ),
          }

   handler = "DhcpServerCliHandler.setDhcpServerGlobalIpv6AddrPrivateOption"

class DhcpServerV4RangeMode( DhcpServerRangeBaseMode, BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv4 Range"

   # param will be set to ( subnetConfig, range, vrf )
   def __init__( self, parent, session, param ):
      param = param + ( 'ipv4', )
      DhcpServerRangeBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dhcpServerConfig = parent.dhcpServerConfig

class DhcpServerV6RangeMode( DhcpServerRangeBaseMode, BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv6 Range"

   # param will be set to ( subnetConfig, range, vrf )
   def __init__( self, parent, session, param ):
      param = param + ( 'ipv6', )
      DhcpServerRangeBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dhcpServerConfig = parent.dhcpServerConfig

class EnterDhcpServerRangeModeBase( CliCommand.CliCommandClass ):
   syntax = '''range START END'''
   noOrDefaultSyntax = syntax

   @staticmethod
   def adapter( mode, args, argsList ):
      '''
      Save the necessary attributes according to specific address family
      to 'args' to pass to handler() and noOrDefaultHandler().
      '''
      ipv4 = isinstance( mode, DhcpServerSubnetV4Mode )
      if ipv4:
         args[ 'rangeMode' ] = DhcpServerV4RangeMode
         args[ 'af' ] = 'ipv4'
         ipAddrFn = Arnet.IpAddr
         tacRangeFn = Tac.Type( "DhcpServer::Range" )
      else:
         args[ 'rangeMode' ] = DhcpServerV6RangeMode
         args[ 'af' ] = 'ipv6'
         ipAddrFn = Arnet.Ip6Addr
         tacRangeFn = Tac.Type( "DhcpServer::Range6" )
      start = ipAddrFn( args[ 'START' ] )
      end = ipAddrFn( args[ 'END' ] )

      if start > end:
         mode.addErrorAndStop( 'Start range cannot be larger than end of range' )

      args[ 'tacRange' ] = tacRangeFn( start, end )

   handler = "DhcpServerCliHandler.doEnterDhcpServerRangeModeBase"

   noOrDefaultHandler = "DhcpServerCliHandler.noEnterDhcpServerRangeModeBase"

class EnterDhcpServerV4RangeMode( EnterDhcpServerRangeModeBase ):
   data = {
      'range': 'Configure range of addresses',
      'START': IpAddrMatcher.IpAddrMatcher(
         'Start of subnet range IPv4 (inclusive)' ),
      'END': IpAddrMatcher.IpAddrMatcher( 'End of subnet range IPv4 (inclusive)' ),
   }

class EnterDhcpServerV6RangeMode( EnterDhcpServerRangeModeBase ):
   data = {
      'range': 'Configure range of addresses',
      'START': Ip6AddrMatcher.Ip6AddrMatcher(
         'Start of subnet range IPv6 (inclusive)' ),
      'END': Ip6AddrMatcher.Ip6AddrMatcher( 'End of subnet range IPv6 (inclusive)' ),
   }

class DhcpServerClientClassAssignmentBase( CliCommand.CliCommandClass ):
   '''
   Command to associate a global client class with a subnet or range.
   '''
   syntax = '''client class assignment { CLIENT_CLASS }'''
   noOrDefaultSyntax = '''client class assignment ...'''


class DhcpServerRangeClientClassAssignment( DhcpServerClientClassAssignmentBase ):
   '''
   Command to associate a global client class with a range.
   '''
   data = {
      'client': 'Client options',
      'class': 'Client class options',
      'assignment': 'Assign one or more client classes',
      'CLIENT_CLASS': CliMatcher.PatternMatcher( pattern=r'[a-zA-Z0-9\._-]+',
                                                 helpname='CLIENT_CLASS',
                                                 helpdesc='Client class name' )
   }

   handler = "DhcpServerCliHandler.setDhcpServerRangeClientClassAssignment"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerRangeClientClassAssignment"

class DhcpServerSubnetClientClassAssignment( DhcpServerClientClassAssignmentBase ):
   '''
   Command to associate a global client class with a subnet.
   '''
   data = {
      'client': 'Client options',
      'class': 'Client class options',
      'assignment': 'Assign one or more client classes',
      'CLIENT_CLASS': CliMatcher.PatternMatcher( pattern=r'[a-zA-Z0-9\._-]+',
                                                 helpname='CLIENT_CLASS',
                                                 helpdesc='Client class name' )
   }

   handler = "DhcpServerCliHandler.setDhcpServerSubnetClientClassAssignment"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerSubnetClientClassAssignment" )

class DhcpServerSubnetRangeBase( CliCommand.CliCommandClass ):
   syntax = '''range START END'''
   noOrDefaultSyntax = syntax

   handler = "DhcpServerCliHandler.setDhcpServerSubnetRangeBase"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerSubnetRangeBase"

class DhcpServerSubnetV4Range( DhcpServerSubnetRangeBase ):
   data = {
      'range': 'Configure range of addresses',
      'START': IpAddrMatcher.IpAddrMatcher(
         'Start of subnet range IPv4 (inclusive)' ),
      'END': IpAddrMatcher.IpAddrMatcher( 'End of subnet range IPv4 (inclusive)' ),
   }

class DhcpServerSubnetV6Range( DhcpServerSubnetRangeBase ):
   data = {
      'range': 'Configure range of addresses',
      'START': Ip6AddrMatcher.Ip6AddrMatcher(
         'Start of subnet range IPv6 (inclusive)' ),
      'END': Ip6AddrMatcher.Ip6AddrMatcher( 'End of subnet range IPv6 (inclusive)' ),
   }

class DhcpServerSubnetName( CliCommand.CliCommandClass ):
   syntax = '''name NAME'''
   noOrDefaultSyntax = '''name ...'''

   data = {
      'name': 'Configure name of subnet',
      'NAME': CliMatcher.StringMatcher( helpname='NAME',
                                        helpdesc='Name of subnet' )
   }

   handler = "DhcpServerCliHandler.setDhcpServerSubnetName"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerSubnetName"

class InterfaceEnterDhcpServerEnable( CliCommand.CliCommandClass ):
   syntax = '''dhcp server ( ipv4 | ipv6 )'''
   noOrDefaultSyntax = syntax

   data = {
      'dhcp': dhcpHelp,
      'server': 'DHCP server interface configuration',
      'ipv4': 'Enable IPv4 DHCP server',
      'ipv6': 'Enable IPv6 DHCP server',
   }

   handler = "DhcpServerCliHandler.setInterfaceEnterDhcpServerEnable"

   noOrDefaultHandler = "DhcpServerCliHandler.noInterfaceEnterDhcpServerEnable"

class DhcpServerClearLeasesBase( CliCommand.CliCommandClass ):
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'dhcp': 'Clear DHCP information',
      'server': 'Clear DHCP server information',
      'ipv4': 'Clear IPv4 DHCP server information',
      'ipv6': 'Clear IPv6 DHCP server information',
      'leases': 'Clear DHCP server leases',
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ipv4' in args:
         args[ 'af' ] = 'ipv4'
         args[ 'addrFn' ] = Arnet.IpAddr
         args[ 'leaseAddr' ] = args.get( 'LEASE_ADDR', None )
      else:
         args[ 'af' ] = 'ipv6'
         args[ 'addrFn' ] = Arnet.Ip6Addr
         args[ 'leaseAddr' ] = args.get( 'LEASE_ADDR6', None )

class DhcpServerClearLeasesVrf( DhcpServerClearLeasesBase ):
   syntax = '''clear dhcp server ( ipv4 | ipv6 ) leases [ VRFNAME ]'''

   data = { 'VRFNAME': VrfCli.VrfExprFactory(
                           helpdesc="Specify VRF for DHCP server",
                           inclAllVrf=True ) }
   data.update( DhcpServerClearLeasesBase.data )

   handler = "DhcpServerCliHandler.setDhcpServerClearLeasesVrf"

class DhcpServerClearIpv4LeaseByAddrVrf( DhcpServerClearLeasesBase ):
   syntax = '''clear dhcp server ipv4 leases [ VRFNAME ] LEASE_ADDR'''

   data = { 'VRFNAME': VrfCli.VrfExprFactory(
                           helpdesc="Specify VRF for DHCP server",
                           inclAllVrf=True ),
            'LEASE_ADDR': IpAddrMatcher.IpAddrMatcher( 'Lease address' ) }
   data.update( DhcpServerClearLeasesBase.data )

   handler = "DhcpServerCliHandler.setDhcpServerClearIpv4LeaseByAddrVrf"

class DhcpServerClearIpv6LeaseByAddrVrf( DhcpServerClearLeasesBase ):
   syntax = '''clear dhcp server ipv6 leases [ VRFNAME ] LEASE_ADDR6'''

   data = { 'VRFNAME': VrfCli.VrfExprFactory(
                           helpdesc="Specify VRF for DHCP server",
                           inclAllVrf=True ),
            'LEASE_ADDR6': Ip6AddrMatcher.Ip6AddrMatcher( 'Lease address' ) }
   data.update( DhcpServerClearLeasesBase.data )

   handler = "DhcpServerCliHandler.setDhcpServerClearIpv6LeaseByAddrVrf"

class EnterDhcpServerSubnetClientClass( EnterDhcpServerClientClassBaseMode ):
   syntax = '''CLIENT_CLASS_EXPR DEFINITION_EXPR'''
   noOrDefaultSyntax = syntax

   data = {
         'CLIENT_CLASS_EXPR': DhcpServerClientClassExpression,
         'DEFINITION_EXPR': SubnetClientClassDefExpr
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      '''
      Save the necessary attributes according to specific address family
      to 'args' to pass to handler() and noOrDefaultHandler().
      '''
      ipv4 = isinstance( mode, DhcpServerSubnetV4Mode )
      args[ 'clientClassConfig' ] = mode.subnetConfig.clientClassConfig
      if ipv4:
         args[ 'clientClassAfMode' ] = (
            clientDynamicSubmodes.DhcpServerSubnetV4ClientClassMode )
      else:
         args[ 'clientClassAfMode' ] = (
            clientDynamicSubmodes.DhcpServerSubnetV6ClientClassMode )

   handler = "DhcpServerCliHandler.doEnterDhcpServerSubnetClientClass"

class EnterDhcpServerRangeClientClass( EnterDhcpServerClientClassBaseMode ):
   syntax = '''client class definition CLIENT_CLASS'''
   noOrDefaultSyntax = syntax

   data = {
         'CLIENT_CLASS_EXPR': DhcpServerClientClassExpression,
         'DEFINITION_EXPR': RangeClientClassDefExpr
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      ipv4 = isinstance( mode, DhcpServerV4RangeMode )
      args[ 'clientClassConfig' ] = mode.rangeConfig.clientClassConfig
      if ipv4:
         args[ 'clientClassAfMode' ] = (
            clientDynamicSubmodes.DhcpServerV4RangeClientClassMode )
      else:
         args[ 'clientClassAfMode' ] = (
            clientDynamicSubmodes.DhcpServerV6RangeClientClassMode )

   handler = "DhcpServerCliHandler.doEnterDhcpServerRangeClientClass"

# reservations mode
class DhcpServerReservationsV4Mode( DhcpServerReservationsBaseMode,
                                    BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv4 Reservations"

   def __init__( self, parent, session, subnetConfig ):
      self.vrf = parent.vrf
      param = ( subnetConfig, self.vrf, 'ipv4' )
      DhcpServerReservationsBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class DhcpServerReservationsV6Mode( DhcpServerReservationsBaseMode,
                                    BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv6 Reservations"

   def __init__( self, parent, session, subnetConfig ):
      self.vrf = parent.vrf
      param = ( subnetConfig, self.vrf, 'ipv6' )
      DhcpServerReservationsBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class EnterDhcpServerReservationsMode( CliCommand.CliCommandClass ):
   syntax = '''reservations'''
   noOrDefaultSyntax = syntax

   data = {
         'reservations': 'DHCP Client reservations configuration'
         }

   @staticmethod
   def adapter( mode, args, argsList ):
      '''
      Save the necessary attributes according to specific address family
      to 'args' to pass to handler() and noOrDefaultHandler().
      '''
      ipv4 = isinstance( mode, DhcpServerSubnetV4Mode )
      if ipv4:
         args[ 'reservationMode' ] = DhcpServerReservationsV4Mode
      else:
         args[ 'reservationMode' ] = DhcpServerReservationsV6Mode

   handler = "DhcpServerCliHandler.doEnterDhcpServerReservationsMode"

   noOrDefaultHandler = "DhcpServerCliHandler.noEnterDhcpServerReservationsMode"

# reservations mac-address mode
class DhcpServerReservationsMacAddressV4Mode(
                            DhcpServerReservationsMacAddressBaseMode,
                            BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv4 Mac Address Reservations"

   def __init__( self, parent, session, subnetConfig, macAddr ):
      macAddr = Arnet.EthAddr( macAddr )
      self.vrf = parent.vrf
      param = ( subnetConfig, macAddr.displayString, self.vrf, 'ipv4' )
      DhcpServerReservationsMacAddressBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class DhcpServerReservationsMacAddressV6Mode(
                            DhcpServerReservationsMacAddressBaseMode,
                            BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv6 Mac Address Reservations"

   def __init__( self, parent, session, subnetConfig, macAddr ):
      macAddr = Arnet.EthAddr( macAddr )
      self.vrf = parent.vrf
      param = ( subnetConfig, macAddr.displayString, self.vrf, 'ipv6' )
      DhcpServerReservationsMacAddressBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class EnterDhcpServerReservationsMacAddressMode( CliCommand.CliCommandClass ):
   syntax = '''mac-address MAC_ADDR'''
   noOrDefaultSyntax = syntax

   data = {
         'mac-address': "Reserve using the client's mac-address",
         'MAC_ADDR': MacAddr.macAddrMatcher,
         }

   @staticmethod
   def adapter( mode, args, argsList ):
      '''
      Save the necessary attributes according to specific address family
      to 'args' to pass to handler() and noOrDefaultHandler().
      '''
      ipv4 = isinstance( mode, DhcpServerReservationsV4Mode )
      if ipv4:
         args[ 'macAddressMode' ] = DhcpServerReservationsMacAddressV4Mode
      else:
         args[ 'macAddressMode' ] = DhcpServerReservationsMacAddressV6Mode

   handler = "DhcpServerCliHandler.doEnterDhcpServerReservationsMacAddressMode"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noEnterDhcpServerReservationsMacAddressMode" )

class DhcpServerReservationsAristaInfoOptionMode(
                           DhcpServerReservationsAristaSwitchBaseMode,
                           BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv4 Arista Information Option Reservations"

   def __init__( self, parent, session, subnetConfig, port, vlan, macAddr ):
      self.vrf = parent.vrf
      param = ( subnetConfig, port.stringValue, vlan, macAddr, self.vrf )
      DhcpServerReservationsAristaSwitchBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   @property
   def af( self ):
      return "ipv4"

class DhcpServerReservationsAristaRemoteIdMode(
                           DhcpServerReservationsAristaSwitchBaseMode,
                           BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv6 Arista Remote Id Reservations"

   def __init__( self, parent, session, subnetConfig, port, vlan, macAddr ):
      self.vrf = parent.vrf
      param = ( subnetConfig, port.stringValue, vlan, macAddr, self.vrf )
      DhcpServerReservationsAristaSwitchBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   @property
   def af( self ):
      return "ipv6"

class EnterDhcpServerReservationsAristaBaseMode( CliCommand.CliCommandClass ):
   @staticmethod
   def adapter( mode, args, argsList ):
      if 'VLAN' in args:
         args[ 'PORT' ] = Arnet.IntfId( args[ 'INTF' ] )
         args[ 'VLAN' ] = args[ 'VLAN' ].id
         mac = args.get( 'MAC', '0000.0000.0000' )
         args[ 'MAC' ] = Arnet.EthAddr( mac ).displayString
      else:
         args[ 'PORT' ] = Arnet.IntfId( '' )
         args[ 'VLAN' ] = 0
         args[ 'MAC' ] = Arnet.EthAddr( args[ 'MAC' ] ).displayString

      ipv4 = isinstance( mode, DhcpServerReservationsV4Mode )
      if ipv4:
         args[ 'mode' ] = DhcpServerReservationsAristaInfoOptionMode
      else:
         args[ 'mode' ] = DhcpServerReservationsAristaRemoteIdMode


   handler = "DhcpServerCliHandler.doEnterDhcpServerReservationsAristaBaseMode"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noEnterDhcpServerReservationsAristaBaseMode" )

reservationHeader = "Reserve using the"
ReservationInformationOptionExpr = (
      InformationOptionExprFactory( reservationHeader ) )
ReservationRemoteIdExpr = RemoteIdExprFactory( reservationHeader )
ReservationV4RemoteIdHexOrStringExpr = (
      RemoteIdHexOrStringExprFactory( ReservationRemoteIdExpr ) )
ReservationV4CircuitIdHexOrStringExpr = (
      CircuitIdHexOrStringFactory( reservationHeader ) )

class EnterDhcpServerReservationsAristaInfoOptionMode(
      EnterDhcpServerReservationsAristaBaseMode ):
   syntax = '''INFORMATION_OPTION ARISTA_SWITCH'''
   noOrDefaultSyntax = syntax

   data = {
         'INFORMATION_OPTION': ReservationInformationOptionExpr,
         'ARISTA_SWITCH': AristaSwitchInformationOptionExpr
   }

class EnterDhcpServerReservationsAristaRemoteIdMode(
      EnterDhcpServerReservationsAristaBaseMode ):
   syntax = 'remote-id ARISTA_SWITCH'
   noOrDefaultSyntax = syntax

   data = {
         'remote-id': 'Reserve by Remote ID (Option 37)',
         'ARISTA_SWITCH': AristaSwitchRemoteIdExpr,
         }

class EnterDhcpServerReservationsInfoOptionMode( CliCommand.CliCommandClass ):
   syntax = '''INFORMATION-OPTION (( CIRCUIT-ID [ REMOTE-ID ] ) | REMOTE-ID )'''
   noOrDefaultSyntax = syntax

   data = {
      'INFORMATION-OPTION': ReservationInformationOptionExpr,
      'CIRCUIT-ID': ReservationV4CircuitIdHexOrStringExpr,
      'REMOTE-ID': ReservationV4RemoteIdHexOrStringExpr,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      circuitIdHex = args.get( 'CIRC_HEX', '' )
      circuitIdStr = args.get( 'CIRC_STR', '' )
      remoteIdHex = args.get( 'REMOTE_HEX', '' )
      remoteIdStr = args.get( 'REMOTE_STR', '' )
      args[ 'infoOpt' ] = tacCircuitIdRemoteIdHexOrStr( circuitIdHex, circuitIdStr,
                                                        remoteIdHex, remoteIdStr )

   handler = "DhcpServerCliHandler.doEnterDhcpServerReservationsInfoOptionMode"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noEnterDhcpServerReservationsInfoOptionMode" )

class DhcpServerReservationsIpAddress( CliCommand.CliCommandClass ):
   syntax = '''ipv4-address IP_ADDR'''
   noOrDefaultSyntax = '''ipv4-address ...'''

   data = {
         'ipv4-address': 'Reserve an IPv4 address',
         'IP_ADDR': IpAddrMatcher.IpAddrMatcher( 'IPv4 address' ),
         }

   adapter = IP_ADDRAdapter

   handler = "DhcpServerCliHandler.setDhcpServerReservationsIpAddress"

   noOrDefaultHandler = handler

class DhcpServerReservationsHostname( CliCommand.CliCommandClass ):
   syntax = '''hostname HOSTNAME'''
   noOrDefaultSyntax = '''hostname ...'''

   data = {
         'hostname': 'Reserve a host name',
         'HOSTNAME': HostnameMatcher( helpname="HOST",
                                      helpdesc="Hostname to reserve" ),
         }

   handler = "DhcpServerCliHandler.setDhcpServerReservationsHostname"

   noOrDefaultHandler = handler

class DhcpServerReservationsIpAddressV6( CliCommand.CliCommandClass ):
   syntax = '''ipv6-address IP_ADDR'''
   noOrDefaultSyntax = '''ipv6-address ...'''

   data = {
         'ipv6-address': 'Reserve an IPv6 address',
         'IP_ADDR': Ip6AddrMatcher.Ip6AddrMatcher( 'IPv6 address' ),
         }

   adapter = IP6_ADDRAdapter

   handler = "DhcpServerCliHandler.setDhcpServerReservationsIpAddressV6"

   noOrDefaultHandler = handler

class DhcpServerGlobalV4ArbitraryOptionStringExpression(
      DhcpServerArbitraryOptionExpression ):
   expression = 'option ipv4 CODE_STRING_V4 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'ipv4': dhcpOptionExpressionData[ 'ipv4' ],
         'CODE_STRING_V4': dhcpOptionExpressionData[ 'CODE_STRING_V4' ]
      }
   )

class DhcpServerGlobalV4ArbitraryOptionFqdnExpression(
      DhcpServerArbitraryOptionExpression ):
   expression = 'option ipv4 CODE_FQDN_V4 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'ipv4': dhcpOptionExpressionData[ 'ipv4' ],
         'CODE_FQDN_V4': dhcpOptionExpressionData[ 'CODE_FQDN_V4' ]
      }
   )

class DhcpServerGlobalV4ArbitraryOptionIpAddrExpression(
      DhcpServerArbitraryOptionExpression ):
   expression = 'option ipv4 CODE_IPADDR_V4 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'ipv4': dhcpOptionExpressionData[ 'ipv4' ],
         'CODE_IPADDR_V4': dhcpOptionExpressionData[ 'CODE_IPADDR_V4' ]
      }
   )

class DhcpServerGlobalV4ArbitraryOptionHexExpression(
      DhcpServerArbitraryOptionExpression ):
   expression = 'option ipv4 CODE_HEX_V4 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'ipv4': dhcpOptionExpressionData[ 'ipv4' ],
         'CODE_HEX_V4': dhcpOptionExpressionData[ 'CODE_HEX_V4' ]
      }
   )

class DhcpServerGlobalV6ArbitraryOptionStringExpression(
      DhcpServerArbitraryOptionExpression ):
   expression = 'option ipv6 CODE_STRING_V6 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'ipv6': dhcpOptionExpressionData[ 'ipv6' ],
         'CODE_STRING_V6': dhcpOptionExpressionData[ 'CODE_STRING_V6' ]
      }
   )

class DhcpServerGlobalV6ArbitraryOptionFqdnExpression(
      DhcpServerArbitraryOptionExpression ):
   expression = 'option ipv6 CODE_FQDN_V6 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'ipv6': dhcpOptionExpressionData[ 'ipv6' ],
         'CODE_FQDN_V6': dhcpOptionExpressionData[ 'CODE_FQDN_V6' ]
      }
   )

class DhcpServerGlobalV6ArbitraryOptionIpAddrExpression(
      DhcpServerArbitraryOptionExpression ):
   expression = 'option ipv6 CODE_IPADDR_V6 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'ipv6': dhcpOptionExpressionData[ 'ipv6' ],
         'CODE_IPADDR_V6': dhcpOptionExpressionData[ 'CODE_IPADDR_V6' ]
      }
   )

class DhcpServerGlobalV6ArbitraryOptionHexExpression(
      DhcpServerArbitraryOptionExpression ):
   expression = 'option ipv6 CODE_HEX_V6 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'ipv6': dhcpOptionExpressionData[ 'ipv6' ],
         'CODE_HEX_V6': dhcpOptionExpressionData[ 'CODE_HEX_V6' ]
      }
   )
# -------------------------------------------------------------------------------
# The "[no|default] option ipv4 CODE [always-send] type string data STRING"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------
class DhcpServerGlobalV4StringArbitraryOption( CliCommand.CliCommandClass ):
   syntax = '''OPTION string data STRING'''
   noOrDefaultSyntax = '''OPTION string...'''
   data = {
      'OPTION': DhcpServerGlobalV4ArbitraryOptionStringExpression,
      'string': 'Type as string',
      'STRING': dhcpOptionExpressionData[ 'stringMatcherV4' ]
   }

   handler = "DhcpServerCliHandler.setDhcpServerGlobalV4StringArbitraryOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerGlobalV4StringArbitraryOption" )

# The "[no|default] option ipv4 CODE [always-send] type fqdn data FQDN"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------

class DhcpServerGlobalV4FqdnArbitraryOption( CliCommand.CliCommandClass ):
   syntax = '''OPTION fqdn data FQDN'''
   noOrDefaultSyntax = '''OPTION fqdn...'''
   data = {
      'OPTION': DhcpServerGlobalV4ArbitraryOptionFqdnExpression,
      'fqdn': 'Type as fully qualified domain name',
      'FQDN': dhcpOptionExpressionData[ 'fqdnMatcherV4' ]
   }

   handler = "DhcpServerCliHandler.setDhcpServerGlobalV4FqdnArbitraryOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerGlobalV4FqdnArbitraryOption" )

# -------------------------------------------------------------------------------
# The "[no|default] option ipv4 CODE [always-send] type hex data HEX"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------
class DhcpServerGlobalV4HexArbitraryOption( CliCommand.CliCommandClass ):
   syntax = '''OPTION hex data HEX'''
   noOrDefaultSyntax = '''OPTION hex...'''
   data = {
      'OPTION': DhcpServerGlobalV4ArbitraryOptionHexExpression,
      'hex': 'Type as hex',
      'HEX': dhcpOptionExpressionData[ 'hexMatcherV4' ]
   }

   handler = "DhcpServerCliHandler.setDhcpServerGlobalV4HexArbitraryOption"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerGlobalV4HexArbitraryOption"

# -------------------------------------------------------------------------------
# The "[no|default] option ipv4 CODE [always-send] type ipv4-address data DATA"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------
class DhcpServerGlobalIpv4AddrArbitraryOption(
                           DhcpServerIpAddrArbitraryOptionBase ):
   syntax = '''OPTION ipv4-address data { IP_ADDRS }'''
   noOrDefaultSyntax = '''OPTION ipv4-address...'''
   data = {
      'OPTION': DhcpServerGlobalV4ArbitraryOptionIpAddrExpression,
      'ipv4-address': 'Type as IPv4 address',
      'IP_ADDRS': IpAddrMatcher.ipAddrMatcher,
   }

   handler = "DhcpServerCliHandler.setDhcpServerGlobalIpv4AddrArbitraryOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerGlobalIpv4AddrArbitraryOption" )

# -------------------------------------------------------------------------------
# The "[no|default] option ipv6 CODE [always-send] type string data STRING"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------
class DhcpServerGlobalV6StringArbitraryOption( CliCommand.CliCommandClass ):
   syntax = '''OPTION string data STRING'''
   noOrDefaultSyntax = '''OPTION string...'''
   data = {
      'OPTION': DhcpServerGlobalV6ArbitraryOptionStringExpression,
      'string': 'Type as string',
      'STRING': dhcpOptionExpressionData[ 'stringMatcherV6' ]
   }

   handler = "DhcpServerCliHandler.setDhcpServerGlobalV6StringArbitraryOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerGlobalV6StringArbitraryOption" )

# -------------------------------------------------------------------------------
# The "[no|default] option ipv6 CODE [always-send] type fqdn data FQDN"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------
class DhcpServerGlobalV6FqdnArbitraryOption( CliCommand.CliCommandClass ):
   syntax = '''OPTION fqdn data FQDN'''
   noOrDefaultSyntax = '''OPTION fqdn...'''
   data = {
      'OPTION': DhcpServerGlobalV6ArbitraryOptionFqdnExpression,
      'fqdn': 'Type as fully qualified domain name',
      'FQDN': dhcpOptionExpressionData[ 'fqdnMatcherV6' ]
   }

   handler = "DhcpServerCliHandler.setDhcpServerGlobalV6FqdnArbitraryOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerGlobalV6FqdnArbitraryOption" )

# -------------------------------------------------------------------------------
# The "[no|default] option ipv6 CODE [always-send] type hex data HEX"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------
class DhcpServerGlobalV6HexArbitraryOption( CliCommand.CliCommandClass ):
   syntax = '''OPTION hex data HEX'''
   noOrDefaultSyntax = '''OPTION hex...'''
   data = {
      'OPTION': DhcpServerGlobalV6ArbitraryOptionHexExpression,
      'hex': 'Type as hex',
      'HEX': dhcpOptionExpressionData[ 'hexMatcherV6' ]
   }

   handler = "DhcpServerCliHandler.setDhcpServerGlobalV6HexArbitraryOption"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerGlobalV6HexArbitraryOption"

# -------------------------------------------------------------------------------
# The "[no|default] option ipv6 CODE [always-send] type ipv6-address data DATA"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------
class DhcpServerGlobalIpv6AddrArbitraryOption(
                           DhcpServerIpAddrArbitraryOptionBase ):
   syntax = '''OPTION ipv6-address data { IP6_ADDRS }'''
   noOrDefaultSyntax = '''OPTION ipv6-address...'''
   data = {
      'OPTION': DhcpServerGlobalV6ArbitraryOptionIpAddrExpression,
      'ipv6-address': 'Type as IPv6 address',
      'IP6_ADDRS': CliCommand.Node( matcher=Ip6AddrMatcher.ip6AddrMatcher ),
   }

   handler = "DhcpServerCliHandler.setDhcpServerGlobalIpv6AddrArbitraryOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerGlobalIpv6AddrArbitraryOption" )

BasicCli.GlobalConfigMode.addCommandClass( EnterDhcpServerVrfMode )
DhcpServerMode.addCommandClass( DhcpServerDisabled )
DhcpServerMode.addCommandClass( DhcpServerGlobalLeaseTime )
DhcpServerMode.addCommandClass( DhcpServerDnsDomainName )
DhcpServerMode.addCommandClass( DhcpServerDnsServer )
DhcpServerMode.addCommandClass( DhcpServerDebugLog )
DhcpServerMode.addCommandClass( DhcpServerTftpOption66 )
DhcpServerMode.addCommandClass( DhcpServerTftpOption150 )
DhcpServerMode.addCommandClass( DhcpServerTftpBootFile )
DhcpServerMode.addCommandClass( EnterDhcpServerSubnetMode )

if featureEchoClientId():
   DhcpServerMode.addCommandClass( DhcpServerDisableEchoClientId )

DhcpServerMode.addCommandClass( DhcpServerGlobalV4StringArbitraryOption )
DhcpServerMode.addCommandClass( DhcpServerGlobalV4FqdnArbitraryOption )
DhcpServerMode.addCommandClass( DhcpServerGlobalV4HexArbitraryOption )
DhcpServerMode.addCommandClass( DhcpServerGlobalIpv4AddrArbitraryOption )

if featureArbitraryOptionV6():
   DhcpServerMode.addCommandClass( DhcpServerGlobalV6StringArbitraryOption )
   DhcpServerMode.addCommandClass( DhcpServerGlobalV6FqdnArbitraryOption )
   DhcpServerMode.addCommandClass( DhcpServerGlobalV6HexArbitraryOption )
   DhcpServerMode.addCommandClass( DhcpServerGlobalIpv6AddrArbitraryOption )

DhcpServerMode.addCommandClass( EnterDhcpServerClientClassMode )
DhcpServerMode.addCommandClass( DhcpServerGlobalV4StringPrivateOption )
DhcpServerMode.addCommandClass( DhcpServerGlobalV6StringPrivateOption )
DhcpServerMode.addCommandClass( DhcpServerGlobalIpv4AddrPrivateOption )
DhcpServerMode.addCommandClass( DhcpServerGlobalIpv6AddrPrivateOption )

DhcpServerMode.addCommandClass( EnterDhcpServerVendorOptionMode )
DhcpServerVendorOptionV4Mode.addCommandClass( DhcpServerVendorOptionSubOption )

DhcpServerSubnetV4Mode.addCommandClass( EnterDhcpServerV4RangeMode )
DhcpServerSubnetV4Mode.addCommandClass( EnterDhcpServerSubnetClientClass )
if featureFlexibleMatchingFuture():
   DhcpServerSubnetV4Mode.addCommandClass( DhcpServerSubnetClientClassAssignment )
   DhcpServerV4RangeMode.addCommandClass( DhcpServerRangeClientClassAssignment )

DhcpServerSubnetV4Mode.addCommandClass( DhcpServerV4DnsServer )
DhcpServerSubnetV4Mode.addCommandClass( DhcpServerSubnetV4DefaultGateway )
DhcpServerSubnetV4Mode.addCommandClass( DhcpServerSubnetName )
DhcpServerSubnetV4Mode.addCommandClass( DhcpServerSubnetTftpOption66 )
DhcpServerSubnetV4Mode.addCommandClass( DhcpServerSubnetTftpOption150 )
DhcpServerSubnetV4Mode.addCommandClass( DhcpServerSubnetTftpBootFile )
DhcpServerSubnetV4Mode.addCommandClass( DhcpServerSubnetLeaseTime )

DhcpServerSubnetV4Mode.addCommandClass( EnterDhcpServerReservationsMode )
DhcpServerReservationsV4Mode.addCommandClass(
                     EnterDhcpServerReservationsMacAddressMode )
DhcpServerReservationsMacAddressV4Mode.addCommandClass(
                     DhcpServerReservationsIpAddress )
DhcpServerReservationsMacAddressV4Mode.addCommandClass(
                     DhcpServerReservationsHostname )

DhcpServerSubnetV6Mode.addCommandClass( DhcpServerSubnetTftpBootFile )
DhcpServerSubnetV6Mode.addCommandClass( EnterDhcpServerV6RangeMode )
DhcpServerSubnetV6Mode.addCommandClass( EnterDhcpServerSubnetClientClass )
DhcpServerSubnetV6Mode.addCommandClass( DhcpServerSubnetClientClassAssignment )
DhcpServerV6RangeMode.addCommandClass( DhcpServerRangeClientClassAssignment )

DhcpServerSubnetV6Mode.addCommandClass( EnterDhcpServerReservationsMode )
DhcpServerReservationsV6Mode.addCommandClass(
               EnterDhcpServerReservationsMacAddressMode )
DhcpServerReservationsMacAddressV6Mode.addCommandClass(
                     DhcpServerReservationsIpAddressV6 )
DhcpServerReservationsMacAddressV6Mode.addCommandClass(
                     DhcpServerReservationsHostname )

if featureFlexibleMatchingFuture():
   DhcpServerReservationsV6Mode.addCommandClass(
         EnterDhcpServerReservationsAristaRemoteIdMode )
   DhcpServerReservationsAristaRemoteIdMode.addCommandClass(
         DhcpServerReservationsIpAddressV6 )
   DhcpServerReservationsV4Mode.addCommandClass(
         EnterDhcpServerReservationsAristaInfoOptionMode )
   DhcpServerReservationsAristaInfoOptionMode.addCommandClass(
         DhcpServerReservationsIpAddress )
   DhcpServerReservationsV4Mode.addCommandClass(
         EnterDhcpServerReservationsInfoOptionMode )
   DhcpServerReservationsInfoOptionMode.addCommandClass(
         DhcpServerReservationsIpAddress )

DhcpServerV6RangeMode.addCommandClass( EnterDhcpServerRangeClientClass )
DhcpServerV4RangeMode.addCommandClass( EnterDhcpServerRangeClientClass )

DhcpServerSubnetV6Mode.addCommandClass( DhcpServerV6DnsServer )
DhcpServerSubnetV6Mode.addCommandClass( DhcpServerSubnetName )
DhcpServerSubnetV6Mode.addCommandClass( DhcpServerSubnetLeaseTime )
DhcpIntfConfigModelet.addCommandClass( InterfaceEnterDhcpServerEnable )

BasicCli.EnableMode.addCommandClass( DhcpServerClearLeasesVrf )

BasicCli.EnableMode.addCommandClass( DhcpServerClearIpv4LeaseByAddrVrf )
BasicCli.EnableMode.addCommandClass( DhcpServerClearIpv6LeaseByAddrVrf )
