#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: library DhcpServer

import Arnet
import CliCommand
import CliMatcher
import CliParserCommon
import re
from CliPlugin import MacAddr
from CliPlugin import IpAddrMatcher
from CliPlugin import Ip6AddrMatcher
from CliPlugin import VlanCli
from CliPlugin import VirtualIntfRule
from CliPlugin import EthIntfCli
from CliPlugin import LagIntfCli
from EosDhcpServerLib import convertDaysHoursMinutes
from EosDhcpServerLib import rangeContains
from EosDhcpServerLib import tacLayer2Info
from EosDhcpServerLib import tacHexOrStr
from EosDhcpServerLib import tacVendorClassOption
from EosDhcpServerLib import tacCircuitIdRemoteIdHexOrStr
from EosDhcpServerLib import OptionType

# set maximum lease time to be a reasonably big number
# that won't cause overflow in Kea
maxLeaseTime = 0x7FFFFFFF

# maximum number of dns servers we allow
_maxDnsServers = 2
dnsHelp = "DHCP DNS configuration"
dnsServerHelp = 'Set the DNS server(s) for %s DHCP clients'

# maximum size of a string that kea can send in an option is 253 bytes
maxKeaStringSize = 253

# Adapters
def tftpBootFileAdapter( mode, args, argsList ):
   bootFile = args.get( 'FILENAME', '' )
   stringSize = len( bootFile.encode( 'utf-8' ) )
   if stringSize > maxKeaStringSize:
      mode.addError( f'Invalid boot file name of size {stringSize}' )

def IP_ADDRAdapter( mode, args, argsList ):
   ipAddrDefault = "0.0.0.0"
   args[ 'IP_ADDR' ] = Arnet.IpAddr( args.get( 'IP_ADDR', ipAddrDefault ) )
   args[ 'IP_DEFAULT' ] = Arnet.IpAddr( ipAddrDefault )

def IP6_ADDRAdapter( mode, args, argsList ):
   ip6AddrDefault = "::"
   args[ 'IP_ADDR' ] = Arnet.Ip6Addr( args.get( 'IP_ADDR', ip6AddrDefault ) )
   args[ 'IP_DEFAULT' ] = Arnet.Ip6Addr( ip6AddrDefault )

# option code set for arbitrary option. See AID10486
stringOptionSetV4 = { 14, 15, 17, 18, 40, 47, 56, 60, 62, 64, 66, 67, 86, 87, 88,
                      98, 100, 101, 113, 114, 160 }
fqdnOptionSetV4 = { 15, 88, 119, 137, 141, 213 }
ipAddrOptionSetV4 = { 3, 4, 5, 6, 7, 8, 9, 10, 11, 16, 21, 28, 32, 33, 41, 42, 44,
                    45, 48, 49, 54, 65, 68, 69, 70, 71, 72, 73, 74, 75, 76, 85,
                    89, 112, 136, 138, 150 }
privateOptionSet = set( range( 224, 255 ) )
stringOptionSetV4.update( privateOptionSet )
fqdnOptionSetV4.update( privateOptionSet )
ipAddrOptionSetV4.update( privateOptionSet )

# excludes not configurable option codes from hex option set
# Kea doesn't support configuration for: 1, 12, 50, 51, 53, 55, 58, 59, 61, 81, 82,
# 90, 91, 92, 118. See aid10486
# We reserve option 222 for subnet name
unsupportedOptionSetv4 = { 1, 12, 50, 51, 53, 55, 58, 59, 61, 81, 82, 90, 91, 92,
                           118, 222 }
hexOptionSetV4 = set( range( 1, 255 ) ) - unsupportedOptionSetv4

# option code set for arbitrary option v6.
stringOptionSetv6 = { 41, 42, 59, 103 }
fqdnOptionSetV6 = { 21, 24, 29, 30, 33, 51, 57, 58, 64, 65 }
ipAddrOptionSetv6 = { 12, 22, 23, 27, 28, 31, 34, 40, 48, 52, 80, 88, 90, 143 }
stringOptionSetv6.update( privateOptionSet )
ipAddrOptionSetv6.update( privateOptionSet )

# excludes fqdn and not configurable option codes from hex option set
# Kea doesn't support configuration for: 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 13, 14, 15,
# 16, 17, 18, 25, 26. See aid10486
# We reserve option 222 for subnet name
unsupportedOptionSetv6 = { 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 13, 14, 15, 16, 17, 18,
                           25, 26, 222 }
hexOptionSetv6 = set( range( 1, 255 ) ) - ( unsupportedOptionSetv6 |
                                            fqdnOptionSetV6 )

physicalOrLagMatcher = VirtualIntfRule.IntfMatcher()
physicalOrLagMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
physicalOrLagMatcher |= LagIntfCli.EthLagIntf.matcher

class SetMatcher( CliMatcher.Matcher ):
   def __init__( self, matchSet=None, helpdesc=None, helpname=None, common=False,
                 priority=CliParserCommon.PRIO_NORMAL, value=None ):
      super().__init__( helpdesc, helpname, common, priority,
                        value )
      self.matchSet = matchSet

   def completions( self, mode, context, token ):
      if token == '' or ( self.match( mode, context, token ) is not
                          CliParserCommon.noMatch ):
         helpname = self.helpname_
         return [ CliParserCommon.Completion( helpname, self.helpdesc_,
                                              False, common=self.common_ ) ]
      return []

   def __str__( self ):
      return '<{}({})>'.format( self.__class__.__name__, 'match' )

   def toValue( self, token ):
      decRegEx = re.compile( r'^-?[0-9]+$' )
      if decRegEx.match( token ):
         return int( token )
      return None

   def match( self, mode, context, token ):
      val = self.toValue( token )
      if val is not None and val in self.matchSet:
         return CliParserCommon.MatchResult( val, str( val ) )
      return CliParserCommon.noMatch

dhcpOptionExpressionData = {
   'ipv4': 'DHCP option for IPv4 DHCP clients',
   'ipv6': 'DHCP option for IPv6 DHCP clients',
   'CODE_STRING_V4': CliCommand.Node( SetMatcher( stringOptionSetV4,
                                               helpname='CODE',
                                               helpdesc='DHCP option code' ),
                                   alias='CODE' ),
   'CODE_FQDN_V4': CliCommand.Node( SetMatcher( fqdnOptionSetV4,
                                               helpname='CODE',
                                               helpdesc='DHCP option code' ),
                                   alias='CODE' ),
   'CODE_IPADDR_V4': CliCommand.Node( SetMatcher( ipAddrOptionSetV4,
                                               helpname='CODE',
                                               helpdesc='DHCP option code' ),
                                   alias='CODE' ),
   'CODE_HEX_V4': CliCommand.Node( SetMatcher( hexOptionSetV4, helpname='CODE',
                                            helpdesc='DHCP option code' ),
                                alias='CODE' ),

   'CODE_STRING_V6': CliCommand.Node( SetMatcher( stringOptionSetv6,
                                                  helpname='CODE',
                                                  helpdesc='DHCP option code' ),
                                      alias='CODE' ),
   'CODE_FQDN_V6': CliCommand.Node( SetMatcher( fqdnOptionSetV6,
                                               helpname='CODE',
                                               helpdesc='DHCP option code' ),
                                   alias='CODE' ),
   'CODE_IPADDR_V6': CliCommand.Node( SetMatcher( ipAddrOptionSetv6,
                                                  helpname='CODE',
                                                  helpdesc='DHCP option code' ),
                                      alias='CODE' ),
   'CODE_HEX_V6': CliCommand.Node( SetMatcher( hexOptionSetv6, helpname='CODE',
                                               helpdesc='DHCP option code' ),
                                   alias='CODE' ),

   # Restrict DHCP option data to be less than or equal to 253 bytes
   'stringMatcherV4': CliMatcher.QuotedStringMatcher( pattern=r'.{1,253}',
                                                      requireQuote=True,
                                                      helpdesc='A quoted string' ),
   'fqdnMatcherV4': CliMatcher.PatternMatcher( pattern=r'.{1,253}',
                                                helpname='WORD',
                                                helpdesc='Domain name' ),
   'hexMatcherV4': CliMatcher.PatternMatcher( pattern=r'^[a-fA-F0-9]{1,506}$',
                                              helpname='HEX',
                                              helpdesc='A hex number' ),

   # Restrict DHCP option data to be less than or equal to 65536 bytes in ipv6
   'stringMatcherV6': CliMatcher.QuotedStringMatcher( pattern=r'.{1,65536}',
                                                      requireQuote=True,
                                                      helpdesc='A quoted string' ),
   'fqdnMatcherV6': CliMatcher.PatternMatcher( pattern=r'.+',
                                                helpname='WORD',
                                                helpdesc='Domain name' ),
   'hexMatcherV6': CliMatcher.PatternMatcher( pattern=r'^[a-fA-F0-9]{1,131072}$',
                                              helpname='HEX',
                                              helpdesc='A hex number' ),
}

class DhcpServerPrivateOptionExpression( CliCommand.CliExpression ):
   data = {
         'private-option': 'Private option',
         'CODE': CliMatcher.IntegerMatcher( 224, 254,
                                            helpdesc='Option code' ),
         'always-send': 'Send the option whether or not the client requested it',
         'type': 'Option type',
         'data': 'Option data'
          }

class DhcpServerIpAddrPrivateOptionBase( CliCommand.CliCommandClass ):
   @staticmethod
   def adapter( mode, args, argsList ):
      # save address type and ip address as list
      if 'ipv4-address' in args:
         ipAddrToken = 'IP_ADDR'
         args[ 'ipAddrType' ] = OptionType.optionIpAddress
      else:
         ipAddrToken = 'IP6_ADDR'
         args[ 'ipAddrType' ] = OptionType.optionIp6Address
      args[ 'IP' ] = args.get( ipAddrToken )
      if args[ 'IP' ] and not isinstance( args[ 'IP' ], list ):
         args[ 'IP' ] = [ args[ 'IP' ] ]

class DhcpServerLeaseTimeExpression( CliCommand.CliExpression ):
   expression = 'DAYS days HOURS hours MINUTES minutes'
   data = {
      'DAYS': CliMatcher.IntegerMatcher( 0, 2000, helpdesc='Days' ),
      'days': 'days',
      'HOURS': CliMatcher.IntegerMatcher( 0, 23, helpdesc='Hours' ),
      'hours': 'hours',
      'MINUTES': CliMatcher.IntegerMatcher( 0, 59, helpdesc='Minutes' ),
      'minutes': 'minutes',
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      # don't parse any information if noOrDefaultHandler is called
      if CliCommand.isNoOrDefaultCmd( args ):
         return

      # Save Lease Time
      days = args.pop( 'DAYS' )
      hours = args.pop( 'HOURS' )
      minutes = args.pop( 'MINUTES' )
      timeSeconds = convertDaysHoursMinutes( days, hours, minutes )
      if 0 < timeSeconds < maxLeaseTime:
         args[ 'TIME' ] = timeSeconds

      # invalid lease time
      elif timeSeconds == 0:
         mode.addError( "Cannot set a 0 duration lease" )

      else:
         mode.addError( "Invalid lease time {} days {} hours {} minutes".format(
                        days, hours, minutes ) )

class DhcpServerTftpOption66Base( CliCommand.CliCommandClass ):
   syntax = '''TFTP_EXPR NAME'''
   noOrDefaultSyntax = '''TFTP_EXPR ...'''

class DhcpServerTftpOption150Base( CliCommand.CliCommandClass ):
   syntax = '''TFTP_EXPR { SERVER_ADDR }'''
   noOrDefaultSyntax = '''TFTP_EXPR ...'''

class DhcpServerTftpBootFileBase( CliCommand.CliCommandClass ):
   syntax = '''TFTP_EXPR file FILENAME'''
   noOrDefaultSyntax = '''TFTP_EXPR file ...'''

   adapter = tftpBootFileAdapter

class DhcpServerV4DefaultGatewayBase( CliCommand.CliCommandClass ):
   syntax = '''default-gateway IP_ADDR'''
   noOrDefaultSyntax = '''default-gateway ...'''

   data = {
      'IP_ADDR': IpAddrMatcher.IpAddrMatcher(
         'Address of default gateway' )
   }

   adapter = IP_ADDRAdapter

def generateTftpServerExpression( option=None ):
   class DhcpServerTftpServerExpression( CliCommand.CliExpression ):
      expression = 'tftp server'
      data = {
         'tftp': 'TFTP option',
         'server': 'Set TFTP server',
      }
      if option:
         expression += f' option {option}'
         data[ 'option' ] = "DHCP option configuration"
         data[ option ] = f"DHCP TFTP option {option}"

   return DhcpServerTftpServerExpression

class DhcpServerLeaseTimeBase( CliCommand.CliCommandClass ):
   syntax = '''lease time TIME'''
   noOrDefaultSyntax = '''lease time ...'''

   @staticmethod
   def leaseTimeIs( mode, config, args ):
      if mode.session.hasError():
         return
      leaseTime = args[ 'TIME' ]
      config.leaseTime = leaseTime

   @staticmethod
   def leaseTimeDel( config ):
      config.leaseTime = 0.0

class DhcpServerArbitraryOptionExpression( CliCommand.CliExpression ):
   data = {
      'option': 'DHCP option',
      'always-send': 'Send the option whether or not the client requested it',
      'type': 'Option type',
      'data': 'Option data'
   }

class DhcpServerClientClassArbitraryOptionStringV4Expression(
                                       DhcpServerArbitraryOptionExpression ):
   expression = 'option CODE_STRING_V4 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'CODE_STRING_V4': dhcpOptionExpressionData[ 'CODE_STRING_V4' ]
      }
   )

class DhcpServerClientClassArbitraryOptionIpAddrV4Expression(
                                       DhcpServerArbitraryOptionExpression ):
   expression = 'option CODE_IPADDR_V4 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'CODE_IPADDR_V4': dhcpOptionExpressionData[ 'CODE_IPADDR_V4' ]
      }
   )

class DhcpServerClientClassArbitraryOptionFqdnV4Expression(
                                       DhcpServerArbitraryOptionExpression ):
   expression = 'option CODE_FQDN_V4 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'CODE_FQDN_V4': dhcpOptionExpressionData[ 'CODE_FQDN_V4' ]
      }
   )

class DhcpServerClientClassArbitraryOptionHexV4Expression(
                                       DhcpServerArbitraryOptionExpression ):
   expression = 'option CODE_HEX_V4 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'CODE_HEX_V4': dhcpOptionExpressionData[ 'CODE_HEX_V4' ]
      }
   )

class DhcpServerClientClassArbitraryOptionStringV6Expression(
                                       DhcpServerArbitraryOptionExpression ):
   expression = 'option CODE_STRING_V6 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'CODE_STRING_V6': dhcpOptionExpressionData[ 'CODE_STRING_V6' ]
      }
   )

class DhcpServerClientClassArbitraryOptionIpAddrV6Expression(
                                       DhcpServerArbitraryOptionExpression ):
   expression = 'option CODE_IPADDR_V6 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'CODE_IPADDR_V6': dhcpOptionExpressionData[ 'CODE_IPADDR_V6' ]
      }
   )

class DhcpServerClientClassArbitraryOptionFqdnV6Expression(
                                       DhcpServerArbitraryOptionExpression ):
   expression = 'option CODE_FQDN_V6 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'CODE_FQDN_V6': dhcpOptionExpressionData[ 'CODE_FQDN_V6' ]
      }
   )

class DhcpServerClientClassArbitraryOptionHexV6Expression(
                                       DhcpServerArbitraryOptionExpression ):
   expression = 'option CODE_HEX_V6 [ always-send ] type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'CODE_HEX_V6': dhcpOptionExpressionData[ 'CODE_HEX_V6' ]
      }
   )
class DhcpServerClientClassMatchArbitraryOptionStringExpression(
                                       DhcpServerArbitraryOptionExpression ):
   expression = 'option CODE_STRING_V4 type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'CODE_STRING_V4': dhcpOptionExpressionData[ 'CODE_STRING_V4' ]
      }
   )

class DhcpServerClientClassMatchArbitraryOptionIpAddrExpression(
                                       DhcpServerArbitraryOptionExpression ):
   expression = 'option CODE_IPADDR_V4 type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'CODE_IPADDR_V4': dhcpOptionExpressionData[ 'CODE_IPADDR_V4' ]
      }
   )

class DhcpServerClientClassMatchArbitraryOptionHexExpression(
                                       DhcpServerArbitraryOptionExpression ):
   expression = 'option CODE_HEX_V4 type'
   DhcpServerArbitraryOptionExpression.data.update(
      {
         'CODE_HEX_V4': dhcpOptionExpressionData[ 'CODE_HEX_V4' ]
      }
   )

class DhcpServerClientClassMatchStringArbitraryOption(
      CliCommand.CliCommandClass ):
   syntax = '''OPTION string data STRING'''
   noOrDefaultSyntax = '''OPTION string...'''
   data = {
      'OPTION': DhcpServerClientClassArbitraryOptionStringV4Expression,
      'string': 'Type as string',
      'STRING': dhcpOptionExpressionData[ 'stringMatcherV4' ]
   }

   handler = ( "DhcpServerCliHandler."
               "setDhcpServerClientClassMatchStringArbitraryOption" )

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassMatchStringArbitraryOption" )

class DhcpServerIpAddrArbitraryOptionBase( CliCommand.CliCommandClass ):
   @staticmethod
   def adapter( mode, args, argsList ):
      # save address type and ip address as list
      if 'ipv4-address' in args:
         ipAddrToken = 'IP_ADDRS'
         args[ 'ipAddrType' ] = OptionType.optionIpAddress
      else:
         ipAddrToken = 'IP6_ADDRS'
         args[ 'ipAddrType' ] = OptionType.optionIp6Address
      args[ 'IP' ] = args.get( ipAddrToken )
      if args[ 'IP' ] and not isinstance( args[ 'IP' ], list ):
         args[ 'IP' ] = [ args[ 'IP' ] ]

class DhcpServerRangeClientClassIpAddrBase( CliCommand.CliCommandClass ):
   @staticmethod
   def handlerBase( mode, args, errorString ):
      ipAddrDefault = args[ 'IP_DEFAULT' ]
      ipAddr = args[ 'IP_ADDR' ]
      subnetRange = mode.range

      # verify the ipv4-address is from the correct range
      if ipAddr != ipAddrDefault and not rangeContains( subnetRange, ipAddr ):
         mode.addError( errorString.format( subnetRange.start, subnetRange.end ) )
         return

      config = mode.clientClassConfig
      config.assignedIp = ipAddr

class DhcpServerDnsServerBase( CliCommand.CliCommandClass ):
   syntax = '''dns server { SERVERS }'''
   noOrDefaultSyntax = '''dns server ...'''

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'af' ] = mode.af

   handler = "DhcpServerCliHandler.setDhcpServerDnsServerBase"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerDnsServerBase"

def generateDnsServerExpression( af ):
   class DhcpServerDnsServerExpression( CliCommand.CliExpression ):
      expression = 'dns server'
      data = {
         'dns': dnsHelp,
         'server': dnsServerHelp % af
      }
   return DhcpServerDnsServerExpression

class DhcpServerV4DnsServer( DhcpServerDnsServerBase ):
   data = {
      'SERVER_EXPR': generateDnsServerExpression( 'IPv4' ),
      'SERVERS': CliCommand.Node(
         matcher=IpAddrMatcher.ipAddrMatcher, maxMatches=_maxDnsServers ),
   }

class DhcpServerV6DnsServer( DhcpServerDnsServerBase ):
   data = {
      'SERVER_EXPR': generateDnsServerExpression( 'IPv6' ),
      'SERVERS': CliCommand.Node(
         matcher=Ip6AddrMatcher.ip6AddrMatcher, maxMatches=_maxDnsServers ),
   }

class InformationOptionExprFactory( CliCommand.CliExpressionFactory ):
   def __init__( self, head ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.head = head

   def generate( self, name ):
      class InformationOptionExpr( CliCommand.CliExpression ):
         expression = 'information option'
         data = {
            'information': f'{self.head} Information Option (Option 82)',
            'option': 'Information Option',
         }
      return InformationOptionExpr

class RemoteIdExprFactory( CliCommand.CliExpressionFactory ):
   def __init__( self, head ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.head = head

   def generate( self, name ):
      class RemoteIdExpr( CliCommand.CliExpression ):
         expression = 'remote-id'
         data = {
            'remote-id': f'{self.head} remote ID',
         }
      return RemoteIdExpr

class RemoteIdHexOrStringExprFactory( CliCommand.CliExpressionFactory ):
   def __init__( self, remoteIdExpr ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.remoteIdExpr = remoteIdExpr

   def generate( self, name ):
      class RemoteIdHexOrStringExpr( CliCommand.CliExpression ):
         expression = ( 'remote-id ( ( REMOTE_HEX_KW REMOTE_HEX ) | '
                        '( REMOTE_STR_KW REMOTE_STR ) )' )
         data = {
            'REMOTE_ID': self.remoteIdExpr,
            'REMOTE_HEX_KW': CliMatcher.KeywordMatcher( 'hex',
                                                        'Match remote ID in hex' ),
            'REMOTE_HEX': CliMatcher.PatternMatcher(
                             pattern=r'[a-fA-F0-9]{1,246}',
                             helpname='HEX',
                             helpdesc='Remote ID to match in hex' ),
            'REMOTE_STR_KW': CliMatcher.KeywordMatcher( 'string',
                                                        'Match remote ID in ASCII' ),
            'REMOTE_STR': CliMatcher.QuotedStringMatcher(
                                requireQuote=True,
                                helpdesc='remote ID to match in a quoted string' ),
         }
      return RemoteIdHexOrStringExpr

class CircuitIdHexOrStringFactory( CliCommand.CliExpressionFactory ):
   def __init__( self, head ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.head = head

   def generate( self, name ):
      class CircuitIdHexOrStringExpr( CliCommand.CliExpression ):
         expression = ( 'circuit-id ( ( CIRC_HEX_KW CIRC_HEX ) | '
                        '( CIRC_STR_KW CIRC_STR ) )' )
         data = {
            'circuit-id': f'{self.head} circuit ID',
            'CIRC_HEX_KW': CliMatcher.KeywordMatcher( 'hex',
                                                      'Match circuit ID in hex' ),
            'CIRC_HEX': CliMatcher.PatternMatcher(
                              pattern=r'[a-fA-F0-9]{1,246}',
                              helpname='HEX',
                              helpdesc='Circuit ID to match in hex' ),
            'CIRC_STR_KW': CliMatcher.KeywordMatcher( 'string',
                                                      'Match circuit ID in ASCII' ),
            'CIRC_STR': CliMatcher.QuotedStringMatcher(
                              requireQuote=True,
                              helpdesc='circuit ID to match in a quoted string' ),
         }
      return CircuitIdHexOrStringExpr

class AristaSwitchExprFactory( CliCommand.CliExpressionFactory ):
   def __init__( self, head ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.head = head

   def generate( self, name ):
      class AristaSwitchExpr( CliCommand.CliExpression ):
         expression = '''arista-switch
                         ( ( INTF vlan VLAN [ switch-mac MAC ] ) |
                           ( switch-mac MAC ) )'''
         data = {
            'arista-switch': 'Use {} data inserted by an Arista switch'.format(
                                 self.head ),
            'INTF': physicalOrLagMatcher,
            'vlan': 'VLAN interface number',
            'VLAN': VlanCli.vlanIdMatcher,
            'switch-mac': 'MAC address to match',
            'MAC': MacAddr.macAddrMatcher,
         }
      return AristaSwitchExpr

clientClassHeader = 'Match client class by'
ClientClassMatchInformationOptionExpr = (
      InformationOptionExprFactory( clientClassHeader ) )
ClientClassRemoteIdExpr = RemoteIdExprFactory( clientClassHeader )
ClientClassMatchRemoteIdHexOrStringExpr = (
      RemoteIdHexOrStringExprFactory( ClientClassRemoteIdExpr ) )
ClientClassMatchV4CircuitIdHexOrStringExpr = (
      CircuitIdHexOrStringFactory( clientClassHeader ) )
infoOptHeader = 'Information Option'
AristaSwitchInformationOptionExpr = ( AristaSwitchExprFactory( infoOptHeader ) )
remoteIdHeader = 'remote ID'
AristaSwitchRemoteIdExpr = ( AristaSwitchExprFactory( remoteIdHeader ) )

class DhcpServerClientClassMatchAristaL2InfoBase( CliCommand.CliCommandClass ):
   @staticmethod
   def adapterBase( mode, args ):
      port = Arnet.IntfId( args.get( 'INTF', '' ) )
      if 'VLAN' in args:
         vlan = args[ 'VLAN' ].id
      else:
         vlan = 0
      mac = Arnet.EthAddr( args.get( 'MAC', '0.0.0' ) )
      args[ 'l2Info' ] = tacLayer2Info( port, vlan, mac )

# -------------------------------------------------------------------------------
# The "[no|default] option CODE [always-send] type ipv4-address data IP_ADDRS"
# command, in "config-dhcp-cls-v4-match" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassMatchIpAddrArbitraryOption(
      DhcpServerIpAddrArbitraryOptionBase ):
   syntax = '''OPTION ipv4-address data { IP_ADDRS }'''
   noOrDefaultSyntax = '''OPTION ipv4-address...'''
   data = {
      'OPTION': DhcpServerClientClassArbitraryOptionIpAddrV4Expression,
      'ipv4-address': 'Type as IPv4 address',
      'IP_ADDRS': IpAddrMatcher.ipAddrMatcher,
   }

   handler = ( "DhcpServerCliHandler."
               "setDhcpServerClientClassMatchIpAddrArbitraryOption" )

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassMatchIpAddrArbitraryOption" )

# -------------------------------------------------------------------------------
# The "[no|default] mac-address MAC_ADDR" command, in
# "config-dhcp-cls-[v4|v6]-match" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassMatchMacAddr( CliCommand.CliCommandClass ):
   syntax = '''mac-address MAC_ADDR'''
   noOrDefaultSyntax = syntax

   data = {
         "mac-address": "Match client class by client's mac-address",
         'MAC_ADDR': MacAddr.macAddrMatcher,
   }

   handler = "DhcpServerCliHandler.setDhcpServerClientClassMatchMacAddr"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerClientClassMatchMacAddr"

# -------------------------------------------------------------------------------
# The "[no|default] option CODE [always-send] type hex data HEX" command, in
# "config-dhcp-cls-v4-match" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassMatchHexArbitraryOption(
      CliCommand.CliCommandClass ):
   syntax = '''OPTION hex data HEX'''
   noOrDefaultSyntax = '''OPTION hex...'''
   data = {
      'OPTION': DhcpServerClientClassArbitraryOptionHexV4Expression,
      'hex': 'Type as hex',
      'HEX': dhcpOptionExpressionData[ 'hexMatcherV4' ]
   }

   handler = ( "DhcpServerCliHandler."
               "setDhcpServerClientClassMatchHexArbitraryOption" )

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassMatchHexArbitraryOption" )

# -------------------------------------------------------------------------------
# The "[no|default] vendor-id VENDOR_ID" command, in "config-dhcp-cls-v4-match" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassMatchVendorId( CliCommand.CliCommandClass ):
   syntax = '''vendor-id VENDOR_ID'''
   noOrDefaultSyntax = syntax

   data = {
         'vendor-id': 'Match client class by vendor identifier',
         'VENDOR_ID': CliMatcher.StringMatcher(
            helpname='VENDOR_ID', helpdesc='Vendor identifier name' ),
   }

   handler = "DhcpServerCliHandler.setDhcpServerClientClassMatchVendorId"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerClientClassMatchVendorId"

# -------------------------------------------------------------------------------
# The "[no|default] duid DUID" command, in "config-dhcp-cls-v6-match" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassMatchDuid( CliCommand.CliCommandClass ):
   syntax = '''duid DUID'''
   noOrDefaultSyntax = syntax

   data = {
         "duid": "Match client class by client's duid",
         "DUID": CliMatcher.PatternMatcher( pattern=r'[a-fA-F0-9]{6,260}',
                                            partialPattern=r'[a-fA-F0-9]{0,260}',
                                            helpname='DUID',
                                            helpdesc='DUID to match in hex' )
   }

   handler = "DhcpServerCliHandler.setDhcpServerClientClassMatchDuid"

   noOrDefaultHandler = "DhcpServerCliHandler.noDhcpServerClientClassMatchDuid"

# -------------------------------------------------------------------------------
# The "[no|default] vendor-class [enterprise-id ENTERPRISE_ID]
# {string QUOTED_STRING}" command, in "config-dhcp-cls-v6-match" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassMatchVendorClass( CliCommand.CliCommandClass ):
   syntax = (
         '''vendor-class [enterprise-id ENTERPRISE_ID] { string QUOTED_STRING }''' )
   noOrDefaultSyntax = syntax

   data = {
         'vendor-class': 'Match client class by the vendor class option',
         'enterprise-id': 'Match client class by enterprise number',
         'ENTERPRISE_ID': CliMatcher.IntegerMatcher( 1, 2**32 - 1,
                                                     helpdesc='Enterprise number' ),
         'string': 'Match the opaque data in ASCII',
         'QUOTED_STRING': CliCommand.Node(
            matcher=CliMatcher.QuotedStringMatcher(
               requireQuote=True,
               helpdesc="Opaque vendor class data" ),
            maxMatches=10 ),
   }

   @staticmethod
   def _tacVendorClassFromArgs( args ):
      enterpriseId = args.get( 'ENTERPRISE_ID', 0 )
      anyEnterpriseId = not enterpriseId
      return tacVendorClassOption( enterpriseId, anyEnterpriseId,
                                   args[ 'QUOTED_STRING' ] )

   handler = "DhcpServerCliHandler.setDhcpServerClientClassMatchVendorClass"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassMatchVendorClass" )

# -------------------------------------------------------------------------------
# The "[no|default] information-option (CIRCUIT-ID|REMOTE-ID)" command, in
# "config-dhcp-cls-v4-match" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassMatchInfoOptionHexOrStr( CliCommand.CliCommandClass ):
   syntax = '''INFORMATION-OPTION ( ( CIRCUIT-ID [ REMOTE-ID ] ) | REMOTE-ID )'''
   noOrDefaultSyntax = syntax

   data = {
         'INFORMATION-OPTION': ClientClassMatchInformationOptionExpr,
         'CIRCUIT-ID': ClientClassMatchV4CircuitIdHexOrStringExpr,
         'REMOTE-ID': ClientClassMatchRemoteIdHexOrStringExpr,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      circuitIdHex = args.get( 'CIRC_HEX', '' )
      circuitIdStr = args.get( 'CIRC_STR', '' )
      remoteIdHex = args.get( 'REMOTE_HEX', '' )
      remoteIdStr = args.get( 'REMOTE_STR', '' )
      args[ 'infoOpt' ] = tacCircuitIdRemoteIdHexOrStr( circuitIdHex, circuitIdStr,
                                                        remoteIdHex, remoteIdStr )

   handler = "DhcpServerCliHandler.setDhcpServerClientClassMatchInfoOptionHexOrStr"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassMatchInfoOptionHexOrStr" )

# -------------------------------------------------------------------------------
# The "[no|default] remote-id (string REMOTE_STR|hex REMOTE_HEX)" command, in
# "config-dhcp-cls-v6-match" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassMatchRemoteIdHexOrStr( CliCommand.CliCommandClass ):
   syntax = '''REMOTE-ID'''
   noOrDefaultSyntax = syntax

   data = {
         'REMOTE-ID': ClientClassMatchRemoteIdHexOrStringExpr,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      remoteIdHex = args.get( 'REMOTE_HEX', '' )
      remoteIdStr = args.get( 'REMOTE_STR', '' )
      args[ 'remoteId' ] = tacHexOrStr( remoteIdHex, remoteIdStr )

   handler = "DhcpServerCliHandler.setDhcpServerClientClassMatchRemoteIdHexOrStr"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassMatchRemoteIdHexOrStr" )

# -------------------------------------------------------------------------------
# The "[no|default] remote-id arista-switch (ETH_PORT|PORT_CHANNEL) vlan VLAN MAC"
# command, in "config-dhcp-cls-v6-match" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassMatchAristaRemoteId(
      DhcpServerClientClassMatchAristaL2InfoBase ):
   syntax = '''REMOTE_ID ARISTA_SWITCH'''
   noOrDefaultSyntax = syntax

   data = {
         'REMOTE_ID': ClientClassRemoteIdExpr,
         'ARISTA_SWITCH': AristaSwitchInformationOptionExpr
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      DhcpServerClientClassMatchAristaL2InfoBase.adapterBase( mode, args )

   handler = "DhcpServerCliHandler.setDhcpServerClientClassMatchAristaRemoteId"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassMatchAristaRemoteId" )

# -------------------------------------------------------------------------------
# The "[no|default] information option arista-switch (ETH_PORT|PORT_CHANNEL) vlan
# VLAN MAC" command, in "config-dhcp-cls-v4-match" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassMatchAristaInfoOption(
      DhcpServerClientClassMatchAristaL2InfoBase ):
   syntax = '''INFORMATION_OPTION ARISTA_SWITCH'''
   noOrDefaultSyntax = syntax

   data = {
         'INFORMATION_OPTION': ClientClassMatchInformationOptionExpr,
         'ARISTA_SWITCH': AristaSwitchInformationOptionExpr
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      DhcpServerClientClassMatchAristaL2InfoBase.adapterBase( mode, args )

   handler = "DhcpServerCliHandler.setDhcpServerClientClassMatchAristaInfoOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassMatchAristaInfoOption" )

# -------------------------------------------------------------------------------
# The "[no|default] option CODE [always-send] type string data STRING"
# command, in "config-dhcp-cls-v4-assignments" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassStringV4ArbitraryOption(
      CliCommand.CliCommandClass ):
   syntax = '''OPTION string data STRING'''
   noOrDefaultSyntax = '''OPTION string...'''
   data = {
      'OPTION': DhcpServerClientClassArbitraryOptionStringV4Expression,
      'string': 'Type as string',
      'STRING': dhcpOptionExpressionData[ 'stringMatcherV4' ]
   }

   handler = "DhcpServerCliHandler.setDhcpServerClientClassStringArbitraryOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassStringArbitraryOption" )

# -------------------------------------------------------------------------------
# The "[no|default] option CODE [always-send] type ipv4-address data IP_ADDRS"
# command, in "config-dhcp-cls-v4-assignments" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassIpAddrV4ArbitraryOption(
      DhcpServerIpAddrArbitraryOptionBase ):
   syntax = '''OPTION ipv4-address data { IP_ADDRS }'''
   noOrDefaultSyntax = '''OPTION ipv4-address...'''
   data = {
      'OPTION': DhcpServerClientClassArbitraryOptionIpAddrV4Expression,
      'ipv4-address': 'Type as IPv4 address',
      'IP_ADDRS': IpAddrMatcher.ipAddrMatcher,
   }

   handler = "DhcpServerCliHandler.setDhcpServerClientClassIpAddrArbitraryOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassIpAddrArbitraryOption" )

# -------------------------------------------------------------------------------
# The "[no|default] option CODE [always-send] type fqdn data FQDN"
# command, in "config-dhcp-cls-v4-assignments" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassFqdnV4ArbitraryOption(
      CliCommand.CliCommandClass ):
   syntax = '''OPTION fqdn data FQDN'''
   noOrDefaultSyntax = '''OPTION fqdn...'''
   data = {
      'OPTION': DhcpServerClientClassArbitraryOptionFqdnV4Expression,
      'fqdn': 'Type as fqdn',
      'FQDN': dhcpOptionExpressionData[ 'fqdnMatcherV4' ]
   }

   handler = "DhcpServerCliHandler.setDhcpServerClientClassFqdnArbitraryOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassFqdnArbitraryOption" )

# -------------------------------------------------------------------------------
# The "[no|default] option CODE [always-send] type hex data HEX"
# command, in "config-dhcp-cls-v4-assignments" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassHexV4ArbitraryOption(
      CliCommand.CliCommandClass ):
   syntax = '''OPTION hex data HEX'''
   noOrDefaultSyntax = '''OPTION hex...'''
   data = {
      'OPTION': DhcpServerClientClassArbitraryOptionHexV4Expression,
      'hex': 'Type as hex',
      'HEX': dhcpOptionExpressionData[ 'hexMatcherV4' ]
   }

   handler = "DhcpServerCliHandler.setDhcpServerClientClassHexArbitraryOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassHexArbitraryOption" )

# -------------------------------------------------------------------------------
# The "[no|default] option CODE [always-send] type string data STRING"
# command, in "config-dhcp-cls-v6-assignments" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassStringV6ArbitraryOption(
      CliCommand.CliCommandClass ):
   syntax = '''OPTION string data STRING'''
   noOrDefaultSyntax = '''OPTION string...'''
   data = {
      'OPTION': DhcpServerClientClassArbitraryOptionStringV6Expression,
      'string': 'Type as string',
      'STRING': dhcpOptionExpressionData[ 'stringMatcherV6' ]
   }

   handler = "DhcpServerCliHandler.setDhcpServerClientClassStringArbitraryOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassStringArbitraryOption" )

# -------------------------------------------------------------------------------
# The "[no|default] option CODE [always-send] type ipv6-address data IP_ADDRS"
# command, in "config-dhcp-cls-v6-assignments" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassIpAddrV6ArbitraryOption(
      DhcpServerIpAddrArbitraryOptionBase ):
   syntax = '''OPTION ipv6-address data { IP6_ADDRS }'''
   noOrDefaultSyntax = '''OPTION ipv6-address...'''
   data = {
      'OPTION': DhcpServerClientClassArbitraryOptionIpAddrV6Expression,
      'ipv6-address': 'Type as IPv6 address',
      'IP6_ADDRS': CliCommand.Node( matcher=Ip6AddrMatcher.ip6AddrMatcher )
   }

   handler = "DhcpServerCliHandler.setDhcpServerClientClassIpAddrArbitraryOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassIpAddrArbitraryOption" )
# -------------------------------------------------------------------------------
# The "[no|default] option CODE [always-send] type fqdn data FQDN"
# command, in "config-dhcp-cls-v6-assignments" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassFqdnV6ArbitraryOption(
      CliCommand.CliCommandClass ):
   syntax = '''OPTION fqdn data FQDN'''
   noOrDefaultSyntax = '''OPTION fqdn...'''
   data = {
      'OPTION': DhcpServerClientClassArbitraryOptionFqdnV6Expression,
      'fqdn': 'Type as fqdn',
      'FQDN': dhcpOptionExpressionData[ 'fqdnMatcherV6' ]
   }

   handler = "DhcpServerCliHandler.setDhcpServerClientClassFqdnArbitraryOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassFqdnArbitraryOption" )

# -------------------------------------------------------------------------------
# The "[no|default] option CODE [always-send] type hex data HEX"
# command, in "config-dhcp-cls-v6-assignments" mode.
# -------------------------------------------------------------------------------
class DhcpServerClientClassHexV6ArbitraryOption(
      CliCommand.CliCommandClass ):
   syntax = '''OPTION hex data HEX'''
   noOrDefaultSyntax = '''OPTION hex...'''
   data = {
      'OPTION': DhcpServerClientClassArbitraryOptionHexV6Expression,
      'hex': 'Type as hex',
      'HEX': dhcpOptionExpressionData[ 'hexMatcherV6' ]
   }

   handler = "DhcpServerCliHandler.setDhcpServerClientClassHexArbitraryOption"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noDhcpServerClientClassHexArbitraryOption" )
