#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: library DhcpServer

import Arnet
import BasicCli
import CliMatcher
from CliPlugin import IpAddrMatcher
from CliPlugin import Ip6AddrMatcher
from CliPlugin.DhcpServerCliModels import (
      DhcpServerIpv4VendorOptionModel,
      DhcpServerSubOptionModel,
      DhcpServerIpv4SubnetModel,
      DhcpServerIpv6SubnetModel,
      DhcpServerIpv4Range,
      DhcpServerIpv6Range,
      DhcpServerShowLeasesModel,
      DhcpServerShowVrfLeasesModel,
      DhcpServerIpv4ShowModel,
      DhcpServerIpv6ShowModel,
      DhcpServerShowModel,
      DhcpServerShowVrfModel,
      DhcpServerInterfaceStatusShowModel,
      DhcpServerIpv4ReservationsMacAddressModel,
      DhcpServerIpv6ReservationsMacAddressModel,
      DhcpServerClientClassModel,
      Ipv4Leases,
      Ipv6Leases,
      Option82,
      Option82SubOption,
      )
from CliPlugin import VrfCli
import CliToken.Ip
import CliToken.Ipv4
import CliToken.Ipv6
from collections import namedtuple
from EosDhcpServerLib import unknownSubnets
from EosDhcpServerLib import KeaDhcpLeaseData
from EosDhcpServerLib import getSubOptionData
from EosDhcpServerLib import vendorSubOptionType
from EosDhcpServerLib import getClientClassInactiveReason
from EosDhcpServerLib import ipAddrToInt
from EosDhcpServerLib import featureEchoClientId
import LazyMount
import ShowCommand
import Tac

# Global Variables
dhcpServerVrfConfig = None
dhcpServerVrfStatus = None
ipAddrZero = Tac.Value( "Arnet::IpAddr" ).ipAddrZero

ConfigData = namedtuple( 'ConfigData', [
   'leaseInfo',
   'serverDisabledDefault',
   'serverDisabled',
   'interfaceDisabled',
   'interfaces',
   'leaseTime',
   'dnsServers',
   'dnsName',
   'tftpServerOption66',
   'tftpServerOption150',
   'tftpBootFile',
   'vendorOption',
   'clientClassConfigs',
   'echoClientId' ] )

disabledReasonsEnum = Tac.Type( 'DhcpServer::DisabledReasonsEnum' )

def _genPrefix( subnet ):
   genPrefix = Arnet.IpGenPrefix( str( subnet ) )
   return genPrefix

def _getReservationsMacAddr( subnetData ):
   reservationsMacAddr = {}
   for macAddr, reservationMacAddr in \
         subnetData.reservationsMacAddr.items():
      reservationModel = DhcpServerIpv4ReservationsMacAddressModel()
      reservationModel.macAddress = macAddr
      if reservationMacAddr.ipAddr != "0.0.0.0":
         reservationModel.ipv4Address = reservationMacAddr.ipAddr
      if reservationMacAddr.hostname != "":
         reservationModel.hostname = reservationMacAddr.hostname
      else:
         reservationModel.hostname = None
      reservationsMacAddr[ macAddr ] = reservationModel

   return reservationsMacAddr or None

def _getReservationsMacAddr6( subnetData ):
   reservationsMacAddr = {}
   for macAddr, reservationMacAddr in \
         subnetData.reservationsMacAddr.items():
      reservationModel = DhcpServerIpv6ReservationsMacAddressModel()
      reservationModel.macAddress = macAddr
      if not reservationMacAddr.ipAddr.isZero:
         reservationModel.ipv6Address = reservationMacAddr.ipAddr
      if reservationMacAddr.hostname != "":
         reservationModel.hostname = reservationMacAddr.hostname
      else:
         reservationModel.hostname = None
      reservationsMacAddr[ macAddr ] = reservationModel

   return reservationsMacAddr or None

def _getClientClasses( clientClassConfigs, af ):
   clientClasses = {}
   for clientClassName, clientClassConfig in clientClassConfigs.items():
      clientClassModel = DhcpServerClientClassModel()
      clientClassModel.inactiveReason = getClientClassInactiveReason(
                                                         clientClassConfig, af )
      clientClasses[ clientClassName ] = clientClassModel

   return clientClasses or None

def _getAllOverlappingSubnet( subnetStatus ):
   """
   Get all the overlapping subnets for the given subnet

   @Input
      subnetStatus (DhcpServer::Subnet(4|6)Status): subnet to be inspected

   @Output
      allOverlappingSubnets list(Arnet::IpGenPrefix): list of all overlapping subnets
      or
      None
   """
   if not subnetStatus or not subnetStatus.overlappingSubnet:
      return None

   allOverlappingSubnets = list( subnetStatus.overlappingSubnet )
   return allOverlappingSubnets

def _getDuplicateReservedIp( subnetStatus ):
   """
   Get duplicate reserved IPv4 address

   @Input
      subnetStatus (DhcpServer::Subnet4Status): subnet to be inspected

   @Output
      duplicateReservedIp (Arnet::IpAddr)
      or
      None
   """
   if not subnetStatus:
      return None

   if disabledReasonsEnum.duplicateReservedIp in subnetStatus.disabledReasons:
      # Duplicate IPv4 address reservation: 1.0.0.12
      return Arnet.IpAddr( subnetStatus.duplicateReservedIp )

   return None

def _getDuplicateReservedIp6( subnetStatus ):
   """
   Get duplicate reserved IPv6 address

   @Input
      subnetStatus (DhcpServer::Subnet6Status): subnet to be inspected

   @Output
      duplicateReservedIp (Arnet::Ip6Addr)
      or
      None
   """
   if not subnetStatus:
      return None

   if disabledReasonsEnum.duplicateReservedIp in subnetStatus.disabledReasons:
      # Duplicate IPv6 address reservation: 1::12
      return Arnet.Ip6Addr( subnetStatus.duplicateReservedIp )

   return None

def _getDisabledReasons( subnetStatus, subnetModel, af='ipv4' ):
   """
   Get/Set all disabled reasons for the given subnetModel

   @Input
      subnetStatus (DhcpServer::SubnetStatus)
      subnetModel (DhcpServerIpv4SubnetModel) or (DhcpServerIpv6SubnetModel)
      af (str)

   @Output
      None
   """
   ipv4 = af == 'ipv4'

   # These attributes need default values
   # overlapping subnets
   subnetModel.overlappingSubnets = _getAllOverlappingSubnet( subnetStatus )

   # The below attributes don't need default values
   if not subnetStatus:
      return

   if ipv4:
      # duplicate Reserved IPv4 address
      subnetModel.duplicateReservedIp = (
                       _getDuplicateReservedIp( subnetStatus ) )
   else:
      # duplicate Reserved IPv6 address
      subnetModel.duplicateReservedIp = (
                       _getDuplicateReservedIp6( subnetStatus ) )

   # overlapping ranges
   rangeModelFunc = DhcpServerIpv4Range if ipv4 else DhcpServerIpv6Range

   if disabledReasonsEnum.overlappingRanges in subnetStatus.disabledReasons:
      # Overlapping range: 192.168.1.110-192.168.1.254
      rangeModel = rangeModelFunc()
      overlappingRange = subnetStatus.overlappingRangeV4 if ipv4 else \
                         subnetStatus.overlappingRangeV6
      rangeModel.startRangeAddress = overlappingRange.start
      rangeModel.endRangeAddress = overlappingRange.end
      subnetModel.overlappingRange = rangeModel

   # invalid ranges
   if disabledReasonsEnum.invalidRanges in subnetStatus.disabledReasons:
      # Invalid range: 192.168.1.110-192.168.1.254
      rangeModel = rangeModelFunc()
      invalidRange = subnetStatus.invalidRangeV4 if ipv4 else \
                     subnetStatus.invalidRangeV6
      rangeModel.startRangeAddress = invalidRange.start
      rangeModel.endRangeAddress = invalidRange.end
      subnetModel.invalidRange = rangeModel

def _getRanges( subnetData, subnet, leaseInfo, af ):
   ranges = []
   ipv4 = af == 'ipv4'
   rangeModelFunc = DhcpServerIpv4Range if ipv4 else DhcpServerIpv6Range
   for r in subnetData.ranges:
      rangeModel = rangeModelFunc()
      rangeModel.startRangeAddress = r.start
      rangeModel.endRangeAddress = r.end
      rangeModel.totalLeases = (
            ipAddrToInt( r.end, v4=ipv4 ) - ipAddrToInt( r.start, v4=ipv4 ) + 1 )
      rangeModel.assignedLeases = 0

      # Range Client Classes
      rangeData = subnetData.rangeConfig[ r ]
      rangeModel.clientClasses = _getClientClasses(
                                          rangeData.clientClassConfig, af )

      ranges.append( rangeModel )
   if leaseInfo.subnet( subnet ):
      for ip in leaseInfo.subnet( subnet ).leases:
         ipInt = ipAddrToInt( ip, v4=ipv4 )
         for r in ranges:
            if ( ipAddrToInt( r.startRangeAddress, v4=ipv4 ) <=
                 ipInt <= ipAddrToInt( r.endRangeAddress, v4=ipv4 ) ):
               r.assignedLeases += 1
               break
   return ranges

def _getSubnet( af, config, status, leaseInfo ):
   subnets = []
   ipv4 = af == 'ipv4'
   subnetConfig = config.subnetConfigIpv4 if ipv4 else \
                  config.subnetConfigIpv6
   # Because the sysdb config and kea config can differ for some (small) amount
   # of time, need to union the subnets
   allSubnets = set( subnetConfig )
   allSubnets = allSubnets | set( leaseInfo.subnetData )
   subnetModelFunc = DhcpServerIpv4SubnetModel if ipv4 else \
                     DhcpServerIpv6SubnetModel
   for subnet in sorted( allSubnets ):
      subnetGenPref = _genPrefix( subnet )
      subnetModel = subnetModelFunc()
      subnetModel.subnet = Tac.const( subnetGenPref )
      subnetData = subnetConfig.get( subnet )
      subnetModel.name = subnetData.subnetName if subnetData else ''
      if subnetData:
         subnetModel.dnsServers = list( subnetData.dnsServers.values() )
         if not subnetModel.dnsServers:
            # use the DNS servers from global config instead
            dnsServers = config.dnsServersIpv4.values() if ipv4 else \
                         config.dnsServersIpv6.values()
            subnetModel.dnsServers = list( dnsServers )

         if subnetData.leaseTime:
            subnetModel.leaseDuration = subnetData.leaseTime

         if ipv4:
            subnetModel.defaultGateway = subnetData.defaultGateway
            subnetStatus = status.subnetStatus.get( subnetGenPref )
            disabledMsg = subnetStatus.disabledMessage if subnetStatus else None

            if subnetData.tftpServerOption66:
               subnetModel.tftpServerOption66 = subnetData.tftpServerOption66

            if subnetData.tftpServerOption150:
               subnetModel.tftpServerOption150 = \
                  list( subnetData.tftpServerOption150.values() )
            else:
               # tftpServerOption150 is an empty list by default.
               # Do this so that it won't show up in CAPI model if not set.
               subnetModel.tftpServerOption150 = None

            if subnetData.tftpBootFileName:
               subnetModel.tftpBootFile = subnetData.tftpBootFileName

            # reservations mac-address
            # If there are no reservations, _getReservationsMacAddr() will return
            # None and subnetModel.reservationsMacAddress will not show up in the
            # CAPI model.
            reservationsMacAddr = _getReservationsMacAddr( subnetData )
            subnetModel.reservationsMacAddress = reservationsMacAddr

         # IPv6
         else:
            subnetStatus = (
                           status.subnet6Status.get( subnetGenPref ) )
            disabledMsg = subnetStatus.disabledMessage if subnetStatus else None

            # Currently, there is no message if direct is active
            directActiveDetail = status.subnetBroadcastStatus.get( subnet )
            if not directActiveDetail:
               subnetModel.directActive = True
            else:
               subnetModel.directActive = False
               subnetModel.directActiveDetail = directActiveDetail
            # If the subnet is active, then we will always reply to relayed requests
            subnetModel.relayActive = True

            # reservations mac-address
            # If there are no reservations, _getReservationsMacAddr6() will return
            # None and subnetModel.reservationsMacAddress will not show up in the
            # CAPI model.
            reservationsMacAddr = _getReservationsMacAddr6( subnetData )
            subnetModel.reservationsMacAddress = reservationsMacAddr

         # IPv4 and IPv6 shared attributes
         if disabledMsg:
            subnetModel.disabledMessage = "Subnet is disabled"

         # fill Disabled Reason(s) attributes
         _getDisabledReasons( subnetStatus, subnetModel, af )

         # subnet client classes
         subnetModel.clientClasses = _getClientClasses(
                                                subnetData.clientClassConfig, af )

         subnetModel.ranges = _getRanges( subnetData, subnet, leaseInfo, af )

      # subnets without subnetConfig
      elif ipv4:
         subnetStatus = status.subnetStatus.get( subnetGenPref )
         disabledMsg = subnetStatus.disabledMessage if subnetStatus else None
         subnetModel.defaultGateway = ipAddrZero
         # tftpServerOption150 is an empty list by default.
         # Do this so that it won't show up in CAPI model if not set.
         subnetModel.tftpServerOption150 = None
         subnetModel.reservationsMacAddress = None
         subnetModel.overlappingSubnets = None
         subnetModel.clientClasses = None
      else:
         subnetStatus = status.subnet6Status.get( subnetGenPref )
         disabledMsg = subnetStatus.disabledMessage if subnetStatus else None
         subnetModel.directActive = False
         subnetModel.relayActive = False
         subnetModel.overlappingSubnets = None
         subnetModel.reservationsMacAddress = None
         subnetModel.clientClasses = None

      # Unknown subnets shared attributes
      if subnet == unknownSubnets[ int( af[ -1 ] ) ]:
         subnetModel.unknownSubnet = True
         if disabledMsg:
            subnetModel.disabledMessage = "Unknown subnet is disabled"

      subnetLeaseData = leaseInfo.subnet( subnet )
      subnetModel.activeLeases = 0
      if subnetLeaseData:
         subnetModel.activeLeases = subnetLeaseData.numActiveLeases()

      subnets.append( subnetModel )
   return subnets

def _getVendorOption( config ):
   vendorOptions = {}
   vendorOption = config.vendorOptionIpv4
   for vendorId in vendorOption:
      subOptions = vendorOption[ vendorId ].subOptionConfig
      subOptionModels = []
      for subOption in subOptions.values():
         subOptionModel = DhcpServerSubOptionModel()
         subOptionModel.optionCode = subOption.code
         subOptionModel.optionType = subOption.type
         data = getSubOptionData( subOption, raw=True )
         if subOption.type == vendorSubOptionType.string:
            subOptionModel.optionDataString = data
            # dataIpAddress is an empty list by default.
            # Do this so that it won't show up in CAPI model if not set
            subOptionModel.optionDataIpAddresses = None
         elif subOption.type == vendorSubOptionType.ipAddress:
            subOptionModel.optionDataIpAddresses = data
         else:
            assert False, 'Unknown sub-option type'
         subOptionModels.append( subOptionModel )

      vendorOptionModel = DhcpServerIpv4VendorOptionModel()
      vendorOptionModel.subOptions = subOptionModels
      vendorOptions[ vendorId ] = vendorOptionModel

   return vendorOptions

def getIpv4ConfigData( config, status, vrf ):
   leaseInfo = KeaDhcpLeaseData( status.ipv4IdToSubnet, vrf=vrf )
   serverDisabledDefault = status.ipv4ServerDisabledDefault
   serverDisabled = status.ipv4ServerDisabled
   interfaceDisabled = status.interfaceIpv4Disabled
   interfaces = config.interfacesIpv4
   leaseTime = config.leaseTimeIpv4
   dnsServers = config.dnsServersIpv4
   dnsName = config.domainNameIpv4
   tftpServerOption66 = config.tftpServerOption66Ipv4
   tftpServerOption150 = list( config.tftpServerOption150Ipv4.values() )
   tftpBootFile = config.tftpBootFileNameIpv4
   vendorOption = config.vendorOptionIpv4
   clientClassConfigs = config.clientClassConfigIpv4
   echoClientId = config.echoClientIdIpv4

   return ConfigData( leaseInfo, serverDisabledDefault, serverDisabled,
                      interfaceDisabled, interfaces, leaseTime, dnsServers,
                      dnsName, tftpServerOption66, tftpServerOption150,
                      tftpBootFile, vendorOption, clientClassConfigs, echoClientId )

def getIpv6ConfigData( config, status, vrf ):
   leaseInfo = KeaDhcpLeaseData( status.ipv6IdToSubnet, ipVersion=6, vrf=vrf )
   serverDisabledDefault = status.ipv6ServerDisabledDefault
   serverDisabled = status.ipv6ServerDisabled
   interfaceDisabled = status.interfaceIpv6Disabled
   interfaces = config.interfacesIpv6
   leaseTime = config.leaseTimeIpv6
   dnsServers = config.dnsServersIpv6
   dnsName = config.domainNameIpv6
   tftpServerOption66 = None
   tftpServerOption150 = None
   tftpBootFile = None
   vendorOption = None
   clientClassConfigs = config.clientClassConfigIpv6
   echoClientId = True

   return ConfigData( leaseInfo, serverDisabledDefault, serverDisabled,
                      interfaceDisabled, interfaces, leaseTime, dnsServers,
                      dnsName, tftpServerOption66, tftpServerOption150,
                      tftpBootFile, vendorOption, clientClassConfigs, echoClientId )

def _afModel( af, config, status, vrf ):
   ipv4 = af == 'ipv4'
   if config is None:
      # This can happen in the VRF case, when we run show leases on a VRF
      # that isn't yet configured for server
      config = Tac.newInstance( "DhcpServer::Config", 'empty' )
   if status is None:
      status = Tac.newInstance( "DhcpServer::Status", 'empty' )

   # Determine the correct data for corresponding address family
   if ipv4:
      model = DhcpServerIpv4ShowModel()
      data = getIpv4ConfigData( config, status, vrf )

      statusDefault = data.serverDisabled == data.serverDisabledDefault
      model.ipv4ServerActive = not data.serverDisabled
      if not model.ipv4ServerActive and not statusDefault:
         model.ipv4DisabledMessage = data.serverDisabled
      if config.debugLogPath:
         model.debugLog = True
      model.ipv4ActiveLeases = data.leaseInfo.totalActiveLeases()
      model.ipv4DnsServers = list( data.dnsServers.values() )
      model.ipv4DomainName = data.dnsName
      model.ipv4LeaseDuration = data.leaseTime

      if featureEchoClientId():
         model.ipv4EchoClientId = data.echoClientId

      if data.tftpServerOption66:
         model.ipv4TftpServerOption66 = data.tftpServerOption66

      # ipv4TftpServerOption150 is an empty list by default.
      # Do this so that it won't show up in CAPI model if not set.
      model.ipv4TftpServerOption150 = data.tftpServerOption150 or None

      if data.tftpBootFile:
         model.ipv4TftpBootFile = data.tftpBootFile

      for intf in data.interfaces:
         intfModel = DhcpServerInterfaceStatusShowModel()
         dhcpIntfStatus = data.interfaceDisabled.get( intf )
         if not dhcpIntfStatus:
            intfModel.active = True
         else:
            intfModel.active = False
            intfModel.detail = dhcpIntfStatus
         model.ipv4Interfaces[ intf ] = intfModel

      # client classes
      model.clientClasses = _getClientClasses( data.clientClassConfigs, af )

      # Vendor Option
      model.ipv4VendorOptions = _getVendorOption( config ) or None
      # subnet
      subnets = _getSubnet( af, config, status, data.leaseInfo )
      model.ipv4Subnets = { subnet.subnet: subnet for subnet in subnets }
   else:
      model = DhcpServerIpv6ShowModel()
      data = getIpv6ConfigData( config, status, vrf )

      statusDefault = data.serverDisabled == data.serverDisabledDefault
      model.ipv6ServerActive = not data.serverDisabled
      if not model.ipv6ServerActive and not statusDefault:
         model.ipv6DisabledMessage = data.serverDisabled
      if config.debugLogPath:
         model.debugLog = True
      model.ipv6ActiveLeases = data.leaseInfo.totalActiveLeases()
      model.ipv6DnsServers = list( data.dnsServers.values() )
      model.ipv6DomainName = data.dnsName
      model.ipv6LeaseDuration = data.leaseTime

      for intf in data.interfaces:
         intfModel = DhcpServerInterfaceStatusShowModel()
         dhcpIntfStatus = data.interfaceDisabled.get( intf )
         if not dhcpIntfStatus:
            intfModel.active = True
         else:
            intfModel.active = False
            intfModel.detail = dhcpIntfStatus
         model.ipv6Interfaces[ intf ] = intfModel

      # client classes
      model.clientClasses = _getClientClasses( data.clientClassConfigs, af )

      # subnet
      subnets = _getSubnet( af, config, status, data.leaseInfo )
      model.ipv6Subnets = { subnet.subnet: subnet for subnet in subnets }

   return model

def getShowModel( af, config, status, vrf ):
   model = DhcpServerShowModel()
   if af == 'ipv4':
      model.ipv4Server = _afModel( af, config, status, vrf )
      model.ipv6Server = DhcpServerIpv6ShowModel()
      # Do this so that it won't show in the CAPI model
      model.ipv6Server.clientClasses = None
   elif af == 'ipv6':
      model.ipv4Server = DhcpServerIpv4ShowModel()
      # Do this so that it won't show in the CAPI model
      model.ipv4Server.clientClasses = None
      model.ipv6Server = _afModel( af, config, status, vrf )
   else:
      model.ipv4Server = _afModel( 'ipv4', config, status, vrf )
      model.ipv6Server = _afModel( 'ipv6', config, status, vrf )
   model.addressFamily = af
   return model

def getShowLeasesModel( args, af, config, status, vrf ):
   v4LeaseInfo = v6LeaseInfo = leaseInfo = None
   afModels = {
      'ipv4': Ipv4Leases,
      'ipv6': Ipv6Leases
   }
   if config is None:
      # This can happen in the VRF case, when we run show leases on a VRF
      # that isn't yet configured for server
      config = Tac.newInstance( "DhcpServer::Config", 'empty' )
   if status is None:
      status = Tac.newInstance( "DhcpServer::Status", 'empty' )

   def addOption82ToLeaseModel( leaseModel, leaseData ):
      if ( leaseData.circuitIds or leaseData.remoteIds or
           leaseData.opt82UnknownSubOpts ):
         opt82 = Option82( circuitIds=leaseData.circuitIds,
                           remoteIds=leaseData.remoteIds )
         if leaseData.opt82UnknownSubOpts:
            opt82.unknownSubOpts = []
            for typeCode, value in leaseData.opt82UnknownSubOpts:
               subOpt = Option82SubOption( typeCode=typeCode, value=value )
               opt82.unknownSubOpts.append( subOpt )
         else:
            opt82.unknownSubOpts = None
         leaseModel.option82 = opt82

   def generateLeases( _leaseInfo, af ):
      leases = []
      if not _leaseInfo:
         return leases
      for subnetData in sorted( _leaseInfo.values(),
                                key=lambda subnetData: subnetData.subnet ):
         for leaseIp in sorted( subnetData.leases ):
            leaseData = subnetData.lease( leaseIp )
            leaseModel = afModels.get( af )()
            leaseModel.ipAddress = leaseIp
            leaseModel.endLeaseTime = float( leaseData.endTime() )
            leaseModel.lastTransaction = float( leaseData.cltt() )
            mac = leaseData.mac()
            leaseModel.macAddress = mac if mac else '0000.0000.0000'
            if af == 'ipv4':
               addOption82ToLeaseModel( leaseModel, leaseData )
            leases.append( leaseModel )
      return leases

   filterType = args.get( 'FILTER' )
   filterWith = args.get( filterType )

   # returns true if a lease is desired
   def toAddLeaseFilter( lease ):
      # get af based on lease information
      ipv4 = lease.get( 'iaid' ) is None
      if ipv4:
         subnetConfigs = config.subnetConfigIpv4
         idToSubnet = status.ipv4IdToSubnet
      else:
         subnetConfigs = config.subnetConfigIpv6
         idToSubnet = status.ipv6IdToSubnet

      toAdd = True
      subnetId = lease[ "subnet-id" ]
      prefix = idToSubnet.get( subnetId )
      if filterType == "NAME" and prefix:
         subnetConfig = subnetConfigs.get( prefix )
         toAdd = subnetConfig.subnetName == filterWith

      elif filterType == "SUBNET" and prefix:
         filterPrefix = Arnet.Prefix( filterWith ) if ipv4 else \
                        Arnet.Ip6Prefix( filterWith )
         toAdd = prefix == filterPrefix

      return toAdd

   model = DhcpServerShowLeasesModel()
   filterFn = toAddLeaseFilter if filterType else None
   if af is None:
      # show both address family
      v4LeaseInfo = KeaDhcpLeaseData( status.ipv4IdToSubnet,
                                      filterFunction=filterFn,
                                      vrf=vrf )

      if v4LeaseInfo.error:
         model.ipv4LeaseError = True
      else:
         v4Leases = generateLeases( v4LeaseInfo.subnetData, 'ipv4' )
         model.ipv4ActiveLeases = v4Leases

      v6LeaseInfo = KeaDhcpLeaseData( status.ipv6IdToSubnet,
                                      ipVersion=6,
                                      filterFunction=filterFn,
                                      vrf=vrf )

      if v6LeaseInfo.error:
         model.ipv6LeaseError = True
      else:
         v6Leases = generateLeases( v6LeaseInfo.subnetData, 'ipv6' )
         model.ipv6ActiveLeases = v6Leases
   else:
      ipVersion = int( af[ -1 ] )
      idToSubnet = status.ipv4IdToSubnet if af == 'ipv4' else \
                   status.ipv6IdToSubnet

      leaseInfo = KeaDhcpLeaseData( idToSubnet, ipVersion=ipVersion,
                                    filterFunction=filterFn,
                                    vrf=vrf )

      if leaseInfo.error:
         model.ipv4LeaseError = af == 'ipv4'
         model.ipv6LeaseError = af == 'ipv6'
      else:
         leases = generateLeases( leaseInfo.subnetData, af )
         if af == 'ipv4':
            model.ipv4ActiveLeases = leases
         else:
            model.ipv6ActiveLeases = leases
   return model

# show commands
class DhcpServerShowVrf( ShowCommand.ShowCliCommandClass ):
   syntax = '''show dhcp server [AF] [ VRFNAME ]'''
   data = {
      'dhcp': 'Show DHCP server and client',
      'server': 'Show DHCP server',
      'AF': CliToken.Ip.afMatcherForShow,
      'VRFNAME': VrfCli.VrfExprFactory( helpdesc="Specify VRF for DHCP server",
                                        inclAllVrf=True ),
   }
   cliModel = DhcpServerShowVrfModel

   @staticmethod
   def handler( mode, args ):
      af = args.pop( 'AF', None )
      vrfModel = DhcpServerShowVrfModel()
      vrf = args.get( 'VRFNAME' ) or VrfCli.vrfMap.lookupCliModeVrf( mode, None )
      if vrf == 'all':
         vrfs = dhcpServerVrfConfig.vrfConfig
         # pylint: disable=protected-access
         vrfModel._all = True
      else:
         vrfs = [ vrf ]

      for vrf in vrfs:
         config = dhcpServerVrfConfig.vrfConfig.get( vrf )
         status = dhcpServerVrfStatus.vrfStatus.get( vrf )
         model = getShowModel( af, config, status, vrf )
         vrfModel.vrfs[ vrf ] = model
      return vrfModel

class DhcpServerShowLeasesVrf( ShowCommand.ShowCliCommandClass ):
   syntax = '''show dhcp server
               ( ( ipv4 leases [ NAME | SUBNET ] )
               | ( ipv6 leases [ NAME | SUBNET6 ] )
               | ( leases [ NAME | SUBNET | SUBNET6 ] ) )
               [ VRFNAME ]
            '''

   data = {
      'dhcp': 'Show DHCP server and client',
      'server': 'Show DHCP server',
      'ipv4': CliToken.Ipv4.ipv4MatcherForShow,
      'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
      'leases': 'Show active leases',
      'NAME': CliMatcher.PatternMatcher( pattern='[a-zA-z0-9_-]+', helpname='NAME',
                                         helpdesc='Subnet name' ),
      'SUBNET': IpAddrMatcher.IpPrefixMatcher( 'IPv4 subnet' ),
      'SUBNET6': Ip6AddrMatcher.Ip6PrefixMatcher( 'IPv6 subnet' ),
      'VRFNAME': VrfCli.VrfExprFactory( helpdesc="Specify VRF for DHCP server",
                                        inclAllVrf=True ),
   }
   cliModel = DhcpServerShowVrfLeasesModel

   @staticmethod
   def adapter( mode, args, argsList ):
      subnet = args.get( 'SUBNET' )
      subnet6 = args.pop( 'SUBNET6', None )

      af = 'ipv4' if args.get( 'ipv4' ) else args.get( 'ipv6' )
      subnet, subnetAF = ( subnet, 'ipv4' ) if subnet else ( subnet6, 'ipv6' )

      # the AF should match the subnetAF if specified
      args[ 'SUBNET' ] = subnet
      args[ 'AF' ] = subnetAF if subnet else af

      filterTypes = [ 'NAME', 'SUBNET' ]
      for f in filterTypes:
         if args.get( f ):
            args[ 'FILTER' ] = f

   @staticmethod
   def handler( mode, args ):
      # Check if address family is specified
      af = args.pop( 'AF', None )
      vrf = args.get( 'VRFNAME' ) or VrfCli.vrfMap.lookupCliModeVrf( mode, None )
      vrfModel = DhcpServerShowVrfLeasesModel()
      if vrf == 'all':
         # pylint: disable=protected-access
         vrfModel._all = True
         vrfs = dhcpServerVrfConfig.vrfConfig
      else:
         vrfs = [ vrf ]
      for vrf in vrfs:
         config = dhcpServerVrfConfig.vrfConfig.get( vrf )
         status = dhcpServerVrfStatus.vrfStatus.get( vrf )
         model = getShowLeasesModel( args, af, config, status, vrf )
         vrfModel.vrfs[ vrf ] = model
      return vrfModel

# Register Commands
BasicCli.addShowCommandClass( DhcpServerShowVrf )
BasicCli.addShowCommandClass( DhcpServerShowLeasesVrf )

def Plugin( entityManager ):
   global dhcpServerVrfConfig
   global dhcpServerVrfStatus
   dhcpServerVrfConfig = LazyMount.mount( entityManager, 'dhcpServer/vrf/config',
                                          'DhcpServer::VrfConfigDir', 'r' )
   dhcpServerVrfStatus = LazyMount.mount( entityManager, 'dhcpServer/vrf/status',
                                          'DhcpServer::VrfStatusDir', 'r' )
