# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import BasicCliModes
import CliCommand
import CliMatcher
import CliParser, LazyMount
import ConfigMount
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.LagIntfCli as LagIntfCli
import CliPlugin.VlanCli as VlanCli
from CliPlugin import IntfCli
from CliToken.Dhcp import dhcpMatcherForConfig
from CliToken.Ip import (
      ipMatcherForConfigIf, ipMatcherForClear, ipMatcherForConfig,
)
from CliToken.Ipv6 import (
      ipv6MatcherForClear, ipv6MatcherForConfig 
)
from CommonGuards import ssoStandbyGuard
from DhcpSnoopingCliLib import dhcpSnoopingHwStatusReady

# Globals written by the Plugin function at the end of this file
dhcpSnoopingConfig = None
dhcpSnoopingHwStatusDir = None
dhcp6SnoopingHwStatusDir = None
platformHardwareSliceDir = None

def dhcpSnoopingSupported():
   if not dhcpSnoopingHwStatusDir:
      return False

   for sliceId, hwStatus in dhcpSnoopingHwStatusDir.items():
      sliceInfo = platformHardwareSliceDir.sliceInfo.get( sliceId )
      if dhcpSnoopingHwStatusReady( sliceInfo, hwStatus ):
         return True
   return False

def dhcpSnoopingSupportedGuard( mode, token ):
   if dhcpSnoopingSupported():
      return None
   return CliParser.guardNotThisPlatform

def dhcp6SnoopingSupported():
   if not dhcp6SnoopingHwStatusDir:
      return False

   for sliceId, hwStatus in dhcp6SnoopingHwStatusDir.items():
      sliceInfo = platformHardwareSliceDir.sliceInfo.get( sliceId )
      if dhcpSnoopingHwStatusReady( sliceInfo, hwStatus ):
         return True
   return False

def dhcp6SnoopingSupportedGuard( mode, token ):
   if dhcp6SnoopingSupported():   
      return None
   return CliParser.guardNotThisPlatform


nodeClear = CliCommand.guardedKeyword( 'clear', helpdesc='Reset functions', 
      guard=ssoStandbyGuard )
nodeSnoopingGuarded = CliCommand.guardedKeyword( 
   'snooping', helpdesc='DHCP Snooping information',
   guard=dhcpSnoopingSupportedGuard )
nodeSnooping6Guarded = CliCommand.guardedKeyword( 
   'snooping', helpdesc='DHCPv6 Snooping information',
   guard=dhcp6SnoopingSupportedGuard )
matcherCounters = CliMatcher.KeywordMatcher( 'counters', helpdesc='Counters' )
matcherDebug = CliMatcher.KeywordMatcher( 'debug',
      helpdesc='Internal counters for debugging' )
matcherNone = CliMatcher.KeywordMatcher( 'none',
      helpdesc='Do not include type in the Circuit ID sub-option' )
matcherSnooping = CliMatcher.KeywordMatcher( 'snooping',
      helpdesc='DHCPv6 Snooping configuration' )
matcherType = CliMatcher.KeywordMatcher( 'type',
      helpdesc='Type encoded in the Circuit ID sub-option' )
matcherVlan = CliMatcher.KeywordMatcher( 'vlan', helpdesc='VLAN' )

class DhcpSnoopingIntfModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, LagIntfCli.EthLagIntf ) or \
            ( isinstance( mode.intf, EthIntfCli.EthPhyIntf ) and
                   mode.intf.name.startswith( ( "Et", 'Sw' ) ) )

IntfCli.IntfConfigMode.addModelet( DhcpSnoopingIntfModelet )

intfModelet = DhcpSnoopingIntfModelet

#-------------------------------------------------------------------------------
# The DhcpSnoopingIntf class is used to remove any interface specific configs
#-------------------------------------------------------------------------------
class DhcpSnoopingIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      if self.intf_.name in dhcpSnoopingConfig.userDefinedCircuitId:
         del dhcpSnoopingConfig.userDefinedCircuitId[ self.intf_.name ]

#-------------------------------------------------------------------------------
# "[no|default] ip dhcp snooping interface information option circuit-id
# type <circuit-id type> value <circuit-id value>" interface command.
#-------------------------------------------------------------------------------
class IpDhcpSnoopingInterfaceInformationNewidCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip dhcp snooping interface information option circuit-id type'
              '( none | ID ) value NEWID' )
   noOrDefaultSyntax = 'ip dhcp snooping interface information option circuit-id ...'
   data = {
      'ip' : ipMatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : CliCommand.guardedKeyword( 'snooping',
         helpdesc='DHCP Snooping configuration',
         guard=dhcpSnoopingSupportedGuard ),
      'interface' : 'Dhcp snooping interface commands',
      'information' : 'Option-82 information',
      'option' : 'Configure Information Option',
      'circuit-id' : 'Circuit ID suboption in Option-82',
      'type' : 'Type encoded in the Circuit ID sub-option',
      'none' : 'Do not include type in the Circuit ID sub-option',
      'ID' : CliMatcher.IntegerMatcher( 0, 255,
         helpdesc='Specify the type in the Circuit ID sub-option' ),
      'value' : 'User defined value Circuit ID subopton in Option-82',
      'NEWID' : CliMatcher.PatternMatcher( pattern='[0-9a-zA-Z._:/\\-]{1,15}',
         helpdesc='Up to 15 alphanumeric characters describing this interface',
         helpname='WORD' ),
   }

   handler = "DhcpSnoopingHandler.setInfoCircuitIdIntf"
   noOrDefaultHandler = "DhcpSnoopingHandler.resetInfoCircuitIdIntf"

DhcpSnoopingIntfModelet.addCommandClass( IpDhcpSnoopingInterfaceInformationNewidCmd )

#--------------------------------------------------------------------------------
# clear ip dhcp snooping counters
#--------------------------------------------------------------------------------
class ClearIpDhcpSnoopingCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip dhcp snooping counters'
   data = {
      'clear' : nodeClear,
      'ip' : ipMatcherForClear,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : nodeSnoopingGuarded,
      'counters' : matcherCounters,
   }

   handler = "DhcpSnoopingHandler.clearDhcpSnoopingCounters"

BasicCliModes.EnableMode.addCommandClass( ClearIpDhcpSnoopingCountersCmd )

#--------------------------------------------------------------------------------
# clear ip dhcp snooping counters debug
#--------------------------------------------------------------------------------
class ClearIpDhcpSnoopingCountersDebugCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip dhcp snooping counters debug'
   data = {
      'clear' : nodeClear,
      'ip' : ipMatcherForClear,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : nodeSnoopingGuarded,
      'counters' : matcherCounters,
      'debug' : matcherDebug,
   }

   handler = "DhcpSnoopingHandler.clearDhcpSnoopingDebugCounters"

BasicCliModes.EnableMode.addCommandClass( ClearIpDhcpSnoopingCountersDebugCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp snooping [ bridging ]
#--------------------------------------------------------------------------------
class IpDhcpSnoopingCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp snooping [ bridging ]'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : CliCommand.guardedKeyword(
         'snooping', helpdesc='DHCP Snooping configuration',
         guard=dhcpSnoopingSupportedGuard ),
      'bridging' : 'Enable DHCP Snooping with bridging',
   }

   handler = "DhcpSnoopingHandler.setDhcpSnooping"
   noOrDefaultHandler = "DhcpSnoopingHandler.noDhcpSnooping"

BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpSnoopingCmd )

#--------------------------------------------------------------------------------
# ip dhcp snooping information option circuit-id type ( none | NONE ) 
#         format ( %h:%p | %p:%v )
#--------------------------------------------------------------------------------
class IpDhcpSnoopingInformationCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip dhcp snooping information option circuit-id type ( none | TYPE ) '
              'format ( %h:%p | %p:%v )' )
   noOrDefaultSyntax = 'ip dhcp snooping information option circuit-id ...'
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : CliCommand.guardedKeyword(
         'snooping', helpdesc='DHCP Snooping configuration',
         guard=dhcpSnoopingSupportedGuard ),
      'information' : 'Option-82 information',
      'format' : 'Specify Circuit ID format',
      'option' : 'Configure Information Option',
      'circuit-id' : 'Circuit ID subopton in Option-82',
      'type' : matcherType,
      'none' : matcherNone,
      'TYPE' : CliMatcher.IntegerMatcher(
         0, 255, helpdesc='Specify the type in the Circuit ID sub-option' ),
      '%h:%p' : 'Hostname and interface name',
      '%p:%v' : 'Interface name and VLAN ID',
   }

   handler = "DhcpSnoopingHandler.setInfoCircuitId"
   noOrDefaultHandler = "DhcpSnoopingHandler.resetInfoCircuitId"

BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpSnoopingInformationCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp snooping information option
#--------------------------------------------------------------------------------
class IpDhcpSnoopingInformationOptionCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp snooping information option'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : CliCommand.guardedKeyword(
         'snooping', helpdesc='DHCP Snooping configuration',
         guard=dhcpSnoopingSupportedGuard ),
      'information' : 'Option-82 information',
      'option' : 'Configure Information Option',
   }

   handler = "DhcpSnoopingHandler.setInfoOption"
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpSnoopingInformationOptionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp snooping vlan VLANSET
#--------------------------------------------------------------------------------
class IpDhcpSnoopingVlanVlansetCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp snooping vlan VLANSET'
   noOrDefaultSyntax = 'ip dhcp snooping vlan [ VLANSET ]'
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : CliCommand.guardedKeyword(
         'snooping', helpdesc='DHCP Snooping configuration',
         guard=dhcpSnoopingSupportedGuard ),
      'vlan' : matcherVlan,
      'VLANSET' : VlanCli.vlanSetMatcher
   }

   handler = "DhcpSnoopingHandler.setDhcpSnoopingVlan"
   noOrDefaultHandler = "DhcpSnoopingHandler.resetDhcpSnoopingVlan"

BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpSnoopingVlanVlansetCmd )

#--------------------------------------------------------------------------------
# clear ipv6 dhcp snooping counters
#--------------------------------------------------------------------------------
class ClearIpv6DhcpSnoopingCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ipv6 dhcp snooping counters'
   data = {
      'clear' : nodeClear,
      'ipv6' : ipv6MatcherForClear,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : nodeSnooping6Guarded,
      'counters' : matcherCounters,
   }

   handler = "DhcpSnoopingHandler.clearDhcp6SnoopingCounters"

BasicCliModes.EnableMode.addCommandClass( ClearIpv6DhcpSnoopingCountersCmd )

#--------------------------------------------------------------------------------
# clear ipv6 dhcp snooping counters debug
#--------------------------------------------------------------------------------
class ClearIpv6DhcpSnoopingCountersDebugCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ipv6 dhcp snooping counters debug'
   data = {
      'clear' : nodeClear,
      'ipv6' : ipv6MatcherForClear,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : nodeSnooping6Guarded,
      'counters' : matcherCounters,
      'debug' : matcherDebug,
   }

   handler = "DhcpSnoopingHandler.clearDhcp6SnoopingDebugCounters"

BasicCliModes.EnableMode.addCommandClass( ClearIpv6DhcpSnoopingCountersDebugCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 dhcp snooping
#--------------------------------------------------------------------------------
class Ipv6DhcpSnoopingCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp snooping'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : nodeSnooping6Guarded,
   }

   handler = "DhcpSnoopingHandler.setDhcp6Snooping"
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6DhcpSnoopingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 dhcp snooping remote-id option
#--------------------------------------------------------------------------------
class Ipv6DhcpSnoopingRemoteIdOptionCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp snooping remote-id option'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : nodeSnooping6Guarded,
      'remote-id' : 'Option-37 information',
      'option' : 'Configure Remote-ID option',
   }

   handler = "DhcpSnoopingHandler.setInfoOptionV6"
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6DhcpSnoopingRemoteIdOptionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 dhcp snooping vlan VLANSET
#--------------------------------------------------------------------------------
class Ipv6DhcpSnoopingVlanVlansetCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp snooping vlan VLANSET'
   noOrDefaultSyntax = 'ipv6 dhcp snooping vlan [ VLANSET ]'
   data = {
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : nodeSnooping6Guarded,
      'vlan' : matcherVlan,
      'VLANSET' : VlanCli.vlanSetMatcher,
   }

   handler = "DhcpSnoopingHandler.setDhcp6SnoopingVlan"
   noOrDefaultHandler = "DhcpSnoopingHandler.resetDhcp6SnoopingVlan"

BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6DhcpSnoopingVlanVlansetCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global dhcpSnoopingHwStatusDir
   global platformHardwareSliceDir
   global dhcp6SnoopingHwStatusDir
   global dhcpSnoopingConfig
   
   dhcpSnoopingConfig = ConfigMount.mount( entityManager,
                                           "bridging/dhcpsnooping/config",
                                           "Bridging::DhcpSnooping::Config", "w" )
   dhcpSnoopingHwStatusDir = LazyMount.mount( entityManager,
                                             "bridging/dhcpsnooping/hardware/status",
                                             "Tac::Dir", "ri" )
   platformHardwareSliceDir = LazyMount.mount( entityManager,
                                               "platform/hardware/slice",
                                               "Hardware::PlatformSliceDir", "r" )
   IntfCli.Intf.registerDependentClass( DhcpSnoopingIntf )
   dhcp6SnoopingHwStatusDir = LazyMount.mount( entityManager,
                                        "bridging/dhcpsnooping/hardware/dhcp6Status",
                                         "Tac::Dir", "ri" )
