# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import BiosLib
import BasicCli
import Cell
import CliCommand
import CliParser
import FileUrl
import LazyMount

storageDevicesStatus = None

def disableUsbGuard( mode, token ):
   baseDir = os.environ.get( 'FILESYSTEM_ROOT', '/mnt' )
   flashDir = os.path.join( baseDir, 'flash' )
   connectedDevices = storageDevicesStatus.connectedDevices
   connectedDisk = connectedDevices.getDiskMountedAt( flashDir )
   mibDeviceType = connectedDevices.disk[ connectedDisk ].mibDeviceType
   if mibDeviceType == 'eUsb':
      return CliParser.guardNotThisPlatform
   baseFeatureRevision = BiosLib.getAbootFeatures()[ 'base' ]
   if baseFeatureRevision < 27:
      return CliParser.guardNotThisAbootVersion
   return None

disabledMatcher = CliCommand.guardedKeyword( 'disabled',
                                              helpdesc='Disable the USB ports',
                                              guard=disableUsbGuard )

def showUsbWarning( mode, doDisable ):
   with open( '/proc/cmdline', 'r' ) as procCmdline:
      if ( ( doDisable
             and ( 'usbcore.nousb' not in procCmdline.read() ) )
         or ( ( not doDisable )
              and ( 'usbcore.nousb' in procCmdline.read() ) ) ):
         if Cell.cellType() == 'supervisor':
            mode.addWarning( 'Change will take effect only after switch '
                             'reboot and only on the current supervisor.' )
         else:
            mode.addWarning( 'Change will take effect only after switch '
                             'reboot.' )

def doNoUsbDisable( mode, args ):
   config = FileUrl.bootConfig( mode )
   config[ 'USB_DISABLED' ] = 'false'
   showUsbWarning( mode, doDisable=False )

def doUsbDisable( mode, args ):
   config = FileUrl.bootConfig( mode )
   config[ 'USB_DISABLED' ] = 'true'
   showUsbWarning( mode, doDisable=True )

# ------------------------------------------------------------------------------
# [ no | default ] boot usb disabled
# ------------------------------------------------------------------------------
class UsbCmd( CliCommand.CliCommandClass ):
   syntax = 'boot usb disabled'
   noOrDefaultSyntax = syntax
   data = {
      'boot': 'System boot configuration',
      'usb': 'Configure the USB ports',
      'disabled': disabledMatcher,
   }
   handler = doUsbDisable
   noOrDefaultHandler = doNoUsbDisable

BasicCli.EnableMode.addCommandClass( UsbCmd )

# --------------------------------------------------
# Plugin method - Mount the objects we need from Sysdb
# --------------------------------------------------
def Plugin( entityManager ):
   global storageDevicesStatus
   storageDevicesStatus = LazyMount.mount(
      entityManager,
      Cell.path( "hardware/storageDevices/status" ),
      "StorageDevices::Status", "r" )
