# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=consider-using-in

from IntfModels import Interface
from Intf.IntfRange import intfListToCanonical
from CliModel import Model, Submodel, Int, List, Enum, Str, Dict, Bool, Float
from TypeFuture import TacLazyType
import TableOutput
import Tac
from time import ctime
from Ark import utcTimeRelativeToNowStr, timestampToStr

FlowType = TacLazyType( "Dmf::Policy::FlowType" )
FlowMatchType = TacLazyType( "Dmf::Policy::FlowMatchType" )
FlowActionType = TacLazyType( "Dmf::Policy::FlowActionType" )
FlowEntryState = TacLazyType( "Dmf::Policy::FlowEntryState" )

####################################################################
# Helper Submodels
####################################################################

class ActionModel( Model ):
   __public__ = False

   actionType = Enum( values=FlowActionType.attributes,
                      help="Flow action" )
   val = Int( help="Action value" )
   valStr = Str( help="Action string" )
   intf = Interface( help="Action interface" )

class FlowModel( Model ):
   __public__ = False

   priority = Int( help="Flow priority" )
   _type = Enum( values=FlowType.attributes,
                     help="Flow type" )
   matchType = Enum( values=FlowMatchType.attributes,
                     help="Flow match type" )
   filterString = Str( help="Filter" )
   ports = List( valueType=Interface,
                   help="List of ports" )
   actions = List( valueType=ActionModel,
                   help="List of actions" )
   state = Enum( values=FlowEntryState.attributes,
                 help="Flow programming status" )
   packetCount = Int( help="Packet counter", optional=True )
   byteCount = Int( help="Byte counter", optional=True )
   lastUpdateTime = Float( help="Counter update time", optional=True )
   _wildcarded = Bool( help="Wildcard flow", optional=True )
   flowId = Int( help="Flow Id", optional=True )

class TableStatsModel( Model ):
   __public__ = False

   total = Int( help="Total number of flows" )
   programmed = Int( help="Total programmed flows" )
   pending = Int( help="Total pending flows" )
   failed = Int( help="Total failed flows" )
   requestedAdds = Int( help="Number of requested flow adds", optional=True )
   requestedMods = \
      Int( help="Number of requested flow modifications", optional=True )
   requestedDels = Int( help="Number of requested flow deletions", optional=True )
   installed = Int( help="Number of installed flows", optional=True )
   pendingAdds = Int( help="Number of pending flow adds", optional=True )
   pendingMods = Int( help="Number of pending flow modifications", optional=True )
   pendingDels = Int( help="Number of pending flow deletions", optional=True )
   pendingCleanups = Int( help="Number of pending flow cleanups", optional=True )
   errorsFull = Int( help="Number of TCAM full errors", optional=True )
   errorsParam = Int( help="Number of parameter errors", optional=True )
   errorsPolicy = Int( help="Number of policy errors", optional=True )
   errorsMisc = Int( help="Number of miscellaneous errors", optional=True )

class FlowErrorModel( Model ):
   __public__ = False
   message = Str( help="Error message" )
   hwError = Str( help="Hardware error message", optional=True )
   time = Float( help="When the error occurred" )
   flow = Submodel( valueType=FlowModel, help="Flow that encountered the error",
                    optional=True )

####################################################################
# show management dmf flow summary
####################################################################
class DmfFlowTableSummaryModel( Model ):
   __public__ = False

   ingress1 = Submodel( valueType=TableStatsModel,
                        help="Ingress Flow 1 summary" )
   ingress2 = Submodel( valueType=TableStatsModel,
                        help="Ingress Flow 2 summary" )
   egress1 = Submodel( valueType=TableStatsModel,
                       help="Egress Flow1 summary" )

   def render( self ):
      ret = "Flow tables:\n\n"
      headers = [ "Table", "Total Flows", "Programmed", "Pending", "Failed" ]
      table = TableOutput.createTable( headers )

      # Table should be left aligned, others should be right aligned as per
      # AID 12
      fmts = [ TableOutput.Format( justify="left" ) ]
      fmts[ -1 ].noPadLeftIs( True )
      fmts += [ TableOutput.Format( justify="right" ) ] * 4
      for fmt in fmts[ 1 : ]:
         fmt.noTrailingSpaceIs( True )
      table.formatColumns( *fmts )

      table.newRow( "Ingress 1", self.ingress1.total,
                    self.ingress1.programmed, self.ingress1.pending,
                    self.ingress1.failed )
      table.newRow( "Ingress 2", self.ingress2.total,
                    self.ingress2.programmed, self.ingress2.pending,
                    self.ingress2.failed )
      table.newRow( "Egress 1", self.egress1.total,
                    self.egress1.programmed, self.egress1.pending,
                    self.egress1.failed )
      ret += table.output()
      print ( ret )

####################################################################
# show management dmf flow (ingress|egress) TABLE [ entry IDS ] [ brief]
####################################################################
class DmfFlowTableModel( Model ):
   __public__ = False

   flows = Dict( keyType=int, valueType=FlowModel,
                 help="Map of flow ID to flow model" )
   summary = Submodel( valueType=TableStatsModel,
                       help="Table summary",
                       optional=True )
   _brief = Bool( help="Brief output", optional=True )
   _showErrors = Bool( help="Error output", optional=True )
   flowErrors = List( valueType=FlowErrorModel, help="Table errors", optional=True )
   _showSysTime = Bool( help="System time output", optional=True )

   def render( self ):
      def _prepareSummary():
         ret = ""
         if self._brief:
            ret += "Total flows: %d\n" % self.summary.total
            ret += "Programmed: %d\n" % self.summary.programmed
            ret += "Pending: %d\n" % self.summary.pending
            ret += "Failed: %d\n" % self.summary.failed
         else:
            ret += "Total flows: %d\n" % self.summary.total
            ret += "Requested adds: %d\n" % self.summary.requestedAdds
            ret += "Requested modifies: %d\n" % self.summary.requestedMods
            ret += "Requested deletes: %d\n\n" % self.summary.requestedDels
            ret += "Programmed: %d\n\n" % self.summary.programmed
            ret += "Pending: %d\n" % self.summary.pending
            ret += "Pendings adds: %d\n" % self.summary.pendingAdds
            ret += "Pending modifies: %d\n" % self.summary.pendingMods
            ret += "Pending deletes: %d\n" % self.summary.pendingDels
            ret += "Pending cleanups: %d\n\n" % self.summary.pendingCleanups
            ret += "Failed: %d\n" % self.summary.failed
            ret += "TCAM full errors: %d\n" % self.summary.errorsFull
            ret += "Parameter errors: %d\n" % self.summary.errorsParam
            ret += "Policy errors: %d\n" % self.summary.errorsPolicy
            ret += "Other errors: %d\n\n" % self.summary.errorsMisc

         return ret

      def _prepareErrorSummary():
         ret = ""
         ret += "Failed: %d\n" % self.summary.failed
         ret += "TCAM full errors: %d\n" % self.summary.errorsFull
         ret += "Parameter errors: %d\n" % self.summary.errorsParam
         ret += "Policy errors: %d\n" % self.summary.errorsPolicy
         ret += "Other errors: %d\n\n" % self.summary.errorsMisc
         return ret

      def _prepareActions( flow, brief=False ):
         ret = ""
         actions = {}
         replicateActionVlans = {}
         for action in flow.actions:
            value = ""
            if action.actionType == FlowActionType.setAggregationGroup:
               value = action.intf.stringValue
            elif action.actionType == FlowActionType.replicate:
               if action.val not in replicateActionVlans:
                  replicateActionVlans[ action.val ] = action.valStr
               continue
            elif action.actionType == FlowActionType.setInnerVlan or \
                 action.actionType == FlowActionType.setOuterVlan or \
                 action.actionType == FlowActionType.copyToCpu:
               value = "%d" % action.val
            if action.actionType not in actions:
               if action.actionType == FlowActionType.copyToCpu:
                  value = "queue %s" % value
               actions[ action.actionType ] = [ value ]
            else:
               actions[ action.actionType ].append( value )
         # generate actions for replicate action
         for vlan, name in replicateActionVlans.items():
            intfString = ""
            for action in flow.actions:
               if action.actionType == FlowActionType.replicate and \
                  action.val == vlan:
                  intfString += "," if intfString else ""
                  intfString += action.intf.stringValue
            value = f"({name};{intfString};setOuterVlan:{vlan};stripOuterVlan)"
            if FlowActionType.replicate not in actions:
               actions[ FlowActionType.replicate ] = [ value ]
            else:
               actions[ FlowActionType.replicate ].append( value )
         # pylint: disable-msg=too-many-nested-blocks
         if not brief:
            actionLabel = "Actions: "
            ret += actionLabel
            first = True
            for action, values in actions.items():
               # align other actions
               if not first:
                  ret += " " * len( actionLabel )
               else:
                  first = False
               # actions that don't have values should be printed alone
               # e.g. stripOuterVlan, drop
               if values[ 0 ] == "":
                  ret += action
               else:
                  valuesString = ""
                  if action == FlowActionType.setAggregationGroup:
                     valuesString = intfListToCanonical( values )[ 0 ]
                  elif action == FlowActionType.replicate:
                     for val in values:
                        val = val.strip( "()" )
                        actVals = val.split( ';' )
                        intfLists = actVals[ 1 ].split( ',' )
                        intfListName = intfListToCanonical( intfLists )[ 0 ]
                        if valuesString:
                           valuesString += "\n"
                           valuesString += " " * len( actionLabel ) + \
                                           f"replicate: To{actVals[ 0 ]}:\n"
                        else:
                           valuesString += f"To{actVals[ 0 ]}:\n"
                        valuesString += " " * ( len( actionLabel ) + 3 ) + \
                                        f"setAggregationGroup: {intfListName}\n"
                        valuesString += " " * ( len( actionLabel ) + 3 ) + \
                              actVals[ 2 ].replace( ":", ": " ) + "\n"
                        valuesString += " " * ( len( actionLabel ) + 3 ) + \
                                        "stripOuterVlan"
                  else:
                     valuesString = ", ".join( values )
                  ret += f"{action}: {valuesString}"
               ret += "\n"
            if not actions:
               # actions is empty
               ret += "\n"
         else:
            first = True
            for action, values in actions.items():
               if not first:
                  ret += ";"
               first = False
               if values[ 0 ] == "":
                  ret += action
               else:
                  retTmp = ""
                  if action == FlowActionType.copyToCpu:
                     retTmp = ",".join( values )
                     retTmp = retTmp.replace( " ", "" )
                  elif action == FlowActionType.setAggregationGroup:
                     retTmp = intfListToCanonical( values )[ 0 ]
                  elif action == FlowActionType.replicate:
                     retTmp = "replicateActions"
                  else:
                     retTmp = "{}:{}".format( action, ",".join( values ) )
                  ret += retTmp
         return ret

      def _prepareFlow( flowId, flow ):
         ret = ""
         ret += "Flow ID: %d\n" % flowId
         ret += "Priority: %d\n" % flow.priority
         ret += "Type: %s\n" % flow.matchType
         ret += "Filter: %s\n" % flow.filterString
         # pylint: disable-msg=protected-access
         flowType = "Ingress" if flow._type == FlowType.ingress else "Egress"
         ret += "%s ports: %s\n" % ( flowType, ",".join(
               intfListToCanonical( flow.ports, noHoleRange=flow._wildcarded ) ) )
         ret += _prepareActions( flow )
         if flow.byteCount is not None:
            ret += "Counters: %d packets, %d bytes\n" % ( flow.packetCount,
                                                          flow.byteCount )
         else:
            ret += "Counters: %d packets\n" % flow.packetCount
         ret += "Counters last update: %s\n" % timestampToStr( flow.lastUpdateTime )
         if flow.state == "pendingProgram":
            ret += "Status: pending program\n\n"
         elif flow.state == "pendingDelete":
            ret += "Status: pending delete\n\n"
         else:
            ret += "Status: %s\n\n" % flow.state
         return ret

      def _prepareFlowBrief( flowId, flow ):
         counterString = ""
         if flow.byteCount is not None:
            counterString = "%d/%d" % ( flow.packetCount, flow.byteCount )
         else:
            counterString = "%d" % flow.packetCount

         statusString = ""
         if flow.state == FlowEntryState.programmed:
            statusString = "S"
         elif flow.state == FlowEntryState.pendingProgram \
               or flow.state == FlowEntryState.pendingDelete:
            statusString = "P"
         else: # failed
            statusString = "F"

         ret = ""
         # pylint: disable-msg=protected-access
         ports = ",".join( intfListToCanonical( flow.ports,
               noHoleRange=flow._wildcarded ) )
         flowType = "in" if flow._type == FlowType.ingress else "out"
         ret += "%d [%s] %s %s:%s %s [%s] [%s]" % ( flowId,
               flow.matchType, flow.filterString, flowType, ports,
               _prepareActions( flow, brief=True ), counterString, statusString )
         ret += "\n"
         return ret

      def _prepareError( error ):
         ret = ""
         if error.flow:
            ret += "Flow: %s" % _prepareFlowBrief( error.flow.flowId, error.flow )
         ret += "Time: %s\n" % utcTimeRelativeToNowStr( error.time )
         ret += "Error: %s\n" % error.message
         if error.hwError:
            ret += "Hardware Error: %s\n\n" % error.hwError
         else:
            ret += "\n"
         return ret

      ret = ""
      if self._showErrors:
         ret += _prepareErrorSummary()
         for error in self.flowErrors:
            ret += _prepareError( error )
         print( ret )
         return

      if self._showSysTime and not self._brief:
         ret += "Current System Time: %s\n" % str( ctime( Tac.utcNow() ) )

      if self.summary:
         ret += _prepareSummary()
         ret += "Flows:\n\n"

      # sort flows so that flow with highest priority in TCAM is printed first
      for flowId, flow in sorted( self.flows.items(),
            key=lambda x: ( ( x[ 1 ].priority << 16 ) | x[ 0 ] ), reverse=True ):
         if not self._brief:
            ret += _prepareFlow( flowId, flow )
         else:
            ret += _prepareFlowBrief( flowId, flow )
      print ( ret )
