# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericAddress
from IntfModels import Interface
from CliModel import Model, Enum, Bool, Dict, Int, Str, List, Float
from TypeFuture import TacLazyType
import TableOutput
from Arnet import sortIntf
import time
from CliPlugin.PtpCliModel import portStateToName
from Vlan import vlanSetToCanonicalString

ConnectionState = TacLazyType( "Dmf::Indigo::ConnectionState" )
ConnectionRole = TacLazyType( "Dmf::Indigo::ConnectionRole" )
HwStatus = TacLazyType( "Dmf::Indigo::HwStatus" )
EthFecEncoding = TacLazyType( "Interface::EthFecEncoding" )
LoopbackMode = TacLazyType( "Interface::LoopbackMode" )
SwitchportMode = TacLazyType( "Bridging::SwitchportMode" )
PtpState = TacLazyType( "Ptp::PortState" )
CoherentFecEncoding = TacLazyType( "Phy::Coherent::CoherentFecEncoding" )

####################################################################
# Helper functions
####################################################################
def getSwitchportModeString( switchPortMode ):
   if switchPortMode == SwitchportMode.dot1qTunnel:
      return "dot1q tunnel"
   elif switchPortMode == SwitchportMode.tapTool:
      return "tap tool"
   return switchPortMode

def getFecEncodingString( ethFecEncoding ):
   if ethFecEncoding == EthFecEncoding.fecEncodingDisabled:
      return "disabled"
   elif ethFecEncoding == EthFecEncoding.fecEncodingReedSolomon544:
      return "reed solomon 544"
   elif ethFecEncoding == EthFecEncoding.fecEncodingReedSolomon:
      return "reed solomon"
   elif ethFecEncoding == EthFecEncoding.fecEncodingFireCode:
      return "fire code"
   elif ethFecEncoding == CoherentFecEncoding.coherentFecEncodingCfec:
      return "concatenated"
   elif ethFecEncoding == CoherentFecEncoding.coherentFecEncodingOfec:
      return "open"
   return "unknown"

def getLoopbackModeString( loopbackMode ):
   if loopbackMode == LoopbackMode.loopbackMac:
      return "mac"
   elif loopbackMode == LoopbackMode.loopbackPhy:
      return "phy"
   elif loopbackMode == LoopbackMode.loopbackPhyRemote:
      return "phy remote"
   return "none"

####################################################################
# Helper Submodels
####################################################################

class ControllerModel( Model ):
   __public__ = False

   ipAddr = IpGenericAddress( help="Controller IP" )
   connectionStatus = Enum( values=ConnectionState.attributes,
                            help="Connection status" )
   connectionRole = Enum( values=ConnectionRole.attributes,
                          help="Connection role" )

class CpuQueueModel( Model ):
   __public__ = False

   mirrorSession = Str( help="Mirroring session name" )
   mirrorIntf = Str( help="Mirroring interface" )
   packetInCount = Int( help="Number of packets trapped to the controller "
                             "via the queue" )

class ConnectionEventModel( Model ):
   __public__ = False
   time = Float( help="Date and time at which the connection event occured" )
   controllerIpAddr = IpGenericAddress( help="Controller IP address" )
   protocol = Str( help="Protocol of connection" )
   port = Int( help="Port at which controller connected or disconnected" )
   event = Enum( values={ "connected", "disconnected", "upgraded" },
                 help="Type of event; whether controller connected or disconnected" )
   auxId = Int( help="Auxiliary connection ID" )

####################################################################
# show management dmf indigo
####################################################################
class DmfIndigoModel( Model ):
   __public__ = False

   enabled = Bool( help="DMF is enabled" )
   active = Bool( help="Indigo agent is active" )
   tcamProfileStatus = Enum( values=HwStatus.attributes,
                             help="TCAM profile programming status" )
   flexCounterStatus = Enum( values=HwStatus.attributes,
                             help="Hardware counters allocation status" )
   forwardingChipDiscovery = Enum( values=[ 'success', 'pending' ],
                                   help="Forwarding chip model discovery status" )

   controllers = Dict( keyType=int, valueType=ControllerModel,
                       help="Map of controller ID to the controller model" )

   def render( self ):
      ret = "DMF: "
      if self.enabled:
         ret += "enabled"
      else:
         ret += "disabled"
      ret += "\n"
      ret += "Indigo agent: "
      if self.active:
         ret += "active"
      else:
         ret += "inactive"
      ret += "\n"
      ret += "TCAM profile programming status: %s\n" % self.tcamProfileStatus
      ret += "Hardware counters status: %s\n" % self.flexCounterStatus
      ret += "Forwarding chip discovery: %s\n\n" % self.forwardingChipDiscovery

      ret += "Controllers:"
      if not self.controllers:
         ret += " None"
      else:
         ret += "\n\n"

         headers = [ "ID", "IP Address", "Connection State", "Connection Role" ]
         table = TableOutput.createTable( headers )

         # ID should be right aligned, others should be left aligned as per AID 12
         fmts = [ TableOutput.Format( justify="right" ) ]
         fmts[ 0 ].noTrailingSpaceIs( True )
         fmts += [ TableOutput.Format( justify="left" ) ] * 3
         for fmt in fmts[ 1 : ]:
            fmt.noPadLeftIs( True )
         table.formatColumns( *fmts )

         for controllerId, controller in sorted( self.controllers.items() ):
            table.newRow( controllerId,
                          controller.ipAddr,
                          controller.connectionStatus,
                          controller.connectionRole )
         ret += table.output()
      print ( ret )

####################################################################
# show management dmf indigo packets
####################################################################
class DmfIndigoPacketsModel( Model ):
   __public__ = False

   packetInCount = Int( help="Number of packets trapped to the controller" )
   packetOutCount = Int( help="Number of packets injected by the controller" )
   lldpInCount = Int( help="Number of LLDP packets snooped to the controller",
                      optional=True )
   queues = Dict( keyType=int, valueType=CpuQueueModel,
                  help="Map of queue ID to its state" )

   def render( self ):
      ret = "Total\n"
      ret += "------------\n"
      ret += "Packets received: %s\n" % self.packetInCount
      ret += "Packets transmitted: %s\n" % self.packetOutCount
      if self.lldpInCount:
         ret += "LLDP packets snooped: %s\n\n" % self.lldpInCount

      headers = [ "Queue", "Session", "Interface", "Packets received" ]
      table = TableOutput.createTable( headers )

      # Queue and Packets received should be right aligned, others should be
      # left aligned as per AID 12
      fmts = [ TableOutput.Format( justify="right" ) ]
      fmts[ 0 ].noTrailingSpaceIs( True )
      fmts += [ TableOutput.Format( justify="left" ) ] * 2
      for fmt in fmts[ 1 : ]:
         fmt.noPadLeftIs( True )
      fmts += [ TableOutput.Format( justify="right" ) ]
      fmts[ -1 ].noTrailingSpaceIs( True )
      table.formatColumns( *fmts )

      for queueId, queue in sorted( self.queues.items() ):
         table.newRow( queueId, queue.mirrorSession,
                       queue.mirrorIntf, queue.packetInCount )
      ret += table.output()
      print ( ret )

####################################################################
# show management dmf controller history
####################################################################
class DmfIndigoConnectionHistoryModel( Model ):
   __public__ = False

   connectionEvents = List( valueType=ConnectionEventModel,
                            help="List of connection events" )

   def render( self ):
      headers = [ "Time", "Controller IP Address", "Event" ]
      table = TableOutput.createTable( headers )

      # Date should be right aligned, others should be left aligned as per AID 12
      fmts = [ TableOutput.Format( justify="right", minWidth=26 ) ]
      fmts[ 0 ].noTrailingSpaceIs( True )
      fmts += [ TableOutput.Format( justify="left", minWidth=15 ) ]
      fmts += [ TableOutput.Format( justify="left" ) ]
      fmts[ -1 ].noTrailingSpaceIs( True )
      table.formatColumns( *fmts )

      for event in self.connectionEvents:
         if event.port == -1:
            details = f"{event.event} by {event.protocol}"
         else:
            details = "{} by {} at {} (auxId={})".format( event.event,
                                                          event.protocol,
                                                          event.port, event.auxId )
         table.newRow(
            time.strftime( '%Y-%m-%d %H:%M:%S', time.localtime( event.time ) ),
            event.controllerIpAddr, details )

      ret = table.output()
      print ( ret )

class InterfaceModel( Model ):
   # Note that fec and loopbackMode below directly uses internal enum values
   # having "fecEncoding" and "loopback" present in its name. HHowever, we are
   # still using them directly since we are setting "__public__" to False below
   # as we don't plan to support CAPI as of now (BUG 543508 )
   __public__ = False

   intf = Interface( help="DMF Interface" )
   dmfConfigured = Bool( help="DMF is configued for this port" )
   switchportMode = Enum( values=SwitchportMode.attributes,
                          help="Switchport mode", optional=True )
   speed = Int( help="Port speed (Mbps)", optional=True )
   stripVlans = Str( help="Strip VLANs", optional=True )
   autoneg = Bool( help="Auto-negotiation is enabled", optional=True )
   fec = Enum( values=EthFecEncoding.attributes + CoherentFecEncoding.attributes,
               help="FEC encoding", optional=True )
   loopbackMode = Enum( values=LoopbackMode.attributes,
                        help="Loopback mode", optional=True )
   rateLimit = Int( help="Rate limit (Kbps)", optional=True )
   forceLinkUp = Bool( help="Force link up enabled", optional=True )
   xmit = Bool( "Transmit enabled", optional=True )
   packetInCount = Int( help="Number of packets ingressing on the dataplane port "
                             "trapped to the controller" )
   packetOutCount = Int( help="Number of packets egressing on the dataplane port "
                              "injected by the controller" )
   lagMembers = List( valueType=Interface,
                      help="Members of this Lag port", optional=True )
   ptpState = Enum( values=list( portStateToName.values() ),
                    help="PTP State", optional=True )
   dynVlans = List( valueType=int, help="Dynamic VLANs", optional=True )
   stripVxlan = Bool( "Strip Vxlan", optional=True )
   udpPort = Int( help="UDP port for VxlanStrip", optional=True )

####################################################################
# show management dmf interfaces summary
####################################################################
class DmfInterfacesSummaryModel( Model ):
   __public__ = False

   portIds = Dict( keyType=int, valueType=InterfaceModel,
                   help="Map of port ID to its DMF configuration" )

   def render( self ):
      headers = [ "Interface", "OpenFlow Port ID", "DMF State",
                  "Packets received",
                  "Packets transmitted" ]
      table = TableOutput.createTable( headers )
      # Interface and DMF state should be left aligned, others should be
      # right aligned as per AID 12
      fmts = [ TableOutput.Format( justify="left" ) ]
      fmts[ -1 ].noPadLeftIs( True )
      fmts += [ TableOutput.Format( justify="right" ) ]
      fmts[ -1 ].noTrailingSpaceIs( True )
      fmts += [ TableOutput.Format( justify="left" ) ]
      fmts[ -1 ].noPadLeftIs( True )
      fmts += [ TableOutput.Format( justify="right" ) ] * 2
      fmts[ -1 ].noTrailingSpaceIs( True )
      fmts[ -2 ].noTrailingSpaceIs( True )
      table.formatColumns( *fmts )

      for portId, port in sorted( self.portIds.items() ):
         state = "Configured" if port.dmfConfigured else "Not configured"
         table.newRow( port.intf.stringValue, portId,
                       state, port.packetInCount,
                       port.packetOutCount )
      print( table.output() )

####################################################################
# show management dmf interfaces [ INTS ]
####################################################################
class DmfInterfacesModel( Model ):
   __public__ = False

   portIds = Dict( keyType=int, valueType=InterfaceModel,
                   help="Map of port ID to its DMF configuration" )

   def render( self ):
      ret = ""
      for portId, port in sorted( self.portIds.items() ):
         ret += "%s\n" % port.intf.stringValue
         ret += "DMF state: %s\n" % \
               ( "configured" if port.dmfConfigured else "not configured" )
         ret += "OpenFlow port ID: %d\n" % portId
         # LAG ports not supported
         if port.dmfConfigured and not port.lagMembers:
            ret += "PTP state: %s\n" % \
                  ( port.ptpState if port.ptpState else "Not configured" )
         if port.switchportMode is not None:
            ret += "Switchport mode: %s\n" % \
                  getSwitchportModeString( port.switchportMode )
         if port.dynVlans is not None:
            dynVlansStr = vlanSetToCanonicalString( port.dynVlans )
            if not dynVlansStr:
               dynVlansStr = "none"
            if dynVlansStr == "1-4094":
               dynVlansStr = "all"
            ret += "Dynamic VLANs: %s\n" % dynVlansStr
         if port.speed is not None:
            ret += "Port speed: "
            if port.speed == 0:
               ret += "unknown\n"
            else:
               ret += "%d Mbps\n" % port.speed
         if port.stripVlans is not None:
            ret += "Strip VLANs: %s\n" % port.stripVlans
         if port.autoneg is not None:
            ret += "Auto-negotiation: %s\n" % \
                  ( "enabled" if port.autoneg else "disabled" )
         if port.fec is not None:
            ret += "FEC: %s\n" % \
                  getFecEncodingString( port.fec )
         if port.loopbackMode is not None:
            ret += "Loopback mode: %s\n" % \
                  getLoopbackModeString( port.loopbackMode )
         if port.rateLimit is not None:
            ret += "Rate limit: %d Kbps\n" % port.rateLimit
         if port.forceLinkUp is not None:
            ret += "Force link up: %s\n" % \
                  ( "enabled" if port.forceLinkUp else "disabled" )
         if port.xmit is not None:
            ret += "Transmit: %s\n" % \
                  ( "enabled" if port.xmit else "disabled" )
         if port.stripVxlan is not None:
            ret += "Strip VXLAN header: %s\n" % \
                  ( "enabled" if port.stripVxlan else "disabled" )
            if port.stripVxlan and port.udpPort:
               ret += "VXLAN UDP port: %d\n" % port.udpPort
         if port.lagMembers is not None:
            ret += "Member ports: "
            # This is empty string if the collection is empty
            ret += " ".join( sortIntf( port.lagMembers ) )
            ret += "\n"

         ret += "Packets received: %d\n" % port.packetInCount
         ret += "Packets transmitted: %d\n" % port.packetOutCount
         ret += "\n"
      print( ret )
