#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import textwrap
import Tac
import Ark
import Arnet
from Ark import utcTimeRelativeToNowStr
from CliMode.Intf import IntfMode
from CliModel import Enum
from CliModel import UncheckedModel
from CliModel import Dict
from CliModel import Float
from CliModel import Int
from CliModel import Bool
from CliModel import Str
from CliModel import List
from CliModel import Model, Submodel
from ArnetModel import Ip4Address
from ArnetModel import IpGenericAddress
from ArnetModel import MacAddress
from collections import OrderedDict
from IntfModels import Interface
from Ethernet import convertMacAddrToDisplay
from TableOutput import createTable, Format
from TypeFuture import TacLazyType
import Toggles.Dot1xToggleLib as Dot1xToggle
from Toggles.MvrpToggleLib import toggleDot1xMvrpVsaEnabled
import Toggles.DhcpLibToggleLib as DhcpLibToggle
from Dot1xLib import ( portControlEnum2Str, portStatusEnum2Str, hostModeEnum2Str,
      authStageDict, cliStageShorten, reAuthBehaviour, aristaWebAuth,
      serviceTypes, framedIpAddrSourceDict, terminationActionDict,
      shortenFallback, shortenAuthMethod, escapedClassHex, enabledDisabled,
      sessionReplaceDetectionActionEnum2Str, escapedString, nasPortTypeEnum2Str )

NasIdType = TacLazyType( "Radius::NasIdType" )
NasPortType = TacLazyType( "Dot1x::NasPortType" )

framedIpSrcL3Nbr = framedIpAddrSourceDict[ 1 ]

def getServiceType( serviceType ):
   if serviceType < len( serviceTypes ):
      return serviceTypes[ serviceType ]
   else:
      return f"Unknown ({serviceType})"

def getFramedIpSource( framedIpAddress, supplicantStatus, status ):
   bestSrcIndex = 0
   framedIpAddrSource = framedIpAddrSourceDict[ bestSrcIndex ]
   if framedIpAddress != Arnet.IpAddr( '0.0.0.0' ):
      for src in supplicantStatus.ipv4Addr:
         srcIndex = framedIpAddrSourceDict.index( src )
         if ( str( framedIpAddress ) == supplicantStatus.ipv4Addr[ src ] and
              bestSrcIndex < srcIndex ):
            framedIpAddrSource = src
            bestSrcIndex = srcIndex
   elif ( framedIpSrcL3Nbr in supplicantStatus.ipv4Addr and
          status.dot1xFramedIpSrcL3Enabled ):
      # Update framedIpAddrSource only if framed ip feature is enabled
      framedIpAddrSource = framedIpSrcL3Nbr
   return framedIpAddrSource
   
#--------------------------------------------------------------------------------
# EAPI Models
#--------------------------------------------------------------------------------
class Dot1xInterfaceInformation( UncheckedModel ):
   portControl = Enum( values=( "controlled", "forceAuth",
                                "forceUnauth" ),
                            help="Controlled -- enable 802.1X for the interface. "
                            "ForceAuth -- disable 802.1X and put the "
                            "interface into authorized state. "
                            "ForceUnauth -- disable 802.1X and put the "
                            "interface into unauthorized state. " )
   forceAuthPhone = Bool( help="Forced authorization of phone devices" )
   hostMode = Enum( values=( "singleHost", "multiHost", "authAllHost" ),
                        help="singleHost -- Allow only one supplicant on an "
                        "authorized port. "
                        "multiHost -- Allow multiple clients on an "
                        "authorized port. "
                        "authAllHost -- Allow multiple dot1x-authenticated clients "
                        "on an authorized port. " )
   sessionReplaceEnabled = Bool(
         help="Dot1x session replace in single-host mode is enabled" )
   quietPeriod = Float( help="Waiting period in seconds "
                             "after a failed authentication. " )
   txPeriod = Float( help="Waiting period in seconds "
                          "before retrying a message to supplicant. " )
   maxReauthReq = Int( help="Maximum number of reauthentication attempts" )
   reauthTimeoutIgnore = Bool( help="Retain current port auth status on "
                               "reauth server or supplicant timeouts "\
                               "or new reauth requests" )
   authFailVlan = Int( help="Authentication failure VLAN" )
   unauthorizedAccessVlanEgress = Bool( help="Give unauthorized access port egress "
                                             "membership of access VLAN " )
   unauthorizedNativeVlanEgress = Bool( help="Give unauthorized trunk port egress "
                                             "membership of native VLAN " )
   eapolDisabled = Bool( help="EAPOL is disabled" )
   mbaEnabled = Bool( help="MAC-based authentication is enabled" )
   mbaTimeout = \
         Float( help="Timeout for starting MAC based authentication in seconds" )
   mbaFallback = Bool( help="MBA fallback for EAPOL authentication failure" )
   mbaHostMode = Bool( help="MAC based authentication host mode is enabled" )
   idleTimeout = Int( help="Idle-Timeout for supplicants in seconds", optional=True )
   mbaAuthAlways = \
         Bool( help="Retain MAC authentication session when EAPOL is in progress" )

   def render( self ):
      print( '--------------------------------------------' )
      # pylint: disable-next=consider-using-f-string
      print( 'Port control: %s' % portControlEnum2Str[ self.portControl ] )
      # pylint: disable-next=consider-using-f-string
      print( 'Forced phone authorization: %s' %
             enabledDisabled[ self.forceAuthPhone ] )
      # pylint: disable-next=consider-using-f-string
      print( 'EAPOL: %s' % enabledDisabled[ not self.eapolDisabled ] )
      # pylint: disable-next=consider-using-f-string
      print( 'Host mode: %s' % hostModeEnum2Str[ self.hostMode ] )
      if self.hostMode == 'singleHost':
         print( 'Session Replace: %s' % # pylint: disable=consider-using-f-string
               enabledDisabled[ self.sessionReplaceEnabled ] )
      # pylint: disable-next=consider-using-f-string
      print( 'MAC-based authentication: %s' % enabledDisabled[ self.mbaEnabled ] )
      if self.idleTimeout:
         # pylint: disable-next=consider-using-f-string
         print( 'Idle-Timeout: %s seconds' % self.idleTimeout )
      mbaHostModeStr = 'MAC-based authentication host mode: '
      if self.mbaHostMode:
         mbaHostModeStr += hostModeEnum2Str[ self.hostMode ]
      else:
         mbaHostModeStr += 'Unconfigured'
      print( mbaHostModeStr )
      # pylint: disable-next=consider-using-f-string
      print( 'MAC-based authentication always: %s' %
             enabledDisabled[ self.mbaAuthAlways ] )
      # pylint: disable-next=consider-using-f-string
      print( 'Quiet period: %d seconds' % self.quietPeriod )
      # pylint: disable-next=consider-using-f-string
      print( 'TX period: %d seconds' % self.txPeriod )
      # pylint: disable-next=consider-using-f-string
      print( 'Maximum reauth requests: %d' % self.maxReauthReq )
      # pylint: disable-next=consider-using-f-string
      print( 'Ignore reauth timeout: %s' % ( 'Yes' if self.reauthTimeoutIgnore
                                              else 'No' ) )
      # pylint: disable-next=consider-using-f-string
      print( 'Auth failure VLAN: %s' % ( 'Unconfigured' if self.authFailVlan == 0
                                          else self.authFailVlan ) )
      # pylint: disable-next=consider-using-f-string
      print( 'Unauthorized access VLAN egress: %s' %
             ( 'Yes' if self.unauthorizedAccessVlanEgress else 'No' ) )
      # pylint: disable-next=consider-using-f-string
      print( 'Unauthorized native VLAN egress: %s' %
             ( 'Yes' if self.unauthorizedNativeVlanEgress else 'No' ) )
      fallbackStr = 'EAPOL authentication failure fallback: '
      if self.mbaFallback and not self.mbaAuthAlways:
         # pylint: disable-next=consider-using-f-string
         fallbackStr += 'MBA, timeout %d seconds' % self.mbaTimeout
      else:
         fallbackStr += 'Unconfigured'
      print( fallbackStr )

      # pylint: disable-msg=W0105
      '''
      we don't support guest vlan and restricted vlan now
      print( 'AuthFailMaxAttempts\t: %d' % self.authFailMaxAttempt )
      if self.guestVlan:
         print( 'GuestVlan\t\t: %d' % self.guestVlan )
      if self.authFailVlan:
         print( 'AuthFailVlan\t\t: %d' % self.authFailVlan )
      '''

def renderIntfs( label, intfs ):
   for intf in Arnet.sortIntf( intfs ):
      print( label, intf )
      intfs[ intf ].render()
      print()

class Dot1xInterfaceRangeInformation( Model ):
   interfaces = Dict( help="A mapping between interfaces and 802.1X "
                           "interface information", keyType=Interface,
                           valueType=Dot1xInterfaceInformation )

   def render( self ):
      renderIntfs( 'Dot1X Information for', self.interfaces )

class Dot1xInterfaceDetails( Dot1xInterfaceInformation ):
   portAuthorizationState = Enum( values=( "blocked", "authorized",
                                           "guestVlan", "restrictedVlan" ),
                                    help="Blocked -- port is unauthorized. "
                                    "Authorized -- port is authorized. "
                                    "GuestVlan -- port is authorized through "
                                    "guest VLAN"
                                    "RestrictedVlan -- port is authorized through "
                                    "restricted VLAN",
                                    optional=True )
   supplicantMacs = Dict( help="Mapping of supplicant MAC address to"
                              " its reauthentication period in seconds",
                              keyType=MacAddress, valueType=int )
   portShutdownStatus = Str( help='Port shutdown by AAA server', optional=True )

   def render( self ):
      Dot1xInterfaceInformation.render( self )
      if Dot1xToggle.toggleDot1xCoaPortShutdownEnabled():
         if self.portShutdownStatus:
            print( f'Port ErrDisabled by CoA: {self.portShutdownStatus}' )
      if self.portAuthorizationState:
         print( '\nDot1X Authenticator Client\n' )
         print( 'Port status: %s' % # pylint: disable=consider-using-f-string
                portStatusEnum2Str[ self.portAuthorizationState ] )
         if self.supplicantMacs:
            print( 'Supplicant MAC\tReauth Period (in seconds)' )
            print( '--------------\t--------------------------' )
            for mac, reauthPeriod in self.supplicantMacs.items():
               print( '%s\t%d' % # pylint: disable=consider-using-f-string
                      ( convertMacAddrToDisplay( mac ), reauthPeriod ) )

class Dot1xInterfaceRangeDetails( Model ):
   interfaces = Dict( help="A mapping between interfaces and 802.1X "
                           "interface detailed information", keyType=Interface,
                           valueType=Dot1xInterfaceDetails )

   def render( self ):
      renderIntfs( 'Dot1X Information for', self.interfaces )

class Dot1xInterfaceStats( UncheckedModel ):
   rxStart = Int( help="EAPOL Start frames received" )
   rxLogoff = Int( help="EAPOL Logoff frames received" )
   rxRespId = Int( help="EAPOL Resp/Id frames received" )
   rxResp = Int( help="EAP Response frames received" )
   rxInvalid = Int( help="Invalid EAPOL frames received" )
   rxTotal = Int( "EAPOL frames received" )
   txReqId = Int( help="EAP Initial Request frames transmitted" )
   txReq = Int( help="EAP Request frames transmitted" )
   txTotal = Int( help="EAPOL frames transmitted" )
   rxVersion = Int( help="Last EAPOL frame version" )
   lastRxSrcMac = MacAddress( help="Last EAPOL frame source" )

   def render( self ):
      print( '-------------------------------------------------' )
      # pylint: disable-next=consider-using-f-string,bad-string-format-type
      print( 'RX start = %d\t RX logoff = %d\t RX response ID = %d' %
             ( self.rxStart, self.rxLogoff, self.rxRespId ) )
      # pylint: disable-next=consider-using-f-string,bad-string-format-type
      print( 'RX response = %d\t RX invalid = %d\t RX total = %d' %
             ( self.rxResp, self.rxInvalid, self.rxTotal ) )
      # pylint: disable-next=consider-using-f-string,bad-string-format-type
      print( 'TX request ID = %d\t TX request = %d\t TX total = %d' %
             ( self.txReqId, self.txReq, self.txTotal ) )
      # pylint: disable-next=consider-using-f-string,bad-string-format-type
      print( 'RX version = %d\t Last RX src MAC = %s' %
                  ( self.rxVersion,
                        convertMacAddrToDisplay( self.lastRxSrcMac.stringValue ) ) )

class Dot1xDroppedCounterStats( UncheckedModel ):
   unauthEapolPort = Int( help="EAPOL unauthorized port" )
   unauthEapolHost = Int( help="EAPOL unauthorized host" )
   unauthMbaHost = Int( help="MBA unauthorized host" )

   def render( self ):
      print( 'Data packet drop counters:' )
      # pylint: disable-next=consider-using-f-string
      print( 'EAPOL unauthorized port = %d' % self.unauthEapolPort )
      # pylint: disable-next=consider-using-f-string
      print( 'EAPOL unauthorized host = %d' % self.unauthEapolHost )
      # pylint: disable-next=consider-using-f-string
      print( 'MBA unauthorized host = %d' % self.unauthMbaHost )

class Dot1xInterfaceRangeStats( Model ):
   interfaces = Dict( help="A mapping between interfaces and 802.1X "
                           "interface statistics", keyType=Interface,
                           valueType=Dot1xInterfaceStats )
   dropCounters = Dict( help="A mapping between interfaces and 802.1X "
                           "interface dropped counter statistics",
                           keyType=Interface, valueType=Dot1xDroppedCounterStats)

   def render( self ):
      if not self.interfaces:
         return

      for key in Arnet.sortIntf( self.interfaces ):
         # pylint: disable-next=consider-using-f-string
         print( 'Dot1X Authenticator Port Statistics for %s' % key )
         self.interfaces[ key ].render()
         if key not in self.dropCounters:
            print( 'Data packet drop counters disabled' )
         else:
            self.dropCounters[ key ].render()
         print()

class _Dot1xAllBase( UncheckedModel ):
   dynAuth = Bool( help="Dot1X Dynamic Authorization is enabled" )
   lldpBypass = Bool( help="Dot1X LLDP Bypass is enabled" )
   bpduBypass = Bool( help="Dot1X BPDU Bypass is enabled" )
   lacpBypass = Bool( help="Dot1X LACP Bypass is enabled" )
   sendIdentityReqLoggedOffMac = Bool(
         help="Send unicast identity request when a MAC is logged off" )
   sendIdentityReqNewMac = Bool( help="Send unicast identity request on a new MAC" )
   sessionMoveEapol = Bool( help="Dot1x EAPOL session move is enabled" )
   sessionReplaceDetectionEnabled = Bool(
         help="Detect frequent session replacements in single host mode. " )
   sessionReplaceDetectionAction = Enum( values=( "detectionDisabled", "logOnly",
                                                  "errdisable" ),
         help="Action to take when frequent session replacements detected. " )
   systemAuthControl = Bool( help="System Authentication Control is enabled" )
   version = Int( help="Dot1X protocol version" )

   def render( self ):
      print( 'System authentication control:',
             enabledDisabled[ self.systemAuthControl ] )
      print( 'Dot1X LLDP bypass:',
             enabledDisabled[ self.lldpBypass ] )
      print( 'Dot1X BPDU bypass:',
             enabledDisabled[ self.bpduBypass ] )
      if Dot1xToggle.toggleDot1xLacpBypassEnabled():
         print( 'Dot1X LACP bypass:',
                enabledDisabled[ self.lacpBypass ] )
      print( 'Dot1X dynamic authorization:',
             enabledDisabled[ self.dynAuth ] )
      print( 'Dot1X protocol version:',
             self.version )
      print( 'Dot1X unicast identity request on new MAC:',
             enabledDisabled[ self.sendIdentityReqNewMac ] )
      print( 'Dot1X unicast identity request when MAC is logged off:',
             enabledDisabled[ self.sendIdentityReqLoggedOffMac ] )
      print( 'Dot1X EAPOL session move:',
             enabledDisabled[ self.sessionMoveEapol ] )
      print( 'Dot1X single host session replace detection:',
             enabledDisabled[ self.sessionReplaceDetectionEnabled ] )
      if self.sessionReplaceDetectionEnabled:
         print( 'Dot1x single host session replace detection action:',
                sessionReplaceDetectionActionEnum2Str[
                   self.sessionReplaceDetectionAction ] )
      print()

      renderIntfs( 'Dot1X Information for', self.interfaces )

class Dot1xAllInformation( _Dot1xAllBase ):
   interfaces = Dict( keyType=Interface, valueType=Dot1xInterfaceInformation,
         help="A mapping of interface to its Dot1X information" )

class Dot1xAllDetails( _Dot1xAllBase ):
   interfaces = Dict( keyType=Interface, valueType=Dot1xInterfaceDetails,
         help="A mapping of interface to its Dot1X detailed information" )

class Dot1xAllSummary( Dot1xAllDetails ):
   def render( self ):
      # pylint: disable-next=consider-using-f-string
      print( '%-24s%-24s%s' % ( 'Interface', 'Client', 'Status' ) )
      print( '-------------------------------------------------------------' )
      for key in Arnet.sortIntf( self.interfaces ):
         details = self.interfaces[ key ]
         if details.portAuthorizationState:
            for mac in details.supplicantMacs:
               client = convertMacAddrToDisplay( mac ) if mac else 'None'
               # pylint: disable-next=consider-using-f-string
               print( '%-24s%-24s%s' % ( key, client,
                       portStatusEnum2Str[ details.portAuthorizationState ] ) )

class Dot1xSupplicant( Model ):
   identity = Str( help="EAP identity in use on this interface" )
   eapMethod = Enum( values=( "unset", "fast", "tls" ),
                     help="unset -- EAP method is currently unset."
                     "fast -- EAP method in use is EAP FAST."
                     "tls -- EAP method in use is TLS." )
   sslProfileName = Str( help="SSL profile name" )
   status = Enum( values=( "unused", "down", "connecting", "failed", "success" ),
                  help="unused -- 802.1X supplicant is uninitialized"
                  "down -- 802.1X supplicant is down."
                  "connecting -- 802.1X supplicant is connecting."
                  "failed -- 802.1X supplicant has failed."
                  "success -- 802.1X supplicant has succeeded." )
   supplicantMac = MacAddress( help="802.1X supplicant MAC address for "
                                    "the interface" )
   authenticatorMac = MacAddress( help="802.1X authenticator MAC address for "
                                       "the interface" )
   eapTlsVersion = Str( help="EAP TLS version" )
   eapTlsCipher = Str( help="EAP TLS cipher" )
   fipsMode = Bool( help="FIPS restrictions enabled", optional=True )

   def fromTacc( self, supplicantIntfStatus, wpaSupplicantIntfStatus ):
      self.identity = supplicantIntfStatus.identity
      self.eapMethod = supplicantIntfStatus.eapMethod
      self.sslProfileName = supplicantIntfStatus.sslProfileName
      self.status = wpaSupplicantIntfStatus.connectionStatus
      eapKey = wpaSupplicantIntfStatus.eapKey
      self.supplicantMac = eapKey.myMacAddr
      self.authenticatorMac = eapKey.peerMacAddr
      eapTlsVersion = re.sub( r'^TLSv', '', wpaSupplicantIntfStatus.eapTlsVersion )
      if eapTlsVersion == '1':
         eapTlsVersion = '1.0'
      self.eapTlsVersion = eapTlsVersion
      self.eapTlsCipher = wpaSupplicantIntfStatus.eapTlsCipher
      self.fipsMode = wpaSupplicantIntfStatus.fipsMode

   def render( self ):
      indentation = ' ' * 4
      # pylint: disable-next=consider-using-f-string
      print( indentation + "Identity: %s" % self.identity )
      eapMethod = self.eapMethod
      if self.eapMethod == "tls":
         eapMethod = "TLS"
      # pylint: disable-next=consider-using-f-string
      print( indentation + "EAP method: %s" % eapMethod )
      # pylint: disable-next=consider-using-f-string
      print( indentation + "SSL profile: %s" % self.sslProfileName )
      # pylint: disable-next=consider-using-f-string
      print( indentation + "Status: %s" % self.status )
      # pylint: disable-next=consider-using-f-string
      print( indentation + "Supplicant MAC: %s" % self.supplicantMac )
      # pylint: disable-next=consider-using-f-string
      print( indentation + "Authenticator MAC: %s" % self.authenticatorMac )
      # pylint: disable-next=consider-using-f-string
      print( indentation + "TLS version: %s" % self.eapTlsVersion )
      # pylint: disable-next=consider-using-f-string
      print( indentation + "TLS cipher: %s" % self.eapTlsCipher )
      # pylint: disable-next=consider-using-f-string
      print( indentation + "FIPS restrictions: %s" %
             ( "enabled" if self.fipsMode else "disabled" ) )

class Dot1xSupplicants( Model ):
   interfaces = Dict( help="A mapping between interfaces and 802.1X supplicant "
                           "interface information", keyType=Interface,
                           valueType=Dot1xSupplicant )

   def render( self ):
      renderIntfs( "Interface:", self.interfaces )

class Dot1XSupplicantAcctAttr( Model ):
   supplicantMac = MacAddress( help="Supplicant MAC address",
                               default=Tac.Type( "Arnet::EthAddr" ).ethAddrZero )
   # Used as NAS-Port-Id
   interface = Interface( help="Supplicant interface", optional=True )
   identity = Str( help="EAP identity in use on this interface", optional=True )
   # Currently always ethernetPort
   nasPortType = Enum( values=( "ethernet", "virtual" ),
                       help="ethernet -- Ethernet port"
                            "virtual -- Virtual port", optional=True )
   nasIdentifier = Str( help="Supplicant NAS-Identifier", optional=True )
   callingStationId = Str( help="Supplicant Calling-Station-Id", optional=True )
   calledStationId = Str( help="Supplicant Called-Station-Id", optional=True )
   supplicantClass = Str( help="Supplicant Class", optional=True )
   framedIpAddress = Ip4Address( help="Supplicant Framed-IP-Address", optional=True )
   framedIpAddrSource = Str( help="Supplicant Framed-IP-Address source",
                             optional=True )
   serviceType = Str( help="Supplicant Service-Type", optional=True )
   lldpSysName = Str( help="Supplicant LLDP system name from LLDP TLV 5",
                      optional=True )
   lldpSysDesc = Str( help="Supplicant LLDP system description from LLDP TLV 6",
                      optional=True )
   hostname = Str( help="Supplicant DHCP hostname from DHCP option 12",
                   optional=True )
   paramReqList = Str( help="Supplicant DHCP parameter request list from DHCP "
                       "option 55", optional=True )
   vendorClassId = Str( help="Supplicant DHCP vendor class identifier from DHCP "
                        "option 60", optional=True )

   def getNasIdentifier( self, radiusConfig, netStatus ):
      if radiusConfig.nasIdType != NasIdType.disabled:
         if radiusConfig.nasIdType == NasIdType.custom:
            nasIdentifier = radiusConfig.nasId
         elif radiusConfig.nasIdType == NasIdType.hostname:
            nasIdentifier = netStatus.hostname
         elif radiusConfig.nasIdType == NasIdType.fqdn:
            nasIdentifier = netStatus.fqdn
         if nasIdentifier and nasIdentifier != "localhost":
            return nasIdentifier
      return ''

   def getCalledStationId( self, ethIntfStatusDir, intfId ):
      portMac = ethIntfStatusDir.intfStatus[ intfId ].addr
      return str( portMac ).upper().replace( ':', '-' )

   def getNasPortType( self, nasPortType ):
      if nasPortType == NasPortType.ethernetPort:
         return 'ethernet'
      elif nasPortType == NasPortType.virtualPort:
         return 'virtual'
      return None

   def fromTacc( self, supplicantStatus, config, intfId, radiusConfig,
                 ethIntfStatusDir, netStatus, status ):
      self.supplicantMac = supplicantStatus.mac
      if supplicantStatus.serviceType != supplicantStatus.serviceTypeUnDefined:
         self.serviceType = getServiceType( supplicantStatus.serviceType )
      elif config.serviceType:
         self.serviceType = ( "Call Check" if supplicantStatus.mbaSupplicant else
                              "Framed" )
      self.lldpSysName = escapedString( supplicantStatus.lldpTlvs.lldpSysName )
      self.lldpSysDesc = escapedString( supplicantStatus.lldpTlvs.lldpSysDesc )
      self.supplicantClass = escapedClassHex( supplicantStatus.classAttr )
      self.identity = supplicantStatus.identity
      self.framedIpAddress = supplicantStatus.framedIpAddr
      self.framedIpAddrSource = getFramedIpSource(
            self.framedIpAddress, supplicantStatus, status )
      self.callingStationId = str( supplicantStatus.mac ).upper().replace( ':', '-' )
      self.calledStationId = self.getCalledStationId( ethIntfStatusDir, intfId )
      self.nasPortType = self.getNasPortType( NasPortType.ethernetPort )
      self.nasIdentifier = self.getNasIdentifier( radiusConfig, netStatus )
      self.hostname = supplicantStatus.dhcpOptions.hostname
      self.paramReqList = supplicantStatus.dhcpOptions.paramReqList
      self.vendorClassId = supplicantStatus.dhcpOptions.vendorClassId

   def render( self ):
      if str( self.supplicantMac ) == Tac.Type( "Arnet::EthAddr" ).ethAddrZero:
         return
      print( "User-Name:", self.identity )
      print( "NAS-Port-Id:", str( self.interface ).strip( "'" ) )
      print( "NAS-Port-Type:", nasPortTypeEnum2Str[ self.nasPortType ] )
      print( "Called-Station-Id:", self.calledStationId )
      print( "Calling-Station-Id:", self.callingStationId )
      print( "Service-Type:", self.serviceType )
      print( "NAS-Identifier:", self.nasIdentifier )
      print( "Class:", self.supplicantClass )
      if self.framedIpAddress != Arnet.IpAddr( '0.0.0.0' ):
         print( "Framed-IP-Address:",
                f'{self.framedIpAddress}, {self.framedIpAddrSource}' )
      elif self.framedIpAddrSource == framedIpSrcL3Nbr:
         print( "Framed-IP-Address:",
                f"Unavailable (locally), {self.framedIpAddrSource}" )
      else:
         print( "Framed-IP-Address:", self.framedIpAddress )
      print( "Arista-LLDP" )
      print( "System name:", self.lldpSysName )
      print( "System description:", self.lldpSysDesc )
      if DhcpLibToggle.toggleDhcpOptionsProfilingEnabled():
         print( "Arista-DHCP" )
         print( "Hostname:", self.hostname )
         print( "Parameter request list:", self.paramReqList )
         print( "Vendor class identifier:", self.vendorClassId )

class Dot1XSupplicantAttr( Model ):
   supplicantMac = MacAddress( help="Supplicant MAC address",
                               default=Tac.Type( "Arnet::EthAddr" ).ethAddrZero )
   identity = Str( help="EAP identity in use on this interface", default='' )
   interface = Interface( help="Supplicant interface", default='' )
   authMethod = Str( help="Authentication method used by the supplicant",
                     default='' )
   authStage = Str( help="Supplicant state", default='' )
   fallback = Str( help="Fallback applied", default='' )
   callingStationId = Str( help="Supplicant Calling-Station-Id", default='' )
   reauthBehavior = Str( help="Supplicant re-authentication behavior", default='' )
   reauthInterval = Int( help="Supplicant re-authentication interval in seconds",
                         default=0 )
   timeToReauth = Int( help="Time remaining for  next reauthentication in seconds",
         optional=True )
   cacheConfTime = Int( help="Cached-results configuration interval in seconds",
                         default=0 )
   timeToCacheExpiry = Int( help="Time remaining for cached-results in seconds",
         optional=True )
   vlanId = Str( help='VLAN ID', default='' )
   accountingSessionId = Str( help="Supplicant Accounting-Session-Id", default='' )
   captivePortal = Str( help="Supplicant Captive portal", optional=True )
   captivePortalSource = Str( help="Supplicant Captive portal source",
                              optional=True )
   aristaMvrp = Str( help="Supplicant Arista-Mvrp", optional=True )
   aristaWebAuth = Str( help="Supplicant Arista-WebAuth", optional=True )
   supplicantClass = Str( help="Supplicant Class", optional=True )
   filterId = Str( help="Supplicant Filter-Id", optional=True )
   framedIpAddress = Ip4Address( help="Supplicant Framed-IP-Address", optional=True )
   framedIpAddrSource = Str( help="Supplicant Framed-IP-Address source",
                             optional=True )
   nasFilterRules = List( valueType=str,
                          help="The list of supplicant NAS-Filter-Rule",
                          optional=True )
   deviceType = Str( help='Device Type classification received from AAA',
           optional=True )

   serviceType = Str( help="Supplicant Service-Type", optional=True )
   sessionTimeout = Int( help="Supplicant Session-Timeout", optional=True )
   idleTimeout = Int( help="Supplicant Idle-Timeout in seconds", optional=True )
   terminationAction = Str( help="Supplicant Termination-Action", optional=True )
   tunnelPrivateGroupId = Str( help="Supplicant Tunnel-Private-GroupId",
                               optional=True )
   aristaTenantId = Str( help="Supplicant Arista-Tenant-Id", optional=True )
   aristaPeriodicIdentity = Str( help="Supplicant Arista-PeriodicIdentity",
                               optional=True )
   cachedAuthAtLinkDown = Bool( help="Cache supplicant on link down", default=False )
   reauthTimeoutSeen = Bool( help="Reauth timeout seen", default=False )
   sessionCached = Bool( help="Supplicant software-cached", default=False )
   aristaSegmentId = Str( help="Supplicant Arista-Segment-Id", optional=True )
   detail_ = Bool( help='Print details', default=False )

   def supplicantConfCacheTime( self, supplicantStatus, config ):
      cacheTimeout = Tac.endOfTime
      dotIntfCfg = config.dot1xIntfConfig[ self.interface ]
      isPhone = supplicantStatus.deviceType == "phone"
      # phone + intf
      if cacheTimeout == Tac.endOfTime and isPhone:
         cacheTimeout = \
            dotIntfCfg.aaaUnresponsivePhoneApplyCachedResultsTimeout.timeout
      # Phone + global
      if cacheTimeout == Tac.endOfTime and isPhone:
         cacheTimeout = \
            config.aaaUnresponsivePhoneApplyCachedResultsTimeout.timeout
      # Intf
      if cacheTimeout == Tac.endOfTime:
         cacheTimeout = \
            dotIntfCfg.aaaUnresponsiveApplyCachedResultsTimeout.timeout
      # Global
      if cacheTimeout == Tac.endOfTime:
         cacheTimeout = config.aaaUnresponsiveApplyCachedResultsTimeout.timeout
      return cacheTimeout

   def fromTacc( self, supplicantStatus, config, detail, status ):
      self.supplicantMac = supplicantStatus.mac
      self.identity = supplicantStatus.identity
      self.authMethod = 'EAPOL'
      if supplicantStatus.forceAuthPhoneApplied:
         self.authMethod = "AUTH-PHONE"
      elif supplicantStatus.mbaSupplicant:
         self.authMethod = 'MAC-BASED-AUTH'
      elif supplicantStatus.authFallBack:
         self.authMethod = "MBA (EAPOL authentication failure fallback)"

      self.authStage = authStageDict[ supplicantStatus.authStage ]
      self.fallback = shortenFallback[ supplicantStatus.fallbackVlan() ]
      if self.fallback == 'AUTH-FAIL-VLAN' and supplicantStatus.inAuthFailAcl:
         self.fallback = 'AUTH-FAIL-ACL'

      if supplicantStatus.authFallBack and self.fallback == "NONE":
         self.fallback = "MBA"

      self.callingStationId = str( supplicantStatus.mac ).upper().replace( ':', '-' )
      self.reauthBehavior = reAuthBehaviour[ supplicantStatus.reauthBehavior ]
      self.reauthInterval = supplicantStatus.sessionTimeout
      if supplicantStatus.lastReauthTimerArmed != Tac.endOfTime:
         self.timeToReauth = int( supplicantStatus.lastReauthTimerArmed -
               Tac.utcNow() )
         if self.timeToReauth < 0: # pylint: disable=consider-using-max-builtin
            self.timeToReauth = 0
      self.vlanId = supplicantStatus.vlan
      self.accountingSessionId = supplicantStatus.acctSessionId
      self.cachedAuthAtLinkDown = supplicantStatus.cachedAuthAtLinkDown
      self.reauthTimeoutSeen = supplicantStatus.reauthTimeoutSeen
      self.sessionCached = supplicantStatus.sessionCached
      self.aristaTenantId = ( str( supplicantStatus.tenantId )
            if supplicantStatus.tenantId else '' )

      if detail:
         cacheCfgTime = self.supplicantConfCacheTime( supplicantStatus, config )
         if Tac.endOfTime not in ( supplicantStatus.cacheUpdationTime,
               cacheCfgTime ):
            self.cacheConfTime = int( cacheCfgTime )
            timeSpentInCache = int( Tac.utcNow() -
                  supplicantStatus.cacheUpdationTime )
            self.timeToCacheExpiry = int( self.cacheConfTime - timeSpentInCache )
            # pylint: disable-next=consider-using-max-builtin
            if self.timeToCacheExpiry < 0:
               self.timeToCacheExpiry = 0
         self.captivePortal = supplicantStatus.captivePortal
         self.captivePortalSource = supplicantStatus.captivePortalSource
         self.aristaWebAuth = aristaWebAuth( supplicantStatus.webAuth )
         self.supplicantClass = escapedClassHex( supplicantStatus.classAttr )
         self.framedIpAddress = supplicantStatus.framedIpAddr
         if supplicantStatus.deviceType == "phone":
            self.deviceType = supplicantStatus.deviceType
         self.framedIpAddrSource = getFramedIpSource(
               self.framedIpAddress, supplicantStatus, status )
         if supplicantStatus.serviceType != supplicantStatus.serviceTypeUnDefined:
            self.serviceType = getServiceType( supplicantStatus.serviceType )
         elif config.serviceType:
            self.serviceType = ( "Call Check" if supplicantStatus.mbaSupplicant else
                                 "Framed" )
         self.sessionTimeout = supplicantStatus.sessionTimeoutSrv
         if supplicantStatus.terminationAction < len( terminationActionDict ):
            self.terminationAction = \
                  terminationActionDict[ supplicantStatus.terminationAction ]
         else:
            self.terminationAction = ''
         self.tunnelPrivateGroupId = supplicantStatus.tunnelPrivateGroupId
         if Dot1xToggle.toggleDot1xSegmentSecurityEnabled():
            self.aristaSegmentId = supplicantStatus.segmentId
         if Tac.endOfTime != supplicantStatus.idleTimeout:
            self.idleTimeout = int( supplicantStatus.idleTimeout )
         self.aristaPeriodicIdentity = supplicantStatus.periodicIdentity
         if toggleDot1xMvrpVsaEnabled():
            # The value of the vsa will never get set to 'invalid' as we would
            # fail authorization of the supplicant and not update mvrpVsa.
            if supplicantStatus.mvrpVsa != 'notReceived':
               self.aristaMvrp = supplicantStatus.mvrpVsa
            else:
               self.aristaMvrp = ''

   def render( self ):
      if str( self.supplicantMac ) == Tac.Type( "Arnet::EthAddr" ).ethAddrZero:
         return
      if self.detail_:
         print( "Operational:" )
      print( "Supplicant MAC:",
             convertMacAddrToDisplay( self.supplicantMac.stringValue ) )
      print( "User name:", self.identity )
      print( "Interface:", str( self.interface ).replace( "'", "" ) )
      print( "Authentication method:", self.authMethod )
      if self.sessionCached:
         authStage = "SOFTWARE-CACHED"
      elif self.cachedAuthAtLinkDown:
         authStage = "DISCONNECTED-CACHED"
      elif self.reauthTimeoutSeen:
         authStage = "SUCCESS-CACHED"
      else:
         authStage = self.authStage
      print( "Supplicant state:", authStage )
      print( "Fallback applied:", self.fallback )
      print( "Calling-Station-Id:", self.callingStationId )
      print( "Reauthentication behaviour:", self.reauthBehavior )
      print( "Reauthentication interval:", self.reauthInterval, "seconds" )
      if self.timeToReauth is not None:
         print( "Time until reauthentication:", self.timeToReauth, "seconds" )
      if self.vlanId:
         print( "VLAN ID:", str( self.vlanId ) )
      else:
         print( "VLAN ID: " )
      print( "Accounting-Session-Id:", self.accountingSessionId )
      if self.detail_:
         print( "Configured cached result timeout:", self.cacheConfTime, "seconds" )
         if self.timeToCacheExpiry is not None:
            print( "Client cached result expiry:",
                  self.timeToCacheExpiry, "seconds" )
         if self.captivePortal:
            # pylint: disable-next=consider-using-f-string
            print( "Captive portal: {} (source: {})".format( self.captivePortal,
                                                    self.captivePortalSource ) )
         else:
            print( "Captive portal:" )
         print( "\nAAA Server Returned:" )
         # Print attributes in sorted order
         if toggleDot1xMvrpVsaEnabled():
            print( "Arista-Mvrp:", self.aristaMvrp )
         if Dot1xToggle.toggleDot1xSegmentSecurityEnabled():
            print( "Arista-Segment-Id:", self.aristaSegmentId )
         print( "Arista-WebAuth:", self.aristaWebAuth )
         print( "Class:", self.supplicantClass )
         print( "Filter-Id:", self.filterId )
         if self.framedIpAddress != Arnet.IpAddr( '0.0.0.0' ):
            print( "Framed-IP-Address:",
                   f'{self.framedIpAddress}, {self.framedIpAddrSource}' )
         elif self.framedIpAddrSource == framedIpSrcL3Nbr:
            print( "Framed-IP-Address: Unavailable (locally),"
                   f'{self.framedIpAddrSource}' )
         else:
            print( "Framed-IP-Address:", self.framedIpAddress )

         nasFilterPrint = "NAS-Filter-Rule: "
         tab = " " * len( nasFilterPrint )
         if self.nasFilterRules:
            for count, rule in enumerate( self.nasFilterRules ):
               if count == 0:
                  nasFilterPrint += rule
               else:
                  nasFilterPrint += '\n' + tab + rule
         print( nasFilterPrint.strip( '\n' ) )
         print( "Service-Type:", self.serviceType )
         print( "Session-Timeout:", self.sessionTimeout, "seconds" )
         if self.idleTimeout:
            print( "Idle-Timeout:", self.idleTimeout, "seconds" )
         print( "Termination-Action:", self.terminationAction )
         print( "Tunnel-Private-GroupId:", self.tunnelPrivateGroupId )
         print( "Arista-Tenant-Id:", self.aristaTenantId )
         print( "Arista-PeriodicIdentity:", self.aristaPeriodicIdentity )
         if self.deviceType == "phone":
            print( "DeviceType:", self.deviceType )

class Dot1XSupplicantExtendedAttr( Model ):
   supplicantAttr = Submodel( valueType=Dot1XSupplicantAttr,
         help="Supplicant operational and AAA returned attributes" )
   vlanName = Str( help="VLAN Name", optional=True )
   staticVlan = Bool( help="Statically configured VLAN", default=False )
   internalVlan = Bool( help="Internally configured VLAN", default=False )
   deviceDomain = Enum( values=( "data", "phone" ),
         help="data - 802.1X supplicant is a data device"
              "phone - 802.1X supplicant is a phone device", default="data" )

   def fromTacc( self, supplicantStatus, config, cliConfig, bridgingConfig,
         mergedHostTable, intfName, status ):
      self.supplicantAttr = Dot1XSupplicantAttr()
      # Set operational attributes
      self.supplicantAttr.interface = intfName
      self.supplicantAttr.fromTacc( supplicantStatus, config, False, status )
      # Set captive portal and the required AAA attributes
      self.supplicantAttr.captivePortal = supplicantStatus.captivePortal
      self.supplicantAttr.captivePortalSource = supplicantStatus.captivePortalSource
      self.supplicantAttr.aristaWebAuth = aristaWebAuth( supplicantStatus.webAuth )
      self.supplicantAttr.framedIpAddress = supplicantStatus.framedIpAddr
      self.supplicantAttr.framedIpAddrSource = getFramedIpSource(
            self.supplicantAttr.framedIpAddress, supplicantStatus, status )
      self.supplicantAttr.sessionTimeout = supplicantStatus.sessionTimeoutSrv
      if Tac.endOfTime != supplicantStatus.idleTimeout:
         self.supplicantAttr.idleTimeout = int( supplicantStatus.idleTimeout )
      # Set VLAN related attributes
      if not supplicantStatus.vlan:
         if hostEntry := mergedHostTable.hostEntry.get( supplicantStatus.mac ):
            if bridgingConfig.vlanConfig[ hostEntry.vlanId ].internal:
               self.internalVlan = True
            else:
               self.staticVlan = True
            self.supplicantAttr.vlanId = str( hostEntry.vlanId )
      if vlanId := self.supplicantAttr.vlanId:
         if vc := cliConfig.vlanConfig.get( int( vlanId ) ):
            self.vlanName = vc.configuredName

   def render( self ):
      suppAttr = self.supplicantAttr
      if str( suppAttr.supplicantMac ) == Tac.Type( "Arnet::EthAddr" ).ethAddrZero:
         return
      print( "Supplicant:", suppAttr.identity, f"({suppAttr.supplicantMac})" )
      print( "Operational:" )
      print( "Supplicant MAC:", convertMacAddrToDisplay(
         suppAttr.supplicantMac.stringValue ) )
      if suppAttr.framedIpAddress != Arnet.IpAddr( '0.0.0.0' ):
         print( "Supplicant IP:",
               f'{suppAttr.framedIpAddress}, {suppAttr.framedIpAddrSource}' )
      elif suppAttr.framedIpAddrSource == framedIpSrcL3Nbr:
         print( "Framed-IP-Address: Unavailable (locally),"
                f'{suppAttr.framedIpAddrSource}' )
      else:
         print( "Supplicant IP:", suppAttr.framedIpAddress )
      print( "User name:", suppAttr.identity )
      print( "Interface:", str( suppAttr.interface ).replace( "'", "" ) )
      print( "Authentication method:", suppAttr.authMethod )
      if suppAttr.sessionCached:
         authStage = "SOFTWARE-CACHED"
      elif suppAttr.cachedAuthAtLinkDown:
         authStage = "DISCONNECTED-CACHED"
      elif suppAttr.reauthTimeoutSeen:
         authStage = "SUCCESS-CACHED"
      else:
         authStage = suppAttr.authStage
      print( "Supplicant state:", authStage )
      print( "Fallback applied:", suppAttr.fallback )
      print( "Reauthentication behaviour:", suppAttr.reauthBehavior )
      print( "Reauthentication interval:", suppAttr.reauthInterval, "seconds" )
      if suppAttr.timeToReauth is not None:
         print( "Time until reauthentication:", suppAttr.timeToReauth, "seconds" )
      if suppAttr.vlanId:
         if self.internalVlan:
            vlanType = "(internal)"
         elif self.staticVlan:
            vlanType = "(static)"
         else:
            vlanType = "(dynamic)"
         print( "VLAN ID:", str( suppAttr.vlanId ), vlanType )
      else:
         print( "VLAN ID: " )
      if self.vlanName:
         print( "VLAN Name:", self.vlanName )
      print( "Device type:", self.deviceDomain.capitalize() )
      print( "Accounting-Session-Id:", suppAttr.accountingSessionId )
      if suppAttr.captivePortal:
         # pylint: disable-next=consider-using-f-string
         print( "Captive portal: {} (source: {})".format( suppAttr.captivePortal,
                                                 suppAttr.captivePortalSource ) )
      else:
         print( "Captive portal:" )

      print( "\nAAA Server Returned:" )
      print( "Arista-WebAuth:", suppAttr.aristaWebAuth )
      print( "Filter-Id:", suppAttr.filterId )
      nasFilterPrint = "NAS-Filter-Rule: "
      tab = " " * len( nasFilterPrint )
      if suppAttr.nasFilterRules:
         for count, rule in enumerate( suppAttr.nasFilterRules ):
            if count == 0:
               nasFilterPrint += rule
            else:
               nasFilterPrint += '\n' + tab + rule
      print( nasFilterPrint.strip( '\n' ) )
      print( "Session-Timeout:", suppAttr.sessionTimeout, "seconds" )
      if suppAttr.idleTimeout:
         print( "Idle-Timeout:", suppAttr.idleTimeout, "seconds" )
      print( "Arista-Tenant-Id:", suppAttr.aristaTenantId )

class Dot1xIntfHosts( Model ):
   supplicants = Dict( help="A mapping betweeen MAC addresses and Dot1X or MBA "
                            "supplicant attributes", keyType=MacAddress,
                            valueType=Dot1XSupplicantExtendedAttr )

   def render( self ):
      for supplicant in self.supplicants.values():
         print()
         supplicant.render()

class Dot1xIntfHostsDetail( Model ):
   interfaces = Dict( help="A mapping between interfaces and their host details",
         keyType=Interface, valueType=Dot1xIntfHosts )

   def render( self ):
      renderIntfs( "Interface host details:", self.interfaces )

class Dot1xVlanHostStats( Model ):
   eapol = Int( help="Count of EAPOL supplicants in VLAN", default=0 )
   mba = Int( help="Count of MBA supplicants in VLAN", default=0 )

   def fromTacc( self, dot1xStatus, vlanId ):
      vlanHostStats = dot1xStatus.vlanHostStats.get( vlanId )
      if vlanHostStats is None:
         return

      self.eapol = vlanHostStats.eapolCount
      self.mba = vlanHostStats.mbaCount

class Dot1xHostStats( Model ):
   webAuth = Int( help="Count of web-auth supplicants", default=0 )
   totalSupplicant = Int( help="Count of all supplicants", default=0 )

   eapolSuccess = Int( help="Count of EAPOL supplicants in success state",
         default=0 )
   mbaSuccess = Int( help="Count of MBA supplicants in success state",
         default=0 )

   eapolFailed = Int( help="Count of EAPOL supplicants in failed state",
         default=0 )
   mbaFailed = Int( help="Count of MBA supplicants in failed state", default=0 )

   eapolCached = Int( help="Count of EAPOL supplicants in cached state",
         default=0 )
   mbaCached = Int( help="Count of MBA supplicants in cached state", default=0 )

   eapolFallbackTotal = Int( help="Count of EAPOL supplicants in all fallback "
         "states", default=0 )
   mbaFallbackTotal = Int( help="Count of MBA supplicants in all fallback "
         "states", default=0 )

class Dot1xInterfaceHostStats( Model ):
   authFailCount = Int( help="Count of supplicants in auth-fail VLAN", default=0 )
   aaaUnrespCount = Int( help="Count of supplicants in AAA unresponsive VLAN",
         default=0 )
   aaaUnrespPhoneCount = Int( help="Count of supplicants in AAA unresponsive "
         "phone VLAN", default=0 )
   guestCount = Int( help="Count of supplicants in guest VLAN", default=0 )
   eapolToMbaFallbackCount = Int( help="Count of supplicants in EAPOL to MBA "
         "fallback", default=0 )
   dot1xEnabled = Bool( help="Dot1x is enabled on the port",
         default=False )
   portControl = Enum( values=( "controlled", "forceAuth",
                                "forceUnauth" ),
                            help="Controlled -- enable 802.1X for the interface. "
                            "ForceAuth -- disable 802.1X and put the "
                            "interface into authorized state. "
                            "ForceUnauth -- disable 802.1X and put the "
                            "interface into unauthorized state. ",
                            default="forceAuth" )
   authFailConfigured = Bool( help="Auth fail vlan is configured",
         default=False )
   guestVlanConfigured = Bool( help="Guest vlan is configured",
         default=False )
   aaaUnrespConfigured = Bool( help="AAA unresponsive agnostic VLAN is configured",
         default=False )
   aaaUnrespPhoneConfigured = Bool( help="AAA unresponsive phone VLAN is configured",
         default=False )
   mbaFallbackConfigured = Bool( help="EAPOL to MBA fallback is configured",
         default=False )

   hostStats = Submodel( valueType=Dot1xHostStats,
         help="Aggregate supplicant counts" )

   def fromTacc( self, dot1xStatus, intfId ):
      intfHostStats = dot1xStatus.interfaceHostStats.get( intfId )
      
      if intfHostStats is None:
         return
      
      self.authFailCount = intfHostStats.authFailCount
      self.aaaUnrespCount = intfHostStats.aaaUnrespCount
      self.aaaUnrespPhoneCount = intfHostStats.aaaUnrespPhoneCount
      self.guestCount = intfHostStats.guestCount
      self.eapolToMbaFallbackCount = intfHostStats.eapolToMbaFallbackCount

      self.hostStats.eapolSuccess = intfHostStats.eapolSuccessCount
      self.hostStats.mbaSuccess = intfHostStats.mbaSuccessCount

      self.hostStats.eapolFailed = intfHostStats.eapolFailedCount
      self.hostStats.mbaFailed = intfHostStats.mbaFailedCount

      self.hostStats.eapolCached = intfHostStats.eapolCachedCount
      self.hostStats.mbaCached = intfHostStats.mbaCachedCount

      self.hostStats.eapolFallbackTotal = intfHostStats.eapolFallbackTotalCount
      self.hostStats.mbaFallbackTotal = intfHostStats.mbaFallbackTotalCount

class Dot1xAllHostStats( Model ):
   interfaces = Dict( help="A mapping between interfaces and their host "
                              "statistics", keyType=str,
                              valueType=Dot1xInterfaceHostStats )
   vlans = Dict( help="A mapping between VLANs and their host statistics",
                         keyType=int, valueType=Dot1xVlanHostStats )

   vlanFiltered_ = Bool( help="Filtered by VLAN", default=False )
   interfaceFiltered_ = Bool( help="Filtered by Interface",
         default=False )
   systemAuthControl = Bool( help="Dot1x is enabled globally" )
   
   hostStats = Submodel( valueType=Dot1xHostStats,
         help="Aggregate supplicant counts" )

   def fromTacc( self, dot1xStatus, intfs=None, vlans=None ):
      self.systemAuthControl = dot1xStatus.dot1xEnabled
      self.vlanFiltered_ = vlans is not None
      self.interfaceFiltered_ = intfs is not None
      self.hostStats = Dot1xHostStats()
      # This case is for the global version of the command which doesn't filter
      # by interface or vlan. Here we display information for all interfaces
      # and vlans, that have non-zero counts.
      if ( not vlans and not intfs ):
         vlans = list( dot1xStatus.vlanHostStats.keys() )
         intfs = []
         for intf, dot1xIntfStatus in dot1xStatus.dot1xIntfStatus.items():
            if len( dot1xIntfStatus.supplicant ) > 0:
               intfs.append( intf )

      if vlans:
         for vlan in vlans:
            vlanHostStats = Dot1xVlanHostStats()
            if vlan in dot1xStatus.vlanHostStats:
               vlanHostStats.fromTacc( dot1xStatus, vlan )
            self.vlans[ vlan ] = vlanHostStats

      if intfs:
         for intf in intfs:
            interfaceHostStats = Dot1xInterfaceHostStats()
            interfaceHostStats.hostStats = Dot1xHostStats()
            # This block sets the values of various configuration flags
            if intf in dot1xStatus.dot1xIntfStatus:
               dot1xIntfStatus = dot1xStatus.dot1xIntfStatus.get( intf )
               if dot1xIntfStatus is None:
                  self.interfaces[ intf ] = interfaceHostStats
                  continue
               interfaceHostStats.dot1xEnabled = True
               interfaceHostStats.portControl = dot1xIntfStatus.portCtrlSetting
               interfaceHostStats.authFailConfigured =\
                     bool( dot1xIntfStatus.authFailVlan )
               interfaceHostStats.guestVlanConfigured =\
                     bool( dot1xIntfStatus.guestVlan )
               interfaceHostStats.aaaUnrespConfigured = \
                     dot1xIntfStatus.aaaUnresponsiveTrafficAllow.enabled()
               interfaceHostStats.aaaUnrespPhoneConfigured = \
                     dot1xIntfStatus.aaaUnresponsivePhoneAllow
               interfaceHostStats.mbaFallbackConfigured = \
                     dot1xIntfStatus.mbaFallbackStatus
               # These counts are directly obtained from dot1xInterfaceStatus
               interfaceHostStats.hostStats.totalSupplicant = \
                     len( dot1xIntfStatus.supplicant )
               interfaceHostStats.hostStats.webAuth = \
                     len( dot1xIntfStatus.webAuthMac )
               
               self.hostStats.totalSupplicant += len( dot1xIntfStatus.supplicant )
               self.hostStats.webAuth += len( dot1xIntfStatus.webAuthMac )

            # This block sets the interface counts and updates the global counts.
            if intf in dot1xStatus.interfaceHostStats:
               interfaceHostStats.fromTacc( dot1xStatus, intf )
               # Compute global counts
               self.hostStats.eapolSuccess += \
                     interfaceHostStats.hostStats.eapolSuccess
               self.hostStats.mbaSuccess += interfaceHostStats.hostStats.mbaSuccess
               self.hostStats.eapolFailed += interfaceHostStats.hostStats.eapolFailed
               self.hostStats.mbaFailed += interfaceHostStats.hostStats.mbaFailed
               self.hostStats.eapolCached += interfaceHostStats.hostStats.eapolCached
               self.hostStats.mbaCached += interfaceHostStats.hostStats.mbaCached
               self.hostStats.eapolFallbackTotal += \
                     interfaceHostStats.hostStats.eapolFallbackTotal
               self.hostStats.mbaFallbackTotal += \
                     interfaceHostStats.hostStats.mbaFallbackTotal

            self.interfaces[ intf ] = interfaceHostStats

   def getTable( self, columns ):
      widths = OrderedDict( columns )
      table = createTable( widths )
      columns = []
      for label, width in widths.items():
         justification = "left" if label == "Interface" else "right"
         formatColumn = Format( justify=justification,
                                minWidth=width )
         if justification == "left":
            formatColumn.noPadLeftIs( True )
         formatColumn.padLimitIs( True )
         columns.append( formatColumn )
      table.formatColumns( *columns )
      return table

   def renderInterface( self, successColumns, failedColumns, fallbackColumns ):
      for intf, stats in self.interfaces.items():
         if not stats.dot1xEnabled:
            print( f"Dot1x is not enabled on { intf }", end="\n\n" )
            continue

         portControl = portControlEnum2Str[ stats.portControl ]
         if portControl != "auto":
            print( f"Dot1x port control on { intf }: { portControl }", end="\n\n" )
            continue

         successTable = self.getTable( successColumns )
         failedTable = self.getTable( failedColumns )
         fallbackTable = self.getTable( fallbackColumns )

         eapolString = \
               f"{ stats.hostStats.eapolSuccess }( { stats.hostStats.eapolCached } )"
         mbaString = \
               f"{ stats.hostStats.mbaSuccess }( { stats.hostStats.mbaCached } )"

         aaaUnrespString = ( f"{ stats.aaaUnrespCount }" if stats.aaaUnrespConfigured
               else "n/a" )
         aaaUnrespPhoneString = ( f"{ stats.aaaUnrespPhoneCount }"
               if stats.aaaUnrespPhoneConfigured else "n/a" )
         mbaFallbackString = ( f"{ stats.eapolToMbaFallbackCount }"
               if stats.mbaFallbackConfigured else "n/a" )
         authFailString = ( f"{ stats.authFailCount }"
               if stats.authFailConfigured else "n/a" )
         guestString = ( f"{ stats.guestCount }" if stats.guestVlanConfigured
               else "n/a" )

         successTable.newRow( eapolString, mbaString )
         failedTable.newRow( str( stats.hostStats.eapolFailed ),
               str( stats.hostStats.mbaFailed ) )
         fallbackTable.newRow( aaaUnrespString, aaaUnrespPhoneString,
               mbaFallbackString, authFailString, guestString )

         totalSuccess = stats.hostStats.eapolSuccess + stats.hostStats.mbaSuccess
         totalFailed = stats.hostStats.eapolFailed + stats.hostStats.mbaFailed
         totalCached = stats.hostStats.eapolCached + stats.hostStats.mbaCached
         totalFallback = stats.hostStats.eapolFallbackTotal + \
               stats.hostStats.mbaFallbackTotal

         print( f"Interface { intf } statistics:" )
         print( f"Total number of supplicants: { stats.hostStats.totalSupplicant }" )
         print( f"Total number of web auth supplicants: "
               f"{ stats.hostStats.webAuth }" )
         print( f"Total number of cached supplicants: { totalCached }",
               end="\n\n" )

         print( f"Successful supplicants: { totalSuccess }" )
         print( successTable.output() )

         print( f"Failed supplicants: { totalFailed }" )
         print( failedTable.output() )

         print( f"Supplicants in fallback: { totalFallback }" )
         print( f"EAPOL supplicants: { stats.hostStats.eapolFallbackTotal }" )
         print( f"MBA supplicants: { stats.hostStats.mbaFallbackTotal }" )
         print( fallbackTable.output() )

   def renderVlan( self, vlanColumns ):
      for vlan, stats in self.vlans.items():
         vlanTable = self.getTable( vlanColumns )
         totalCount = stats.eapol + stats.mba
         vlanTable.newRow( stats.eapol, stats.mba, totalCount )
         print( f"VLAN { vlan } statistics:" )
         print( vlanTable.output() )

   def render( self ):
      if not self.systemAuthControl:
         print( "Dot1x system authentication control: disabled" )
         return

      successColumns = [ ( "Interface", 15 ), ( "EAPOL( cached )", 16 ),
         ( "MBA( cached )", 16 ) ]
      failedColumns = [ ( "Interface", 15 ), ( "EAPOL", 7 ), ( "MBA", 7 ) ]
      fallbackColumns = [ ( "Interface", 15 ), ( "AAA-UNRESPONSIVE", 16 ),
         ( "AAA-UNRESPONSIVE-PHONE", 22 ), ( "MBA", 7 ), ( "AUTH-FAIL", 9 ),
         ( "GUEST", 7 ) ]
      vlanColumns = [ ( "VLAN", 4 ), ( "EAPOL", 7 ), ( "MBA", 7 ), ( "Total", 8 ) ]

      if self.vlanFiltered_:
         self.renderVlan( vlanColumns[ 1 : ] )
         return

      if self.interfaceFiltered_:
         self.renderInterface( successColumns[ 1 : ], failedColumns[ 1 : ],
               fallbackColumns[ 1 : ] )
         return

      successTable = self.getTable( successColumns )
      failedTable = self.getTable( failedColumns )
      fallbackTable = self.getTable( fallbackColumns )
      vlanTable = self.getTable( vlanColumns )

      for intf, stats in self.interfaces.items():
         eapolString = \
               f"{ stats.hostStats.eapolSuccess }( { stats.hostStats.eapolCached } )"
         mbaString = \
               f"{ stats.hostStats.mbaSuccess }( { stats.hostStats.mbaCached } )"

         aaaUnrespString = ( f"{ stats.aaaUnrespCount }" if stats.aaaUnrespConfigured
               else "n/a" )
         aaaUnrespPhoneString = ( f"{ stats.aaaUnrespPhoneCount }"
               if stats.aaaUnrespPhoneConfigured else "n/a" )
         mbaFallbackString = ( f"{ stats.eapolToMbaFallbackCount }"
               if stats.mbaFallbackConfigured else "n/a" )
         authFailString = ( f"{ stats.authFailCount }"
               if stats.authFailConfigured else "n/a" )
         guestString = ( f"{ stats.guestCount }" if stats.guestVlanConfigured
               else "n/a" )

         successTable.newRow( intf, eapolString, mbaString )
         failedTable.newRow( intf, str( stats.hostStats.eapolFailed ),
               str( stats.hostStats.mbaFailed ) )
         fallbackTable.newRow( intf, aaaUnrespString, aaaUnrespPhoneString,
               mbaFallbackString, authFailString, guestString )

      for vlan, stats in self.vlans.items():
         totalCount = stats.eapol + stats.mba
         vlanTable.newRow( vlan, stats.eapol, stats.mba, totalCount )

      totalSuccess = self.hostStats.eapolSuccess + self.hostStats.mbaSuccess
      totalFailed = self.hostStats.eapolFailed + self.hostStats.mbaFailed
      totalCached = self.hostStats.eapolCached + self.hostStats.mbaCached
      totalFallback = self.hostStats.eapolFallbackTotal + \
            self.hostStats.mbaFallbackTotal
      
      print( f"Total number of supplicants: { self.hostStats.totalSupplicant }" )
      print( f"Total number of web auth supplicants: { self.hostStats.webAuth }" )
      print( f"Total number of cached supplicants: { totalCached }",
            end="\n\n" )

      print( f"Successful supplicants: { totalSuccess }" )
      print( f"EAPOL supplicants: { self.hostStats.eapolSuccess }" )
      print( f"MBA supplicants: { self.hostStats.mbaSuccess }" )
      print( successTable.output() )

      print( f"Failed supplicants: { totalFailed }" )
      print( f"EAPOL supplicants: { self.hostStats.eapolFailed }" )
      print( f"MBA supplicants: { self.hostStats.mbaFailed }" )
      print( failedTable.output() )

      print( f"Supplicants in fallback: { totalFallback }" )
      print( f"EAPOL supplicants: { self.hostStats.eapolFallbackTotal }" )
      print( f"MBA supplicants: { self.hostStats.mbaFallbackTotal }" )
      print( fallbackTable.output() )

      print( "Supplicants per VLAN:" )
      print( vlanTable.output() )


class Dot1xHost( Model ):
   supplicantMac = MacAddress( help="supplicant MAC address" )
   username = Str( help="EAP identity in use on this interface" )
   authMethod = Str( help="Authentication method used by the supplicant" )
   authStage = Str( help="Supplicant state" )
   fallback = Str( help="Fallback applied" )
   vlanId = Str( help='VLAN ID' )
   vlanName = Str( help="VLAN Name", optional=True )
   cachedAuthAtLinkDown = Bool( help="Cache supplicant on link down" )
   reauthTimeoutSeen = Bool( help="Reauth timeout seen" )
   sessionCached = Bool( help="Supplicant software-cached", default=False )
   staticVlan = Bool( help="Statically configured VLAN", default=False )
   internalVlan = Bool( help="Internally configured VLAN", default=False )

   def fromTacc( self, supplicantStatus ):
      self.supplicantMac = supplicantStatus.mac
      self.username = supplicantStatus.identity
      self.cachedAuthAtLinkDown = supplicantStatus.cachedAuthAtLinkDown
      self.reauthTimeoutSeen = supplicantStatus.reauthTimeoutSeen
      self.sessionCached = supplicantStatus.sessionCached
      self.authMethod = 'EAPOL'
      if supplicantStatus.forceAuthPhoneApplied:
         self.authMethod = "AUTH-PHONE"
      elif supplicantStatus.mbaSupplicant:
         self.authMethod = 'MAC-BASED-AUTH'
      self.vlanId = supplicantStatus.vlan
      self.authStage = authStageDict[ supplicantStatus.authStage ]
      if self.sessionCached:
         self.authStage = "SOFTWARE-CACHED"
      elif self.cachedAuthAtLinkDown:
         self.authStage = "DISCONNECTED-CACHED"
      elif self.reauthTimeoutSeen:
         self.authStage = "SUCCESS-CACHED"
      self.fallback = supplicantStatus.fallbackVlan()
      self.fallback = shortenFallback[ self.fallback ]
      if self.fallback == 'AUTH-FAIL-VLAN' and supplicantStatus.inAuthFailAcl:
         self.fallback = 'AUTH-FAIL-ACL'

      # 'MBA' Fallback should show up only if no other fallback is applied
      if supplicantStatus.authFallBack and self.fallback == "NONE":
         self.fallback = "MBA"

   def getAuthStageDetails( self ):
      macString = convertMacAddrToDisplay( self.supplicantMac.stringValue )

      authMethod = shortenAuthMethod[ self.authMethod ]

      authStage = self.authStage

      for state in cliStageShorten: # pylint: disable=consider-using-dict-items
         authStage = authStage.replace( state, cliStageShorten[ state ] )

      vlanId = 'none' if self.vlanId == 'None' else self.vlanId

      if self.internalVlan:
         vlanId += '+'
      elif self.staticVlan:
         vlanId += "*"

      vlanName = self.vlanName if self.vlanName else ""

      return ( macString, self.username, authMethod, authStage, self.fallback,
            vlanId, vlanName )

class Dot1xHosts( Model ):
   supplicants = Dict( help="A mapping betweeen MAC addresses and Dot1X or MBA "
                            "supplicant information", keyType=MacAddress,
                            valueType=Dot1xHost )

   def populate( self, dot1xHostsTable, intf ):
      if not self.supplicants:
         return

      for supplicant in self.supplicants:
         ( macString, username, authMethod, authStage, fallback, vlanId,
               vlanName ) = self.supplicants[ supplicant ].getAuthStageDetails()
         dot1xHostsTable.newRow( intf, macString, username, authMethod, authStage,
               fallback, vlanId, vlanName )


class Dot1xAllHosts( Model ):
   intfSupplicantsDict = Dict( help="A mapping between the interfaces and all the "
                                    "MBA or Dot1X supplicants for that interface",
                               keyType=Interface, valueType=Dot1xHosts )
   def render( self ):
      if not self.intfSupplicantsDict:
         return

      widths = OrderedDict( [
         ( 'Port', 10 ),
         ( 'Supplicant MAC', 15 ),
         ( 'Username', 30 ),
         ( 'Auth', 6 ),
         ( 'State', 24 ),
         ( 'Fallback', 23 ),
         ( 'VLAN', 6 ),
         ( 'VLAN Name', 10 ) ] )

      dot1xHostsTable = createTable( widths )
      columns = []
      for column in widths:
         formatColumn = Format( justify='left', minWidth=widths[ column ],
               maxWidth=widths[ column ], wrap=True )
         formatColumn.noPadLeftIs( True )
         formatColumn.padLimitIs( True )
         formatColumn.noTrailingSpaceIs( True )
         columns.append( formatColumn )
      dot1xHostsTable.formatColumns( *columns )

      for key in Arnet.sortIntf( self.intfSupplicantsDict ):
         self.intfSupplicantsDict[ key ].populate( dot1xHostsTable,
               IntfMode.getShortname( key ) )
      print( "Legend:" )
      print( "* - Statically configured VLAN" )
      print( "+ - Internally configured VLAN" )
      print( dot1xHostsTable.output() )

class Dot1xDeadTimers( Model ):
   hosts = Dict( help="A mapping of hostname with the timestamp when it is /"
                      "will be marked as failed", keyType=str, valueType=float )
   radiusConfigDeadTime = Float( help="Configured radius deadtime" )

   def render( self ):
      if not self.hosts:
         return
      widths = OrderedDict( [
         ( 'RADIUS Server', 50 ), ( 'Status', 13 ), ( 'Time', 11 ) ] )
      alignment = [ 'left', 'left', 'right' ]
      dot1xDeadTimersTable = createTable( widths )
      columns = []
      for column, justify in zip( widths, alignment ):
         formatColumn = Format( justify=justify, minWidth=widths[ column ],
                                maxWidth=widths[ column ], wrap=True )
         formatColumn.noPadLeftIs( True )
         formatColumn.padLimitIs( True )
         columns.append( formatColumn )
         dot1xDeadTimersTable.formatColumns( *columns )

      def formatTime( t ):
         return f'{t:.0f} sec'

      now = float( Tac.now() )
      for host, t in sorted( self.hosts.items() ):
         if t > now:
            # Let's say we have a server going down at T=10 with a configured dead
            # time of 1 min (60 seconds), then this command should show the server
            # status as "failed for" until T=70 seconds, viz 't' in this case.
            # So at time T=20 ==> "failed for 10 seconds"
            #            T=30 ==> "failed for 20 seconds"
            time = formatTime( now - ( t - self.radiusConfigDeadTime ) )
            status = 'failed for'
         else:
            time = 'n/a'
            status = 'active'
         dot1xDeadTimersTable.newRow( host, status, time )
      print( dot1xDeadTimersTable.output() )

class Dot1xBlockedMacEntry( Model ):
   interfaces = Dict( help="A mapping between interface and VLAN",
                      keyType=Interface, valueType=int )
   def renderEntry( self, table, macAddr ):
      first = True
      for intf in Arnet.sortIntf( self.interfaces ):
         if first:
            first = False
            table.startRow()
            table.newRow( convertMacAddrToDisplay( macAddr ), intf,
                          self.interfaces[ intf ] )
         else:
            table.newRow( "", intf, self.interfaces[ intf ] )

class Dot1xBlockedMacTable( Model ):
   macAddresses = Dict( help="A mapping between blocked MAC address and list of "
                        "intferace and VLAN pair on which the MAC is blocked"
                        " through dynamic authorization",
                        keyType=MacAddress, valueType=Dot1xBlockedMacEntry )

   def render( self ):
      headings = ( "MAC Address", "Interface", "VLAN ID" )
      table = createTable( headings )
      for macAddr in sorted( self.macAddresses ):
         self.macAddresses[ macAddr ].renderEntry( table, macAddr )
      print( table.output() )

class Dot1xVlanAssignmentGroupNames( Model ):
   vlanGroupNameDict = Dict( help="A mapping of group name to that group's VLANs",
                         keyType=str, valueType=str )
   def render( self ):
      if not self.vlanGroupNameDict:
         return
      fields = [ 'VLAN Assignment Group', 'VLANs' ]
      fmt = "%-33s%s"
      print( fmt % ( fields[ 0 ], fields[ 1 ] ) )
      print( fmt % ( len( fields[ 0 ] ) * "-", len( fields[ 1 ] ) * "-" ) )
      for key in sorted( self.vlanGroupNameDict ):
         print( fmt % ( key, self.vlanGroupNameDict[ key ] ) )

class Dot1xCaptivePortalBypassIpMatch( Model ):
   addresses = List( valueType=IpGenericAddress,
                     help="Addresses matching the FQDN" )

class Dot1xCaptivePortalBypass( Model ):
   fqdns = Dict( keyType=str,
                 valueType=Dot1xCaptivePortalBypassIpMatch,
                 help="A mapping of the configured bypass FQDNs to the matching "
                 "addresses" )

   def render( self ):
      first = True
      for fqdn in sorted( self.fqdns ):
         ipmatch = self.fqdns.get( fqdn )
         if not ipmatch:
            continue
         if first:
            print( "Captive portal bypass:" )
         print( textwrap.fill( f"{fqdn} matched by " +
                               ', '.join( str( ip ) for ip in ipmatch.addresses ),
                               width=79, subsequent_indent='  ' ) )
         first = False

# -------------------------------------------------------------------------

class Dot1xCaptivePortalResolutionsEntry( Model ):
   ready = Bool( help="Resolution result is ready" )
   hostnames = List( valueType=str, help="Hostnames returned by DNS" )
   expiration = Float( help="Expiration time of the hostnames", optional=True )

   def fromTacc( self, reverseDnsEntry ):
      self.ready = reverseDnsEntry.resolution.ready
      self.hostnames = sorted( list( reverseDnsEntry.resolution.hostname ) )
      if self.ready:
         self.expiration = Ark.switchTimeToUtc( reverseDnsEntry.validUntil )

class Dot1xCaptivePortalResolutions( Model ):
   addresses = Dict( keyType=IpGenericAddress,
                     valueType=Dot1xCaptivePortalResolutionsEntry,
                     help="A mapping of addresses to resolutions" )

   def fromTacc( self, reverseDnsEntries ):
      for addr, reverseDnsEntry in reverseDnsEntries.entry.items():
         modelEntry = Dot1xCaptivePortalResolutionsEntry()
         modelEntry.fromTacc( reverseDnsEntry )
         self.addresses[ addr ] = modelEntry

   def render( self ):
      table = createTable( ( 'Address', 'Hostnames', 'Expiration' ) )
      table.formatColumns( Format( justify='left' ),
                           Format( justify='left' ),
                           Format( justify='right' ) )
      ips = [ Arnet.IpGenAddr( ip ) for ip in self.addresses ]
      ips.sort()
      for ipAddr in ips:
         ip = str( ipAddr )
         entry = self.addresses.get( ip )
         if not entry.ready:
            table.newRow( str( ip ), '(pending)', '' )
         elif entry.hostnames:
            hostnames = sorted( entry.hostnames )
            timestamp = Ark.utcTimestampToStr( entry.expiration )
            table.newRow( str( ip ), hostnames[ 0 ], timestamp )
            if len( hostnames ) > 1:
               for hostname in hostnames[ 1 : ]:
                  table.newRow( '', hostname, '' )
         else:
            timestamp = Ark.utcTimestampToStr( entry.expiration )
            table.newRow( str( ip ), '-', timestamp )
      print( table.output() )

# --------------------------------------------------------------------------------
# Captive Portal Counter Models
# --------------------------------------------------------------------------------

class CaptivePortalCountersInterface( Model ):
   httpRequest = Int( help="HTTP requests to dot1x web server" )
   httpRedirect = Int( help="HTTP redirections to captive portal" )
   httpInvalid = Int( help="HTTP errors/dropped/timeouts" )
   httpsRequest = Int( help="HTTPS requests to dot1x web server" )
   httpsRedirect = Int( help="HTTPS redirections to captive portal" )
   httpsInvalid = Int( help="HTTPS errors/dropped/timeouts" )
   lastClearTime = Float( help="Last cleared time for counters" )

   def fromTacc( self, intfCaptivePortalCounters ):
      self.httpRequest = intfCaptivePortalCounters.httpRequest
      self.httpRedirect = intfCaptivePortalCounters.httpRedirect
      self.httpInvalid = intfCaptivePortalCounters.httpInvalid
      self.httpsRequest = intfCaptivePortalCounters.httpsRequest
      self.httpsRedirect = intfCaptivePortalCounters.httpsRedirect
      self.httpsInvalid = intfCaptivePortalCounters.httpsInvalid
      self.lastClearTime = intfCaptivePortalCounters.lastClearTime

   def populateTable( self, table ):
      '''
      Populates the table for the counter data of a given interface 
      Each table has two rows (HTTP and HTTPS)
      '''
      # Adding HTTP Counter data row to the table
      table.newRow( 'HTTP', self.httpRequest, self.httpRedirect, self.httpInvalid )
      # Adding HTTPS Counter data row to the table
      table.newRow( 'HTTPS', self.httpsRequest, self.httpsRedirect,
                    self.httpsInvalid )

   def render( self ):
      '''
      creates, populates and prints the counter data table
      '''
      widths = OrderedDict( [
         ( 'Request Type', [ 12, 'left' ] ),
         ( 'Requests', [ 12, 'right' ] ),
         ( 'Redirects', [ 12, 'right' ] ),
         ( 'Invalids', [ 12, 'right' ] ) ] )

      captivePortalCounterTable = createTable( widths )
      columns = []
      for column in widths:
         formatColumn = Format( justify=widths[ column ][ 1 ],
                                minWidth=widths[ column ][ 0 ] )
         if widths[ column ][ 1 ] == 'left':
            formatColumn.noPadLeftIs( True )
         formatColumn.padLimitIs( True )
         columns.append( formatColumn )
      captivePortalCounterTable.formatColumns( *columns )

      self.populateTable( captivePortalCounterTable )
      print( captivePortalCounterTable.output(), end="" )
      utcTimeRelativeToNow = utcTimeRelativeToNowStr( self.lastClearTime )
      print( "Last cleared:", utcTimeRelativeToNow, end="\n\n" )

class Dot1xCaptivePortalCounters( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=CaptivePortalCountersInterface,
                      help="A mapping from interface to its counters", )

   def render( self ):
      for intf in Arnet.sortIntf( self.interfaces ):
         print( 'Interface:', intf )
         self.interfaces[ intf ].render()
