# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliMode.Dot1x import Dot1xSuppProfileBaseMode
from CliPlugin import Dot1xModel
from CliPlugin import IntfCli
from CliPlugin import Ssl
from CliPlugin.Dot1xModeCli import Dot1xMode
import BasicCli, Cell, ConfigMount, LazyMount
import CliCommand
import ShowCommand
import CliParser
import CliMatcher
import ReversibleSecretCli
from Intf.IntfRange import IntfRangeMatcher
from IntfRangePlugin.EthIntf import EthPhyAutoIntfType
from IntfRangePlugin.SwitchIntf import SwitchAutoIntfType
import Tac
import TacSigint
import Dot1xLib

config = None
status = None
hwstatus = None
wpaSupplicantStatus = None

# Dot1x Supplicant profile mode
class Dot1xSuppProfileMode( Dot1xSuppProfileBaseMode, BasicCli.ConfigModeBase ):
   name = 'Dot1x supplicant profile configuration'

   def __init__( self, parent, session, profileName ):
      self.profileName = profileName
      Dot1xSuppProfileBaseMode.__init__( self, profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.updateInterfaceList()

   def profile( self ):
      return config.supplicantProfile[ self.profileName ]

   def updateInterfaceList( self ):
      # Walk through all interfaces which have this profile enabled and update the
      # intf collection.
      profile = self.profile()
      for intf in config.supplicantIntfConfig.values():
         if intf.profileName == self.profileName:
            profile.intf.add( intf.intfId )

   def setIdentity( self, identity ):
      if identity and len( identity ) > Dot1xLib.MAX_SUPPLICANT_ID_SIZE:
         # pylint: disable-next=consider-using-f-string
         self.addError( "Maximum identity size is %d" % \
               Dot1xLib.MAX_SUPPLICANT_ID_SIZE )
         return

      # Update EAP identity in the profile.
      self.profile().identity = identity

   def noIdentity( self ):
      # Delete identity
      self.profile().identity = ""

   def setPassphrase( self, key ):
      if key and key.clearTextSize() > Dot1xLib.MAX_SUPPLICANT_KEY_SIZE:
         # pylint: disable-next=consider-using-f-string
         self.addError( "Maximum key size is %d" % Dot1xLib.MAX_SUPPLICANT_KEY_SIZE )
         return

      # Set EAP passphrase
      self.profile().encryptedPassword = key

   def noPassphrase( self ):
      # Delete password
      self.profile().encryptedPassword = ReversibleSecretCli.getDefaultSecret()

   def setEapMethod( self, eapMethod ):
      self.profile().eapMethod = eapMethod

   def noEapMethod( self ):
      self.profile().eapMethod = self.profile().eapMethodDefault

   def setSslProfile( self, sslProfileName ):
      self.profile().sslProfileName = sslProfileName or ''

#-------------------------------------------------------------------------------
# [no|default] profile <profileName>
#-------------------------------------------------------------------------------
matcherSupplicant = CliMatcher.KeywordMatcher( 'supplicant', 
                                 helpdesc='Dot1x supplicant commands' )

def getProfileNames( mode=None ):
   return sorted( config.supplicantProfile.members() )

profileExpression = CliMatcher.DynamicNameMatcher( getProfileNames, 
                                                   'Profile name' )

def gotoDot1xSuppProfileMode( mode, args ):
   # Create a new profile object if it doesnt exist.
   profileName = args[ 'PROFILE' ]
   config.supplicantProfile.newMember( profileName )
   # Transition into the new mode.
   childMode = mode.childMode( Dot1xSuppProfileMode, profileName=profileName )
   mode.session_.gotoChildMode( childMode )

def noDot1xSuppProfileMode( mode, args ):
   if profile := args.get( 'PROFILE' ):
      del config.supplicantProfile[ profile ]
   else:
      # We got called from `Dot1xMode.clear`; no args.
      config.supplicantProfile.clear()

class SupplicantProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'supplicant profile PROFILE'
   noOrDefaultSyntax = 'supplicant profile PROFILE ...'
   data = {
      'supplicant': 'Dot1x supplicant commands',
      'profile': 'Dot1x supplicant profile commands',
      'PROFILE': profileExpression,
   }
   handler = gotoDot1xSuppProfileMode
   noOrDefaultHandler = noDot1xSuppProfileMode

Dot1xMode.addCommandClass( SupplicantProfileCmd )

# Dot1x profile specific commands.

#--------------------------------------------------------------------------------
# [no|default] eap-method fast|tls
#--------------------------------------------------------------------------------
class EapMethodCmd( CliCommand.CliCommandClass ):
   syntax = 'eap-method fast|tls'
   noOrDefaultSyntax = 'eap-method ...'
   data = {
      'eap-method': 'Extensible Authentication Protocol (EAP) method',
      'fast': 'EAP Flexible Authentication via Secure Tunneling (FAST)',
      'tls': 'EAP with Transport Layer Security (TLS)',
   }

   @staticmethod
   def handler( mode, args ):
      if 'fast' in args:
         mode.setEapMethod( args[ 'fast' ] )
      elif 'tls' in args:
         mode.setEapMethod( args[ 'tls' ] )
   
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noEapMethod()

Dot1xSuppProfileMode.addCommandClass( EapMethodCmd )

#--------------------------------------------------------------------------------
# [no|default] identity IDENTITY
#--------------------------------------------------------------------------------
class IdentityCmd( CliCommand.CliCommandClass ):
   syntax = 'identity IDENTITY'
   noOrDefaultSyntax = 'identity ...'
   data = {
      'identity': 'Extensible Authentication Protocol (EAP) user identity',
      'IDENTITY': CliMatcher.PatternMatcher( pattern=r'(.+)',
                           helpdesc='user identity', helpname='WORD' ),
   }
   
   @staticmethod
   def handler( mode, args ):
      mode.setIdentity( args[ 'IDENTITY' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noIdentity()

Dot1xSuppProfileMode.addCommandClass( IdentityCmd )

class PassphraseCommand( CliCommand.CliCommandClass ):
   syntax = "passphrase <PASSWORD>"
   noOrDefaultSyntax = "passphrase ..."
   data = {
      'passphrase' : 'Extensible Authentication Protocol (EAP) password',
      '<PASSWORD>': ReversibleSecretCli.defaultReversiblePwdCliExpr,
   }
   @staticmethod
   def handler( mode, args ):
      mode.setPassphrase( args[ '<PASSWORD>' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noPassphrase()

Dot1xSuppProfileMode.addCommandClass( PassphraseCommand )

#--------------------------------------------------------------------------------
# [no|default] ssl profile <profile>
#--------------------------------------------------------------------------------
class SslProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'ssl profile PROFILE_NAME'
   noOrDefaultSyntax = 'ssl profile ...'
   data = {
            'ssl': Ssl.sslMatcher,
            'profile': Ssl.profileMatcher,
            'PROFILE_NAME': Ssl.profileNameMatcher
          }

   @staticmethod
   def handler( mode, args ):
      mode.setSslProfile( args[ 'PROFILE_NAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.setSslProfile( None )

Dot1xSuppProfileMode.addCommandClass( SslProfileCmd )

#-------------------------------------------------------------------------------
# [no|default] dot1x pae supplicant <profileName>
#-------------------------------------------------------------------------------

def setDot1xSupplicant( mode, profileName ):
   # If an intfConfig does not exist yet, create it now.
   intfId = mode.intf.name
   newProfile = True

   if intfId not in config.supplicantIntfConfig:
      intfConfig = config.supplicantIntfConfig.newMember( intfId )
   else:
      intfConfig = config.supplicantIntfConfig[ intfId ]

   # Get the associated profile.
   if profileName not in config.supplicantProfile:
      # pylint: disable-next=consider-using-f-string
      mode.addWarning( 'Dot1x supplicant profile %s does not exist.' % profileName )
      newProfile = False

   # If profile associated with the interface has not changed then there is nothing
   # to do.
   if intfConfig.profileName != profileName:
      # Disassociate the interface from its old profile.
      if intfConfig.profileName in config.supplicantProfile:
         del config.supplicantProfile[ intfConfig.profileName ].intf[ intfId ]

      # Associate the interface with its new profile.
      if newProfile:
         config.supplicantProfile[ profileName ].intf.add( intfId )

      # Update the interface.
      intfConfig.profileName = profileName

def delIntfConfig( intfId ):
   # Get the intfConfig object.
   if intfId in config.supplicantIntfConfig:
      intfConfig = config.supplicantIntfConfig[ intfId ]
   else:
      # Nothing to do.
      return
   profileName = intfConfig.profileName
   if not profileName:
      # Nothing to do.
      return
   if profileName in config.supplicantProfile:
      del config.supplicantProfile[ profileName ].intf[ intfId ]

   # Delete the interface config.
   del config.supplicantIntfConfig[ intfId ]

def noDot1xSupplicant( mode ):
   # Disassociate the interface from its current profile.
   intfId = mode.intf.name

   delIntfConfig( intfId )

#-------------------------------------------------------------------------------
# Cleanup per-interface configuration
#-------------------------------------------------------------------------------
class IntfDot1xSuppConfigCleaner( IntfCli.IntfDependentBase ):
   """This class is responsible for removing per-interface Dot1x supplicant config
   when the interface is deleted."""
   def setDefault( self ):
      delIntfConfig( self.intf_.name )

#-------------------------------------------------------------------------------
# The "show dot1x supplicant [ intf <intfName> ]" command
#-------------------------------------------------------------------------------
def showSupplicant( mode, args ):
   intfs = args.get( 'INTFS', status.supplicantIntfStatus )
   model = Dot1xModel.Dot1xSupplicants()
   for intf in intfs:
      supplicantStatus = status.supplicantIntfStatus.get( intf )
      wpaStatus = wpaSupplicantStatus.wpaSupplicantIntfStatus.get( intf )
      if supplicantStatus and wpaStatus:
         dot1xSupplicant = Dot1xModel.Dot1xSupplicant()
         dot1xSupplicant.fromTacc( supplicantStatus, wpaStatus )
         model.interfaces[ intf ] = dot1xSupplicant
      TacSigint.check()

   return model

def dot1xSupportedGuard( mode, token ):
   if hwstatus.dot1xSupported:
      return None
   return  CliParser.guardNotThisPlatform

matcherDot1x = CliMatcher.KeywordMatcher( 'dot1x',
               helpdesc='IEEE 802.1X port authentication' )
nodeDot1x = CliCommand.Node( matcherDot1x, guard=dot1xSupportedGuard )

intfRangeMatcher = IntfRangeMatcher( explicitIntfTypes=( EthPhyAutoIntfType,
                                                         SwitchAutoIntfType ) )

class Dot1XSupplicantCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x supplicant [ INTFS ]'
   data = {
      'dot1x': nodeDot1x,
      'supplicant': 'Show 802.1X supplicant information summary',
      'INTFS': intfRangeMatcher,
   }
   handler = showSupplicant
   cliModel = Dot1xModel.Dot1xSupplicants

BasicCli.addShowCommandClass( Dot1XSupplicantCmd )
#------------------------------------------------------------
# [no|default] dot1x supplicant logging
#------------------------------------------------------------
def enableSupplicantLogging( mode, args ):
   config.supplicantLoggingEnabled = True

def disableSupplicantLogging( mode, args ):
   config.supplicantLoggingEnabled = False

class SupplicantLoggingCmd( CliCommand.CliCommandClass ):
   syntax = 'supplicant logging'
   noOrDefaultSyntax = 'supplicant logging ...'
   data = {
      'supplicant': matcherSupplicant,
      'logging': 'Enable supplicant logging',
   }
   handler = enableSupplicantLogging
   noOrDefaultHandler = disableSupplicantLogging

Dot1xMode.addCommandClass( SupplicantLoggingCmd )

#------------------------------------------------------------
# [no|default]
#  dot1x supplicant supplicant disconnect cached-results timeout <secs> seconds
#------------------------------------------------------------
def supplicantDisconnectTimeout( mode, args ):
   timeout = args[ 'TIMEOUT' ]
   config.disconnectTimeout = timeout

def noSupplicantDisconnectTimeout( mode, args ):
   config.disconnectTimeout = config.disconnectTimeoutDefault

matcherTimeoutPeriod = CliMatcher.IntegerMatcher( 60, 65535,
      helpdesc='Timeout period in seconds' )

class SupplicantDisconnectTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'supplicant disconnect cached-results timeout TIMEOUT seconds'
   noOrDefaultSyntax = 'supplicant disconnect cached-results timeout ...'
   data = {
      'supplicant': matcherSupplicant,
      'disconnect': 'Configure supplicant disconnect behavior',
      'cached-results': 'Configure cached supplicant disconnect behavior',
      'timeout': 'Timeout for removing a disconnected supplicant',
      'TIMEOUT': matcherTimeoutPeriod,
      'seconds': 'Timeout period in seconds',
   }
   handler = supplicantDisconnectTimeout
   noOrDefaultHandler = noSupplicantDisconnectTimeout

Dot1xMode.addCommandClass( SupplicantDisconnectTimeoutCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global config
   global status
   global hwstatus
   global wpaSupplicantStatus

   config = ConfigMount.mount( entityManager, "dot1x/config",
                               "Dot1x::Config", "w" )
   status = LazyMount.mount( entityManager, 
                             Cell.path( "dot1x/supplicantStatus/dot1x" ),
                             "Dot1x::SupplicantStatus", "r" )
   wpaSupplicantStatus = \
         LazyMount.mount( entityManager, 
                          Cell.path( "dot1x/supplicantStatus/superserver" ), 
                          "Dot1x::SuperServerSupplicantStatus", "r" )
   hwstatus = LazyMount.mount( entityManager, "dot1x/hwstatus",
                               "Dot1x::HwStatus", "r" )

   # register interface-deletion handler
   IntfCli.Intf.registerDependentClass( IntfDot1xSuppConfigCleaner, priority=10 )
