# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

roleMap = {
   "pllTimeSync": "timeSync",
   "pllTimeSyncStandalone": "timeSyncFreeRun",
   "pllPhy": "phy",
   "pllServo": "servo",
   "pllSystem": "system",
}

stateMap = {
   "pllStateUnknown": "disabled",
   "pllStateHwNotPresent": "disabled",
   "pllStateInitializing": "initializing",
   "pllStateFailed": "failed",
   "pllStateLocking": "initializing",
   "pllStateFreeRunning": "freeRunning",
   "pllStateClkInput": "active",
   "pllStateClkInputWithHoldover": "active",
}

moduleOrder = {
   "Supervisor": "0",
   "Fabric": "1",
   "Linecard": "2",
}

def updatePllResetConfig( allPllConfig, resetConfig ):
   r"""
   Writes the reset config in a way that the agent understands
   the root pllConfig path might look like this:
   |
   \- Lmk05318
      \- Linecard1/0
      \- Linecard2/0
      \- Linecard3/0
   \- Lmk5b33216
   \- Rc32312
      \- Supervisor1/0
   this function will update the reset config for each PLL in each "model type"
   """
   for pllModel in allPllConfig:
      if pllModel not in resetConfig:
         resetConfig.newEntity( "Hardware::Dpll::ResetConfigDir", pllModel )

      thisPllType = allPllConfig.entity[ pllModel ]
      for pllName in resetConfig.entity[ pllModel ].pll:
         if pllName not in thisPllType.pllConfig:
            del resetConfig.pll[ pllName ]
      for pllName in thisPllType.pllConfig:
         if pllName not in resetConfig.entity[ pllModel ].pll:
            resetConfig.entity[ pllModel ].newPll( pllName )

def clockNames( allPllConfig ):
   res = set()
   for pllModel in allPllConfig:
      thisPllType = allPllConfig.entity[ pllModel ]
      for pll in thisPllType.pllConfig.values():
         res.add( pll.name )
   return sorted( res, key=parseClockName )

clockNameRe = re.compile( "^([A-Za-z]+)([0-9]*)/([0-9]+)$" )

def parseClockName( clockName ):
   m = clockNameRe.match( clockName )
   if m:
      module = m.group( 1 )
      module = moduleOrder.get( module, module )
      return ( module, int( m.group( 2 ) or "0" ), int( m.group( 3 ) ) )
   return ( "!" + clockName, 0, 0 )
