# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
from CliPlugin import EthIntfCli
from CliPlugin import SubIntfCli
from CliPlugin import LagIntfCli
from CliPlugin import IpAddrMatcher
from CliPlugin.VrfCli import VrfExprFactory
from CliPlugin.VirtualIntfRule import IntfMatcher
from CliPlugin.WanTEShowCli import pathSelectionSupportedGuard
from CliPlugin.DpsCliLib import ( RouterPathSelectionContext,
                        DpsPathGroupContext,
                        DpsIntfContext,
                        DpsPolicyContext,
                        DpsLoadBalanceProfileContext,
                        DpsVrfConfigContext,
                        DpsEncapConfigContext,
                        DpsPeerDynamicStunConfigContext,
                        DpsPathGroupStunConfigContext,
                        DpsPeerDynamicConfigContext )
import LazyMount
from Toggles.DpsToggleLib import ( toggleFlowAssignmentLanEnabled,
                                   toggleDpsUnderlayIntfInNonDefaultVrfEnabled )
from Toggles.WanTECommonToggleLib import toggleAvtLowestLoadMetricEnabled
from CliMode.Dps import ( RouterPathSelectionConfigMode,
                          DpsPathGroupConfigMode,
                          DpsPathGroupRemoteRouterConfigMode,
                          DpsPolicyConfigMode,
                          DpsPolicyRuleKeyConfigMode,
                          DpsPolicyDefaultRuleConfigMode,
                          DpsLoadBalanceProfileConfigMode,
                          DpsVrfConfigMode,
                          DpsPathGroupStunConfigMode,
                          DpsPathGroupPeerDynamicConfigMode,
                          DpsIntfConfigMode,
                          DEFAULT_PRIORITY,
                          DEFAULT_JITTER, JITTER_SCALE,
                          DEFAULT_LATENCY, LATENCY_SCALE,
                          DEFAULT_LOSSRATE, LOSS_RATE_SCALE, LOSS_RATE_ADJUSTMENT,
                          DEFAULT_UDP_PORT )
from CliToken.Router import routerMatcherForConfig

Constants = Tac.Type( "Dps::DpsConstants" )
constants = Constants()

MAX_RULES = 255

appRecognitionConfig = None
dpsCliConfig = None
dpsStatus = None
entityManager = None
ipsecConfig = None
peerStatus = None
stunServerProfileConfig = None

flowAssignmentLanErrorDynamicPeer = 'Flow assignement LAN can not be'\
   ' configured in a path group with dynamic peer.'
flowAssignmentLanErrorStaticPeer = 'Flow assignement LAN can not be'\
   ' configured in a path group with more than one static peer.'

tokenRuleKey = CliMatcher.IntegerMatcher( 1, MAX_RULES, helpdesc='rule key' )

tokenMilliSecond = CliMatcher.IntegerMatcher( 0, 10000, helpdesc='millisecond' )

tokenPercentage = CliMatcher.FloatMatcher( 0, 100,
                                  helpdesc='percentage',
                                  precisionString='%.2f' )

pathGroupIdMatcher = CliMatcher.IntegerMatcher( 1, 65535, helpdesc='path group id' )

pathSelectionMatcher = CliMatcher.KeywordMatcher( 'path-selection',
                                     helpdesc='Configure dynamic path selection' )
localMatcherForPathGroup = CliMatcher.KeywordMatcher( 'local',
                                 helpdesc='Configure local info for DPS path group' )
pathSelectionNode = CliCommand.Node( matcher=pathSelectionMatcher,
                                     guard=pathSelectionSupportedGuard )
publicKeywordMatcher = CliMatcher.KeywordMatcher( 'public',
                                     helpdesc='Configure public IP assigned by NAT' )
serverProfileMatcher = CliMatcher.KeywordMatcher( 'server-profile',
                                     helpdesc='Configure STUN server profile name' )
priorityForPathGroupMatcher = CliMatcher.IntegerMatcher( 1, 65535,
                                     helpdesc='Priority of this group' )
pathGroupConfigNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [] if dpsCliConfig is None else dpsCliConfig.pathGroupConfig,
      helpdesc='name of the path group',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

policyConfigNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [] if dpsCliConfig is None else dpsCliConfig.policyConfig,
      helpdesc='name of the Dps policy',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

loadBalanceProfileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [] if dpsCliConfig is None else dpsCliConfig.loadBalanceProfile,
      helpdesc='name of the load-balance policy',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

ipsecProfileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [] if ipsecConfig is None else ipsecConfig.ipsecProfile,
      helpdesc='name of the ipsec profile',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

vrfConfigNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [] if dpsCliConfig is None else dpsCliConfig.vrfConfig,
      helpdesc='name of the vrf',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

nameToIdNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [] if appRecognitionConfig is None
            else appRecognitionConfig.appProfile,
      helpdesc='name of the application profile',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

siteNameMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                             helpname='WORD',
                                             helpdesc='name of the site' )

serverProfileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: stunServerProfileConfig.serverProfile,
      helpdesc='name of the STUN server profile',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

matcherTrafficClass = CliMatcher.KeywordMatcher( 'traffic-class',
        'Set traffic class for matched traffic' )
matcherTrafficClassNum = CliMatcher.IntegerMatcher( 0, 7,
        helpdesc='Traffic class value' )

matcherDscp = CliMatcher.KeywordMatcher( 'dscp',
        'Set DSCP for matched traffic' )
matcherDscpValue = CliMatcher.IntegerMatcher( 0, 63,
        helpdesc='DSCP value' )

speedRateMatcher = CliMatcher.IntegerMatcher( 1, 0xFFFFFFFF, helpdesc='Bandwidth' )

pathGroupIntfMatcher = IntfMatcher()
pathGroupIntfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
pathGroupIntfMatcher |= SubIntfCli.subMatcher
pathGroupIntfMatcher |= LagIntfCli.EthLagIntf.matcher
pathGroupIntfMatcher |= LagIntfCli.subMatcher

def getPathGroupNames():
   return dpsCliConfig.pathGroupConfig.iterKeys()

def guardFlowAssignmentLanConfig( mode, token ):
   if not toggleFlowAssignmentLanEnabled():
      return CliParser.guardNotThisEosVersion
   return None

def guardLocalIntfVrfConfig( mode, token ):
   if not toggleDpsUnderlayIntfInNonDefaultVrfEnabled():
      return CliParser.guardNotThisEosVersion
   return None

def dpsPathResponseCheck( mode, response ):
   if response:
      warning = "Same interface or IP address cannot be used in multiple "\
                "path groups"
      mode.addWarning( warning )
      return False
   return True

def dpsStunConfigVerify( mode, context ):
   if context.publicIp:
      warning = "The configured public IP will be overriden by the STUN "\
                "provided IP address."
      mode.addWarning( warning )

class RouterPathSelectionConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''router path-selection'''
   noOrDefaultSyntax = syntax
   data = {
         'router': routerMatcherForConfig,
         'path-selection': pathSelectionNode,
   }

   @staticmethod
   def handler( mode, args ):
      dpsCliConfig.dpsConfigured = True
      context = RouterPathSelectionContext( dpsCliConfig )
      childMode = mode.childMode( RouterPathSelectionConfigMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dpsCliConfig.udpPortConfig = DEFAULT_UDP_PORT
      dpsCliConfig.pathGroupConfig.clear()
      dpsCliConfig.loadBalanceProfile.clear()
      dpsCliConfig.policyConfig.clear()
      dpsCliConfig.vrfConfig.clear()
      dpsCliConfig.tcpMssIngressConfig = constants.tcpMssIngressRewriteDisabled
      dpsCliConfig.mtuDiscInterval = constants.mtuDiscDefaultInterval
      dpsCliConfig.peerDynamicStun = False
      dpsCliConfig.dpsConfigured = False
      dpsCliConfig.icmpFragNeededEnabled = False
      dpsCliConfig.icmpFragNeededRateLimit = constants.icmpFragNeededDefaultRateLimit
      if toggleAvtLowestLoadMetricEnabled():
         dpsCliConfig.intfConfig.clear()

intfSpeedMatcher = IntfMatcher()
intfSpeedMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfSpeedMatcher |= SubIntfCli.subMatcher
intfSpeedMatcher |= LagIntfCli.EthLagIntf.matcher
intfSpeedMatcher |= LagIntfCli.subMatcher

class DpsIntfSpeedConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'interface INTF'
   noOrDefaultSyntax = 'interface INTF'

   data = { 'interface': 'Interface',
            'INTF': intfSpeedMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF' ]
      context = DpsIntfContext( dpsCliConfig, dpsStatus, intf.name )

      context.addOrRemoveIntf( intf.name )
      childMode = mode.childMode( DpsIntfConfigMode, context=context )

      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = args[ 'INTF' ]
      context = DpsIntfContext( dpsCliConfig, dpsStatus, intf.name )
      context.addOrRemoveIntf( intf.name, False )

class DpsIntfTxBandWidthConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'metric bandwidth transmit RATE Mbps'
   noOrDefaultSyntax = 'metric bandwidth transmit ...'

   data = { 'metric': 'Configure metric',
            'bandwidth': 'Configure max bandwidth',
            'transmit': 'Configure max transmit bandwidth',
            'RATE': speedRateMatcher,
            'Mbps': 'Rate unit is Mbps'
   }

   @staticmethod
   def handler( mode, args ):
      txBandwidth = args.get( "RATE", 0 )
      mode.context.setTxBandwidth( txBandwidth )

   noOrDefaultHandler = handler

class DpsIntfRxBandWidthConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'metric bandwidth receive RATE Mbps'
   noOrDefaultSyntax = 'metric bandwidth receive ...'
   data = { 'metric': 'Configure metric',
            'bandwidth': 'Configure max bandwidth',
            'receive': 'Configure max receive bandwidth',
            'RATE': speedRateMatcher,
            'Mbps': 'Rate unit is Mbps'
   }

   @staticmethod
   def handler( mode, args ):
      rxBandwidth = args.get( "RATE", 0 )
      mode.context.setRxBandwidth( rxBandwidth )

   noOrDefaultHandler = handler

class DpsPathGroupConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''path-group <path-group-name> [ id <id> ]'''
   noOrDefaultSyntax = syntax
   data = {
         'path-group': 'Configure path group',
         '<path-group-name>': pathGroupConfigNameMatcher,
         'id': 'Path group id',
         '<id>': pathGroupIdMatcher
   }

   @staticmethod
   def handler( mode, args ):
      # DpsPathGroupConfigCmd.registerNoHandler(
      #  lambda: self.noDpsPathGroupHandler( mode ) )
      name = args.get( '<path-group-name>' )
      pgId = args.get( '<id>', 0 )
      context = DpsPathGroupContext( dpsCliConfig, dpsStatus, peerStatus, name )

      context.addOrRemovePathGroup( name, pgId, True )
      childMode = mode.childMode( DpsPathGroupConfigMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args.get( '<path-group-name>' )
      context = DpsPathGroupContext( dpsCliConfig, dpsStatus, peerStatus, name )

      context.addOrRemovePathGroup( name, 0, False )

class DpsPathGroupIpsecConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''ipsec profile <profile>'''
   noOrDefaultSyntax = '''ipsec profile ...'''

   data = {
         'ipsec': 'Configure ipsec for Path Group',
         'profile': 'Configure ipsec profile for Path Group',
         '<profile>': ipsecProfileNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      profile = args[ '<profile>' ]
      mode.context.addOrRemovePathGroupIpsec( profile, True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.addOrRemovePathGroupIpsec( '', False )

class DpsIntfConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'local interface INTF [ VRF ] [ public address IPADDR ]'
   noOrDefaultSyntax = 'local interface INTF ...'

   data = {
         'local': localMatcherForPathGroup,
         'interface': 'interface',
         'INTF': pathGroupIntfMatcher,
         'VRF': VrfExprFactory( guard=guardLocalIntfVrfConfig,
                                inclDefaultVrf=True ),
         'public': publicKeywordMatcher,
         'address': 'IP address',
         'IPADDR': IpAddrMatcher.ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF' ]
      vrf = args.get( 'VRF', Tac.Type( 'L3::VrfName' ).defaultVrf )
      pubIp = args.get( 'IPADDR' )
      response = mode.context.addOrRemoveIntf( intf.name, vrf=vrf,
                                               publicIp=pubIp )
      if dpsPathResponseCheck( mode, response ):
         stunProfiles = mode.context.pgCfg.intfStunConfig
         pgName = mode.context.pathGroupName()
         context = DpsPathGroupStunConfigContext( dpsCliConfig, dpsStatus,
                                                  pgName, intf.name, vrf,
                                                  pubIp, stunProfiles,
                                                  'interface' )
         childMode = mode.childMode( DpsPathGroupStunConfigMode, context=context )
         mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = args[ 'INTF' ]
      mode.context.addOrRemoveIntf( intf.name, add=False )

class DpsPathGroupIpConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'local ip IPADDR [ public address PIPADDR ]'
   noOrDefaultSyntax = 'local ip IPADDR ...'

   data = {
         'local': localMatcherForPathGroup,
         'ip': 'ip',
         'IPADDR': IpAddrMatcher.ipAddrMatcher,
         'public': publicKeywordMatcher,
         'address': 'IP address',
         'PIPADDR': IpAddrMatcher.ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      localIp = args[ 'IPADDR' ]
      pubIp = args.get( 'PIPADDR' )
      response = mode.context.addOrRemoveLocalIp( localIp, publicIp=pubIp )
      if dpsPathResponseCheck( mode, response ):
         stunProfiles = mode.context.pgCfg.ipStunConfig
         pgName = mode.context.pathGroupName()
         defaultVrf = Tac.Type( "L3::VrfName" ).defaultVrf
         ipContext = DpsPathGroupStunConfigContext( dpsCliConfig, dpsStatus,
                                                    pgName, localIp,
                                                    defaultVrf, pubIp,
                                                    stunProfiles, 'ip' )
         childMode = mode.childMode( DpsPathGroupStunConfigMode, context=ipContext )
         mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      localIp = args[ 'IPADDR' ]
      mode.context.addOrRemoveLocalIp( localIp, add=False )

class DpsPathGroupIntfServerProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'stun server-profile { PROFILES }'
   noOrDefaultSyntax = 'stun server-profile [ { PROFILES } ]'
   data = {
      'stun': 'stun',
      'server-profile': serverProfileMatcher,
      'PROFILES': CliCommand.Node( matcher=serverProfileNameMatcher, maxMatches=12 )
   }

   @staticmethod
   def _parseProfileNames( mode, args ):
      warning = 'Duplicate server-profile will be ignored.'
      profiles = args.get( 'PROFILES', [] )
      uniqueProfileSet = set( profiles )
      if len( profiles ) != len( uniqueProfileSet ):
         mode.addWarning( warning )
      return uniqueProfileSet

   @staticmethod
   def handler( mode, args ):
      profileSet = DpsPathGroupIntfServerProfileCmd._parseProfileNames( mode, args )
      mode.context.addOrRemoveStunProfile( profileSet )
      dpsStunConfigVerify( mode, mode.context )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileSet = DpsPathGroupIntfServerProfileCmd._parseProfileNames( mode, args )
      mode.context.addOrRemoveStunProfile( profileSet, add=False )

class DpsPathGroupRemoteRouterCmd( CliCommand.CliCommandClass ):
   syntax = '''peer static router-ip IPADDR'''
   noOrDefaultSyntax = syntax
   data = {
         'peer': 'Configure Peer Router Info for DPS Path Group',
         'static': 'Static peer router',
         'router-ip': 'Peer router-IP',
         'IPADDR': IpAddrMatcher.ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      flowAssignmentLan = mode.context.isPathGroupFlowAssignmentLan()
      peers = mode.context.getPathGroupPeers()
      routerIp = args[ 'IPADDR' ]
      if flowAssignmentLan and peers[ 'static' ] and\
         peers[ 'static' ][ 0 ].stringValue != routerIp:
         mode.addErrorAndStop( flowAssignmentLanErrorStaticPeer )

      mode.context.addOrRemoveRouterIp( routerIp, True )
      childMode = mode.childMode( DpsPathGroupRemoteRouterConfigMode,
                                  context=mode.context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      routerIp = args[ 'IPADDR' ]
      mode.context.addOrRemoveRouterIp( routerIp, False )

class DpsPathGroupPeerDynamicCmd( CliCommand.CliCommandClass ):
   syntax = '''peer dynamic'''
   noOrDefaultSyntax = syntax
   data = {
         'peer': 'Configure Peer Router Info for DPS Path Group',
         'dynamic': 'Peer discovered dynamically throught BGP',
   }

   @staticmethod
   def handler( mode, args ):
      flowAssignmentLan = mode.context.isPathGroupFlowAssignmentLan()
      if flowAssignmentLan:
         mode.addErrorAndStop( flowAssignmentLanErrorDynamicPeer )

      mode.context.addOrRemoveDynamicPeer( True )

      pgName = mode.context.pathGroupName()
      context = DpsPeerDynamicConfigContext( dpsCliConfig, dpsStatus,
                                             pgName )
      childMode = mode.childMode( DpsPathGroupPeerDynamicConfigMode,
                                  context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.addOrRemoveDynamicPeer( False )

class DpsPeerDynamicIpLocalCmd( CliCommand.CliCommandClass ):
   syntax = '''ip local'''
   noOrDefaultSyntax = syntax
   data = {
         'ip': 'Set IP preference for dynamic peer connection',
         'local': 'Prefer local IP address',
   }

   @staticmethod
   def handler( mode, args ):
      mode.context.addOrRemovePreferLocalIp( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.addOrRemovePreferLocalIp( False )

class DpsPeerDynamicIpsecCmd( CliCommand.CliCommandClass ):
   syntax = '''ipsec [ disabled ]'''
   noOrDefaultSyntax = syntax
   data = {
         'ipsec': 'IPsec configuration for dynamic peers',
         'disabled': 'Disable IPsec',
   }

   @staticmethod
   def handler( mode, args ):
      mode.context.updateDynamicPeerIpsec( 'disabled' in args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.updateDynamicPeerIpsec( False, default=True )

class DpsPathGroupPeerName( CliCommand.CliCommandClass ):
   syntax = '''name NAME'''
   noOrDefaultSyntax = '''name ...'''
   data = {
      'name': 'Configure Peer Router Name',
      'NAME': siteNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      peerName = args[ 'NAME' ]
      curName = mode.context.checkPeerName( peerName )
      if curName:
         warning = "Peer %s has been configured with name %s." % \
                   ( curName[ 0 ], curName[ 1 ] )
         mode.addWarning( warning )
      else:
         mode.context.addOrRemovePeerName( peerName, True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.addOrRemovePeerName( '', False )

class DpsPathGroupRemoteViaCmd( CliCommand.CliCommandClass ):
   syntax = '''ipv4 address IPADDR'''
   noOrDefaultSyntax = syntax
   data = {
      'ipv4': 'Configure Peer Router Via for DPS Path Group',
      'address': 'Static ipv4 address',
      'IPADDR': IpAddrMatcher.ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      viaIp = args[ 'IPADDR' ]
      response = mode.context.addOrRemoveRouterVia( viaIp, True )
      dpsPathResponseCheck( mode, response )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      viaIp = args[ 'IPADDR' ]
      mode.context.addOrRemoveRouterVia( viaIp, False )

class DpsPathGroupTcpMssCeilingCmd( CliCommand.CliCommandClass ):
   syntax = '''tcp mss ceiling
               ipv4 MSSv4 egress'''
   noOrDefaultSyntax = '''tcp mss ceiling ...'''
   data = {
      'tcp': 'TCP',
      'mss': 'Maximum segment size',
      'ceiling': 'Set maximum limit',
      'ipv4': 'Internet Protocol version 4',
      'MSSv4': CliMatcher.IntegerMatcher( 64, 65515,
                                          helpdesc='Segment size' ),
      'egress': 'Enforce on packets forwarded to the network',
   }

   @staticmethod
   def handler( mode, args ):
      ipv4TcpMss = args.get( 'MSSv4', 0 )
      mode.context.setMss( ipv4TcpMss )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setMss( 0 )

class DpsPathGroupMtuCmd( CliCommand.CliCommandClass ):
   syntax = '''mtu ( MTU_VAL | disabled )'''
   noOrDefaultSyntax = '''mtu'''
   data = {
      'mtu': 'Configure MTU parameters',
      # Min val = 68 bytes as per RFC 791
      # Max val = 9194 - DPS overhead = 9194 - 36 = 9158
      'MTU_VAL': CliMatcher.IntegerMatcher( 68, 9158,
                                            helpdesc='MTU value' ),
      'disabled': 'Disable PMTU auto discovery',
   }

   @staticmethod
   def handler( mode, args ):
      mode.context.setMtu( args.get( 'MTU_VAL', constants.pmtuDisabled ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # auto case
      mode.context.setMtu( 0 )

class DpsPathGroupItsInterval( CliCommand.CliCommandClass ):
   syntax = '''keepalive interval ( auto ) | ( INTERVAL milliseconds
               [ failure-threshold THRESHOLD intervals ] )'''
   noOrDefaultSyntax = '''keepalive interval ...'''
   data = {
      'keepalive': 'Keepalive',
      'interval': 'Interval',
      'auto': 'Adaptive keepalive and feedback interval',
      'INTERVAL': CliMatcher.IntegerMatcher( 50, 60000,
                                             helpdesc='Interval ( in ms )' ),
      'milliseconds': 'Milliseconds',
      'failure-threshold': 'Failure threshold',
      'THRESHOLD': CliMatcher.IntegerMatcher( 2, 100,
                                              helpdesc='Interval Count' ),
      'intervals': 'Intervals'
   }

   @staticmethod
   def handler( mode, args ):
      keepaliveInterval = args.get( 'INTERVAL', -1 )
      scale = args.get( 'THRESHOLD', 5 )
      mode.context.setItsInterval( keepaliveInterval, scale )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setItsInterval( 0, 0 )

class DpsPathGroupImportCmd( CliCommand.CliCommandClass ):
   syntax = '''import path-group { ( remote REMOTE_PG ) | ( local LOCAL_PG ) }'''
   noOrDefaultSyntax = syntax
   data = {
      'import': 'import remote and local info from other path-groups',
      'path-group': 'Configure path group to import from',
      'remote': CliCommand.singleKeyword(
                  'remote', helpdesc='import remote info from path-group' ),
      'REMOTE_PG': pathGroupConfigNameMatcher,
      'local': CliCommand.singleKeyword(
                  'local', helpdesc='import local info from path-group' ),
      'LOCAL_PG': pathGroupConfigNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      pathGroupRemote = args.get( 'REMOTE_PG', [ None ] )[ 0 ]
      pathGroupLocal = args.get( 'LOCAL_PG', [ None ] )[ 0 ]
      if not pathGroupRemote:
         pathGroupRemote = mode.context.pathGroupName()
      if not pathGroupLocal:
         pathGroupLocal = mode.context.pathGroupName()
      if pathGroupRemote == pathGroupLocal:
         error = "Local and remote path groups cannot be the same"
         mode.addErrorAndStop( error )
      if mode.context.pathGroupImportExists( pathGroupRemote, pathGroupLocal ):
         warning = "The same import configuration exists in another path group " \
                   "configuration. Only one will be effective."
         mode.addWarning( warning )
      mode.context.setPathGroupImport( pathGroupRemote, pathGroupLocal )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pathGroupRemote = args.get( 'REMOTE_PG', [ None ] )[ 0 ]
      pathGroupLocal = args.get( 'LOCAL_PG', [ None ] )[ 0 ]
      if not pathGroupRemote:
         pathGroupRemote = mode.context.pathGroupName()
      if not pathGroupLocal:
         pathGroupLocal = mode.context.pathGroupName()
      if pathGroupRemote != pathGroupLocal:
         mode.context.removePathGroupImport( pathGroupRemote, pathGroupLocal )

class DpsPathGroupFlowAssignmentLanCmd( CliCommand.CliCommandClass ):
   syntax = '''flow assignment lan'''
   noOrDefaultSyntax = syntax
   data = {
      'flow': CliCommand.guardedKeyword( 'flow',
                                         helpdesc='Enable flow assignement feature',
                                         guard=guardFlowAssignmentLanConfig ),
      'assignment': 'Assignment',
      'lan': 'LAN path group'
   }

   @staticmethod
   def handler( mode, args ):
      pg = mode.context.checkFlowAssignmentLanPathGroup()
      if pg:
         warning = 'Path group %s has already been configured with'\
         ' flow assignment lan' % pg
         mode.addErrorAndStop( warning )
      else:
         peers = mode.context.getPathGroupPeers()
         if 'dynamic' in peers:
            mode.addErrorAndStop( flowAssignmentLanErrorDynamicPeer )
         elif len( peers[ 'static' ] ) > 1:
            mode.addErrorAndStop( flowAssignmentLanErrorStaticPeer )

      mode.context.addOrRemoveFlowAssignmentLan( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.addOrRemoveFlowAssignmentLan( False )

class DpsPolicyConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''policy <policy-name>'''
   noOrDefaultSyntax = syntax
   data = {
         'policy': 'Configure Dps policy',
         '<policy-name>': policyConfigNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      name = args.get( '<policy-name>' )
      context = DpsPolicyContext( dpsCliConfig, dpsStatus, name )

      context.addOrRemovePolicy( name, True )
      childMode = mode.childMode( DpsPolicyConfigMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args.get( '<policy-name>' )
      context = DpsPolicyContext( dpsCliConfig, dpsStatus, name )
      context.addOrRemovePolicy( name, False )

class DpsPolicyRuleKeyConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''<rule-key> application-profile <app-profile>'''
   noOrDefaultSyntax = syntax
   data = {
         '<rule-key>': tokenRuleKey,
         'application-profile': 'application-profile',
         '<app-profile>': nameToIdNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      ruleKey = args.get( '<rule-key>' )
      appProfile = args.get( '<app-profile>' )
      mode.context.addOrRemoveRuleKey( ruleKey, appProfile, True )
      mode.rulesChanged = True
      childMode = mode.childMode( DpsPolicyRuleKeyConfigMode,
                                  context=mode.context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ruleKey = args.get( '<rule-key>' )
      appProfile = args.get( '<app-profile>' )
      mode.context.addOrRemoveRuleKey( ruleKey, appProfile, False )
      mode.rulesChanged = True

class DpsPolicyDefaultRuleConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''default-match'''
   noOrDefaultSyntax = syntax
   data = {
         'default-match': 'default matching rule',
   }

   @staticmethod
   def handler( mode, args ):
      mode.context.addOrRemoveDefaultRule( True )
      childMode = mode.childMode( DpsPolicyDefaultRuleConfigMode,
                                  context=mode.context )
      mode.session_.gotoChildMode( childMode )
      mode.rulesChanged = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.addOrRemoveDefaultRule( False )
      mode.rulesChanged = True

class DpsPolicyRuleLbCmd( CliCommand.CliCommandClass ):
   syntax = '''load-balance <load-balance-grp>'''
   noOrDefaultSyntax = '''load-balance ...'''
   data = {
         'load-balance': 'attach load-balance group',
         '<load-balance-grp>': loadBalanceProfileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      actionName = args.get( '<load-balance-grp>' )
      mode.context.setLbGrpName( actionName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setLbGrpName( "" )

class DpsLoadBalanceProfileConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''load-balance policy <profile-name>'''
   noOrDefaultSyntax = syntax
   data = {
         'load-balance': 'Configure load balance parameters',
         'policy': 'Configure load balance policy parameters',
         '<profile-name>': loadBalanceProfileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      name = args.get( '<profile-name>' )
      context = DpsLoadBalanceProfileContext( dpsCliConfig, dpsStatus, name )
      context.addProfile( name )
      childMode = mode.childMode( DpsLoadBalanceProfileConfigMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args.get( '<profile-name>' )
      context = DpsLoadBalanceProfileContext( dpsCliConfig, dpsStatus, name )
      context.delProfile( name )

class DpsLbpLatencyConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''latency MILLISECONDS'''
   noOrDefaultSyntax = 'latency ...'

   data = {
         'latency': 'Configure one way delay requirement'
                    ' for this load balance policy',
         'MILLISECONDS': tokenMilliSecond,
   }

   @staticmethod
   def handler( mode, args ):
      timeArg = args.get( 'MILLISECONDS' )
      mode.context.setLatency( int( timeArg ) * LATENCY_SCALE )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setLatency( DEFAULT_LATENCY )

class DpsLbpHopCountLowestConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''hop count lowest'''
   noOrDefaultSyntax = syntax
   data = {
         'hop': 'Configure path hop requirement for this load balance policy',
         'count': 'Configure hop count limit for multi hop path',
         'lowest': 'Prefer paths with lowest hop-count'
   }

   @staticmethod
   def handler( mode, args ):
      mode.context.setHopCountLowest( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setHopCountLowest( False )

class DpsLbpJitterConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''jitter MILLISECONDS'''
   noOrDefaultSyntax = 'jitter ...'

   data = {
         'jitter': 'Configure jitter requirement for this load balance policy',
         'MILLISECONDS': tokenMilliSecond,
   }

   @staticmethod
   def handler( mode, args ):
      timeArg = args.get( 'MILLISECONDS' )
      mode.context.setJitter( int( timeArg ) * JITTER_SCALE )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setJitter( DEFAULT_JITTER )

class DpsLbpLossRateConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''loss-rate <percentage>'''
   noOrDefaultSyntax = 'loss-rate ...'

   data = {
         'loss-rate': 'Configure loss rate requirement for this load balance policy',
         '<percentage>': tokenPercentage,
   }

   @staticmethod
   def handler( mode, args ):
      percentArg = args.get( '<percentage>' )
      loss = int( float( percentArg ) * LOSS_RATE_SCALE )
      loss += LOSS_RATE_ADJUSTMENT
      mode.context.setLossRate( loss )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setLossRate( DEFAULT_LOSSRATE )

class DpsLbpPathGroupConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''path-group <name> [ priority <priority> ]'''
   noOrDefaultSyntax = '''path-group <name> ...'''

   data = {
      'path-group': 'Configure the path-group to use with this load balance policy',
      '<name>': pathGroupConfigNameMatcher,
      'priority': 'Configure priority for this path group',
      '<priority>': priorityForPathGroupMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      pathGroup = args.get( '<name>' )
      priority = args.get( '<priority>', DEFAULT_PRIORITY )
      mode.context.addPathGroup( pathGroup, priority )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pathGroup = args.get( '<name>' )
      mode.context.delPathGroup( pathGroup )

class DpsVrfConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'VRF'
   noOrDefaultSyntax = syntax
   data = {
         'VRF': VrfExprFactory( inclDefaultVrf=True ),
   }

   @staticmethod
   def handler( mode, args ):
      name = args.get( 'VRF' )
      context = DpsVrfConfigContext( dpsCliConfig, dpsStatus, name )

      context.addVrfConfig( name )
      childMode = mode.childMode( DpsVrfConfigMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args.get( 'VRF' )
      context = DpsVrfConfigContext( dpsCliConfig, dpsStatus, name )

      context.delVrfConfig( name )

# XXX: The cpu keyword may not be used in the future.
# a design change may let the customer use the same policy for both data
# path and cpu path.
class DpsVrfPathSelectionPolicyConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''path-selection-policy <policy-name>'''
   noOrDefaultSyntax = 'path-selection-policy ...'

   data = {
      'path-selection-policy': 'Configure policy name to use for this vrf',
      '<policy-name>': policyConfigNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      policyName = args.get( '<policy-name>' )
      mode.context.setPolicy( policyName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setPolicy( "" )

class DpsEncapConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''encapsulation path-telemetry udp port PORT'''
   noOrDefaultSyntax = '''encapsulation path-telemetry udp port ...'''
   data = {
      'encapsulation': 'Configure encapsulation parameters',
      'path-telemetry': 'Configure encapsulation path-telemetry parameters',
      'udp': 'Configure UDP parameters',
      'port': 'Configure UDP port number',
      'PORT': CliMatcher.IntegerMatcher( 1, 65536, helpdesc='Destination UDP port' ),
   }

   @staticmethod
   def handler( mode, args ):
      portNum = args.get( 'PORT', DEFAULT_UDP_PORT )
      context = DpsEncapConfigContext( dpsCliConfig, dpsStatus )
      context.setUdpPort( portNum )

   noOrDefaultHandler = handler

class DpsPeerDynamicStunConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''peer dynamic source stun'''
   noOrDefaultSyntax = syntax
   data = {
      'peer': 'Configure peer router discovery',
      'dynamic': 'Peer discovered dynamically',
      'source': 'Source of dynamic peer discovery',
      'stun': 'Use STUN clients as peers',
   }

   @staticmethod
   def handler( mode, args ):
      context = DpsPeerDynamicStunConfigContext( dpsCliConfig, dpsStatus )
      context.addOrRemovePeerStunConfig( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      context = DpsPeerDynamicStunConfigContext( dpsCliConfig, dpsStatus )
      context.addOrRemovePeerStunConfig( False )

class DpsTcpMssCeilingIngressCmd( CliCommand.CliCommandClass ):
   syntax = '''tcp mss ceiling ipv4 [ MSS4 ] ingress'''
   noOrDefaultSyntax = '''tcp mss ceiling ...'''
   data = {
      'tcp': 'TCP',
      'mss': 'Maximum segment size',
      'ceiling': 'Set maximum limit',
      'ipv4': 'Internet Protocol version 4',
      'MSS4': CliMatcher.IntegerMatcher( 64, 65515,
                                         helpdesc='Segment size' ),
      'ingress': 'Enforce on packets ingressing from DPS tunnel',
   }

   @staticmethod
   def handler( mode, args ):
      ipv4TcpMss = args.get( 'MSS4', constants.tcpMssIngressRewriteAuto )
      dpsCliConfig.tcpMssIngressConfig = ipv4TcpMss

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dpsCliConfig.tcpMssIngressConfig = constants.tcpMssIngressRewriteDisabled

class DpsMtuDiscIntervalConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''mtu discovery interval MTU_INTERVAL seconds'''
   noOrDefaultSyntax = '''mtu discovery interval ...'''
   data = {
      'mtu': 'Configure MTU parameters',
      'discovery': 'Configure MTU discovery parameters',
      'interval': 'Set interval',
      'MTU_INTERVAL': CliMatcher.IntegerMatcher( 60, 600,
                                                 helpdesc='Interval in secs' ),
      'seconds': 'Interval in units of seconds',
   }

   @staticmethod
   def handler( mode, args ):
      mode.context.setMtuDiscInterval( args.get( 'MTU_INTERVAL' ) )

   noOrDefaultHandler = handler

class DpsMtuDiscHostsConfigCmd( CliCommand.CliCommandClass ):
   syntax = "mtu discovery hosts " \
            "[ fragmentation-needed rate-limit RATE_LIMIT packets-per-second ]"
   noOrDefaultSyntax = "mtu discovery hosts ..."
   data = {
      'mtu': 'Configure MTU parameters',
      'discovery': 'Configure MTU discovery parameters',
      'hosts': 'Enable MTU discovery for hosts on the LAN',
      'fragmentation-needed': 'Set ICMP fragmentation needed rate limit',
      'rate-limit': 'Maximum rate of ICMP packet generation per CPU core',
      'RATE_LIMIT': CliMatcher.IntegerMatcher( 1, 500,
                                         helpdesc='packets per second' ),
      'packets-per-second': 'Rate limit in units of packets per second',
   }

   @staticmethod
   def handler( mode, args ):
      rateLimit = args.get( 'RATE_LIMIT', constants.icmpFragNeededDefaultRateLimit )
      dpsCliConfig.icmpFragNeededRateLimit = rateLimit
      dpsCliConfig.icmpFragNeededEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dpsCliConfig.icmpFragNeededEnabled = False
      dpsCliConfig.icmpFragNeededRateLimit = constants.icmpFragNeededDefaultRateLimit

# router path-selection
BasicCli.GlobalConfigMode.addCommandClass( RouterPathSelectionConfigCmd )

# tcp mss ingress command
RouterPathSelectionConfigMode.addCommandClass( DpsTcpMssCeilingIngressCmd )

# mtu discovery hosts command
RouterPathSelectionConfigMode.addCommandClass( DpsMtuDiscHostsConfigCmd )

# path-group sub-commands
RouterPathSelectionConfigMode.addCommandClass( DpsPathGroupConfigCmd )
DpsPathGroupConfigMode.addCommandClass( DpsIntfConfigCmd )
DpsPathGroupConfigMode.addCommandClass( DpsPathGroupIpConfigCmd )
DpsPathGroupStunConfigMode.addCommandClass( DpsPathGroupIntfServerProfileCmd )
DpsPathGroupConfigMode.addCommandClass( DpsPathGroupTcpMssCeilingCmd )
DpsPathGroupConfigMode.addCommandClass( DpsPathGroupItsInterval )
DpsPathGroupConfigMode.addCommandClass( DpsPathGroupIpsecConfigCmd )
DpsPathGroupConfigMode.addCommandClass( DpsPathGroupRemoteRouterCmd )
DpsPathGroupConfigMode.addCommandClass( DpsPathGroupPeerDynamicCmd )
DpsPathGroupConfigMode.addCommandClass( DpsPathGroupMtuCmd )
DpsPathGroupConfigMode.addCommandClass( DpsMtuDiscIntervalConfigCmd )
DpsPathGroupConfigMode.addCommandClass( DpsPathGroupFlowAssignmentLanCmd )
DpsPathGroupConfigMode.addCommandClass( DpsPathGroupImportCmd )
DpsPathGroupRemoteRouterConfigMode.addCommandClass( DpsPathGroupRemoteViaCmd )
DpsPathGroupRemoteRouterConfigMode.addCommandClass( DpsPathGroupPeerName )
DpsPathGroupPeerDynamicConfigMode.addCommandClass( DpsPeerDynamicIpLocalCmd )
DpsPathGroupPeerDynamicConfigMode.addCommandClass( DpsPeerDynamicIpsecCmd )

# policy sub-commands
RouterPathSelectionConfigMode.addCommandClass( DpsPolicyConfigCmd )
DpsPolicyConfigMode.addCommandClass( DpsPolicyRuleKeyConfigCmd )
DpsPolicyConfigMode.addCommandClass( DpsPolicyDefaultRuleConfigCmd )
DpsPolicyRuleKeyConfigMode.addCommandClass( DpsPolicyRuleLbCmd )
DpsPolicyDefaultRuleConfigMode.addCommandClass( DpsPolicyRuleLbCmd )

# load-balance
RouterPathSelectionConfigMode.addCommandClass( DpsLoadBalanceProfileConfigCmd )
DpsLoadBalanceProfileConfigMode.addCommandClass( DpsLbpLatencyConfigCmd )
DpsLoadBalanceProfileConfigMode.addCommandClass( DpsLbpJitterConfigCmd )
DpsLoadBalanceProfileConfigMode.addCommandClass( DpsLbpLossRateConfigCmd )
DpsLoadBalanceProfileConfigMode.addCommandClass( DpsLbpPathGroupConfigCmd )
DpsLoadBalanceProfileConfigMode.addCommandClass( DpsLbpHopCountLowestConfigCmd )

# Vrf
RouterPathSelectionConfigMode.addCommandClass( DpsVrfConfigCmd )
DpsVrfConfigMode.addCommandClass( DpsVrfPathSelectionPolicyConfigCmd )

# UDP port
RouterPathSelectionConfigMode.addCommandClass( DpsEncapConfigCmd )

# dynamic peers from STUN
RouterPathSelectionConfigMode.addCommandClass( DpsPeerDynamicStunConfigCmd )

# mtu discovery interval config
RouterPathSelectionConfigMode.addCommandClass( DpsMtuDiscIntervalConfigCmd )

if toggleAvtLowestLoadMetricEnabled():
   # interface max bandwidth
   RouterPathSelectionConfigMode.addCommandClass( DpsIntfSpeedConfigCmd )
   DpsIntfConfigMode.addCommandClass( DpsIntfTxBandWidthConfigCmd )
   DpsIntfConfigMode.addCommandClass( DpsIntfRxBandWidthConfigCmd )

def Plugin( em ):
   global dpsCliConfig, dpsStatus, peerStatus, entityManager, ipsecConfig
   global stunServerProfileConfig, appRecognitionConfig

   entityManager = em
   mountGroup = entityManager.mountGroup()
   dpsStatus = mountGroup.mount(
      'dps/status',
      'Tac::Dir', 'ri' )
   mountGroup.close( callback=None, blocking=False )

   appRecognitionConfig = LazyMount.mount( entityManager,
         'classification/app-recognition/config',
         'Classification::AppRecognitionConfig', 'r' )
   dpsCliConfig = ConfigMount.mount( entityManager,
        'dps/input/cli',
        'Dps::DpsCliConfig', 'wi' )
   ipsecConfig = LazyMount.mount(
      entityManager, 'ipsec/ike/config',
      'Ipsec::Ike::Config', 'r' )
   peerStatus = LazyMount.mount(
      entityManager, 'dps/peer/status',
      'Dps::PeerStatus', 'r' )
   stunServerProfileConfig = LazyMount.mount(
      entityManager, 'stun/server-profile',
      'Stun::ServerProfileConfig', 'r' )
