# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
Dps interface type implementation.

This module provides the DpsIntf class.
"""

from __future__ import absolute_import

import re
import Tac
import LazyMount
import ConfigMount
import CliParser
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin import IntfCli
from CliPlugin import VirtualIntfRule
from CliPlugin.WanTEShowCli import pathSelectionSupportedGuard
from IntfRangePlugin.DpsIntf import DpsAutoIntfType
from DpsIntfUtils import minDpsIntfNum, maxDpsIntfNum

# pathSelectionSupportedGuard is part of WanTECommon-cli. And it uses
# Dps::HwCapability which is part of WanTECommon-lib. So, have this pkgdeps to
# introduce a explicit dependency on WanTECommon-lib, otherwise at times the
# DpsIntfCliTest.py breadth test would fail saying no library has been loaded
# that includes this type...
# pkgdeps: rpm WanTECommon-lib

dpsIntfConfigDir = None
dpsIntfStatusDir = None

class DpsIntf( IntfCli.VirtualIntf ):

   def __init__( self, name, mode ):
      """Creates a new Dps interface instance of the specified name."""
      m = re.match( r'Dps(\d+)$', name )
      self.dpsId = int( m.group( 1 ) )
      IntfCli.VirtualIntf.__init__( self, name, mode )
      self.intfStatus = None

   # ----------------------------------------------------------------------------
   # The rule for matching Dps interface names. When this pattern matches,
   # it returns an instance of the DpsIntf class.
   #
   # This rule gets added to the Intf.rule when this class is registered with
   # the Intf class by calling Intf.addPhysicalIntfType, below.
   # ----------------------------------------------------------------------------
   matcher = VirtualIntfRule.VirtualIntfMatcher(
      'Dps', minDpsIntfNum, maxDpsIntfNum, value=lambda mode,
      intf: DpsIntf( intf, mode ), # pylint: disable-msg=undefined-variable
      helpdesc='DPS Tunnel Interface', guard=pathSelectionSupportedGuard )

   def createPhysical( self, startupConfig=False ):
      """Creates the DpsIntfConfig object for this interface."""
      if not dpsIntfConfigDir:
         return
      dpsIntfConfigDir.intfConfig.newMember( str( self ) )

   def lookupPhysical( self ):
      """Determines whether the DpsIntfStatus object exists."""
      if not dpsIntfStatusDir:
         return False
      self.intfStatus = dpsIntfStatusDir.intfStatus.get( str( self ) )
      return self.intfStatus is not None

   def destroyPhysical( self ):
      """Destroys the DpsIntfConfig object for this interface."""
      if not dpsIntfConfigDir:
         return
      del dpsIntfConfigDir.intfConfig[ str( self ) ]

   def config( self ):
      """Returns the DpsIntfConfig object for this interface."""
      if not dpsIntfConfigDir:
         return None
      return dpsIntfConfigDir.intfConfig.get( str( self ) )

   def status( self ):
      """Returns the DpsIntfStatus object for this interface."""
      return self.intfStatus

   def getIntfCounterDir( self ):
      """Returns the DpsIntfCounterDir object for this interface.

      We aren't supporting counters on dps interfaces
      """
      return None

   def showPhysical( self, mode, intfStatusModel ):
      """Outputs information about this intf in an intf type specific manner."""
      pass

   def bandwidth( self ):
      return 0

   def hardware( self ):
      return "dps"

   def mtu( self ):
      if self.lookupPhysical():
         return self.intfStatus.mtu
      return 9194

   def addrStr( self ):
      return None

   @staticmethod
   def getAllPhysical( mode ):
      """Returns unsorted list of DpsIntf objects.

      Returns the list of intfs for which the status object exists.
      """
      intfs = []
      if not dpsIntfStatusDir:
         return intfs
      iDir = dpsIntfStatusDir.intfStatus
      for name in iDir:
         intf = DpsIntf( name, mode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

   def countersSupported( self ):
      """We don't support counters on Dps interfaces."""
      return False

   def vrfSupported( self ):
      """We don't allow vrf forwarding command on the Dps interface."""
      return False

   def isValidHostInetAddress( self, a1 ):
      if a1.len > 32:
         return ( False, "Prefix length must be less than or equal to 32" )
      if a1.len == 32:
         # do what the industry standard seems to do
         if ( not IpAddrMatcher.validateMulticastIpAddr( a1.address ) or
               not IpAddrMatcher.splitIpAddrToInts( a1.address )[ 0 ] ):
            return ( False, "Not a valid host address - %s" % ( a1.address, ) )

         return ( True, "" )
      if IpAddrMatcher.isLoopbackIpAddr( a1.address ):
         return ( False, "IP address must not be loopback" )
      return self.isUnicastInetAddress( a1 )

   def isValidHostInet6Address( self, a1 ):
      return ( False, "Not supported yet" )

# -------------------------------------------------------------------------------
# Register the DpsIntf class as a type of physical interface.
# -------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( DpsIntf, DpsAutoIntfType, withIpSupport=True,
                                  withRoutingProtoSupport=False )

# -------------------------------------------------------------------------------
# Adds Dps-specific CLI commands to the "config-if" mode for DPS interface.
# -------------------------------------------------------------------------------
class DpsIntfConfigModelet ( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, DpsIntf )

# -------------------------------------------------------------------------------
# Associate the DpsIntfConfigModelet with the "config-if" mode.
# -------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( DpsIntfConfigModelet )

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global dpsIntfConfigDir, dpsIntfStatusDir
   dpsIntfConfigDir = ConfigMount.mount( entityManager,
                                         "interface/config/dps/intf",
                                         "Interface::DpsIntfConfigDir", "w" )
   dpsIntfStatusDir = LazyMount.mount( entityManager,
                                       "interface/status/dps/intf",
                                       "Interface::DpsIntfStatusDir", "r" )
