# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import Int
from CliModel import Model
from CliModel import Str
from CliModel import Dict
from CliModel import Enum
from CliModel import Bool
from CliModel import List
from CliModel import Submodel
import CliExtensions
import EthIntfLib
# pylint: disable-next=consider-using-from-import
import CliPlugin.EthIntfModel as EthIntfModel
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
import CliPlugin.IntfModel as IntfModel # pylint: disable=consider-using-from-import
import TableOutput

class Encapsulation( Model ):
   encapsulationType = Enum( values=( "dot1q", ),
                             help="Encapsulation type" )
   dot1qVlanTags = List( valueType=int,
                         help="Encapsulation dot1q vlan tags",
                         optional=True )
   # Encapsulation information is incomplete (Flex Encap).
   # The user needs to get the complete information from
   # 'show interface encapsulation vlan'.
   _incompleteInformation = Bool(
         help='Incomplete encapsulation information. '
              'Use show interface encapsulation vlan instead' )

class InterfacesStatus( Model ):
   class InterfaceStatusEntry( Model ):
      class VlanInformation( Model ):
         vlanId = Int( optional=True, help="Vlan ID if interface is bridged, "
                                           "None otherwise" )
         interfaceMode = Enum( values=( "bridged", "routed", "inactive", "trunk",
                                        "tap", "tool", "tapTool", "encap",
                                        "fabric" ),
                               optional=True, help="If not in DataLink mode, what "
                                                   "mode." )
         vlanExplanation = EthIntfModel.EthPhyInterfaceStatus.interfaceMembership
         interfaceForwardingModel = \
                                 EthIntfModel.EthPhyInterfaceStatus.forwardingModel
         
         def getVlanStr( self ):        
            if ( self.interfaceForwardingModel and
                 self.interfaceForwardingModel not in \
                       ( "dataLink", "quietDataLink", "unauthorized", 
                         "recirculation", "layer1Patch", "linkQualification" ) ):
               if self.interfaceMode == "bridged":
                  return str( self.vlanId )
               elif self.interfaceMode == "encap":
                  if self.interfaceForwardingModel == "bridged": # l2SubIntf
                     return str( self.vlanId )
                  else: # l3SubIntf
                     assert self.interfaceForwardingModel == "routed"
                     return "routed"
               else:
                  return self.interfaceMode
            else:
               return self.vlanExplanation 
 
      description = EthIntfModel.EthPhyInterfaceStatus.description
      linkStatus = EthIntfModel.EthPhyInterfaceStatus.interfaceStatus
      duplex = EthIntfModel.EthPhyInterfaceStatus.duplex
      bandwidth = EthIntfModel.EthPhyInterfaceStatus.bandwidth
      autoNegotigateActive = Bool( help="Is autonegotiate active. "
                                        "Misspelled autoNegotiateActive (legacy)" )
      autoNegotiateActive = Bool( help="Is autonegotiate active", optional=True ) 
      interfaceType = Str( help="Interface Type" )
      vlanInformation = Submodel( valueType=VlanInformation, 
                                  help="Model to describe the VLAN setting" )
      encapsulation = Submodel( valueType=Encapsulation,
                                help="Encapsulation information", 
                                optional=True )
      lineProtocolStatus = Enum( help="Line protocol status",
                                 values=IntfModel.intfOperStatuses, optional=True )
      underMaintenance = Bool( help="Indicates an Interface is under "
                                "maintenance or not", optional=True )
      trafficAboveThreshold = Bool( help="Indicates that the traffic on the "
                                    "Interface is above traffic threshold",
                                    optional=True )
      interfaceDamped = Bool( help="Indicates that the interface is damped "
                                   "as a result of link flaps",
                                    optional=True )
   interfaceStatuses = Dict( valueType=InterfaceStatusEntry,
                            help="Collection of interfaces statuses, "
                                 "keyed by interface name" )
   
   def render( self ):
      # don't print anything if we don't have any interfaces
      if len( self.interfaceStatuses ) == 0:
         return

      # To avoid content truncation, we set the column width dynamically. 
      # minWidth is needed to avoid table shrinking ( tableWidth >= 80 )
      numOfColumns = 9
      # Port
      portMinLen = 10
      portMaxLen = 10
      # Name
      nameMinLen = 6
      nameMaxLen = 6
      # Status
      statusMinLen = 12
      statusMaxLen = 12
      # Vlan
      vlanMinLen = 8
      vlanMaxLen = 8
      # Duplex
      duplexMinLen = 6
      duplexMaxLen = 6
      # Speed
      speedMinLen = 6
      speedMaxLen = 6
      # Type
      typeMinLen = 12
      typeMaxLen = 12
      # FLags
      flagsMinLen = 5
      flagsMaxLen = 5
      # Encapsulation
      encapsulationMinLen = 13
      encapsulationMaxLen = 13

      # Calculate dynamic columns' widths first
      for interface in Arnet.sortIntf( self.interfaceStatuses ):
         interfaceStatusEntry = self.interfaceStatuses[ interface ]
         # Update max len for dynamically sized columns
         portMaxLen = max( len( IntfCli.Intf.getShortname( interface ) ), 
                           portMaxLen )
         nameMaxLen = max( len( interfaceStatusEntry.description ), nameMaxLen )
         typeMaxLen = max( len( interfaceStatusEntry.interfaceType ), typeMaxLen )
         vlanStr = interfaceStatusEntry.vlanInformation.getVlanStr()
         if vlanStr is not None:
            vlanMaxLen = max( len( vlanStr ), vlanMaxLen )

      tableMaxWidth = ( portMaxLen + nameMaxLen + statusMaxLen + vlanMaxLen +
                        duplexMaxLen + speedMaxLen + typeMaxLen + flagsMaxLen +
                        encapsulationMaxLen +
                        # spaces between columns
                        numOfColumns - 1 )
      # Give spaces left to the Name column, making tableWidth is at least 80.
      if tableMaxWidth < 80:
         nameMinLen = nameMaxLen + ( 80 - tableMaxWidth )
         nameMaxLen = nameMinLen
         tableMaxWidth = 80

      # tableWidth is set to enforce a "terminal wrap" instead of a "table wrap"
      table = TableOutput.createTable( ( "Port", "Name", "Status", "Vlan",
                                         "Duplex", "Speed", "Type", "Flags",
                                         "Encapsulation" ),
                                         tableWidth=tableMaxWidth )

      # Port
      f1 = TableOutput.Format( justify="left", minWidth=portMinLen, 
                               maxWidth=portMaxLen, isHeading=True )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )

      # Name
      f2 = TableOutput.Format( justify="left", minWidth=nameMinLen,
                               maxWidth=nameMaxLen )
      f2.noPadLeftIs( True )
      f2.padLimitIs( True )

      # Status
      f3 = TableOutput.Format( justify="left", minWidth=statusMinLen, 
                               maxWidth=statusMaxLen )
      f3.noPadLeftIs( True )
      f3.padLimitIs( True )

      # Vlan
      f4 = TableOutput.Format( justify="left", minWidth=vlanMinLen, 
                               maxWidth=vlanMaxLen )
      f4.noPadLeftIs( True )
      f4.padLimitIs( True )

      # Duplex
      f5 = TableOutput.Format( justify="left", minWidth=duplexMinLen, 
                               maxWidth=duplexMaxLen )
      f5.noPadLeftIs( True )
      f5.padLimitIs( True )

      # Speed
      f6 = TableOutput.Format( justify="left", minWidth=speedMinLen, 
                               maxWidth=speedMaxLen )
      f6.noPadLeftIs( True )
      f6.padLimitIs( True )

      # Type
      f7 = TableOutput.Format( justify="left", minWidth=typeMinLen, 
                               maxWidth=typeMaxLen )
      f7.noPadLeftIs( True )
      f7.padLimitIs( True )

      # Flags
      f8 = TableOutput.Format( justify="left", minWidth=flagsMinLen, 
                               maxWidth=flagsMaxLen )
      f8.noPadLeftIs( True )
      f8.padLimitIs( True )

      # Encapsulation
      f9 = TableOutput.Format( justify="left", minWidth=encapsulationMinLen, 
                               maxWidth=encapsulationMaxLen )
      f9.noPadLeftIs( True )
      f9.padLimitIs( True )
      f9.noTrailingSpaceIs( True )

      table.formatColumns( f1, f2, f3, f4, f5, f6, f7, f8, f9 )
      table.formatRows( TableOutput.Format( isHeading=True, border=False ) )
 
      showLegend = False
      for interface in Arnet.sortIntf( self.interfaceStatuses ):
         flag = ''
         interfaceStatusEntry = self.interfaceStatuses[ interface ]
         duplex = EthIntfLib.showIntStatusDuplexStr( 
                                       interfaceStatusEntry.autoNegotiateActive,
                                       interfaceStatusEntry.duplex )
         bandwidth = EthIntfLib.showIntStatusSpeedStr( 
                                          interfaceStatusEntry.autoNegotiateActive, 
                                          interfaceStatusEntry.bandwidth )

         encap = interfaceStatusEntry.encapsulation
         encapStr = ''
         if encap:
            # pylint: disable=protected-access
            if encap._incompleteInformation:
               encapStr = '*'
            elif encap.encapsulationType == "dot1q":
               encapStr = ','.join( [ str( x ) for x in encap.dot1qVlanTags ] )

         if interfaceStatusEntry.underMaintenance:
            showLegend = True
            flag = 'm'
            if interfaceStatusEntry.trafficAboveThreshold:
               flag += 'v'
         if interfaceStatusEntry.interfaceDamped:
            showLegend = True
            flag += 'd'
         table.newRow( IntfCli.Intf.getShortname( interface ), 
                       interfaceStatusEntry.description,
                       IntfModel.formatIntfStatus(
                          interfaceStatusEntry.linkStatus ),
                       interfaceStatusEntry.vlanInformation.getVlanStr(),
                       duplex, bandwidth, interfaceStatusEntry.interfaceType, flag,
                       encapStr ) 

      print( table.output() )
      if showLegend:
         print( 'Flags : m - Under Maintenance' )
         print( '        v - Violating traffic threshold' )
         print( '        d - Interface state is damped' )
interfaceStatusHook = CliExtensions.CliHook()
