# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import OrderedDict
import CliParser
import ConfigMount
from EnvironmentUtils import (
   actionOnOverheatEnumMap,
   numZonesToProfileName,
   MAX_NUM_OF_COOLING_ZONES,
)
import EntityMib
import CliPlugin.TechSupportCli              # pylint: disable-msg=F0401
import CliPlugin.FruCli as FruCli # pylint: disable=consider-using-from-import
from CliPlugin import EnvironmentModels
import CliExtensions
import LazyMount
import Tac
import Tracing
from TypeFuture import TacLazyType
from ArPyUtils import naturalOrderKey

t0 = Tracing.trace0

DEFAULT_TEMPERATURE_POLL_INTERVAL = 5

temperatureConfig = None
temperatureStatus = None
archerCellTempStatus = None
archerSystemTempStatus = None
powerStatusDir = None
coolingConfig = None
coolingDomainDir = None
archerCoolingStatus = None
thermostatConfig = None
thermostatHwConfig = None
thermostatStatus = None
entityMib = None

#--------------------------------------------------
# 'show system environment temperature'
#
# [legacy]
# 'show environment temperature'
#--------------------------------------------------

ActionOnOverheat = TacLazyType( "Environment::Thermostat::ActionOnOverheat" )

def tempSanitize( temperature ):
   # We cannot set model attributes to -infinity
   return temperature if temperature != float( "-INFINITY" ) else None

def _populateModuleTempSensorInfo( modelSlot, tempSensors, tempConfig,
                                   archerCellStatus, archerSystemStatus,
                                   detail,
                                   isXcvrDomTempSensor=False ):

   tempSensors.sort( key=lambda ts: ts.relPos )
   for tempSensor in tempSensors:
      name = EntityMib.componentName( tempSensor )
      ths = thermostatStatus.tempSensorPidStatus.get( name )
      tsc = tempConfig.tempSensor.get( name )
      tss = archerSystemStatus.get( name )
      if not tss:
         for cell in archerCellStatus.values():
            tss = cell.get( name )
            if tss:
               break

      if not tsc or not tss:
         # sensor has been removed
         continue

      sensor = EnvironmentModels.TempSensor( _renderSensorDetail=bool( detail ),
                                             _isXcvrDomTempSensor=bool(
                                                isXcvrDomTempSensor ) )

      sensor.relPos = tempSensor.label
      sensor.name = name
      sensor.description = tsc.description
      if tss.hwStatus != 'disabled':
         sensor.currentTemperature = tempSanitize( tss.temperature )
         sensor.maxTemperature = tempSanitize( tss.maxTemperature )
      sensor.overheatThreshold = tempSanitize( tsc.overheatThreshold )
      sensor.criticalThreshold = tempSanitize( tsc.criticalThreshold )
      sensor.targetTemperature = tempSanitize( tsc.targetTemperature )

      #Tac.now() is a monotonic counter, we need to convert the value into utc
      #converting a given TS into utc
      #TS in utc = Tac.utcNow() - ( Tac.noww() - (TS in monotonic time ) )
      utcConversionFactor = Tac.utcNow() - Tac.now()

      # There is a short period (a few seconds ) of time when the thermostat agent
      # has not added the new temp sensor to the tempSensorPidStatus list yet.
      if ths is not None:
         sensor.setPointTemperature = tempSanitize( ths.setPointTemperature )
         sensor.isPidDriver = bool( ths.pidDriver )
         sensor.pidDriverCount = ths.pidDriverCount

         if ths.lastPidDriverTime != 0:
            sensor.lastPidDriverTime = ( utcConversionFactor
                                         + ths.lastPidDriverTime )

      sensor.hwStatus = tss.hwStatus
      if tss.maxTemperatureTime != 0:
         sensor.maxTemperatureLastChange = ( utcConversionFactor
                                             + tss.maxTemperatureTime )
      sensor.inAlertState = bool( tss.alertRaised )
      sensor.alertCount = tss.alertRaisedCount

      if tss.lastAlertRaisedTime != 0:
         sensor.alertLastChange = ( utcConversionFactor + tss.lastAlertRaisedTime )

      modelSlot.tempSensors.append( sensor )

def _powerLoss( slot ):
   supply = slot.powerSupply
   psName = 'PowerSupply%s' % supply.label # pylint: disable=consider-using-f-string
   supplyStatus = powerStatusDir.get( psName )
   return supplyStatus.state == 'powerLoss'

def doShowTemperature( mode, args ):
   module = args.get( 'MODULE' )
   detail = 'detail' in args
   # module is a named tuple with slot, tag and label as its members
   entityMibRoot = entityMib.root
   model = EnvironmentModels.SystemTemperature( _renderDetail=detail,
         _renderModule=bool( module ) )
   model.systemStatus = thermostatStatus.temperatureAlarmLevel
   action = thermostatStatus.actionOnOverheat
   model.shutdownOnOverheat = action == ActionOnOverheat.actionShutdown
   model.powercycleOnOverheat = action == ActionOnOverheat.actionPowercycle
   model.actionOnOverheat = action
   model.recoveryModeOnOverheat = thermostatConfig.recoveryModeOnOverheat
   model.ambientThreshold = thermostatConfig.ambientThreshold

   def _populatePsSensors( model, detail ):
      powerSupplySlots = sorted( entityMibRoot.powerSupplySlot.values(),
         key=lambda slot: slot.relPos )

      for slot in powerSupplySlots:
         supply = slot.powerSupply
         if supply:
            modelPsSlot = EnvironmentModels.FruSlot()
            tempSensors = [ sensor for sensor in supply.sensor.values()
                            if sensor.tag == "TempSensor" ]
            if tempSensors:
               # Power supply is inserted
               modelPsSlot.entPhysicalClass = slot.tag
               modelPsSlot.relPos = slot.label
               _populateModuleTempSensorInfo( modelPsSlot, tempSensors,
                                              temperatureConfig,
                                              archerCellTempStatus,
                                              archerSystemTempStatus,
                                              detail=detail )
               model.powerSupplySlots.append( modelPsSlot )

   def _populateCardSlotSensors( model, detail ):
      cardSlots = sorted( entityMibRoot.cardSlot.values(),
         key=lambda slot: slot.relPos )
      for slot in cardSlots:
         if slot.card:
            # Card is inserted
            modelCardSlot = EnvironmentModels.FruSlot()
            modelCardSlot.entPhysicalClass = slot.tag
            modelCardSlot.relPos = slot.label
            tempSensors = [ sensor for sensor in slot.card.sensor.values()
                            if sensor.tag == "TempSensor" ]
            if tempSensors:
               _populateModuleTempSensorInfo( modelCardSlot, tempSensors,
                                              temperatureConfig,
                                              archerCellTempStatus,
                                              archerSystemTempStatus,
                                              detail=detail )
            model.cardSlots.append( modelCardSlot )

   if entityMibRoot is None or entityMibRoot.initStatus != "ok":
      if ( mode.session_ and
            mode.session_.entityManager_.redundancyStatus().mode == 'active' ):
         mode.addError( "System is not yet initialized." )
         return None

   if module:
      tempSensors = []
      modelCardSlot = EnvironmentModels.FruSlot()
      if module.tag == 'PowerSupply':
         if module.powerSupply:
            modelCardSlot.entPhysicalClass = module.tag
            modelCardSlot.relPos = module.label
            tempSensors = [ sensor for sensor in module.powerSupply.sensor.values()
                            if sensor.tag == "TempSensor" ]
      elif module.slot.card:
         modelCardSlot.entPhysicalClass = module.slot.tag
         modelCardSlot.relPos = module.slot.label
         tempSensors = [ sensor for sensor in module.slot.card.sensor.values()
                         if sensor.tag == "TempSensor" ]
      if tempSensors:
         _populateModuleTempSensorInfo( modelCardSlot, tempSensors,
                                        temperatureConfig, archerCellTempStatus,
                                        archerSystemTempStatus, detail=detail )
      else:
         mode.addError( "Invalid module" )
         return None
      model.cardSlots.append( modelCardSlot )

   elif entityMibRoot.tacType.fullTypeName == "EntityMib::Chassis":
      _populateCardSlotSensors( model, detail=detail )
      _populatePsSensors( model, detail=detail )
   elif entityMibRoot.tacType.fullTypeName == "EntityMib::FixedSystem":
      tempSensors = [ sensor for sensor in entityMibRoot.sensor.values()
                      if sensor.tag == "TempSensor" ]
      #We can pass model here instead of a CardSlot since model also
      #has an optional TempSensors List (for fixed systems)
      _populateModuleTempSensorInfo( model, tempSensors, temperatureConfig,
                                     archerCellTempStatus, archerSystemTempStatus,
                                     detail=detail )
      _populateCardSlotSensors( model, detail=detail )
      _populatePsSensors( model, detail=detail )

   return model

#--------------------------------------------------
# 'show system environment temperature transceiver'
#
# [legacy]
# 'show environment temperature transceiver'
#--------------------------------------------------

def doShowTemperatureXcvr( mode, args ):
   module = args.get( 'MODULE' )
   detail = 'detail' in args
   # module is a named tuple with slot, tag and label as its members
   entityMibRoot = entityMib.root
   model = EnvironmentModels.SystemXcvrTemperature( _renderDetail=detail,
                                                    _renderModule=bool( module ) )

   if entityMibRoot is None or entityMibRoot.initStatus != "ok":
      if ( mode.session_ and
            mode.session_.entityManager_.redundancyStatus().mode == 'active' ):
         mode.addError( "System is not yet initialized." )
         return None

   def _getXcvrDomTempSensor( xcvrSlots ):
      if xcvrSlots is None:
         return []
      xcvrSlots = sorted( xcvrSlots.values(), key=lambda slot: slot.relPos )
      tempSensors = []
      for slot in xcvrSlots:
         if slot.xcvr != None: # pylint: disable=singleton-comparison
            sensors = slot.xcvr.sensor
            for sensor in sensors.values():
               if sensor.tag == "DomTemperatureSensor":
                  tempSensors.append( sensor )
      return tempSensors

   if module:
      modelCardSlot = EnvironmentModels.FruSlot()
      if module.tag == 'PowerSupply':
         pass
      elif module.slot.card:
         modelCardSlot.entPhysicalClass = module.slot.tag
         modelCardSlot.relPos = module.slot.label
         xcvrDomTempSensors = _getXcvrDomTempSensor( module.slot.card.xcvrSlot )
         _populateModuleTempSensorInfo( modelCardSlot,
                                        xcvrDomTempSensors, temperatureConfig,
                                        archerCellTempStatus,
                                        archerSystemTempStatus, detail=detail,
                                        isXcvrDomTempSensor=True )
         model.cardSlots.append( modelCardSlot )
   elif entityMibRoot.tacType.fullTypeName == "EntityMib::Chassis":
      cardSlots = sorted( entityMibRoot.cardSlot.values(),
         key=lambda slot: slot.relPos )
      for slot in cardSlots:
         if slot.card:
            # Card is inserted
            modelCardSlot = EnvironmentModels.FruSlot()
            if slot.tag.lower().find( "linecard" ) == -1:
               continue
            modelCardSlot.entPhysicalClass = slot.tag
            modelCardSlot.relPos = slot.label
            xcvrDomTempSensors = _getXcvrDomTempSensor( slot.card.xcvrSlot )
            _populateModuleTempSensorInfo( modelCardSlot, xcvrDomTempSensors,
                                           temperatureConfig,
                                           archerCellTempStatus,
                                           archerSystemTempStatus,
                                           detail=detail,
                                           isXcvrDomTempSensor=True )
            model.cardSlots.append( modelCardSlot )
   elif entityMibRoot.tacType.fullTypeName == "EntityMib::FixedSystem":
      xcvrDomTempSensors = _getXcvrDomTempSensor( entityMibRoot.xcvrSlot )
      _populateModuleTempSensorInfo( model, xcvrDomTempSensors,
                                     temperatureConfig, archerCellTempStatus,
                                     archerSystemTempStatus, detail=detail,
                                     isXcvrDomTempSensor=True )

   return model

#--------------------------------------------------
# 'show system environment cooling'
#
# [legacy]
# 'show environment cooling'
#--------------------------------------------------

def _populateFanTrayInfo( coolingModel, fanTraySlots, powerSupplySlots, coolConfig,
      archerCoolStatus, thermoStatus ):
   def labelNaturalOrderKey( item ):
      return naturalOrderKey( item.label )

   def _populateFanTrayInfoHelper( modelSlot, slot, labelPrefix="" ):
      supplyStatus = None
      if hasattr( slot, "fanTray" ):
         fanTray = slot.fanTray
      elif hasattr( slot, "powerSupply" ):
         fanTray = slot.powerSupply
         if fanTray:
            # pylint: disable-next=consider-using-f-string
            psName = 'PowerSupply%s' % fanTray.label
            supplyStatus = powerStatusDir.get( psName )
      else:
         assert 0, "Invalid parameter passed to _printFanTrayInfoHelper"
      if not fanTray or fanTray.initStatus != "ok":
         modelSlot.label = labelPrefix + EntityMib.formatEntityMibName(
            EntityMib.componentName( slot ) )
         modelSlot.status = "notInserted"
      else:
         fans = sorted( fanTray.fan.values(), key=labelNaturalOrderKey )
         modelSlot.label = labelPrefix + EntityMib.formatEntityMibName(
            EntityMib.componentName( fanTray ) )
         _populateFanAggregatedInfo( modelSlot, fans, coolConfig, thermoStatus,
                                     archerCoolStatus, supplyStatus )
         for fan in fans:
            modelFan = EnvironmentModels.Fan()
            _populateFanDetailedInfo( modelFan, fan, coolConfig,
                                      thermoStatus, archerCoolStatus,
                                      supplyStatus )
            modelFan.label = labelPrefix + EntityMib.formatEntityMibName(
               EntityMib.componentName( fan ) )
            modelSlot.fans.append( modelFan )

   # populate the info for all fan tray slots
   fanTraySlots.sort( key=labelNaturalOrderKey )
   for fanTraySlot in fanTraySlots:
      modelFanCollection = EnvironmentModels.FanCollection()
      _populateFanTrayInfoHelper( modelFanCollection, fanTraySlot )
      coolingModel.fanTraySlots.append( modelFanCollection )

   # populate info for all power supply slots if they are managed
   powerSupplySlots.sort( key=labelNaturalOrderKey )
   for powerSupplySlot in powerSupplySlots:
      if _powerSupplyFansAreManaged( powerSupplySlot, coolConfig ):
         modelPsSlot = EnvironmentModels.FanCollection()
         _populateFanTrayInfoHelper( modelPsSlot, powerSupplySlot,
                                             labelPrefix="PowerSupply" )
         coolingModel.powerSupplySlots.append( modelPsSlot )

def _powerSupplyFansAreManaged( powerSupplySlot, coolConfig ):
   if not powerSupplySlot.powerSupply:
      # Not inserted. Assume fans are managed
      return True
   for fan in powerSupplySlot.powerSupply.fan.values():
      name = EntityMib.componentName( fan )
      fanConfig = coolConfig.fan.get( name )
      if fanConfig.managed:
         # If 1 fan is managed, assume they're all managed
         return True
   return False

# get the fan's status and speed
def _fanStatusStr( fanConfig, fanStatus, supplyStatus=None ):
   if not fanStatus or not fanConfig or fanStatus.hwStatus == "unknownHwStatus":
      return "unknownHwStatus"
   elif not fanConfig.properties:
      return "unsupported"
   elif fanStatus.hwStatus == "failed":
      return "failed"
   elif ( fanConfig.offOnPowerLoss and
          supplyStatus and supplyStatus.state == "powerLoss" ):
      return "powerLoss"
   else:
      return "ok"

def _populateFanAggregatedInfo( modelSlot, fans, coolConfig,
                       thermoStatus, archerCoolStatus, supplyStatus=None ):

   # Get the fan speed from one of the fans
   name = EntityMib.componentName( fans[ 0 ] )
   fanConfig = coolConfig.fan.get( name )
   fanStatus = archerCoolStatus.get( name )
   if fanConfig:
      if not fanConfig.readOnly:
         cfg = thermoStatus.fanConfig.get( name )
         if cfg:
            modelSlot.speed = int( cfg.speed )
      if ( fanConfig.offOnPowerLoss and
           supplyStatus and supplyStatus.state == "powerLoss" ):
         modelSlot.speed = None

   fanStatusStrs = []
   for fan in fans:
      name = EntityMib.componentName( fan )
      fanConfig = coolConfig.fan.get( name )
      fanStatus = archerCoolStatus.get( name )
      fanStatusStrs.append( _fanStatusStr( fanConfig, fanStatus, supplyStatus ) )

   sev = { "failed" : 0,
           "unsupported" : 1,
           "unknownHwStatus" : 2,
           "powerLoss" : 3,
           "ok" : 4 }
   fanStatusStrs.sort( key=lambda s: sev[ s ] )
   modelSlot.status = fanStatusStrs[0]

def _populateFanDetailedInfo( modelFan, fan, coolConfig,
                       thermoStatus, archerCoolStatus, supplyStatus=None ):
   name = EntityMib.componentName( fan )
   fanConfig = coolConfig.fan.get( name )
   fanStatus = archerCoolStatus.get( name )

   if fanStatus:
      modelFan.actualSpeed = int( fanStatus.speed )

   if fanConfig:
      if not fanConfig.readOnly:
         cfg = thermoStatus.fanConfig.get( name )
         if cfg:
            modelFan.configuredSpeed = int( cfg.speed )
      if fanConfig.properties:
         modelFan.maxSpeed = int( fanConfig.properties.maxRpm )
      if ( fanConfig.offOnPowerLoss and
           supplyStatus and supplyStatus.state == "powerLoss" ):
         modelFan.configuredSpeed = None
         modelFan.actualSpeed = None

   modelFan.status = _fanStatusStr( fanConfig, fanStatus, supplyStatus )
   if modelFan.status == "ok" and fanStatus:
      # Saved TS was in local time, convert that to UTC
      deltaTime = Tac.now() - fanStatus.lastHwStatusChangeTime
      modelFan.uptime = Tac.utcNow() - deltaTime
      modelFan.speedStable = fanStatus.speedStable
      if modelFan.speedStable:
         deltaTime = Tac.now() - fanStatus.lastSpeedStableChangeTime
         modelFan.lastSpeedStableChangeTime = Tac.utcNow() - deltaTime
      modelFan.speedHwOverride = fanStatus.fanSpeedOverriddenByHardware

def _isNorthfaceChassis():
   return entityMib and entityMib.chassis and \
         entityMib.chassis.modelName == "DCS-7808-CH"

def _hasAmbientTempSensor():
   if thermostatStatus.airflowDirection == 'frontToBackAirflow':
      expectedPosition = 'inlet'
   else:
      expectedPosition = 'outlet'

   return any( s.position == expectedPosition
            for s in temperatureConfig.tempSensor.values() )

def doShowCooling( mode, args ):
   entityMibRoot = entityMib.root
   model = EnvironmentModels.SystemCooling( _renderFanDetail='detail' in args )
   if entityMibRoot is None or entityMibRoot.initStatus != "ok":
      if ( mode.session_ and
            mode.session_.entityManager_.redundancyStatus().mode == 'standby' ):
         mode.addError( "This command only works on the active supervisor." )
      else:
         mode.addError( "System is not yet initialized." )
      return None

   model.numCoolingZones.extend( coolingDomainDir.supportedProfile.values() )

   if coolingDomainDir.coolingDomainProfileName and \
         coolingDomainDir.coolingDomainProfileName in \
         coolingDomainDir.supportedProfile:
      model.currentZones = coolingDomainDir.supportedProfile[
            coolingDomainDir.coolingDomainProfileName ]
      if thermostatConfig.cliNumCoolingZones:
         model.configuredZones = thermostatConfig.cliNumCoolingZones
         model.defaultZones = False
      else:
         model.configuredZones = coolingDomainDir.supportedProfile[
               coolingDomainDir.defaultCoolingDomainProfileName ]
         model.defaultZones = True
   else:
      model.currentZones = 1
      model.defaultZones = False
      model.configuredZones = 0

   # Do not print configured cooling zones on platforms where
   # cooling zones are not configurable
   if not _isNorthfaceChassis():
      model.configuredZones = 0

   model.systemStatus = thermostatStatus.coolingAlarmLevel
   model.fansStatus = thermostatStatus.fanAlarmLevel
   model.ambientTemperature = tempSanitize( thermostatStatus.inletTemperature )
   model.airflowDirection = thermostatStatus.airflowDirection
   model.shutdownOnInsufficientFans = thermostatConfig.shutdownOnInsufficientFans
   model.coolingMode = thermostatConfig.mode
   model.overrideFanSpeed = int( thermostatConfig.fanSpeed )
   model.minFanSpeed = int( thermostatConfig.userMinFanSpeed )

   fanTraySlots = list( entityMibRoot.fanTraySlot.values() )
   powerSupplySlots = list( entityMibRoot.powerSupplySlot.values() )
   _populateFanTrayInfo( model, fanTraySlots, powerSupplySlots, coolingConfig,
         archerCoolingStatus, thermostatStatus )
   return model

#--------------------------------------------------
# 'show system environment all'
#
# [legacy]
# 'show environment all'
#--------------------------------------------------

# This hook allows to add list of handlers which return Models
# with extra environment information
extraShowSystemEnvAllModelHandlerHook = CliExtensions.CliHook()

def doShowEnvironmentAll( mode, args ):
   model = EnvironmentModels.SystemEnvAll()
   model.temperature = doShowTemperature( mode, {} )
   model.temperatureXcvr = doShowTemperatureXcvr( mode, {} )
   model.cooling = doShowCooling( mode, {} )
   model.extra = []
   for handler in extraShowSystemEnvAllModelHandlerHook.extensions():
      extraModel = handler( mode, {} )
      if extraModel:
         model.extra.append( extraModel )
   return model

#--------------------------------------------------
# 'environment cooling zones <NUM>'
# '[no|default] environment cooling zones'
#--------------------------------------------------

def coolingZonesGuard( mode, token ):
   if _isNorthfaceChassis():
      return None

   return CliParser.guardNotThisPlatform

def getSupportedCoolingZones( mode, context ):
   def getDesc( numZones ):
      # pylint: disable-next=consider-using-f-string
      return "Use %d %s" % ( numZones, "zones" if numZones > 1 else "zone" )

   if not coolingDomainDir.supportedProfile:
      # List of supported profiles is not ready when startup-config commands
      # are executed. Fru plugins will validate profile name
      res = list( range( 1, MAX_NUM_OF_COOLING_ZONES + 1 ) )
   else:
      res = list( coolingDomainDir.supportedProfile.values() )

   return OrderedDict( [ ( str( i ), getDesc( i ) ) for i in res ] )

coolingDomainWarnings = ( "WARNING: Change will take effect only after the switch " \
      "reboot.", "Save config and reboot the switch." )

def setNumCoolingZones( mode, args ):
   numZones = int( args[ 'NUM' ] )
   profileName = numZonesToProfileName( numZones,
         coolingDomainDir.supportedProfile )
   thermostatConfig.cliNumCoolingZones = numZones
   if coolingDomainDir.coolingDomainProfileName != profileName:
      mode.addWarning( coolingDomainWarnings[ 0 ] )
      mode.addWarning( coolingDomainWarnings[ 1 ] )

def setDefaultNumCoolingZones( mode, args ):
   changed = ( coolingDomainDir.coolingDomainProfileName != \
         coolingDomainDir.defaultCoolingDomainProfileName )
   thermostatConfig.cliNumCoolingZones = 0

   if changed:
      mode.addWarning( coolingDomainWarnings[ 0 ] )
      mode.addWarning( coolingDomainWarnings[ 1 ] )

#--------------------------------------------------
# 'environment fan-speed override <fan speed>'
# 'environment fan-speed minimum <fan speed>'
# 'environment fan-speed auto'
# '[no|default] environment fan-speed'
#--------------------------------------------------

def overrideFanSpeed( mode, args ):
   print( "====================================================================" )
   print( "WARNING: Overriding the system fan speed is unsupported and should only "
      )
   print( "be done under the direction of an Arista Networks engineer." )
   print( "You can risk damaging hardware by setting the fan speed too low" )
   print( "and doing so without direction from Arista Networks can be grounds" )
   print( "for voiding your warranty." )
   print( "To set the fan speed back to automatic mode, use the" )
   print( "'environment fan-speed auto' command" )
   print( "====================================================================" )
   fanSpeed = args.get( 'FANSPEED' )
   thermostatConfig.fanSpeed = fanSpeed
   thermostatConfig.userMinFanSpeed = 0
   thermostatConfig.mode = 'manual'

def setMinFanSpeed( mode, args ):
   fanSpeed = args.get( 'FANSPEED' )
   thermostatConfig.fanSpeed = 0
   thermostatConfig.userMinFanSpeed = fanSpeed
   thermostatConfig.mode = 'automatic'

def noMinFanSpeed( mode, args ):
   thermostatConfig.fanSpeed = 0
   thermostatConfig.userMinFanSpeed = 0
   thermostatConfig.mode = 'automatic'

def autoFanSpeed( mode, args ):
   thermostatConfig.mode = 'automatic'
   thermostatConfig.userMinFanSpeed = 0
   thermostatConfig.fanSpeed = 0

#--------------------------------------------------
# '[no|default] environment overheat action power-cycle recovery'
#--------------------------------------------------

recoveryModeEnum = Tac.Type( "Environment::Thermostat::RecoveryModeOnOverheat" )

def overheatRecoveryGuard( mode, token ):
   if _hasAmbientTempSensor():
      return None

   return CliParser.guardNotThisPlatform

def setRecoveryOnOverheat( mode, args ):
   defaultAmbientThreshold = 45
   thermostatConfig.recoveryModeOnOverheat = recoveryModeEnum.recoveryModeRestricted
   thermostatConfig.actionOnOverheat = ActionOnOverheat.actionPowercycle
   thermostatConfig.ambientThreshold = defaultAmbientThreshold

def setNoOrDefaultRecoveryOnOverheat( mode, args ):
   defaultAmbientThreshold = 45
   thermostatConfig.recoveryModeOnOverheat = recoveryModeEnum.recoveryModeNA
   if thermostatHwConfig.supportsPowerOff:
      thermostatConfig.actionOnOverheat = ActionOnOverheat.actionShutdown
   else:
      thermostatConfig.actionOnOverheat = ActionOnOverheat.actionPowercycle
   thermostatConfig.ambientThreshold = defaultAmbientThreshold

#--------------------------------------------------
# '[no|default] environment overheat action power-cycle recovery restricted'
# ' ambient-threshold <THRESHOLD>'
#--------------------------------------------------

def setRecoveryAmbientThresholdOnOverheat( mode, args ):
   ambTh = args.get( 'THRESHOLD' )
   thermostatConfig.recoveryModeOnOverheat = recoveryModeEnum.recoveryModeRestricted
   thermostatConfig.actionOnOverheat = ActionOnOverheat.actionPowercycle
   thermostatConfig.ambientThreshold = ambTh

def setNoOrDefaultRecoveryAmbientThresholdOnOverheat( mode, args ):
   defaultAmbientThreshold = 45
   thermostatConfig.recoveryModeOnOverheat = recoveryModeEnum.recoveryModeNA
   if thermostatHwConfig.supportsPowerOff:
      thermostatConfig.actionOnOverheat = ActionOnOverheat.actionShutdown
   else:
      thermostatConfig.actionOnOverheat = ActionOnOverheat.actionPowercycle
   thermostatConfig.ambientThreshold = defaultAmbientThreshold

#--------------------------------------------------
# '[no|default] environment overheat action (shutdown|ignore|power-cycle)'
# '[no|default] environment insufficient-fans action (shutdown|ignore)'
#--------------------------------------------------

warning = """====================================================================
WARNING: Overriding the system shutdown behavior when the system
%s is unsupported and should only be done under
the direction of an Arista Networks engineer. You risk damaging
hardware by not shutting down the system in this situation, and doing
so without direction from Arista Networks can be grounds for voiding
your warranty. If in any circumstances the system loses all the fans,
it will shut down to prevent accidental burnout.
To re-enable the shutdown-on-overheat behavior, use the
'environment %s action shutdown' command.
===================================================================="""

def shutdownActionGuard( mode, token ):
   if thermostatHwConfig.supportsPowerOff:
      return None

   return CliParser.guardNotThisPlatform

def powercycleActionGuard( mode, token ):
   if not thermostatHwConfig.supportsPowerOff:
      return None

   return FruCli.fixedSystemGuard( mode, token )

def setActionOnOverheat( mode, args ):
   defaultAmbientThreshold = 45
   action = ''
   configAction = ActionOnOverheat.actionUnknown
   # Search args for any of the valid overheat actions
   for supportedAction in actionOnOverheatEnumMap:
      action = args.get( supportedAction )
      if action:
         configAction = actionOnOverheatEnumMap[ action ]
         break
   if action == 'ignore':
      mode.addWarning( warning % ( "is overheating", "overheat" ) )

   thermostatConfig.actionOnOverheat = configAction

   if thermostatConfig.actionOnOverheat == ActionOnOverheat.actionPowercycle:
      thermostatConfig.recoveryModeOnOverheat = recoveryModeEnum.recoveryModeNormal
   else:
      thermostatConfig.recoveryModeOnOverheat = recoveryModeEnum.recoveryModeNA
   thermostatConfig.ambientThreshold = defaultAmbientThreshold

def setNoOrDefaultActionOnOverheat( mode, args ):
   defaultAmbientThreshold = 45
   if thermostatHwConfig.supportsPowerOff:
      thermostatConfig.actionOnOverheat = ActionOnOverheat.actionShutdown
   else:
      thermostatConfig.actionOnOverheat = ActionOnOverheat.actionPowercycle
   thermostatConfig.recoveryModeOnOverheat = recoveryModeEnum.recoveryModeNA
   thermostatConfig.ambientThreshold = defaultAmbientThreshold

def setShutdownOnInsufficientFans( mode, args ):
   # the default is that we shutdown on insufficient fans
   action = args.get( 'ACTION', 'shutdown' )
   if action == 'ignore':
      mode.addWarning( warning % ( "has insufficient fans inserted",
                                   "insufficient-fans" ) )
   thermostatConfig.shutdownOnInsufficientFans = ( action == 'shutdown' )

#------------------------------------------------------
# Register env show commands into "show tech-support".
#------------------------------------------------------

# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2010-01-01 00:15:00',
   cmds=[ 'show system env cooling',
          'show system env temperature',
          'show system env temperature transceiver' ],
   summaryCmds=[ 'show system env temperature transceiver',
                 'show system environment all' ] )

#--------------------------------------------------------------------------------
# '[ no | default ] environment temperature invalid action ignore'
# '[ no | default ] environment temperature transceiver third-party invalid
#  action ignore'
#--------------------------------------------------------------------------------

def setIgnoreTempTrue( mode, args ):
   thermostatConfig.ignoreInvalidTemp = True

def setIgnoreTempFalse( mode, args ):
   thermostatConfig.ignoreInvalidTemp = False

def setIgnoreThirdPartyXcvrTrue( mode, args ):
   thermostatConfig.ignoreInvalidThirdPartyXcvrTemp = True

def setIgnoreThirdPartyXcvrFalse( mode, args ):
   thermostatConfig.ignoreInvalidThirdPartyXcvrTemp = False

#--------------------------------------------------------------------------------
# [ no | default ] environment temperature poll-interval
#--------------------------------------------------------------------------------
def setTemperaturePollInterval( mode, args ):
   interval = args.get( 'INTERVAL', DEFAULT_TEMPERATURE_POLL_INTERVAL )
   if 'milliseconds' in args:
      interval /= 1000.0
   thermostatConfig.pollInterval = interval

#--------------------------------------------------
# Plugin method - Mount the objects we need from Sysdb
#--------------------------------------------------
def Plugin( entityManager ):
   global temperatureConfig, coolingConfig, coolingDomainDir
   global archerCellTempStatus, archerSystemTempStatus
   global archerCoolingStatus, powerStatusDir
   global thermostatConfig, thermostatHwConfig, thermostatStatus, entityMib
   temperatureConfig = LazyMount.mount( entityManager,
                                        "environment/temperature/config",
                                        "Environment::Temperature::Config", "r" )
   archerCellTempStatus = LazyMount.mount( 
      entityManager, "environment/archer/temperature/status/cell", "Tac::Dir", "ri" )
   archerSystemTempStatus = LazyMount.mount( 
      entityManager, "environment/archer/temperature/status/system",
      "Tac::Dir", "ri" )
   coolingConfig = LazyMount.mount( entityManager, "environment/cooling/config",
                                    "Environment::Cooling::Config", "r" )
   archerCoolingStatus = LazyMount.mount(
      entityManager, "environment/archer/cooling/status", "Tac::Dir", "ri" )
   thermostatConfig = ConfigMount.mount( entityManager,
                                         "environment/thermostat/config",
                                         "Environment::Thermostat::Config", "w" )
   thermostatHwConfig = LazyMount.mount( entityManager,
                                         "environment/thermostat/hwconfig",
                                         "Environment::Thermostat::HwConfig", "r" )
   thermostatStatus = LazyMount.mount( entityManager,
                                       "environment/thermostat/status",
                                       "Environment::Thermostat::Status", "r" )
   powerStatusDir = LazyMount.mount( entityManager,
                                     'environment/archer/power/status/powerSupply',
                                     'Tac::Dir', 'ri' )
   entityMib = LazyMount.mount( entityManager, "hardware/entmib",
                                "EntityMib::Status", "r" )
   coolingDomainDir = LazyMount.mount( entityManager,
         "environment/thermostat/coolingDomain",
         "Environment::Thermostat::CoolingDomainDir", "r" )
