#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=bad-string-format-type

from CliModel import Bool
from CliModel import Enum
from CliModel import Float
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from CliModel import Tac
import Ark
import EnvironmentUtils
import datetime
from TableOutput import createTable, Format

def _printXcvrDomTempSensorBanner():
   print( """\
                                                         Alert  Critical
                                       Temp    Setpoint  Limit     Limit
Description                             (C)         (C)    (C)       (C)
----------------------------------- ------- ----------- ------ ---------""" )

def _printSensorBanner():
   print ( "                                                   "
      "              Alert  Critical" )
   print ( "                                               Temp"
      "    Setpoint  Limit     Limit" )
   print ( "Sensor  Description                             (C)"
      "         (C)    (C)       (C)" )
   print ( "------- ----------------------------------- -------"
      " ----------- ------ ---------" )

def makeTable( header ):
   t = createTable( header )
   fl = Format( justify="left" )
   fl.noPadLeftIs( True )
   fl.padLimitIs( True )
   fr = Format( justify="right" )
   fr.noPadLeftIs( True )
   fr.padLimitIs( True )
   t.formatColumns( fl, fl, fr, fr, fr, fl, fr )
   return t

def printTable( table ):
   # Strip whitespace from the end of the lines, because that's how it was
   # before and it's what some parsers expect. Also remove the last newline.
   print( '\n'.join( map( str.rstrip, table.output().split( '\n' )[ :-1 ] ) ) )


class TempSensor( Model ):
   """Temperature sensor in the system"""

   _renderSensorDetail = Bool(
         help="Render details of the temperature sensor if True",
         default=False )
   relPos = Str( help="Temperature sensor relative position" )
   name = Str( help="Temperature sensor name" )
   description = Str( help="Temperature sensor description" )
   overheatThreshold = Float(
         help="Configured temperature threshold to trigger an overheating event (in"
         "Celsius)", optional=True )
   criticalThreshold = Float(
         help="Configured critical temperature threshold to trigger a system "
         "shutdown (in Celsius)", optional=True )
   targetTemperature = Float(
         help="Configured target temperature used for PID tuning (in Celsius) ",
         optional=True )
   hwStatus = Enum( values=EnvironmentUtils.getSensorHwStatusStrings(),
         help="Hardware status reported by the temperature sensor" )
   currentTemperature = Float(
         help="Current temperature reported by the temperature sensor "
         "(in Celsius)", optional=True )
   setPointTemperature = Float(
         help="Current setpoint temperature used by the temperature sensor "
         "(in Celsius)", optional=True )
   maxTemperature = Float( help="Maximum temperature recorded by the temperature "
   "sensor (in Celsius)", optional=True )
   maxTemperatureLastChange = Float( help="Last time the maximum temperature "
         "recorded by the temperature sensor changed in POSIX time",
         optional=True )
   inAlertState = Bool( help="Whether the system is in alert state" )
   alertCount = Int( help="Number of times the alert has been triggered" )
   alertLastChange = Float(
         help="Last time the alert status changed in POSIX time",
         optional=True )
   isPidDriver = Bool( help="Whether the sensor is currently the PID driver",
                       default=False )
   pidDriverCount = Int( help="Number of times the sensor has been the PID driver",
                         default=0 )
   lastPidDriverTime = Float(
         help="Last time the sensor was the PID driver",
         optional=True )
   _isXcvrDomTempSensor = Bool( help="Whether the sensor is on Xcvr", default=False )
   def _tempStr( self, temperature, fmt, default="Unknown" ):
      # pylint: disable-next=singleton-comparison
      return fmt % temperature if temperature != None else default

   def render( self ):

      if self._renderSensorDetail:
         print( "%s - %s" % ( self.name, self.description ) )
         fmt = "  %-27.27s %16.16s %10.10s %22.22s"

         tempStr = self._tempStr( self.currentTemperature, "%0.3fC" )
         maxTempStr =  self._tempStr( self.maxTemperature, "%0.3fC" )
         #Convert TS to monotonic time as returned by Tac.now()
         utcConversionFactor = Tac.utcNow() - Tac.now()
         maxTempLastChangeUtc = self.maxTemperatureLastChange

         if maxTempLastChangeUtc:
            maxTempLastChange = maxTempLastChangeUtc - utcConversionFactor
         else:
            maxTempLastChange = 0

         alertLastChangeUtc = self.alertLastChange
         if alertLastChangeUtc:
            alertLastChange = alertLastChangeUtc - utcConversionFactor
         else:
            alertLastChange = 0

         lastPidDriverTimeUtc = self.lastPidDriverTime
         if lastPidDriverTimeUtc:
            lastPidDriverTime = lastPidDriverTimeUtc - utcConversionFactor
         else:
            lastPidDriverTime = 0

         print( fmt % ( "", "Current State", "Count", "Last Change" ) )
         print( fmt % ( "Temperature", tempStr, "", "" ) )
         print( fmt % ( "Max Temperature", maxTempStr, "",
                      Ark.timestampToStr( maxTempLastChange ) ) )
         print( fmt % ( "Alert", self.inAlertState,
                      self.alertCount,
                      Ark.timestampToStr( alertLastChange ) ) )
         print( fmt % ( "PID Driver", self.isPidDriver,
                      self.pidDriverCount,
                      Ark.timestampToStr( lastPidDriverTime ) ) )
         print( "" )

      else:
         if self.currentTemperature is None:
            tempStr = 'N/A'
         else:
            if self.hwStatus == "ok":
               tempStr = self._tempStr( self.currentTemperature, "%0.1f" )
            else:
               tempStr = EnvironmentUtils.getSensorHwStatusDisplayString(
                     self.hwStatus )
         if self.targetTemperature is None:
            setPointTempStr = '(N/A)'
         elif self.targetTemperature == self.setPointTemperature:
            # Only display target temp if different from set point temp
            setPointTempStr = ""
         else:
            setPointTempStr = self._tempStr( self.targetTemperature, "(%d)" )
         if self.setPointTemperature is None:
            setPointTempStr += ' N/A'
         else:
            setPointTempStr += self._tempStr( self.setPointTemperature, " %0.1f" )
         if self._isXcvrDomTempSensor:
            print( "%-35.35s %7.7s %11.11s %6.6s %9.9s" % (
               self.description, tempStr, setPointTempStr,
               self._tempStr( self.overheatThreshold, "%d" ),
               self._tempStr( self.criticalThreshold, "%d" ) ) )
         else:
            print( "%-7.7s %-35.35s %7.7s %11.11s %6.6s %9.9s" % (
               self.relPos, self.description, tempStr, setPointTempStr,
               self._tempStr( self.overheatThreshold, "%d" ),
               self._tempStr( self.criticalThreshold, "%d" ) ) )

class FruSlot( Model ):
   """Information for a generic fru slot in the system"""

   relPos = Str( help = "Slot relative position" )
   entPhysicalClass = Str( help="Physical entity class of the slot" )
   tempSensors = List( valueType=TempSensor,
         help="The temperatures sensors in the card slot" )

   def render( self ):
      for tempSensor in self.tempSensors:
         tempSensor.render()

class SystemXcvrTemperature( Model ):
   """Transceiver temperature information for the system"""
   _renderDetail = Bool(
      help="Render details of the temperature sensors if True",
         default=False )
   _renderModule = Bool( help="Render details for a given module if True",
         default=False )
   tempSensors = List( valueType=TempSensor,
         help="Temperature sensors on a fixed system" )
   cardSlots = List( valueType=FruSlot,
         help="Card slots on a modular system" )

   def render( self ):
      for slot in self.cardSlots:
         if not self._renderModule:
            print( "" )
            print( "%s %s:" % ( slot.entPhysicalClass, slot.relPos ) )
         if not slot.tempSensors:
            continue
         if not self._renderDetail:
            _printXcvrDomTempSensorBanner()
         slot.render( )

      if self.tempSensors and not self._renderDetail:
         _printXcvrDomTempSensorBanner()
      for sensor in self.tempSensors:
         sensor.render()

class SystemTemperature( Model ):
   """Temperature information for the system"""

   _renderDetail = Bool(
         help="Render details of the temperature sensors if True",
         default=False )
   _renderModule = Bool( help="Render details for a given module if True",
         default=False )
   systemStatus = Enum(
         values=EnvironmentUtils.getTempAlarmLevelStrings(),
         help="Temperature status of the system" )
   shutdownOnOverheat = Bool( help="Whether the system is configured to "
         "shutdown on an overheat event" )
   powercycleOnOverheat = Bool( help="Whether the system is configured to "
         "power-cycle on an overheat event" )
   actionOnOverheat = Enum(
         values=EnvironmentUtils.getActionOnOverheatStrings(),
         help="System action for an overheat event" )
   recoveryModeOnOverheat = Enum(
         values=EnvironmentUtils.getRecoveryModeOnOverheatStrings(),
         help="Specify recovery behavior after power-cycle" )
   ambientThreshold = Int( help="Ambient temperature (in Celsius) above which EOS"
                                " will run in reduced capacity" )
   tempSensors = List( valueType=TempSensor,
         help="Temperature sensors on a fixed system" )
   cardSlots = List( valueType=FruSlot,
         help="Card slots on a modular system" )
   powerSupplySlots = List( valueType=FruSlot,
         help="Power Supply Slots on the system" )

   def render( self ):

      if not self._renderModule and not self._renderDetail:
         print ( "System temperature status is: %s"
               % EnvironmentUtils.getTempAlarmLevelDisplayString( self.systemStatus )
               )

         actionOnOverheat = EnvironmentUtils.getActionOnOverheatDisplayString(
                              self.actionOnOverheat )
         print( "Action on overheat:", actionOnOverheat )

         recoveryMode = EnvironmentUtils.getRecoveryModeOnOverheatDisplayString(
                              self.recoveryModeOnOverheat )
         print( "Recovery mode when power-cycle upon overheat:", recoveryMode )
         if self.recoveryModeOnOverheat == 'recoveryModeRestricted':
            print ( "Restrict recovery when ambient temperature is at or above: %sC"
                  % self.ambientThreshold )
         else:
            print ( "Restrict recovery when ambient temperature is at or above: "
                  "not applicable" )

      if self.tempSensors:
         if not self._renderDetail:
            _printSensorBanner()
         for tempSensor in self.tempSensors:
            tempSensor.render()

      if self.cardSlots:
         for slot in self.cardSlots:
            if not self._renderModule:
               print( "" )
               print( "%s %s:" % ( slot.entPhysicalClass, slot.relPos ) )
            if not self._renderDetail:
               _printSensorBanner()
            slot.render( )

      if self.powerSupplySlots:
         for slot in self.powerSupplySlots:
            if not self._renderModule:
               print( "" )
               print( "%s %s:" % ( slot.entPhysicalClass, slot.relPos ) )
            if not self._renderDetail:
               _printSensorBanner()
            slot.render( )

class Fan( Model ):
   """A Fan in the system"""

   label = Str( help="Label for the fan" )
   maxSpeed = Int( help="Maximum speed for the fan (in rpm)",
         optional=True )
   configuredSpeed = Int( help="Configured speed for the fan (as %)",
         optional=True )
   actualSpeed = Int( help="Actual speed of the fan (as %)",
         optional=True )
   status = Enum( values=EnvironmentUtils.getFanStatusStrings(),
         help="Status of the fan" )
   uptime = Float( help="Timestamp in UTC (in seconds) when the last change"
         " in Fan's status occurred",
         optional=True )
   speedStable = Bool( help="If the fan's speed is stable",
         optional=True )
   speedHwOverride = Bool( help="If the fan's speed is overridden by hardware",
         optional=True )
   lastSpeedStableChangeTime = Float( help="Timestamp in UTC (in seconds) when"
         " last time the fan's speed was stable.",
         optional=True )

class FanCollection( Model ):
   """Information for a collection of fans in the system"""

   label = Str( help = "Slot Label" )
   fans = List( valueType=Fan, 
         help="The fans in the fan tray" )
   status = Enum( values=EnvironmentUtils.getFanTrayStatusStrings(),
         help="The status of the fan tray" )
   speed = Int( help="The aggregated speed of the fans in the tray (as %)",
         optional=True )

def _renderFanCollection( slot, tableOutput, detail ):

   # if fanTray is missing or not initialized, print Not Inserted
   # since this is not a fan specific status, don't split into Tray/#
   # even if detail = True
   if slot.status == "notInserted":
      if detail:
         tableOutput.newRow( slot.label,
               EnvironmentUtils.getFanTrayStatusDisplayString( slot.status ),
               "N/A", "N/A", "N/A", "Offline", "N/A", "N/A" )
      else:
         tableOutput.newRow( slot.label,
               EnvironmentUtils.getFanTrayStatusDisplayString( slot.status ),
               "N/A", "N/A", "Offline", "N/A", "N/A" )
   else:
      for fan in slot.fans:
         configuredSpeedStr = ( "N/A"
               if fan.configuredSpeed is None
               else "%d%%" % fan.configuredSpeed )
         actualSpeedStr = ( "N/A"
               if fan.actualSpeed is None
               else "%d%%" % fan.actualSpeed )
         fanStatusString = EnvironmentUtils.getFanStatusDisplayString( fan.status )
         if fanStatusString == "Ok":
            upt = Tac.utcNow() - fan.uptime
            td = datetime.timedelta( seconds=int( upt ) )
            uptime = str( td )
            if fan.speedHwOverride:
               speedStable = "FW Override"
               stableTime = "N/A"
            elif fan.speedStable:
               speedStable = "Stable"
               stTime = Tac.utcNow() - fan.lastSpeedStableChangeTime
               td = datetime.timedelta( seconds=int( stTime ) )
               stableTime = str( td )
            else:
               speedStable = "Stabilizing"
               stableTime = "N/A"
         else:
            uptime = "Offline"
            speedStable = "N/A"
            stableTime = "N/A"
         if detail:
            maxSpeedStr = ( "N/A"
                  if fan.maxSpeed is None
                  else "%d" % fan.maxSpeed )
            tableOutput.newRow( fan.label, fanStatusString, maxSpeedStr,
                                configuredSpeedStr, actualSpeedStr, uptime,
                                speedStable, stableTime )
         else:
            tableOutput.newRow( fan.label, fanStatusString, configuredSpeedStr,
                                actualSpeedStr, uptime, speedStable, stableTime )

class SystemCooling( Model ):
   """Cooling information for the system"""

   systemStatus = Enum(
         values=EnvironmentUtils.getCoolingAlarmLevelStrings(),
         help="System cooling status" )
   fansStatus = Enum(
         values=EnvironmentUtils.getFanAlarmLevelStrings(),
         help="Fans status" )
   ambientTemperature = Float(
         help="Ambient temperature detected by the cooling system (in Celsius)", 
         optional=True )
   airflowDirection = Enum(
         values=EnvironmentUtils.getAirflowDirectionStrings(),
         help="The direction of air flow in the system (front refers to port-side)" )
   currentZones = Int(
         help="Current number of cooling zones",
         optional=True )
   configuredZones = Int(
         help="Configured number of cooling zones",
         optional=True )
   defaultZones = Bool(
         help="Default number of cooling zones is used",
         optional=True )
   numCoolingZones = List( valueType=int,
         help="Supported number of cooling zones" )
   shutdownOnInsufficientFans = Bool(
         help="Shut the system down on insufficient fans if True" )
   overrideFanSpeed = Int(
         help="Fan speed at which manual override is configured (as %)",
         optional=True )
   minFanSpeed = Int(
         help="User-set minimum fan speed (as %)",
         optional=True )
   coolingMode = Enum( values=( 'manual', 'automatic', 'debug' ),
         help="Mode of operation of the cooling system" )
   fanTraySlots = List( valueType=FanCollection,
         help="Fan tray slots in the system" )
   powerSupplySlots = List( valueType=FanCollection,
         help="Power supply slots in the system" )
   _renderFanDetail = Bool( 
         help="Render details of the fans if True",
         default=False )

   def render( self ):
      print ( "System cooling status is: %s"
            % EnvironmentUtils.getCoolingAlarmLevelDisplayString( self.systemStatus )
            )
      if self.ambientTemperature is not None:
         print( "Ambient temperature: %dC" % self.ambientTemperature )
      if not self.shutdownOnInsufficientFans:
         print( "Shutdown due to insufficient fans disabled" )
      if self.coolingMode == "manual":
         print( "Fan speed override mode enabled at %d%%" % self.overrideFanSpeed ) 
      if self.minFanSpeed > 0:
         print( "User minimum fan speed enabled at %d%%" % self.minFanSpeed ) 
      if self.coolingMode not in [ "automatic", "manual" ]:
         print( "Fan speed mode (%s) unsupported" % self.coolingMode )
      if self.airflowDirection != "unknownAirflowDirection":
         print( "Airflow: %s" % EnvironmentUtils.getAirflowDirectionDisplayString(
               self.airflowDirection ) )

      if self.currentZones:
         print( "Number of cooling zones is: %d" % self.currentZones )
         if self.configuredZones:
            defaultZones = ""
            if self.defaultZones:
               defaultZones = " (default)"
            print( "Configured number of cooling zones is: %d%s" % \
                   ( self.configuredZones, defaultZones ) )

      tableColumn = [ [ "", "hl", [ "Fan" ], ],
                      [ "", "hl", [ "Status" ], ],
                      [ "Config", "hr", [ "Speed" ], ],
                      [ "Actual", "hr", [ "Speed" ], ],
                      [ "", "hr", [ "Uptime" ], ],
                      [ "Speed", "hl", [ "Stability" ], ],
                      [ "Stable", "hr", [ "Uptime" ], ], ]
      # Add detailed data(only max speed for now) into the table
      if self._renderFanDetail:
         tableColumn.insert( 2, [ "Max", "hr", [ "Speed" ], ] )
      tableShowEnvironmentCooling = makeTable( tableColumn )

      for slot in self.fanTraySlots:
         _renderFanCollection( slot, tableShowEnvironmentCooling,
                               self._renderFanDetail )
      
      for slot in self.powerSupplySlots:
         _renderFanCollection( slot, tableShowEnvironmentCooling,
                               self._renderFanDetail )

      printTable( tableShowEnvironmentCooling )

class SystemEnvAllExtra( Model ):
   pass

class SystemEnvAll( Model ):
   temperature = Submodel( valueType=SystemTemperature,
         help="Temperature information for the system" )
   temperatureXcvr = Submodel( valueType=SystemXcvrTemperature,
         help="Transceiver temperature information for the system" )
   cooling = Submodel( valueType=SystemCooling,
         help="Cooling information for the system" )
   extra = List( valueType=SystemEnvAllExtra,
         help="Extended environment information" )

   def render( self ):
      for model in [ self.temperature, self.cooling ]:
         if model is not None:
            model.render()
            print()

      for model in self.extra:
         if model is not None:
            model.render()
            print()
