#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=bad-string-format-type
# pylint: disable=consider-using-f-string

from collections import OrderedDict

import Ark
import Arnet
from CliModel import (
   Bool,
   Dict,
   Enum,
   Float,
   Int,
   List,
   Model,
   Str,
   Submodel
)
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
from CliPlugin.IntfModel import (
   InterfaceCounters,
   InterfaceStatus
)
from datetime import datetime
import EthIntfLib
import HumanReadable
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
from IntfModels import Interface
import Tac
from TypeFuture import TacLazyType

class CapabilitiesMixIn:
   '''
      Provides utility functions used by InterfaceCapabilitiesData and
      InterfaceCapabilitiesDefaultData
   '''

   # speedValues used in CliModel EthSpeed
   speedValues = [ speedVal[ len( 'speed' ): ]
                 for speedVal in Tac.Type( 'Interface::EthSpeed' ).attributes ]

   flowControlCapsToStr = { 'flowControlCapable' : 'Capable',
                            'flowControlNotCapable' : 'Incapable',
                            'flowControlUnknown' : 'Unknown' }

   errorCorrectionEncodingToLbls = { 'ReedSolomon' : 'reed-solomon',
                                     'ReedSolomon544' : 'reed-solomon',
                                     'FireCode' : 'fire-code',
                                     'Disabled' : 'disabled' }

   errorCorrectionEncodingToDisplay = [ 'reed-solomon',
                                        'fire-code',
                                        'disabled' ]

   errorCorrectionValues = [ fecVal[ len( 'fecEncoding' ): ]
                 for fecVal in Tac.Type( 'Interface::EthFecEncoding' ).attributes 
                 if fecVal != 'fecEncodingUnknown' ]

   coherentModes = [ 'none', '16QAM', '8QAM', 'DP-QPSK' ]

   coherentFecs = [ 'RS G.709', 'SC 7%', 'SD 15%', 'SD 20%', 'SD 25%',
                    'SD 25% BCH', 'C-FEC', 'O-FEC' ]

   coherentLineRates = [ '100G', '200G', '300G', '400G' ]

   # expect: List of CapList
   # return: List of EthIntfMode, autoneg supported flag
   @staticmethod
   def linkModeCapsToEthSpeedModel( linkModeCaps, ethMode=False ):
      spModel = []
      autonegSupported = False
      if not linkModeCaps:
         return spModel, autonegSupported 

      for spMed in linkModeCaps: 
         esp = EthSpeedLanesDuplex()
         spMedSpeed = getattr( spMed, 'speed', 'speedUnknown' )
         if spMedSpeed == 'auto':
            esp.autoIs( True )
            autonegSupported = True
         else:
            esp.speed = CapabilitiesMixIn.ethSpeedToSpeedValue( spMedSpeed )
         spMedDuplex = getattr( spMed, 'duplex', 'duplexUnknown' )
         esp.duplex = spMedDuplex.lstrip( 'duplex' ).lower()
         esp.showLanesIs( getattr( spMed, 'showLanes', False ) )
         if esp.showLanes():
            esp.laneCount = int( CapabilitiesMixIn.ethLaneCountToValue(
                                 spMed.lanes ) )

         if ethMode:
            em = EthIntfMode()
            em.link = esp
            em.defaultLinkMode = getattr( spMed, 'default', False )
            em.pllIncompatibleIs( getattr( spMed, 'pllIncompatible', False ) )
            if em not in spModel:
               spModel.append( em )
         else:
            if esp not in spModel:
               spModel.append( esp )

      return spModel, autonegSupported

   # return: formated string to be printed out
   @staticmethod
   def formatCapabilitiesFieldStr( indtLvl, header, data ):
      output = ''
      indtStr = ' ' * indtLvl
      if header:
         headerStr = indtStr + header + ':'
         # Indent space is decided by the existing Cli outputs. We make it a fixed
         # number to avoid possible breaking of existing scripts
         fmtStr = '{hdr} {data}'
         output += fmtStr.format( hdr=headerStr, data=data ) + '\n'
      else:
         output += indtStr + data + '\n'
      return output

   # convert Interface::EthFecEncoding to errorCorrectionValues
   @staticmethod
   def ethFecEncodingToErrorCorrectionValue( fecEncoding ):
      assert fecEncoding in Tac.Type( 'Interface::EthFecEncoding' ).attributes
      return fecEncoding[ len( 'fecEncoding' ): ]

   # convert Interface::EthSpeed to speedValues
   @staticmethod
   def ethSpeedToSpeedValue( ethSpeed ):
      assert ethSpeed in Tac.Type( 'Interface::EthSpeed' ).attributes, \
         f"Unknown speed: '{ethSpeed}'"
      return ethSpeed[ len( 'speed' ): ]

   # convert Interface::EthLaneCount to laneCount values
   @staticmethod
   def ethLaneCountToValue( ethLaneCount ):
      assert ethLaneCount in Tac.Type( 'Interface::EthLaneCount' ).attributes, \
         f"Unknown lane count: '{ethLaneCount}'"
      if not ethLaneCount == 'laneCountUnknown':
         return int( ethLaneCount[ len( 'laneCount' ): ] )
      else:
         return 0

   # return: formated spped-lanes/duplex string for printing
   @staticmethod
   def formatSpeedLanesDuplex( spStr, lanes, showLanes, duplex, default,
                               pllIncompatible ):
      if not spStr:
         return ''
      if spStr.endswith( 'bps' ):
         tmpStr = spStr[ :-len( 'bps' ) ].replace( 'p', '.' )
      else:
         tmpStr = spStr.replace( 'p', '.' )

      if lanes and showLanes and not EthIntfLib.cliNeverShowLanes( tmpStr ):
         tmpStr = tmpStr + '-' + str( lanes )

      if duplex:
         tmpStr += '/'

      if pllIncompatible:
         tmpStr = tmpStr.replace( '/', '*/' )

      tmpStr += duplex

      if tmpStr == 'Unknown/unknown':
         tmpStr = 'unknown'

      if default:
         tmpStr = tmpStr + '(default)'

      return tmpStr

   # input: FlowControlCapabilities
   # return: flow control capabilities string
   @staticmethod
   def formatFlowControl( flowControl ):
      if flowControl.flowControl == 'Capable':
         fcSt = 'off,on'
         if flowControl.autoNegotiationPause:
            fcSt += ',desired'
      elif flowControl.flowControl == 'Incapable':
         fcSt = 'off'
      else:
         fcSt = 'unknown'

      return fcSt

   # input: list of EthIntfMode
   # return: dict with { fecType : speedStr } from EthIntfMode
   @staticmethod
   def ethIntfModeToFecTypeSpeedDict( ethIntfModes, defaultCheck=False ):
      if not ethIntfModes:
         return {}

      fecDict = OrderedDict()
      for mode in ethIntfModes:
         if not mode.errorCorrections or not mode.link:
            continue
         for fecSet in mode.errorCorrections:
            fecType = CapabilitiesMixIn.errorCorrectionEncodingToLbls[ fecSet ]
            speedStr = CapabilitiesMixIn.formatSpeedLanesDuplex(
               mode.link.speed, mode.link.laneCount, mode.link.showLanes(), "",
               defaultCheck and fecSet == mode.defaultErrorCorrection, False )
            tmpStr = fecDict.get( fecType, '' )
            if tmpStr != '':
               tmpStr = fecDict[ fecType ]
               tmpStr += ','
               tmpStr += speedStr 
            else:
               tmpStr = speedStr
            fecDict.update( { fecType: tmpStr } )
      return fecDict

   # input: dict with { fecType : speedStr }
   # return: formated string for printing out
   @staticmethod
   def fecTypeSpeedDictToFmtStr( fecDict, clientFecSet=False ):
      coherentEncodingSupported = False
      fecCaps = []
      for lbl in CapabilitiesMixIn.errorCorrectionEncodingToDisplay:
         speedStr = fecDict.get( lbl, 'none' )
         if lbl != 'disabled':
            coherentEncodingSupported |= speedStr != 'none'
         fecCaps.append( '%s(%s)' % ( lbl, speedStr ) )
      fmtStr = ', '.join( fecCaps )
      if not coherentEncodingSupported and not clientFecSet:
         return ''
      else:
         return fmtStr

   @staticmethod
   def coherentFecTypeDictToFmtStr( typeDict ):
      fmtList = []
      for ftype in sorted( typeDict ):
         dataStr = ','.join( typeDict[ ftype ] )
         tempStr = '%s(%s)' % ( ftype, dataStr )
         fmtList.append( tempStr )
      return ', '.join( fmtList )
#-------------------------------------------------------------------------------
# EthIntf specific Eapi Models
#-------------------------------------------------------------------------------
class EthInterfaceCounters( InterfaceCounters ):
   class PhysicalInputErrors( Model ):
      runtFrames = Int( help='Number of runt frames received')
      giantFrames = Int( help='Number of giant frames received' )
      fcsErrors = Int( help='FCS errors' )
      alignmentErrors = Int( help='Alignment errors' )
      symbolErrors = Int( help='Symbol errors' )
      rxPause = Int( help='Flow control pause frames received' )  
   
   class PhysicalOutputErrors( Model ):
      collisions = Int( help='Sum of single and multiple collisions' )
      lateCollisions = Int( help='Late collision frames' )
      deferredTransmissions = Int( help='Deferred transmissions' )
      txPause = Int( help='Flow control pause frames sent' )
   
   linkStatusChanges = Int( help='Number of link status changes since last clear' )
   lastClear = Float( help='Last clear of interface counters', optional=True )
   totalInErrors = Int( help='Total number of input errors' )
   inputErrorsDetail = Submodel( valueType=PhysicalInputErrors, optional=True, 
                                 help='Details on the types of input errors with '
                                      'physical layer counters' )
   totalOutErrors = Int( help='Total number of output errors' )
   outputErrorsDetail = Submodel( valueType=PhysicalOutputErrors, optional=True, 
                                  help='Details on the types of output errors with '
                                       'physical layer counters' )
   counterRefreshTime = Float( help="UTC of last counter refresh", optional=True  )
   
   def renderCounters( self ):
      self._renderInputStatistics()
      self._renderInputErrors()
      self._renderOutputStatistics()
      self._renderOutputErrors()
   
   def _renderInputStatistics( self ):
      print( '     %d packets input, %d bytes' % ( self.inTotalPkts,
                                                   self.inOctets ) )
      print( '     Received %d broadcasts, %d multicast' % ( self.inBroadcastPkts,
                                                             self.inMulticastPkts ) )

   def _renderDiscards( self, value, valueType ):
      name = '%s discards' % valueType
      if value is None:
         value = 'N/A'
      return '%s %s' % ( value, name )

   def _renderInputErrors( self ):
      if self.inputErrorsDetail is not None:
         print( '     %d runts, %d giants' % ( self.inputErrorsDetail.runtFrames,
                                               self.inputErrorsDetail.giantFrames ) )
         print( '     %d input errors, %d CRC, %d alignment, %d symbol, %s' % (
                self.totalInErrors, self.inputErrorsDetail.fcsErrors,
                self.inputErrorsDetail.alignmentErrors,
                self.inputErrorsDetail.symbolErrors,
                self._renderDiscards( self.inDiscards, 'input' ) ) )
         print( '     %d PAUSE input' % self.inputErrorsDetail.rxPause )
      else:
         print( '     %d input errors, %s' %
                ( self.totalInErrors,
                  self._renderDiscards( self.inDiscards, 'input' ) ) )

   def _renderOutputStatistics( self ):
      print( '     %d packets output, %s bytes' % ( self.outTotalPkts,
                                                    self.outOctets ) )
      print( '     Sent %d broadcasts, %d multicast' % ( self.outBroadcastPkts,
                                                         self.outMulticastPkts ) )

   def _renderOutputErrors( self ):
      if self.outputErrorsDetail is not None:
         print( '     %d output errors, %d collisions' % (
                self.totalOutErrors, self.outputErrorsDetail.collisions ) )
         print( '     %d late collision, %d deferred, %s' % (
                self.outputErrorsDetail.lateCollisions,
                self.outputErrorsDetail.deferredTransmissions,
                self._renderDiscards( self.outDiscards, 'output' ) ) )
         print( '     %d PAUSE output' % self.outputErrorsDetail.txPause )
      else:
         print( '     %d output errors, %s' %
                ( self.totalOutErrors,
                  self._renderDiscards( self.outDiscards, 'output' ) ) )

   def renderLastClearCounter( self ):
      if self.lastClear is not None:
         lastClear = Ark.timestampToStr( self.lastClear  - Tac.utcNow() + Tac.now() )
      else:
         lastClear = 'never'
      print( '  Last clearing of "show interface" counters %s' % lastClear )

   def renderLinkStatusChanges( self ):
      print( '  %d link status changes since last clear' % self.linkStatusChanges )

class EthInterfaceStatistics( Model ):
   updateInterval = Float( help='Stats update interval in seconds' )
   inBitsRate = Float( help='input bits rate' )
   inPktsRate = Float( help='input packets rate' )
   outBitsRate = Float( help='output bits rate' )
   outPktsRate = Float( help='output packets rate' )
   
    # Helper functions
   def ratePct( self, bps, pps, bandwidth ):
      # Calculate the percentage of theoretical bitrate achieved
      # which includes 12 bytes of inter-frame gap (IFG) and 8 bytes
      # of preamble.
      per = '-'
      # In case the speed is undefined don't print anything
      if bandwidth != 0:
         per = '%.1f%%' % ( ( bps + 160 * pps ) * 100 / bandwidth )
      return per
   
   def printRate( self, direction, bps, pps, bandwidth ):
      ( scaledBps, bpsPrefix ) = HumanReadable.scaleValueSi( bps )

      # If the number has been scaled with some SI prefix
      # (e.g. Mbps) then it's meaningful to display digits after
      # the decimal point
      if bpsPrefix:
         formattedBps = HumanReadable.formatValueSigFigs( scaledBps, 3 )
      else:
         formattedBps = '%.0f' % scaledBps

      fmt = '  %s %s rate %s %sbps (%s with framing overhead), '
      fmt += '%.0f packets/sec'
      intervalStr = IntfCli.getLoadIntervalPrintableString( self.updateInterval )
      print( fmt % ( intervalStr, direction, formattedBps, bpsPrefix,
                     self.ratePct( bps, pps, bandwidth ), pps ) )

   def renderStatistics( self, bandwidth ):
      self.printRate( 'input', self.inBitsRate, self.inPktsRate, bandwidth )
      self.printRate( 'output', self.outBitsRate, self.outPktsRate, bandwidth )
   
   
class EthInterfaceStatus( InterfaceStatus ):  
   interfaceMembership = Str( help='Shows whether the interface is part of a '
                                    'monitor destination or a Port-channel',
                                    optional=True )
   interfaceStatistics = Submodel( valueType=EthInterfaceStatistics, optional=True, 
                                   help='Interface statistics, such as rates' )
   interfaceCounters = Submodel( valueType=EthInterfaceCounters, optional=True,
                                 help='Interface Counters and related information')
   
   def renderIntfMembership( self ):
      if self.interfaceMembership:
         print( '  %(interfaceMembership)s' % self )

   def renderLinkTypeSpecific( self ):
      raise NotImplementedError()
   
   def renderLoopbackMode( self ):
      raise NotImplementedError()

   def render( self ):
      self.renderHeader()
      self.renderHardware()
      self.renderIntfMembership()
      self.renderInterfaceAddress()
      self.renderMtuMruAndBw()
      self.renderLinkTypeSpecific()
      self.renderUptime()
      self.renderMaintEnterTime()
      self.renderLoopbackMode()
      
      # Stats
      if self.interfaceStatistics is None:
         print( '  Statistics unavailable' )
         return

      self.interfaceCounters.renderLinkStatusChanges()
      self.interfaceCounters.renderLastClearCounter()
      self.interfaceStatistics.renderStatistics( self.bandwidth )
      self.interfaceCounters.renderCounters()
      

class EthPhyInterfaceStatus( EthInterfaceStatus ):
   duplex = Enum( values=( 'duplexHalf', 'duplexFull', 'duplexUnknown' ), 
                  help='Duplex status of the interface' )
   _autoNegotiateActive = Bool( help='Is autonegotiate active. We store this '
                                       'field here for now because autonegActive '
                                       'function in EthIntfCli is currently '
                                       'hacky.' )
   autoNegotiate = Enum( values=( 'unknown', 'off', 'negotiating', 'success', 
                                  'failed' ), help='Current auto-negotiation state' )
   _uniLinkMode = Enum( values=( 'n/a', 'disabled', 'send-only', 'receive-only',
                                 'send-receive' ),
                                 help='Current unidirectional link mode' )
   loopbackMode = Enum( values=( 'loopbackNone', 'loopbackPhy', 'loopbackMac',
                                 'loopbackPhyRemote' ), optional=True,
                                 help="Loopback mode of the interface" )
   lanes = Int( optional=True, help='Number of lanes on the interface' )

   def renderLoopbackMode( self ):
      if self.loopbackMode:
         lbMode = self.loopbackMode
         prefix = "  Loopback Mode : "
         if lbMode == 'loopbackPhyRemote':
            print( prefix + "Network PHY" )
         elif lbMode == 'loopbackMac':
            print( prefix + "System MAC" )
         elif lbMode == 'loopbackPhy':
            print( prefix + "System PHY" )
         else:
            print( prefix + "None" )
      else:
         print( "  " )

   def renderLinkTypeSpecific( self ):
      duplex = EthIntfLib.generateOperDuplex( 'Format1', 
                                              self._autoNegotiateActive,
                                              self.duplex )
      autoNegotiate = EthIntfLib.generateAutoNegotiation( self.autoNegotiate )
      bandwidth = EthIntfLib.getSpeedString( self.bandwidth, 
                                            self._autoNegotiateActive,
                                            'Format1' )
      laneCount = EthIntfLib.getLaneCountString( self.lanes )
      print( '  %s, %s%s, auto negotiation: %s, uni-link: %s' %
             ( duplex, bandwidth, laneCount, autoNegotiate, self._uniLinkMode ) )

class InterfaceDebounceTime( Model ):
   linkUpTime = Int( help='Interface linkup debounce time in milliseconds, '
                           '0 means disabled' )
   linkDownTime = Int( help='Interface linkdown debounce time in milliseconds, '
                           '0 means disabled' )

class InterfacesDebounceTime( Model ):
   interfaces = Dict( keyType=Interface, valueType=InterfaceDebounceTime,
                 help="Mapping between an interface and debounce times" )

   def render( self ):
      if not self.interfaces:
         return

      hdrFmt = '%-10.10s  %-20.20s'
      print( hdrFmt % ( 'Port', 'Debounce Time' ) )
      print( hdrFmt % ( '-' * 10, '-' * 20 ) )

      fmt = '%-10.10s  %-10.10s %-10.10s'
      print( fmt % ( '', 'link-up', 'link-down' ) )
      print( fmt % ( '', '-' * 10, '-' * 10 ) )

      for key in Arnet.sortIntf( self.interfaces ):
         debounceInfo = self.interfaces[ key ]
         print( fmt % ( IntfCli.Intf.getShortname( key ),
                        debounceInfo.linkUpTime,
                        debounceInfo.linkDownTime ) )


class InterfaceBinsCounters( Model ):

   frames64Octet = Int( help='counter bin 64 bytes' )
   frames65To127Octet = Int( help='counter bin 65-127 bytes' )
   frames128To255Octet = Int( help='counter bin 128-255 bytes' )
   frames256To511Octet = Int( help='counter bin 256-511 bytes' )
   frames512To1023Octet = Int( help='counter bin 512-1023 bytes' )
   frames1024To1522Octet = Int( help='counter bin 1024-1522 bytes' )
   frames1523ToMaxOctet = Int( help='counter bin 1524-Max bytes' )

class InterfaceInOutBinsCounters( Model ):
   inBinsCounters = Submodel( valueType=InterfaceBinsCounters, optional=True,
                          help='counter bins for in frames' ) 
   outBinsCounters = Submodel( valueType=InterfaceBinsCounters, optional=True,
                          help='counter bins for out frames' ) 

class InterfacesBinsCounters( Model ):
   interfaces = Dict( keyType=Interface, valueType=InterfaceInOutBinsCounters,
                 help="Mapping between an interface and counter bins" )

   def printDirection( self, interfaces, direction, header ):

      printedHeader = False

      fmt = '%-10s %16s %16s %16s %16s'
      def printHdr():
         print( header )
         print( fmt % ( 'Port', '64 Byte', '65-127 Byte', '128-255 Byte',
                        '256-511 Byte' ) )
         print( '-' * 78 )

      for intf in Arnet.sortIntf( self.interfaces ):
         inOutBinsCounters = interfaces[ intf ]

         if direction == 'in':
            binsCounters = inOutBinsCounters.inBinsCounters
         else:
            binsCounters = inOutBinsCounters.outBinsCounters

         if binsCounters:
            if not printedHeader:
               printedHeader = True
               printHdr()

            print( fmt % ( IntfCli.Intf.getShortname( intf ),
                           binsCounters.frames64Octet,
                           binsCounters.frames65To127Octet,
                           binsCounters.frames128To255Octet,
                           binsCounters.frames256To511Octet ) )

      if not printedHeader:
         # There are no interfaces with this stat
         return

      print()
      fmt = '%-10s %16s %16s %16s'
      print( fmt % ( 'Port', '512-1023 Byte', '1024-1522 Byte',
                     '1523-MAX Byte' ) )
      print( '-' * 61 )
      for intf in Arnet.sortIntf( self.interfaces ):
         inOutBinsCounters = interfaces[ intf ]

         if direction == 'in':
            binsCounters = inOutBinsCounters.inBinsCounters
         else:
            binsCounters = inOutBinsCounters.outBinsCounters

         if binsCounters:
            print( fmt % ( IntfCli.Intf.getShortname( intf ),
                           binsCounters.frames512To1023Octet,
                           binsCounters.frames1024To1522Octet,
                           binsCounters.frames1523ToMaxOctet ) )

   def render( self ):
      if not self.interfaces:
         return

      self.printDirection( self.interfaces, 'in', 'Input' )
      self.printDirection(  self.interfaces, 'out', '\nOutput' )

# Flow control data for interfaces
class FlowControlData( Model ):
   txAdminState = Enum( values=( "on", "off", "desired", "unknown", "unsupported" ),
                        help="Send Flowcontrol admin state" )
   txOperState = Enum( values=( "on", "off", "disagree", "unknown", "unsupported" ),
                       help="Send Flowcontrol operational state" )
   rxAdminState = Enum( values=( "on", "off", "desired", "unknown", "unsupported" ),
                        help="Receive Flowcontrol admin state" )
   rxOperState = Enum( values=( "on", "off", "disagree", "unknown", "unsupported" ),
                       help="Receive Flowcontrol operational state" )
   txPause = Int( help="Pause packets sent" )
   rxPause = Int( help="Pause packets received" )
   
#Interface flow control
class InterfacesFlowControl( Model ):
   interfaceFlowControls = Dict( keyType=Interface, valueType=FlowControlData,
                      help="Mapping between an interface and it's flow control data")
   syslogInterval = Float( optional=True,
            help='Syslog interval for flow control pasuse frames in seconds' )

   def render( self ):
      if not self.interfaceFlowControls:
         return
      if self.syslogInterval:
         print( 'Pause frame received logging: enabled,'
                ' Interval: %.f seconds' % ( self.syslogInterval ) )
      elif self.syslogInterval == 0:
         print( 'Pause frame received logging: disabled' )
      hdrFmt = '%-10.10s  %-17.17s %-20.20s %-13.13s %-13.13s'
      print( hdrFmt % ( 'Port', 'Send FlowControl', 'Receive FlowControl',
                        'RxPause', 'TxPause' ) )

      fmt = '%-10.10s  %-8.8s %-8.8s %-8.8s %-8.8s    %-13.13s %-13.13s'
      print( fmt % ( '', 'admin', 'oper', 'admin', 'oper', '', '' ) )

      print( fmt % ( '-' * 10, '-' * 8, '-' * 8, '-' * 8, '-' * 8, '-' * 13,
                     '-' * 13 ) )

      def _convertFcState( state ):
         retState = "Unsupp." if state == "unsupported" else state
         return retState

      for intf in Arnet.sortIntf( self.interfaceFlowControls ):
         x = self.interfaceFlowControls[ intf ]
         print( fmt % ( IntfCli.Intf.getShortname( intf ),
                        _convertFcState( x.txAdminState ),
                        _convertFcState( x.txOperState ),
                        _convertFcState( x.rxAdminState ),
                        _convertFcState( x.rxOperState ),
                        x.rxPause,
                        x.txPause ) )

class L2Mtu( Model ):
   l2MtuCfg = Int( help="Configured L2 MTU (bytes)" )
   l2MtuStatus = Int( help="L2 MTU Status (bytes)" )


# Class that holds L2 MTU values for given interfaces
class InterfacesL2Mtu( Model ):

   interfaceL2Mtus = Dict( keyType=Interface, valueType=L2Mtu,
                  help="Mapping between an interface and errors" )

   def render( self ):
      if not self.interfaceL2Mtus:
         return

      hdrFmt = '%-10s  %10s %10s'
      print( hdrFmt % ( 'Port', 'MTU Config', 'MTU Status' ) )
      print( hdrFmt % ( '-' * 10, '-' * 10, '-' * 10 ) )
      for intf in Arnet.sortIntf( self.interfaceL2Mtus ):
         x = self.interfaceL2Mtus[ intf ]
         shortName = IntfCli.Intf.getShortname( intf )
         if shortName[ :2 ] == 'Lo':
            continue

         print( hdrFmt % ( shortName, str( x.l2MtuCfg ) if x.l2MtuCfg else '',
                          str( x.l2MtuStatus ) ) )

class InterfaceUnidirectional( Model ):
   status = InterfaceStatus.interfaceStatus
   # pylint: disable-msg=protected-access
   uniLinkMode = EthPhyInterfaceStatus._uniLinkMode

class InterfacesUnidirectional( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=InterfaceUnidirectional,
                      help='Mapping between an interface '
                           'and its unidirectional link information' )

   def render( self ):

      hdrFmt = '%-10.10s  %-12.10s %-12.12s'
      print( hdrFmt % ( 'Port', 'Status', 'UniLinkMode' ) )
      print( hdrFmt % ( '-' * 10, '-' * 10, '-' * 12 ) )

      for intf in Arnet.sortIntf( self.interfaces ):
         data = self.interfaces[ intf ]
         shortName = IntfCli.Intf.getShortname( intf )
         print( hdrFmt % ( shortName,
                           data.status,
                           data.uniLinkMode ) )

#-------------------------------------------------------------------------------
# Error-correction related models.
#-------------------------------------------------------------------------------
class InterfaceErrorCorrectionStatus( Model ):

   _nonCoherentIntf = Bool(
      help="True if it is a non coherent Interface" )
   _coherentIntf = Bool(
      help="True if it is a coherent Interface" )
   _noCoherentConfiguration = Bool(
      help="True if user hasn't applied any coherent configuration", optional=True )
   # The user applied configuration.
   fecEncodingConfigDisabled = Bool(
      help="True if error correction is disabled."
      " When set, all other configured encoding values are ignored"
      " and should be False." )
   fecEncodingConfigReedSolomon544Enabled = Bool(
      help="True if Reed-Solomon FEC (544), is enabled." )
   fecEncodingConfigReedSolomonEnabled = Bool(
      help="True if Reed-Solomon FEC (528), is enabled." )
   fecEncodingConfigFireCodeEnabled = Bool(
      help="True if Fire code (BASE-R) FEC, is enabled." )
   coherentEncodingConfigured = Enum(
       values=( 'default', 'reedSolomonG709', 'staircaseOverhead7Percent',
                'softDecisionOverhead15Percent', 'softDecisionOverhead20Percent',
                'softDecisionOverhead25Percent',
                'softDecisionOverhead25PercentBch', 'concatenatedFec', 'openFec' ),
       help="Configured coherent error-correction encoding.", optional=True )

   def clientEncodingConfigString( self ):
      """Returns a string representing the encoding configuration"""
      rsFecEnabled = ( self.fecEncodingConfigReedSolomon544Enabled or
                       self.fecEncodingConfigReedSolomonEnabled )
      valueToStr = { ( False, False, False ) : "Default",
                     ( True, False, False )  : "Disabled",
                     ( False, True, False ) : "Reed-Solomon",
                     ( False, False, True ) : "Fire-code",
                     ( False, True, True ) : "Reed-Solomon, Fire-code" }

      try:
         value = valueToStr[
            ( self.fecEncodingConfigDisabled,
              rsFecEnabled,
              self.fecEncodingConfigFireCodeEnabled )
            ]
      except KeyError:
         # The table above doesn't handle the 3 invalid combinations where
         # encodingNone and another encoding is set. We catch the error
         # and only honor the encodingNone value.
         value = "Disabled"

      return value

   def coherentEncodingConfigString( self ):
      """Returns a string representing the coherent encoding configuration"""
      if self._noCoherentConfiguration:
         return 'Default'
      valueToStr = { 'reedSolomonG709' : 'RS G.709',
                     'staircaseOverhead7Percent' : 'SC 7%',
                     'softDecisionOverhead15Percent' : 'SD 15%',
                     'softDecisionOverhead20Percent' : 'SD 20%',
                     'softDecisionOverhead25Percent' : 'SD 25%',
                     'softDecisionOverhead25PercentBch' : 'SD 25% BCH',
                     'concatenatedFec' : 'C-FEC',
                     'openFec' : 'O-FEC', }
      return valueToStr.get( self.coherentEncodingConfigured, 'Default' )

   def encodingConfigString( self ):
      retConfigString = ""
      if self._nonCoherentIntf and not self._coherentIntf:
         retConfigString += self.clientEncodingConfigString()
      if not self._nonCoherentIntf and self._coherentIntf:
         retConfigString += self.coherentEncodingConfigString()
      else:
         # This will hit on a slot capable of accepting both coherent
         # and client transceivers.
         clientConfigString = self.clientEncodingConfigString()
         coherentConfigString = self.coherentEncodingConfigString()
         if ( clientConfigString in ( "Default", "Disabled", ) and
              coherentConfigString != "Default" ):
            retConfigString = coherentConfigString
         elif ( clientConfigString not in ( "Default", "Disabled", ) and
                coherentConfigString == "Default" ):
            retConfigString = clientConfigString
         elif clientConfigString == "Disabled" and coherentConfigString == "Default":
            retConfigString = clientConfigString
         elif clientConfigString == "Default" and coherentConfigString == "Default":
            retConfigString = "Default"
         else:
            retConfigString = clientConfigString + ", " + coherentConfigString
      return retConfigString

   # Which encodings were available based on the hardware and configuration.
   fecEncodingReedSolomon544Available = Bool(
      help="True if RS-FEC-544 is available" )
   fecEncodingReedSolomonAvailable = Bool(
      help="True if RS-FEC-528 is available" )
   fecEncodingFireCodeAvailable = Bool(
      help="True if Fire code (BASE-R) FEC, is available" )

   # Coherent FECs
   reedSolomonG709Available = Bool( help="G.709 Reed-Solomon FEC", optional=True )
   staircaseOverhead7PercentAvailable = Bool(
      help="True if Staircase 7% overhead FEC is available",
      optional=True )
   softDecisionOverhead15PercentAvailable = Bool(
      help="True if Soft Decision 15% overhead FEC is available",
      optional=True )
   softDecisionOverhead20PercentAvailable = Bool(
      help="True if Soft Decision 20% overhead FEC is available",
      optional=True )
   softDecisionOverhead25PercentAvailable = Bool(
      help="True if Soft Decision 25% overhead FEC is available",
      optional=True )
   softDecisionOverhead25PercentBchAvailable = Bool(
      help="True if Soft Decision 25% overhead BCH FEC is available",
      optional=True )
   openFecAvailable = Bool(
      help="True if Open FEC is available",
      optional=True )
   concatenatedFecAvailable = Bool(
      help="True if C-FEC is available",
      optional=True )

   def clientEncodingAvailString( self ):
      """Returns a string representing the available client encodings"""
      rsFecAvailable = ( self.fecEncodingReedSolomon544Available or
                         self.fecEncodingReedSolomonAvailable )
      valueToStr = { ( False, False ) : "None",
                     ( True, False ) : "Reed-Solomon",
                     ( False, True ) : "Fire-code",
                     ( True, True ) : "Reed-Solomon, Fire-code" }

      value = valueToStr[ ( rsFecAvailable,
                            self.fecEncodingFireCodeAvailable ) ]
      return value

   def coherentEncodingAvailString( self ):
      """Returns a string representing the available coherent encodings"""
      fecDict = OrderedDict(
                [ ( 'RS G.709', self.reedSolomonG709Available ),
                  ( 'SC 7%', self.staircaseOverhead7PercentAvailable ),
                  ( 'SD 15%', self.softDecisionOverhead15PercentAvailable ),
                  ( 'SD 20%', self.softDecisionOverhead20PercentAvailable ),
                  ( 'SD 25%', self.softDecisionOverhead25PercentAvailable ),
                  ( 'SD 25% BCH', self.softDecisionOverhead25PercentBchAvailable ),
                  ( 'C-FEC', self.concatenatedFecAvailable ),
                  ( 'O-FEC', self.openFecAvailable ),
                ] )
      available = [ fecStr for fecStr in fecDict if fecDict[ fecStr ] ]
      return ', '.join( available ) if available else 'None'

   def encodingAvailString( self ):
      retAvailString = ""
      if self._nonCoherentIntf and not self._coherentIntf:
         retAvailString += self.clientEncodingAvailString()
      if not self._nonCoherentIntf and self._coherentIntf:
         retAvailString += self.coherentEncodingAvailString()
      else:
         # This will hit on a slot capable of accepting both coherent
         # and client transceivers.
         clientAvailString = self.clientEncodingAvailString()
         coherentAvailString = self.coherentEncodingAvailString()
         if ( clientAvailString == "None" and coherentAvailString != "None" ):
            retAvailString = coherentAvailString
         elif ( coherentAvailString == "None" and clientAvailString != "None" ):
            retAvailString = clientAvailString
         elif ( clientAvailString == "None" and coherentAvailString == "None" ):
            retAvailString = "None"
         else:
            retAvailString = clientAvailString + ", " + coherentAvailString
      return retAvailString

   # Status - which encoding is in use
   errCorrEncodingStatus = Enum(
      # Value 'none' is for coherent interfaces that weren't active due to
      # modulation config.
      values=( 'unknown', 'none', 'disabled', 'reedSolomon', 'fireCode',
               'reedSolomonG709', 'staircaseOverhead7Percent',
               'softDecisionOverhead15Percent', 'softDecisionOverhead20Percent',
               'softDecisionOverhead25Percent',
               'softDecisionOverhead25PercentBch', 'openFec', 'concatenatedFec' ),
      help="Current error correction encoding state" )

   def statusString( self ):
      """Returns a string representing the currently active
      error correction encoding"""
      valueToStr = { "unknown" : "Unknown",
                     "none" : "None",
                     "disabled" : "Disabled",
                     "reedSolomon" : "Reed-Solomon",
                     "fireCode" : "Fire-code",
                     "reedSolomonG709" : "RS G.709",
                     "staircaseOverhead7Percent" : "SC 7%",
                     "softDecisionOverhead15Percent" : "SD 15%",
                     "softDecisionOverhead20Percent" : "SD 20%",
                     "softDecisionOverhead25Percent" : "SD 25%",
                     "softDecisionOverhead25PercentBch" : "SD 25% BCH",
                     "openFec" : "O-FEC",
                     "concatenatedFec" : "C-FEC" ,}
      return valueToStr[ self.errCorrEncodingStatus ]

class InterfacesErrorCorrectionStatus( Model ):
   _hasCoherentIntf = Bool( help = "interfaceErrCorrStatuses contains at"
                               " least one coherent interface", optional=True )
   interfaceErrCorrStatuses = Dict(
      keyType=Interface,
      valueType=InterfaceErrorCorrectionStatus,
      help="Mapping between an interface and error correction information" )

   def render( self ):
      if not self.interfaceErrCorrStatuses:
         return

      helpText = ( "RS G.709:    Reed-Solomon G.709 FEC\n"
                   "SC 7%:       Staircase 7% overhead FEC\n"
                   "SD 15%:      Soft Decision 15% overhead FEC\n"
                   "SD 20%:      Soft Decision 20% overhead FEC\n"
                   "SD 25%:      Soft Decision 25% overhead FEC\n"
                   "SD 25 BCH%:  Soft Decision 25% overhead with outer BCH FEC\n"
                   "                                                            " )
      if self._hasCoherentIntf:
         print( helpText )
      hdrFmt = '%-18s  %-23s %-54s %-15s'
      print( hdrFmt % ( 'Interface', 'Configured', 'Available', 'Operational' ) )
      print( hdrFmt % ( '-' * 16, '-' * 20, '-' * 52, '-' * 11 ) )
      for intf in Arnet.sortIntf( self.interfaceErrCorrStatuses ):
         status = self.interfaceErrCorrStatuses[ intf ]
         print( hdrFmt % ( intf,
                           status.encodingConfigString(),
                           status.encodingAvailString(),
                           status.statusString() ) )

class FlowControlCapabilities( Model ):
   flowControl = Enum( values=( 'Capable', 'Incapable', 'Unknown' ),
                       help='Flow control capability' )
   autoNegotiationPause = Bool( help='True if we have pause capability'
                                     ' for auto negotiation' )

class EthSpeedLanesDuplex( Model ):
   speed = Enum( values=CapabilitiesMixIn.speedValues, optional=True,
                 help='Interface speed' )
   # legacy speeds will have laneCount is None
   laneCount = Int( optional=True, help='Number of lanes on the interface' )
   duplex = Enum( values=( 'full', 'half', 'unknown' ), optional=True,
                  help='Duplex communication capabilities with this speed' )
   _showLanes = Bool( optional=True, help='Lane count is set' )
   auto = Bool( optional=True, help='This speed is auto negotiated' )

   def showLanesIs( self, showLanes ):
      self._showLanes = showLanes

   def showLanes( self ):
      return self._showLanes

   def autoIs( self, auto ):
      self.auto = auto

class EthIntfMode( Model ):
   link = Submodel( valueType=EthSpeedLanesDuplex, optional=True,
                    help='Interface speed and communication mode' )
   errorCorrections = List( valueType=str, optional=True,
                            help='List of error-correction types capable of'
                                 ' running with this speed' )
   defaultErrorCorrection = Enum( values=CapabilitiesMixIn.errorCorrectionValues,
                                  optional=True,
                                  help='Default error correction' )
   defaultLinkMode = Bool( optional=True,
                           help='This is the default link mode for this interface' )
   pllIncompatible = Bool( optional=True, 
                           help='This link mode is incompatible for this'
                                ' interface' )
   _errorCorrectionsIsValid = Bool( optional=True, 
                                    help='True if defaultErrorCorrection is set' )
         
   def pllIncompatibleIs( self, incompatible ):
      self.pllIncompatible = incompatible

   def errorCorrectionsIsValidIs( self, errorCorrectionsIsValid ):
      self._errorCorrectionsIsValid = errorCorrectionsIsValid

   def errorCorrectionsIsValid( self ):
      return self._errorCorrectionsIsValid

class CoherentMode( Model ):
   modulation = Enum( values=CapabilitiesMixIn.coherentModes,
                      help='Modulation type' )
   defaultModulation = Enum( values=CapabilitiesMixIn.coherentModes,
                             optional=True,
                             help='Default modulation type' )
   defaultErrorCorrection = Enum( values=CapabilitiesMixIn.coherentFecs,
                                  optional=True,
                                  help='Default error correction' )
   errorCorrections = List( valueType=str,
                            help='List of coherent error-correction types'
                                 ' available while using the corresponding'
                                 ' modulation' )
   defaultLineRate = Enum( values=CapabilitiesMixIn.coherentLineRates,
                           optional=True,
                           help='Default line rate' )
   lineRates = List( valueType=str,
                     help='List of coherent line rate types'
                     ' available while using the corresponding'
                     ' modulation' )

class CoherentTributary( Model ):
   lineRate = Enum( values=CapabilitiesMixIn.coherentLineRates,
                    optional=True,
                    help='Data rate of coherent optical link' )

   defaultTributary = Int( help='Default tributary', optional=True )

   tributaries = List( valueType=int,
                       help='List of tributaries available while using the '
                       'corresponding line rate' )

class AutonegClause73Capabilities( Model ):
   ieee = List( valueType=EthSpeedLanesDuplex,
                help='List of speed/lane count/duplex triplets capable of IEEE'
                     ' defined Clause 73 auto negotiation' )
   consortium = List( valueType=EthSpeedLanesDuplex,
                      help='List of speed/lane count/duplex triplets capable of'
                           ' Consortium defined Clause 73 auto negotiation' )

class SpeedGroup( Model ):
   groupId = Str( help='Speed group Id' )
   intfList = List( valueType=Interface,
                    help='List of speed group interface members' )
   _masterIntf = Interface( optional=True,
                           help='Master interface of this speed group' )

   def masterIntfIs( self, masterIntf ):
      self._masterIntf = masterIntf

   def masterIntf( self ):
      return self._masterIntf

class InterfaceCapabilitiesBaseModel( Model ):
   switchModelName = Str( help='Model name of the switch' )
   transceiverType = Str( help='Type of the inserted transceiver' )
   modes = List( valueType=EthIntfMode,
                 help='List of link-mode capabilities for this interface' )
   rxFlowControl = Submodel( valueType=FlowControlCapabilities,
                             help='The rx flow control capabilities of this'
                                  ' interface' )
   txFlowControl = Submodel( valueType=FlowControlCapabilities,
                             help='The tx flow control capabilities of this'
                                  ' interface' )
   coherentModes = List( valueType=CoherentMode, optional=True,
                         help='List of modulation and coherent error-correction'
                              ' capabilities for this interface' )
   coherentTributaries = List(
      valueType=CoherentTributary, optional=True,
      help='List of tributaries which may be mapped at each line rate' )

   autoNegotiationClause28 = List( valueType=EthSpeedLanesDuplex, optional=True,
                                   help='List of speed/lane count/duplex triplets at'
                                        ' which this interface can perform clause 28'
                                        ' auto negotiation. A missing entry'
                                        ' indicates auto negotiation capabilities'
                                        ' are unknown.' )
   autoNegotiationClause37 = List( valueType=EthSpeedLanesDuplex, optional=True,
                                   help='List of speed/lane count/duplex triplets at'
                                        ' which this interface can perform clause 37'
                                        ' auto negotiation. A missing entry'
                                        ' indicates auto negotiation capabilities'
                                        ' are unknown.' )
   autoNegotiationClause73 = Submodel( valueType=AutonegClause73Capabilities, 
                                       optional=True,
                                       help='List of speed/lane count/duplex'
                                            ' triplets at which this interface can'
                                            ' perform clause 73 auto negotiation. A'
                                            ' missing entry indicates auto'
                                            ' negotiation capabilities are unknown.'
                                            )
   speedGroup = Submodel( valueType=SpeedGroup, optional=True,
                          help='Speed Group Id and range of member interfaces.' )

   hasAutoneg = Bool( optional=True,
                      help='This interface supports speed auto negotiated' )
   _hasCoherentIntf = Bool( optional=True,
                            help = 'The interfaceErrCorrStatuses contains at'
                                   ' least one coherent interface' )
   _hasCmisCoherentIntf = Bool( optional=True,
                                help='A modulation was added on a CMIS '
                                     'coherent slot. When this is set the '
                                     'modulation label is ommited from output.' )
   _hasClientFec = Bool( optional=True,
                         help = 'The fecCapabilities contains client error'
                                ' correction' )
   _transceiverPresence = Bool( optional=True,
                                help = 'Transceiver is present' )

   def modelIs( self, switchModelName ):
      self.switchModelName = switchModelName

   def typeIs( self, transceiverPresence, transceiverType ):
      self._transceiverPresence = transceiverPresence
      self.transceiverType = transceiverType

   # expect: List of CapList
   def speedLanesDuplexIs( self, linkModeCaps ):
      ethMode, hasAutoneg = CapabilitiesMixIn.linkModeCapsToEthSpeedModel(
            linkModeCaps, True )
      if ethMode:
         self.modes = ethMode
         self.hasAutoneg = self.hasAutoneg or hasAutoneg
      else:
         # create an empty record
         ethMode = []
         mode = EthIntfMode()
         ethMode.append( mode )
         self.modes = ethMode
         self.hasAutoneg = hasAutoneg

   def speedGroupIs( self, groupId, intfList, masterIntf=None ):
      if not self.speedGroup:
         self.speedGroup = SpeedGroup()
      self.speedGroup.groupId = groupId
      self.speedGroup.masterIntfIs( masterIntf )
      for intf in intfList:
         self.speedGroup.intfList.append( intf )

   def flowControlIs( self, flowControlCaps, pauseNegCaps, rx=True ):
      fcCap = CapabilitiesMixIn.flowControlCapsToStr.get( flowControlCaps,
                                                          'Unknown' )
      if rx:
         if not self.rxFlowControl:
            self.rxFlowControl = FlowControlCapabilities()
         self.rxFlowControl.flowControl = fcCap 
         self.rxFlowControl.autoNegotiationPause = pauseNegCaps
      else:   
         if not self.txFlowControl:
            self.txFlowControl = FlowControlCapabilities()
         self.txFlowControl.flowControl = fcCap 
         self.txFlowControl.autoNegotiationPause = pauseNegCaps

   def clause28Is( self, linkModeCaps ):
      self.autoNegotiationClause28, _ = \
                  CapabilitiesMixIn.linkModeCapsToEthSpeedModel( linkModeCaps )

   def clause37Is( self, linkModeCaps ):
      self.autoNegotiationClause37, _ = \
                  CapabilitiesMixIn.linkModeCapsToEthSpeedModel( linkModeCaps )

   def clause73Is( self, ieeeLinkModeCaps, consortiumLinkModeCaps ):
      anegCl73 = AutonegClause73Capabilities()
      anegCl73.ieee, _ = CapabilitiesMixIn.linkModeCapsToEthSpeedModel(
                                                         ieeeLinkModeCaps ) 
      anegCl73.consortium, _ = CapabilitiesMixIn.linkModeCapsToEthSpeedModel(
                                                         consortiumLinkModeCaps )
      self.autoNegotiationClause73 = anegCl73

   # expect: List of FecCapList
   def errorCorrectionIs( self, linkModeCaps, defaultCheck=False ):
      supportedErrorCorrections = set()
      for lmc in linkModeCaps:
         for ethMode in self.modes:
            if defaultCheck and not ethMode.defaultErrorCorrection and \
               not ethMode.link.auto:
               # Unless there's a default fec set, we should set Disabled as the
               # default error correction for the given ethMode
               ethMode.defaultErrorCorrection = 'Disabled'
            if ethMode.link.speed == CapabilitiesMixIn.ethSpeedToSpeedValue(
                  lmc.speed ):
               if ethMode.link.showLanes() and \
                  ethMode.link.laneCount != CapabilitiesMixIn.ethLaneCountToValue(
                        lmc.lanes ):
                  continue
               # pylint: disable-next=no-else-continue
               if ethMode.link.duplex != lmc.duplex:
                  continue
               else:
                  ethMode.errorCorrections.append(
                        CapabilitiesMixIn.ethFecEncodingToErrorCorrectionValue(
                                                                  lmc.encoding ) )
                  supportedErrorCorrections.add( 
                        CapabilitiesMixIn.ethFecEncodingToErrorCorrectionValue(
                                                                  lmc.encoding ) )
                  ethMode.errorCorrectionsIsValidIs( True )
                  if lmc.default:
                     ethMode.defaultErrorCorrection = \
                           CapabilitiesMixIn.ethFecEncodingToErrorCorrectionValue(
                                                                  lmc.encoding )
            elif ethMode.link.speed in [ '2p5Gbps', '5Gbps' ]:
               # Since these speeds cannot be forced, they don't have entries in
               # fec capabilities set. We should still set error corrections to
               # [ 'disabled' ] for these speeds
               ethMode.errorCorrections = [ 'Disabled' ]
               if defaultCheck:
                  ethMode.defaultErrorCorrection = 'Disabled'

      # If we end up seeing that the interface is only capable of fecDisabled,
      # Then we don't really support client fec.
      onlyDisabledSupported = "ReedSolomon" not in supportedErrorCorrections and \
                              "FireCode" not in supportedErrorCorrections
      self._hasClientFec = not onlyDisabledSupported

   def coherentErrorCorrectionIs( self, fecDict ):
      self._hasCoherentIntf = True
      if not fecDict:
         return
      modDict = {}
      defaultDict = {}
      modsList = []
      for fecMode in fecDict:
         for mode in fecDict[ fecMode ].split( ',' ):
            fecMode = fecMode.strip()
            fecMode = fecMode.rstrip( ':' )

            if mode.endswith( '(default)' ):
               mode = mode[ :-len( '(default)' ) ]
               defaultDict.update( { mode : fecMode } )
            types = []
            types = modDict.get( mode, [] )

            types.append( fecMode )
            modDict.update( { mode : types } )
            if mode not in modsList:
               modsList.append( mode )

      if not self.coherentModes:
         for mode in modDict: # pylint: disable=consider-using-dict-items
            coherentMode = CoherentMode()
            coherentMode.modulation = mode
            if defaultDict[ mode ]:
               coherentMode.defaultErrorCorrection = defaultDict[ mode ]
            for ftype in reversed( modDict[ mode ] ):
               coherentMode.errorCorrections.append( ftype )
            self.coherentModes.append( coherentMode )
      else:
         self.modulationIs( ','.join( modsList ) )

         for cmode in self.coherentModes:
            if not cmode.modulation in modDict:
               continue
            for fecType in modDict[ cmode.modulation ]:
               cmode.errorCorrections.append( fecType )
               if cmode.modulation in defaultDict and \
                  defaultDict[ cmode.modulation ]:
                  cmode.defaultErrorCorrection = defaultDict[ cmode.modulation ]

   # expect string of modulation divided by ','
   # ex: 'DP-QPSK,8QAM,16QAM(default)'
   def modulationIs( self, modStr ):
      for modulation in modStr.split( ',' ):
         defaultMod = ''
         if modulation.endswith( '(default)' ):
            modulation = modulation[ :-len( '(default)' ) ]
            defaultMod = modulation
         if modulation in CapabilitiesMixIn.coherentModes:
            # add only modulation does not exist
            modFound = False
            for curMod in self.coherentModes:
               if curMod.modulation == modulation:
                  modFound = True
                  break
            if modFound:
               continue
            coherentMode = CoherentMode()
            coherentMode.modulation = modulation
            if defaultMod:
               coherentMode.defaultModulation = defaultMod
            self.coherentModes.append( coherentMode )

   def lineRateIs( self, lrDict ):
      '''
      Updates this object's current set of coherent modes (self.coherentModes) with
      the line rate capabilities specified in the passed in lrDict.

      Parameters
      ----------
      lrDict : { str : list( str ) }
        A mapping from modulation to list of line-rates supported by that modulation.
      '''
      for modulation in lrDict:
         if modulation in CapabilitiesMixIn.coherentModes:
            # Add new modulations to self.coherentMode, skip existing ones
            modFound = False
            curMode = None
            for curMode in self.coherentModes:
               if curMode.modulation == modulation:
                  modFound = True
                  break
            if modFound:
               coherentMode = curMode
            else:
               # If no mode with this modulation exists, create it
               coherentMode = CoherentMode()
               coherentMode.modulation = modulation
               self.coherentModes.append( coherentMode )

            # Update the mode corresponding to this modulation with supported
            # line rates
            for lineRate in lrDict[ modulation ]:
               defaultLr = ''
               if lineRate.endswith( '(default)' ):
                  lineRate = lineRate.replace( '(default)', '' )
                  defaultLr = lineRate
                  coherentMode.defaultLineRate = defaultLr
               coherentMode.lineRates.append( lineRate )

   def tributaryMappingIs( self, tmDict ):
      '''
      Updates this object's current set of coherent tribs (self.coherentTributaries)
      with the tributary mapping capabilities specified in the passed in tmDict.

      Parameters
      ----------
      tmDict : { str : list( int ) }
        A mapping from line rate to list of tributaries supported by that line rate.
        Example: Assume tributaries 1 and 2 are supported at 100g line rate,
                 and only tributary 1 and 200g line rate
        { 'lineRate100g': [ 1, 2 ],
          'lineRate200g': [1]
        }
      '''
      self.coherentTributaries = []
      for lineRate in tmDict:
         # Update the object corresponding to this line rate with supported
         # tributaries
         curTribs = CoherentTributary()
         curTribs.lineRate = lineRate
         defaultTrib = TacLazyType( 'Xcvr::TributaryId' ).tributaryIdUnknown
         for tributary in tmDict[ lineRate ]:
            if '(default)' in tributary:
               tributary = tributary.replace( '(default)', '' )
               defaultTrib = int( tributary )
            curTribs.tributaries.append( int( tributary ) )
         curTribs.defaultTributary = defaultTrib
         self.coherentTributaries.append( curTribs )

   def renderModel( self ):
      return CapabilitiesMixIn.formatCapabilitiesFieldStr(
                     indtLvl=2, header='Model',
                     data=self.switchModelName or 'unknown' )

   def renderType( self ):
      # pylint: disable-next=consider-using-in
      if self.transceiverType == 'Unknown' or self.transceiverType == 'Not Present':
         dataStr = self.transceiverType.lower()
      else:
         dataStr = self.transceiverType
      return CapabilitiesMixIn.formatCapabilitiesFieldStr(
                     indtLvl=2, header='Type',
                     data=dataStr or 'unknown' )

   def renderFlowControl( self ):
      rxStr = CapabilitiesMixIn.formatFlowControl( self.rxFlowControl )
      txStr = CapabilitiesMixIn.formatFlowControl( self.txFlowControl )

      dataFmt = 'rx-(%s),tx-(%s)' % ( rxStr, txStr )
      return CapabilitiesMixIn.formatCapabilitiesFieldStr(
                  indtLvl=2, header='Flowcontrol', data=dataFmt )

   def renderSpeedLanesDuplex( self ):
      raise NotImplementedError

   def renderClientErrorCorrectionToStr( self ):
      raise NotImplementedError

   def renderCoherentErrorCorrectionToStr( self ):
      raise NotImplementedError

   def renderErrorCorrections( self ):
      _fmtStr = self.renderClientErrorCorrectionToStr() + \
                self.renderCoherentErrorCorrectionToStr()
      if not _fmtStr:
         if self._hasCoherentIntf:
            _fmtStr = 'none'
         else:
            _fmtStr = 'unsupported'

      fmtStr = CapabilitiesMixIn.formatCapabilitiesFieldStr(
                        indtLvl=2, header='Error correction', data=_fmtStr )
      return fmtStr

   def renderModulation( self, defaultCheck=False ):
      if not self.coherentModes or self._hasCmisCoherentIntf:
         return ''
      fmtStr=''
      for mod in self.coherentModes:
         if fmtStr:
            fmtStr += ','
         fmtStr += mod.modulation
         if defaultCheck and mod.modulation == mod.defaultModulation:
            fmtStr += '(default)'

      return CapabilitiesMixIn.formatCapabilitiesFieldStr(
                  indtLvl=2, header='Modulation', data=fmtStr )

   def renderLineRates( self, defaultCheck=False ):
      rateToMod = {}
      for mode in self.coherentModes:
         for lr in mode.lineRates:
            if lr not in rateToMod:
               rateToMod[ lr ] = []
            modStr = mode.modulation
            if mode.defaultLineRate == lr and defaultCheck:
               modStr += '(default)'
            rateToMod[ lr ].append( modStr )
      lineRateCaps = []
      for lr, mods in iter( rateToMod.items() ):
         # The convention is to print the modulation in reverse lexical order
         mods.sort( reverse=True )
         lineRateCaps.append( lr + '(' + ','.join( mods ) + ')' )
      if lineRateCaps:
         # print line rates in ascending order
         lineRateCaps.sort()
         return CapabilitiesMixIn.formatCapabilitiesFieldStr(
            indtLvl=2, header='Line rate',
            data=','.join( lineRateCaps ) )
      return ''

   def renderTributaries( self, defaultCheck=False ):
      tribToRate = {}
      for mode in self.coherentTributaries:
         for trib in mode.tributaries:
            if trib not in tribToRate:
               tribToRate[ trib ] = []
            lineRateStr = mode.lineRate
            if mode.defaultTributary == trib and defaultCheck:
               lineRateStr += '(default)'
            tribToRate[ trib ].append( lineRateStr )
      tribCaps = []
      for trib, lineRates in tribToRate.items():
         # print line rates in ascending order
         lineRates.sort()
         tribCaps.append( str( trib ) + '(' + ','.join( lineRates ) + ')' )
      if tribCaps:
         # print tributaries in ascending order
         tribCaps.sort()
         return CapabilitiesMixIn.formatCapabilitiesFieldStr(
            indtLvl=2, header='Tributary',
            data=','.join( tribCaps ) )
      return ''

   def renderClause28( self ):
      if not self.autoNegotiationClause28:
         return ''
      capStr = self._renderSpeedLanesDuplexFormat( self.autoNegotiationClause28 )

      return CapabilitiesMixIn.formatCapabilitiesFieldStr(
                  indtLvl=2, header='Autoneg CL28', data=capStr )

   def renderClause37( self ):
      if not self.autoNegotiationClause37:
         return ''
      capStr = self._renderSpeedLanesDuplexFormat( self.autoNegotiationClause37 )

      return CapabilitiesMixIn.formatCapabilitiesFieldStr(
                  indtLvl=2, header='Autoneg CL37', data=capStr )

   def _renderClause73Format( self, anegVariants, ethMode=False ):
      anegVarStrList = []
      for anegVariantLbl, anegCaps in anegVariants.items():
         if not anegCaps:
            continue
         fmtStr = anegVariantLbl + '('
         anegStrList = []
         for sd in anegCaps:
            capStr = CapabilitiesMixIn.formatSpeedLanesDuplex(
                                             sd.speed, sd.laneCount,
                                             sd.showLanes(), sd.duplex,
                                             False, False )
            anegStrList.append( capStr )
         fmtStr += ','.join( anegStrList )
         fmtStr += ')'
         anegVarStrList.append( fmtStr )
      return ', '.join( anegVarStrList )

   def renderClause73( self ):
      if not self.autoNegotiationClause73:
         return ''
      if ( not self.autoNegotiationClause73.ieee and
           not self.autoNegotiationClause73.consortium ):
         return ''
      fmtStr = ''
      anegVariants = { 'IEEE': self.autoNegotiationClause73.ieee,
                       'consortium': self.autoNegotiationClause73.consortium }
      ieeeConsortiumCapsStr = self._renderClause73Format( anegVariants )
      if ieeeConsortiumCapsStr:
         fmtStr += CapabilitiesMixIn.formatCapabilitiesFieldStr(
                           indtLvl=2, header='Autoneg CL73',
                           data=ieeeConsortiumCapsStr )
      return fmtStr

   def renderAutoneg( self ):
      fmtStrCl28 = self.renderClause28()
      fmtStrCl37 = self.renderClause37()
      fmtStrCl73 = self.renderClause73()

      return fmtStrCl28 + fmtStrCl37 + fmtStrCl73

class InterfaceCapabilitiesDefaultData( InterfaceCapabilitiesBaseModel ):
   def _renderSpeedLanesDuplexFormat( self, speedDuplex, ethMode=False ):
      if not speedDuplex:
         return ''
      capList = []
      for sd in speedDuplex:
         if ethMode:
            if sd.link:
               sd = sd.link
            else:
               continue

         tmpStr = CapabilitiesMixIn.formatSpeedLanesDuplex(
                                             sd.speed, sd.laneCount,
                                             sd.showLanes(), sd.duplex,
                                             False, False )

         capList.append( tmpStr )
      capStr = ','.join( capList )
      return capStr

   def renderSpeedLanesDuplex( self ):
      if not self.modes:
         return CapabilitiesMixIn.formatCapabilitiesFieldStr(
                     indtLvl=2, header='Speed/duplex', data='' )
      capStr = self._renderSpeedLanesDuplexFormat( self.modes, ethMode=True )
        
      return CapabilitiesMixIn.formatCapabilitiesFieldStr(
                  indtLvl=2, header='Speed/duplex', data=capStr )

   def renderSpeedGroup( self ):
      if not self.speedGroup:
         return ''
      intfs = IntfRange.intfListToCanonical(
            self.speedGroup.intfList, noHoleRange=True )[ 0 ]
      sgStr = '%s (%s)' % ( self.speedGroup.groupId, intfs )
      return CapabilitiesMixIn.formatCapabilitiesFieldStr(
                  indtLvl=2, header='Speed group', data=sgStr )

   def renderClientErrorCorrectionToStr( self ):
      if not self.modes:
         return ''

      fecDict = CapabilitiesMixIn.ethIntfModeToFecTypeSpeedDict(
                                          self.modes, defaultCheck=False )
      try:
         self._hasClientFec
      except TypeError:
         _hasClientFec = False
      else:
         _hasClientFec = bool( self._hasClientFec )
      return CapabilitiesMixIn.fecTypeSpeedDictToFmtStr(
                                       fecDict, clientFecSet=_hasClientFec )

   def renderCoherentErrorCorrectionToStr( self ):
      if not self.coherentModes:
         return ''

      typeDict = {}
      for cmode in self.coherentModes:
         for fecType in cmode.errorCorrections:
            modes = [] 
            if fecType in typeDict: # pylint: disable=consider-using-get
               modes = typeDict[ fecType ]
            modes.append( cmode.modulation )
            typeDict.update( { fecType: modes } )

      return CapabilitiesMixIn.coherentFecTypeDictToFmtStr( typeDict )

   def renderData( self ):
      fmtStr = self.renderModel()
      fmtStr += self.renderType()
      fmtStr += self.renderSpeedLanesDuplex()
      fmtStr += self.renderSpeedGroup()
      fmtStr += self.renderFlowControl()
      fmtStr += self.renderAutoneg()
      fmtStr += self.renderModulation( defaultCheck=False )
      fmtStr += self.renderLineRates( defaultCheck=False )
      fmtStr += self.renderTributaries( defaultCheck=False )
      fmtStr += self.renderErrorCorrections()
      return fmtStr

class InterfacesCapabilitiesDefault( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=InterfaceCapabilitiesDefaultData,
                      help='A mapping of interfaces to their capabilities defaults'
                           ' information' )

   def render( self ):
      if not self.interfaces:
         return

      fmtStr = ''
      for intf in Arnet.sortIntf( self.interfaces ):
         fmtStr += '%s\n' % ( intf )
         fmtStr += self.interfaces[ intf ].renderData()

      print( fmtStr, end=( '' if fmtStr[ -1 ].isspace() else ' ' ) )

class InterfaceCapabilitiesData( InterfaceCapabilitiesBaseModel ):
   def _renderSpeedLanesDuplexFormat( self, speedDuplex, ethMode=False ):
      if not speedDuplex:
         return ''
      capList = []
      for sd in speedDuplex:
         if ethMode:
            if not sd.link:
               continue
            if sd.link.auto:
               tmpStr = 'auto'
               if sd.defaultLinkMode:
                  tmpStr += '(default)'
            else:
               tmpStr = CapabilitiesMixIn.formatSpeedLanesDuplex(
                                                sd.link.speed, sd.link.laneCount,
                                                sd.link.showLanes(), sd.link.duplex,
                                                sd.defaultLinkMode,
                                                sd.pllIncompatible )
         else:
            tmpStr = CapabilitiesMixIn.formatSpeedLanesDuplex(
                                             sd.speed, sd.laneCount,
                                             sd.showLanes(), sd.duplex,
                                             False, False )
         capList.append( tmpStr )
      capStr = ','.join( capList )
      return capStr

   def renderSpeedLanesDuplex( self ):
      if not self.modes:
         return CapabilitiesMixIn.formatCapabilitiesFieldStr(
                     indtLvl=2, header='Speed/duplex', data='unknown' )

      capStr = self._renderSpeedLanesDuplexFormat( self.modes, ethMode=True )
      if not capStr and not self._transceiverPresence:
         capStr = 'unknown'

      return CapabilitiesMixIn.formatCapabilitiesFieldStr(
                  indtLvl=2, header='Speed/duplex', data=capStr )

   def renderClientErrorCorrectionToStr( self ):
      if not self.modes:
         fmtStr = ''
         if self.switchModelName != 'Unknown':
            fmtList = []
            for lbl in CapabilitiesMixIn.errorCorrectionEncodingToDisplay:
               tempStr = lbl
               tempStr += "(none)"
               fmtList.append( tempStr )
            fmtStr = ', '.join( fmtList )
         return fmtStr

      fecDict = CapabilitiesMixIn.ethIntfModeToFecTypeSpeedDict(
                                          self.modes, defaultCheck=True )
      try:
         self._hasClientFec
      except TypeError:
         _hasClientFec = False
      else:
         _hasClientFec = bool( self._hasClientFec )
      return CapabilitiesMixIn.fecTypeSpeedDictToFmtStr(
                                    fecDict, clientFecSet=_hasClientFec )

   def renderCoherentErrorCorrectionToStr( self ):
      if not self.coherentModes:
         return ''

      typeDict = {}
      for cmode in self.coherentModes:
         for fecType in cmode.errorCorrections:
            modes = [] 
            if fecType in typeDict: # pylint: disable=consider-using-get
               modes = typeDict[ fecType ]
            cmodeStr = cmode.modulation
            if fecType == cmode.defaultErrorCorrection:
               cmodeStr += '(default)'
            modes.append( cmodeStr )
            typeDict.update( { fecType: modes } )

      return CapabilitiesMixIn.coherentFecTypeDictToFmtStr( typeDict )

   def renderData( self ):
      fmtStr = self.renderModel()
      fmtStr += self.renderType()
      fmtStr += self.renderSpeedLanesDuplex()
      fmtStr += self.renderFlowControl()
      fmtStr += self.renderAutoneg()
      fmtStr += self.renderModulation( defaultCheck=True )
      fmtStr += self.renderLineRates( defaultCheck=True )
      fmtStr += self.renderTributaries( defaultCheck=True )
      fmtStr += self.renderErrorCorrections()
      return fmtStr

class InterfacesCapabilities( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=InterfaceCapabilitiesData,
                      help='A mapping of interfaces to their capabilities'
                           ' information' )

   def render( self ):
      if not self.interfaces:
         return

      fmtStr = ''
      speedGroupStr = ''
      for intf in Arnet.sortIntf( self.interfaces ):
         fmtStr += '%s\n' % ( intf )
         fmtStr += self.interfaces[ intf ].renderData()

         if self.interfaces[ intf ].speedGroup:
            if self.interfaces[ intf ].speedGroup.masterIntf():
               speedGroupStr = '*  = Requires speed group or master interface' \
                               ' speed change\n'
            else:
               speedGroupStr = '*  = Requires speed group setting change\n'

      fmtStr = speedGroupStr + fmtStr
      print( fmtStr, end=( '' if fmtStr[ -1 ].isspace() else ' ' ) )

class LinkFlapDampingProfileData( Model ):
   demerit = Int( help='Current penalty value' )
   suppressed = Float( help='Last suppression time' )
   reuse = Float( help='Expected reuse time' )

class LinkFlapDampingIntfData( Model ):
   profiles = Dict( keyType=str,
                    valueType=LinkFlapDampingProfileData,
                    help='Mapping between profile names and profile statuses' )

class InterfaceLinkFlapDamping( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=LinkFlapDampingIntfData,
                      help='Mapping between interfaces and profiles active on them' )

   def render( self ):
      if not self.interfaces:
         return

      hdrFmt = '%-15.15s %-15.15s %-10.10s %-20.20s %-20.20s'
      print( hdrFmt % ( 'Interface', 'Profile', 'Penalty', 'Suppressed', 'Re-use' ) )
      print( hdrFmt % ( '-' * 15, '-' * 15, '-' * 10, '-' * 20, '-' * 20 ) )

      fmt = '%-15.15s %-15.15s %-10.10s %-20.20s %-20.20s'
      dateFormat = '%Y-%m-%d %H:%M:%S'
      for intf in Arnet.sortIntf( self.interfaces ):
         intfData = self.interfaces[ intf ]
         for profileName in sorted( intfData.profiles ):
            profileData = intfData.profiles[ profileName ]

            if profileData.suppressed != 0.0:
               suppressDate = datetime.fromtimestamp(
                                 profileData.suppressed
                              ).strftime( dateFormat )
            else:
               suppressDate = ''

            if profileData.reuse != 0.0:
               reuseDate = datetime.fromtimestamp(
                              profileData.reuse
                           ).strftime( dateFormat )
            else:
               reuseDate = ''

            print( fmt % (
               intf, profileName, profileData.demerit, suppressDate, reuseDate
            ) )
