#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import re
from CliModel import Model, Submodel, Bool, Dict, Int, Str, Float, List, Enum
from EventLib import maintenanceOperTaccToCli, triggerTypeTaccToCli, \
      actionKindTaccToCli
from IntfModels import Interface
from CliPlugin.IpGenAddrMatcher import isIpv4, isIpv6
from ArnetModel import Ip4Address, Ip6Address

commonStages = [ 'begin', 'end', 'all' ]
uniqueStages = [ 'bgp', 'ratemon', 'mlag', 'linkdown' ]

capiStages = commonStages + \
             [ "before" + y.capitalize() for y in uniqueStages ] + \
             [ "after" + y.capitalize() for y in uniqueStages ]

cliStages = commonStages + \
            [ "before stage " + y for y in uniqueStages ] + \
            [ "after stage " + y for y in uniqueStages ]
            
taccStages = commonStages + \
             [ "before_" + y for y in uniqueStages ] + \
             [ "after_" + y for y in uniqueStages ]

eventStageTaccToCapi = { taccStages[ x ] : capiStages[ x ]
                         for x in range ( 0, len( taccStages ) ) }

eventStageCapiToCli = { capiStages[ x ] : cliStages[ x ]
                        for x in range ( 0, len( capiStages ) ) }

eventStageCapiToTacc = { capiStages[ x ] : taccStages[ x ]
                         for x in range ( 0, len( capiStages ) ) }

def _formatTimeDelta( tacSeconds ):
   prettyString = []
   totalSecs = int( tacSeconds )
   tSecs = totalSecs

   # Split up the seconds into days, hours, minutes and seconds
   tDays, tSecs = divmod( tSecs, 86400 )
   tHours, tSecs = divmod( tSecs, 3600 )
   tMins, tSecs = divmod( tSecs, 60 )

   # Format the time
   if tDays:
      prettyString.append( str( tDays ) )
      prettyString.append( 'day' if tDays == 1 else 'days' )
   if tHours:
      prettyString.append( str( tHours ) )
      prettyString.append( 'hour' if tHours == 1 else 'hours' )
   if tMins:
      prettyString.append( str( tMins ) )
      prettyString.append( 'minute' if tMins == 1 else 'minutes' )
   if totalSecs == 0 or ( tSecs > 0 and tDays == 0 ):
      prettyString.append( str( tSecs ) )
      prettyString.append( 'second' if tSecs == 1 else 'seconds' )
   prettyString.append( 'ago' )
   return ' '.join( prettyString )

class Bgp( Model ):
   groupName = Str( help="Name of peer group", optional=True )
   ip4Neighbor = Ip4Address( help="Neighbor address", optional=True )
   ip6Neighbor = Ip6Address( help="Neighbor address", optional=True )
   vrfName = Str( help="Configured BGP VRF", optional=True )

class OnMaintenance( Model ):
   onMaintenanceType = Enum( help="The type of maintenance interface",
                             values=[ 'bgp', 'unit', 'intf' ] )
   onMode = Enum( help="Maintenance operation that causes a trigger",
                  values=list( maintenanceOperTaccToCli.values() ) )
   onStage = Enum( help="Trigger condition occurs on maintenance stage", 
                   values=capiStages )
   bgp = Submodel( help="BGP peer whose maintenance causes a trigger",
                   valueType=Bgp, optional=True )
   unit = Str( help="The unit whose maintenance causes a trigger",
               optional=True )
   interface = Interface( help="Name of interface", optional=True )

class OnIntf( Model ):
   __revision__ = 2
   interface = List(
      help="List of monitored interfaces", valueType=Interface )
   onChangeOf = List( help="Action is triggered upon these changes "
                      "to the interface", valueType=str )
   def degrade( self, dictRepr, revision ):
      if revision < 2:
         try:
            _ = Tac.Value( "Arnet::IntfId", dictRepr[ "interface" ][0] )
            dictRepr[ "interface" ] = dictRepr[ "interface" ][0]
         except IndexError:
            dictRepr[ "interface" ] = ""
      return dictRepr

class OnCounters( Model ):
   pollInterval = Float( help="The counter polling interval in seconds" )
   condition = Str( help="The counter condition expression" )  
    
class OnLogging( Model ):
   pollInterval = Float( help="The poll interval in seconds" )
   logRegex = Str( help="Regular expression to use for searching log messages" )

class onCustomCondition( Model ):
   pollInterval = Float( help="The poll interval in seconds" )
   scriptTrigger = Str( help="Multi-line script to evaluate" )

class Trigger ( Model ):
   __revision__ = 2
   triggerType = Enum( help="Configured event trigger condition",
                       values=list( triggerTypeTaccToCli ) )
   onIntf = Submodel( help="Trigger on interface",
                      valueType=OnIntf, optional=True )
   onMaintenance = Submodel( help="Trigger on maintenance",
                             valueType=OnMaintenance, optional=True )
   onCounters = Submodel( help="Trigger on counters",
                             valueType=OnCounters, optional=True )
   onLogging = Submodel( help="Trigger on logging",
                             valueType=OnLogging, optional=True )
   onCustomCondition = Submodel( help="Trigger on Custom Condition",
                             valueType=onCustomCondition, optional=True )

   def fromTacc( self, hConf, hStat ):
      self.triggerType = hConf.triggerType
      if hConf.triggerType == 'onIntf':
         self.onIntf = OnIntf()
         # hConf.intfName is the name of a single interface, or a comma-separated
         # interface names with the first entry is the interface range as entered
         # by the user in the Cli config command
         intfs = hConf.intfName.split()
         if len( intfs ) > 1:
            self.onIntf.interface = intfs[1:]
         else:
            self.onIntf.interface = intfs[0]
         if hConf.operstatus:
            self.onIntf.onChangeOf.append( "operStatus" )
         if hConf.ip:
            self.onIntf.onChangeOf.append( "ipAddress" )
         if hConf.ip6:
            self.onIntf.onChangeOf.append( "ip6Address" )
      elif hConf.triggerType == 'onMaintenance':
         self.onMaintenance = OnMaintenance()
         cliOper = maintenanceOperTaccToCli[ hConf.maintenanceOper ]
         self.onMaintenance.onMode = cliOper
         self.onMaintenance.onStage = eventStageTaccToCapi[ hConf.maintenanceStage ]
         if hConf.maintenanceBgpPeer:
            self.onMaintenance.onMaintenanceType = "bgp"
            self.onMaintenance.bgp = Bgp()
            if isIpv4( hConf.maintenanceBgpPeer ):
               self.onMaintenance.bgp.ip4Neighbor = hConf.maintenanceBgpPeer
            elif isIpv6( hConf.maintenanceBgpPeer ):
               self.onMaintenance.bgp.ip6Neighbor = hConf.maintenanceBgpPeer
            else:
               self.onMaintenance.bgp.groupName = hConf.maintenanceBgpPeer
            if hConf.vrfName != 'default':
               self.onMaintenance.bgp.vrfName = hConf.vrfName
         elif hConf.intfName:
            self.onMaintenance.onMaintenanceType = "intf"
            self.onMaintenance.interface = hConf.intfName 
         else:
            self.onMaintenance.onMaintenanceType = "unit"
            self.onMaintenance.unit = hConf.maintenanceUnitName
      elif hConf.triggerType == 'onCounters':
         self.onCounters = OnCounters()
         self.onCounters.pollInterval = hConf.pollInterval
         self.onCounters.condition = hConf.countersCondition
      elif hConf.triggerType == 'onLogging':
         self.onLogging = OnLogging()
         self.onLogging.pollInterval = hConf.pollInterval
         self.onLogging.logRegex = hConf.logRegex
      elif hConf.triggerType == 'onCustomCondition':
         self.onCustomCondition = onCustomCondition()
         self.onCustomCondition.pollInterval = hConf.pollInterval
         self.onCustomCondition.scriptTrigger = hConf.scriptTrigger

   def doRender( self, delay ):
      # Build the trigger string for the "show event-handler" command.
      # First we build a list of strings in triggerList, and then join them
      # at the end.
      triggerList = []
      # Convert the trigger type from the Tacc name to the CLI name
      triggerList.append( triggerTypeTaccToCli[ self.triggerType ] )
      if self.triggerType == 'onIntf':
         # Build the on-interface trigger type show response
         triggerList.append( ','.join( self.onIntf.interface ) )
         eventStr = [ re.sub( "address", "",
                              self.onIntf.onChangeOf[ x ].lower() )
                      for x in range ( 0, len( self.onIntf.onChangeOf ) ) ]  
         if eventStr:
            triggerList.append( 'on' )
            if len( eventStr ) > 1:
               triggerList.append( ', '.join( eventStr[ :-1 ] ) )
               triggerList.append( 'and' )
               triggerList.append( eventStr[ -1 ] )
            else:
               triggerList += eventStr
         else:
            triggerList.append( 'with no trigger set' )
      elif self.triggerType == 'onMaintenance':
         # Build the on-maintenance trigger type show resonse
         operationStr = self.onMaintenance.onMode
         triggerList.append( operationStr )
         if self.onMaintenance.bgp:
            triggerList.append( 'bgp' )
            if self.onMaintenance.bgp.groupName:
               triggerList.append( '%s' % self.onMaintenance.bgp.groupName )
            elif self.onMaintenance.bgp.ip4Neighbor:
               triggerList.append( '%s' % self.onMaintenance.bgp.ip4Neighbor )
            elif self.onMaintenance.bgp.ip6Neighbor:
               triggerList.append( '%s' % self.onMaintenance.bgp.ip6Neighbor )
            if self.onMaintenance.bgp.vrfName:
               triggerList.append( 'vrf %s' %
                                   self.onMaintenance.bgp.vrfName )
         elif self.onMaintenance.interface:
            triggerList.append( 'interface %s' % 
                                self.onMaintenance.interface.stringValue )
         elif self.onMaintenance.unit:
            triggerList.append( 'unit %s' % self.onMaintenance.unit )

         triggerList.append( eventStageCapiToCli[ self.onMaintenance.onStage ] )

      triggerList.append( 'delay %d second%s'
                            % ( delay, 's' if delay != 1 else '' ) )

      # Create one string from the list, seperate each string with a space
      triggerString = ' '.join( triggerList )
      # Add on-counters additional lines if required
      if self.triggerType == 'onCounters':
         triggerString += '\n  Polling Interval: %d second%s' % \
            ( self.onCounters.pollInterval,
              "s" if self.onCounters.pollInterval > 1 else "" )
         triggerString += '\n  Condition: %s' % self.onCounters.condition

      # Add on-logging additional lines if required
      if self.triggerType == 'onLogging':
         triggerString += '\n  Polling Interval: %d second%s' % \
            ( self.onLogging.pollInterval,
              "s" if self.onLogging.pollInterval > 1 else "" )
         triggerString += '\n  RegEx: %s' % self.onLogging.logRegex

      # Add on-custom-condition additional lines if required
      if self.triggerType == 'onCustomCondition':
         triggerString += '\n  Polling Interval: %d second%s' % \
            ( self.onCustomCondition.pollInterval,
              "s" if self.onCustomCondition.pollInterval > 1 else "" )
         triggerString += '\n  Condition bash: '
         if '\n' not in self.onCustomCondition.scriptTrigger:
            triggerString += self.onCustomCondition.scriptTrigger
         else:
            for line in self.onCustomCondition.scriptTrigger.split( "\n" ):
               if line.strip():
                  triggerString += '\n    ' + line
      return triggerString

class EventSubhandler( Model ):
   lastTriggerTime = Float( help="UTC timestamp of last trigger activation",
                            optional=True )
   lastDetectionTime = Float( help="UTC timetamp of last trigger detection",
                              optional=True )
   triggerCount = Int( help="The number of triggers that have occured" )
   detectionCount = Int( help="The number of trigger detections that "
                         "have occured" )
   lastActionTime = Float( help="UTC timestamp of last action",
                           optional=True )
   actionCount = Int( help="The number of actions that have occured" )
   pollingCount = Int( help="The number of times the event-handler has been polled" )
   def fromTacc( self, hStat ):
      timeNow = Tac.now()
      utcNow = Tac.utcNow()
      if hStat.lastNotificationTime > 0:
         self.lastTriggerTime = hStat.lastNotificationTime + utcNow - timeNow
      self.triggerCount = hStat.notificationCount
      if hStat.lastDetectionTime > 0:
         self.lastDetectionTime = hStat.lastDetectionTime + utcNow - timeNow
      self.detectionCount = hStat.detectionCount

      if hStat.lastActionTime > 0:
         self.lastActionTime = hStat.lastActionTime + utcNow - timeNow

      self.actionCount = hStat.actionCount
      self.pollingCount = hStat.pollingCount

   def doRender( self, sourceName ):
      # Build the "show event-handler" Cli output

      utcNow = Tac.utcNow()
      print( '  Source: %s' % sourceName )
      print( '    Total Polls: %d' % self.pollingCount )
      print( '    Last Trigger Detection Time: %s' %
          ( ( self.lastDetectionTime ) and
               _formatTimeDelta( utcNow - self.lastDetectionTime ) or 'Never' ) )
      print( '    Total Trigger Detections: %d' % self.detectionCount )
      print( '    Last Trigger Activation Time: %s' %
          ( ( self.lastTriggerTime ) and
               _formatTimeDelta( utcNow - self.lastTriggerTime ) or 'Never' ) )
      print( '    Total Trigger Activations: %d' % self.triggerCount )
      print( '    Last Action Time: %s' %
          ( ( self.lastActionTime ) and
               _formatTimeDelta( utcNow - self.lastActionTime ) or 'Never' ) )
      print( '    Total Actions: %d' % self.actionCount )

class EventHandler( Model ):
   __revision__ = 2
   trigger = Submodel( help="The condition that causes a trigger",
                       valueType=Trigger, optional=True )
   delay = Float( help="The configured event-handler delay in seconds" )
   repeatInterval = Float( help="The period in seconds for which repeating "
                           "events are ignored" )
   maxRepeatActionCount = Int( help="Maximum actions permitted in the repeat "
                     "interval window" )
   action = Str( help="The action to take when a trigger occurs" )
   healthAction = Str( help="The health action to take when a trigger occurs",
                       optional=True)
   asynchronous = Bool( help="Indicates if action is non-blocking" )
   maxActionDuration = Float( help="The maximum time in seconds the action is "
                              "allowed to execute before being aborted" )
   terminateActionOnTimeout = Bool( help="Indicates that the action should be "
                                    "terminated if it times out" )
   thresholdWindow = Int( help="The window in seconds where the number of "
                             "events must match or exceed the threshold count" )
   thresholdCount = Int( help="The number of events that must occur "
                         "within the threshold window" )
   lastTriggerTime = Float( help="UTC timestamp of last trigger activation",
                            optional=True )
   lastDetectionTime = Float( help="UTC timestamp of last trigger detection",
                              optional=True )
   triggerCount = Int( help="The number of triggers that have occured" )
   detectionCount = Int( help="The number of trigger detections that "
                         "have occured" )
   lastActionTime = Float( help="UTC timestamp of last action",
                           optional=True )
   actionCount = Int( help="The number of actions that have occured" )
   builtinHandler = Bool( help="Indicates that this handler is built-in, not "
                          "user configured" )
   builtinHandlerDisabled = Bool( help="Indicates that this built-in handler is"
                                  "manually disabled by the user" )
   pollingCount = Int( help="The number of times the event-handler has been polled" )
   hasSubhandlers = Bool( help="The handler has per-source granularity" )
   eventSubhandlers = Dict( help="A mapping between a subhandler for this handler"
                                 " and its status information",
                            valueType=EventSubhandler )
   actionKind = Str( "Type of action to take when a trigger occurs" )
   runUponConfig = Bool( help="Indicates if action is to carried out when this"
                              " handler is configured" )

   def fromTacc( self, hConf, hStat, healthStatus ):
      timeNow = Tac.now()
      utcNow = Tac.utcNow()
      self.asynchronous = hStat.asynchronous
      if hConf.triggerType != "none":
         self.trigger = Trigger()
         self.trigger.fromTacc( hConf, hStat )
      self.delay = hStat.delay
      self.repeatInterval = hConf.repeatInterval
      self.maxRepeatActionCount = hConf.maxRepeatActionCount
      self.actionKind = actionKindTaccToCli[ hConf.actionKind ]
      self.action = hConf.command
      self.healthAction = None
      if hConf.metricName != '' and hConf.metricName in healthStatus.metric:
         self.healthAction = "increment metric %s" % hConf.metricName
      self.maxActionDuration = hConf.timeout
      self.terminateActionOnTimeout = hConf.terminateActionOnTimeout
      if hStat.lastNotificationTime > 0:
         # store time in seconds from UTC from epoch
         self.lastTriggerTime = hStat.lastNotificationTime +  utcNow - timeNow
      self.triggerCount = hStat.notificationCount
      if hStat.lastDetectionTime > 0:
         # store time in seconds from UTC from epoch
         self.lastDetectionTime = hStat.lastDetectionTime +  utcNow - timeNow
      self.detectionCount = hStat.detectionCount
 
      if hStat.lastActionTime > 0:
         self.lastActionTime = hStat.lastActionTime + utcNow - timeNow

      self.actionCount = hStat.actionCount

      self.thresholdWindow = int( hConf.threshold )
      self.thresholdCount = hConf.thresholdCount

      self.builtinHandler = hConf.builtinHandler
      self.builtinHandlerDisabled = hConf.builtinHandlerDisabled
      self.hasSubhandlers = hConf.hasSubhandlers
      self.pollingCount = hStat.pollingCount
      self.runUponConfig = hConf.runUponConfig

   def doRender( self, handlerName ):

      utcNow = Tac.utcNow()

      # Show all the handler information
      if self.builtinHandler:
         builtinDesc = ' (BUILT-IN'
         if self.builtinHandlerDisabled:
            builtinDesc += '/DISABLED'
         builtinDesc += ')'
      else:
         builtinDesc = ''
      print( f'Event-handler {handlerName}{builtinDesc}' )

      # Build the trigger string and display it
      triggerList = [ "Trigger:" ]
      if self.asynchronous:
         triggerList.append( 'Asynchronous' )
      triggerList.append( self.trigger.doRender( self.delay )
                          if self.trigger else 'None' )
      print( ' '.join( triggerList ) )

      # Display the remainder of the trigger information
      if self.repeatInterval == 0:
         repeatInterval = "None"
      else:
         repeatInterval = "%.0f seconds" % self.repeatInterval
      print( 'Repeat Interval: %s' % repeatInterval )
      print( 'Maximum Actions Permitted %d in Repeat Interval' %
            self.maxRepeatActionCount )
      print( 'Threshold Time Window: %d Seconds, Event Count: %d times' %
             ( self.thresholdWindow, self.thresholdCount ) )
      print( 'Run action upon config: %s' %
             ( 'Yes' if self.runUponConfig else 'No' ) )
      print( 'Terminate action on timeout: %s' %
             ( 'Yes' if self.terminateActionOnTimeout else 'No' ) )
      if self.actionKind == 'log':
         print( 'Action: %s' % self.actionKind )
      elif '\n' in self.action:
         print( 'Action:' )
         for line in self.action.rstrip().split( "\n" ):
            print( '  ' + line )
      else:
         print( 'Action: %s' % self.action )
      print( 'Device-health Action: %s' % self.healthAction )
      print( 'Action expected to finish in less than %d seconds' %
             self.maxActionDuration )
      if not self.hasSubhandlers:
         if self.trigger and ( self.trigger.triggerType == 'onCounters' ):
            print( 'Total Polls: %d' % self.pollingCount )
         print( 'Last Trigger Detection Time: %s' %
            ( ( self.lastDetectionTime ) and
                _formatTimeDelta( utcNow - self.lastDetectionTime ) or 'Never' ) )
         print( 'Total Trigger Detections: %d' % self.detectionCount )
         print( 'Last Trigger Activation Time: %s' %
            ( ( self.lastTriggerTime ) and
               _formatTimeDelta( utcNow - self.lastTriggerTime ) or 'Never' ) )
         print( 'Total Trigger Activations: %d' % self.triggerCount )
         print( 'Last Action Time: %s' %
            ( ( self.lastActionTime ) and
                _formatTimeDelta( utcNow - self.lastActionTime ) or 'Never' ) )
         print( 'Total Actions: %d' % self.actionCount )
      elif not self.eventSubhandlers:
         print( 'Subhandlers: None' )
      else:
         print( 'Subhandlers:' )
         for key in  sorted( self.eventSubhandlers.keys() ):
            self.eventSubhandlers[ key ].doRender( key )
      print( '' )

class EventHandlers( Model ):
   __revision__ = 2
   eventHandlers = Dict( help="A mapping between a handler and hander information",
                         valueType=EventHandler )
   def render( self ):
      if not self.eventHandlers:
         return
      # Print interfaces
      for key in  self.eventHandlers:
         self.eventHandlers[ key ].doRender( key )

class EventHandlerTriggerCounters( Model ):
   counters = List( help="The list of counters supported by this on-counters "
                    "event handler counter source",
                    valueType=str )

class EventHandlerTrigger( Model ):
   eventHandlerTriggerCounterSources = List( help="The list of counter sources "
                                             "supported by the on-counters event "
                                             "handler", 
                                             valueType=str, optional=True )
   eventHandlerTriggerCounters = Dict( help="A mapping of counter sources to " 
                                       "counters",
                                       valueType=EventHandlerTriggerCounters, 
                                       optional=True )
   def render( self ):
      # pylint: disable-next=singleton-comparison
      if self.eventHandlerTriggerCounterSources != None:
         print( "\nThe list of counter sources supported by the on-counters event "
                "handler\n" )
         for source in sorted ( self.eventHandlerTriggerCounterSources ):
            print( "  " + source )
         print()
      # pylint: disable-next=singleton-comparison
      if self.eventHandlerTriggerCounters != None:
         print( "\nThe list of counters supported by the on-counters event handler" )
         for source in  sorted ( self.eventHandlerTriggerCounters ):
            print()
            for counter in \
                   sorted( self.eventHandlerTriggerCounters[ source ].counters ):
               if counter:
                  separator = '.'
               else:
                  separator = ''
               print( "    " + source + separator + counter )
               print( "    " + source + separator + counter + ".delta" )
         print()
