#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-from-import

import CliCommand
import CliMatcher
import CliPlugin.EventCli as EventCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.VrfCli as VrfCli
import CliPlugin.MaintenanceCliLib as MaintenanceCliLib
import Toggles.EventMgrToggleLib

matcherAction = CliMatcher.KeywordMatcher( 'action',
      helpdesc='Define event-handler action' )
matcherTrigger = CliMatcher.KeywordMatcher( 'trigger',
      helpdesc='Configure event trigger condition' )
matcherOnMaintenance = CliMatcher.KeywordMatcher( 'on-maintenance',
      helpdesc='Trigger condition occurs on maintenance operation' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Trigger condition occurs on maintenance operation of specified '
               'interface' )
matcherUnit = CliMatcher.KeywordMatcher( 'unit',
      helpdesc=( 'Trigger condition occurs on maintenance operation of '
                 'specified unit' ) )
matcherBgp = CliMatcher.KeywordMatcher( 'bgp',
      helpdesc=( 'Trigger condition occurs on maintenance operation of specified '
                 'BGP peer' ) )
matcherStage = CliMatcher.KeywordMatcher( 'stage',
   helpdesc='Maintenance stage name' )
matcherOperation = CliMatcher.EnumMatcher( {
      'enter' : 'Trigger condition occurs on-maintenance enter operation',
      'exit' : 'Trigger condition occurs on-maintenance exit operation',
   } )
matcherStageType = CliMatcher.EnumMatcher( {
      'bgp' : 'Maintenance stage BGP where BGP maintenance profile is applied',
      'ratemon' : ( 'Maintenance stage ratemon where interface maintenance profile '
                    'is applied' ),
      'mlag' : 'Maintenance stage MLAG where MLAG forceful failover is done',
      'linkdown' : ( 'Maintenance stage linkdown where MLAG port-channels are '
                     'shutdown' ),
   } )
matcherBeginEndAll = CliMatcher.EnumMatcher( {
      'begin' : 'Action is triggered at the begining of maintenance operation',
      'end' : 'Action is triggered at the end of maintenance operation',
      'all' : 'Action is triggered at all stages of maintenance operation',
   } )
matcherBeforeAfter = CliMatcher.EnumMatcher( {
      'before' : 'Action is triggered before specified stage',
      'after' : 'Action is triggered after specified stage',
   } )
vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='VRF name' )

class PeerExpr( CliCommand.CliExpression ):
   expression = 'ADDR | V6ADDR | PEER_NAME'
   data = {
      'ADDR' : IpAddrMatcher.IpAddrMatcher( helpdesc='Neighbor address' ),
      'V6ADDR' : Ip6AddrMatcher.Ip6AddrMatcher(  helpdesc='Neighbor address' ),
      'PEER_NAME' : CliMatcher.PatternMatcher( pattern='[-A-Za-z0-9_]+',
         helpdesc='Name of peer group', helpname='WORD' ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'PEER' in args:
         return
      args[ 'PEER' ] = args.get( 'ADDR', args.get( 'V6ADDR',
                        args.get( 'PEER_NAME' ) ) )
#--------------------------------------------------------------------------------
# action bash BASH
#--------------------------------------------------------------------------------
class ActionBashCmd( CliCommand.CliCommandClass ):
   syntax = 'action bash [ CMD ]'
   data = {
      'action': 'Define event-handler action',
      'bash': 'Define BASH command action. <CR> for multi-line',
      'CMD': CliMatcher.StringMatcher( helpdesc='BASH command',
         helpname='COMMAND' ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'CMD' in args:
         EventCli.EventHandlerConfigMode.setAction( mode, args[ 'CMD' ] )
      else:
         EventCli.EventHandlerConfigMode.setActionMultiline( mode )

EventCli.EventHandlerConfigMode.addCommandClass( ActionBashCmd )

#--------------------------------------------------------------------------------
# [ no | default ] action increment device-health metric METRIC
#--------------------------------------------------------------------------------
class ActionIncrementDeviceMetricCmd( CliCommand.CliCommandClass ):
   syntax = 'action increment device-health metric METRIC'
   noOrDefaultSyntax = 'action increment device-health metric ...'
   data = {
      'action': matcherAction,
      'increment': 'Define INCREMENT command action',
      'device-health': 'Actions related to monitoring the health of the system',
      'metric': 'Define METRIC keyword',
      'METRIC': CliMatcher.DynamicNameMatcher(
         lambda m: list( EventCli.healthStatus.metric ),
         helpdesc=( 'Name of device-health metric. '
                    'Use \'show monitor device-health\' to list metrics.' ),
         helpname='METRIC' ),
   }
   handler = EventCli.EventHandlerConfigMode.setHealthAction
   noOrDefaultHandler = EventCli.EventHandlerConfigMode.noOrDefaultHealthAction

EventCli.EventHandlerConfigMode.addCommandClass( ActionIncrementDeviceMetricCmd )

#--------------------------------------------------------------------------------
# [ no | default ] asynchronous
#--------------------------------------------------------------------------------
class AsynchronousCmd( CliCommand.CliCommandClass ):
   syntax = 'asynchronous'
   noOrDefaultSyntax = 'asynchronous ...'
   data = {
      'asynchronous': 'Set the action to be non-blocking',
   }
   handler = EventCli.EventHandlerConfigMode.cmdAsynchronous
   noOrDefaultHandler = EventCli.EventHandlerConfigMode.cmdNoAsynchronous

EventCli.EventHandlerConfigMode.addCommandClass( AsynchronousCmd )

#--------------------------------------------------------------------------------
# delay DELAY
#--------------------------------------------------------------------------------
class DelayCmd( CliCommand.CliCommandClass ):
   syntax = 'delay DELAY'
   data = {
      'delay': 'Configure event-handler delay',
      'DELAY': CliMatcher.IntegerMatcher( 0, 3600,
         helpdesc='Delay period (seconds)' ),
   }
   handler = EventCli.EventHandlerConfigMode.setDelay

EventCli.EventHandlerConfigMode.addCommandClass( DelayCmd )

#--------------------------------------------------------------------------------
# repeat interval DELAY
#--------------------------------------------------------------------------------
class RepeatIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'repeat interval INTERVAL [ count COUNT ]'
   data = {
      'repeat': 'Configure event-handler repetition',
      'interval': 'Set interval for which repeating events are ignored',
      'INTERVAL': CliMatcher.FloatMatcher( 0, 1e11,
         helpdesc='Interval (seconds)', precisionString='%.0f' ),
      'count': ( 'The maximum number of action executions '
                 'during the repeat interval window' ),
      'COUNT': CliMatcher.IntegerMatcher( 1, 100000000,
         helpdesc=( 'Action count' ) ),

   }
   handler = EventCli.EventHandlerConfigMode.setRepeatInterval

EventCli.EventHandlerConfigMode.addCommandClass( RepeatIntervalCmd )

#--------------------------------------------------------------------------------
# threshold THRESHOLD count COUNT
#--------------------------------------------------------------------------------
class ThresholdCountCmd( CliCommand.CliCommandClass ):
   syntax = 'threshold THRESHOLD count COUNT'
   data = {
      'threshold': 'Threshold time window where a number of events should happen',
      'THRESHOLD': CliMatcher.IntegerMatcher( 0, 100000000,
         helpdesc=( 'Threshold time window where a number of '
                    'events should happen (seconds)' ) ),
      'count': ( 'The number of events that should happen within '
                 'the threshold time window' ),
      'COUNT': CliMatcher.IntegerMatcher( 1, 100000000,
         helpdesc=( 'The number of events that should happen within the '
                    'threshold time window' ) ),
   }
   handler = EventCli.EventHandlerConfigMode.setThreshold

EventCli.EventHandlerConfigMode.addCommandClass( ThresholdCountCmd )

#--------------------------------------------------------------------------------
# timeout TIMEOUT
#--------------------------------------------------------------------------------
class TimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'timeout TIMEOUT [ terminate ]'
   data = {
      'timeout': 'Set the expected time the action should finish in',
      'TIMEOUT': CliMatcher.IntegerMatcher( 10, 600,
         helpdesc='Action duration (seconds)' ),
      'terminate': 'Terminate the action if it times out',
   }
   handler = EventCli.EventHandlerConfigMode.setTimeout

EventCli.EventHandlerConfigMode.addCommandClass( TimeoutCmd )

#--------------------------------------------------------------------------------
# trigger ( on-boot | on-startup-config )
#--------------------------------------------------------------------------------
class TriggerOnBootCmd( CliCommand.CliCommandClass ):
   syntax = 'trigger ( on-boot | on-startup-config )'
   data = {
      'trigger': matcherTrigger,
      'on-boot': CliMatcher.KeywordMatcher(
         'on-boot',
         helpdesc='Trigger condition occurs on system boot',
         alternates=[ 'onBoot' ] ) ,
      'on-startup-config': 'Trigger condition occurs on startup config changes'
   }
   handler = EventCli.EventHandlerConfigMode.setTriggerType

EventCli.EventHandlerConfigMode.addCommandClass( TriggerOnBootCmd )

#--------------------------------------------------------------------------------
# trigger on-counters
#--------------------------------------------------------------------------------
class TriggerOnCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'trigger on-counters'
   data = {
      'trigger': matcherTrigger,
      'on-counters': 'Trigger condition occurs on evaluating statistical counters',
   }
   handler = EventCli.EventHandlerConfigMode.gotoTriggerOnCountersMode

EventCli.EventHandlerConfigMode.addCommandClass( TriggerOnCountersCmd )

#--------------------------------------------------------------------------------
# trigger on-logging
#--------------------------------------------------------------------------------
class TriggerOnLoggingCmd( CliCommand.CliCommandClass ):
   syntax = 'trigger on-logging'
   data = {
      'trigger': matcherTrigger,
      'on-logging': 'Trigger condition occurs when regex match any log message',
   }
   handler = EventCli.EventHandlerConfigMode.gotoTriggerOnLoggingMode

EventCli.EventHandlerConfigMode.addCommandClass( TriggerOnLoggingCmd )

#--------------------------------------------------------------------------------
# trigger on-custom-condition
#--------------------------------------------------------------------------------
class TriggerOnCustomCmd( CliCommand.CliCommandClass ):
   syntax = 'trigger on-custom-condition'
   data = {
      'trigger': matcherTrigger,
      'on-custom-condition': 'Trigger based on the return value of a custom script',
   }
   handler = EventCli.EventHandlerConfigMode.gotoTriggerOnCustomMode

if Toggles.EventMgrToggleLib.toggleOnCustomEventEnabled():
   EventCli.EventHandlerConfigMode.addCommandClass( TriggerOnCustomCmd )

#--------------------------------------------------------------------------------
# trigger vm-tracer vm
#--------------------------------------------------------------------------------
class TriggerVmTracerVmCmd( CliCommand.CliCommandClass ):
   syntax = 'trigger vm-tracer vm'
   data = {
      'trigger': matcherTrigger,
      'vm': 'Action is triggered upon changes in VMs in VmTracer',
      'vm-tracer': 'Trigger condition occurs on VmTracer events',
   }
   handler = EventCli.EventHandlerConfigMode.setVmTracerVm

EventCli.EventHandlerConfigMode.addCommandClass( TriggerVmTracerVmCmd )

#--------------------------------------------------------------------------------
# [ default ] condition CONDITION
#--------------------------------------------------------------------------------
class ConditionCmd( CliCommand.CliCommandClass ):
   syntax = 'condition CONDITION'
   defaultSyntax = 'condition ...'
   data = {
      'condition': 'Set the counters condition expression',
      'CONDITION': CliMatcher.StringMatcher(
         helpdesc='Logical expression to evaluate', helpname='EXPRESSION' ),
   }
   handler = EventCli.EventTriggerOnCountersMode.setCondition
   defaultHandler = EventCli.EventTriggerOnCountersMode.defaultCondition

EventCli.EventTriggerOnCountersMode.addCommandClass( ConditionCmd )

#--------------------------------------------------------------------------------
# [ default ] poll interval INTERVAL
#--------------------------------------------------------------------------------
class PollIntervalOnCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'poll interval INTERVAL'
   defaultSyntax = 'poll interval ...'
   data = {
      'poll': 'Set the polling interval',
      'interval': 'Set the polling interval',
      'INTERVAL': CliMatcher.IntegerMatcher( 1, 1000000,
         helpdesc='Poll Interval in seconds' ),
   }
   handler = EventCli.EventTriggerOnCountersMode.setPollingInterval
   defaultHandler = EventCli.EventTriggerOnCountersMode.defaultPollingInterval

EventCli.EventTriggerOnCountersMode.addCommandClass( PollIntervalOnCountersCmd )

#--------------------------------------------------------------------------------
# [ no | default ] trigger on-config disabled
#--------------------------------------------------------------------------------
class TriggerOnConfigDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'trigger on-config disabled'
   noOrDefaultSyntax = syntax
   data = {
      'trigger': matcherTrigger,
      'on-config': 'Trigger the event handler when it is configured',
      'disabled': 'Disable the trigger of the event handler when it is configured',
   }
   handler = EventCli.EventHandlerConfigMode.setTriggerOnConfigDisabled
   noOrDefaultHandler = EventCli.EventHandlerConfigMode.setTriggerOnConfigEnabled

EventCli.EventHandlerConfigMode.addCommandClass( TriggerOnConfigDisabledCmd )

#--------------------------------------------------------------------------------
# [ no | default ] granularity per-source
#--------------------------------------------------------------------------------
class GranularityCmd( CliCommand.CliCommandClass ):
   syntax = 'granularity per-source'
   noOrDefaultSyntax = syntax
   data = { 'granularity': 'Set the granularity of event counting for '
                           'a wildcarded condition',
            'per-source': 'Create a subhandler for each source that matches '
                          'the wildcarded condition' }
   handler = EventCli.EventTriggerOnCountersMode.setPerSourceGranularity
   noOrDefaultHandler = EventCli.EventTriggerOnCountersMode.defaultGranularity

EventCli.EventTriggerOnCountersMode.addCommandClass( GranularityCmd )

#--------------------------------------------------------------------------------
# [ default ] poll interval INTERVAL
#--------------------------------------------------------------------------------
class PollIntervalOnLoggingCmd( CliCommand.CliCommandClass ):
   syntax = 'poll interval INTERVAL'
   defaultSyntax = 'poll interval ...'
   data = {
      'poll': 'Set the maximum polling interval',
      'interval': 'Set the maximum polling interval',
      'INTERVAL': CliMatcher.IntegerMatcher( 1, 1000000,
         helpdesc='Maximum poll Interval in seconds' ),
   }
   handler = EventCli.EventTriggerOnLoggingMode.setMaxPollingInterval
   defaultHandler = EventCli.EventTriggerOnLoggingMode.defaultMaxPollingInterval

EventCli.EventTriggerOnLoggingMode.addCommandClass( PollIntervalOnLoggingCmd )

#--------------------------------------------------------------------------------
# [ default ] regex REGEX
#--------------------------------------------------------------------------------
class RegexCmd( CliCommand.CliCommandClass ):
   syntax = 'regex REGEX'
   defaultSyntax = 'regex ...'
   data = {
      'regex': 'Regular expression to use for searching log messages',
      'REGEX': CliMatcher.StringMatcher(
         helpdesc='Regular expression to use for searching log messages',
         helpname='REGEXP' ),
   }
   handler = EventCli.EventTriggerOnLoggingMode.setLogRegex
   defaultHandler = EventCli.EventTriggerOnLoggingMode.defaultLogRegex

EventCli.EventTriggerOnLoggingMode.addCommandClass( RegexCmd )

#--------------------------------------------------------------------------------
# [ default ] poll interval INTERVAL
#--------------------------------------------------------------------------------
class PollIntervalOnCustomCmd( CliCommand.CliCommandClass ):
   syntax = 'poll interval INTERVAL'
   defaultSyntax = 'poll interval ...'
   data = {
      'poll': 'Set the polling interval',
      'interval': 'Set the polling interval',
      'INTERVAL': CliMatcher.IntegerMatcher( 1, 1000000,
         helpdesc='Poll Interval in seconds' ),
   }
   handler = EventCli.EventTriggerOnCustomMode.setPollingInterval
   defaultHandler = EventCli.EventTriggerOnCustomMode.defaultPollingInterval

EventCli.EventTriggerOnCustomMode.addCommandClass( PollIntervalOnCustomCmd )

#--------------------------------------------------------------------------------
# [ default ] condition bash SCRIPT
#--------------------------------------------------------------------------------
class CustomScriptCmd( CliCommand.CliCommandClass ):
   syntax = 'condition bash [ SCRIPT ]'
   defaultSyntax = 'condition bash ...'
   data = {
      'condition': 'Custom condition script to be evaluated',
      'bash': 'Custom condition script to be evaluated',
      'SCRIPT': CliMatcher.StringMatcher(
         helpdesc='Trigger script Path. Or <CR> for embedded multi-line',
         helpname='SCRIPT' ),
   }
   defaultHandler = EventCli.EventTriggerOnCustomMode.defaultScriptTrigger

   @staticmethod
   def handler( mode, args ):
      if 'SCRIPT' in args:
         EventCli.EventTriggerOnCustomMode.setScriptTrigger( mode, args )
      else:
         EventCli.EventTriggerOnCustomMode.setScriptTriggerMultiline( mode )

EventCli.EventTriggerOnCustomMode.addCommandClass( CustomScriptCmd )

#--------------------------------------------------------------------------------
# trigger on-intf INTFS INPUTS
#--------------------------------------------------------------------------------
class SetOnIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'trigger on-intf INTFS INPUTS'
   data = {
      'trigger' : matcherTrigger,
      'on-intf' : CliMatcher.KeywordMatcher( 'on-intf',
         helpdesc='Trigger condition occurs on specified interface changes',
         alternates=[ 'onIntf' ] ),
      'INTFS' : IntfCli.Intf.rangeMatcher,
      'INPUTS' : CliCommand.SetEnumMatcher( {
         'operstatus' : 'Action is triggered upon changes to interface operStatus',
         'ip' : ( 'Action is triggered upon changes to interface IP address '
                  'assignment' ),
         'ip6' : ( 'Action is triggered upon changes to interface ip6 address '
                   'assignment' ),
      } )
   }

   handler = EventCli.EventHandlerConfigMode.setOnIntf

EventCli.EventHandlerConfigMode.addCommandClass( SetOnIntfCmd )

#--------------------------------------------------------------------------------
# trigger on-maintenance OPERATION bgp PEER [ VRF ] HOOK_TYPE
#--------------------------------------------------------------------------------
class TriggerOnMaintenanceBgpCmd( CliCommand.CliCommandClass ):
   syntax = 'trigger on-maintenance OPERATION bgp PEER [ VRF ] HOOK_TYPE'
   data = {
      'trigger' : matcherTrigger,
      'on-maintenance' : matcherOnMaintenance,
      'OPERATION' : matcherOperation,
      'bgp' : matcherBgp,
      'PEER' : PeerExpr,
      'VRF' : vrfExprFactory,
      'HOOK_TYPE' : matcherBeginEndAll
   }

   handler = EventCli.EventHandlerConfigMode.setOnMaintenanceBgp

EventCli.EventHandlerConfigMode.addCommandClass( TriggerOnMaintenanceBgpCmd )

#--------------------------------------------------------------------------------
# trigger on-maintenance OPERATION bgp PEER [ VRF ] HOOK_TYPE stage STAGE
#--------------------------------------------------------------------------------
class TriggerOnMaintenanceBgpStageCmd( CliCommand.CliCommandClass ):
   syntax = 'trigger on-maintenance OPERATION bgp PEER [ VRF ] HOOK_TYPE stage STAGE'
   data = {
      'trigger' : matcherTrigger,
      'on-maintenance' : matcherOnMaintenance,
      'OPERATION' : matcherOperation,
      'bgp' : matcherBgp,
      'PEER' : PeerExpr,
      'VRF' : vrfExprFactory,
      'HOOK_TYPE' : matcherBeforeAfter,
      'stage' : matcherStage,
      'STAGE' : matcherStageType,
   }

   handler = EventCli.EventHandlerConfigMode.setOnMaintenanceBgp

EventCli.EventHandlerConfigMode.addCommandClass( TriggerOnMaintenanceBgpStageCmd )

#--------------------------------------------------------------------------------
# trigger on-maintenance OPERATION interface INTF HOOK_TYPE
#--------------------------------------------------------------------------------
class OnMaintenanceInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'trigger on-maintenance OPERATION interface INTF HOOK_TYPE'
   data = {
      'trigger' : matcherTrigger,
      'on-maintenance' : matcherOnMaintenance,
      'OPERATION' : matcherOperation,
      'interface' : matcherInterface,
      'INTF' : IntfCli.Intf.matcher,
      'HOOK_TYPE' : matcherBeginEndAll
   }

   handler = EventCli.EventHandlerConfigMode.setOnMaintenanceInterface

EventCli.EventHandlerConfigMode.addCommandClass( OnMaintenanceInterfaceCmd )

#--------------------------------------------------------------------------------
# trigger on-maintenance OPERATION interface INTF HOOK_TYPE stage STAGE
#--------------------------------------------------------------------------------
class OnMaintenanceInterfaceStageCmd( CliCommand.CliCommandClass ):
   syntax = 'trigger on-maintenance OPERATION interface INTF HOOK_TYPE stage STAGE'
   data = {
      'trigger' : matcherTrigger,
      'on-maintenance' : matcherOnMaintenance,
      'OPERATION' : matcherOperation,
      'interface' : matcherInterface,
      'INTF' : IntfCli.Intf.matcher,
      'HOOK_TYPE' : matcherBeforeAfter,
      'stage' : matcherStage,
      'STAGE' : matcherStageType,
   }

   handler = EventCli.EventHandlerConfigMode.setOnMaintenanceInterface

EventCli.EventHandlerConfigMode.addCommandClass( OnMaintenanceInterfaceStageCmd )

#--------------------------------------------------------------------------------
# trigger on-maintenance OPERATION unit UNIT_NAME HOOK_TYPE
#--------------------------------------------------------------------------------
class OnMaintenanceUnitCmd( CliCommand.CliCommandClass ):
   syntax = 'trigger on-maintenance OPERATION unit UNIT_NAME HOOK_TYPE'
   data = {
      'trigger' : matcherTrigger,
      'on-maintenance' : matcherOnMaintenance,
      'OPERATION' : matcherOperation,
      'unit' : matcherUnit,
      'UNIT_NAME' : MaintenanceCliLib.matcherUnitName,
      'HOOK_TYPE' : matcherBeginEndAll
   }

   handler = EventCli.EventHandlerConfigMode.setOnMaintenanceUnit

EventCli.EventHandlerConfigMode.addCommandClass( OnMaintenanceUnitCmd )

#--------------------------------------------------------------------------------
# trigger on-maintenance OPERATION unit UNIT_NAME HOOK_TYPE stage STAGE
#--------------------------------------------------------------------------------
class OnMaintenanceUnitStageCmd( CliCommand.CliCommandClass ):
   syntax = 'trigger on-maintenance OPERATION unit UNIT_NAME HOOK_TYPE stage STAGE'
   data = {
      'trigger' : matcherTrigger,
      'on-maintenance' : matcherOnMaintenance,
      'OPERATION' : matcherOperation,
      'unit' : matcherUnit,
      'UNIT_NAME' : MaintenanceCliLib.matcherUnitName,
      'HOOK_TYPE' : matcherBeforeAfter,
      'stage' : matcherStage,
      'STAGE' : matcherStageType,
   }

   handler = EventCli.EventHandlerConfigMode.setOnMaintenanceUnit

EventCli.EventHandlerConfigMode.addCommandClass( OnMaintenanceUnitStageCmd )

