#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=ungrouped-imports

import Tac
import CliCommand
import CliMatcher
import Url
# pylint: disable-next=consider-using-from-import
import CliPlugin.EventMonCli as EventMonCli
import BasicCliModes
import Toggles.EventMonToggleLib as Toggle

from EventMonCliTables import allTableMatcher, qtTableMatcher

matcherBackup = CliMatcher.KeywordMatcher( 'backup', 
      helpdesc='Backed up log files' )
matcherBuffer = CliMatcher.KeywordMatcher( 
      'buffer', helpdesc='Local buffer settings' )
matcherEventMonitor = CliMatcher.KeywordMatcher( 
      'event-monitor', helpdesc='Enable event-monitor agent' )
matcherMaxSize = CliMatcher.KeywordMatcher( 
      'max-size', helpdesc='Maximum number of stored backup logs' )
matcherMaxBufSize = CliMatcher.KeywordMatcher( 
      'max-size', helpdesc='Maximum size of the primary event monitor log' )
matcherPath = CliMatcher.KeywordMatcher( 'path', helpdesc='Path to backup logs' )


BackupSizeType = Tac.Type( 'EventMon::BackupSize' )
BufferSizeType = Tac.Type( 'EventMon::BufferSize' )
TableSizeType = Tac.Type( 'EventMon::TableSize' )

#--------------------------------------------------------------------------------
# ( no | default ) event-monitor backup max-size SIZE
#--------------------------------------------------------------------------------
class EventMonitorBackupMaxSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'event-monitor backup max-size SIZE'
   noOrDefaultSyntax = 'event-monitor backup max-size ...'
   data = {
      'event-monitor': matcherEventMonitor,
      'backup': matcherBackup,
      'max-size': matcherMaxSize,
      'SIZE': CliMatcher.IntegerMatcher( BackupSizeType.min, BackupSizeType.max, 
         helpdesc='Maximum number of stored backup logs' )
   }
   handler = EventMonCli.setForeverLogMaxSize
   noOrDefaultHandler = EventMonCli.setForeverLogMaxSize

BasicCliModes.GlobalConfigMode.addCommandClass( EventMonitorBackupMaxSizeCmd )

#--------------------------------------------------------------------------------
# ( no | default ) event-monitor TABLE max-size SIZE
#--------------------------------------------------------------------------------
class EventMonitorMaxTableSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'event-monitor TABLE max-size [ SIZE ]'
   noOrDefaultSyntax = 'event-monitor TABLE max-size ...'
   data = {
      'event-monitor': matcherEventMonitor,
      'TABLE': allTableMatcher,
      'max-size': CliMatcher.KeywordMatcher( 'max-size',
         helpdesc='Table max size.' ),
      'SIZE': CliMatcher.PatternMatcher( pattern='[0-9]+[b|k|m|g|B|K|M|G]?',
         helpdesc='Maximum size of table in bytes'
         'k for kilobytes, m for megabytes, g for gigabytes', helpname='WORD' )
   }
   handler = EventMonCli.setDbMaxTableSize
   noOrDefaultHandler = EventMonCli.setDbMaxTableSize

BasicCliModes.GlobalConfigMode.addCommandClass( EventMonitorMaxTableSizeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] event-monitor 
#  [route|route6|mroute|lacp|arp|neighbor|mac|igmpsnooping|
#  stpunstable|vxlanmac|nat] backup path URL
#--------------------------------------------------------------------------------
class EventMonitorBackupPathCmd( CliCommand.CliCommandClass ):
   syntax = \
      ('event-monitor [ TABLE ] backup path URL')
   noOrDefaultSyntax = ('event-monitor [ TABLE ] backup [ path ] ...')
   data = {
      'event-monitor': matcherEventMonitor,
      'TABLE': qtTableMatcher,
      'backup': matcherBackup,
      'path': matcherPath,
      'URL': Url.UrlMatcher(
         fsFunc=lambda fs: fs.fsType == 'flash', helpdesc='forever log URL' ), 

   }
   handler = EventMonCli.setForeverLogPath
   noOrDefaultHandler = EventMonCli.restoreForeverLogPath

BasicCliModes.GlobalConfigMode.addCommandClass( EventMonitorBackupPathCmd )

#--------------------------------------------------------------------------------
# [ no | default ] event-monitor buffer max-size SIZE
#--------------------------------------------------------------------------------
class EventMonitorBufferMaxSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'event-monitor buffer max-size SIZE'
   noOrDefaultSyntax = 'event-monitor buffer max-size ...'
   data = {
      'event-monitor': matcherEventMonitor,
      'buffer': matcherBuffer,
      'max-size': matcherMaxBufSize,
      'SIZE': CliMatcher.IntegerMatcher( BufferSizeType.min, BufferSizeType.max, 
         helpdesc='Maximum size of the primary event monitor log' ),
   }
   handler = EventMonCli.setBufferSize
   noOrDefaultHandler = EventMonCli.setBufferSize

BasicCliModes.GlobalConfigMode.addCommandClass( EventMonitorBufferMaxSizeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] event-monitor
#--------------------------------------------------------------------------------
class EventMonitorCmd( CliCommand.CliCommandClass ):
   syntax = 'event-monitor'
   noOrDefaultSyntax = syntax
   data = {
      'event-monitor': 'Enable event-monitor agent',
   }

   @staticmethod
   def handler( mode, args ):
      EventMonCli.config.agentEnabled = True

   @staticmethod
   def noHandler ( mode, args ):
      EventMonCli.config.agentEnabled = False

   @staticmethod
   def defaultHandler ( mode, args ):
      EventMonCli.config.agentEnabled = EventMonCli.config.defaultAgentEnabled

BasicCliModes.GlobalConfigMode.addCommandClass( EventMonitorCmd )

#--------------------------------------------------------------------------------
# ( no | default ) event-monitor nexthop-group ( backup-activation )
#--------------------------------------------------------------------------------
class EventMonitorNexthopGroupTablesCmd( CliCommand.CliCommandClass ):
   syntax = ('event-monitor nexthop-group ( backup-activation )')
   noOrDefaultSyntax = syntax
   data = {
      'event-monitor': matcherEventMonitor,
      'nexthop-group': 'Nexthop group events',
      'backup-activation': 'Add nexthop group backup activation events to table',
   }

   @staticmethod
   def handler( mode, args ):
      backupActivationTable = args.get( 'backup-activation' )
      if backupActivationTable in EventMonCli.cliTokenToTableNameMapping:
         mappedTable = EventMonCli.cliTokenToTableNameMapping[ \
                       backupActivationTable ]
         EventMonCli.config.table[ mappedTable ].enabled = True

   @staticmethod
   def noHandler ( mode, args ):
      backupActivationTable = args.get( 'backup-activation' )
      if backupActivationTable in EventMonCli.cliTokenToTableNameMapping:
         mappedTable = EventMonCli.cliTokenToTableNameMapping[ \
                       backupActivationTable ]
         EventMonCli.config.table[ mappedTable ].enabled = False

   @staticmethod
   def defaultHandler ( mode, args ):
      backupActivationTable = args.get( 'backup-activation' )
      if backupActivationTable in EventMonCli.cliTokenToTableNameMapping:
         mappedTable = EventMonCli.cliTokenToTableNameMapping[ \
                       backupActivationTable ]
         table = EventMonCli.config.table[ mappedTable ]
         table.enabled = table.initialEnabledValue

if Toggle.toggleFecBackupactivationEnabled():
   BasicCliModes.GlobalConfigMode.addCommandClass( \
       EventMonitorNexthopGroupTablesCmd )

#--------------------------------------------------------------------------------
# [ no | default ] event-monitor 
#   [all|route|route6|mroute|lacp|arp|neighbor|mac|
#   igmpsnooping|stpunstable|vxlanmac|nat]
#--------------------------------------------------------------------------------
class EventMonitorTableCmd( CliCommand.CliCommandClass ):
   syntax = ('event-monitor TABLE')
   noOrDefaultSyntax = syntax
   data = {
      'event-monitor': 'Enable event-monitor agent',
      'TABLE': allTableMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      enabledTable = args.get( 'TABLE' )
      if enabledTable and enabledTable == 'all':
         for table in EventMonCli.config.table.values():
            table.enabled = True
      else:
         if enabledTable in EventMonCli.cliTokenToTableNameMapping:
            mappedTable = EventMonCli.cliTokenToTableNameMapping[ enabledTable ]
            EventMonCli.config.table[ mappedTable ].enabled = True

         EventMonCli.config.table[enabledTable].enabled = True

   @staticmethod
   def noHandler ( mode, args ):
      disabledTable = args.get( 'TABLE' )
      if disabledTable and disabledTable == 'all':
         for table in EventMonCli.config.table.values():
            table.enabled = False
      else:
         if disabledTable in EventMonCli.cliTokenToTableNameMapping:
            mappedTable = EventMonCli.cliTokenToTableNameMapping[ disabledTable ]
            EventMonCli.config.table[ mappedTable ].enabled = False
         EventMonCli.config.table[disabledTable].enabled = False

   @staticmethod
   def defaultHandler ( mode, args ):
      defaultTable = args.get( 'TABLE' )
      if defaultTable == 'all':
         for table in EventMonCli.config.table.values():
            table.enabled = table.initialEnabledValue
      else:
         if defaultTable in EventMonCli.cliTokenToTableNameMapping:
            mappedTable = EventMonCli.cliTokenToTableNameMapping[ defaultTable ]
            table = EventMonCli.config.table[ mappedTable ]
            table.enabled = table.initialEnabledValue
         table = EventMonCli.config.table[defaultTable]
         table.enabled = table.initialEnabledValue

BasicCliModes.GlobalConfigMode.addCommandClass( EventMonitorTableCmd )

