#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=ungrouped-imports

import BasicCli
import CliCommand
import CliMatcher
import ShowCommand
# pylint: disable-next=consider-using-from-import
import CliPlugin.EventMonCli as EventMonCli
import CommonGuards
from CliMatcher import StringMatcher
from EventMonCliTables import allTableMatcher
from Intf.IntfRange import IntfRangeMatcher
import Toggles.EventMonToggleLib as Toggle

matcherEventMonitor = CliMatcher.KeywordMatcher( 'event-monitor', 
      helpdesc='Show event monitor logs' )
matcherGroupBy = CliCommand.Node( CliMatcher.KeywordMatcher( 'group-by', 
      helpdesc='Group the results by attribute' ), maxMatches=1 )
matcherLimit = CliCommand.Node( CliMatcher.KeywordMatcher( 'limit', 
      helpdesc='Limit the number of messages' ), maxMatches=1 )
matcherMatchInterface = CliCommand.Node( CliMatcher.KeywordMatcher( 
      'match-interface', helpdesc='Filter results by interface' ), 
      maxMatches=1 )
matcherMatchIPv4 = CliMatcher.KeywordMatcher( 'match-ip', 
      helpdesc='Filter results by IPv4 address' )
matcherMatchIPv6 = CliMatcher.KeywordMatcher( 'match-ipv6', 
      helpdesc='Filter results by IPv6 address' )
matcherMatchMac = CliMatcher.KeywordMatcher( 'match-mac', 
      helpdesc='Filter results by MAC address' )
matcherMatchTime = CliMatcher.KeywordMatcher( 'match-time', 
      helpdesc='Filter results by time' )
matcherMatchType = CliMatcher.KeywordMatcher( 'match-type', 
      helpdesc='Filter results by MAC address entry type' )
matcherMatchVlan = CliMatcher.KeywordMatcher( 'match-vlan', 
      helpdesc='Filter results by VLAN number' )
matcherMatchVtep = CliMatcher.KeywordMatcher( 'match-vtep', 
      helpdesc='Filter results by VTEP address' )
matcherMatchVrf = CliCommand.Node( CliMatcher.KeywordMatcher( 'match-vrf', 
      helpdesc='Filter results by VRF name' ), maxMatches=1 )
matcherMatchNexthopGroup = CliCommand.Node( CliMatcher.KeywordMatcher( 
      'match-nexthop-group', helpdesc='Filter results by nexthop group name' ),
       maxMatches=1 )
matcherMatchNexthopGroupVersion = CliCommand.Node( CliMatcher.KeywordMatcher( 
      'match-nexthop-group-version',
      helpdesc='Filter results by nexthop group version' ),
      maxMatches=1 )

matcherMatchLimit = CliCommand.Node( matcher=CliMatcher.IntegerMatcher( 1, 
      15000, helpdesc='Message limit' ), maxMatches=1 )
nhgVersionPatternMatch = CliCommand.Node( matcher=CliMatcher.IntegerMatcher( 1, 
      65535, helpdesc='VersionId' ), maxMatches=1 )

nodeEventMonitor = CliCommand.Node( matcher=matcherEventMonitor, 
      guard=CommonGuards.standbyGuard )
macPatternMatch = CliMatcher.PatternMatcher( pattern='.+',
      helpdesc='Enter MAC address regular expression. Ex: aaaa.bbbb.cccc.%',
      helpname='REGEX' )
ipPatternMatch = CliMatcher.PatternMatcher( pattern='.+',
      helpdesc='Enter IPv4 address regular expression. Ex: 10.10.10.%',
      helpname='REGEX' )
ip6PatternMatch = CliMatcher.PatternMatcher( pattern='.+',
      helpdesc='Enter IPv6 address regular expression. Ex: aaaa:bbbb::%',
      helpname='REGEX' )
typePatternMatch = CliMatcher.PatternMatcher( pattern='.+',
      helpdesc='Enter entry type regular expression', helpname='REGEX' )
vtepPatternMatch = CliMatcher.PatternMatcher( pattern='.+',
      helpdesc='Enter VTEP address regular expression', helpname='REGEX' )
vrfPatternMatch = CliMatcher.PatternMatcher( pattern='.+',
      helpdesc='Enter VRF name regular expression', helpname='REGEX' )
nhgNamePatternMatch = CliMatcher.PatternMatcher( pattern='.+',
      helpdesc='Enter NHG name regular expression', helpname='REGEX' )
matcherTimeEnum = CliCommand.Node(
      matcher=CliMatcher.EnumMatcher( {
         'last-minute': 'Match events in the last minute',
         'last-hour': 'Match events in the last hour',
         'last-day': 'Match events in the last day',
         'last-week': 'Match events in the last week'
      } ), maxMatches=1 )


#--------------------------------------------------------------------------------
# show event-monitor all
#--------------------------------------------------------------------------------
class EventMonitorAllCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show event-monitor all'
   data = {
      'event-monitor': nodeEventMonitor,
      'all': 'Monitor all events',
   }
   handler = EventMonCli.showTables
   privileged = True

BasicCli.addShowCommandClass( EventMonitorAllCmd )

#--------------------------------------------------------------------------------
# show event-monitor arp [ { ( ( group-by GROUPBY ) | 
# ( limit LIMIT ) | ( match-mac MACADDRESS ) |
# match-time ( last-minute | last-hour | last-day | last-week ) | 
# ( match-interface INTFS ) | 
# ( match-vrf VRF ) | ( match-ip IPADDRESS ) ) } ]
#--------------------------------------------------------------------------------
def showArpTable( mode, args ):
   EventMonCli.showTable( mode, args, "arp" )

class EventMonitorArpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show event-monitor arp 
               [ { ( group-by GROUPBY ) 
                 | ( limit LIMIT ) 
                 | ( match-mac MACADDRESS ) 
                 | ( match-time TIME ) 
                 | ( match-interface INTFS ) 
                 | ( match-vrf VRF ) 
                 | ( match-ip IPADDRESS ) } ]'''
   data = {
      'event-monitor': nodeEventMonitor,
      'arp': 'Monitor ARP table events',
      'group-by': matcherGroupBy,
      'GROUPBY': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            'interface': 'Group by interface',
            'vrf': 'Group by VRF',
            'mac': 'Group by MAC address',
            'ip': 'Group by IPv4 address'
         } ), maxMatches=1 ),
      'limit': matcherLimit,
      'LIMIT': matcherMatchLimit,
      'match-mac': matcherMatchMac,
      'MACADDRESS': macPatternMatch,
      'match-time': matcherMatchTime,
      'TIME': matcherTimeEnum, 
      'match-interface': matcherMatchInterface,
      'INTFS': IntfRangeMatcher( helpdesc="interface of the ARP events" ),
      'match-vrf': matcherMatchVrf,
      'VRF': vrfPatternMatch, 
      'match-ip': matcherMatchIPv4,
      'IPADDRESS': ipPatternMatch, 
   }
   handler = showArpTable
   privileged = True

BasicCli.addShowCommandClass( EventMonitorArpCmd )

#--------------------------------------------------------------------------------
# show event-monitor mac [ { ( ( group-by GROUPBY ) | ( limit LIMIT ) |
# match-time ( last-minute | last-hour | last-day | last-week ) |
# ( match-mac MACADDRESS ) | ( match-interface INTFS ) |
# ( match-type TYPE ) | ( match-vlan VLANID ) | ( match-vtep VTEP ) ) } ]
#--------------------------------------------------------------------------------
def showMacTable( mode, args ):
   EventMonCli.showTable( mode, args, "mac" )

class EventMonitorMacCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show event-monitor mac
               [ { ( group-by GROUPBY )
                 | ( limit LIMIT )
                 | ( match-time TIME )
                 | ( match-mac MACADDRESS )
                 | ( match-interface INTFS )
                 | ( match-type TYPE )
                 | ( match-vtep VTEP )
                 | ( match-vlan VLANID ) } ]'''
   data = {
      'event-monitor': nodeEventMonitor,
      'mac': 'Monitor MAC table events',
      'group-by': matcherGroupBy,
      'GROUPBY': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            'interface': 'Group by interface',
            'mac': 'Group by MAC address',
            'vlan': 'Group by VLAN',
            'vtep': 'Group by VTEP IP address'
         } ), maxMatches=1 ),      
      'limit': matcherLimit,
      'LIMIT': matcherMatchLimit,
      'match-time': matcherMatchTime,
      'TIME': matcherTimeEnum,
      'match-mac': matcherMatchMac,
      'MACADDRESS': macPatternMatch, 
      'match-interface': matcherMatchInterface,
      'INTFS': IntfRangeMatcher( helpdesc="Interface of the MAC events" ), 
      'match-type': matcherMatchType,
      'TYPE': typePatternMatch,
      'match-vtep': matcherMatchVtep,
      'VTEP': vtepPatternMatch,
      'match-vlan': matcherMatchVlan,
      'VLANID': CliMatcher.IntegerMatcher( 1, 4094, 
            helpdesc='Identifier for a Virtual LAN' )
   }
   handler = showMacTable
   privileged = True

BasicCli.addShowCommandClass( EventMonitorMacCmd )

#--------------------------------------------------------------------------------
# show event-monitor neighbor [ { ( ( group-by GROUPBY ) | ( limit LIMIT ) | 
# ( match-mac MACADDRESS ) | 
# match-time ( last-minute | last-hour | last-day | last-week ) | 
# ( match-interface INTFS ) | ( match-vrf VRF ) | 
# ( match-ipv6 IPV6ADDRESS ) ) } ]
#--------------------------------------------------------------------------------
def showNeighborTable( mode, args ):
   EventMonCli.showTable( mode, args, "neighbor" )

class EventMonitorNeighborCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ('show event-monitor neighbor [ { ( ( group-by GROUPBY ) |' 
   ' ( limit LIMIT ) | ( match-mac MACADDRESS ) |' 
   ' ( match-time TIME ) |' 
   ' ( match-interface INTFS ) | ( match-vrf VRF ) |' 
   ' ( match-ipv6 IPV6ADDRESS ) ) } ]')
   data = {
      'event-monitor': nodeEventMonitor,
      'neighbor': 'Monitor neighbor events',
      'group-by': matcherGroupBy,
      'GROUPBY': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            'interface': 'Group by interface',
            'vrf': 'Group by VRF',
            'mac': 'Group by MAC address',
            'ip6': 'Group by IPv6 address'
         } ), maxMatches=1 ),
      'limit': matcherLimit,
      'LIMIT': matcherMatchLimit,
      'match-mac': matcherMatchMac,
      'MACADDRESS': macPatternMatch, 
      'match-time': matcherMatchTime,
      'TIME': matcherTimeEnum, 
      'match-interface': matcherMatchInterface,
      'INTFS': IntfRangeMatcher( helpdesc="Interface of the neighbor events" ),
      'match-vrf': matcherMatchVrf,
      'VRF': vrfPatternMatch, 
      'match-ipv6': matcherMatchIPv6,
      'IPV6ADDRESS': ip6PatternMatch, 
   }
   handler = showNeighborTable
   privileged = True

BasicCli.addShowCommandClass( EventMonitorNeighborCmd )

#--------------------------------------------------------------------------------
# show event-monitor route6 [ { ( ( group-by GROUPBY ) | 
# ( limit LIMIT ) | match-time ( last-minute | last-hour | last-day | last-week ) |
# ( match-ipv6 IPV6ADDRESS ) ) } ]
#--------------------------------------------------------------------------------
def showRoute6Table( mode, args ):
   EventMonCli.showTable( mode, args, "route6" )

class EventMonitorRoute6Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ('show event-monitor route6 [ { ( ( group-by GROUPBY ) |' 
         ' ( limit LIMIT ) |' 
         ' ( match-time TIME ) |' 
         ' ( match-ipv6 IPV6ADDRESS ) ) } ]')
   data = {
      'event-monitor': nodeEventMonitor,
      'route6': 'Monitor route6 events',
      'group-by': matcherGroupBy,
      'GROUPBY': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            'ip6': 'Group by IPv6 address'
         } ), maxMatches=1 ),
      'limit': matcherLimit,
      'LIMIT': matcherMatchLimit,
      'match-time': matcherMatchTime,
      'TIME': matcherTimeEnum,
      'match-ipv6': matcherMatchIPv6,
      'IPV6ADDRESS': ip6PatternMatch, 
   }
   handler = showRoute6Table
   privileged = True

BasicCli.addShowCommandClass( EventMonitorRoute6Cmd )

#--------------------------------------------------------------------------------
# show event-monitor route [ { ( ( group-by GROUPBY ) | ( limit LIMIT ) | 
# match-time ( last-minute | last-hour | last-day | last-week ) | 
# ( match-ip IPADDRESS ) ) } ]
#--------------------------------------------------------------------------------
def showRouteTable( mode, args ):
   EventMonCli.showTable( mode, args, "route" )

class EventMonitorRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show event-monitor route [ { ( ( group-by GROUPBY ) | '
   '( limit LIMIT ) |' 
   ' ( match-time TIME ) |'
   ' ( match-ip IPADDRESS ) ) } ]' )
   data = {
      'event-monitor': nodeEventMonitor,
      'route': 'Monitor routing events',
      'group-by': matcherGroupBy,
      'GROUPBY': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            'ip': 'Group by IPv4 address',
         } ), maxMatches=1 ),
      'limit': matcherLimit,
      'LIMIT': matcherMatchLimit,
      'match-time': matcherMatchTime,
      'TIME': matcherTimeEnum,
      'match-ip': matcherMatchIPv4,
      'IPADDRESS': ipPatternMatch, 
   }
   handler = showRouteTable
   privileged = True

BasicCli.addShowCommandClass( EventMonitorRouteCmd )

#--------------------------------------------------------------------------------
# show event-monitor sqlite { STRING }
#--------------------------------------------------------------------------------
class EventMonitorSqliteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show event-monitor sqlite { QUERY }'
   data = {
      'event-monitor': nodeEventMonitor,
      'sqlite': 'Enter a SQLite statment',
      'QUERY': StringMatcher( helpdesc='Enter a full SQL statement', 
         helpname='SQL Statement' ),
   }
   handler = EventMonCli.showEventMon
   prepareFunction = EventMonCli.prepShowEventMon
   privileged = True

BasicCli.addShowCommandClass( EventMonitorSqliteCmd )

#--------------------------------------------------------------------------------
# show event-monitor stpunstable [ { ( ( limit LIMIT ) | 
# match-time ( last-minute | last-hour | last-day | last-week ) ) } ]
#--------------------------------------------------------------------------------
def showStpunstableTable( mode, args ):
   EventMonCli.showTable ( mode, args, "stpunstable" )

class EventMonitorStpunstableCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show event-monitor stpunstable [ { ( ( limit LIMIT )'
         ' | ( match-time TIME ) ) } ]' )
   data = {
      'event-monitor': nodeEventMonitor,
      'stpunstable': 'Monitor STP unstable events',
      'limit': matcherLimit,
      'LIMIT': matcherMatchLimit,
      'match-time': matcherMatchTime,
      'TIME': matcherTimeEnum   
      }
   handler = showStpunstableTable
   privileged = True

BasicCli.addShowCommandClass( EventMonitorStpunstableCmd )

#--------------------------------------------------------------------------------
# show event-monitor igmpsnooping [ { ( ( group-by GROUPBY ) | ( limit LIMIT ) | 
# ( match-mac MACADDRESS ) | 
# match-time ( last-minute | last-hour | last-day | last-week ) | 
# ( match-interface INTFS ) | ( match-vlan VLANID ) ) } ]
#--------------------------------------------------------------------------------
def showIgmpSnoopingTable( mode, args ):
   EventMonCli.showTable( mode, args, "igmpsnooping" )

class EventMonitorIgmpsnoopingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show event-monitor igmpsnooping [ { ( ( group-by GROUPBY ) |' 
         ' ( limit LIMIT ) | ( match-mac MACADDRESS ) |' 
         ' ( match-time TIME ) |' 
         ' ( match-interface INTFS ) | ( match-vlan VLANID ) ) } ]' )
   data = {
      'event-monitor': nodeEventMonitor,
      'igmpsnooping': 'Monitor IGMP snooping table events',
      'group-by': matcherGroupBy,
      'GROUPBY': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            'interface': 'Group by interface',
            'vlan': 'Group by VLAN',
            'mac': 'Group by MAC address',
         } ), maxMatches=1 ),
      'limit': matcherLimit,
      'LIMIT': matcherMatchLimit,
      'match-mac': matcherMatchMac,
      'MACADDRESS': macPatternMatch, 
      'match-time': matcherMatchTime,
      'TIME': matcherTimeEnum,
      'match-interface': matcherMatchInterface,
      'INTFS': IntfRangeMatcher( helpdesc="Interface of the IGMP events" ), 
      'match-vlan': 'Filter results by VLAN',
      'VLANID': CliMatcher.IntegerMatcher( 1, 4094, 
            helpdesc='Identifier for a Virtual LAN' )
   }
   handler = showIgmpSnoopingTable
   privileged = True

BasicCli.addShowCommandClass( EventMonitorIgmpsnoopingCmd )

#--------------------------------------------------------------------------------
# show event-monitor mroute [ { ( ( group-by GROUPBY ) | ( limit LIMIT ) |
# match-time ( last-minute | last-hour | last-day | last-week ) | 
# ( match-interface INTFS ) | ( match-source-ip IPADDRESS ) | 
# ( match-group-ip IPADDRESS ) ) } ]
#--------------------------------------------------------------------------------
def showMrouteTable( mode, args ):
   EventMonCli.showTable( mode, args, "mroute" )

class EventMonitorMrouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ('show event-monitor mroute '
   '[ { ( ( group-by GROUPBY ) | ( limit LIMIT ) |' 
   ' ( match-time TIME ) |' 
   ' ( match-interface INTFS ) | ( match-source-ip SRCADDR ) |' 
   ' ( match-group-ip GRPADDR ) ) } ]')
   data = {
      'event-monitor': nodeEventMonitor,
      'mroute': 'Monitor mroute table events',
      'group-by': matcherGroupBy,
      'GROUPBY': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            'interface': 'Group by interface',
            'source-ip': 'Group by source IP address',
            'group-ip': 'Group by group IP address',
         } ), maxMatches=1 ),
      'limit': matcherLimit,
      'LIMIT': matcherMatchLimit,
      'match-time': matcherMatchTime,
      'TIME': matcherTimeEnum,
      'match-interface': matcherMatchInterface,
      'INTFS': IntfRangeMatcher( helpdesc="interface of the mroute events" ), 
      'match-source-ip': 'Filter results by source IP',
      'SRCADDR': ipPatternMatch, 
      'match-group-ip': 'Filter results by group IP',
      'GRPADDR': ipPatternMatch
   }
   handler = showMrouteTable
   privileged = True

BasicCli.addShowCommandClass( EventMonitorMrouteCmd )

#--------------------------------------------------------------------------------
# show event-monitor nat [ { ( ( limit LIMIT ) | 
# match-time ( last-minute | last-hour | last-day | last-week ) ) } ]
#--------------------------------------------------------------------------------
def showNatTable( mode, args ):
   if EventMonCli.sfeTableStatus.active:
      EventMonCli.showTable( mode, args, "sfe" )
   else:
      EventMonCli.showTable( mode, args, "nat" ) 

class EventMonitorNatCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show event-monitor nat [ { ( ( limit LIMIT )'
         ' | ( match-time TIME ) ) } ]' )
   data = {
      'event-monitor': nodeEventMonitor,
      'nat': 'Monitor nat table events',
      'limit': matcherLimit,
      'LIMIT': matcherMatchLimit,
      'match-time': matcherMatchTime,
      'TIME': matcherTimeEnum   
   }
   handler = showNatTable
   privileged = True

BasicCli.addShowCommandClass( EventMonitorNatCmd )

#--------------------------------------------------------------------------------
# show event-monitor nhg [ { ( ( limit LIMIT ) |
# match-time ( last-minute | last-hour | last-day | last-week ) ) } ]
#--------------------------------------------------------------------------------
def showNhgTable( mode, args ):
   EventMonCli.showTable( mode, args, "nhg" )

class EventMonitorNhgCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show event-monitor nhg [ { ( ( limit LIMIT )'
         ' | ( match-time TIME ) ) } ]' )
   data = {
      'event-monitor': nodeEventMonitor,
      'nhg': 'Monitor NHG events',
      'limit': matcherLimit,
      'LIMIT': matcherMatchLimit,
      'match-time': matcherMatchTime,
      'TIME': matcherTimeEnum
   }
   handler = showNhgTable
   privileged = True

if Toggle.toggleNhgEventTrackingEnabled():
   BasicCli.addShowCommandClass( EventMonitorNhgCmd )

#--------------------------------------------------------------------------------
# show event-monitor nexthop-group backup-activation [ { ( ( group-by GROUPBY ) | 
# ( limit LIMIT ) | ( match-nexthop-group NHGNAME ) | 
# ( match-nexthop-group-version NHGVERSION ) |
# ( match-time ( last-minute | last-hour | last-day | last-week ) ) ) } ]
#--------------------------------------------------------------------------------
def showBackupactivationTable( mode, args ):
   if Toggle.toggleFecBackupactivationEnabled():
      EventMonCli.showTable( mode, args, "backupactivation" )

class EventMonitorBackupactivationCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show event-monitor nexthop-group backup-activation 
               [ { ( group-by GROUPBY )
                 | ( limit LIMIT ) 
                 | ( match-nexthop-group NHGNAME ) 
                 | ( match-nexthop-group-version NHGVERSION )
                 | ( match-time TIME ) } ]'''
   data = {
      'event-monitor': nodeEventMonitor,
      'nexthop-group': 'Monitor nexthop group events',
      'backup-activation': 'Monitor nexthop group backup activation events',
      'group-by': matcherGroupBy,
      'GROUPBY': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
           'nexthop-group': 'Group by nexthop group name',
           'nexthop-group-version': 'Group by nexthop group version',
         } ), maxMatches=1 ),
      'limit': matcherLimit,
      'LIMIT': matcherMatchLimit,
      'match-nexthop-group': matcherMatchNexthopGroup,
      'NHGNAME': nhgNamePatternMatch,
      'match-nexthop-group-version': matcherMatchNexthopGroupVersion,
      'NHGVERSION': nhgVersionPatternMatch,
      'match-time': matcherMatchTime,
      'TIME': matcherTimeEnum
   }
   handler = showBackupactivationTable
   privileged = True

if Toggle.toggleFecBackupactivationEnabled():
   BasicCli.addShowCommandClass( EventMonitorBackupactivationCmd )

#--------------------------------------------------------------------------------
# show event-monitor status [ TABLE ]
#--------------------------------------------------------------------------------
class EventMonitorShowStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show event-monitor status [ TABLE ]'
   data = {
      'event-monitor': nodeEventMonitor,
      'status': 'Report table metadata',
      'TABLE': allTableMatcher,
   }
   cliModel = 'EventMonStatusModel.EventMonStatus'
   handler = 'EventMonHandler.showEventMonStatus'
   privileged = True

BasicCli.addShowCommandClass( EventMonitorShowStatusCmd )
