# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CliPlugin module for Evpn commands
import BasicCli
import CliCommand
import ShowCommand
import CliParser
import CliMatcher
import ConfigMount
import Ethernet
import LazyMount
import Tac
from Toggles import (
   ArBgpToggleLib,
   BgpCommonToggleLib,
   EvpnToggleLib,
   McastVpnLibToggleLib,
   RoutingLibToggleLib,
)
from CliDynamicSymbol import (
   CliDynamicPlugin,
   LazyCallback
)
from CliMode.Evpn import (
      EthernetSegmentMode,
      EvpnMulticastMode,
      EvpnMulticastAfMode )
import CliToken.Clear
from CliToken.RoutingBgpShowCliTokens import bgpAfterShow
from CliToken import Evpn as EvpnTokens
from CliToken import RoutingBgp as bgpTokens
from CliPlugin import IntfCli
from CliPlugin.BgpVpnCli import (
   RouteImportMatchFailureDiscardCmd,
   tokenRoute,
   tokenVpnImport,
   tokenVpnExport,
   VpnClientIbgpAttributeTunneling,
)
from CliPlugin import BgpVpnDomainIdCli
from CliPlugin import TechSupportCli
from CliPlugin.BgpMacVrfConfigCli import (
   dfKw,
   electionKw,
)
from CliPlugin import ConfigConvert
from CliPlugin import Esid
from CliPlugin.MlagWarningCli import ( canShutdownMlagHook,
      canHitfullyReconfigureMlagHook )
from CliPlugin import RoutingBgpCli
from CliPlugin.EthIntfCli import EthPhyIntf
from CliPlugin.LagIntfCli import EthLagIntf
from CliPlugin.LagIntfCli import LagSubIntf
from CliPlugin.SubIntfCli import SubIntf
from CliPlugin.RoutingBgpCli import (
   RouterBgpBaseAfEvpnMode,
   deleteRouterBgpMacVrfHook,
   deleteRouterBgpAfiSafiHook,
   BgpCmdBaseClass,
)
from CliPlugin.RoutingBgpInstanceCli import (
   NexthopResolutionDisabledCmd,
)
from CliPlugin.RoutingBgpNeighborCli import (
   NeighborReflectorClientOrrPositionCmd
)
from CliPlugin.RoutingBgpShowCli import (
   ArBgpShowOutput,
)
from CliPlugin.VlanCli import vlanSetMatcher
from CliPlugin.VxlanCli import vxlanMcastIpv6OverlaySupportedGuard
from CliPlugin.VxlanShowCommands import nodeVxlan
from CliPlugin import VxlanModel
from IpLibTypes import ProtocolAgentModelType
from TypeFuture import TacLazyType
import Tracing


evpnConfig = None
vtiConfigDir = None
vtiStatusDir = None
bgpMacVrfConfigDir = None
vxlanCtrlConfig = None
vxlanCtrlConfigBox = []
bgpConfig = None
vrfConfigDir = None
evpnOismConfig = None
l3ProtocolAgentModelStatus = None

traceHandle = Tracing.Handle( 'Evpn' )
t0 = traceHandle.trace0
# pkgdeps: rpm Evpn-lib

#-------------------------------------------------------------------------------
# EVPN Ethernet Segment Configuration Commands
# config-if
#   evpn ethernet-segment
#       identifier <10 octet>
#       designated-forwarder election hold-time <N>
#       redundancy [all-active | single-active]
#       route-target import 12:34:45:67:89:01
#       mpls tunnel flood filter time <N>
#-------------------------------------------------------------------------------

U32_MAX_VALUE = 0xFFFFFFFF
ethernetSegmentConfigDir = None
allIntfStatusDir = None
interconnectEsCliConfigDir = None
macDuplicationBlacklist = None
bridgingHwCapabilities = None
electionHoldTimeVal = \
      Tac.Value( 'Evpn::DFElectionHoldTime' )
MacVrfTypeEnum = TacLazyType( "Routing::Bgp::MacVrfType" )
dfPreferenceMatcher = CliMatcher.IntegerMatcher( 0, 65535,
                                                 helpdesc='Preference value' )
ArBgpCliHandler = CliDynamicPlugin( "ArBgpCliHandler" )

# This global variable is used to describe the underlying interface for the WAN
# Interconnect ESI (I-ESI) in EVPN VXLAN DCI MH deployments
WAN_INTERCONNECT_ES_INTF = "Vxlan1"

# 'evpn ethernet-segment' can be configured under L2 interface or
# 'address-family evpn' mode
class EvpnEthernetSegmentModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      # evpn ethernet-segment only makes sense for L2 interfaces. Command should not
      # be available for SVIs, Managment interfaces, peer interfaces...etc.
      allowedTypes = ( EthPhyIntf, EthLagIntf, SubIntf, LagSubIntf )
      return isinstance( mode.intf, allowedTypes )

def evpnEthernetSegmentSupportedGuard( mode, token ):
   # this guard only applies to this command when it's in RouterBgpBaseAfEvpnMode
   if not isinstance( mode, RouterBgpBaseAfEvpnMode ):
      return None

   cmdSupported = bridgingHwCapabilities.evpnDciGatewayMultihomingSupported
   return None if cmdSupported else CliParser.guardNotThisPlatform

def evpnEthernetSegmentDomainSupportedGuard( mode, token ):
   return None if isinstance( mode, RouterBgpBaseAfEvpnMode ) else \
          CliParser.guardNotThisPlatform

def evpnEthernetSegmentSingleActiveSupportedGuard( mode, token ):
   if isinstance( mode.parent_, RouterBgpBaseAfEvpnMode ):
      return CliParser.guardNotThisPlatform
   else:
      return None

def evpnEthernetSegmentDisabledSupportedGuard( mode, token ):
   if isinstance( mode.parent_, RouterBgpBaseAfEvpnMode ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def evpnEthernetSegmentAliasingSupportedGuard( mode, token ):
   if isinstance( mode.parent_, RouterBgpBaseAfEvpnMode ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def evpnEthernetSegmentType1EsiSupportedGuard( mode, token ):
   allowedTypes = ( EthPhyIntf, EthLagIntf, SubIntf, LagSubIntf )
   return ( None if
            ( hasattr( mode.parent_, 'intf' ) and
              isinstance( mode.parent_.intf, allowedTypes ) ) else
            CliParser.guardNotThisPlatform )

class EvpnIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del ethernetSegmentConfigDir.intfConfig[ self.intf_.name ]

class EthernetSegmentConfigMode( EthernetSegmentMode, BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, param ):
      ( self.intf_, domain ) = param
      EthernetSegmentMode.__init__( self, domain )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.config()

   def tacEsDomainType( self, esdType ):
      return Tac.newInstance( 'Evpn::EthernetSegmentDomainType', esdType )

   def config( self ):
      config = None
      if self.domain == "domain local":
         config = interconnectEsCliConfigDir.localDomainIEsConfig
         if config is None:
            intfId = Tac.Value( "Arnet::IntfId", "" )
            interconnectEsCliConfigDir.localDomainIEsConfig = ( intfId, )
            config = interconnectEsCliConfigDir.localDomainIEsConfig
            config.esDomainType = self.tacEsDomainType( 'domain_local' )
      elif self.domain == "domain remote":
         config = interconnectEsCliConfigDir.remoteDomainIEsConfig
         if config is None:
            intfId = Tac.Value( "Arnet::IntfId", "" )
            interconnectEsCliConfigDir.remoteDomainIEsConfig = ( intfId, )
            config = interconnectEsCliConfigDir.remoteDomainIEsConfig
            config.esDomainType = self.tacEsDomainType( 'domain_remote' )
      elif self.domain == "domain all":
         config = interconnectEsCliConfigDir.allDomainIEsConfig
         if config is None:
            intfId = Tac.Value( "Arnet::IntfId", "" )
            interconnectEsCliConfigDir.allDomainIEsConfig = ( intfId, )
            config = interconnectEsCliConfigDir.allDomainIEsConfig
            config.esDomainType = self.tacEsDomainType( 'domain_all' )
      else:
         config = ethernetSegmentConfigDir.intfConfig.get( self.intf_, None )
         if config is None:
            config = ethernetSegmentConfigDir.newIntfConfig( self.intf_ )
            config.esDomainType = self.tacEsDomainType( 'domain_none' )
      # Set the name of the ethernet segment (default is the interface name).
      # Note that if the command is executed in the RouterBgpBaseAfEvpnMode,
      # the intf name would be '' or 'Vxlan1'.
      if config.esName == '':
         config.esName = self.intf_
      return config

#--------------------------------------------------------------------------------
# [ no | default ] evpn ethernet-segment
#--------------------------------------------------------------------------------
class EvpnEthernetSegmentCmd( CliCommand.CliCommandClass ):
   syntax = 'evpn ethernet-segment'
   noOrDefaultSyntax = 'evpn ethernet-segment ...'
   data = {
      'evpn' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'evpn',
            helpdesc='EVPN Configuration' ),
         guard=evpnEthernetSegmentSupportedGuard ),
      'ethernet-segment' : 'Ethernet segment configuration',
   }
   handler = "EvpnCliHandler.EvpnEthernetSegmentCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.EvpnEthernetSegmentCmd_noOrDefaultHandler"

class EvpnEthernetSegmentDomainCmd( CliCommand.CliCommandClass ):
   syntax = 'evpn ethernet-segment domain all'
   noOrDefaultSyntax = 'evpn ethernet-segment ...'
   data = {
      'evpn' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'evpn',
            helpdesc='EVPN Configuration' ),
         guard=evpnEthernetSegmentSupportedGuard ),
      'ethernet-segment' : 'Ethernet segment configuration',
      'domain' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'domain',
            helpdesc='EVPN WAN interconnect ethernet segment domain' ),
         guard=evpnEthernetSegmentDomainSupportedGuard ),
      'all' : 'Apply config for both remote domain and local domain',
   }
   handler = "EvpnCliHandler.EvpnEthernetSegmentDomainCmd_handler"
   noOrDefaultHandler = \
      "EvpnCliHandler.EvpnEthernetSegmentDomainCmd_noOrDefaultHandler"

class EvpnEthernetSegmentDualEncapDomainCmd( CliCommand.CliCommandClass ):
   syntax = 'evpn ethernet-segment domain ( all | local | remote )'
   noOrDefaultSyntax = 'evpn ethernet-segment [ domain ( all | local | remote ) ]'
   data = {
      'evpn' : CliCommand.guardedKeyword(
            'evpn',
            helpdesc='EVPN Configuration',
         guard=evpnEthernetSegmentSupportedGuard ),
      'ethernet-segment' : 'Ethernet segment configuration',
      'domain' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'domain',
            helpdesc='EVPN WAN interconnect ethernet segment domain' ),
         guard=evpnEthernetSegmentDomainSupportedGuard ),
      'all' : 'Apply config for both remote domain and local domain',
      'local' : 'Apply config for the local domain',
      'remote' : 'Apply config for the remote domain',
   }
   handler = "EvpnCliHandler.EvpnEthernetSegmentDomainCmd_handler"
   noOrDefaultHandler = \
      "EvpnCliHandler.EvpnEthernetSegmentDomainCmd_noOrDefaultHandler"


EvpnEthernetSegmentModelet.addCommandClass( EvpnEthernetSegmentCmd )
IntfCli.IntfConfigMode.addModelet( EvpnEthernetSegmentModelet )
RouterBgpBaseAfEvpnMode.addCommandClass( EvpnEthernetSegmentDualEncapDomainCmd )

#--------------------------------------------------------------------------------
# identifier ( ESI | auto lacp )
#--------------------------------------------------------------------------------
class IdentifierEsiCmd( CliCommand.CliCommandClass ):
   if EvpnToggleLib.toggleEvpnType1EsiEnabled():
      syntax = 'identifier ( ESI | ( auto lacp ) )'
   else:
      syntax = 'identifier ESI'
   noOrDefaultSyntax = 'identifier ...'
   data = {
      'identifier': '10 octet ethernet segment identifier',
      'ESI' : Esid.esiMatcher
   }
   if EvpnToggleLib.toggleEvpnType1EsiEnabled():
      data.update( {
         'auto' : 'Automatically generate ESI value',
         'lacp' : CliCommand.Node(
            matcher=CliMatcher.KeywordMatcher( 'lacp',
               helpdesc='ESI Type 1 value obtained from LACP parameters' ),
            guard=evpnEthernetSegmentType1EsiSupportedGuard ) } )
   handler = "EvpnCliHandler.IdentifierEsiCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.IdentifierEsiCmd_noOrDefaultHandler"

EthernetSegmentConfigMode.addCommandClass( IdentifierEsiCmd )

#--------------------------------------------------------------------------------
# name ETH_SEGMENT_NAME
# --------------------------------------------------------------------------------
class EthSegmentNameCmd( CliCommand.CliCommandClass ):
   syntax = 'name ETH_SEGMENT_NAME'
   noOrDefaultSyntax = 'name ...'
   data = {
      'name' : 'Name of the ethernet segment (default: interface name)',
      'ETH_SEGMENT_NAME' : CliMatcher.PatternMatcher(
         pattern=r".+",
         helpname="WORD",
         helpdesc="Ethernet segment name" ),
   }
   handler = "EvpnCliHandler.EthSegmentNameCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.EthSegmentNameCmd_noOrDefaultHandler"

EthernetSegmentConfigMode.addCommandClass( EthSegmentNameCmd )

# --------------------------------------------------------------------------------
# [ no | default ] redundancy ( all-active | single-active )
#--------------------------------------------------------------------------------
class RedundancyCmd( CliCommand.CliCommandClass ):
   syntax = 'redundancy ( all-active | single-active )'
   noOrDefaultSyntax = 'redundancy ...'
   data = {
      'redundancy': 'Multihoming redundancy mode configuration',
      'all-active': 'All PEs attached to an Ethernet segment are forwarding',
      'single-active' : CliCommand.guardedKeyword( 'single-active',
         helpdesc='Only a single PE attached to an Ethernet segment is '
         'forwarding', guard=evpnEthernetSegmentSingleActiveSupportedGuard ),
   }
   handler = "EvpnCliHandler.RedundancyCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.RedundancyCmd_noOrDefaultHandler"

EthernetSegmentConfigMode.addCommandClass( RedundancyCmd )

#--------------------------------------------------------------------------------
# designated-forwarder election algorithm ( hrw | modulus | preference PREFERENCE )
#   [ dont-preempt ]
#
# Note: dont-preempt is currently only supported for preference mode, so right
# now the option only appears when the preference algorithm is chosen.
#--------------------------------------------------------------------------------
class ElectionAlgorithmCmd( CliCommand.CliCommandClass ):
   syntax = ( 'designated-forwarder election algorithm '
              '( hrw | modulus | ( preference PREFERENCE [ dont-preempt ] ) )' )
   noOrDefaultSyntax = 'designated-forwarder election algorithm ...'
   data = {
      'designated-forwarder' : dfKw,
      'election' : electionKw,
      'algorithm' : 'Algorithm used to elect a designated forwarder',
      'hrw' : 'Selection based on highest random weight',
      'modulus' : 'Default selection based on VLAN ID modulo number of candidates',
      'preference' : 'Selection based on a configured preference value',
      'PREFERENCE' : dfPreferenceMatcher,
      'dont-preempt' : 'Inform other candidates not to preempt this device as DF',
   }
   handler = "EvpnCliHandler.ElectionAlgorithmCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.ElectionAlgorithmCmd_noOrDefaultHandler"

EthernetSegmentConfigMode.addCommandClass( ElectionAlgorithmCmd )

#--------------------------------------------------------------------------------
# designated-forwarder election hold-time HOLD_TIME
#                               [ subsequent-hold-time DELAY_TIME ]
#--------------------------------------------------------------------------------
class HoldTimeCmd( CliCommand.CliCommandClass ):
   syntax = ( 'designated-forwarder election hold-time HOLD_TIME '
              '[ subsequent-hold-time DELAY_TIME ]' )
   noOrDefaultSyntax = 'designated-forwarder election hold-time ...'
   data = {
      'designated-forwarder' : dfKw,
      'election' : electionKw,
      'hold-time': 'Timer waiting to receive updates from other PE',
      'HOLD_TIME': CliMatcher.IntegerMatcher( electionHoldTimeVal.min,
                                              electionHoldTimeVal.max,
                                              helpdesc='Number of seconds' ),
      'subsequent-hold-time' : 'Time to wait before running each election',
      'DELAY_TIME' : CliMatcher.IntegerMatcher( 10, 10000,
                                                helpdesc='Time in milliseconds' ),
   }
   handler = "EvpnCliHandler.HoldTimeCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.HoldTimeCmd_noOrDefaultHandler"

EthernetSegmentConfigMode.addCommandClass( HoldTimeCmd )

#--------------------------------------------------------------------------------
# designated-forwarder election candidate reachability required
#--------------------------------------------------------------------------------
class ReachabilityRequiredCmd( CliCommand.CliCommandClass ):
   syntax = 'designated-forwarder election candidate reachability required'
   noOrDefaultSyntax = syntax
   data = {
      'designated-forwarder' : dfKw,
      'election' : electionKw,
      'candidate' : 'Election candidate',
      'reachability' : ( 'The reachability of BGP nexthop in Type-4 route'
                         ' from election candidate' ),
      'required' : 'Only reachable candidates are considered for election'
   }
   handler = "EvpnCliHandler.ReachabilityRequiredCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.ReachabilityRequiredCmd_noOrDefaultHandler"

EthernetSegmentConfigMode.addCommandClass( ReachabilityRequiredCmd )

#--------------------------------------------------------------------------------
# mpls tunnel flood filter time TIME
#--------------------------------------------------------------------------------
class EncapFloodFilterTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls tunnel flood filter time TIME'
   noOrDefaultSyntax = 'mpls tunnel flood filter time ...'
   data = {
      'mpls' : 'MPLS config',
      'tunnel' : 'MPLS tunnel config',
      'flood' : 'Broadcast traffic',
      'filter' : ( 'Filter MPLS encapsulated broadcast traffic into tunnel when DF'
                   ' election is triggered' ),
      'time' : 'Time after running triggered DF election when filter is removed',
      'TIME' : CliMatcher.IntegerMatcher( 10, 10000,
                                         helpdesc='Time in milliseconds' ),
   }
   handler = "EvpnCliHandler.EncapFloodFilterTimeCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.EncapFloodFilterTimeCmd_noOrDefaultHandler"

EthernetSegmentConfigMode.addCommandClass( EncapFloodFilterTimeCmd )

#--------------------------------------------------------------------------------
# mpls shared index INDEX
#--------------------------------------------------------------------------------
class SharedEsiLabelIndexCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls shared index INDEX'
   noOrDefaultSyntax = 'mpls shared index ...'
   data = {
      'mpls' : 'MPLS config',
      'shared' : 'Shared MPLS label config',
      'index' : 'MPLS label index into l2evpn shared ethernet-segment range',
      'INDEX' : CliMatcher.IntegerMatcher( 1, 1024,
                                           helpdesc='Label index value' )
   }
   handler = "EvpnCliHandler.SharedEsiLabelIndexCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.SharedEsiLabelIndexCmd_noOrDefaultHandler"

EthernetSegmentConfigMode.addCommandClass( SharedEsiLabelIndexCmd )

#--------------------------------------------------------------------------------
# route-target import RT
#--------------------------------------------------------------------------------
class RouteTargetImportCmd( CliCommand.CliCommandClass ):
   syntax = 'route-target import RT'
   noOrDefaultSyntax = 'route-target import ...'
   data = {
      'route-target': 'Route Target',
      'import': 'Route Import',
      'RT': CliMatcher.PatternMatcher( Ethernet.macAddrPattern,
                                       helpname='H.H.H',
                                       helpdesc='Low-order 6 bytes of ES-Import '
                                                'Route Target' ),
   }
   handler = "EvpnCliHandler.RouteTargetImportCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.RouteTargetImportCmd_noOrDefaultHandler"

EthernetSegmentConfigMode.addCommandClass( RouteTargetImportCmd )

# --------------------------------------------------------------------------------
# Ethernet Segment disabled
# --------------------------------------------------------------------------------
class EsDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
      'disabled' : CliCommand.guardedKeyword( 'disabled',
         helpdesc='Set default admin state of ethernet segment to disabled',
         guard=evpnEthernetSegmentDisabledSupportedGuard ),
   }
   handler = "EvpnCliHandler.EsDisabledCmd_Handler"
   noOrDefaultHandler = "EvpnCliHandler.EsDisabledCmd_noOrDefaultHandler"

if EvpnToggleLib.toggleEvpnInterconnectEsDisabledEnabled():
   EthernetSegmentConfigMode.addCommandClass( EsDisabledCmd )

# --------------------------------------------------------------------------------
# Ethernet Segment aliasing
# --------------------------------------------------------------------------------
class EsAliasingCmd( CliCommand.CliCommandClass ):
   syntax = 'aliasing'
   noOrDefaultSyntax = syntax
   data = {
      'aliasing' : CliCommand.guardedKeyword( 'aliasing',
         helpdesc='Enable aliasing support',
         guard=evpnEthernetSegmentAliasingSupportedGuard ),
   }
   handler = "EvpnCliHandler.EsAliasingCmd_Handler"
   noOrDefaultHandler = "EvpnCliHandler.EsAliasingCmd_noOrDefaultHandler"

if EvpnToggleLib.toggleEvpnDciNoAliasingEnabled():
   EthernetSegmentConfigMode.addCommandClass( EsAliasingCmd )

#-------------------------------------------------------------------------------
# END EVPN Ethernet Segment Configuration Commands
#-------------------------------------------------------------------------------

class EvpnMulticastConfigMode( EvpnMulticastMode, BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, vrfName ):
      self.session_ = session
      EvpnMulticastMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      config = RoutingBgpCli.configForVrf( vrfName )
      if not config.afEvpnOismExplicitlyDisabled:
         config.afEvpnOismEnabled = True
         evpnOismConfig.afEvpnOismEnabledVrf[ vrfName ] = True

class EvpnMulticastAfIpMode( EvpnMulticastAfMode, BasicCli.ConfigModeBase ):
   name = 'Evpn Multicast address family IP configuration'

   def __init__( self, parent, session, addrFamily, vrfName ):
      EvpnMulticastAfMode.__init__( self, addrFamily, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class EvpnMulticastAfIpv6Mode( EvpnMulticastAfMode, BasicCli.ConfigModeBase ):
   name = 'Evpn Multicast address family IPV6 configuration'

   def __init__( self, parent, session, addrFamily, vrfName ):
      EvpnMulticastAfMode.__init__( self, addrFamily, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#--------------------------------------------------------------------------------
# [ no | default ] evpn multicast
#--------------------------------------------------------------------------------
class EvpnMulticastModeCmd( CliCommand.CliCommandClass ):
   syntax = 'evpn multicast'
   noOrDefaultSyntax = 'evpn multicast'
   data = { 'evpn' : CliMatcher.KeywordMatcher( 'evpn', 'EVPN configuration' ),
            'multicast' : CliMatcher.KeywordMatcher(
               'multicast', helpdesc='Configure EVPN multicast' ),
   }
   handler = "EvpnCliHandler.EvpnMulticastModeCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.EvpnMulticastModeCmd_noOrDefaultHandler"

#--------------------------------------------------------------------------------
# '[ no | default ] address-family ( ipv4 | ipv6 )' in evpn-mcast mode
#--------------------------------------------------------------------------------
class EvpnMulitcastAfCommand( BgpCmdBaseClass ):
   _afs = { 'ipv4' : 'IPv4 configuration',
            'ipv6' : 'IPv6 configuration' }
   syntax = 'address-family AF'
   noOrDefaultSyntax = syntax
   data = {
      'address-family' : 'address-family configuration',
      'AF' : CliCommand.Node( matcher=CliMatcher.EnumMatcher( _afs ),
                              guard=vxlanMcastIpv6OverlaySupportedGuard ) }
   handler = "EvpnCliHandler.EvpnMulitcastAfCommand_handler"
   noOrDefaultHandler = "EvpnCliHandler.EvpnMulitcastAfCommand_noOrDefaultHandler"

# --------------------------------------------------------------------------------
# gateway dr election algorithm ( modulus | hrw | preference PREFERENCE )
# --------------------------------------------------------------------------------
class EvpnGatewayDrElectionAlgorithmCmd( CliCommand.CliCommandClass ):

   syntax = ( 'gateway dr election algorithm '
              '( hrw | modulus | ( preference PREFERENCE ) )' )

   noOrDefaultSyntax = 'gateway dr election algorithm ...'

   data = {
      'gateway' : 'Gateway configuration',
      'dr' : 'Designated-router configuration',
      'election' : 'Designated-router election parameters',
      'algorithm' : 'Algorithm used to elect a gateway designated-router',
      'hrw' : 'Default selection based on highest random weight',
      'modulus' : 'Selection based on VLAN ID modulo number of candidates',
      'preference' : 'Selection based on a configured preference value',
      'PREFERENCE' : dfPreferenceMatcher,
   }
   handler = "EvpnCliHandler.EvpnGatewayDrElectionAlgorithmCmd_handler"
   noOrDefaultHandler = \
      "EvpnCliHandler.EvpnGatewayDrElectionAlgorithmCmd_noOrDefaultHandler"

EvpnMulticastConfigMode.addCommandClass( EvpnGatewayDrElectionAlgorithmCmd )

#--------------------------------------------------------------------------------
# transit
#--------------------------------------------------------------------------------
class OismTransitCmd( CliCommand.CliCommandClass ):
   syntax = 'transit'
   noOrDefaultSyntax = syntax
   data = {
      'transit' : 'Enable EVPN domain as a transit '
                  'network for multicast traffic',
   }
   handler = "EvpnCliHandler.OismTransitCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.OismTransitCmd_noOrDefaultHandler"

if McastVpnLibToggleLib.toggleMcastVpnTransitEnabled():
   EvpnMulticastConfigMode.addCommandClass( EvpnMulitcastAfCommand )
   EvpnMulticastAfIpMode.addCommandClass( OismTransitCmd )

#--------------------------------------------------------------------------------
# disabled
#--------------------------------------------------------------------------------
class OismDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
      'disabled' : 'Disable EVPN Multicast'
   }
   handler = "EvpnCliHandler.OismDisabledCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.OismDisabledCmd_noOrDefaultHandler"

EvpnMulticastAfIpMode.addCommandClass( OismDisabledCmd )
EvpnMulticastAfIpv6Mode.addCommandClass( OismDisabledCmd )

def convertEvpnMulticastSyntax( mode ):
   RoutingBgpCli.bgpConfig.evpnOismLegacy = False

ConfigConvert.registerConfigConvertCallback( convertEvpnMulticastSyntax )

RoutingBgpCli.RouterBgpVrfMode.addCommandClass( EvpnMulticastModeCmd )

# -------------------------------------------------------------------------------
# "show bgp evpn summary [ route-type auto-discovery | mac-ip | imet |
#                                     ethernet-segment | ( ip-prefix ipv4 | ipv6 ) |
#                                     smet | spmsi |join-sync | leave-sync ]"
# -------------------------------------------------------------------------------
# Show BGP EVPN peers and optionally filter by route-type.
# When route-type is not specified, summary statistics reflect the number of routes
# received and accepted for all evpn route-types.
# When a route-type is specified, summary statistics reflect the number of routes
# received and accepted for the specified route-type.

@ArBgpShowOutput( 'doShowBgpEvpnSummary', arBgpModeOnly=True )
def doShowBgpEvpnSummary( *args, **kwargs ):
   mode = args[ 0 ]
   argsValue = kwargs.pop( 'args', None )

   nlriTypeValue = argsValue.pop( 'nlriTypeValue', None )
   if nlriTypeValue:
      nlriType = nlriTypeValue[ 'nlriType' ]
      return ArBgpCliHandler.doShowIpBgpSummaryAcrImpl( mode, nlriType=nlriType )
   else:
      return ArBgpCliHandler.doShowIpBgpSummaryAcrImpl( mode,
                                                       nlriAfiSafi='l2vpnEvpn' )

# TODO remove these once we remove the AgentCommandRequest implementation
def esiLabelExtComm( esiLabel ):
   c = ( 0x0601 << 48 )
   if esiLabel[ 'esiLabelRedundancyMode' ] == 'single-active':
      c |= ( 0x0100 << 32 )
   c |= int( esiLabel[ 'esiLabelValue' ] )
   return c

def routerMacExtComm( macStr ):
   mac = Tac.Value( 'Arnet::EthAddr', stringValue=macStr )
   c = ( 0x0603 << 48 ) | ( mac.word0 << 32 ) | ( mac.word1 << 16 ) | mac.word2
   return c

@ArBgpShowOutput( 'doShowBgpEvpn', arBgpModeOnly=True )
def doShowBgpEvpn( mode, peerAddrValue=None, bgpRouteTypeValue=None,
                   commValuesAndExact=None, extCommValuesAndExact=None,
                   largeCommValuesAndExact=None,
                   nlriTypeAndPrefixValues=None, rdValue=None, vniValue=None,
                   nexthopValue=None, esiValue=None, detail=None, internal=None ):
   raise NotImplementedError( "Something went wrong with EvpnCliHelperCli loading" )

# show bgp evpn mac and show bgp evpn arp
@ArBgpShowOutput( 'doShowBgpEvpnL2', arBgpModeOnly=True )
def doShowBgpEvpnL2( mode, peerAddrValue=None, bgpRouteTypeValue=None,
                     commValuesAndExact=None, extCommValuesAndExact=None,
                     largeCommValuesAndExact=None,
                     nlriTypeAndPrefixValues=None, rdValue=None, vniValue=None,
                     nexthopValue=None, esiValue=None, detail=None, internal=None ):
   raise NotImplementedError( "Something went wrong with EvpnCliHelperCli loading" )
   
#------------------------------------------------------------------
# "show bgp evpn instance"
# "show bgp evpn instance vlan VLANID"
# "show bgp evpn instance vlan-aware-bundle BUNDLE"
# "show bgp evpn instance vpws VPWS"
# "show bgp evpn instance sbd <SBD-VRF-NAME>"
#
# Shows the details of corresponding mac vrf.
#------------------------------------------------------------------
class ShowBgpEvpnInstanceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show bgp evpn instance
   [ ( vlan VLANID )
   | ( vlan-aware-bundle BUNDLE )
   | ( vpws VPWS )
   | ( sbd SBD_VRF_NAME ) ]"""

   cliModel = "EvpnCliModels.BgpEvpnInstances"
   data = {
      "bgp" : bgpAfterShow,
      "evpn" : EvpnTokens.evpnShow,
      "instance" : "EVPN instance",
      "vlan" : "Show information for VLAN-based EVPN instance",
      "VLANID" : CliMatcher.IntegerMatcher( 1, 4094,
         helpdesc="Identifier for a Virtual LAN" ),
      "vlan-aware-bundle" : "Show information for VLAN-aware bundle EVPN instance",
      "BUNDLE" : CliMatcher.QuotedStringMatcher(
         helpname="WORD",
         helpdesc="Unique name to identify VLAN-aware bundle" ),
      "vpws" : ( "Show information for EVPN virtual private wire service (VPWS) "
                 "instance" ),
      "VPWS" : CliMatcher.PatternMatcher(
         pattern=r".+",
         helpname="WORD",
         helpdesc="Virtual private wire service (VPWS) instance name" ),
      "sbd" : "Show information for SBD VLAN EVPN instance",
      "SBD_VRF_NAME" : CliMatcher.PatternMatcher(
         pattern=r".+",
         helpname="WORD",
         helpdesc="IP-VRF name" )
   }

   syntax = syntax[ : -1 ] + '| ( vrf ( VRF | all ) ) ]'
   data[ "vrf" ] = "Show information for IP VRF EVPN instance"
   data[ "VRF" ] = CliMatcher.PatternMatcher(
      pattern=r".+",
      helpname="WORD",
      helpdesc="IP-VRF name" )
   data[ "all" ] = "All IP-VRFs"
   handler = "EvpnCliHandler.ShowBgpEvpnInstanceCmd_handler"

BasicCli.addShowCommandClass( ShowBgpEvpnInstanceCmd )

#--------------------------------------------------------------------------
# register show tech-support extended evpn
#--------------------------------------------------------------------------
def _showEvpnGuard():
   def isMultiAgentMode():
      return ( l3ProtocolAgentModelStatus.protocolAgentModel ==
               ProtocolAgentModelType.multiAgent )

   return isMultiAgentMode() and bgpConfig.asNumber != 0

def evpnActivatedForPeer():
   for peer in bgpConfig.neighborConfig.values():
      if peer.afEvpn == 'afActive':
         return True
   return False


TechSupportCli.registerShowTechSupportCmd(
   '2017-11-03 12:06:06',
   cmds=[ 'show bgp evpn',
          'show bgp evpn instance',
          'show bgp evpn detail',
          'show bgp evpn summary' ],
   cmdsGuard=_showEvpnGuard,
   extended='evpn' )

# Register cmd for 'show tech-support summary' cmd
TechSupportCli.registerShowTechSupportCmd(
   '2020-05-14 23:49:42',
   summaryCmds=[ 'show bgp evpn summary' ],
   summaryCmdsGuard=_showEvpnGuard,
)

TechSupportCli.registerShowTechSupportCmd(
   '2023-03-13 13:49:42',
   cmds=[ 'show bgp evpn summary',
          'show bgp evpn host-flap' ],
   cmdsGuard=_showEvpnGuard,
)

TechSupportCli.registerShowTechSupportCmd(
   '2023-06-16 13:02:32',
   cmds=[ 'show bgp evpn sanity' ],
   cmdsGuard=evpnActivatedForPeer,
)

# ------------------------------------------------------------------
#  routing-bgp <asn>
#     address-family evpn
#       "[ no | default ] route import overlay-index gateway"
#
#  Configures handling of EVPN overlay-index routes, ignoring them
#  if disabled or recursively resolving them if enabled
# ------------------------------------------------------------------
class RouteImportOverlayIndexCmd( CliCommand.CliCommandClass ):
   syntax = "route import overlay-index gateway"
   noOrDefaultSyntax = syntax
   data = {
      'route' : tokenRoute,
      'import' : tokenVpnImport,
      'overlay-index' : 'Overlay index for recursive resolution',
      'gateway' : 'Gateway IP'
   }
   handler = "EvpnCliHandler.RouteImportOverlayIndexCmd_handler"
   noOrDefaultHandler = \
      "EvpnCliHandler.RouteImportOverlayIndexCmd_noOrDefaultHandler"

RouterBgpBaseAfEvpnMode.addCommandClass( RouteImportOverlayIndexCmd )

# ------------------------------------------------------------------
#  routing-bgp <asn>
#     address-family evpn
#       "[ no | default ] unknown-mac-route install"
#  To be configured on TORs to indicate they can install UMR
# ------------------------------------------------------------------
class UmrInstallCommand( CliCommand.CliCommandClass ):
   syntax = 'unknown-mac-route install'
   noOrDefaultSyntax = syntax
   data = {
      'unknown-mac-route' : CliMatcher.KeywordMatcher(
         'unknown-mac-route', helpdesc='EVPN Type-2 Unknown MAC route (UMR)' ),
      'install' : CliMatcher.KeywordMatcher(
         'install', helpdesc='Install received Unknown MAC routes' ),
   }
   handler = "EvpnCliHandler.UmrInstallCliCommand_handler"
   noOrDefaultHandler = \
      "EvpnCliHandler.UmrInstallCliCommand_noOrDefaultHandler"

if ArBgpToggleLib.toggleEvpnUmrEnabled():
   RouterBgpBaseAfEvpnMode.addCommandClass( UmrInstallCommand )

# -------------------------------------------------------------------------------
#  routing-bgp <asn>
#     address-family evpn
#       "[ no | default ] route-type mac-ip re-advertise domain local [disabled]"
#  To be configured on Gateways to prevent re-advertising remote domain mac-ip
#  routes into local domain
# -------------------------------------------------------------------------------
class ReAdvertiseMacIpDisabledCommand( CliCommand.CliCommandClass ):
   syntax = 'route-type mac-ip re-advertise domain local [ DISABLED ]'
   noOrDefaultSyntax = 'route-type mac-ip re-advertise domain local ...'
   data = {
      'route-type' : CliMatcher.KeywordMatcher(
         'route-type', helpdesc='NLRI route type' ),
      'mac-ip' : CliMatcher.KeywordMatcher(
         'mac-ip', helpdesc='MAC-IP route (type 2)' ),
      're-advertise' : CliMatcher.KeywordMatcher(
         're-advertise', helpdesc='re-advertise imported routes' ),
      'domain' : CliMatcher.KeywordMatcher(
         'domain', helpdesc='Select scope for multi-domain gateway config' ),
      'local' : CliMatcher.KeywordMatcher(
         'local',
         helpdesc='Apply config to paths being re-advertised into local domain' ),
      'DISABLED' : CliMatcher.KeywordMatcher(
         'disabled',
         helpdesc='Disable re-advertisement of imported paths into local domain' ),
   }
   handler = "EvpnCliHandler.ReAdvertiseMacIpDisabledCommand_handler"
   noOrDefaultHandler = \
      "EvpnCliHandler.ReAdvertiseMacIpDisabledCommand_noOrDefaultHandler"

if ArBgpToggleLib.toggleEvpnUmrEnabled():
   RouterBgpBaseAfEvpnMode.addCommandClass( ReAdvertiseMacIpDisabledCommand )

# -------------------------------------------------------------------------------
# route export ethernet-segment ip mass-withdraw
# -------------------------------------------------------------------------------
class EvpnRouteExportEsIpMassWithdrawCmd( CliCommand.CliCommandClass ):
   syntax = 'route export ethernet-segment ip mass-withdraw'
   noOrDefaultSyntax = syntax

   data = {
     'route' : tokenRoute,
      'export' : tokenVpnExport,
      'ethernet-segment' : 'Ethernet segment configuration',
      'ip' : 'L3 multi-homing configuration',
      'mass-withdraw' : 'Enables the export of IP mass withdrawal routes'
   }
   handler = "EvpnCliHandler.EvpnRouteExportEsIpMassWithdrawCmd_handler"
   noOrDefaultHandler = \
      "EvpnCliHandler.EvpnRouteExportEsIpMassWithdrawCmd_noOrDefaultHandler"

RouterBgpBaseAfEvpnMode.addCommandClass( EvpnRouteExportEsIpMassWithdrawCmd )

# -------------------------------------------------------------------------------
# route import ethernet-segment ip mass-withdraw
# -------------------------------------------------------------------------------
class EvpnRouteImportEsIpMassWithdrawCmd( CliCommand.CliCommandClass ):
   syntax = 'route import ethernet-segment ip mass-withdraw'
   noOrDefaultSyntax = syntax

   data = {
     'route' : tokenRoute,
      'import' : tokenVpnImport,
      'ethernet-segment' : 'Ethernet segment configuration',
      'ip' : 'L3 multi-homing configuration',
      'mass-withdraw' : 'Enables the import of IP mass withdrawal routes'
   }
   handler = "EvpnCliHandler.EvpnRouteImportEsIpMassWithdrawCmd_handler"
   noOrDefaultHandler = \
      "EvpnCliHandler.EvpnRouteImportEsIpMassWithdrawCmd_noOrDefaultHandler"

RouterBgpBaseAfEvpnMode.addCommandClass( EvpnRouteImportEsIpMassWithdrawCmd )

#------------------------------------------------------------------
#  routing-bgp <asn>
#     address-family evpn
#        "[ no | default ] host-flap detection window SECONDS"
#        "[ no | default ] host-flap detection threshold MOVES"
#
#  Configures the parameters used to determine if flapping MAC
#  addresses should be blacklisted according to RFC 7432.
#  If a MAC address flaps more than MOVES times within
#  SECONDS seconds it will be blacklisted.
#------------------------------------------------------------------
class HostFlapDetectionParamsCmd( CliCommand.CliCommandClass ):
   syntax = """host-flap detection
               ( ( window SECONDS [ threshold MOVES ]
                                  [ expiry timeout TIMEOUT seconds ] )
               | ( threshold MOVES [ window SECONDS ]
                                   [ expiry timeout TIMEOUT seconds ] )
               | ( expiry timeout TIMEOUT seconds [ window SECONDS ]
                                                  [ threshold MOVES ] ) )"""
   noOrDefaultSyntax = """host-flap detection ..."""
   data = {
         'host-flap' : 'Host-flap',
         'detection' : 'Detection',
         'window' : 'Time (in seconds) to detect a MAC duplication issue',
         'SECONDS' : CliMatcher.FloatMatcher( 0, U32_MAX_VALUE,
                                              precisionString='%.25g',
                                              helpdesc='Time (in seconds) to detect '
                                                       'a MAC duplication issue' ),
         'threshold' : 'Minimum number of MAC moves that indicate '
                       'a MAC Duplication issue',
         'MOVES' : CliMatcher.IntegerMatcher( 0, U32_MAX_VALUE,
                                              helpdesc='Minimum number of MAC moves '
                                                       'that indicate a MAC '
                                                       'duplication issue' ),
         'expiry' : 'Time to wait before purging a MAC duplication issue',
         'timeout' : 'Time to wait before purging a MAC duplication issue',
         'TIMEOUT' : CliMatcher.FloatMatcher(
            0, U32_MAX_VALUE, precisionString='%.0f',
            helpdesc='Time (in seconds) to purge a MAC duplication issue' ),
         'seconds' : 'seconds',
      }
   handler = "EvpnCliHandler.HostFlapDetectionParamsCmd_handler"
   noHandler = "EvpnCliHandler.HostFlapDetectionParamsCmd_noHandler"
   defaultHandler = "EvpnCliHandler.HostFlapDetectionParamsCmd_defaultHandler"

RouterBgpBaseAfEvpnMode.addCommandClass( HostFlapDetectionParamsCmd )

deleteRouterBgpMacVrfHook.addExtension(
   LazyCallback( "EvpnCliHandler.deleteEvpnConfigHook" ) )
deleteRouterBgpAfiSafiHook.addExtension(
   LazyCallback( "EvpnCliHandler.deleteEvpnConfigHook" ) )
canShutdownMlagHook.addExtension(
   LazyCallback( "EvpnCliHandler.canShutdownMlag" ) )
canHitfullyReconfigureMlagHook.addExtension(
   LazyCallback( "EvpnCliHandler.canReconfigureMlag" ) )

#-----------------------------------------------------------------------------------
# "[no|default] encapsulation vxlan layer-3 set next-hop igp-cost"
# under 'address-family evpn' mode.
# For L3 NLRI it sets IGP cost in the best path selection to underlay next hop
# IGP cost. In the future L2 related config can be accomodated here.
#-----------------------------------------------------------------------------------
class EvpnVxlanL3UseNhIgpCostCmd( BgpCmdBaseClass ):
   syntax = 'encapsulation vxlan layer-3 set next-hop igp-cost'
   noOrDefaultSyntax = syntax
   data = BgpCmdBaseClass._createSyntaxData( {
         'encapsulation' : bgpTokens.encap,
         'vxlan' : bgpTokens.vxlan,
         'layer-3' : EvpnTokens.layer3,
         'set' : bgpTokens.setForBestPathUse,
         'next-hop' : bgpTokens.setNhAttrForBestPath,
         'igp-cost' : bgpTokens.setNhIgpCost
   } )
   handler = "EvpnCliHandler.EvpnVxlanL3UseNhIgpCostCmd_handler"
   noOrDefaultHandler = \
      "EvpnCliHandler.EvpnVxlanL3UseNhIgpCostCmd_noOrDefaultHandler"

RouterBgpBaseAfEvpnMode.addCommandClass( EvpnVxlanL3UseNhIgpCostCmd )

#------------------------------------------------------------------------------
# "[no|default] layer-2 fec in-place update timeout TIMEOUT seconds
# under 'address-family evpn' mode
# Sets the maximum time to wait for incoming AD routes before refreshing macs
# on the segment
#------------------------------------------------------------------------------
class EvpnL2FecInPlaceUpdateTimeoutCmd( BgpCmdBaseClass ):
   syntax = 'layer-2 fec in-place update [ timeout TIMEOUT seconds ]'
   noOrDefaultSyntax = 'layer-2 fec in-place update [ timeout ] ...'
   data = BgpCmdBaseClass._createSyntaxData( {
         'layer-2' : 'Layer 2 information',
         'fec' : bgpTokens.fec,
         'in-place' : bgpTokens.inPlace,
         'update' : bgpTokens.iarUpdate,
         'timeout' : bgpTokens.iarTimeout,
         'TIMEOUT' : CliMatcher.IntegerMatcher( 0, 300,
                        helpdesc='Number of seconds' ),
         'seconds' : 'Seconds'
   } )
   handler = "EvpnCliHandler.EvpnL2FecInPlaceUpdateTimeoutCmd_handler"
   noOrDefaultHandler = \
      "EvpnCliHandler.EvpnL2FecInPlaceUpdateTimeoutCmd_noOrDefaultHandler"

RouterBgpBaseAfEvpnMode.addCommandClass( EvpnL2FecInPlaceUpdateTimeoutCmd )

#------------------------------------------------------------------
# "[ show | clear ] bgp evpn host-flap"
#
# Shows or clears flapping MAC addresses from the blacklist.
#------------------------------------------------------------------
class ShowBgpEvpnHostFlapCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp evpn host-flap'
   data = {
         'bgp': 'BGP information',
         'evpn' : EvpnTokens.evpnShow,
         'host-flap': 'MAC addresses blacklisted due to duplication'
         }
   cliModel = "EvpnCliModels.BgpEvpnHostFlapEntries"
   handler = "EvpnCliHandler.ShowBgpEvpnHostFlapCmd_handler"

class ClearBgpEvpnHostFlapCmd( CliCommand.CliCommandClass ):
   syntax = 'clear bgp evpn host-flap'
   data = {
         'clear': CliToken.Clear.clearKwNode,
         'bgp': 'Bgp',
         'evpn': 'Evpn',
         'host-flap': 'MAC addresses blacklisted due to duplication'
         }
   handler = "EvpnCliHandler.ClearBgpEvpnHostFlapCmd_handler"

BasicCli.addShowCommandClass( ShowBgpEvpnHostFlapCmd )
BasicCli.EnableMode.addCommandClass( ClearBgpEvpnHostFlapCmd )

# VPN domain id configuration under EVPN address-family.
# "domain identifier ASN:LOCAL_ADMIN"
class SetEvpnDomainIdCmd( CliCommand.CliCommandClass ):
   syntax = 'domain identifier DOMAIN_ID'
   noOrDefaultSyntax = 'domain identifier ...'
   data = { 'domain' : BgpVpnDomainIdCli.tokenDomain,
            'identifier' : BgpVpnDomainIdCli.tokenIdentifier,
            'DOMAIN_ID' : BgpVpnDomainIdCli.DomainIdExpression }
   handler = "EvpnCliHandler.SetEvpnDomainIdCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.SetEvpnDomainIdCmd_noOrDefaultHandler"

# EVPN local and remote domain id configuration under EVPN address-family.
# "domain identifier ASN:LOCAL_ADMIN [ remote ]"
class SetEvpnInterDomainIdCmd( CliCommand.CliCommandClass ):
   syntax = 'domain identifier DOMAIN_ID [ remote ]'
   noOrDefaultSyntax = 'domain identifier [ [DOMAIN_ID] ( remote | DOMAIN_ID ) ]'
   data = { 'domain' : BgpVpnDomainIdCli.tokenDomain,
            'identifier' : BgpVpnDomainIdCli.tokenIdentifier,
            'DOMAIN_ID' : BgpVpnDomainIdCli.DomainIdExpression,
            'remote' : BgpVpnDomainIdCli.tokenRemote }
   handler = "EvpnCliHandler.SetEvpnInterDomainIdCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.SetEvpnInterDomainIdCmd_noOrDefaultHandler"

# Local, remote and attach domain ID configuration under EVPN address-family.
# "domain identifier ASN:LOCAL_ADMIN [ remote | attach ]"
class SetEvpnAttachInterDomainIdCmd( CliCommand.CliCommandClass ):
   syntax = 'domain identifier DOMAIN_ID [ remote | attach ]'
   noOrDefaultSyntax = ( 'domain identifier [ [DOMAIN_ID] '
                         '( remote | attach | DOMAIN_ID ) ]' )
   data = { 'domain' : BgpVpnDomainIdCli.tokenDomain,
            'identifier' : BgpVpnDomainIdCli.tokenIdentifier,
            'DOMAIN_ID' : BgpVpnDomainIdCli.DomainIdExpression,
            'attach' : BgpVpnDomainIdCli.tokenAttach,
            'remote' : BgpVpnDomainIdCli.tokenRemote }
   handler = "EvpnCliHandler.SetEvpnAttachInterDomainIdCmd_handler"
   noOrDefaultHandler = \
      "EvpnCliHandler.SetEvpnAttachInterDomainIdCmd_noOrDefaultHandler"

if EvpnToggleLib.toggleEvpnAttachDomainPathEnabled() and \
   EvpnToggleLib.toggleEvpnMacIpDomainPathEnabled():
   RouterBgpBaseAfEvpnMode.addCommandClass( SetEvpnAttachInterDomainIdCmd )
elif EvpnToggleLib.toggleEvpnMacIpDomainPathEnabled():
   RouterBgpBaseAfEvpnMode.addCommandClass( SetEvpnInterDomainIdCmd )
else:
   RouterBgpBaseAfEvpnMode.addCommandClass( SetEvpnDomainIdCmd )

#---------------------------------------------------------------------------------
# "[no|default] route import match-failure action discard"
# in "address-family evpn" mode.
#
# Enables the discarding of EVPN paths that won't be imported into any VRF (AID6625).
#---------------------------------------------------------------------------------

RouterBgpBaseAfEvpnMode.addCommandClass( RouteImportMatchFailureDiscardCmd )

#---------------------------------------------------------------------------------
# "[no|default] route type smet advertised-by ( designated-forwarder | all )"
# in "address-family evpn" mode.
#
# When "designated-forwarder" option is chosen, only the DF will advertise SMETs in
# a multi-homed scenario.  This is the default behavior.  When "all" is chosen, all
# PEs in a multi-homed scenario will advertise SMETs.
#---------------------------------------------------------------------------------

class RouteTypeSmetAdvertiseCmd( CliCommand.CliCommandClass ):
   syntax = 'route type smet advertised-by ( designated-forwarder | all )'
   noOrDefaultSyntax = 'route type smet advertised-by ...'
   data = {
         'route' : EvpnTokens.route,
         'type' : EvpnTokens.routeType,
         'smet' : EvpnTokens.smet,
         'advertised-by' : 'Advertised by',
         'designated-forwarder' : dfKw,
         'all' : 'All PEs'
         }
   handler = "EvpnCliHandler.RouteTypeSmetAdvertiseCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.RouteTypeSmetAdvertiseCmd_noOrDefaultHandler"

RouterBgpBaseAfEvpnMode.addCommandClass( RouteTypeSmetAdvertiseCmd )

class RouteTypeEsImportAutoRouteTargetCmd( CliCommand.CliCommandClass ):
   syntax = 'route type ethernet-segment route-target auto'
   noOrDefaultSyntax = 'route type ethernet-segment route-target auto'
   data = {
      'route' : EvpnTokens.route,
      'type' : EvpnTokens.routeType,
      'ethernet-segment' : EvpnTokens.es,
      'route-target' : EvpnTokens.rt,
      'auto' : EvpnTokens.auto
      }
   handler = "EvpnCliHandler.RouteTypeEsImportAutoRouteTargetCmd_handler"
   noOrDefaultHandler = \
      "EvpnCliHandler.RouteTypeEsImportAutoRouteTargetCmd_noOrDefaultHandler"

if EvpnToggleLib.toggleEvpnType1EsiEnabled():
   RouterBgpBaseAfEvpnMode.addCommandClass( RouteTypeEsImportAutoRouteTargetCmd )

# ---------------------------------------------------------------------------------
# "[no|default] route type smet proxy inter-domain"
# in "address-family evpn" mode.
#
# When "proxy inter-domain" option is chosen, the IES DF will proxy SMETs on behalf
# of a domain to the other domain.
# AID/12426
# ---------------------------------------------------------------------------------
class RouteTypeSmetProxyInterDomainCmd( CliCommand.CliCommandClass ):
   syntax = 'route type smet proxy inter-domain'
   noOrDefaultSyntax = syntax
   data = {
         'route' : EvpnTokens.route,
         'type' : EvpnTokens.routeType,
         'smet' : EvpnTokens.smet,
         'proxy' : 'Proxy SMETs',
         'inter-domain' : 'Agregate state and proxy SMETs aross domain',
         }
   handler = "EvpnCliHandler.RouteTypeSmetProxyCmd_handler"
   noOrDefaultHandler = "EvpnCliHandler.RouteTypeSmetProxyCmd_noOrDefaultHandler"


if McastVpnLibToggleLib.toggleOismGatewayEnabled():
   RouterBgpBaseAfEvpnMode.addCommandClass( RouteTypeSmetProxyInterDomainCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor default next-hop-self received-evpn-routes route-type
# TYPE [inter-domain]" command, in "address-family evpn" mode
#-------------------------------------------------------------------------------
class SetEvpnNeighborDefaultNextHopSelfCmd( BgpCmdBaseClass ):
   syntax = ( 'neighbor default next-hop-self received-evpn-routes '
              'route-type ROUTE_TYPES [ inter-domain ]' )
   noOrDefaultSyntax = syntax
   data = {
         'neighbor' : bgpTokens.neighbor,
         'default' : bgpTokens.default,
         'next-hop-self' : bgpTokens.nextHopSelf,
         'received-evpn-routes' : bgpTokens.receivedEvpnRoutes,
         'route-type' : bgpTokens.evpnRouteTypeNhSelf,
         'ROUTE_TYPES' : bgpTokens.EvpnRouteTypeExpression,
         'inter-domain' : bgpTokens.interDomain
   }
   handler = "EvpnCliHandler.SetEvpnNeighborDefaultNextHopSelfCmd_handler"
   noOrDefaultHandler = \
      "EvpnCliHandler.SetEvpnNeighborDefaultNextHopSelfCmd_noOrDefaultHandler"

RouterBgpBaseAfEvpnMode.addCommandClass( SetEvpnNeighborDefaultNextHopSelfCmd )

# -------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR route-reflector-client \
#     optimal-route-reflection position peer-address | POSITION-NAME" \
#     route-type ip-prefix
# command, in "address-family evpn" mode.
# -------------------------------------------------------------------------------
class EvpnNeighborReflectorClientOrrPositionCmd(
      NeighborReflectorClientOrrPositionCmd ):
   syntax = NeighborReflectorClientOrrPositionCmd.syntax + ' route-type ip-prefix'
   data = NeighborReflectorClientOrrPositionCmd.data.copy()
   data.update( { 'route-type' : 'Enable ORR for route type',
                  'ip-prefix' : 'Enable ORR for IP prefix routes (type 5)' } )

if RoutingLibToggleLib.toggleBgpOrrEvpnType5Enabled():
   RouterBgpBaseAfEvpnMode.addCommandClass(
      EvpnNeighborReflectorClientOrrPositionCmd )

# ---------------------------------------------------------------------------------
# "[no | default] vpn client ibgp attribute tunneling"
# command, in "address-family evpn" mode.
# Enables or disables the EVPN version of RFC 6368 behavior.
# ---------------------------------------------------------------------------------
if BgpCommonToggleLib.toggleEvpnIBgpAsPeCeProtocolEnabled():
   RouterBgpBaseAfEvpnMode.addCommandClass( VpnClientIbgpAttributeTunneling )

# NOTE: This more properly belongs in the Vxlan package, but because it needs to
# mount both VCS and Evpn paths to source its data, and Vxlan can't depend
# on ArBgp because of dependency cycles, putting it here instead
#-------------------------------------------------------------------------------
# show vxlan control-plane [ import | export ] [ vlan <multi-range-rule> ]
#-------------------------------------------------------------------------------
class ShowVxlanControlPlaneCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan control-plane [ import | export ] [ vlan VLANSET ]'
   data = {
      'vxlan': nodeVxlan,
      'control-plane': 'See control planes active in specific VLANs',
      'import': 'Restrict output to VLANs importing from a control plane',
      'export': 'Restrict output to VLANs exporting to a control plane',
      'vlan': 'Specify specific VLANs to show control plane information for',
      'VLANSET': vlanSetMatcher
   }
   cliModel = VxlanModel.VxlanControlPlaneConfigModel
   handler = "EvpnCliHandler.ShowVxlanControlPlaneCmd_handler"

BasicCli.addShowCommandClass( ShowVxlanControlPlaneCmd )

# ---------------------------------------------------------------------------------
# "[no|default] next-hop resolution disabled" in "router-bgp-af" mode for EVPN
#---------------------------------------------------------------------------------
RouterBgpBaseAfEvpnMode.addCommandClass( NexthopResolutionDisabledCmd )

def Plugin( entityManager ):
   global evpnConfig
   global ethernetSegmentConfigDir
   global allIntfStatusDir
   global interconnectEsCliConfigDir
   global vxlanCtrlConfig
   global macDuplicationBlacklist
   global vtiConfigDir
   global vtiStatusDir
   global bgpMacVrfConfigDir
   global bridgingHwCapabilities
   global bgpConfig
   global vrfConfigDir
   global evpnOismConfig
   global l3ProtocolAgentModelStatus

   evpnConfig = ConfigMount.mount(
      entityManager, 'routing/bgp/evpnconfig',
      'Routing::Bgp::MacPlugin::EvpnConfig', 'w' )

   ethernetSegmentConfigDir = ConfigMount.mount(
         entityManager, 'evpn/interface/config',
         'Evpn::EthernetSegmentCliConfigDir', 'w' )
   allIntfStatusDir = LazyMount.mount( entityManager,
         'interface/status/all', 'Interface::AllIntfStatusDir', 'r' )
   interconnectEsCliConfigDir = ConfigMount.mount(
         entityManager, 'evpn/interface/domainConfig',
         'Evpn::InterconnectEsCliConfigDir', 'w' )
   IntfCli.Intf.registerDependentClass( EvpnIntf )
   vxlanCtrlConfig = LazyMount.mount( entityManager,
                                      "vxlancontroller/config",
                                      "VxlanController::Config", "r" )

   # Boxing vxlanCtrlConfig since VniMatcher needs it at mod-load
   vxlanCtrlConfigBox.append( vxlanCtrlConfig )
   macDuplicationBlacklist = LazyMount.mount(
         entityManager, 'routing/bgp/macdupblacklist',
         'Routing::Bgp::MacPlugin::MacDuplicationBlacklist', 'r' )

   # Need read access to vtiConfigDir and vtiStatusDir for the implementation of
   # the vxlan show control-plane command and Evpn Sanity
   vtiConfigDir = LazyMount.mount( entityManager,
         "interface/config/eth/vxlan", "Vxlan::VtiConfigDir", "r" )
   vtiStatusDir = LazyMount.mount( entityManager,
         "interface/status/eth/vxlan", "Vxlan::VtiStatusDir", "r" )
   # Need read access to macVrfConfigDir for the implementation of
   # the vxlan show control-plane command
   bgpMacVrfConfigDir = LazyMount.mount( entityManager,
         "routing/bgp/macvrf", "Routing::Bgp::MacVrfConfigDir", "r" )

   # Need read access to hardware capabilities for CLI guards
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )

   # Need read access to BgpConfig for Evpn Sanity for Send community check
   bgpConfig = LazyMount.mount( entityManager, "routing/bgp/config",
                                "Routing::Bgp::Config", "r" )

   # Need read access to vrfConfigDir for Evpn sanity ip-vrf checks
   vrfConfigDir = LazyMount.mount( entityManager, 'routing/bgp/vrf/config',
                                   'Routing::Bgp::VrfConfigDir', 'r' )

   # Need write access to evpnOismConfig
   evpnOismConfig = LazyMount.mount( entityManager,
                                     'routing/bgp/evpnOismConfig',
                                     'Routing::Bgp::EvpnOismConfig', 'w' )

   # Need access to l3ProtocolAgentModelStatus for multi-agent check
   l3ProtocolAgentModelStatus = LazyMount.mount( entityManager,
      'l3/status/protocolAgentModelStatus',
      'L3::ProtocolAgentModelStatus', 'r' )
