# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Int, Bool, Dict, Enum
from ArnetModel import IpGenericAddress
import TableOutput
from TypeFuture import TacLazyType

class EvpnGatewayVrf( Model ):
   vlan = Int( help="Internal VLAN ID for this VRF" )

class EvpnGatewayVrfs( Model ):
   vrfs = Dict( help="Mapping of VRFs to their VLAN IDs", keyType=str,
                valueType=EvpnGatewayVrf )
   _detail = Bool( help="Detailed PIM gateway info", optional=True )

   def detailIs( self, detail ):
      self._detail = detail

   def render( self ):
      if self.vrfs:
         if self._detail:
            headers = ( "VRF Name", "VLAN" )
            formatLeft = TableOutput.Format( justify="left" )
            formatLeft.noPadLeftIs( True )
            table = TableOutput.createTable( headers )
            table.formatColumns( formatLeft, formatLeft )
            for vrf, vlan in sorted( self.vrfs.items() ):
               table.newRow( vrf, vlan.vlan )
            print( table.output() )
         else:
            print( "VRF Name" )
            print( "--------------" )
            for vrf in sorted( self.vrfs ):
               print( vrf )

# The input algorithm can be the TACC Evpn::DFAlgorithm enum or the UI algorithm
# string
def drAlgorithmForModel( algorithm ):
   if algorithm == 'dfaDefault' or algorithm == 'modulus':
      return 'modulus'
   elif algorithm == 'dfaHrw' or algorithm == 'HRW':
      return 'hrw'
   elif algorithm == 'dfaPreference' or algorithm == 'preference':
      return 'preference'
   else:
      return 'invalid'

# The input algorithm can be the TACC Evpn::DFAlgorithm enum or the CAPI
# EvpnGatewayDr::algorithm enum
def drAlgorithmForUI( algorithm ):
   if algorithm == 'dfaDefault' or algorithm == 'modulus':
      return 'modulus'
   elif algorithm == 'dfaHrw' or algorithm == 'hrw':
      return 'HRW'
   elif algorithm == 'dfaPreference' or algorithm == 'preference':
      return 'preference'
   else:
      return 'invalid'

class EvpnGatewayDr( Model ):
   role = Enum( values=TacLazyType( "Routing::Multicast::PegDrRole" ).attributes,
                help="Router role on this VLAN" )
   addr = IpGenericAddress( help="Address of Designated Router" )
   algorithm = Enum( values=( 'modulus', 'hrw', 'preference' ),
                     help='Algorithm used to elect DR', optional=True )

class EvpnGatewayDrs( Model ):
   vlans = Dict( help="Mapping of VLAN to external gateway DR information",
                 keyType=int,
                 valueType=EvpnGatewayDr )

   def render( self ):
      if self.vlans:
         headers = ( "VLAN", "Role", "DR Address", "Algorithm" )
         formatLeft = TableOutput.Format( justify="left" )
         formatLeft.noPadLeftIs( True )
         table = TableOutput.createTable( headers )
         table.formatColumns( formatLeft, formatLeft, formatLeft, formatLeft )
         for vlan, dr in sorted( self.vlans.items() ):
            role = 'DR' if dr.role == \
                  TacLazyType( "Routing::Multicast::PegDrRole" ).dr else 'Non-DR'
            algorithm = drAlgorithmForUI( dr.algorithm )
            table.newRow( vlan, role, dr.addr, algorithm )
         print( table.output() )
