#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliGlobal
import CliParser
import LazyMount
from CliPlugin import (
   IntfCli,
   LagCliLib,
   LagIntfCli,
)
from CliPlugin.IntfStatusMatcher import DefaultIntfMatcher
from Intf import IntfRange
from Toggles.SwagCoreToggleLib import toggleSwagPhase1Enabled
if toggleSwagPhase1Enabled():
   from IntfRangePlugin.FabricChannelIntf import FabricChannelAutoIntfType

gv = CliGlobal.CliGlobal( ethLagIntfStatusDir=None )
FabricChannelIntfId = LagCliLib.FabricChannelIntfId
fabricChannelIntfMatcher = DefaultIntfMatcher(
   'Fabric-Channel',
   LagCliLib.FabricChannelIntfNumberMatcher,
   helpdesc="Fabric-channel interface",
   value=lambda mode, intf:
   FabricChannelIntf( intf, mode ),
   guard=LagCliLib.fabricChannelGuard )

#-------------------------------------------------------------------------------
# A subclass for fabric-channel interfaces
#-------------------------------------------------------------------------------
class FabricChannelIntf( LagIntfCli.EthLagIntf ):
   #----------------------------------------------------------------------------
   # The rule for matching Lag interface names.  When this pattern matches, it
   # returns an instance of the EthLagIntf class.
   #
   # This rule gets added to the Intf.rule when this class is registered with
   # the Intf class by calling Intf.addPhysicalIntfType, below.
   #----------------------------------------------------------------------------
   matcher = fabricChannelIntfMatcher

   def createShortname( self ):
      return f'Fc{self.lagId}'

   def getLagId( self ):
      memberId = FabricChannelIntfId.memberId( self.name )
      channelId = FabricChannelIntfId.channelId( self.name )
      return f"{memberId}/{channelId}"

   @staticmethod
   def getAllPhysical( mode ):
      intfs = []
      for name in gv.ethLagIntfStatusDir.intfStatus:
         if not FabricChannelIntfId.isFabricChannelIntfId( name ):
            continue
         intf = FabricChannelIntf( name, mode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

   def hardware( self ):
      return "fabricChannel"

   def switchportEligible( self ):
      return False

   def countersSupported( self ):
      return False

   def routingSupported( self ):
      return False

   def getIntfCounterDir( self ):
      return None

class FabricChannelIntfConfigModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, FabricChannelIntf )

#-------------------------------------------------------------------------------
# Register the EthLagIntf class as a type of physical interface.
#-------------------------------------------------------------------------------
if toggleSwagPhase1Enabled():
   IntfCli.Intf.addPhysicalIntfType( FabricChannelIntf, FabricChannelAutoIntfType )
   IntfRange.registerIntfTypeGuard( FabricChannelAutoIntfType,
                                    LagCliLib.fabricChannelGuard )
   IntfCli.IntfConfigMode.addModelet( FabricChannelIntfConfigModelet )

def Plugin( entityManager ):
   gv.ethLagIntfStatusDir = LazyMount.mount(
      entityManager, "interface/status/eth/lag",
      "Interface::EthLagIntfStatusDir", "r" )
