# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines the config-if-fabric CLI mode and all commands that blong to
it.

As per AID 8164 the config-if-fabric mode is designed for fabric configuration
commands that span the entirety of fabric ( not just specific interfaces or interface
ranges ). Therefore, its not really an "interface" configuration mode. However, due
to legacy reasons we have to stick with it being under `config-if-fabric`.

The config-if-fabric is only enabled if the PD layer has set the
`configIfModeEnabled` capability in the fabric system capabilities.
'''

from __future__ import absolute_import, division, print_function

import BasicCli
import CliParserCommon
import CliCommand
import CliGlobal
import CliMode.Fabric
import CliParser
import LazyMount

# The following is needed to introduce a dependency between the cli and the main rpm.
# pkgdeps: rpm FabricIntf-lib

gv = CliGlobal.CliGlobal( dict( fabricCliCapabilities=None ) )

def isConfigurableFabricSystem( mode, token ):
   if gv.fabricCliCapabilities.globalQosConfigIfModeSupported:
      return None
   return CliParser.guardNotThisPlatform

# The guarded "fabric" keyword needs to be set to normal priority,
# since it overlaps with the intf range inside "show interfaces fabric ...".
# Otherwise, we never match on the fabric interface range.
ifConfigFabricKw = CliCommand.guardedKeyword( 'Fabric',
                                              'Fabric interface',
                                               isConfigurableFabricSystem,
                                               priority=CliParserCommon.PRIO_NORMAL )

#--------------------------------------------------------------------------------
# [ no | default ] interface Fabric ...
#--------------------------------------------------------------------------------
class ConfigFabricModeCmd( CliCommand.CliCommandClass ):
   syntax = 'interface Fabric'
   noOrDefaultSyntax = syntax
   data = {
      'interface' : 'Interface Configuration',
      'Fabric' : ifConfigFabricKw,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( CliMode.Fabric.FabricMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pass

BasicCli.GlobalConfigMode.addCommandClass( ConfigFabricModeCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( em ):
   gv.fabricCliCapabilities = LazyMount.mount(
      em,
      'hardware/fabric/capabilities/cli',
      'Fabric::CliCapabilities',
      'r' )
