# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import TableOutput

from CliModel import Dict, Int, Model, Str

class InterfaceFabricConnectivityInfo( Model ):
   peerDeviceName = Str( help="Name of the peer device" )
   peerDeviceId = Int( help="ID for the peer device" )
   peerInterface = Str( help="Peer interface name" )

class DeviceFabricConnectivityInfo( Model ):
   interfaces = Dict( keyType=str, valueType=InterfaceFabricConnectivityInfo,
                      help="Maps interface name to its peer connectivity info" )
   deviceId = Int( help="ID for host device" )

class ShowFabricPeers ( Model ):
   devices = Dict( keyType=str, valueType=DeviceFabricConnectivityInfo,
                   help="Maps device name to its fabric connectivity info" )

   def render( self ):
      headers = ( "Device",
                  ( "Device", ( "Sys/Mem ID", ) ),
                  "Interface",
                  "Peer Device",
                  ( "Peer Device", ( "Sys/Mem ID", ) ),
                  "Peer Interface" )
      table = TableOutput.createTable( headers )
      formatIntfColumn = TableOutput.Format( justify="left", minWidth=5 )
      formatDeviceColumn = TableOutput.Format( justify="left", minWidth=5 )
      formatDeviceIdColumn = TableOutput.Format( justify="right", minWidth=5 )

      # Remove additional column paddings
      formatIntfColumn.padLimitIs( True )
      formatDeviceColumn.padLimitIs( True )
      formatDeviceIdColumn.padLimitIs( True )

      table.formatColumns( formatDeviceColumn,
                           formatDeviceIdColumn,
                           formatIntfColumn,
                           formatDeviceColumn,
                           formatDeviceIdColumn,
                           formatIntfColumn )

      sortedDevices = sorted( self.devices )
      for device in sortedDevices:
         deviceInfo = self.devices[ device ]
         sortedIntfs = Arnet.sortIntf( deviceInfo.interfaces )
         for intfId in sortedIntfs:
            intfInfo = deviceInfo.interfaces[ intfId ]
            peerDeviceId = intfInfo.peerDeviceId if intfInfo.peerDeviceName else ""
            table.newRow( device,
                          deviceInfo.deviceId,
                          intfId,
                          intfInfo.peerDeviceName,
                          peerDeviceId,
                          intfInfo.peerInterface )
      print( table.output() )
