#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os

import BasicCli
import CliCommand
import CliParser
import LazyMount
import ShowCommand
import Tac

from CliPlugin.IntfCli import Intf, interfacesShowKw
from CliPlugin.FabricIntfModels import ( ShowFabricPeers,
                                         InterfaceFabricConnectivityInfo,
                                         DeviceFabricConnectivityInfo )
from TypeFuture import TacLazyType

dsfHwCapabilities = None
fabricConnectivityInfoBaseDir = None
fciAggrSm = None
fciAggrDir = None

ArexSmBehavior = TacLazyType( "Arx::ArexSmBehavior" )

def fabricKeywordGuard( mode, token ):
   # TODO: This should also include checks for SWAG capable devices.
   if ( ( dsfHwCapabilities.valid and dsfHwCapabilities.capable ) or
        os.environ.get( 'SIMULATION_PEERS' ) ):
      return None
   return CliParser.guardNotThisPlatform

fabricKw = CliCommand.guardedKeyword( 'fabric',
                                      helpdesc='Fabric interface information',
                                      guard=fabricKeywordGuard )

class ShowFabricPeerCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show fabric peers [ interfaces INTFS ]'
   data = {
      'fabric': fabricKw,
      'peers': 'Fabric peer connctivity information',
      'interfaces': interfacesShowKw,
      'INTFS': Intf.rangeMatcher,
   }

   cliModel = ShowFabricPeers

   @staticmethod
   def handler( mode, args ):
      global fciAggrSm
      global fciAggrDir

      fabricInfoDict = {}
      intfs = set( args.get( 'INTFS', [] ) )
      if not fciAggrSm:
         fciAggrDir = Tac.Type( "Fabric::FabricConnectivityInfoPtrDir" )( "all" )
         fciBaseDirForced = LazyMount.force( fabricConnectivityInfoBaseDir )
         smControl = Tac.newInstance( "Arx::SmControl" )
         # Arex SMs run asyncronously by default, which has caused some assertions
         # in the multi-threaded ConfigAgent that verifies certain tasks are executed
         # on the same thread as the task scheduler. To make the task scheduler
         # happy, the SM should run synchronously.
         smControl.forceBehavior = ArexSmBehavior.arexSmBehaviorForceSynchronous
         fciAggrSm = Tac.newInstance( "Fabric::FabricConnectivityInfoAggrSm",
                                      fciBaseDirForced,
                                      fciAggrDir,
                                      smControl )
      for fci in fciAggrDir.fabricConnectivityInfo.values():
         hostEndpoint = fci.hostEndpoint
         peerEndpoint = fci.peerEndpoint
         if intfs and hostEndpoint.intfId not in intfs:
            continue
         fciModel = InterfaceFabricConnectivityInfo()
         devName = hostEndpoint.deviceName
         deviceInfoModel = fabricInfoDict.get( devName )
         if deviceInfoModel is None:
            deviceInfoModel = DeviceFabricConnectivityInfo()
            fabricInfoDict[ devName ] = deviceInfoModel
         deviceInfoModel.deviceId = hostEndpoint.deviceId
         deviceInfoDict = deviceInfoModel.interfaces
         fciModel.peerDeviceName = peerEndpoint.deviceName
         fciModel.peerDeviceId = peerEndpoint.deviceId
         fciModel.peerInterface = peerEndpoint.intfId
         deviceInfoDict[ hostEndpoint.intfId ] = fciModel
      return ShowFabricPeers( devices=fabricInfoDict )

BasicCli.addShowCommandClass( ShowFabricPeerCmd )

def Plugin( em ):
   global dsfHwCapabilities
   global fabricConnectivityInfoBaseDir

   dsfHwCapabilities = LazyMount.mount( em,
      "dsf/hardware/capabilities", "Dsf::DsfHwCapabilities", "r" )
   fabricConnectivityInfoBaseDir = LazyMount.mount( em,
      "hardware/fabric/connectivity/slice", "Tac::Dir", "ri" )
