# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import re
from socket import AF_INET, AF_INET6

import Arnet
import AgentCommandRequest
import BasicCli
import CliParser
import Ethernet
import FhrpUtils
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin import FruCli
from CliPlugin import IntfCli
from CliPlugin import IraCommonCli
from CliPlugin import IraIp6IntfCli
from CliPlugin import IraIpCli
from CliPlugin import IraIpIntfCli
from CliPlugin import IraVrfCli
from CliPlugin import TechSupportCli
from CliPlugin import VlanIntfCli
from CliPlugin.FhrpModels import TrackedObject
from CliPlugin.FhrpModels import VarpModel
from CliPlugin.FhrpModels import VarpSourceNatV4Model
from CliPlugin.FhrpModels import VarpSourceNatV6Model
from CliPlugin.FhrpModels import VrrpBriefModel
from CliPlugin.FhrpModels import VrrpDetailModel
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliPlugin.Ip6AddrMatcher import ip6PrefixMatcher
from CliPlugin.MacAddr import MacAddrMatcher
from CliPlugin.VrfCli import ALL_VRF_NAME, DEFAULT_VRF, VrfExprFactory, vrfExists
import IpUtils
import LazyMount
import ConfigMount
import Tac
import Tracing
import FhrpAgent
import CliExtensions
import CliCommand
import ShowCommand
import functools

import CliMatcher
from CliMatcher import StringMatcher
from CliMatcher import KeywordMatcher


from BasicCliModes import GlobalConfigMode
from CliToken.Ip import ipMatcherForConfig
from CliToken.Ip import ipMatcherForConfigIf
from CliToken.Ip import ipMatcherForShow
from CliToken.Ipv6 import ipv6MatcherForConfig
from CliToken.Ipv6 import ipv6MatcherForConfigIf
from CliToken.Ipv6 import ipv6MatcherForShow
from CliToken.Mac import macMatcherForConfigIf
from CommonGuards import standbyGuard
from IpLibConsts import MAX_SECONDARIES_PER_INTERFACE
import ReversibleSecretCli

# Cli models
from ArnetModel import IpAddrAndMask
from ArnetModel import Ip6AddrAndMask

import Toggles.FhrpToggleLib

traceHandle = Tracing.Handle( 'Fhrp' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2

# Common matchers
matcherAdvertisementInterval = CliMatcher.KeywordMatcher( 'advertisement-interval',
      helpdesc='Configure advertisement interval in seconds' )
matcherAdvertisement = CliMatcher.KeywordMatcher( 'advertisement',
      helpdesc='Configure VRRP advertisement' )
matcherMacAddress = CliMatcher.KeywordMatcher( 'mac-address',
      helpdesc='Virtual router MAC address' )
matcherSourceNat = CliMatcher.KeywordMatcher( 'source-nat',
      helpdesc='Configure source NAT for virtual IP' )
matcherVirtual = CliMatcher.KeywordMatcher( 'virtual',
      helpdesc='Set virtual IP address of an interface' )
matcherVirtualRouter = CliMatcher.KeywordMatcher( 'virtual-router',
      helpdesc='Configure a virtual router' )
matcherAll = CliMatcher.KeywordMatcher( 'all',
      helpdesc='Display all VRRP Instances (including offline)' )
matcherAllVrf = CliMatcher.KeywordMatcher( ALL_VRF_NAME,
      helpdesc='All configured VRFs' )
matcherFhrp = CliMatcher.KeywordMatcher( 'fhrp',
      helpdesc='FHRP (VRRP and VARP) status' )
matcherGroup = CliMatcher.KeywordMatcher( 'group',
      helpdesc='Display VRRP Instances with group ID' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Display per-interface VRRP info' )
matcherInternal = CliMatcher.KeywordMatcher( 'internal',
      helpdesc='Show the Fhrp agent internal information' )
matcherVarpForShow = CliMatcher.KeywordMatcher( 'virtual-router',
      helpdesc='Virtual router status' )
matcherShowVrrp = CliMatcher.KeywordMatcher( 'vrrp',
      helpdesc='VRRP information' )
matcherVrrp = CliMatcher.KeywordMatcher( 'vrrp',
      helpdesc='Virtual Router Redundancy Protocol (VRRP)' )
matcherVrId = CliMatcher.IntegerMatcher( 1, 255,
      helpdesc='Virtual router Id' )
matcherIp = CliMatcher.KeywordMatcher( 'ip',
      helpdesc='Enable VRRP for IPv4' )
matcherIpv4 = CliMatcher.KeywordMatcher( 'ipv4',
      helpdesc='Enable VRRP for IPv4' )
matcherIpv6 = CliMatcher.KeywordMatcher( 'ipv6',
      helpdesc='Enable VRRP for IPv6' )
matcherObjName = CliMatcher.PatternMatcher( pattern=r'[a-zA-Z0-9]+',
      helpdesc='String name', helpname='WORD' )
matcherSourceNatAddr = CliMatcher.KeywordMatcher( 'address',
      helpdesc='Source NAT IP address' )
matcherBfd = CliMatcher.KeywordMatcher( 'bfd',
      helpdesc='Enable BFD for the VRRP Router' )
# TODO: Use the matcher from IraIpIntfCli when it gets converted
matcherAddress = CliMatcher.KeywordMatcher( 'address',
      helpdesc='Set IP address of an interface' )
matcherHourInterval = CliMatcher.IntegerMatcher( 0, 3600,
      helpdesc='Interval in units of seconds' )
matcherIpWithMaskExpr = IpAddrMatcher.ipAddrWithMaskExpr(
      'IP address', 'Subnet\'s mask value',
      'IP address with mask length' )

authVersionWarningMessage = 'Authentication is not compatible with version 3 ' \
                            'of VRRP'
advtIntervalWarningMessage = 'VRRP version 3 with advertisement intervals over ' \
                             '40 seconds is not supported and will be treated ' \
                             'as 40 seconds'

MAX_VARP_IPV4_PER_INTERFACE = 500
MAX_VARP_IPV6_PER_INTERFACE = 500
MAX_VRRP_SECONDARY_IPV4_PER_VRID = 254
MAX_VRRP_IPV6_PER_VRID = 255
MAX_VRRP_V3_ADVERTISE_INTERVAL = 40

VARP_VRID = 256

modelet = IraIpIntfCli.RoutingProtocolIntfConfigModelet

fhrpConfigDir = None
varpConfigDir = None
fhrpStatusDirV4 = None
fhrpStatusDirV6 = None
fhrpVirtualMacStatus = None
fhrpHardwareStatus = None
routing6HardwareStatus = None
l3ConfigDir = None
ipConfig = None
ip6Config = None
ipStatus = None
ip6Status = None
hwEntMibStatus = None
mlagConfigDir = None
trackingConfig = None
securityConfig = None

authTypeEnum = Tac.Type( 'Routing::Fhrp::Lib::VrrpAuthType' )
intfEnabledStateEnum = Tac.Type( 'Interface::IntfEnabledState' )
fhrpStateEnum = Tac.Type( 'Routing::Fhrp::FhrpState' )
bfdStateEnum = Tac.Type( "Routing::Fhrp::BfdVrState" )
EthAddrWithMask = Tac.Type( "Arnet::EthAddrWithMask" )
ipAddrZero = Arnet.IpAddress( 0 )
ip6AddrZero = Arnet.Ip6Addr( '::' )

def isSubIntf( intfName ):
   return '.' in intfName

def fhrpSupportedGuard( mode, token ):
   if fhrpHardwareStatus.vrrpSupported or fhrpHardwareStatus.varpSupported:
      return None
   return CliParser.guardNotThisPlatform

def vrrpSupportedGuard( mode, token ):
   if isinstance( mode, IntfCli.IntfConfigMode ):
      if ( not fhrpHardwareStatus.vrrpSubinterfaceSupported and
            isSubIntf( mode.intf.name ) ):
         return CliParser.guardNotThisPlatform
   if fhrpHardwareStatus.vrrpSupported:
      return None
   return CliParser.guardNotThisPlatform

def varpMacMaskSupportedGuard( mode, token ):
   if fhrpHardwareStatus.varpMacMaskSupported:
      return None
   return CliParser.guardNotThisPlatform

def vrrpV6SupportedGuard( mode, token ):
   if vrrpSupportedGuard( mode, token ) is None:
      if routing6HardwareStatus.routingSupported:
         return None
   return CliParser.guardNotThisPlatform

def varpSupportedGuard( mode, token ):
   if fhrpHardwareStatus.varpSupported:
      return None
   return CliParser.guardNotThisPlatform

def varpV6SupportedGuard( mode, token ):
   if varpSupportedGuard( mode, token ) is None:
      if routing6HardwareStatus.routingSupported:
         return None
   return CliParser.guardNotThisPlatform

def varpV6IntfGuard( mode, token ):
   if varpV6SupportedGuard( mode, token ) is not None:
      return varpV6SupportedGuard( mode, token )
   return None

def forceIpv4MacPrefixGuard( mode, token ):
   if vrrpSupportedGuard( mode, token ) is not None:
      return vrrpSupportedGuard( mode, token )
   if fhrpHardwareStatus.ipv6ForceIpv4MacPrefixSupported:
      return None
   return CliParser.guardNotThisPlatform

# Common nodes
nodeVirtualRouter = CliCommand.Node( matcherVirtualRouter,
                                     guard=varpSupportedGuard )
nodeVirtualRouterV6 = CliCommand.Node( matcherVirtualRouter,
                                       guard=varpV6SupportedGuard )
nodeFhrp = CliCommand.Node( matcherFhrp, guard=fhrpSupportedGuard )
nodeInternal = CliCommand.Node( matcherInternal, guard=standbyGuard )
nodeShowVrrp = CliCommand.Node( matcherShowVrrp, guard=vrrpSupportedGuard )
nodeVrrp = CliCommand.Node( matcherVrrp, guard=vrrpSupportedGuard )
nodeVrrpIp = CliCommand.Node( matcherIp, deprecatedByCmd='vrrp ipv4' )
nodeVrrpIpv4 = CliCommand.Node( matcherIpv4 )
nodeVrrpIpv6 = CliCommand.Node( matcherIpv6, guard=vrrpV6SupportedGuard )

def systemMacConflict( mac ):
   mac = Arnet.EthAddr( mac )
   # Skip conflict check for default MAC address. Otherwise this creates trouble
   # with saving and installing default config on non MLAG systems where
   # systemId is the default MAC address.
   if str( mac ) == Tac.Type( "Arnet::EthAddr" ).ethAddrZero:
      return False
   systemMac = Arnet.EthAddr( hwEntMibStatus.systemMacAddr )
   return mac == systemMac

def fhrpStateEnumString( state ):
   if state == fhrpStateEnum.fhrpActive:
      return 'master'
   elif state == fhrpStateEnum.fhrpBackup:
      return 'backup'
   elif state == fhrpStateEnum.fhrpStopped:
      return 'stopped'
   return 'stopped'

def varpStateEnumString( state ):
   if state == fhrpStateEnum.fhrpActive:
      return 'active'
   elif state == fhrpStateEnum.fhrpStopped:
      return 'stopped'
   return 'stopped'

def getNumOfVrrpIntfs():
   if fhrpHardwareStatus.vrrpSupported:
      vcd = _vrrpConfig()
      return len( vcd.vrrpIntfConfig )
   else:
      return 0

IraIpCli.fhrpVrrpIntfsHook.addExtension( getNumOfVrrpIntfs )

def _fhrpConfig():
   return fhrpConfigDir

def _mlagConfig():
   return mlagConfigDir

def _varpConfig():
   return varpConfigDir

def _vrrpConfig():
   fhrp = _fhrpConfig()
   return fhrp.vrrpConfig

def _varpIntfConfig( mode, createIfMissing=True ):
   varp = _varpConfig()
   if varp is None:
      return None

   name = mode.intf.name
   intfConfig = varp.varpIntfConfig.get( name )
   if intfConfig is None and createIfMissing:
      t2( 'Creating VarpIntfConfig:', name )
      intfConfig = varp.varpIntfConfig.newMember( name )
   return intfConfig

def _vrrpIntfConfig( mode, createIfMissing=True ):
   name = mode.intf.name
   vrrp = _vrrpConfig()
   if vrrp is None:
      return None

   intfConfig = vrrp.vrrpIntfConfig.get( name )
   if intfConfig is None and createIfMissing:
      t2( 'Creating VrrpIntfConfig:', name )
      intfConfig = vrrp.vrrpIntfConfig.newMember( name )
   return intfConfig

def _vrConfig( mode, vrId, createIfMissing=True ):
   intf = _vrrpIntfConfig( mode, createIfMissing=createIfMissing )
   if intf is None:
      return None

   vr = intf.virtualRouterConfig
   config = vr.get( vrId )
   if config is None:
      if not createIfMissing:
         return None
      if not _newVirtualRouterIdAllowed( mode, vrId ):
         deleteVrrpIntfConfigIfDefault( mode.intf.name )
         return None
      t2( 'Creating VirtualRouterConfig:', vrId )
      config = vr.newMember( vrId )
   return config

def _newVirtualRouterIdAllowed( mode, vrId ):
   config = _vrrpConfig()

   numCreated = 0
   intfUsingVrid = None
   for intfConfig in config.vrrpIntfConfig.values():
      numCreated += len( intfConfig.virtualRouterConfig )
      if vrId in intfConfig.virtualRouterConfig:
         intfUsingVrid = intfConfig.intfId

   if fhrpHardwareStatus.vrrpSupportedMaxNum and \
      numCreated >= fhrpHardwareStatus.vrrpSupportedMaxNum:
      mode.addError( "More than %s virtual routers are not supported" %
                     ( fhrpHardwareStatus.vrrpSupportedMaxNum, ) )
      return False
   if intfUsingVrid and fhrpHardwareStatus.useUniqueGroupIds:
      mode.addError( "VRRP group %d is already in use on %s" %
                     ( vrId, intfUsingVrid ) )
      return False
   return True

def _fhrpVmacStatus():
   return fhrpVirtualMacStatus

def _fhrpStatusV4():
   return fhrpStatusDirV4

def _fhrpStatusV6():
   return fhrpStatusDirV6

def _allowDuplicate():
   fhrp = _fhrpConfig()
   return fhrp.allowDuplicate

def getVrfName( intfName ):
   # we check the ipIntfConfig for the vrfName because checking the
   # status causes potential issues when a vrf has been configured
   # on an interface but the ipIntfStatus has not yet been updated
   ipIntfConfig = ipConfig.ipIntfConfig.get( intfName )
   if ipIntfConfig:
      return ipIntfConfig.vrf
   return DEFAULT_VRF

def fhrpIntfIdPair( intfId, vrId ):
   return Tac.Value( 'Routing::Fhrp::FhrpIntfIdPair', intfId, vrId )

def vrfIpPair( vrf, ip ):
   return Tac.Value( "VirtualIp::VrfIpPair", vrf,
                     Arnet.IpGenAddr( str( ip ) ) )

def vrfIpWithMaskPair( vrf, ip, maskLen=None ):
   if maskLen:
      ipAddr = '%s/%d' % ( str( ip ), maskLen )
   else:
      ipAddr = str( ip )
   return Tac.Value( "VirtualIp::VrfIpWithMaskPair", vrf,
                     Arnet.IpGenAddrWithMask( ipAddr ) )

def subnetStr( a ):
   return Arnet.IpAddress( Arnet.Subnet( a.address, a.len ).toNum() ).stringValue

def deleteVarpConfigIfDefault( intfName ):
   varpConfig = _varpConfig()
   intfConfig = varpConfig.varpIntfConfig.get( intfName )
   if intfConfig is None:
      return

   if ( intfConfig and
        not intfConfig.virtualIpAddr and
        not intfConfig.virtualIp6Addr and
        not intfConfig.varpTrackedObject and
        not intfConfig.useVirtualMac ):
      t2( 'Deleting VarpIntfConfig:', intfName )
      del varpConfig.varpIntfConfig[ intfName ]

def deleteVrConfigIfDefault( intfName, vrId ):
   vrrpConfig = _vrrpConfig()
   intfConfig = vrrpConfig.vrrpIntfConfig.get( intfName )
   if intfConfig is None:
      return
   vr = intfConfig.virtualRouterConfig.get( vrId )
   if vr is None:
      return

   if ( vr.priority == vr.priorityDefault and
        vr.version == vr.versionDefault and
        vr.advInterval == vr.advIntervalDefault and
        vr.arpOrNAInterval == vr.arpOrNAIntervalDefault and
        vr.preempt == vr.preemptDefault and
        vr.preemptDelay == vr.preemptDelayDefault and
        vr.preemptReloadDelay == vr.preemptReloadDelayDefault and
        vr.reloadDelay == vr.reloadDelayDefault and
        vr.authType == vr.authTypeDefault and
        vr.authKey == vr.authKeyDefault and
        vr.bfdPeerAddrV4 == vr.bfdPeerAddrV4Default and
        vr.bfdPeerAddrV6 == vr.bfdPeerAddrV6Default and
        vr.description == vr.descriptionDefault and
        vr.adminEnabledState == vr.adminEnabledStateDefault and
        not vr.vrrpTrackedObject and
        not vr.ip6Addrs and
        vr.primaryAddr == vr.primaryAddrDefault and
        not vr.secondaryAddrs and
        vr.ipv4ExcludePseudoChecksum == vr.ipv4ExcludePseudoChecksumDefault and
        vr.advMismatchLog == vr.advMismatchLogDefault and
        vr.advMismatchDrop == vr.advMismatchDropDefault ):
      t2( 'Deleting VirtualRouterConfig:', vrId )
      del intfConfig.virtualRouterConfig[ vrId ]
   deleteVrrpIntfConfigIfDefault( intfName )

def deleteVrrpIntfConfigIfDefault( intfName ):
   vrrpConfig = _vrrpConfig()
   intfConfig = vrrpConfig.vrrpIntfConfig.get( intfName )
   if intfConfig and not intfConfig.virtualRouterConfig:
      t2( 'Deleting VrrpIntfConfig:', intfName )
      del vrrpConfig.vrrpIntfConfig[ intfName ]

def isVrrpMac( macAddr ):
   vrrp2Pattern = '00:00:5e:00:01:%s' % Ethernet.pair
   vrrp3Pattern = '00:00:5e:00:02:%s' % Ethernet.pair
   colonAddr = Ethernet.convertMacAddrToCanonical( macAddr )
   if re.match( vrrp2Pattern, colonAddr ) or re.match( vrrp3Pattern, colonAddr ):
      return True
   return False

# This check is needed because in vrrp v2 there are 8 bits for advInterval field
# and it was configured in seconds allowing for up to 255 seconds. With vrrp v3,
# there are 12 bits for advInterval, but is actually in centiseconds which allows
# a max value of 40.95 seconds.
def checkAdvtInterval( mode, config ):
   ip6Count = len( config.ip6Addrs )
   primaryDefault = config.primaryAddr == config.primaryAddrDefault
   vrrpV3 = config.version == 3
   if ( config.advInterval // 100 > MAX_VRRP_V3_ADVERTISE_INTERVAL and
        ( ( ip6Count == 1 and primaryDefault ) or
          ( ip6Count == 0 and vrrpV3 and not primaryDefault ) ) ):
      mode.addWarning( advtIntervalWarningMessage )

def checkAuthAndVrrpVersion( mode, config ):
   if config.version == 3 and config.authType != authTypeEnum.vrrpAuthNone:
      mode.addWarning( authVersionWarningMessage )

# Checks if there are more virtual routers configured than
# fhrpHardwareStatus.vrrpSupportedMaxNum.
def maxVrCountExceeded():
   if not fhrpHardwareStatus.vrrpSupportedMaxNum:
      return False
   vrrpConfig = _vrrpConfig()
   vrCount = 0
   for intfConfig in vrrpConfig.vrrpIntfConfig.values():
      for config in intfConfig.virtualRouterConfig.values():
         if config.primaryAddr != config.primaryAddrDefault:
            vrCount += 1
         if config.ip6Addrs:
            vrCount += 1
   if vrCount >= fhrpHardwareStatus.vrrpSupportedMaxNum:
      return True
   return False

# Checks if VARP is configured.
def varpConfigured():
   varpConfig = _varpConfig()
   return varpConfig.virtualMacWithMask != varpConfig.virtualMacWithMaskDefault

def vrrpIntfConfigured():
   vrrpConfig = _vrrpConfig()
   return bool( vrrpConfig.vrrpIntfConfig )

def varpIntfConfigured():
   varpConfig = _varpConfig()
   return bool( varpConfig.varpIntfConfig )

def mlagPeerMacConfigured():
   mlagConfig = _mlagConfig()
   return mlagConfig.peerMacRoutingEnabled

# Check whether Fhrp agent should be enabled or disabled. This is used to
# set the runnability of Fhrp, such that it can be read by Launcher, which
# can stop/start the Fhrp agent.
def evaluateFhrpRunnability():
   # Fhrp is runnable in the following cases:
   # 1) A non-default virtual MAC address is configured.
   # 2) A vrrp interface config has been created
   # 3) A varp interface config has been created
   # Ip Address Virtual configuration is not used in this check, as it would
   # involve iterating over all interfaces, which will be slow, and since the
   # configuration doesn't take effect, unless a VMAC is configured.
   fhrpConfig = _fhrpConfig()
   fhrpConfig.fhrpConfigured = ( varpConfigured() or vrrpIntfConfigured() or
                               varpIntfConfigured() )

# This function checks the platform limitations in programming the mac address
# corresponding to the virtual router id and returns a warning (if any).
def getVrrpPlatformLimitations( vrId, addrFamily=AF_INET ):
   vrIdVariableBitLimit = fhrpHardwareStatus.vrIdVariableBitLimit
   fullMacLimit = fhrpHardwareStatus.fullMacLimit
   warning = ""
   if vrIdVariableBitLimit:
      # vrIdVariableBitLimit set to non zero in fhrpHardwareStatus (Arad platforms)
      # results in three limitations:
      # 1. VARP and VRRP cannot coexist
      # 2. IPv4 and IPv6 VRRP cannot be configured simulatenously
      # 3. vrId's which differ in more than the most significant bits of the smallest
      #    vrId get disabled.
      if varpConfigured():
         warning = "VARP and VRRP cannot coexist"

      vrrpConfig = _vrrpConfig()

      def getValidVrIdAndIp():
         # Find the  valid virtual router id, the mac address for which has been
         # programmed in the hardware.
         unprogrammedVrIds = fhrpHardwareStatus.vrrpUnprogrammedVrId
         for vrfIp in vrrpConfig.virtualIpAddr:
            vrId = vrrpConfig.virtualIpAddr[ vrfIp ].vrId
            if vrId not in unprogrammedVrIds:
               return ( vrId, vrfIp.ip )
         return None

      vrIdIpPair = getValidVrIdAndIp()
      af = 'ipv6' if addrFamily == AF_INET6 else 'ipv4'
      if vrIdIpPair:
         if not vrrpConfig.ipv6ForceIpv4MacPrefix and vrIdIpPair[ 1 ].af != af:
            warning = "IPv4 and IPv6 VRRP cannot be configured simultaneously"
         elif ( vrId ^ vrIdIpPair[ 0 ] ) > ( 2 ** vrIdVariableBitLimit - 1 ):
            warning = "The virtual router Id can only vary in the last %d bits" \
                     % ( vrIdVariableBitLimit )
      return warning
   elif fullMacLimit:
      # fullMacLimit set to non zero in fhrpHardwareStatus removes some of the
      # limitations from above (vridVariableBitLimit):
      # 1. VARP and VRRP can coexist
      # 2. IPv4 and IPv6 VRRP can be configured simultaneously
      # 3. Up to 16 different MACs can be programmed, with no limitation on range

      # Add address family in along with vrId
      vrrpConfig = _vrrpConfig()
      # Use dict for speed
      uniqueMacs = {}
      for vrfIp, virtualIpAddr in vrrpConfig.virtualIpAddr.items():
         otherVrId = virtualIpAddr.vrId
         # if we are forcing ipv4 mac prefixs, address family can be disregarded.
         otherAddrFamily = ( AF_INET if vrrpConfig.ipv6ForceIpv4MacPrefix
                             else vrfIp.ip.af )
         vrIdAddrPair = ( otherAddrFamily, otherVrId )
         uniqueMacs[ vrIdAddrPair ] = True
      # convert back to list
      uniqueMacs = list( uniqueMacs )
      # sort by address family, and then by vrid
      uniqueMacs = sorted( uniqueMacs )

      numUniqueMacs = len( uniqueMacs )
      programmedVrrpMacCount = fullMacLimit
      # Sorting by address family (ipv4/ipv6) in ascending order puts all ipv4
      # addresses before ipv6 addresses. This adequately represents the
      # priority of the macs to be programmed in hardware. The bottom 16
      # (or 15, or 14, if VARP/MLAG Peer are configured) are the ones programmed
      # in hardware.

      if varpConfigured():
         numUniqueMacs += 1
         programmedVrrpMacCount -= 1
      if mlagPeerMacConfigured():
         numUniqueMacs += 1
         programmedVrrpMacCount -= 1
      # If we've reached the unique MAC limit, and this vrId/AddrFamily pair will
      # not fit (or will kick out a currently existing MAC), print the warning.
      # This condition also works for show vrrp.
      af = ( 'ipv6' if ( addrFamily == AF_INET6 and not
                         vrrpConfig.ipv6ForceIpv4MacPrefix )
             else 'ipv4' )
      if ( numUniqueMacs >= fullMacLimit and
           ( af, vrId ) not in uniqueMacs[ : programmedVrrpMacCount ] ):
         warning = "A maximum of %d unique routable MAC addresses can be " \
                   "configured on this device" % ( fullMacLimit )
      return warning
   else:
      return ""

# The following functions checks the corresponding configurations for already
# assigned ip or ipv6 addresses. If the corresponding interfaces are passed,
# the function will skip the check on those parameters. This is because
# if the same configuration is passed twice, the command should not return
# an error. If vrId is set to VARP_VRID, the function will only check for already
# configured vrrp addresses. This is because varp allows duplicate virtual ips
# on the same vrf.
def uniqueFhrpIpAssigned( vrfIp, intfName=None, vrId=None ):
   ipGenAddr = vrfIp.ip
   if ipGenAddr.isAddrZero:
      return True, None

   # Check for matching VRRP IPs
   vrrpConfig = _vrrpConfig()
   fhrpIntf = vrrpConfig.virtualIpAddr.get( vrfIp )
   if fhrpIntf:
      if fhrpIntf.intfId != intfName or fhrpIntf.vrId != vrId:
         return False, '%s is assigned as a VRRP address on interface %s ' \
                       'in vrf %s for virtual router %s' \
                       % ( ipGenAddr, fhrpIntf.intfId, vrfIp.vrfName,
                           fhrpIntf.vrId )

   # This isn't VARP, let's check if VARP is configured
   if vrId != VARP_VRID:
      varpConfig = _varpConfig()
      intfCol = varpConfig.virtualIpAddr.get( vrfIp )
      if intfCol:
         intfId, mask = next( iter( intfCol.intfIdToMask.items() ) )
         return False, '%s conflicts with VARP address %s/%s on interface %s' \
                       'in vrf %s' % ( ipGenAddr, ipGenAddr, mask,
                                       intfId, vrfIp.vrfName )

   return True, None

# the allowOverlap parameter is for varp addresses because varp addresses
# are allowed to be in physical subnets of other interfaces.
def uniqueIntfIpAssigned( ipAddr, intfName, allowOverlap=False ):
   if ipAddr == ipAddrZero:
      return True, None

   # mgmt prefix is '' to allow overlap for all interfaces
   mgmtPrefix = '' if allowOverlap else 'Management'
   primaryMatch = 'allowNoMatch'
   secondaryMatch = 'allowNoMatch'
   ipConfigEntity = ConfigMount.force( ipConfig )
   query = Tac.newInstance( 'Ira::AddressQueryParams', ipConfigEntity, intfName,
                            mgmtPrefix, primaryMatch, secondaryMatch,
                            Arnet.AddrWithMask( ipAddr ) )
   checker = Tac.Value( 'Ira::AddressAssignmentChecker' )
   ans = checker.check( query )
   return ans.acceptable, ans.msg

def uniqueIntfIp6Assigned( ip6Addr, intfName, allowOverlap=False ):
   if ip6Addr.isUnspecified:
      return True, None

   # mgmt prefix is '' to allow overlap for all interfaces
   mgmtPrefix = '' if allowOverlap else 'Management'
   ipConfigEntity = ConfigMount.force( ip6Config )
   query = Tac.newInstance( 'Ira::Address6QueryParams', ipConfigEntity, intfName,
                            mgmtPrefix, Arnet.Ip6AddrWithMask( ip6Addr ) )
   checker = Tac.Value( 'Ira::Address6AssignmentChecker' )
   ans = checker.check( query )
   return ans.acceptable, ans.msg

# Check if an ipv6 link-local address has been assigned to the interface, or any
# other vr ( VRRP or VARP ) on the interface
def uniqueIntfLinkLocalAssigned( vrfIp, ip6Addr, intfName, vrId ):
   intf = ip6Config.intf.get( intfName )
   if intf and intf.linkLocalAddrWithMask.address == ip6Addr:
      return False, '%s is assigned as a link-local address for this ' \
                    'interface' % ip6Addr
   vrrpConfig = _vrrpConfig()
   fhrpIntfCol = vrrpConfig.virtualIp6LinkLocalAddr.get( vrfIp )
   if fhrpIntfCol:
      for vrIdIntf in fhrpIntfCol.vrIdIntf:
         if vrIdIntf.intfId == intfName and vrIdIntf.vrId != vrId:
            return False, '%s is assigned as a VRRP address on this ' \
                          'interface in vrf %s for virtual router %s' \
                          % ( ip6Addr, vrfIp.vrfName, vrIdIntf.vrId )
   # This isn't VARP, let's check if VARP is using the same link-local address
   # on this interface
   if vrId != VARP_VRID:
      varpConfig = _varpConfig()
      intfCol = varpConfig.virtualIpAddr.get( vrfIp )
      if intfCol:
         if intfName in intfCol.intfIdToMask:
            return False, '%s is assigned as a VARP address on this ' \
                          'interface in vrf %s' % ( ip6Addr, vrfIp.vrfName )
   return True, None

# checks to see whether ipAddrWithMask overlaps with any configured varp
# addresses on interface intfName
def overlappingVarpIpAssigned( ipAddrWithMask, intfName ):
   varpConfig = _varpConfig()
   varpIntfConfig = varpConfig.varpIntfConfig.get( intfName )
   if varpIntfConfig:
      ipAddrWithMask = Arnet.AddrWithMask( ipAddrWithMask )
      for ip in varpIntfConfig.virtualIpAddr:
         sameIp = ip.address == ipAddrWithMask.address
         overlapIp = ( ip.contains( ipAddrWithMask.address ) or
                       ipAddrWithMask.contains( ip.address ) ) and \
                     ( ipAddrWithMask.len != 32 and ip.len != 32 )
         if not sameIp and overlapIp:
            return True, '%s overlaps with varp address %s' \
                         % ( ipAddrWithMask, ip )
   return False, None

def ipSubnetOverlappingIntfSubnet( ipAddrWithMask, intfName ):
   if ipAddrWithMask.len != _fhrpConfig().defaultIpMaskLen:
      ipIntf = ipConfig.ipIntfConfig.get( intfName )
      if ipIntf:
         if ipIntf.addrWithMask.contains( ipAddrWithMask.address ) or \
            ipAddrWithMask.contains( ipIntf.addrWithMask.address ):
            return True
         for secAddr in ipIntf.secondaryWithMask:
            if secAddr.contains( ipAddrWithMask.address ) or \
               ipAddrWithMask.contains( secAddr.address ):
               return True
   return False

def inIp4Subnet( ipIntf, ipAddr ):
   if ipIntf.addrWithMask.contains( ipAddr ):
      return True
   for secAddr in ipIntf.secondaryWithMask:
      if secAddr.contains( ipAddr ):
         return True
   return False

def inIp6Subnet( ipIntf, ip6Addr ):
   if not ipIntf.addr:
      return True
   for ip in ipIntf.addr:
      if ip.contains( ip6Addr ):
         return True
   return False

# FHRP Commands
#--------------------------------------------------------------------------------
# "[ no | default ] ip fhrp accept-mode" in config mode
#--------------------------------------------------------------------------------

def setFhrpAcceptMode( mode, args ):
   config = _fhrpConfig()
   config.acceptMode = 'enabled'

def noFhrpAcceptMode( mode, args ):
   config = _fhrpConfig()
   config.acceptMode = config.acceptModeDefault

class IpFhrpAcceptModeCmd( CliCommand.CliCommandClass ):
   syntax = 'ip fhrp accept-mode'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipMatcherForConfig,
      'fhrp': CliCommand.Node( CliMatcher.KeywordMatcher( 'fhrp',
                               helpdesc='First Hop Redundancy Protocol (FHRP)' ),
                               guard=fhrpSupportedGuard ),
      'accept-mode': 'Enable virtual IP accept mode',
   }

   handler = setFhrpAcceptMode
   noOrDefaultHandler = noFhrpAcceptMode

GlobalConfigMode.addCommandClass( IpFhrpAcceptModeCmd )

# VRRP Commands
#-------------------------------------------------------------------------------
# "no vrrp VRID" in config-if mode
#-------------------------------------------------------------------------------

def noVrrp( mode, vrId ):
   intfConfig = _vrrpIntfConfig( mode, createIfMissing=False )
   if intfConfig is None:
      return
   vrConfig = intfConfig.virtualRouterConfig.get( vrId )
   if vrConfig:
      deleteVrrpGlobalIp( vrConfig.primaryAddr, mode.intf.name )
      for ipAddr in vrConfig.secondaryAddrs:
         deleteVrrpGlobalIp( ipAddr, mode.intf.name )
      for ip6Addr in vrConfig.ip6Addrs:
         deleteVrrpGlobalIp( ip6Addr, mode.intf.name )
      del intfConfig.virtualRouterConfig[ vrId ]
   deleteVrrpIntfConfigIfDefault( mode.intf.name )
   evaluateFhrpRunnability()

class NoVrrpCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'vrrp VRID'
   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
   }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrId = args[ 'VRID' ]
      noVrrp( mode, vrId )

modelet.addCommandClass( NoVrrpCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID session description DESCRIPTION" in config-if mode
#
# legacy:
# "[ no | default ] vrrp VRID description DESCRIPTION" in config-if mode
#-------------------------------------------------------------------------------

def setVrrpDescription( mode, vrId, description ):
   config = _vrConfig( mode, vrId )
   if config is None:
      return
   config.description = description
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpDescription( mode, vrId ):
   config = _vrConfig( mode, vrId, createIfMissing=False )
   if config is None:
      return
   config.description = config.descriptionDefault
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpSessionDescriptionCmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID ( ( session description ) | DESCRIPTION_DEPRECATED ) ' \
            'DESCRIPTION'
   noOrDefaultSyntax = 'vrrp VRID ( ( session description ) | ' \
                       'DESCRIPTION_DEPRECATED ) ...'

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'session': 'VRRP session Information',
      'description': 'Virtual router description',
      'DESCRIPTION_DEPRECATED': CliCommand.Node( KeywordMatcher( 'description',
         helpdesc='Virtual router description' ),
         deprecatedByCmd='vrrp session description' ),
      'DESCRIPTION': StringMatcher( helpdesc='Virtual router description' ),
   }

   @staticmethod
   def handler( mode, args ):
      vrId = args[ 'VRID' ]
      description = args[ 'DESCRIPTION' ]
      setVrrpDescription( mode, vrId, description )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrId = args[ 'VRID' ]
      noVrrpDescription( mode, vrId )

modelet.addCommandClass( VrrpSessionDescriptionCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID ipv4 IPADDR [ secondary ]" in config-if
#
# legacy:
# "[ no | default ] vrrp VRID ip IPADDR [ secondary ]" in config-if
#-------------------------------------------------------------------------------

def setVrrpIp( mode, vrId, ipAddr, secondary=None ):
   intfName = mode.intf.name
   vrf = getVrfName( intfName )
   vrfIp = vrfIpPair( vrf, ipAddr )

   # Checking for used addresses on any interface
   if ( not mode.session.skipConfigCheck() and
        ipAddr != ipAddrZero.stringValue and not _allowDuplicate() ):
      # Checking vrrp config for used vips
      unique, msg = uniqueFhrpIpAssigned( vrfIp, intfName, vrId )
      if not unique:
         mode.addError( msg )
         return

      def isOnesOrZeroAddr( ipAddrWithMask ):
         if ipAddrWithMask.allOnesAddr == ipAddr:
            return 'Virtual IP address must not be a broadcast address'
         elif ipAddrWithMask.allZerosAddr == ipAddr:
            return 'Virtual IP address must not be the network address'
         return None

      ipIntf = ipConfig.ipIntfConfig.get( intfName )
      if ipIntf:
         warningMessage = None
         # Checking if address is a zero address or broadcast address
         for secAddr in ipIntf.secondaryWithMask:
            msg = isOnesOrZeroAddr( secAddr )
            if msg:
               warningMessage = msg

         msg = isOnesOrZeroAddr( ipIntf.addrWithMask )
         if msg:
            warningMessage = msg

         # Only want to print one of the relevant messages
         if warningMessage:
            mode.addWarning( warningMessage )

         # Checking ipConfig for same or overlapping in an ip on different interface
         if not inIp4Subnet( ipIntf, ipAddr ):
            mode.addWarning( '%s is not within a subnet on this interface'
                             % ipAddr )

      # Check if ip address is used as other physical addresses
      if not Toggles.FhrpToggleLib.toggleVrrpSamePhysicalIpAsVipEnabled():
         unique, msg = uniqueIntfIpAssigned( ipAddr, intfName )
         if unique:
            if msg:
               mode.addWarning( msg )
         else:
            mode.addError( msg )
            return

   # Check the platform limitations if any.
   warning = getVrrpPlatformLimitations( vrId, AF_INET )
   if warning:
      mode.addWarning( warning )

   vrrpConfig = _vrrpConfig()
   config = _vrConfig( mode, vrId )
   if config is None:
      return

   if secondary:
      # If the ip is default, do nothing
      if ipAddr != config.primaryAddrDefault:
         # This is secondary, fail if address is already primary
         if ipAddr == config.primaryAddr:
            mode.addError( '%s is assigned as the primary address' % ipAddr )
         # If the ip is already in secondary, do nothing
         elif ipAddr not in config.secondaryAddrs:
            # Check for max assigned addresses
            if len( config.secondaryAddrs ) == MAX_VRRP_SECONDARY_IPV4_PER_VRID:
               mode.addError( 'Maximum number of IPv4 virtual addresses (%d) per '
                              'VRRP group assigned' %
                              MAX_VRRP_SECONDARY_IPV4_PER_VRID )
            # Success
            else:
               # Add to global collection
               vrfIp = vrfIpPair( vrf, ipAddr )
               vrrpConfig.virtualIpAddr[ vrfIp ] = fhrpIntfIdPair( intfName,
                                                                       vrId )
               # Add to secondary address collection
               config.secondaryAddrs[ ipAddr ] = True
   else:
      # This is primary, fail if address is already in secondary
      if ipAddr in config.secondaryAddrs:
         mode.addError( '%s is assigned as a secondary address' % ipAddr )
      # If the ip is the same as configured primary, do nothing
      elif ipAddr != config.primaryAddr:
         oldIpAddr = config.primaryAddr
         # Check for max number of vrrp virtual routers created
         if ( ipAddr != config.primaryAddrDefault and
              oldIpAddr == config.primaryAddrDefault and
              maxVrCountExceeded() ):
            mode.addError( "More than %s virtual routers not supported"
                           % ( fhrpHardwareStatus.vrrpSupportedMaxNum ) )
         # Success and assigning default ip is the same as clearing the config
         else:
            # If ip is not default, add to global collection
            if ipAddr != config.primaryAddrDefault:
               vrfIp = vrfIpPair( vrf, ipAddr )
               vrrpConfig.virtualIpAddr[ vrfIp ] = fhrpIntfIdPair( intfName, vrId )
            # Set the primary to ipAddr
            config.primaryAddr = ipAddr
            # Prevent multiple warnings
            if oldIpAddr == config.primaryAddrDefault:
               checkAdvtInterval( mode, config )
            else:
               # If old ip is not default, delete from global collection
               oldVrfIp = vrfIpPair( vrf, oldIpAddr )
               del vrrpConfig.virtualIpAddr[ oldVrfIp ]

   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpIp( mode, vrId, ipAddr, secondary=None ):
   config = _vrConfig( mode, vrId, createIfMissing=False )
   if config is None:
      return
   if secondary:
      if ipAddr in config.secondaryAddrs:
         deleteVrrpGlobalIp( ipAddr, mode.intf.name )
         del config.secondaryAddrs[ ipAddr ]
      elif ipAddr != config.primaryAddrDefault:
         mode.addError( 'Cannot find secondary IPv4 address %s' % ipAddr )
   else:
      if ipAddr == config.primaryAddr:
         # If ip is not default, delete from global collection
         if ipAddr != config.primaryAddrDefault:
            deleteVrrpGlobalIp( ipAddr, mode.intf.name )
         config.primaryAddr = config.primaryAddrDefault
      else:
         mode.addError( 'Primary address does not match %s' % ipAddr )

   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpIpv4Cmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID ( ipv4 | ip ) IPADDR [ secondary ]'
   noOrDefaultSyntax = syntax

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'ipv4': nodeVrrpIpv4,
      'ip': nodeVrrpIp,
      'IPADDR': IpAddrMatcher.ipAddrMatcher,
      'secondary': 'Specify an additional VRRP address',
   }

   @staticmethod
   def handler( mode, args ):
      vrId = args[ 'VRID' ]
      ipAddr = args[ 'IPADDR' ]
      secondary = 'secondary' in args
      setVrrpIp( mode, vrId, ipAddr, secondary )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrId = args[ 'VRID' ]
      ipAddr = args[ 'IPADDR' ]
      secondary = 'secondary' in args
      noVrrpIp( mode, vrId, ipAddr, secondary )

modelet.addCommandClass( VrrpIpv4Cmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID bfd ip IPADDR" in config-if
#-------------------------------------------------------------------------------

def setVrrpBfdPeerIp( mode, args ):
   vrId = args[ 'VRID' ]
   ipAddr = args[ 'IPADDR' ]
   intfName = mode.intf.name
   warningMessage = None

   # Checks are disabled on startup/config-replace to improve performance
   if ( not mode.session.skipConfigCheck() and
        ipAddr != ipAddrZero.stringValue ):
      # Checking that ipAddr is not configured as any physical/virtual address
      for intf in _vrrpConfig().vrrpIntfConfig.values():
         for vr in intf.virtualRouterConfig.values():
            if vr.primaryAddr == ipAddr:
               mode.addError( '%s is already configured as the primary '
                              'address on %s interface as virtual '
                              'router %s' % ( ipAddr, intf.intfId, vr.vrId ) )
               return
            elif vr.secondaryAddrs and ipAddr in vr.secondaryAddrs:
               mode.addError( '%s is already configured as a secondary '
                              'address on %s interface as virtual '
                              'router %s' % ( ipAddr, intf.intfId, vr.vrId ) )
               return
      primaryMatch = 'allowNoMatch'
      secondaryMatch = 'allowNoMatch'
      ipConfigEntity = ConfigMount.force( ipConfig )
      query = Tac.newInstance( 'Ira::AddressQueryParams', ipConfigEntity, intfName,
                               'Management', primaryMatch, secondaryMatch,
                                Arnet.AddrWithMask( ipAddr ) )
      checker = Tac.Value( 'Ira::AddressAssignmentChecker' )
      ans = checker.check( query )
      if ans.acceptable:
         if ans.msg:
            mode.addWarning( '%s' % ans.msg )
      else:
         mode.addError( '%s' % ans.msg )
         return

      ipIntf = ipConfig.ipIntfConfig.get( intfName )
      if ipIntf:
         if not inIp4Subnet( ipIntf, ipAddr ):
            warningMessage = '%s is not within a subnet on this interface' % ipAddr

   config = _vrConfig( mode, vrId )
   if config is None:
      return

   config.bfdPeerAddrV4 = ipAddr

   if warningMessage:
      mode.addWarning( warningMessage )

   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpBfdPeerIp( mode, args ):
   vrId = args[ 'VRID' ]
   config = _vrConfig( mode, vrId, createIfMissing=False )
   if config is None:
      return
   config.bfdPeerAddrV4 = config.bfdPeerAddrV4Default
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpBfdPeerIpCmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID bfd ip IPADDR'
   noOrDefaultSyntax = 'vrrp VRID bfd ip ...'

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'bfd': matcherBfd,
      'ip': 'Configure peer IPv4 address',
      'IPADDR': IpAddrMatcher.ipAddrMatcher,
   }

   handler = setVrrpBfdPeerIp
   noOrDefaultHandler = noVrrpBfdPeerIp

modelet.addCommandClass( VrrpBfdPeerIpCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID bfd ipv6 IP6ADDR" in config-if
#-------------------------------------------------------------------------------

def setVrrpBfdPeerIp6( mode, args ):
   vrId = args[ 'VRID' ]
   ip6Addr = args[ 'IP6ADDR' ]
   intfName = mode.intf.name
   warningMessage = None

   if not mode.session.skipConfigCheck() and not ip6Addr.isUnspecified:
      # Checking that ip6Addr is not configured as any physical or virtual ip
      for intf in ip6Config.intf.values():
         for ip6 in intf.addr:
            if ip6.address == ip6Addr:
               mode.addError( '%s is already configured on interface %s'
                              % ( ip6Addr, intf.intfId ) )
               return
      vrrpConfig = _vrrpConfig()
      for intf in vrrpConfig.vrrpIntfConfig.values():
         for vr in intf.virtualRouterConfig.values():
            if ip6Addr in vr.ip6Addrs:
               mode.addError( '%s is already configured on %s '
                              'interface as virtual router %s'
                              % ( ip6Addr, intf.intfId, vrId ) )
               return
      intf = ip6Config.intf.get( intfName )
      if intf and not inIp6Subnet( intf, ip6Addr ):
         warningMessage = '%s is not within a subnet on this interface' % ip6Addr

   config = _vrConfig( mode, vrId )
   if config is None:
      return

   config.bfdPeerAddrV6 = ip6Addr

   if warningMessage:
      mode.addWarning( warningMessage )

   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpBfdPeerIp6( mode, args ):
   vrId = args[ 'VRID' ]
   config = _vrConfig( mode, vrId, createIfMissing=False )
   if config is None:
      return
   config.bfdPeerAddrV6 = config.bfdPeerAddrV6Default
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpBfdPeerIpv6Cmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID bfd ipv6 IP6ADDR'
   noOrDefaultSyntax = 'vrrp VRID bfd ipv6 ...'

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'bfd': matcherBfd,
      'ipv6': 'Configure peer IPv6 address',
      'IP6ADDR': Ip6AddrMatcher( helpdesc='IPv6 address' ),
   }

   handler = setVrrpBfdPeerIp6
   noOrDefaultHandler = noVrrpBfdPeerIp6

modelet.addCommandClass( VrrpBfdPeerIpv6Cmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID ipv4 version { 2 | 3 }" in config-if
#
# legacy:
# "[ no | default ] vrrp VRID ip version { 2 | 3 }" in config-if
#-------------------------------------------------------------------------------

def setVrrpVersion( mode, vrId, version ):
   config = _vrConfig( mode, vrId )
   if config is None:
      return
   oldVersion = config.version
   config.version = version
   if oldVersion == config.versionDefault:
      checkAdvtInterval( mode, config )
   checkAuthAndVrrpVersion( mode, config )
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpVersion( mode, vrId ):
   config = _vrConfig( mode, vrId, createIfMissing=False )
   if config is None:
      return
   config.version = config.versionDefault
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpIpv4VersionCmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID ( ipv4 | ip ) version VRRPVERSION'
   noOrDefaultSyntax = 'vrrp VRID ( ipv4 | ip ) version ...'

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'ipv4': nodeVrrpIpv4,
      'ip': nodeVrrpIp,
      'version': 'Set the VRRP version',
      'VRRPVERSION': CliMatcher.IntegerMatcher( 2, 3, helpdesc='Version 2 or 3' ),
   }

   @staticmethod
   def handler( mode, args ):
      vrId = args[ 'VRID' ]
      version = args[ 'VRRPVERSION' ]
      setVrrpVersion( mode, vrId, version )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrId = args[ 'VRID' ]
      noVrrpVersion( mode, vrId )

modelet.addCommandClass( VrrpIpv4VersionCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID ipv4 checksum pseudo-header exclude" in config-if
#-------------------------------------------------------------------------------

def setVrrpExcludePseudoSum( mode, args ):
   vrId = args[ 'VRID' ]
   config = _vrConfig( mode, vrId )
   if config is None:
      return
   config.ipv4ExcludePseudoChecksum = True
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpExcludePseudoSum( mode, args ):
   vrId = args[ 'VRID' ]
   config = _vrConfig( mode, vrId, createIfMissing=False )
   if config is None:
      return
   config.ipv4ExcludePseudoChecksum = config.ipv4ExcludePseudoChecksumDefault
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpIpv4ExcludePseudoSumCmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID ipv4 checksum pseudo-header exclude'
   noOrDefaultSyntax = syntax

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'ipv4': nodeVrrpIpv4,
      'checksum': 'Modify the checksum',
      'pseudo-header': 'Modify the pseudo-header checksum',
      'exclude': 'Exclude pseudo header from the checksum',
   }

   handler = setVrrpExcludePseudoSum
   noOrDefaultHandler = noVrrpExcludePseudoSum

modelet.addCommandClass( VrrpIpv4ExcludePseudoSumCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID ipv6 IP6ADDR" in config-if
#-------------------------------------------------------------------------------

def setVrrpIp6( mode, args ):
   vrId = args[ 'VRID' ]
   ip6Addr = args[ 'IP6ADDR' ]
   intfName = mode.intf.name
   vrf = getVrfName( intfName )
   vrfIp = vrfIpPair( vrf, ip6Addr )

   # Checking for used addresses on any interface
   if ( not mode.session.skipConfigCheck() and not ip6Addr.isUnspecified
        and not _allowDuplicate() ):
      intf = ip6Config.intf.get( intfName )
      if not ip6Addr.isLinkLocal:
         unique, msg = uniqueFhrpIpAssigned( vrfIp, intfName, vrId )
         if not unique:
            mode.addError( msg )
            return
         if intf and not inIp6Subnet( intf, ip6Addr ):
            mode.addWarning( '%s is not within a subnet on this interface'
                              % ip6Addr )

         if not Toggles.FhrpToggleLib.toggleVrrpSamePhysicalIpAsVipEnabled():
            unique, msg = uniqueIntfIp6Assigned( ip6Addr, intfName )
            if unique:
               if msg:
                  mode.addWarning( msg )
            else:
               mode.addError( msg )
               return
      else:
         if not Toggles.FhrpToggleLib.toggleVrrpSamePhysicalIpAsVipEnabled():
            unique, msg = uniqueIntfLinkLocalAssigned( vrfIp, ip6Addr, intfName,
                                                       vrId )
            if not unique:
               mode.addError( msg )
               return

   # Check the platform limitations if any.
   warning = getVrrpPlatformLimitations( vrId, AF_INET6 )
   if warning:
      mode.addWarning( warning )

   config = _vrConfig( mode, vrId )
   if config is None:
      return

   # If address is default or already configured, do nothing
   if not ip6Addr.isUnspecified and ip6Addr not in config.ip6Addrs:
      # Check if max ip6 addresses already configured
      if len( config.ip6Addrs ) == MAX_VRRP_IPV6_PER_VRID:
         mode.addError( 'Maximum number of IPv6 virtual addresses (%d) per '
                        'VRRP group assigned' % MAX_VRRP_IPV6_PER_VRID )
         return
      # Check if max non-link-local ip6 addresses already configured
      if len( config.ip6Addrs ) == MAX_VRRP_IPV6_PER_VRID - 1 and \
         not ip6Addr.isLinkLocal:
         hasConfiguredLinkLocal = False
         for addr in config.ip6Addrs:
            if addr.isLinkLocal:
               hasConfiguredLinkLocal = True
               break
         if not hasConfiguredLinkLocal:
            mode.addError( 'Maximum number of non-link-local IPv6 virtual '
                           'addresses (%d) per VRRP group already assigned' %
                           ( MAX_VRRP_IPV6_PER_VRID - 1 ) )
            return
      # Check if able to add another virtual router
      if not config.ip6Addrs and maxVrCountExceeded():
         mode.addError( 'More than %s virtual routers not supported'
                        % ( fhrpHardwareStatus.vrrpSupportedMaxNum ) )
         deleteVrConfigIfDefault( mode.intf.name, vrId )
         return
      else:
         if ip6Addr not in config.ip6Addrs:
            vrfIp = vrfIpPair( vrf, ip6Addr )
            vrrpConfig = _vrrpConfig()
            fhrpIntfId = fhrpIntfIdPair( intfName, vrId )
            vrrpConfig.virtualIpAddr[ vrfIp ] = fhrpIntfId
            virtualIp6LinkLocalAddr = vrrpConfig.virtualIp6LinkLocalAddr
            # Since same IPv6 link local address can be configured as a VRRP address
            # accross multiple virtual routers, virtualIp6LinkLocalAddr collection
            # used for bookkeeping these addresses need to be updated accordingly.
            if ip6Addr.isLinkLocal:
               if vrfIp not in virtualIp6LinkLocalAddr:
                  virtualIp6LinkLocalAddr.newMember( vrfIp )
               virtualIp6LinkLocalAddr[ vrfIp ].vrIdIntf[ fhrpIntfId ] = True

            config.ip6Addrs[ ip6Addr ] = True
            checkAdvtInterval( mode, config )

   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpIp6( mode, args ):
   vrId = args[ 'VRID' ]
   ip6Addr = args[ 'IP6ADDR' ]
   config = _vrConfig( mode, vrId, createIfMissing=False )
   if config is None:
      return

   if ip6Addr in config.ip6Addrs:
      deleteVrrpGlobalIp( ip6Addr, mode.intf.name, vrId=vrId )
      del config.ip6Addrs[ ip6Addr ]
   elif not ip6Addr.isUnspecified:
      mode.addError( 'Cannot find IPv6 address %s' % ip6Addr )
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpIpv6Cmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID ipv6 IP6ADDR'
   noOrDefaultSyntax = syntax

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'ipv6': nodeVrrpIpv6,
      'IP6ADDR': Ip6AddrMatcher( helpdesc='IPv6 address' ),
   }

   handler = setVrrpIp6
   noOrDefaultHandler = noVrrpIp6

modelet.addCommandClass( VrrpIpv6Cmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID preempt" in config-if
#-------------------------------------------------------------------------------

def setVrrpPreempt( mode, args ):
   vrId = args[ 'VRID' ]
   config = _vrConfig( mode, vrId )
   if config is None:
      return
   config.preempt = True
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpPreempt( mode, args ):
   vrId = args[ 'VRID' ]
   config = _vrConfig( mode, vrId )
   if config is None:
      return
   config.preempt = False
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpPreemptCmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID preempt'
   noSyntax = syntax
   defaultSyntax = syntax

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'preempt': 'Configure the preempt mode',
   }

   handler = setVrrpPreempt
   noHandler = noVrrpPreempt
   defaultHandler = handler

modelet.addCommandClass( VrrpPreemptCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID preempt delay { ( minimum <0-3600> ) |
# ( reload <0-3600> ) }" in config-if
#-------------------------------------------------------------------------------

def setVrrpPreemptDelay( mode, args ):
   vrId = args[ 'VRID' ]

   config = _vrConfig( mode, vrId )
   if config is None:
      return

   if 'minimum' in args:
      interval = args[ 'DELAYTIME' ]
      config.preemptDelay = interval * 100
   if 'reload' in args:
      interval = args[ 'RELOADTIME' ]
      config.preemptReloadDelay = interval * 100

   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpPreemptDelay( mode, args ):
   vrId = args[ 'VRID' ]

   config = _vrConfig( mode, vrId )
   if config is None:
      return

   if 'minimum' in args:
      config.preemptDelay = config.preemptDelayDefault
   elif 'reload' in args:
      config.preemptReloadDelay = config.preemptReloadDelayDefault
   else:
      config.preemptReloadDelay = config.preemptReloadDelayDefault
      config.preemptDelay = config.preemptDelayDefault

   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpPreemptDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID preempt delay { ( minimum DELAYTIME ) | ' \
            '( reload RELOADTIME ) }'
   noOrDefaultSyntax = 'vrrp VRID preempt delay [ minimum | reload ] ...'

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'preempt': 'Configure the preempt mode',
      'delay': 'Required if either the minimum or reload keywords are specified',
      'minimum': CliCommand.Node( KeywordMatcher( 'minimum',
         helpdesc='Specify the minimum time in seconds for the local '
                  'router to wait before taking over the active role. '
                  'Default is 0 seconds.' ), maxMatches=1 ),
      'DELAYTIME': CliCommand.Node( matcherHourInterval, maxMatches=1 ),
      'reload': CliCommand.Node( KeywordMatcher( 'reload',
         helpdesc='Specify the time after a reload in seconds for the local router '
                  'to wait before taking over the active role. '
                  'Default is 0 seconds' ), maxMatches=1 ),
      'RELOADTIME': CliCommand.Node( matcherHourInterval, maxMatches=1 ),
   }

   handler = setVrrpPreemptDelay
   noOrDefaultHandler = noVrrpPreemptDelay

modelet.addCommandClass( VrrpPreemptDelayCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID advertisement interval <1-255>" in config-if
#
# legacy:
# "[ no | default ] vrrp VRID timers advertise <1-255>" in config-if
#-------------------------------------------------------------------------------

def setVrrpAdvertisementInterval( mode, vrId, time ):
   config = _vrConfig( mode, vrId )
   if config is None:
      return
   oldAdvInterval = config.advInterval // 100
   config.advInterval = time * 100
   # Prevent multiple warnings
   # pylint: disable-next=chained-comparison
   if ( oldAdvInterval <= MAX_VRRP_V3_ADVERTISE_INTERVAL and
        ( config.ip6Addrs or
          ( config.primaryAddr != config.primaryAddrDefault and
            config.version == 3 ) ) and
        time > MAX_VRRP_V3_ADVERTISE_INTERVAL ):
      mode.addWarning( advtIntervalWarningMessage )
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpAdvertisementInterval( mode, vrId ):
   config = _vrConfig( mode, vrId, createIfMissing=False )
   if config is None:
      return
   config.advInterval = config.advIntervalDefault
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpAdvIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID ( ( advertisement interval ) | ' \
            '( timers advertise ) ) INTERVAL'
   noOrDefaultSyntax = 'vrrp VRID ( ( advertisement interval ) | ' \
                       '( timers advertise ) ) ...'

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'advertisement': matcherAdvertisement,
      'interval': 'Configure timer interval',
      'timers': 'Configure the VRRP timer',
      'advertise': CliCommand.Node( KeywordMatcher( 'advertise',
            helpdesc='Set advertisement timer interval' ),
         deprecatedByCmd='vrrp advertisement interval' ),
      'INTERVAL': CliMatcher.IntegerMatcher( 1, 255,
         helpdesc='Interval in units of seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      vrId = args[ 'VRID' ]
      time = args[ 'INTERVAL' ]
      setVrrpAdvertisementInterval( mode, vrId, time )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrId = args[ 'VRID' ]
      noVrrpAdvertisementInterval( mode, vrId )

modelet.addCommandClass( VrrpAdvIntervalCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID advertisement mismatch action { log | drop }"
# in config-if
#-------------------------------------------------------------------------------

def setVrrpAdvMismatchAction( mode, args ):
   vrId = args[ 'VRID' ]
   config = _vrConfig( mode, vrId )
   if config is None:
      return
   if 'drop' in args:
      config.advMismatchDrop = True
   else:
      config.advMismatchDrop = config.advMismatchDropDefault
   if 'log' in args:
      config.advMismatchLog = True
   else:
      config.advMismatchLog = config.advMismatchLogDefault
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpAdvMismatchAction( mode, args ):
   vrId = args[ 'VRID' ]
   config = _vrConfig( mode, vrId, createIfMissing=False )
   if config is None:
      return
   config.advMismatchDrop = config.advMismatchDropDefault
   config.advMismatchLog = config.advMismatchLogDefault
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpAdvMismatchActionCmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID advertisement mismatch action { log | drop }'
   noOrDefaultSyntax = 'vrrp VRID advertisement mismatch action [ { log | drop } ]'

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'advertisement': matcherAdvertisement,
      'mismatch': 'Configure the advertisement mismatch detection',
      'action': 'Configure the actions when mismatches are detected',
      'log': CliCommand.Node( KeywordMatcher( 'log',
             helpdesc='Log the mismatched advertisement packets' ),
             maxMatches=1 ),
      'drop': CliCommand.Node( KeywordMatcher( 'drop',
              helpdesc='Drop the mismatched advertisement packets' ),
              maxMatches=1 ),
   }

   handler = setVrrpAdvMismatchAction
   noOrDefaultHandler = noVrrpAdvMismatchAction

modelet.addCommandClass( VrrpAdvMismatchActionCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID timers delay reload <0-3600>" in config-if mode
#
# legacy:
# "[ no | default ] vrrp VRID delay reload <0-3600>" in config-if mode
#-------------------------------------------------------------------------------

def setVrrpReloadDelay( mode, vrId, reloadDelay ):
   config = _vrConfig( mode, vrId )
   if config is None:
      return
   config.reloadDelay = reloadDelay
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpReloadDelay( mode, vrId ):
   config = _vrConfig( mode, vrId )
   if config is None:
      return
   config.reloadDelay = config.reloadDelayDefault
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpDelayReloadCmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID ( ( timers delay ) | delayDeprecated ) reload INTERVAL'
   noOrDefaultSyntax = 'vrrp VRID ( ( timers delay ) | delayDeprecated ) reload ...'

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'timers': 'Configure the VRRP timer',
      'delay': 'VRRP initialization delay',
      'delayDeprecated': CliCommand.Node( KeywordMatcher( 'delay',
                          helpdesc='VRRP initialization delay' ),
                          deprecatedByCmd='vrrp timers delay reload' ),
      'reload': 'Delay after reload. Default is 0 seconds.',
      'INTERVAL': matcherHourInterval,
   }

   @staticmethod
   def handler( mode, args ):
      vrId = args[ 'VRID' ]
      reloadDelay = args[ 'INTERVAL' ]
      setVrrpReloadDelay( mode, vrId, reloadDelay )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrId = args[ 'VRID' ]
      noVrrpReloadDelay( mode, vrId )

modelet.addCommandClass( VrrpDelayReloadCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID priority-level <1-254>" in config-if
#
# legacy:
# "[ no | default ] vrrp VRID priority <1-254>" in config-if
#-------------------------------------------------------------------------------

def setVrrpPriority( mode, vrId, priority ):
   config = _vrConfig( mode, vrId )
   if config is None:
      return
   config.priority = priority
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpPriority( mode, vrId ):
   config = _vrConfig( mode, vrId, createIfMissing=False )
   if config is None:
      return
   config.priority = config.priorityDefault
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID ( priority-level | priority ) PRIORITY'
   noOrDefaultSyntax = 'vrrp VRID ( priority-level | priority ) ...'

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'priority-level': 'Priority of this VRRP Router',
      'priority': CliCommand.Node( KeywordMatcher( 'priority',
            helpdesc='Priority of this VRRP Router' ),
         deprecatedByCmd='vrrp priority-level' ),
      'PRIORITY': CliMatcher.IntegerMatcher( 1, 254,
         helpdesc='VRRP router priority value' ),
   }

   @staticmethod
   def handler( mode, args ):
      vrId = args[ 'VRID' ]
      priority = args[ 'PRIORITY' ]
      setVrrpPriority( mode, vrId, priority )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrId = args[ 'VRID' ]
      noVrrpPriority( mode, vrId )

modelet.addCommandClass( VrrpPriorityCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID mac-address advertisement-interval <0-3600>"
# in config-if mode
#-------------------------------------------------------------------------------

def setVrrpMacAdvertise( mode, args ):
   vrId = args[ 'VRID' ]
   interval = args[ 'INTERVAL' ]
   config = _vrConfig( mode, vrId )
   if config is None:
      return
   config.arpOrNAInterval = interval * 100
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpMacAdvertise( mode, args ):
   vrId = args[ 'VRID' ]
   config = _vrConfig( mode, vrId, createIfMissing=False )
   if config is None:
      return
   config.arpOrNAInterval = config.arpOrNAIntervalDefault
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpMacAdvIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID mac-address advertisement-interval INTERVAL'
   noOrDefaultSyntax = 'vrrp VRID mac-address ...'

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'mac-address': 'Configure the mac address advertisement interval',
      'advertisement-interval': 'Set the time between mac-address advertisement '
      'packets',
      'INTERVAL': matcherHourInterval,
   }

   handler = setVrrpMacAdvertise
   noOrDefaultHandler = noVrrpMacAdvertise

modelet.addCommandClass( VrrpMacAdvIntervalCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID disabled" in config-if mode
#
# legacy:
# "[ no | default ] vrrp VRID shutdown" in config-if mode
#-------------------------------------------------------------------------------

def setVrrpDisabled( mode, vrId ):
   config = _vrConfig( mode, vrId )
   if config is None:
      return
   config.adminEnabledState = intfEnabledStateEnum.shutdown
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpDisabled( mode, vrId ):
   config = _vrConfig( mode, vrId, createIfMissing=False )
   if config is None:
      return
   config.adminEnabledState = config.adminEnabledStateDefault
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID ( disabled | shutdown )'
   noOrDefaultSyntax = syntax

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'disabled': 'Disable VRRP configuration',
      'shutdown': CliCommand.Node( KeywordMatcher( 'shutdown',
                                    helpdesc='Disable VRRP configuration' ),
                                    deprecatedByCmd='vrrp disabled' ),
   }

   @staticmethod
   def handler( mode, args ):
      vrId = args[ 'VRID' ]
      setVrrpDisabled( mode, vrId )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrId = args[ 'VRID' ]
      noVrrpDisabled( mode, vrId )

modelet.addCommandClass( VrrpDisabledCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID track OBJECTNAME decrement PRIORITY"
# in config-if mode
#
# legacy:
# "[ no | default ] vrrp VRID track OBJECTNAME decrement PRIORITY"
# in config-if mode
#-------------------------------------------------------------------------------

def setVrrpTrackPriority( mode, vrId, objectName, priority ):
   config = _vrConfig( mode, vrId )
   if config is None:
      return

   trackedObject = config.vrrpTrackedObject.get( objectName )
   if trackedObject is None:
      trackedObject = config.vrrpTrackedObject.newMember( objectName )
   trackedObject.trackPriorityDecrement = priority
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpTrackPriority( mode, vrId, objectName ):
   config = _vrConfig( mode, vrId, createIfMissing=False )
   if config is None:
      return

   if objectName not in config.vrrpTrackedObject:
      return
   trackedObject = config.vrrpTrackedObject[ objectName ]
   trackedObject.trackPriorityDecrement = \
         trackedObject.trackPriorityDecrementDefault

   if ( trackedObject.trackActionIsShutdown ==
        trackedObject.trackActionIsShutdownDefault ):
      del config.vrrpTrackedObject[ objectName ]
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpTrackPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID ( tracked-object | track ) OBJECTNAME decrement PRIORITY'
   noOrDefaultSyntax = \
         'vrrp VRID ( tracked-object | track ) OBJECTNAME decrement ...'

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'tracked-object': 'Track an object',
      'track': CliCommand.Node( KeywordMatcher( 'track',
            helpdesc='Track an object' ),
         deprecatedByCmd='vrrp tracked-object' ),
      'OBJECTNAME': matcherObjName,
      'decrement': 'Decrement VRRP priority',
      'PRIORITY': CliMatcher.IntegerMatcher( 1, 254,
         helpdesc='VRRP router priority value' ),
   }

   @staticmethod
   def handler( mode, args ):
      vrId = args[ 'VRID' ]
      objectName = args[ 'OBJECTNAME' ]
      priority = args[ 'PRIORITY' ]
      setVrrpTrackPriority( mode, vrId, objectName, priority )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrId = args[ 'VRID' ]
      objectName = args[ 'OBJECTNAME' ]
      noVrrpTrackPriority( mode, vrId, objectName )

modelet.addCommandClass( VrrpTrackPriorityCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID tracked-object OBJNAME shutdown"
# in config-if mode
#
# legacy:
# "[ no | default ] vrrp VRID track OBJNAME shutdown" in config-if mode
#-------------------------------------------------------------------------------

def setTrackShutdown( mode, vrId, objectName ):
   config = _vrConfig( mode, vrId )
   if config is None:
      return

   if objectName not in config.vrrpTrackedObject:
      config.vrrpTrackedObject.newMember( objectName )
   trackedObject = config.vrrpTrackedObject[ objectName ]
   trackedObject.trackActionIsShutdown = True
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noTrackShutdown( mode, vrId, objectName ):
   config = _vrConfig( mode, vrId, createIfMissing=False )
   if config is None:
      return

   if objectName not in config.vrrpTrackedObject:
      return
   trackedObject = config.vrrpTrackedObject[ objectName ]
   trackedObject.trackActionIsShutdown = trackedObject.trackActionIsShutdownDefault
   if ( trackedObject.trackPriorityDecrement ==
        trackedObject.trackPriorityDecrementDefault ):
      del config.vrrpTrackedObject[ objectName ]
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

class VrrpTrackShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID ( tracked-object | track ) OBJECTNAME shutdown'
   noOrDefaultSyntax = syntax

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'tracked-object': 'Track an object',
      'track': CliCommand.Node( KeywordMatcher( 'track',
                                 helpdesc='Track an object' ),
                                 deprecatedByCmd='vrrp tracked-object' ),
      'OBJECTNAME': matcherObjName,
      'shutdown': 'Disable VRRP configuration',
   }

   @staticmethod
   def handler( mode, args ):
      vrId = args[ 'VRID' ]
      objectName = args[ 'OBJECTNAME' ]
      setTrackShutdown( mode, vrId, objectName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrId = args[ 'VRID' ]
      objectName = args[ 'OBJECTNAME' ]
      noTrackShutdown( mode, vrId, objectName )

modelet.addCommandClass( VrrpTrackShutdownCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] vrrp VRID peer authentication (
# ( [ text ]  ( ( [ 0 ] CLEARTEXT ) | ( 7 TYPE7KEY ) | ( 8a TYPE8AKEY ) ) ) |
# ( ietf-md5 key-string ( ( 7 TYPE7KEY ) | ( [ 0 ] CLEARTEXT ) |
#                         ( 8A TYPE8AKEY ) ) ) )"
# in config-if mode
#
# legacy:
# "[ no | default ] vrrp VRID authentication
# ( [ text ]  ( ( [ 0 ] CLEARTEXT ) | ( 7 TYPE7KEY ) | ( 8a TYPE8AKEY ) ) ) |
# ( ietf-md5 key-string ( ( 7 TYPE7KEY ) | ( [ 0 ] CLEARTEXT ) |
#                         ( 8A TYPE8AKEY ) ) ) )"
# in config-if mode
#-------------------------------------------------------------------------------

def setVrrpAuthenticationKey( mode, vrId, authType, password ):
   config = _vrConfig( mode, vrId )
   if config is None:
      return

   if password.clearTextSize() > authKeyMaxLen:
      mode.addErrorAndStop( 'Invalid encrypted keystring' )

   config.authKey = password
   config.authType = authType
   checkAuthAndVrrpVersion( mode, config )
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

def noVrrpAuthentication( mode, vrId ):
   config = _vrConfig( mode, vrId, createIfMissing=False )
   if config is None:
      return
   config.authKey = config.authKeyDefault
   config.authType = config.authTypeDefault
   deleteVrConfigIfDefault( mode.intf.name, vrId )
   evaluateFhrpRunnability()

authKeyMinLen = 1
authKeyMaxLen = 8
authKeyLenRegex = f"{{{authKeyMinLen},{authKeyMaxLen}}}"
authTextStringMatcher = CliMatcher.PatternMatcher(
      r'[a-zA-Z0-9!#$%%&\'()*+,-./:;<=>@[\\\]^_`{|}]%s' % authKeyLenRegex,
      helpname='WORD',
      helpdesc='Authentication string (up to %d characters)' % authKeyMaxLen )
ietfMd5KeyMatcher = CliMatcher.PatternMatcher(
      r'.%s' % authKeyLenRegex,
      helpname='WORD',
      helpdesc='IETF-MD5 key (up to %d characters)' % authKeyMaxLen )

def generateVrrpEncryptionKey( mode, args ):
   return FhrpUtils.getVrrpEncryptionKey( mode.intf.name )

class VrrpPeerAuthenticationCmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp VRID ( ( peer authentication ) | authenticationDeprecated ) ' \
            '( [ text ] AUTH_PASS ) | ( ietf-md5 key-string AUTH_IPAH )'
   noOrDefaultSyntax = 'vrrp VRID ( ( peer authentication ) | ' \
                       'authenticationDeprecated ) ...'

   data = {
      'vrrp': nodeVrrp,
      'VRID': matcherVrId,
      'peer': 'VRRP Peer Information',
      'authentication': 'Use authentication (VRRPv2 only)',
      'authenticationDeprecated': CliCommand.Node( KeywordMatcher( 'authentication',
         helpdesc='Use authentication (VRRPv2 only)' ),
         deprecatedByCmd='vrrp peer authentication' ),
      'text': 'Plain text authentication',
      'AUTH_PASS': ReversibleSecretCli.ReversiblePasswordCliExpression(
                     cleartextMatcher=authTextStringMatcher,
                     obfuscatedTextMatcher=ReversibleSecretCli.type7AuthMatcher,
                     uniqueKeyGenerator=generateVrrpEncryptionKey,
                     algorithm='DES' ),
      'ietf-md5': 'Use IP authentication header with MD5 authentication',
      'key-string': 'Authentication key',
      'AUTH_IPAH': ReversibleSecretCli.ReversiblePasswordCliExpression(
                     cleartextMatcher=ietfMd5KeyMatcher,
                     obfuscatedTextMatcher=ReversibleSecretCli.type7AuthMatcher,
                     uniqueKeyGenerator=generateVrrpEncryptionKey,
                     algorithm='DES' ),
   }

   @staticmethod
   def handler( mode, args ):
      vrId = args[ 'VRID' ]
      if 'ietf-md5' in args:
         authType = authTypeEnum.vrrpAuthIpAh
         password = args[ 'AUTH_IPAH' ]
      else:
         authType = authTypeEnum.vrrpAuthPasswd
         password = args[ 'AUTH_PASS' ]
      setVrrpAuthenticationKey( mode, vrId, authType, password )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrId = args[ 'VRID' ]
      noVrrpAuthentication( mode, vrId )

modelet.addCommandClass( VrrpPeerAuthenticationCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] vrrp ipv6 mac range ipv4" in config mode
#--------------------------------------------------------------------------------

def setForceIpv4MacPrefix( mode, args ):
   _vrrpConfig().ipv6ForceIpv4MacPrefix = True

def noForceIpv4MacPrefix( mode, args ):
   vrrpConfig = _vrrpConfig()
   vrrpConfig.ipv6ForceIpv4MacPrefix = vrrpConfig.ipv6ForceIpv4MacPrefixDefault

class VrrpIpv6MacRangeIpv4Cmd( CliCommand.CliCommandClass ):
   syntax = 'vrrp ipv6 mac range ipv4'
   noOrDefaultSyntax = syntax
   data = {
      'vrrp': CliCommand.Node( CliMatcher.KeywordMatcher(
              'vrrp', helpdesc='Virtual Router Redundancy Protocol (VRRP)' ),
              guard=vrrpSupportedGuard ),
      'ipv6': CliCommand.Node( CliMatcher.KeywordMatcher( 'ipv6',
              helpdesc='VRRP IPv6 configuration' ),
              guard=forceIpv4MacPrefixGuard ),
      'mac': 'VRRP MAC address configuration',
      'range': 'VRRP MAC address range configuration',
      'ipv4': 'VRRP IPv4 MAC prefix for IPv6 virtual routers',
   }

   handler = setForceIpv4MacPrefix
   noOrDefaultHandler = noForceIpv4MacPrefix

GlobalConfigMode.addCommandClass( VrrpIpv6MacRangeIpv4Cmd )

# VARP Commands
#-------------------------------------------------------------------------------
# VARP Modelet
#-------------------------------------------------------------------------------

class VarpIntfConfigModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return FhrpUtils.intfSupportsVarp( mode.intf.name )

IntfCli.IntfConfigMode.addModelet( VarpIntfConfigModelet )

#--------------------------------------------------------------------------------
# "[ no | default ] ip virtual-router mac-address MACADDR [ mask MASK ]"
# in config mode
#--------------------------------------------------------------------------------

def canSetSystemMacAddr( systemMac ):
   varpVirtualMac = Arnet.EthAddr( fhrpVirtualMacStatus.varpVirtualMac )
   if ( str( varpVirtualMac ) != Tac.Type( "Arnet::EthAddr" ).ethAddrZero and
        Arnet.EthAddr( systemMac ) == varpVirtualMac ):
      message = 'System MAC address cannot be same as the '\
                'virtual router MAC address'
      return ( False, message )
   return ( True, None )

def setVarpMacAddr( mode, args ):
   macAddr = args[ 'MACADDR' ]
   config = _varpConfig()
   mask = args.get( 'MASK', config.virtualMacMaskDefault )

   eth = Arnet.EthAddr( macAddr )
   if eth.isMulticast or eth.isIp6Multicast():
      mode.addError( 'Virtual router MAC address cannot be assigned as '
                     'a multicast MAC address' )
      return

   if isVrrpMac( macAddr ):
      if ( mode.session.guardsEnabled() and
           not fhrpHardwareStatus.vrrpMacAsVarpMacSupported ):
         mode.addError( 'Virtual router MAC address assigned as a VRRP MAC address '
                        'not supported on this platform' )
         return
      mode.addWarning( 'Virtual router MAC address assigned as a VRRP MAC address '
                       'may not work if VRRP is enabled' )

   if systemMacConflict( macAddr ):
      mode.addError( 'System MAC address cannot be configured as the Virtual '
                     'router MAC address' )
      return

   vmacWithMask = EthAddrWithMask( macAddr, mask )
   config.virtualMacWithMask = vmacWithMask
   evaluateFhrpRunnability()

def noVarpMacAddr( mode, args ):
   macAddr = args[ 'MACADDR' ]
   config = _varpConfig()
   mask = args.get( 'MASK', config.virtualMacMaskDefault )

   vmacWithMask = EthAddrWithMask( macAddr, mask )
   errorMsgFmt = 'MAC address could not be removed: %s'
   if vmacWithMask == config.virtualMacWithMask:
      config.virtualMacWithMask = config.virtualMacWithMaskDefault
   elif mask == config.virtualMacWithMask.mask:
      mode.addWarning( errorMsgFmt % 'Address not found' )
   elif macAddr == config.virtualMacWithMask.mac:
      mode.addWarning( errorMsgFmt % 'Mask not found' )
   else:
      mode.addWarning( errorMsgFmt % 'Address and mask not found' )
   evaluateFhrpRunnability()

class VarpMacAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'ip virtual-router mac-address MACADDR [ mask MASK ]'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipMatcherForConfig,
      'virtual-router': nodeVirtualRouter,
      'mac-address': 'Virtual router MAC address',
      'MACADDR': MacAddrMatcher( helpdesc='Ethernet address' ),
      'mask': CliCommand.Node( CliMatcher.KeywordMatcher( 'mask',
         helpdesc='MAC mask' ), guard=varpMacMaskSupportedGuard ),
      'MASK': MacAddrMatcher( helpdesc='mask' ),
   }

   handler = setVarpMacAddr
   noOrDefaultHandler = noVarpMacAddr

GlobalConfigMode.addCommandClass( VarpMacAddressCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] ip virtual-router mac-address advertisement-interval INTERVAL"
# in config mode
#--------------------------------------------------------------------------------

def setVarpMacAdvertise( mode, args ):
   interval = args[ 'INTERVAL' ]
   config = _varpConfig()
   config.advtInterval = interval

def noVarpMacAdvertise( mode, args ):
   config = _varpConfig()
   config.advtInterval = config.advtIntervalDefault

class VarpMacAdvertisementCmd( CliCommand.CliCommandClass ):
   syntax = 'ip virtual-router mac-address advertisement-interval INTERVAL'
   noOrDefaultSyntax = 'ip virtual-router mac-address ' \
                       'advertisement-interval ...'
   data = {
      'ip': ipMatcherForConfig,
      'virtual-router': nodeVirtualRouter,
      'mac-address': matcherMacAddress,
      'advertisement-interval': matcherAdvertisementInterval,
      'INTERVAL': CliMatcher.IntegerMatcher( 0, 86400, helpdesc='Seconds' ),
   }

   handler = setVarpMacAdvertise
   noOrDefaultHandler = noVarpMacAdvertise

GlobalConfigMode.addCommandClass( VarpMacAdvertisementCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] ip virtual-router address subnet-routes"
# in config mode
#--------------------------------------------------------------------------------

def setVarpSubnetRoutes( mode, args ):
   config = _varpConfig()
   config.subnetRoutes = True

def noVarpSubnetRoutes( mode, args ):
   config = _varpConfig()
   config.subnetRoutes = False

class VarpAddressSubnetRoutesCmd( CliCommand.CliCommandClass ):
   syntax = 'ip virtual-router address subnet-routes'
   noOrDefaultSyntax = syntax

   data = {
      'ip': ipMatcherForConfig,
      'virtual-router': nodeVirtualRouter,
      'address': 'Virtual router address',
      'subnet-routes': 'Enable connected routes for configured virtual-router subnet'
   }

   handler = setVarpSubnetRoutes
   noOrDefaultHandler = noVarpSubnetRoutes

GlobalConfigMode.addCommandClass( VarpAddressSubnetRoutesCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] ip virtual-router address ( IPADDR | IPPREFIX )"
# in config-if mode
#--------------------------------------------------------------------------------

def setVarpIpAddress( mode, args ):
   ipAddr = args.get( 'IPADDR', args.get( 'IPPREFIX' ) )
   if '/' not in ipAddr:
      maxLen = _fhrpConfig().defaultIpMaskLen
      ipAddr = '%s/%d' % ( ipAddr, maxLen )
   ipAddrWithMask = Arnet.AddrWithMask( ipAddr )
   ipAddr = ipAddrWithMask.address
   intfName = mode.intf.name
   vrf = getVrfName( intfName )
   vrfIp = vrfIpPair( vrf, ipAddr )

   # Zeroed ip
   if ipAddr == ipAddrZero.stringValue:
      return

   if not mode.session.skipConfigCheck():
      unique, msg = uniqueFhrpIpAssigned( vrfIp, intfName, VARP_VRID )
      if not unique:
         mode.addError( msg )
         return

      if not Toggles.FhrpToggleLib.toggleVrrpSamePhysicalIpAsVipEnabled():
         unique, msg = uniqueIntfIpAssigned( ipAddr, intfName, allowOverlap=True )
         if unique:
            if msg:
               mode.addWarning( msg )
         else:
            mode.addError( msg )
            return

      overlap, msg = overlappingVarpIpAssigned( ipAddrWithMask, intfName )
      if overlap:
         mode.addWarning( msg )

      overlap = ipSubnetOverlappingIntfSubnet( ipAddrWithMask, intfName )
      if overlap:
         overlapWarning = ( 'Virtual router subnets overlapping an interface subnet '
                            'are not supported' )
         mode.addWarning( overlapWarning )

   t0( 'Enabling Virtual Router IP address', ipAddr, 'on', mode.intf.name )
   config = _varpIntfConfig( mode )

   # VIPs that exist should do nothing
   if ipAddrWithMask not in config.virtualIpAddr:
      # Over the limit needs to block
      if len( config.virtualIpAddr ) == MAX_VARP_IPV4_PER_INTERFACE:
         mode.addError( 'Maximum number of IPv4 virtual addresses (%d) per '
                        'interface assigned' % MAX_VARP_IPV4_PER_INTERFACE )
      else:
         varpConfig = _varpConfig()

         intfCol = varpConfig.virtualIpAddr.get( vrfIp )
         if not intfCol:
            intfCol = varpConfig.virtualIpAddr.newMember( vrfIp )
         if intfName in intfCol.intfIdToMask:
            # Delete old vip with mask
            oldMask = intfCol.intfIdToMask[ intfName ]
            oldIpWithMask = Arnet.AddrWithMask( '%s/%s' % ( ipAddr, oldMask ) )
            del config.virtualIpAddr[ oldIpWithMask ]

         intfCol.intfIdToMask[ intfName ] = ipAddrWithMask.len
         config.virtualIpAddr[ ipAddrWithMask ] = True
   deleteVarpConfigIfDefault( mode.intf.name )
   evaluateFhrpRunnability()

def noVarpIpAddress( mode, args ):
   ipAddr = args.get( 'IPADDR', args.get( 'IPPREFIX' ) )
   if ipAddr:
      t0( 'Disabling Virtual Router IP address', ipAddr, 'on', mode.intf.name )
   else:
      t0( 'Disabling all Virtual Router IP addresses on', mode.intf.name )

   config = _varpIntfConfig( mode, createIfMissing=False )
   if config is None:
      return

   if ipAddr:
      if '/' not in ipAddr:
         maxLen = _fhrpConfig().defaultIpMaskLen
         ipAddr = '%s/%d' % ( ipAddr, maxLen )
      ipAddrWithMask = Arnet.AddrWithMask( ipAddr )
      ipAddr = ipAddrWithMask.address

      if ipAddrWithMask in config.virtualIpAddr:
         deleteVarpGlobalIp( ipAddrWithMask, mode.intf.name )
         del config.virtualIpAddr[ ipAddrWithMask ]
      elif ipAddr != ipAddrZero.stringValue:
         mode.addWarning( 'Virtual address %s was not found for deletion'
                          % ipAddrWithMask )
   else:
      for addr in config.virtualIpAddr:
         deleteVarpGlobalIp( addr, mode.intf.name )
      config.virtualIpAddr.clear()

   deleteVarpConfigIfDefault( mode.intf.name )
   evaluateFhrpRunnability()

class VarpIpAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'ip virtual-router address ( IPADDR | IPPREFIX )'
   noOrDefaultSyntax = 'ip virtual-router address [ IPADDR | IPPREFIX ]'
   data = {
      'ip': ipMatcherForConfigIf,
      'virtual-router': nodeVirtualRouter,
      'address': 'Virtual router address',
      'IPADDR': IpAddrMatcher.ipAddrMatcher,
      'IPPREFIX': IpAddrMatcher.ipPrefixMatcher
   }

   handler = setVarpIpAddress
   noOrDefaultHandler = noVarpIpAddress

VarpIntfConfigModelet.addCommandClass( VarpIpAddressCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] ip virtual-router track OBJECTNAME shutdown" in config-if mode
#--------------------------------------------------------------------------------

def setVarpTrackShutdown( mode, args ):
   objectName = args[ 'OBJECTNAME' ]
   config = _varpIntfConfig( mode )
   if config is None:
      return

   if objectName not in config.varpTrackedObject:
      config.varpTrackedObject.newMember( objectName )
   trackedObject = config.varpTrackedObject[ objectName ]
   trackedObject.trackActionIsShutdown = True
   deleteVarpConfigIfDefault( mode.intf.name )
   evaluateFhrpRunnability()

def noVarpTrackShutdown( mode, args ):
   objectName = args[ 'OBJECTNAME' ]
   config = _varpIntfConfig( mode, createIfMissing=False )
   if config is None:
      return

   if objectName not in config.varpTrackedObject:
      return
   del config.varpTrackedObject[ objectName ]
   deleteVarpConfigIfDefault( mode.intf.name )
   evaluateFhrpRunnability()

class VarpTrackShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'ip virtual-router track OBJECTNAME shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipMatcherForConfigIf,
      'virtual-router': nodeVirtualRouter,
      'track': 'Track an object',
      'OBJECTNAME': matcherObjName,
      'shutdown': 'Disable virtual router configuration',
   }

   handler = setVarpTrackShutdown
   noOrDefaultHandler = noVarpTrackShutdown

VarpIntfConfigModelet.addCommandClass( VarpTrackShutdownCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] ip address virtual source-nat [ vrf VRF ] address IPADDR"
# in config mode
#--------------------------------------------------------------------------------
def setSourceNatForVirtualIp( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   ipAddr = args[ 'IPADDR' ]

   if not vrfExists( vrfName ):
      mode.addWarning( "VRF %s does not exist. Assigning anyway." % vrfName )

   varpConfig = _varpConfig()
   varpConfig.borrowedIpAddr[ vrfName ] = ipAddr

def noSourceNatForVirtualIp( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   varpConfig = _varpConfig()
   if vrfName in varpConfig.borrowedIpAddr:
      del varpConfig.borrowedIpAddr[ vrfName ]

class IpAddressVirtualSourceNatAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'ip address virtual source-nat [ VRF ] sourceNatAddress IPADDR'
   noOrDefaultSyntax = syntax.replace( 'IPADDR', '...' )
   data = {
      'ip': ipMatcherForConfig,
      'address': matcherAddress,
      'virtual': matcherVirtual,
      'source-nat': matcherSourceNat,
      'VRF': VrfExprFactory( helpdesc='Configure source NAT IP address in a VRF' ),
      'sourceNatAddress': matcherSourceNatAddr,
      'IPADDR': IpAddrMatcher.ipAddrMatcher,
   }

   handler = setSourceNatForVirtualIp
   noOrDefaultHandler = noSourceNatForVirtualIp

GlobalConfigMode.addCommandClass( IpAddressVirtualSourceNatAddressCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] ipv6 address virtual source-nat [ vrf VRF ] address IP6ADDR"
# in config mode
#--------------------------------------------------------------------------------
def setSourceNatForVirtualIp6( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   ip6Addr = args[ 'IP6ADDR' ]

   if not vrfExists( vrfName ):
      mode.addWarning( "VRF %s does not exist. Assigning anyway." % vrfName )

   varpConfig = _varpConfig()
   varpConfig.borrowedIp6Addr[ vrfName ] = ip6Addr

def noSourceNatForVirtualIp6( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   varpConfig = _varpConfig()
   del varpConfig.borrowedIp6Addr[ vrfName ]

class Ipv6AddressVirtualSourceNatAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 address virtual source-nat [ VRF ] sourceNatAddress IP6ADDR'
   noOrDefaultSyntax = syntax.replace( 'IP6ADDR', '...' )
   data = {
      'ipv6': ipv6MatcherForConfig,
      'address': IraIp6IntfCli.matcherAddress,
      'virtual': matcherVirtual,
      'source-nat': matcherSourceNat,
      'VRF': VrfExprFactory( helpdesc='Configure source NAT IP address in a VRF' ),
      'sourceNatAddress': matcherSourceNatAddr,
      'IP6ADDR': Ip6AddrMatcher( helpdesc='IPv6 address' ),
   }

   handler = setSourceNatForVirtualIp6
   noOrDefaultHandler = noSourceNatForVirtualIp6

GlobalConfigMode.addCommandClass( Ipv6AddressVirtualSourceNatAddressCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] ipv6 virtual-router address IP6ADDR" in config-if mode
#--------------------------------------------------------------------------------

def setVarpIpv6Address( mode, args ):
   ip6Addr = args[ 'IP6ADDR' ]
   maskLen = 128
   intfName = mode.intf.name
   if ip6Addr.isUnspecified:
      return

   vrf = getVrfName( intfName )
   vrfIp = vrfIpPair( vrf, ip6Addr )

   if not mode.session.skipConfigCheck():
      if not ip6Addr.isLinkLocal:
         unique, msg = uniqueFhrpIpAssigned( vrfIp, intfName, VARP_VRID )
         if not unique:
            mode.addError( msg )
            return

         unique, msg = uniqueIntfIp6Assigned( ip6Addr, intfName, allowOverlap=True )
         if unique:
            if msg:
               mode.addWarning( msg )
         else:
            mode.addError( msg )
            return
      else:
         unique, msg = uniqueIntfLinkLocalAssigned( vrfIp, ip6Addr, intfName,
                                                    VARP_VRID )
         if not unique:
            mode.addError( msg )
            return

   config = _varpIntfConfig( mode )
   if ip6Addr not in config.virtualIp6Addr:
      t0( 'Enabling Virtual Router IP address', ip6Addr, 'on', intfName )
      if len( config.virtualIp6Addr ) == MAX_VARP_IPV6_PER_INTERFACE:
         mode.addError( 'Maximum number of IPv6 virtual addresses (%d) per '
                        'interface assigned' % MAX_VARP_IPV6_PER_INTERFACE )
      else:
         # remove original virtual ip in global collection if the new virtual ip has
         # the same address but a different subnet
         varpConfig = _varpConfig()
         intfCol = varpConfig.virtualIpAddr.get( vrfIp )
         if not intfCol:
            intfCol = varpConfig.virtualIpAddr.newMember( vrfIp )
         intfCol.intfIdToMask[ intfName ] = maskLen
         config.virtualIp6Addr[ ip6Addr ] = True
   deleteVarpConfigIfDefault( mode.intf.name )
   evaluateFhrpRunnability()

def noVarpIpv6Address( mode, args ):
   ip6Addr = args.get( 'IP6ADDR' )
   config = _varpIntfConfig( mode, createIfMissing=False )
   if config is None:
      return

   if ip6Addr:
      t0( 'Disabling Virtual Router IP address', ip6Addr, 'on',
          mode.intf.name )
      if ip6Addr in config.virtualIp6Addr:
         deleteVarpGlobalIp( ip6Addr, mode.intf.name )
         del config.virtualIp6Addr[ ip6Addr ]
      elif not ip6Addr.isUnspecified:
         mode.addWarning( 'Virtual address %s was not found for deletion'
                          % ip6Addr )
   else:
      t0( 'Disabling all Virtual Router IP addresses on', mode.intf.name )
      for ip6 in config.virtualIp6Addr:
         deleteVarpGlobalIp( ip6, mode.intf.name )
      config.virtualIp6Addr.clear()
   deleteVarpConfigIfDefault( mode.intf.name )
   evaluateFhrpRunnability()

class VarpIpv6AddressCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 virtual-router address IP6ADDR'
   noOrDefaultSyntax = 'ipv6 virtual-router address [ IP6ADDR ]'
   data = {
      'ipv6': ipv6MatcherForConfigIf,
      'virtual-router': nodeVirtualRouterV6,
      'address': 'Virtual router address',
      'IP6ADDR': Ip6AddrMatcher( helpdesc='IPv6 address' ),
   }

   handler = setVarpIpv6Address
   noOrDefaultHandler = noVarpIpv6Address

VarpIntfConfigModelet.addCommandClass( VarpIpv6AddressCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] ip address virtual ADDRWITHMASK" in config-if mode
#-------------------------------------------------------------------------------

class IpVirtIntfConfigModelet( VlanIntfCli.VlanIntfModelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( VlanIntfCli.VlanIntfModelet.shouldAddModeletRule( mode ) or
               mode.intf.name.startswith( 'Test' ) )

IntfCli.IntfConfigMode.addModelet( IpVirtIntfConfigModelet )

def isValidIntfAddrWithMask( addrWithMask, intfName ):
   arnetSubnet = Arnet.Subnet( addrWithMask )
   addr = addrWithMask.address
   maxLen = _fhrpConfig().defaultIpMaskLen
   if addrWithMask.len >= maxLen:
      return ( False, "Prefix length must be less than 32" )
   if IpAddrMatcher.validateMulticastIpAddr( addr ) is None:
      return ( False, "IP address must be unicast" )
   if arnetSubnet.isBroadcastAddress():
      return ( False, "IP address must not be broadcast" )
   if arnetSubnet.isAllZerosAddress():
      return ( False, "IP address must not have a zero host number" )
   if IpAddrMatcher.isMartianIpAddr( addr ):
      return ( False, "Not a valid host address" )

   primaryMatch = 'allowNoMatch'
   secondaryMatch = 'allowNoMatch'
   ipConfigEntity = ConfigMount.force( ipConfig )
   query = Tac.newInstance( 'Ira::AddressQueryParams', ipConfigEntity, intfName,
                            'Management', primaryMatch, secondaryMatch,
                            addrWithMask )
   checker = Tac.Value( 'Ira::AddressAssignmentChecker' )
   ans = checker.check( query )
   if ans.acceptable:
      return ( True, "" )
   return ( False, ans.msg )

def setIpAddrVirtual( mode, args ):
   addressWithMask = args[ 'ADDRWITHMASK' ]
   intfName = mode.intf.name

   ipIntf = ipConfig.ipIntfConfig.get( intfName )
   if ipIntf and addressWithMask == ipIntf.virtualAddrWithMask:
      # nothing to do
      return

   if not mode.session.skipConfigCheck():
      ( valid, errMsg ) = isValidIntfAddrWithMask( addressWithMask, intfName )
      if not valid:
         mode.addError( errMsg )
         return

   if ipIntf is None:
      ipIntf = ipConfig.ipIntfConfig.newMember( intfName )
      ipIntf.l3Config = l3ConfigDir.newIntfConfig( intfName )

   if not mode.session.skipConfigCheck():
      if ipIntf.addrWithMask != Arnet.AddrWithMask( "0.0.0.0/0" ):
         mode.addError( 'Non-virtual address %s is already configured. '
                        'Mixing virtual and non-virtual addresses on an interface '
                        'is not supported.' % ipIntf.addrWithMask )
         return

      if addressWithMask == ipIntf.virtualAddrWithMask:
         # nothing to do
         return

      # a secondary address might be promoted
      if ( addressWithMask and
           addressWithMask in ipIntf.secondaryWithMask ):
         # for a short time the same address is both primary and secondary!
         ipIntf.virtualAddrWithMask = addressWithMask
         del ipIntf.virtualSecondaryWithMask[ addressWithMask ]
         return

   ipIntf.virtualAddrWithMask = addressWithMask

def noIpAddrVirtual( mode, args ):
   addressWithMask = args.get( 'ADDRWITHMASK' )
   intfName = mode.intf.name
   ipIntf = ipConfig.ipIntfConfig.get( intfName )
   if ipIntf:
      if addressWithMask:
         zap = addressWithMask
         pri = ipIntf.virtualAddrWithMask
         if zap != pri:
            mode.addError( "Address %s/%s does not match primary virtual address "
                           "%s/%s" %
                           ( zap.address, zap.len, pri.address, pri.len ) )
            return

         if ipIntf.virtualSecondaryWithMask:
            mode.addError( "Primary virtual address cannot be deleted before "
                           "secondary" )
         else:
            ipIntf.virtualAddrWithMask = Arnet.AddrWithMask( "0.0.0.0/0" )
      else:
         ipIntf.virtualSecondaryWithMask.clear()
         ipIntf.virtualAddrWithMask = Arnet.AddrWithMask( "0.0.0.0/0" )

class IpAddrVirtualCmd( CliCommand.CliCommandClass ):
   syntax = 'ip address virtual ADDRWITHMASK'
   noOrDefaultSyntax = 'ip address virtual [ ADDRWITHMASK ]'
   data = {
      'ip': ipMatcherForConfigIf,
      'address': matcherAddress,
      'virtual': matcherVirtual,
      'ADDRWITHMASK': matcherIpWithMaskExpr,
   }

   handler = setIpAddrVirtual
   noOrDefaultHandler = noIpAddrVirtual

IpVirtIntfConfigModelet.addCommandClass( IpAddrVirtualCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] ip address virtual ADDRWITHMASK secondary" in config-if mode
#-------------------------------------------------------------------------------

def setSecondaryIpAddrVirtual( mode, args ):
   addressWithMask = args[ 'ADDRWITHMASK' ]
   intfName = mode.intf.name

   ipIntf = ipConfig.ipIntfConfig.get( intfName )
   if ipIntf is None:
      ipIntf = ipConfig.ipIntfConfig.newMember( intfName )
      ipIntf.l3Config = l3ConfigDir.newIntfConfig( intfName )

   if not mode.session.skipConfigCheck():
      pri = ipIntf.virtualAddrWithMask
      if pri == Arnet.AddrWithMask( "0.0.0.0/0" ):
         mode.addError( "Primary virtual address must be assigned first" )
         return
      if pri.address == addressWithMask.address:
         mode.addError( "Address %s is assigned as primary" %
                        ( pri.address, ) )
         return

      if addressWithMask in ipIntf.virtualSecondaryWithMask:
         # nothing to do
         return

      if ( len( ipIntf.virtualSecondaryWithMask ) >=
           MAX_SECONDARIES_PER_INTERFACE ):
         mode.addError( "This interface already has %d secondary virtual "
                        "addresses assigned" % (
               MAX_SECONDARIES_PER_INTERFACE, ) )
         return

      ( valid, errMsg ) = isValidIntfAddrWithMask( addressWithMask, intfName )
      if not valid:
         mode.addError( errMsg )
         return

   ipIntf.virtualSecondaryWithMask[ addressWithMask ] = True

def noSecondaryIpAddrVirtual( mode, args ):
   addressWithMask = args[ 'ADDRWITHMASK' ]
   intfName = mode.intf.name
   ipIntf = ipConfig.ipIntfConfig.get( intfName )

   if ipIntf:
      del ipIntf.virtualSecondaryWithMask[ addressWithMask ]

class IpAddrVirtualSecondaryCmd( CliCommand.CliCommandClass ):
   syntax = 'ip address virtual ADDRWITHMASK secondary'
   noOrDefaultSyntax = 'ip address virtual ADDRWITHMASK secondary'
   data = {
      'ip': ipMatcherForConfigIf,
      'address': matcherAddress,
      'virtual': matcherVirtual,
      'ADDRWITHMASK': matcherIpWithMaskExpr,
      'secondary': 'Specify that address is secondary',
   }

   handler = setSecondaryIpAddrVirtual
   noOrDefaultHandler = noSecondaryIpAddrVirtual

IpVirtIntfConfigModelet.addCommandClass( IpAddrVirtualSecondaryCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] ipv6 address virtual IP6PREFIX" in config-if mode
#-------------------------------------------------------------------------------

def isValidIntfAddr6WithMask( addrWithMask, intfName, mode ):
   # Checks taken from IntfCli.Intf.isValidHostInet6Address()
   if addrWithMask.len >= 128:
      return ( False, "Prefix length must be less than 128" )

   addr = addrWithMask.address
   if addr.isMulticast:
      return ( False, "Address must be unicast" )
   if addr.isUnspecified:
      return ( False, "Invalid unicast address" )
   if addr.isLoopback:
      return ( False, "Invalid address for this interface" )

   mgmtPrefix = "Management"
   query = Tac.newInstance( "Ira::Address6QueryParams", ip6Config, intfName,
                            mgmtPrefix, addrWithMask )
   checker = Tac.Value( 'Ira::Address6AssignmentChecker' )
   ans = checker.check( query )
   if ans.acceptable:
      if ans.msg:
         mode.addWarning( ans.msg )
   else:
      return ( False, ans.msg )

   return ( True, "" )

def setIpv6AddrVirtual( mode, args ):
   addressWithMask = args[ 'IP6PREFIX' ]
   intfName = mode.intf.name
   ipIntf = ip6Config.intf.get( intfName )
   addrInfo = Tac.Value( "Ip6::AddressInfo", key=addressWithMask )

   if not mode.session.skipConfigCheck():
      ( valid, errMsg ) = isValidIntfAddr6WithMask( addressWithMask, intfName, mode )
      if not valid:
         mode.addError( errMsg )
         return

      if ipIntf:
         if ipIntf.addr:
            mode.addError( 'At least 1 non-virtual address is already configured. '
                           'Mixing virtual and non-virtual addresses on an '
                           'interface is not supported.' )
            return
         if addressWithMask in ipIntf.virtualAddr:
            # nothing to do
            return
   if not ipIntf:
      ipIntf = ip6Config.intf.newMember( intfName )
      ipIntf.l3Config = l3ConfigDir.newIntfConfig( intfName )

   ipIntf.virtualAddr.addMember( addrInfo )

def noIpv6AddrVirtual( mode, args ):
   addressWithMask = args.get( 'IP6PREFIX' )
   intfName = mode.intf.name
   ipIntf = ip6Config.intf.get( intfName )

   if not ipIntf:
      return
   if not addressWithMask:
      ipIntf.virtualAddr.clear()
      return
   if addressWithMask not in ipIntf.virtualAddr:
      mode.addError( "Virtual address not configured on interface" )
      return
   del ipIntf.virtualAddr[ addressWithMask ]

class Ipv6AddrVirtualCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 address virtual IP6PREFIX'
   noOrDefaultSyntax = 'ipv6 address virtual [ IP6PREFIX ]'
   data = {
      'ipv6': ipv6MatcherForConfigIf,
      'address': IraIp6IntfCli.matcherAddress,
      'virtual': matcherVirtual,
      'IP6PREFIX': ip6PrefixMatcher,
   }

   handler = setIpv6AddrVirtual
   noOrDefaultHandler = noIpv6AddrVirtual

IpVirtIntfConfigModelet.addCommandClass( Ipv6AddrVirtualCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] mac address virtual-router" in config-if mode
#-------------------------------------------------------------------------------

def setUseVirtualMac( mode, args ):
   setVal = not CliCommand.isNoCmd( args )
   config = _varpIntfConfig( mode, createIfMissing=setVal )
   if config:
      config.useVirtualMac = setVal
      if not setVal:
         deleteVarpConfigIfDefault( mode.intf.name )
   evaluateFhrpRunnability()

class MacAddrVirtualRouterCmd( CliCommand.CliCommandClass ):
   syntax = 'mac address virtual-router'
   noOrDefaultSyntax = syntax
   data = {
      'mac': macMatcherForConfigIf,
      'address': 'Configure MAC address',
      'virtual-router': CliCommand.Node( matcherVarpForShow,
                                         guard=varpSupportedGuard ),
   }

   handler = setUseVirtualMac
   noOrDefaultHandler = setUseVirtualMac

IpVirtIntfConfigModelet.addCommandClass( MacAddrVirtualRouterCmd )

# Show VRRP commands

def parseArgs( args ):
   vrfName = args.get( 'VRF' )
   interface = args.get( 'INTF' )
   vrId = args.get( 'VRID' )
   allVrrp = 'all' in args

   if interface:
      intfList = [ interface.name ]
   else:
      vrfName = vrfName if vrfName else DEFAULT_VRF
      vrrpIntfs = list( _fhrpConfig().vrrpConfig.vrrpIntfConfig )
      intfList = []
      for intf in vrrpIntfs:
         if vrfName == ALL_VRF_NAME or vrfName == getVrfName( intf ):
            intfList.append( intf )
   return ( intfList, vrId, allVrrp )

def showVrrpBrief( mode, args ):
   intfList, vrId, allVrrp = parseArgs( args )
   return showBrief( intfList, vrId, allVrrp )

def showVrrp( mode, args ):
   intfList, vrId, allVrrp = parseArgs( args )
   return showDetail( intfList, vrId, allVrrp )

def showBrief( intfList, vrId=None, allVrrp=False, ):
   vrs = []
   for intfId in Arnet.sortIntf( intfList ):
      vrs += _showBrief( intfId, vrId, allVrrp, _fhrpStatusV4() )
      vrs += _showBrief( intfId, vrId, allVrrp, _fhrpStatusV6() )
   return VrrpBriefModel( virtualRouters=vrs )

def _showBrief( intfId, vrId, allVrrp, fhrpStatus ):
   vrIntfStatus = fhrpStatus.vrIntfStatus.get( intfId )
   intfConfig = _vrrpConfig().vrrpIntfConfig.get( intfId )
   vrs = []
   if vrIntfStatus and intfConfig:
      vrList = []
      if vrId:
         if vrId in vrIntfStatus.vrStatus:
            vrList = [ vrId ]
      else:
         vrList = sorted( vrIntfStatus.vrStatus.keys() )
      for vid in vrList:
         if vid not in intfConfig.virtualRouterConfig:
            continue
         config = intfConfig.virtualRouterConfig[ vid ]
         status = vrIntfStatus.vrStatus[ vid ]
         if ( not vrId and not allVrrp and status.state not in
              ( fhrpStateEnum.fhrpBackup, fhrpStateEnum.fhrpActive ) ):
            continue
         vr = showBriefVrId( config, status, intfId )
         if vr:
            vrs.append( vr )
   return vrs

def showDetail( intfList, vrId=None, allVrrp=False ):
   vrs = []
   for intfId in Arnet.sortIntf( intfList ):
      vrs += _showDetail( intfId, vrId, allVrrp, _fhrpStatusV4() )
      vrs += _showDetail( intfId, vrId, allVrrp, _fhrpStatusV6() )
   return VrrpDetailModel( virtualRouters=vrs )

def _showDetail( intfId, vrId, allVrrp, fhrpStatus ):
   vrIntfStatus = fhrpStatus.vrIntfStatus.get( intfId )
   intfConfig = _vrrpConfig().vrrpIntfConfig.get( intfId )
   vrs = []
   if vrIntfStatus and intfConfig:
      vrList = []
      if vrId:
         if vrId in vrIntfStatus.vrStatus:
            vrList = [ vrId ]
      else:
         vrList = sorted( vrIntfStatus.vrStatus.keys() )
      for vid in vrList:
         config = intfConfig.virtualRouterConfig.get( vid )
         if config is None:
            continue
         status = vrIntfStatus.vrStatus[ vid ]
         if ( not vrId and not allVrrp and status.state not in
              ( fhrpStateEnum.fhrpBackup, fhrpStateEnum.fhrpActive ) ):
            continue
         vr = showDetailVrId( config, status, intfId )
         if vr:
            vrs.append( vr )
   return vrs

def showPriority( config, status ):
   if status.priority != 0:
      return status.priority
   return config.priority

def showPrimaryAddr( config, status ):
   if status.primaryAddr != '0.0.0.0':
      return status.primaryAddr
   return config.primaryAddr

def showIpv4Addrs( config, status ):
   if status.ipAddrs:
      return sorted( status.ipAddrs.keys(),
                     key=functools.cmp_to_key( IpUtils.compareIpAddress ) )
   return sorted( config.secondaryAddrs.keys(),
                  key=functools.cmp_to_key( IpUtils.compareIpAddress ) )

def showIpv6Addrs( config, status ):
   if status.ip6Addrs:
      return sorted( status.ip6Addrs.keys(),
                     key=functools.cmp_to_key( IpUtils.compareIp6Address ) )
   return sorted( config.ip6Addrs.keys(),
                  key=functools.cmp_to_key( IpUtils.compareIp6Address ) )

def showBriefVrId( config, status, intfName ):
   if 'ipAddrs' in dir( status ):
      ipAddr = showPrimaryAddr( config, status )
      version = config.version
      vr = VrrpBriefModel.VrrpRouterBriefV4( virtualIp=ipAddr )
   else:
      ipAddrs = showIpv6Addrs( config, status )
      if not ipAddrs:
         ipAddrs = [ ip6AddrZero ]
      version = 3
      vr = VrrpBriefModel.VrrpRouterBriefV6( virtualIps=ipAddrs )

   vr.interface = intfName
   vr.vrfName = getVrfName( intfName )
   vr.groupId = status.vrId
   vr.version = version
   vr.priority = showPriority( config, status )
   # get a masterDownInterval in milliseconds for CLI by multiply 10, since
   # the masterDownInterval in status is in centisecons.
   vr.masterDownInterval = status.masterDownInterval * 10
   vr.state = fhrpStateEnumString( status.state )
   vr.stateTransitionTime = status.lastStateTransitionTime
   return vr

def showDetailVrId( config, status, intfName ):
   vrId = status.vrId
   description = config.description

   # in rare case when the interface is shutdown, fhrp might still be in the
   # active state, but ip6IntfStatus/ipIntfStatus might be gone already
   address = None
   if 'ip6Addrs' in dir( status ):
      addrFamily = AF_INET6
      intf = ip6Status.intf.get( intfName )
      if intf:
         address = intf.linkLocalAddrWithMask().address
   else:
      addrFamily = AF_INET
      intf = ipStatus.ipIntfStatus.get( intfName )
      if intf:
         address = intf.activeAddrWithMask.address

   actualState = status.state
   if address is None:
      actualState = fhrpStateEnum.fhrpStopped
   state = fhrpStateEnumString( actualState )

   virtualMac = status.virtualMac

   macAdvtInterval = config.arpOrNAInterval // 100

   advtInterval = config.advInterval // 100
   preempt = config.preempt
   delay = config.preemptDelay // 100
   reloadDelay = config.preemptReloadDelay // 100
   priority = showPriority( config, status )

   if actualState == fhrpStateEnum.fhrpActive:
      mprio = priority
      maddr = address
   elif actualState == fhrpStateEnum.fhrpBackup:
      mprio = status.masterPriorityReceived
      maddr = status.srcAddr
   else:
      mprio = 0
      maddr = status.srcAddrDefault

   masterInterval = status.masterAdverInterval // 100
   skewTime = status.skewTime / 100.0
   downInterval = status.masterDownInterval * 10

   if 'ip6Addrs' in dir( status ):
      addrs = showIpv6Addrs( config, status )
      if not addrs:
         addrs = [ ip6AddrZero ]
      if status.ip6LinkLocal in addrs:
         addrs.remove( status.ip6LinkLocal )
      version = 3
      bfdState = status.bfdState
      vr = VrrpDetailModel.VrrpRouterDetailV6( masterAddr=maddr, virtualIpv6=addrs,
                                               linkLocalAddr=status.ip6LinkLocal,
                                               bfdPeerAddr=config.bfdPeerAddrV6,
                                               bfdPeerState=bfdState )
   else:
      primaryAddr = showPrimaryAddr( config, status )
      addrs = showIpv4Addrs( config, status )
      addrs = [ addr for addr in addrs if addr != primaryAddr ]
      version = config.version
      bfdState = status.bfdState
      vr = VrrpDetailModel.VrrpRouterDetailV4( masterAddr=maddr,
                                               virtualIp=primaryAddr,
                                               virtualIpSecondary=addrs,
                                               bfdPeerAddr=config.bfdPeerAddrV4,
                                               bfdPeerState=bfdState )
      if version == 2:
         if config.authType != authTypeEnum.vrrpAuthNone:
            if config.authType == authTypeEnum.vrrpAuthPasswd:
               vr.authType = 'text'
            elif config.authType == authTypeEnum.vrrpAuthIpAh:
               vr.authType = 'md5'
            encryptionAlgo = ReversibleSecretCli.getEncryptionAlgo( securityConfig,
                                                                    'DES' )
            if encryptionAlgo == 'AES-256-GCM':
               vr.authSringEncrytType = '8a'
            else:
               vr.authSringEncrytType = '7'
            vr.authString = ReversibleSecretCli.encodePassword(
                  config.authKey, FhrpUtils.getVrrpEncryptionKey( intfName ),
                  encryptionAlgo, securityConfig )

   if version == 3 and advtInterval > MAX_VRRP_V3_ADVERTISE_INTERVAL:
      advtInterval = MAX_VRRP_V3_ADVERTISE_INTERVAL

   if vrId in fhrpHardwareStatus.vrrpUnprogrammedVrId:
      vr.vrIdDisabled = True
      vr.vrIdDisabledReason = getVrrpPlatformLimitations( vrId, addrFamily )
      if vr.vrIdDisabledReason == "":
         vr.vrIdDisabled = False
   else:
      vr.vrIdDisabled = False
      vr.vrIdDisabledReason = ""

   trackedObjects = []
   for trackedObject in config.vrrpTrackedObject:
      trackedObjectConfig = config.vrrpTrackedObject[ trackedObject ]
      if trackedObject not in trackingConfig.interfaceObjectConfig:
         continue
      interfaceObjectConfig = trackingConfig.interfaceObjectConfig[ trackedObject ]
      trackedInterface = interfaceObjectConfig.intfName
      trackedObjects.append( TrackedObject( trackedObject=trackedObject,
                             interface=trackedInterface,
                             shutdown=trackedObjectConfig.trackActionIsShutdown,
                             priority=trackedObjectConfig.trackPriorityDecrement ) )
   vr.interface = intfName
   vr.vrfName = getVrfName( intfName )
   vr.groupId = vrId
   vr.version = version
   vr.description = description
   vr.state = state
   vr.stateTransitionTime = status.lastStateTransitionTime
   vr.virtualMac = virtualMac
   vr.macAddressInterval = macAdvtInterval
   vr.vrrpAdvertInterval = advtInterval
   vr.preempt = preempt
   vr.preemptDelay = delay
   vr.preemptReload = reloadDelay
   vr.priority = priority
   vr.masterPriority = mprio
   vr.masterInterval = masterInterval
   vr.skewTime = skewTime
   vr.masterDownInterval = downInterval
   vr.trackedObjects = trackedObjects
   return vr

#--------------------------------------------------------------------------------
# "show vrrp [ { ( ( vrf VRF ) | ( interface INTF ) ) | ( group VRID ) } ]
# brief [ all ]" in enable mode
#--------------------------------------------------------------------------------

vrfIntfSharedObj = object()
vrfExprFactory = VrfExprFactory(
      helpdesc='VRF name',
      inclDefaultVrf=True,
      inclAllVrf=True,
      sharedMatchObj=vrfIntfSharedObj,
      maxMatches=1 )

class ShowVrrpBriefCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show vrrp
               [ { ( VRF )
                 | ( interface INTF )
                 | ( group VRID ) } ]
               brief [ all ]'''
   data = {
      'vrrp': nodeShowVrrp,
      'VRF': vrfExprFactory,
      'group': CliCommand.Node( matcherGroup, maxMatches=1 ),
      'VRID': CliCommand.Node( matcherVrId, maxMatches=1 ),
      'interface': CliCommand.Node( matcherInterface, maxMatches=1,
                                    sharedMatchObj=vrfIntfSharedObj ),
      'INTF': CliCommand.Node( IntfCli.Intf.matcher, maxMatches=1 ),
      'all': matcherAll,
      'brief': 'Display VRRP status summary',
   }
   cliModel = VrrpBriefModel
   privileged = True
   handler = showVrrpBrief

BasicCli.addShowCommandClass( ShowVrrpBriefCmd )

#--------------------------------------------------------------------------------
# "show vrrp [ { ( ( vrf VRF ) | ( interface INTF ) ) | ( group VRID ) } ]
# [ all ]" in enable mode
#--------------------------------------------------------------------------------

class ShowVrrpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show vrrp
               [ { ( ( VRF )
                   | ( interface INTF ) )
                 | ( group VRID ) } ]
               [ all ]'''
   data = {
      'vrrp': nodeShowVrrp,
      'VRF': vrfExprFactory,
      'group': CliCommand.Node( matcherGroup, maxMatches=1 ),
      'VRID': CliCommand.Node( matcherVrId, maxMatches=1 ),
      'interface': CliCommand.Node( matcherInterface, maxMatches=1,
                                    sharedMatchObj=vrfIntfSharedObj ),
      'INTF': CliCommand.Node( IntfCli.Intf.matcher, maxMatches=1 ),
      'all': matcherAll,
   }

   cliModel = VrrpDetailModel
   privileged = True
   handler = showVrrp

BasicCli.addShowCommandClass( ShowVrrpCmd )

# Show VARP commands
#-------------------------------------------------------------------------------
# "show ip virtual-router [ vrf ( default | all | VRF ) ]" in enable mode
#-------------------------------------------------------------------------------

def _getAllIntf( mode ):
   intfList = IntfCli.Intf.getAll( mode )
   if not intfList:
      intfList = []
   intfDict = { intf.name: intf for intf in intfList }
   return intfDict

# Hook for extending 'show ip virtual-router' cmd
showVirtualRouterMacHooks = CliExtensions.CliHook()

def showVirtualRouter( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   config = _varpConfig()
   vmacStatus = _fhrpVmacStatus()
   vmac = vmacStatus.varpVirtualMacWithMask
   virtualMac = Ethernet.convertMacAddrToDisplay( str( vmac.mac ) )
   mask = Ethernet.convertMacAddrToDisplay( str( vmac.mask ) )
   subnetRoutes = config.subnetRoutes
   varp = VarpModel()
   fhrpStatus = _fhrpStatusV4()
   varp.virtualMacs.append( VarpModel.VarpRouterMac( macAddress=virtualMac,
                                                     mask=mask,
                                                     macType='varp',
                                                     subnetRoutes=subnetRoutes ) )
   varp.advertiseInterval = config.advtInterval
   vrs = []
   if vmacStatus.varpVirtualMacWithMask != config.virtualMacWithMaskDefault:
      intfs = _getAllIntf( mode )
      varpIntfConfig = config.varpIntfConfig
      for intfName, intfConfig in sorted( varpIntfConfig.items() ):
         # Get Interface Object
         intf = intfs.get( intfName )
         if not ( intf and intf.routingSupported() and
                  intf.routingCurrentlySupported() ):
            continue

         # Get Vrf
         vrf = getVrfName( intfName )
         # pylint: disable-next=consider-using-in
         if vrfName != ALL_VRF_NAME and vrf != vrfName:
            continue

         # Get the virtual IPs
         virtualIps = sorted( intfConfig.virtualIpAddr.keys(),
                              key=functools.cmp_to_key( IpUtils.compareIpPrefix ) )
         if virtualIps:
            # Get physical addresses
            ipIntfConfig = IraIpIntfCli.IpIntf( intf, mode.sysdbRoot,
                                                createIfMissing=False )
            addr = ipIntfConfig.config().addrWithMask
            ipAddrAndMask = IpAddrAndMask( ip=addr.address,
                                           mask=addr.len )

            # Format the virtual IPs
            vipsAndMask = [ IpAddrAndMask( ip=vip.address, mask=vip.len )
                            for vip in virtualIps ]

            # Get VARP state
            state = varpStateEnumString( fhrpStateEnum.fhrpStopped )
            vrIntfStatus = fhrpStatus.vrIntfStatus.get( intfName )
            if vrIntfStatus:
               vrStatus = vrIntfStatus.vrStatus.get( VARP_VRID )
               if vrStatus:
                  state = varpStateEnumString( vrStatus.state )

            vr = VarpModel.VarpRouterV4( ipAddress=ipAddrAndMask,
                                         interface=intf.name,
                                         vrfName=vrf,
                                         interfaceStatus=intf.getIntfState(),
                                         protocolStatus=intf.lineProtocolState(),
                                         virtualIps=vipsAndMask,
                                         state=state )
            vrs.append( vr )
   varp.virtualRouters = vrs

   for hook in showVirtualRouterMacHooks.extensions():
      otherVirtualMac = hook( mode )
      if otherVirtualMac:
         varp.virtualMacs.append( otherVirtualMac )

   return varp

class ShowIpVirtualRouterCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip virtual-router [ VRF ]'
   data = {
      'ip': ipMatcherForShow,
      'virtual-router': CliCommand.Node( matcherVarpForShow,
                                         guard=varpSupportedGuard ),
      'VRF': vrfExprFactory,
   }
   cliModel = VarpModel
   handler = showVirtualRouter

BasicCli.addShowCommandClass( ShowIpVirtualRouterCmd )

#-------------------------------------------------------------------------------
# "show ipv6 virtual-router [ vrf ( default | all | VRF ) ]" in enable mode
#-------------------------------------------------------------------------------

def showVirtualRouterV6( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   config = _varpConfig()
   vmacStatus = _fhrpVmacStatus()
   vmac = vmacStatus.varpVirtualMacWithMask
   virtualMac = Ethernet.convertMacAddrToDisplay( str( vmac.mac ) )
   mask = Ethernet.convertMacAddrToDisplay( str( vmac.mask ) )
   subnetRoutes = config.subnetRoutes
   varp = VarpModel()
   fhrpStatus = _fhrpStatusV6()
   varp.virtualMacs.append( VarpModel.VarpRouterMac( macAddress=virtualMac,
                                                     mask=mask,
                                                     macType='varp',
                                                     subnetRoutes=subnetRoutes ) )
   varp.advertiseInterval = config.advtInterval
   vrs = []
   if vmacStatus.varpVirtualMacWithMask != config.virtualMacWithMaskDefault:
      intfs = _getAllIntf( mode )
      varpIntfConfig = config.varpIntfConfig
      for intfName, intfConfig in sorted( varpIntfConfig.items() ):
         # Get Interface Object
         intf = intfs.get( intfName )
         if not ( intf and intf.routingSupported() and
                  intf.routingCurrentlySupported() ):
            continue

         # Get Vrf
         vrf = getVrfName( intfName )
         # pylint: disable-next=consider-using-in
         if vrfName != ALL_VRF_NAME and vrf != vrfName:
            continue

         # Get the virtual IPs
         virtualIps = sorted( intfConfig.virtualIp6Addr.keys(),
                              key=functools.cmp_to_key( IpUtils.compareIp6Address ) )
         if virtualIps:
            # Get physical addresses
            ip6IntfConfig = IraIp6IntfCli.Ip6Intf( intf, mode.sysdbRoot,
                                                   createIfMissing=False )
            addrs = [ Ip6AddrAndMask( ip=ip6.address, mask=ip6.len )
                      for ip6 in ip6IntfConfig.config().addr ]
            if not addrs:
               addrs = [ Ip6AddrAndMask( ip=ip6AddrZero, mask=0 ) ]

            # Get VARP state
            state = varpStateEnumString( fhrpStateEnum.fhrpStopped )
            vrIntfStatus = fhrpStatus.vrIntfStatus.get( intfName )
            if vrIntfStatus:
               vrStatus = vrIntfStatus.vrStatus.get( VARP_VRID )
               if vrStatus:
                  state = varpStateEnumString( vrStatus.state )

            vr = VarpModel.VarpRouterV6( ipv6Addresses=addrs,
                                         interface=intf.name,
                                         vrfName=getVrfName( intf.name ),
                                         interfaceStatus=intf.getIntfState(),
                                         protocolStatus=intf.lineProtocolState(),
                                         virtualIpv6=virtualIps,
                                         state=state )
            vrs.append( vr )
   varp.virtualRouters = vrs

   for hook in showVirtualRouterMacHooks.extensions():
      otherVirtualMac = hook( mode )
      if otherVirtualMac:
         varp.virtualMacs.append( otherVirtualMac )

   return varp

class ShowIpv6VirtualRouterCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 virtual-router [ VRF ]'
   data = {
      'ipv6': ipv6MatcherForShow,
      'virtual-router': CliCommand.Node( matcherVarpForShow,
                                         guard=varpV6SupportedGuard ),
      'VRF': vrfExprFactory,
   }
   cliModel = VarpModel
   handler = showVirtualRouterV6

BasicCli.addShowCommandClass( ShowIpv6VirtualRouterCmd )

#-------------------------------------------------------------------------------
# "show ip address virtual source-nat address" in enable mode
#-------------------------------------------------------------------------------

def showSourceNatForVirtualIp( mode, args ):
   fhrpStatus = _fhrpStatusV4()

   varpSourceNatModel = VarpSourceNatV4Model()
   varpSourceNatModel.vrfs = {}
   for vrf, ipAddr in fhrpStatus.borrowedIpAddr.items():
      if ipAddr != '0.0.0.0':
         varpSourceNatModel.vrfs[ vrf ] = ipAddr

   return varpSourceNatModel

class ShowIpVirtualSnatAddressCmd( ShowCommand.ShowCliCommandClass ):
   # 'sourceNatAddr' below is the second 'address' in command.
   # Have to use this KeywordMatcher workaround because we have
   # two 'address' in this command
   syntax = 'show ip address virtual source-nat sourceNatAddr'
   data = {
      'ip': ipMatcherForShow,
      'address': 'Virtual router address',
      'virtual': 'Source NAT for IP address virtual',
      'source-nat': 'Source NAT for virtual IP status',
      'sourceNatAddr': matcherSourceNatAddr,
   }
   cliModel = VarpSourceNatV4Model
   handler = showSourceNatForVirtualIp

BasicCli.addShowCommandClass( ShowIpVirtualSnatAddressCmd )

#-------------------------------------------------------------------------------
# "show ipv6 address virtual source-nat address" in enable mode
#-------------------------------------------------------------------------------

def showSourceNatForVirtualIpv6( mode, args ):
   fhrpStatus = _fhrpStatusV6()

   varpSourceNatModel = VarpSourceNatV6Model()
   varpSourceNatModel.vrfs = {}
   for vrf, ip6Addr in fhrpStatus.borrowedIp6Addr.items():
      if ip6Addr != '::':
         varpSourceNatModel.vrfs[ vrf ] = ip6Addr

   return varpSourceNatModel

class ShowIpv6VirtualSnatAddressCmd( ShowCommand.ShowCliCommandClass ):
   # 'sourceNatAddr' below is the second 'address' in command.
   # Have to use this KeywordMatcher workaround because we have
   # two 'address' in this command
   syntax = 'show ipv6 address virtual source-nat sourceNatAddr'
   data = {
      'ipv6': ipv6MatcherForShow,
      'address': 'Virtual router address',
      'virtual': 'Source NAT for IP address virtual',
      'source-nat': 'Source NAT for virtual IP status',
      'sourceNatAddr': matcherSourceNatAddr,
   }
   cliModel = VarpSourceNatV6Model
   handler = showSourceNatForVirtualIpv6

BasicCli.addShowCommandClass( ShowIpv6VirtualSnatAddressCmd )

# Show FHRP internal commands
#-------------------------------------------------------------------------------
# "show fhrp internal pam [ vrf VRF ]" in enable mode
#-------------------------------------------------------------------------------

def showInternalPam( mode, args ):
   vrfName = args.get( 'VRF', '' )

   if not _fhrpConfig().fhrpConfigured:
      return

   # if vrfName is 'vrf all', it's the same as the no vrf behavior
   if vrfName is ALL_VRF_NAME:
      vrfName = ''

   AgentCommandRequest.runSocketCommand( mode.entityManager,
                                         FhrpAgent.name,
                                         'pam', vrfName )

class ShowFhrpInternalPamCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show fhrp internal pam [ VRF ]'
   data = {
      'fhrp': nodeFhrp,
      'internal': nodeInternal,
      'pam': 'PAM counters',
      'VRF': vrfExprFactory,
   }
   hidden = True
   privileged = True
   handler = showInternalPam

BasicCli.addShowCommandClass( ShowFhrpInternalPamCmd )

#-------------------------------------------------------------------------------
# "show fhrp internal state [ interface INTF [ group VRID ] ]" in enable mode
#-------------------------------------------------------------------------------

def showInternalState( mode, args ):
   intfId = args.get( 'INTF' )
   vrId = args.get( 'VRID' )

   if not _fhrpConfig().fhrpConfigured:
      return

   if intfId and vrId:
      cmd = 'intfId %s vrId %s' % ( intfId, vrId )
   elif intfId:
      cmd = 'intfId %s' % intfId
   elif vrId:
      cmd = 'vrId %s' % vrId
   else:
      cmd = ''

   AgentCommandRequest.runSocketCommand( mode.entityManager,
                                         FhrpAgent.name,
                                         'state', cmd )

class ShowFhrpInternalStateCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show fhrp internal state [ interface INTF ] [ group VRID ]'
   data = {
      'fhrp': nodeFhrp,
      'internal': nodeInternal,
      'state': 'VRRP state and counters',
      'interface': 'Interface',
      'INTF': IntfCli.Intf.matcher,
      'group': 'VRRP group',
      'VRID': CliMatcher.IntegerMatcher( 1, 256,
         helpdesc='Virtual router ID (256 for VARP)' ),
   }
   hidden = True
   privileged = True
   handler = showInternalState

BasicCli.addShowCommandClass( ShowFhrpInternalStateCmd )

#-------------------------------------------------------------------------------
# "show fhrp internal counters" in enable mode
#-------------------------------------------------------------------------------

def showInternalCounters( mode, args ):
   if not _fhrpConfig().fhrpConfigured:
      return

   AgentCommandRequest.runSocketCommand( mode.entityManager,
                                         FhrpAgent.name,
                                         'counters', '' )

class ShowFhrpInternalCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show fhrp internal counters'
   data = {
      'fhrp': nodeFhrp,
      'internal': nodeInternal,
      'counters': 'Fhrp agent counters',
   }
   hidden = True
   privileged = True
   handler = showInternalCounters

BasicCli.addShowCommandClass( ShowFhrpInternalCountersCmd )

# Helper function to clean the virtualIp6LinkLocalAddr collection for a virtual
# router( vrId, intfId ). virtualIp6LinkLocalAddr keeps track of all the IPv6
# link local addresses configured as VRRP address accross the virtual routers.
# This function clears up an address from this collection for a virtual router
# (and all the virtual routers using that intfId if vrId is passed as None).
def cleanupVirtualIp6LinkLocalAddr( vrfIp, intfId, vrId=None ):
   vrrpConfig = _vrrpConfig()
   if vrfIp in vrrpConfig.virtualIp6LinkLocalAddr:
      fhrpIntfCol = vrrpConfig.virtualIp6LinkLocalAddr[ vrfIp ]
      for fhrpId in fhrpIntfCol.vrIdIntf:
         if not vrId:
            if fhrpId.intfId == intfId:
               del fhrpIntfCol.vrIdIntf[ fhrpId ]
         else:
            if fhrpId.intfId == intfId and fhrpId.vrId == vrId:
               del fhrpIntfCol.vrIdIntf[ fhrpId ]
      if not fhrpIntfCol.vrIdIntf:
         del vrrpConfig.virtualIp6LinkLocalAddr[ vrfIp ]

#-------------------------------------------------------------------------------
# The FhrpIntf class is used to remove the fhrp IntfConfig object when an
# interface is deleted. The Intf class will create a new instance of FhrpIntf
# and call destroy when the interface is deleted
#-------------------------------------------------------------------------------

class FhrpIntf( IntfCli.IntfDependentBase ):
   # Destroys the fhrp IntfConfig object for this interface if it exists.
   def setDefault( self ):
      vrrp = _vrrpConfig()
      varp = _varpConfig()
      intf = self.intf_.name

      deleteFhrpIntfIps( intf )

      # Force delete because there might be some non-ip lingering config
      if intf in vrrp.vrrpIntfConfig:
         del vrrp.vrrpIntfConfig[ intf ]
      if intf in varp.varpIntfConfig:
         del varp.varpIntfConfig[ intf ]

      evaluateFhrpRunnability()

#-------------------------------------------------------------------------------
# Hook for Ira CLI to check if an address conflicts with VRRP addresses
#-------------------------------------------------------------------------------

def canSetIntfIp( intfName, ipAddrWithMask, secondary=False, mode=None ):
   vrf = getVrfName( intfName )
   ipAddr = ipAddrWithMask.address
   vrfIp = vrfIpPair( vrf, ipAddr )

   if ipAddr == ipAddrZero:
      return [ True, None ]

   message = None
   if not Toggles.FhrpToggleLib.toggleVrrpSamePhysicalIpAsVipEnabled():
      unique, message = uniqueFhrpIpAssigned( vrfIp )
      if not unique:
         return [ False, message ]

   # check for 'ip address virtual' config when configuring ip addr
   ipIntfConfig = ipConfig.ipIntfConfig.get( intfName )
   if ipIntfConfig and ipIntfConfig.virtualAddrWithMask.address != '0.0.0.0':
      return [ False, 'Virtual IP address is already configured with %s' %
               ipIntfConfig.virtualAddrWithMask ]
   return [ True, message ]

def canSetIntfIp6( intfName, ip6AddrWithMask, mode=None ):
   vrf = getVrfName( intfName )
   ip6Addr = ip6AddrWithMask.address
   vrfIp = vrfIpPair( vrf, ip6Addr )

   if ip6Addr.isUnspecified:
      return [ True, None ]

   message = None
   if not Toggles.FhrpToggleLib.toggleVrrpSamePhysicalIpAsVipEnabled():
      unique, message = uniqueFhrpIpAssigned( vrfIp )
      if not unique:
         return [ False, message ]

   # check for 'ipv6 address virtual' config when configuring ipv6 addr
   ipIntf = ip6Config.intf.get( intfName )
   if ipIntf and ipIntf.virtualAddr:
      return [ False, 'At least 1 virtual address is already configured. '
               'Mixing virtual and non-virtual addresses on an interface is not'
               ' supported.' ]

   return [ True, message ]

def canSetIntfLinkLocalAddr( intfName, ip6AddrWithMask ):
   vrf = getVrfName( intfName )
   ip6Addr = ip6AddrWithMask.address
   vrfIp = vrfIpPair( vrf, ip6Addr )

   if ip6Addr.isUnspecified:
      return [ True, None ]

   vrrpConfig = _vrrpConfig()
   fhrpIntfCol = vrrpConfig.virtualIp6LinkLocalAddr.get( vrfIp )
   if fhrpIntfCol:
      for vrIdIntf in fhrpIntfCol.vrIdIntf:
         if vrIdIntf.intfId == intfName:
            return False, '%s is assigned as a VRRP address on interface %s ' \
                  'in vrf %s for virtual router %s' \
                  % ( ip6Addr, vrIdIntf.intfId, vrfIp.vrfName, vrIdIntf.vrId )
   varpConfig = _varpConfig()
   intfCol = varpConfig.virtualIpAddr.get( vrfIp )
   if intfCol:
      if intfName in intfCol.intfIdToMask:
         return False, '%s is assigned as a VARP address on interface %s ' \
               'in vrf %s' \
                % ( ip6Addr, intfName, vrfIp.vrfName )

   return [ True, None ]

def isValidNextHop( hop, vrfName ):
   vrrpConfig = _vrrpConfig()
   varpConfig = _varpConfig()
   vrfIp = vrfIpPair( vrfName, hop )

   if vrfIp in vrrpConfig.virtualIpAddr or vrfIp in varpConfig.virtualIpAddr:
      return False
   return True

def isValidNextHop6( hop, vrfName ):
   vrrpConfig = _vrrpConfig()
   varpConfig = _varpConfig()
   vrfIp = vrfIpPair( vrfName, hop )

   if vrfIp in vrrpConfig.virtualIpAddr or vrfIp in varpConfig.virtualIpAddr:
      return False
   return True

#-------------------------------------------------------------------------------
# Hook invoked by Ira before setting or deleting VRF
#-------------------------------------------------------------------------------

# deletes the varp ip in the global collection
def deleteVarpGlobalIp( ipWithMask, intfName ):
   ip = str( ipWithMask ).split( '/' )[ 0 ] # pylint: disable=use-maxsplit-arg
   vrfName = getVrfName( intfName )
   vrfIp = vrfIpPair( vrfName, ip )
   varpConfig = _varpConfig()
   intfCol = varpConfig.virtualIpAddr.get( vrfIp )

   if intfCol:
      if intfName in intfCol.intfIdToMask:
         del intfCol.intfIdToMask[ intfName ]
      if not intfCol.intfIdToMask:
         del varpConfig.virtualIpAddr[ vrfIp ]

# deletes the vrrp ip in the global collection
def deleteVrrpGlobalIp( ip, intfName, vrId=None ):
   vrrpConfig = _vrrpConfig()
   vrfName = getVrfName( intfName )
   vrfIp = vrfIpPair( vrfName, ip )
   if vrfIp in vrrpConfig.virtualIpAddr:
      if intfName == vrrpConfig.virtualIpAddr[ vrfIp ].intfId:
         del vrrpConfig.virtualIpAddr[ vrfIp ]
   cleanupVirtualIp6LinkLocalAddr( vrfIp, intfName, vrId=vrId )

def deleteVrrpIntfIps( intfName ):
   vrrpConfig = _vrrpConfig()
   config = vrrpConfig.vrrpIntfConfig.get( intfName )
   if config is None:
      return False

   vipsDeleted = False
   for vrid in config.virtualRouterConfig:
      if deleteVrrpVrIps( intfName, vrid ):
         vipsDeleted = True
      deleteVrConfigIfDefault( intfName, vrid )
   deleteVrrpIntfConfigIfDefault( intfName )
   return vipsDeleted

def deleteVrrpVrIps( intfName, vrid ):
   vrrpConfig = _vrrpConfig()
   config = vrrpConfig.vrrpIntfConfig[ intfName ].virtualRouterConfig[ vrid ]
   if config is None:
      return False
   vipsDeleted = config.primaryAddr != config.primaryAddrDefault or \
                 config.secondaryAddrs or config.ip6Addrs

   deleteVrrpGlobalIp( config.primaryAddr, intfName )
   config.bfdPeerAddrV4 = config.bfdPeerAddrV4Default
   config.bfdPeerAddrV6 = config.bfdPeerAddrV6Default
   config.primaryAddr = config.primaryAddrDefault
   for ipAddr in config.secondaryAddrs:
      deleteVrrpGlobalIp( ipAddr, intfName )
   config.secondaryAddrs.clear()
   for ip6Addr in config.ip6Addrs:
      deleteVrrpGlobalIp( ip6Addr, intfName )
   config.ip6Addrs.clear()

   return vipsDeleted

def deleteVarpIntfIps( intfName ):
   varpConfig = _varpConfig()
   config = varpConfig.varpIntfConfig.get( intfName )
   if config is None:
      return False
   vipsDeleted = config.virtualIpAddr or config.virtualIp6Addr

   for ipAddrWithMask in config.virtualIpAddr:
      deleteVarpGlobalIp( ipAddrWithMask, intfName )
   config.virtualIpAddr.clear()
   for ip6Addr in config.virtualIp6Addr:
      deleteVarpGlobalIp( ip6Addr, intfName )
   config.virtualIp6Addr.clear()
   deleteVarpConfigIfDefault( intfName )

   return vipsDeleted

def deleteFhrpIntfIps( intfName ):
   vrrpVipsDeleted = deleteVrrpIntfIps( intfName )
   varpVipsDeleted = deleteVarpIntfIps( intfName )
   return vrrpVipsDeleted or varpVipsDeleted

def canSetVrf( intfName, oldVrf, newVrf, vrfDelete ):
   vipsDeleted = deleteFhrpIntfIps( intfName )
   if not vipsDeleted:
      msg = None
   elif newVrf != DEFAULT_VRF:
      msg = 'Virtual IP addresses from interface %s have been removed due to ' \
            'enabling VRF %s' % ( intfName, newVrf )
   else:
      msg = 'Virtual IP addresses from interface %s have been removed due to ' \
            'disabling VRF %s' % ( intfName, newVrf )

   evaluateFhrpRunnability()
   return ( True, msg )

def canDeleteVrf( vrfName ):
   varpConfig = _varpConfig()
   vipsDeleted = False
   for intfName in varpConfig.varpIntfConfig:
      if getVrfName( intfName ) == vrfName:
         if deleteFhrpIntfIps( intfName ):
            vipsDeleted = True

   if vipsDeleted:
      msg = 'Virtual IP addresses from all interfaces in VRF %s have been removed' \
            % vrfName
   else:
      msg = None

   evaluateFhrpRunnability()
   return ( True, msg )

#-------------------------------------------------------------------------------
# VRRP commands in "show tech-support"
#-------------------------------------------------------------------------------
# These commands were in the original "show tech" command list from the EOS
# package.  To avoid having them move around in the "show tech" output,
# we use a made up timestamp.
TechSupportCli.registerShowTechSupportCmd(
   '2010-05-31 00:00:00',
   cmds=[ 'show vrrp vrf all all' ],
   cmdsGuard=lambda: vrrpSupportedGuard( mode=None, token=None ) is None )

#-------------------------------------------------------------------------------
# VARP commands in "show tech-support"
#-------------------------------------------------------------------------------
TechSupportCli.registerShowTechSupportCmd(
   '2010-06-01 00:00:00',
   cmds=[ 'show ip virtual-router vrf all' ],
   cmdsGuard=lambda: varpSupportedGuard( None, None ) is None )

TechSupportCli.registerShowTechSupportCmd(
   '2010-06-01 00:00:00',
   cmds=[ 'show ipv6 virtual-router vrf all' ],
   cmdsGuard=lambda: varpV6SupportedGuard( None, None ) is None )

def Plugin( entMan ):
   global fhrpConfigDir, fhrpStatusDirV4, fhrpStatusDirV6, fhrpHardwareStatus
   global varpConfigDir
   global fhrpVirtualMacStatus, routing6HardwareStatus
   global ipConfig, ip6Config, ipStatus, ip6Status, l3ConfigDir
   global mlagConfigDir
   global hwEntMibStatus
   global trackingConfig
   global securityConfig

   fhrpConfigDir = ConfigMount.mount( entMan, 'routing/fhrp/config',
                                      'Routing::Fhrp::Config', 'w' )
   varpConfigDir = ConfigMount.mount( entMan, 'routing/fhrp/varp/config',
                                      'Routing::Fhrp::VarpConfig', 'w' )
   fhrpStatusDirV4 = LazyMount.mount( entMan, 'routing/fhrp/status',
                                      'Routing::Fhrp::StatusV4', 'r' )
   fhrpStatusDirV6 = LazyMount.mount( entMan, 'routing6/fhrp/status',
                                      'Routing::Fhrp::StatusV6', 'r' )
   fhrpVirtualMacStatus = LazyMount.mount( entMan, 'routing/fhrp/vrMacStatus',
                                           'Routing::Fhrp::VirtualRouterMacStatus',
                                           'r' )
   fhrpHardwareStatus = LazyMount.mount( entMan, 'routing/fhrp/hardware/status',
                                         'Routing::Fhrp::Hardware::Status', 'r' )
   routing6HardwareStatus = LazyMount.mount( entMan, 'routing6/hardware/status',
                                             'Routing6::Hardware::Status', 'r' )
   ipConfig = LazyMount.mount( entMan, 'ip/config', 'Ip::Config', 'w' )
   ip6Config = LazyMount.mount( entMan, 'ip6/config', 'Ip6::Config', 'w' )
   ipStatus = LazyMount.mount( entMan, 'ip/status', 'Ip::Status', 'r' )
   ip6Status = LazyMount.mount( entMan, 'ip6/status', 'Ip6::Status', 'r' )
   l3ConfigDir = ConfigMount.mount( entMan, 'l3/intf/config', 'L3::Intf::ConfigDir',
                                    'w' )
   mlagConfigDir = LazyMount.mount( entMan, 'mlag/config', 'Mlag::Config', 'r' )
   hwEntMibStatus = LazyMount.mount( entMan, 'hardware/entmib',
                                     "EntityMib::Status", 'r' )
   trackingConfig = LazyMount.mount( entMan, 'tracking/config', 'Tracking::Config',
                                     'r' )
   securityConfig = LazyMount.mount( entMan, 'mgmt/security/config',
                                     "Mgmt::Security::Config", 'r' )

   IraIpIntfCli.canSetIntfIpHook.addExtension( canSetIntfIp )
   IraIp6IntfCli.canSetIntfIpHook.addExtension( canSetIntfIp6 )
   IraIp6IntfCli.canSetIntfLLAddrHook.addExtension( canSetIntfLinkLocalAddr )
   FruCli.canSetSystemMacAddrHook.addExtension( canSetSystemMacAddr )
   IraCommonCli.isValidNextHopHook4.addExtension( isValidNextHop )
   IraCommonCli.isValidNextHopHook6.addExtension( isValidNextHop6 )
   IraIpIntfCli.canSetVrfHook.addExtension( canSetVrf )
   IraVrfCli.canDeleteVrfHook.addExtension( canDeleteVrf )

   # The intf config is deleted with priority 10, so have this run on priority 9 so
   # we can read the vrf from it before that info is lost (yes, lower priority
   # activities run first, don't ask me why).
   IntfCli.Intf.registerDependentClass( FhrpIntf, priority=9 )
