# Copyright (c) 2005-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# pylint: disable=ungrouped-imports

import ArPyUtils
import BasicCli
import BasicCliModes
import BasicCliUtil
import Cell
import CliCommand
import CliExtensions
from CliGlobal import CliGlobal
import CliMatcher
import CliModel
import CliPlugin.TechSupportCli
import CliPrint
from CliPrint import (
      Printer,
      IntAttrInfo,
      StringAttrInfo,
      FloatAttrInfo,
)
import CliSave
import CliToken.Configure
import CliToken.Service
import CommonGuards
import ConfigMount
from CliPlugin.FileCliModels import (
   FileInformation,
   FileSystem,
   FileSystems,
   FileDigest,
   FileDir,
)
# pylint: disable-next=consider-using-from-import
import CliPlugin.ShowRunModel as ShowRunModel
import FileCliUtil
import FileUrl
import LazyMount
import Logging
from SectionCliLib import sectionFilter, sectionFilterAll
import ShowCommand
import ShowRunOutputModel
import SwiSignLib
import Tac
import TacSigint
import Url
import UtmpDump
import UrlPlugin.FlashUrl as FlashUrl # pylint: disable=consider-using-from-import
from UrlPlugin.SystemUrl import SystemUrl

import errno
import hashlib
import os
import re
import stat
import sys
import time

# pkgdeps: rpm FileCli-lib

SYS_CONFIG_STARTUP = Logging.LogHandle(
              "SYS_CONFIG_STARTUP",
              severity=Logging.logNotice,
              fmt="Startup config saved from %s by %s on %s (%s).",
              explanation=( "A network administrator has saved the system's "
                            "startup configuration." ),
              recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_BOOT_NEW_SWI = Logging.LogHandle(
              "SYS_BOOT_NEW_SWI",
              severity=Logging.logInfo,
              fmt="Boot image has been updated and has a SHA-512 hash of:%s",
              explanation=( "The SWI used to boot has been overwritten. The hash"
                            " of the new SWI is displayed in the log message." ),
              recommendedAction=Logging.NO_ACTION_REQUIRED )
SYS_BOOT_COPY_FAILED = Logging.LogHandle(
              "SYS_BOOT_COPY_FAILED",
              severity=Logging.logInfo,
              fmt="Failed to copy new SWI into place due to: %s",
              explanation=( "The system was unable to copy the new SWI file "
                            "into the boot location." ),
              recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS )

em = BasicCli.EnableMode

abootSb = CliGlobal( status=None )
cliStatus = None
serviceConfig = None

# Print command-name and date/time for some show commands.
def showCommandAndTime( mode, cmdStr ):
   mode.addWarning( 'Command: %s' % cmdStr )
   if serviceConfig.timeInRunningConfig:
      mode.addWarning( 'Time: %s' % time.asctime() )

def showRunningConfigCommon( mode, cmdStr, **kwargs ):
   # Inserting the info printed by showAndCommandTime into the
   # file-stream is hard to achieve for each command.
   if mode.session_.shouldPrint():
      showCommandAndTime( mode, cmdStr )

   CliSave.saveRunningConfig( mode.entityManager, sys.stdout, **kwargs )

   # Return an empty model to keep CAPI happy
   return CliModel.noValidationModel( ShowRunOutputModel.Mode )

def copyFile( mode, surl, durl ):
   errorStr = FileCliUtil.copyFile( cliStatus, mode, surl, durl )
   bootConfig = FileUrl.bootConfig( mode, createIfMissing=False )
   if ( bootConfig and 'SWI' in bootConfig and 
        str( durl ) == bootConfig[ 'SWI' ] ):
      if errorStr:
         Logging.log( SYS_BOOT_COPY_FAILED, errorStr )
      else:
         localPath = durl.localFilename()
         if localPath:
            # Use SHA-512 since that is what is published on arista.com
            swiHash = FileCliUtil.chunkedHashCompute( durl, hashlib.sha512 )
            mode.addMessage( "Boot SWI has been updated to a file " +\
                             "with hash: %s" %\
                             swiHash )
            Logging.log( SYS_BOOT_NEW_SWI, swiHash )

copyRunningHandler = None

#####################################################################################
## Implementation of the copy command
##     copy <src-url> <dst-url>
#####################################################################################

secureMonitorKw = CliCommand.Node( CliMatcher.KeywordMatcher(
   'secure-monitor',
   helpdesc='Secure monitor running or startup configuration' ),
                                   guard=CommonGuards.secureMonitorGuard )

class ConfigExpressionFactory( CliCommand.CliExpressionFactory ):
   # Generate "[ secure-monitor ] running-config | startupConfig"

   def __init__( self, isRunningConfig, guard=None ):
      self.isRunningConfig_ = isRunningConfig
      self.guard_ = guard
      CliCommand.CliExpressionFactory.__init__( self )

   def generate( self, name ):
      secureMonitorName = "secure-monitor_" + name
      if self.isRunningConfig_:
         configName = "running-config_"
      else:
         configName = "startup-config_"
      configName += name

      if self.isRunningConfig_:
         matcher = CliMatcher.KeywordMatcher(
            "running-config",
            helpdesc="System operating configuration" )
      else:
         matcher = CliMatcher.KeywordMatcher(
            "startup-config",
            helpdesc="Startup configuration" )

      class ConfigExpression( CliCommand.CliExpression ):
         expression = f"[ {secureMonitorName} ] {configName}"
         data = {
            secureMonitorName : secureMonitorKw,
            configName : CliCommand.Node( matcher,
                                          guard=self.guard_ )
         }
         @classmethod
         def adapter( cls, mode, args, argsList ):
            if configName in args:
               urlArgs = Url.urlArgsFromMode( mode )
               # Note: this is a bit special. If either source or destination has
               # 'secure-monitor' we apply it to both to avoid inconsistent usage
               # and leak sensitive information (e.g., copying secure config to
               # normal startup-config). This is a bit hackish. Unfortunately,
               # we cannot use the same 'secure-monitor' name in source and
               # destination as it'll assert in expression expansion.
               secureMonitor = ( secureMonitorName in args or
                                 any( name.startswith( 'secure-monitor_' ) and
                                      value == 'secure-monitor'
                                      for name, value in args.items() ) )
               if self.isRunningConfig_:
                  val = FileUrl.localRunningConfig( *urlArgs,
                                                    secureMonitor=secureMonitor )
               else:
                  val = FileUrl.localStartupConfig( *urlArgs,
                                                    secureMonitor=secureMonitor )

               args[ name ] = val

      return ConfigExpression

FileCliUtil.registerCopySource(
   'startup-config',
   ConfigExpressionFactory( False ),
   'startup-config' )

FileCliUtil.registerCopySource(
   'running-config',
   ConfigExpressionFactory( True ),
   'running-config' )

FileCliUtil.registerCopyDestination(
   'startup-config',
   ConfigExpressionFactory( False, guard=CommonGuards.ssoStandbyGuard ),
   'startup-config' )

def logExitConfig( mode, srcUrl, dstUrl, commandSource ):
   if dstUrl.url == "system:/running-config":
      BasicCliUtil.logConfigSource( str( srcUrl ), BasicCliUtil.SYS_CONFIG_I )
   return True

def handleFileCopyToSourceConfig( mode, surl, durl, commandSource ):
   startupConfig = FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ) )
   runningConfig = FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ) )
   if mode.session_.standalone_ and surl == runningConfig and durl == startupConfig:
      mode.addError( "Copy from running-config to startup-config "
                      "not allowed in standalone mode" )
      return False
   return True

# Log appropriate SYS_CONFIG_I pair when copying to running-config
FileCliUtil.copyFileNotifiers.addExtension( logExitConfig )

FileCliUtil.copyFileCheckers.addExtension( handleFileCopyToSourceConfig )

# there is really no difference as copy destination between running-config
# and secure-monitor running-config, so we just register the 'running-config' token.
FileCliUtil.registerCopyDestination(
   'running-config',
   CliCommand.Node(
      CliMatcher.KeywordMatcher( 'running-config',
           helpdesc='Update (merge with) system running configuration',
                                 value=lambda mode, match: \
                                 FileUrl.localRunningConfig(
                                    *Url.urlArgsFromMode( mode ) ) ),
      guard=CommonGuards.ssoStandbyGuard ),
   'running-config' )

# copy command is run with aaa disabled in verifyConfig() eapi. Caching it will
# save the disableAaa_ in the Url context and be reused by subsequent copy commands
# and thereby bypassing aaa.
class CopyCmd( CliCommand.CliCommandClass ):
   syntax = "copy SRC_URL DEST_URL"
   data = { 'copy' : 'Copy files',
            'SRC_URL' : FileCliUtil.copySourceUrlExpr(),
            'DEST_URL' : FileCliUtil.copyDestinationUrlExpr()
            }
   allowCache = False
   @staticmethod
   def handler( mode, args ):
      surl = args[ 'SRC_URL' ]
      durl = args[ 'DEST_URL' ]
      if ( isinstance( durl, SystemUrl ) and
           durl.pathname == '/running-config' and
           serviceConfig.useSessionInCopyRunningConfig ):
         if copyRunningHandler:
            if copyRunningHandler( mode, surl ): # pylint: disable-msg=E1102
               mode.addMessage( "Copy with config session completed successfully." )
            return
         else:
            # This shouldn't happen, print an error and continue with default
            # implementation.
            mode.addError( "Cannot copy to running-config with config session" )

      copyFile( mode, surl, durl )

em.addCommandClass( CopyCmd )

#####################################################################################
## Implementation of the diff command
##     diff FIRST_URL SECOND_URL
#####################################################################################
diffFirstUrl = FileCliUtil.diffFirstUrlExpr()
diffFirstUrl.registerExtension(
   'startup-config',
   CliMatcher.KeywordMatcher(
      'startup-config',
      helpdesc='Diff from startup configuration',
      value=lambda mode, match: \
      FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ) ) ),
   'startup-config' )
diffFirstUrl.registerExtension(
   'running-config',
   CliMatcher.KeywordMatcher(
      'running-config',
      helpdesc='Diff from current system configuration',
      value=lambda mode, match: \
      FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ) ) ),
   'running-config' )

diffSecondUrl = FileCliUtil.diffSecondUrlExpr()
diffSecondUrl.registerExtension(
   'startup-config',
   CliMatcher.KeywordMatcher (
      'startup-config',
      helpdesc='Diff to startup configuration',
      value=lambda mode, match: \
         FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ) ) ),
   'startup-config' )
diffSecondUrl.registerExtension(
   'running-config',
   CliMatcher.KeywordMatcher(
      'running-config',
      helpdesc='Diff to current system configuration',
      value=lambda mode, match: \
         FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ) ) ),
   'running-config' )

class DiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "diff FIRST_URL SECOND_URL"
   data = {
      "diff" : 'Diff one file with another',
      'FIRST_URL' : diffFirstUrl,
      'SECOND_URL' : diffSecondUrl
      }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      firstUrl = args[ 'FIRST_URL' ]
      secondUrl = args[ 'SECOND_URL' ]
      FileCliUtil.diffFile( mode, firstUrl, secondUrl )

BasicCli.addShowCommandClass( DiffCmd )

#####################################################################################
## Implementation of the "obsolete" write command
## Too bad we can't just retrain everyone's fingers
##     write [memory]
##     write terminal
##     write network
##     write erase [ secure-monitor ]
#####################################################################################
writeKwMatcher = CliMatcher.KeywordMatcher( 'write', helpdesc='Write configuration' )
nowKwMatcher = CliMatcher.KeywordMatcher( 'now',
                          helpdesc="Perform action immediately without prompting" )

eraseHelp = 'Erase startup Configuration'
# erase for `write erase`
eraseKwMatcher = CliMatcher.KeywordMatcher( 'erase',
                                            helpdesc='Erase startup Configuration' )
eraseKw = CliCommand.Node( eraseKwMatcher, guard=CommonGuards.ssoStandbyGuard )
# erase for `erase startup-config`
eraseDeprecatedKw = CliCommand.Node( eraseKwMatcher,
                                     guard=CommonGuards.ssoStandbyGuard,
                                     deprecatedByCmd='delete' )
# delete for `delete startup-config`
deletekKw = CliCommand.guardedKeyword( 'delete',
                                       eraseHelp,
                                       CommonGuards.ssoStandbyGuard )

def eraseLocalStartupConfig( mode, delete ):
   if delete:
      try:
         FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ) ).empty()
      except OSError as e:
         mode.addError( "Error erasing startup configuration (%s)" % e.strerror )


FileCliUtil.registerPersistentConfiguration( "startup-config",
                                 eraseLocalStartupConfig,
                                 "saved startup configuration", default=True )

def erasePersistentConfig( mode, now=False, preserveOrDelete=None,
                           preserveOnly=None, deleteOnly=None ):
   if preserveOrDelete is None:
      preserveOrDelete = [] # Unspecified, but it must be of type list
   # If we specify delete and preserve for the same configuration, preserve it
   # just to be safe, and print a warning!
   delete = None
   preserve = None
   if preserveOnly:
      # assert to catch regressions in future rules; Cli should guarantee
      assert not deleteOnly
      preserve = preserveOnly
      configs = set( FileCliUtil.__persistentConfigInfo__.info().keys() )
      delete = list( configs.difference( preserve ) )
   elif deleteOnly:
      assert not preserveOnly
      delete = deleteOnly
      configs = set( FileCliUtil.__persistentConfigInfo__.info().keys() )
      preserve = list( configs.difference( delete ) )
   else:
      for name, value in preserveOrDelete:
         if name == 'delete':
            delete = value
         elif name == 'preserve':
            preserve = value
         else:
            assert False, "%s must be either 'preserve' or 'delete'" % name
   if not now:
      if not BasicCliUtil.confirm( mode, \
            "Proceed with erasing startup configuration? [confirm]" ):
         return
   conflicts = ( isinstance( delete, list ) and
                 isinstance( preserve, list ) and
                 set( delete ).intersection( preserve ) )
   if conflicts:
      pronouns = ( ( "They", "them" ) if ( len( conflicts ) > 1 )
                   else ( "It", "it" ) )
      msg = ( "specified both preserve and delete for: " +
              ",".join( conflicts ) +
              ( ". %s will be preserved." +
               " Re-issue command unambiguously to delete %s." ) % pronouns )
      mode.addWarning( msg )
      delete = list( set( delete ).difference( conflicts ) )
   print( "cleanup", delete, preserve )
   FileCliUtil.__persistentConfigInfo__.cleanup( mode, delete, preserve )

# Just erase the default configs, with no option for user to have any
# finer grained user control.  Defaults for now are startup-config
# and ribd-config.
def eraseStartupConfig( mode, now=False, commandSource="commandLine" ):
   # Write-erase event is added to config history table (CONFIG-MAN-MIB)
   # commandSource = erase, configSource = erase, configDest = "none"
   # sourceURL = "", destURL = ""
   mode.session_.addToHistoryEventTable( commandSource, "erase",
                                         "none" )
   erasePersistentConfig( mode, now=now )

def eraseSecureMonitorStartupConfig( mode, now=False ):
   # delete the secure-monitor startup
   if not now and not BasicCliUtil.confirm( mode, \
         "Proceed with erasing secure-monitor startup configuration? [confirm]" ):
      return
   try:
      if not mode.session.secureMonitor():
         raise OSError( errno.EPERM, os.strerror( errno.EPERM ) )
      config = FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ),
                                           secureMonitor=True )
      if config.exists():
         config.delete( False )
   except OSError as e:
      mode.addError( "Error erasing secure-monitor startup configuration (%s)" %
                     e.strerror )

class WriteTerminal( CliCommand.CliCommandClass ):
   syntax = "write terminal"
   data = { 'write' : writeKwMatcher,
            'terminal' : 'Show Running Configuration'
   }
   @staticmethod
   def handler( mode, args ):
      showRunningConfig( mode )

em.addCommandClass( WriteTerminal )

class WriteMemory( CliCommand.CliCommandClass ):
   syntax = "write [ memory ]"
   data = { 'write' : writeKwMatcher,
            'memory' : CliCommand.guardedKeyword( 'memory',
                                                  'Write to Startup Configuration',
                                                  CommonGuards.ssoStandbyGuard ),
   }
   @staticmethod
   def handler( mode, args ):
      if ( 'memory' not in args and
           CommonGuards.ssoStandbyGuard( mode, 'memory' ) is not None ):
         # "write <cr>" cannot be guarded. do the check here
         mode.addError( 'Command is unavailable on this supervisor' )
      else:
         copyFile( mode,
                   FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ) ),
                   FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ) ) )

em.addCommandClass( WriteMemory )

class WriteErase( CliCommand.CliCommandClass ):
   syntax = "write erase [ secure-monitor] [ now ]"
   data = { 'write' : writeKwMatcher,
            'erase' : eraseKw,
            'secure-monitor' : CliCommand.Node(
               CliMatcher.KeywordMatcher(
                  'secure-monitor',
                  helpdesc='Erase secure monitor startup configuration' ),
               guard=CommonGuards.secureMonitorGuard ),
            'now' : nowKwMatcher
   }
   @staticmethod
   def handler( mode, args ):
      now = 'now' in args
      if 'secure-monitor' in args:
         eraseSecureMonitorStartupConfig( mode, now=now )
      else:
         eraseStartupConfig( mode, now=now )

em.addCommandClass( WriteErase )

class WriteNetwork( CliCommand.CliCommandClass ):
   syntax = "write network [ URL ]"
   data = { 'write' : writeKwMatcher,
            'network' : 'Copy Running Configuration to the Network',
            'URL' : Url.UrlMatcher( lambda fs: fs.supportsWrite(),
                                    'Destination file URL',
                                    notAllowed=[ 'startup-config',
                                                 'running-config' ],
                                    allowAllPaths=True )
            }
   @staticmethod
   def handler( mode, args ):
      url = args.get( 'URL' )
      if url is None or url.localFilename():
         mode.addError( 'Please use "copy running-config URL"' )
      else:
         copyFile( mode, FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ) ),
                   url )

em.addCommandClass( WriteNetwork )

##############################################################################
## Implementation of hidden erase persistent-config command
##
##############################################################################
def pcHelp( includeDefaults ):
   helpDict = {}
   for keyword, entry in \
         FileCliUtil.__persistentConfigInfo__.info().items():
      _, description, default = entry
      helpStr = description
      if includeDefaults:
         helpStr += "[ default: %s ]" % ( "delete" if default else "preserve" )
      helpDict[ keyword ] = helpStr
   return helpDict

@Tac.memoize
def pcHelpNoDefaults():
   return pcHelp( False )

@Tac.memoize
def pcHelpDefaults():
   return pcHelp( True )

pcMatcher = CliMatcher.DynamicKeywordMatcher( lambda mode: pcHelpDefaults() )

class ErasePersistent( CliCommand.CliCommandClass ):
   syntax = ( "erase persistent-config [ now ] "
              "[ { ( preserve { PRESERVE } ) | ( delete { DELETE } ) } | "
              "( preserve-only { PRESERVE } ) | ( delete-only { DELETE } ) ]" )
   data = {
      'erase' : eraseKw,
      'persistent-config' : CliCommand.Node( CliMatcher.KeywordMatcher(
         'persistent-config', helpdesc='Erase contents of persistent config files' ),
                                             hidden=True ),
      'now' : nowKwMatcher,
      'preserve' : CliCommand.singleKeyword( 'preserve',
                                  helpdesc='Persistent configurations to preserve' ),
      'delete' : CliCommand.singleKeyword( 'delete',
                                  helpdesc='Persistent configuration to delete' ),
      'preserve-only' : ( 'Persistent configuration(s) to delete, '
                          'preserving all others' ),
      'delete-only' : 'Persistent configuration(s) to delete, preserving all others',
      'PRESERVE' : pcMatcher,
      'DELETE' : pcMatcher
   }
   @staticmethod
   def handler( mode, args ):
      now = 'now' in args
      preserveOnly = deleteOnly = preserveOrDelete = None

      if 'preserve-only' in args:
         preserveOnly = args[ 'PRESERVE' ]
      elif 'delete-only' in args:
         deleteOnly = args[ 'DELETE' ]
      else:
         preserveOrDelete = []
         if 'preserve' in args:
            preserveOrDelete.append( ( 'preserve', args[ 'PRESERVE' ] ) )
         if 'delete' in args:
            preserveOrDelete.append( ( 'delete', args[ 'DELETE' ] ) )

      erasePersistentConfig( mode, now=now,
                             preserveOrDelete=preserveOrDelete,
                             preserveOnly=preserveOnly,
                             deleteOnly=deleteOnly )

em.addCommandClass( ErasePersistent )

#####################################################################################
## Implementation of the "obsolete" configure network command
##     configure network <src-url>
#####################################################################################
class ConfigureNetwork( CliCommand.CliCommandClass ):
   syntax = "configure network [ URL ]"
   data = {
      'configure' : CliToken.Configure.configureParseNode,
      'network' : 'Read Configuration from the Network',
      'URL' : Url.UrlMatcher( lambda fs: True, 'Source file URL',
                              notAllowed=[ 'startup-config', 'running-config' ],
                              allowAllPaths=True )
      }
   @staticmethod
   def handler( mode, args ):
      url = args.get( 'URL' )
      if url is None or url.localFilename():
         mode.addError( 'Please use copy <url> running-config' )
      else:
         copyFile( mode, url,
                   FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ) ) )

em.addCommandClass( ConfigureNetwork )

#####################################################################################
## Implementation of the rename command
##     rename <src-url> <dst-url>
#####################################################################################
class RenameCmd( CliCommand.CliCommandClass ):
   syntax = "rename SRC_URL DST_URL"
   data = {
      "rename" : 'Move a file',
      "SRC_URL" : Url.UrlMatcher( lambda fs: fs.supportsRename(),
                                  'Source file name',
                                  allowAllPaths=True ),
      "DST_URL" : Url.UrlMatcher( lambda fs: fs.supportsRename(),
                                  'Destination file name',
                                  allowAllPaths=True )
      }
   @staticmethod
   def handler( mode, args ):
      surl = args[ 'SRC_URL' ]
      durl = args[ 'DST_URL' ]
      FileCliUtil.checkUrl( surl )
      FileCliUtil.checkUrl( durl )
      try:
         if surl.fs != durl.fs:
            raise OSError( 0, "Cannot rename between filesystems" )
         surl.renameto( durl )

      except OSError as e:
         mode.addError( "Error renaming {} to {} ({})".format(
            surl.url, durl.url, e.strerror ) )

em.addCommandClass( RenameCmd )

#####################################################################################
## Implementation of the dir command
##     dir [/recursive] [/all] [<url> | all-filesystems]
#####################################################################################
monthNameMap = { 1: 'Jan', 2: 'Feb', 3: 'Mar', 4: 'Apr', 5: 'May', 6: 'Jun',
                 7: 'Jul', 8: 'Aug', 9: 'Sep', 10: 'Oct', 11: 'Nov', 12: 'Dec' }

def getFileEntry( url ):
   filename = url.basename()

   if not url.exists():
      if url.islink():
         permissionStr = '----'
         size = 0
         mtime = None
         date = '-'
      else:
         raise OSError( errno.ENOENT, os.strerror( errno.ENOENT ) )
   else:
      mtime = url.date()
      if not mtime:
         date = '<no date>'
      else:
         t = time.localtime( mtime )
         currentTime = time.localtime()

         date = monthNameMap[ t[ 1 ] ] + " " + str( t[ 2 ] ) + " "
         if t[ 0 ] != currentTime[ 0 ]:
            date += '%5d' % t[ 0 ]
         else:
            date += '%02d:%02d' % ( t[ 3 ], t[ 4 ] )

      permissions = [ '-' ] * 4
      if url.permission()[ 0 ]:
         permissions[ 0 ] = 'd'
      if url.permission()[ 1 ]:
         permissions[ 1 ] = 'r'
      if url.permission()[ 2 ]:
         permissions[ 2 ] = 'w'
      if url.permission()[ 3 ]:
         permissions[ 3 ] = 'x'
      permissionStr = ''.join( permissions )
      size = url.size()

   return ( permissionStr, size, mtime, date, filename )

def printFileEntry( printer, permissionStr, size, mtime, date,
                    filename, url, recursive, showHidden ):
   file = StringAttrInfo( printer, "file" )
   file.setValue( filename )
   file.isKey()
   fPermissions = StringAttrInfo( printer, "permissions" )
   fPermissions.setValue( permissionStr )
   fSize = IntAttrInfo( printer, "size" )
   fSize.setValue( size )
   if mtime is not None:
      fmtime = FloatAttrInfo( printer, "mtime" )
      fmtime.setValue( mtime )
   try:
      # We want to raise the OSError before we start streaming if there is any
      if url.isdir() and recursive:
         urls = [ url.child( f ) for f in url.listdir() ]

      # This is not the root directory, we include this under current "entries"
      with printer.dictEntry( file.getArg() ):
         printer.addAttributes( "%s", fPermissions.getArg() )
         printer.addAttributes( "%d", fSize.getArg() )
         if mtime is not None:
            printer.addAttributes( "%f", fmtime.getArg() )
         if url.islink():
            link = StringAttrInfo( printer, "link" )
            link.setValue( str( url.readlink() ) )
            printer.addAttributes( "%s", link.getArg() )
         # Recursively print out file entries only if
         # it is a directory and we want to display it
         if url.isdir() and recursive:
            with printer.dict( "entries" ):
               # Avoid following links which leads into infinite listing issue
               # If current url is link, it means that we already streamed the
               # link once and we don't want to stream it again.
               if urls and not url.islink():
                  doListUrls( printer, urls, recursive, showHidden )
   except OSError as e:
      # Display the error under the entry to continue listing files and directories
      fErrorMsg = StringAttrInfo( printer, "errorMsg" )
      fErrorMsg.setValue( e.strerror )
      with printer.dictEntry( file.getArg() ):
         printer.addAttributes( "%s", fErrorMsg.getArg() )

def handleUrls( urls, recursive, showHidden ):
   dirs = []
   files = []
   entries = []
   for u in urls:
      if u.basename().startswith( "." ) and not showHidden:
         continue
      entries.append( u )
      if u.isdir() and recursive:
         dirs.append( u )
      else:
         files.append( u )

   return ( dirs, files, entries )

def doListUrls( printer, urls, recursive, showHidden, explicit=False ):
   ( dirs, files, entries ) = handleUrls( urls, recursive, showHidden )
   if files:
      printer.addFrills( "\n" )

   # Both files and directories will be handled by printFileEntry(...)
   # for streaming json output recursively
   if printer.outputFormat == CliPrint.JSON:
      files = entries

   for u in sorted( files ):
      try:
         TacSigint.check()
         ( permissionStr, size,
           mtime, date, filename ) = getFileEntry( u )
         # Streaming for text and json output is separated because for
         # text ouput file modification time is printed out as string
         # in date format whereas for json output it is a float.
         if printer.outputFormat == CliPrint.TEXT:
            printer.addFrills( '       %4s %11d %22s  %s\n' % ( permissionStr, size,
                                                             date, filename ) )
         else:
            # Streaming file entries recursively for JSON output
            printFileEntry( printer, permissionStr, size, mtime, date,
                            filename, u, recursive, showHidden )
      except OSError as e:
         if ( not explicit and ( e.errno == errno.ENOENT ) ):
            # This just means that a file was deleted after we made our
            # pass to collect file names, and before we got back to actually
            # extract the size and permissions etc.
            pass
         else:
            # Print error message under the currrent file entry without disturbing
            # the streaming process
            fErrorMsg = StringAttrInfo( printer, "errorMsg" )
            fErrorMsg.setValue( e.strerror )
            printer.addAttributes( f"Error listing file entry {u} %s",
                                   fErrorMsg.getArg() )

   # This is for text output, we recurse into sub-directories
   # to continue printing out file entries
   if printer.outputFormat == CliPrint.TEXT:
      for url in sorted( dirs ):
         try:
            printer.addFrills( "\nDirectory of %s\n", url )
            urls = [ url.child( f ) for f in url.listdir() ]
            if not urls:
               # Avoid recursing if the directory is empty
               printer.addFrills( "\nNo files in directory\n" )
            elif not url.islink():
               # Avoid following links which leads into infinite listing issue
               # If the current url is link, it means that we already list it
               # once and we don't want to list it again.
               doListUrls( printer, urls, recursive, showHidden )
         except OSError as e:
            # Print error message under the currrent directory without disturbing
            # the streaming process
            printer.addFrills( "\n" )
            printer.addFrills( f"Error listing directory {url} ({e.strerror})" )
            printer.addFrills( "\n" )

def doListDirOnFilesystem( mode, printer, fUrl, url,
                           recursive, showHidden, explicit ):
   fUrl.setValue( url )
   printer.addAttributes( "Directory of %s\n", fUrl.getArg() )
   with printer.dictEntry( fUrl.getArg() ):
      if url.isWildcard():
         explicit = False
         urls = url.expandWildcard()
         if not urls:
            printer.addFrills( "\nNo such file or directory\n" )
         with printer.dict( "entries" ):
            doListUrls( printer, urls, recursive, showHidden, explicit=explicit )
      else:
         try:
            if url.isdir():
               urls = [ url.child( f ) for f in url.listdir() ]
               if not urls:
                  printer.addFrills( "\nNo files in directory\n" )
            else:
               urls = [ url ]
            with printer.dict( "entries" ):
               doListUrls( printer, urls, recursive, showHidden, explicit=explicit )
         except OSError as e:
            fErrorMsg = StringAttrInfo( printer, "errorMsg" )
            fErrorMsg.setValue( e.strerror )
            printer.addFrills( "\n" )
            printer.addAttributes( f"Error listing directory {url} (%s)",
                                   fErrorMsg.getArg() )
            printer.addFrills( "\n" )

      if url.localFilename( False ):
         urlStr = Url.filenameToUrl( url.localFilename( False ) )
         fs = Url.getFilesystem( urlStr.split( ':', 1 )[ 0 ] + ':' )
      else:
         fs = url.fs
      ( size, free ) = fs.stat()

      printer.addFrills( "\n" )
      if printer.outputFormat == CliPrint.JSON:
         fSysName = StringAttrInfo( printer, "name" )
         fSysSize = IntAttrInfo( printer, "totalBytes" )
         fSysFree = IntAttrInfo( printer, "freeBytes" )
         fSysName.setValue( fs.scheme )
         fSysSize.setValue( size )
         fSysFree.setValue( free )
         with printer.dict( "filesystem" ):
            printer.addAttributes( "%s", fSysName.getArg() )
            printer.addAttributes( "%d", fSysSize.getArg() )
            printer.addAttributes( "%d", fSysFree.getArg() )
      if size != 0:
         printer.addFrills( f"{size} bytes total ({free} bytes free)"
                            f" on {fs.scheme}\n" )
      else:
         printer.addFrills( "No space information available\n" )

def listDirStreaming( mode, printer, degrade, url, recursive, showHidden ):
   # If degrade to revision 1, output goes to msg as a single text string
   # We fool the printer to generate text output first and grab the content.
   # Then switch to JSON printer at the end.
   if degrade:
      msg = ''
      printer = Printer( CliPrint.TEXT )

   fUrl = StringAttrInfo( printer, "urls" )
   fUrl.isKey()

   if url == 'all-filesystems':
      for fs in sorted( Url.filesystems() ):
         if fs.fsType in [ 'flash', 'system' ]:
            # Note that the directory listing on the current filesystem will be of
            # the current directory, not the root directory.  This is
            # industry-standard.
            try:
               u = Url.parseUrl( fs.scheme,
                                 Url.Context( *Url.urlArgsFromMode( mode ) ) )
               if degrade:
                  with ArPyUtils.StdoutAndStderrInterceptor() as f:
                     doListDirOnFilesystem( mode, printer, fUrl, u,
                                            recursive, showHidden, False )
                  msg += f.contents()
                  msg += '\n'
               else:
                  doListDirOnFilesystem( mode, printer, fUrl, u,
                                         recursive, showHidden, False )
                  printer.addFrills( "\n" )
            except ValueError:
               # The filesystem was removed in the window between calling
               # Url.filesystems() and calling Url.parseUrl().
               pass
   else:
      if degrade:
         with ArPyUtils.StdoutAndStderrInterceptor() as f:
            doListDirOnFilesystem( mode, printer, fUrl, url,
                                   recursive, showHidden, False )
         msg += f.contents()
      else:
         doListDirOnFilesystem( mode, printer, fUrl, url,
                                recursive, showHidden, True )

   # Streaming for revision 1
   if degrade:
      printer = Printer( CliPrint.JSON )
      message = StringAttrInfo( printer, "" )
      message.setValue( msg )
      printer.addAttributes( "%s", message.getArg() )

def listDir( mode, url, recursive=False, showHidden=False ):
   # List the files and directories at a URL.  If the recursive flag is true, all
   # files under the directory are listed, recursively.  If the all flag is true,
   # hidden files will also be listed.
   if url is None:
      url = mode.session.currentDirectory()
   if hasattr( url, 'checkAllowedPathEnabledIs' ):
      url.checkAllowedPathEnabledIs( False )
   FileCliUtil.checkUrl( url )
   revision = mode.session_.requestedModelRevision()
   printer = Printer( mode.session_.outputFormat() )
   degrade = printer.outputFormat == CliPrint.JSON and revision == 1
   # If the url does not exist, raise error before starting to stream
   # A wildcard url might not exist, but its parent dir path must
   # (no wildcard match is not an error).
   if url != 'all-filesystems' and not url.exists() and \
      ( not url.isWildcard() or not url.parent().exists() ):
      raise OSError( errno.ENOENT, os.strerror( errno.ENOENT ) )
   if degrade:
      printer.start()
      with printer.list( "messages" ):
         listDirStreaming( mode, printer, degrade, url, recursive, showHidden )
      printer.end()
   else:
      printer.start()
      with printer.dict( "urls" ):
         listDirStreaming( mode, printer, degrade, url, recursive, showHidden )
      printer.end()

class DirCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "dir [ /recursive ] [ /all ] [ all-filesystems | URL ]"
   data = {
      'dir' : 'Disply details about files',
      '/recursive' : 'List files recursively',
      '/all' : 'List all files, including hidden files',
      'all-filesystems' : 'List files on all filesystems',
      'URL' : Url.UrlMatcher( lambda fs: fs.supportsListing(),
                              'The file or directory or interest',
                              notAllowed=[ '/all', '/recursive', 'all-filesystems' ],
                              allowAllPaths=True )
      }
   cliModel = FileDir

   @staticmethod
   def handler( mode, args ):
      url = args.get( 'URL' ) or args.get( 'all-filesystems' )
      recursive = '/recursive' in args
      showHidden = '/all' in args
      try:
         listDir( mode, url, recursive, showHidden )
         return FileDir
      except OSError as e:
         if e.errno != errno.EPIPE:
            if e.errno == errno.ENOENT:
               mode.addMessage( 'Directory of %s\n' % url )
            mode.addError( f"Error listing directory {url} ({e.strerror})" )
         return None

em.addCommandClass( DirCmd )

#####################################################################################
## Implementation of the show file information command
##     show file information <url>
#####################################################################################
showFileKw = CliMatcher.KeywordMatcher( 'file',
                                        helpdesc='Details about files' )

class ShowFileInfoCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show file information URL"
   data = {
      'file' : showFileKw,
      'information' : 'Details about the file',
      'URL' : Url.UrlMatcher( lambda fs: fs.supportsListing(), 'File path',
                              allowAllPaths=True )
   }
   privileged = True
   cliModel = FileInformation

   @staticmethod
   def handler( mode, args ):
      url = args[ 'URL' ]
      FileCliUtil.checkUrl( url )
      info = FileInformation()
      try:
         # permission() raises an EnvironmentError if the file doesn't exist.
         ( isdir, readable, _, executable ) = url.permission()
         if isdir:
            info.isDir = True
         else:
            if executable:
               info.isExecutable = True
            filename = url.localFilename()  # None if file is remote.
            if not url.size():
               info.isEmpty = True
            elif readable and filename:
               info.fileName = filename
               with url.open( mode='rb' ) as contents:
                  magic = contents.read( 4 )
               if magic == b'\x7fELF':
                  info.fileType = 'elf'
               elif magic == b'PK\x03\x04':
                  if filename.endswith( '.swi' ):
                     import EosVersion # pylint: disable=import-outside-toplevel
                     info.fileType = 'swi'
                     version = EosVersion.swiVersion( filename )
                     info.version = version.internalVersion() or '<unknown>'
                  elif filename.endswith( '.swix' ):
                     info.fileType = 'swix'
                  else:
                     info.fileType = 'zip'
               elif magic.startswith( b'#!' ):
                  if filename.endswith( '.py' ):
                     info.fileType = 'pythonScript'
                  else:
                     info.fileType = 'script'
         info.path = str( url )
         return info
      except OSError as e:
         if isinstance( url, Url.Url ):
            url = url.url
         # don't commit a double fault (can't catch them twice!?!)
         if e.errno != errno.EPIPE:
            mode.addError( "Error getting information about %s (%s)" \
                           % ( url, e.strerror ) )
         return None

BasicCli.addShowCommandClass( ShowFileInfoCmd )

#####################################################################################
## Implementation of the show file systems command
##     show file systems
#####################################################################################
class ShowFileSystemCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show file systems"
   data = {
      'file' : showFileKw,
      'systems' : 'List filesystems'
   }
   privileged = True
   cliModel = FileSystems

   @staticmethod
   def handler( mode, args ):
      fSystems = FileSystems()
      try:
         currentFs = mode.session.currentDirectory().fs
         for fs in sorted( Url.filesystems() ):
            # those have no size/perm (not interesting)
            if fs.scheme in [ 'terminal:' ]:
               continue
            fs1 = FileSystem()
            fs1.currentFs = fs == currentFs
            ( size, free ) = fs.stat()
            _, _, linuxFs = fs.mountInfo()
            if fs.fsType == 'flash':
               try:
                  fs1.linuxFs = linuxFs
               except ValueError:
                  pass
            size //= 1024
            free //= 1024
            fs1.size = size
            fs1.free = free
            fs1.fsType = fs.fsType
            fs1.permission = fs.permission
            fs1.prefix = fs.scheme
            fSystems.fileSystems.append( fs1 )
         return fSystems

      except OSError as e:
         # don't commit a double fault (can't catch them twice!?!)
         if e.errno != errno.EPIPE:
            mode.addError( "Error listing filesystems (%s)" % e.strerror )
         return None

BasicCli.addShowCommandClass( ShowFileSystemCmd )

#####################################################################################
## Implementation of the eject command
##     eject <filesystem>
#####################################################################################
def _isEjectable( fs ):
   return fs.fsType == 'flash' and fs.removable

def eject( mode, targetFsName ):
   def _unregisterUrlFilesystem( fs, mntPoint ):
      # Unmount first so Url filesystems are in a consistent state
      # (eject might fail in the middle of unmounting filesystems)
      try:
         Tac.run(
            [ 'umount', mntPoint ], stdout=Tac.DISCARD, stderr=Tac.DISCARD,
            asRoot=True )
      except Tac.SystemCommandError:
         mode.addError( 'Failed to unmount ' + str( fs.scheme ) )
         raise

      Url.unregisterFilesystem( fs )

      # Clean up mount point and .conf file
      # (if both are present, the Url filesystem will be re-registered)
      failedRmdir = False
      try:
         Tac.run(
            [ 'rmdir', mntPoint ], stdout=Tac.DISCARD, stderr=Tac.DISCARD,
            asRoot=True )
      except Tac.SystemCommandError:
         failedRmdir = True
      try:
         Tac.run(
            [ 'rm', mntPoint + '.conf' ], stdout=Tac.DISCARD, stderr=Tac.DISCARD,
            asRoot=True )
      except Tac.SystemCommandError:
         if failedRmdir:
            mode.addError( 'Failed to remove record of ' + str( fs.scheme ) )

   def _getDeviceDevPath( fsDevPath ):
      deviceName = os.path.basename( os.path.dirname( os.path.realpath(
         os.path.join( '/sys/class/block', os.path.basename( fsDevPath ) ) ) ) )
      deviceDevPath = os.path.join( '/dev', deviceName )
      if ( os.path.exists( deviceDevPath ) and
           stat.S_ISBLK( os.stat( deviceDevPath ).st_mode ) ):
         return deviceDevPath
      return fsDevPath

   targetFs = Url.getFilesystem( targetFsName )
   if targetFs is None or not _isEjectable( targetFs ):
      mode.addError( 'Filesystem ' + targetFsName + ' no longer valid' )
      return
   targetDevPath, mntPoint, _ = targetFs.mountInfo()
   toUnregister = [ ( targetFs, mntPoint ) ]
   deviceDevPath = _getDeviceDevPath( targetDevPath )
   for fs in Url.filesystems():
      if fs is not targetFs:
         devPath, mntPoint, _ = fs.mountInfo()
         if ( isinstance( devPath, str ) and
              _getDeviceDevPath( devPath ) == deviceDevPath ):
            toUnregister.append( ( fs, mntPoint ) )
   try:
      for fs, mntPoint in toUnregister:
         _unregisterUrlFilesystem( fs, mntPoint )
      Tac.run(
         [ 'eject', targetDevPath ], stdout=Tac.DISCARD, stderr=Tac.DISCARD,
         asRoot=True )
   except Tac.SystemCommandError:
      mode.addError( 'Failed to eject device with ' + targetFsName )
      return
   # Give device buffer a chance to flush
   time.sleep( 1 )

class EjectCmd( CliCommand.CliCommandClass ):
   syntax = "eject TARGET"
   data = {
      'eject' : 'Eject storage device for safe removal',
      'TARGET' : CliMatcher.DynamicKeywordMatcher(
         lambda mode: { str( fs.scheme ): 'Filesystem' for fs in Url.filesystems() if
                        _isEjectable( fs ) } )
      }
   @staticmethod
   def handler( mode, args ):
      eject( mode, args[ 'TARGET' ] )

em.addCommandClass( EjectCmd )

#####################################################################################
## Implementation of the cd command
##     cd [<url>]
#####################################################################################
def setCurrentDirectory( mode, url ):
   session = mode.session
   if url is None:
      url = url or session.homeDirectory()
   FileCliUtil.checkUrl( url )
   try:
      try:
         if not url.exists():
            raise OSError( errno.ENOENT, os.strerror( errno.ENOENT ) )
         if not url.isdir():
            raise OSError( errno.ENOTDIR, os.strerror( errno.ENOTDIR ) )

         mode.session.currentDirectoryIs(
            Url.parseUrl( str( url ),
                          Url.Context( *Url.urlArgsFromMode( mode ) ) ) )
         # The reason we call Url.parseUrl( str( url ) ) is so that the .url
         # attribute of the Url object we store is the canonical string
         # representation of the URL, not whatever the user happened to type in after
         # 'cd' (which might be a relative path like 'foo/bar').  This rarely
         # matters, but otherwise any error messages from 'dir' are a bit odd.
      except ValueError:
         # Unfortunately, doing this opens up a tiny window where the filesystem
         # we're cd'ing to is removed between Url.parseUrl() being called by the CLI
         # rule, and the call to Url.parseUrl() above.  We handle that here.
         # pylint: disable-next=raise-missing-from
         raise OSError( 0, "Filesystem no longer exists" )
   except OSError as e:
      mode.addError( 
         "Error changing current directory to {} ({})".format(
            url.url, e.strerror ) )

class CdCmd( CliCommand.CliCommandClass ):
   syntax = "cd [ URL ]"
   data = {
      'cd' : 'Move to another directory',
      'URL' : Url.UrlMatcher( lambda fs: fs.supportsListing(), 'Directory name',
                              allowAllPaths=True )
      }
   @staticmethod
   def handler( mode, args ):
      setCurrentDirectory( mode, args.get( 'URL' ) )

em.addCommandClass( CdCmd )

#####################################################################################
## Implementation of the pwd command
##     pwd
#####################################################################################
class PwdCmd( CliCommand.CliCommandClass ):
   syntax = "pwd"
   data = {
      'pwd' : 'Show the CLI shell current directory'
   }
   @staticmethod
   def handler( mode, args ):
      print( mode.session.currentDirectory() )

em.addCommandClass( PwdCmd )

#####################################################################################
## Implementation of the more command
##     more <url>
## 
## We do not currently support the industry-standard /ascii, /binary or /ebcdic 
## flags.
#####################################################################################
moreUrl = FileCliUtil.registerUrlExpression( 'Name of file to print',
      lambda fs: fs.scheme != 'terminal:' and fs.supportsRead(),
      notAllowed=[], allowAllPaths=True )

class MoreCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "more URL"
   data = {
      "more" : 'Print file contents',
      "URL" : moreUrl
      }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      FileCliUtil.showFile( mode, args[ 'URL' ] )

BasicCli.addShowCommandClass( MoreCmd )

#####################################################################################
## Implementation of the delete command
##     delete [/recursive] <url>
## 
## We do not currently support the industry-standard /force flag.
#####################################################################################
deleteKwMatcher = CliMatcher.KeywordMatcher( 'delete', helpdesc='Remove a file' )

def deleteFile( mode, url, recursive=False ):
   FileCliUtil.checkUrl( url )
   try:
      if url.isWildcard():
         urls = url.expandWildcard()
         if not urls:
            print( "No such file" )
            return
      else:
         urls = [ url ]

      for u in urls:
         u.delete( recursive )
   except OSError as e:
      mode.addError( "Error deleting {} ({})".format(
            url.url, e.strerror ) )

class DeleteCmd( CliCommand.CliCommandClass ):
   syntax = "delete [ /recursive ] URL"
   data = {
      'delete' : deleteKwMatcher,
      '/recursive' : 'Delete files recursively',
      'URL' : Url.UrlMatcher( lambda fs: fs.supportsDelete(),
                              'Name of file being removed',
                              notAllowed=[ '/recursive', 'startup-config' ],
                              allowAllPaths=True )
   }
   @staticmethod
   def handler( mode, args ):
      deleteFile( mode, args[ 'URL' ], '/recursive' in args )

em.addCommandClass( DeleteCmd )

#####################################################################################
## Implementation of the mkdir command
##     mkdir <url>
#####################################################################################
class MkdirCmd( CliCommand.CliCommandClass ):
   syntax = "mkdir URL"
   data = {
      'mkdir' : 'Create a directory',
      'URL' : Url.UrlMatcher( lambda fs: fs.supportsMkdir(), 'Directory name',
                              allowAllPaths=True ),
      }
   @staticmethod
   def handler( mode, args ):
      url = args[ 'URL' ]
      FileCliUtil.checkUrl( url )
      try:
         url.mkdir()
      except OSError as e:
         mode.addError( "Error creating directory {} ({})".format(
            url.url, e.strerror ) )

em.addCommandClass( MkdirCmd )

#####################################################################################
## Implementation of the rmdir command
##     rmdir <url>
#####################################################################################
class RmdirCmd( CliCommand.CliCommandClass ):
   syntax = "rmdir URL"
   data = {
      'rmdir' : 'Delete a directory',
      'URL' : Url.UrlMatcher( lambda fs: fs.supportsMkdir(), 'Directory name',
                              allowAllPaths=True ),
      }
   @staticmethod
   def handler( mode, args ):
      url = args[ 'URL' ]
      FileCliUtil.checkUrl( url )
      try:
         url.rmdir()
      except OSError as e:
         mode.addError( "Error deleting directory {} ({})".format(
            url.url, e.strerror ) )

em.addCommandClass( RmdirCmd )

#####################################################################################
## Implementation of the verify command
##     verify [ /md5 | /sha512 ] URL
##
## Verifies that <url> is a SWI signed by Arista such that it is authorized by Arista
## for release and has not been altered post release by checking the swi-signature.
#####################################################################################
verifyKwMatcher = CliMatcher.KeywordMatcher( 'verify',
                             helpdesc='Verify the state of an object in the system' )
hashUrlMatcher = Url.UrlMatcher( lambda fs: fs.supportsRead(), "File to verify",
                                 allowAllPaths=True )

def verifySwi( mode, url ):
   FileCliUtil.checkUrl( url )

   localFilename = url.localFilename()
   if not localFilename:
      mode.addError( "Error: file %s not found" % url )
      return

   sigValid, sigError, _ = SwiSignLib.verifySwiSignature( localFilename,
                                                          userHint=True )
   if sigValid:
      print( "Verifying %s successful." % url.url )
   else:
      mode.addError( sigError )

verifyHook = CliExtensions.CliHook()

def verifyHash( mode, url, hashInitializer, hashName ):
   """
   Generic function for verifying a hash.
   Takes an openssl hashInitializer function pointer
   and the name of the hash function pointer.
   """
   FileCliUtil.checkUrl( url )

   for i in verifyHook.extensions():
      if not i( mode, hashName ):
         return None

   if not url.exists():
      mode.addError( "Error: file %s not found" % url )
      return None

   try:
      hashValue = url.verifyHash( hashName, mode, hashInitializer )
      return hashValue
   except OSError as e:
      mode.addError( f"Error reading {url.url} ({e.strerror})" )
      return None

class VerifyCmd( CliCommand.CliCommandClass ):
   syntax = "verify [ /md5 | /sha512 ] URL"
   data = {
      'verify' : verifyKwMatcher,
      '/md5' : 'Calculate MD5 hash',
      '/sha512' : 'Calculate SHA-512 hash',
      'URL' : hashUrlMatcher
      }
   @staticmethod
   def handler( mode, args ):
      url = args[ 'URL' ]
      if '/md5' in args:
         hashValue = verifyHash( mode, url, hashlib.md5, "md5" )
         if hashValue is not None:
            print( "verify /{} ({}) = {}".format( "md5", url.url, hashValue ) )
      elif '/sha512' in args:
         hashValue = verifyHash( mode, url, hashlib.sha512, "sha512" )
         if hashValue is not None:
            print( "verify /{} ({}) = {}".format( "sha512", url.url, hashValue ) )
      else:
         verifySwi( mode, url )

em.addCommandClass( VerifyCmd )

#####################################################################################
## Implementation of the show file digest HASHTYPE URL command
##      show file digest HASHTYPE URL
##
## Similiar to Verify command, this command calculate and shows the hash value of the
## file specified. Structured json format output for displaying file's hash value is
## demanded. However, converting Verify command to support CAPI changes the default
## eapi request responses which will affect existing users.
#####################################################################################
runningConfigDigestKw = CliMatcher.KeywordMatcher(
   'digest', 'Display message digest of running-config',
   # Do not prevent "di" from autocompleting to "diffs"
   autoCompleteMinChars=3 )

class ShowFileDigest( ShowCommand.ShowCliCommandClass ):
   syntax = "show file digest HASHTYPE URL"
   data = {
      'file': showFileKw,
      'digest': runningConfigDigestKw,
      'HASHTYPE': CliMatcher.EnumMatcher( {
         'md5': 'Calculate MD5 hash',
         'sha512': 'Calculate SHA-512 hash',
      } ),
      'URL': hashUrlMatcher
      }
   cliModel = FileDigest

   @staticmethod
   def handler( mode, args ):
      url = args[ 'URL' ]
      hashType = args[ 'HASHTYPE' ]
      fDigest = FileDigest()
      fDigest.path = url.url
      fDigest.hashType = hashType
      if hashType == 'md5':
         fDigest.hashValue = verifyHash( mode, url, hashlib.md5, "md5" )
      elif hashType == 'sha512':
         fDigest.hashValue = verifyHash( mode, url, hashlib.sha512, "sha512" )

      return fDigest

BasicCli.addShowCommandClass( ShowFileDigest )

#####################################################################################
##     show [ secure-monitor ] running-config [sanitized]
#####################################################################################
runningConfigAfterShowKw = CliMatcher.KeywordMatcher(
   'running-config',
   helpdesc='System running configuration' )
sanitizedKwMatcher = CliMatcher.KeywordMatcher(
   "sanitized", helpdesc='Sanitized Output' )

runningConfigAfterShowKw = CliMatcher.KeywordMatcher(
   'running-config',
   helpdesc='System running configuration' )

def showRunningConfig( mode, secureMonitor=False,
                       showDiffs=False, showSanitized=False,
                       showFilteredRoot=False ):
   # `shouldPrint` tells us the caller requested text output
   showJson = not mode.session_.shouldPrint()

   if showDiffs:
      assert not showJson, 'JSON is not supported for diffs'
      runningConfigUrl = FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ),
            secureMonitor=secureMonitor, showSanitized=showSanitized,
            showJson=showJson, showFilteredRoot=showFilteredRoot )
      startupConfigUrl = FileUrl.localStartupConfig(
         *Url.urlArgsFromMode( mode ), secureMonitor=secureMonitor )
      # For 'show run diff', Normalize spaces: remove at end of line, squizze double
      # spaces inside, but keep leading spaces. See BUG384651: sometimes extra spaces
      # are indaventely added/removed, for example when code like "%s %s" % (a, b)
      # where b can be "" is refactored and no longer leads to a double space.
      FileCliUtil.diffFile( mode, startupConfigUrl, runningConfigUrl )
      return None

   if secureMonitor:
      runningConfigUrl = FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ),
            secureMonitor=secureMonitor, showSanitized=showSanitized,
            showJson=showJson, showFilteredRoot=showFilteredRoot )
      if mode.session_.shouldPrint():
         showCommandAndTime( mode, 'show secure-monitor running-config' )

      FileCliUtil.showFile( mode, runningConfigUrl )
      return CliModel.noValidationModel( ShowRunOutputModel.Mode )
   else:
      if showFilteredRoot:
         cmd = 'show running-config (dynamic)'
      else:
         cmd = 'show running-config'

      return showRunningConfigCommon( mode, cmd,
            secureMonitor=secureMonitor,
            showSanitized=showSanitized,
            showFilteredRoot=showFilteredRoot,
            showJson=showJson )

class ShowRunningConfig( ShowCommand.ShowCliCommandClass ):
   syntax = "show [ secure-monitor ] running-config [ sanitized ]"
   data = {
      "secure-monitor" : secureMonitorKw,
      "running-config" : runningConfigAfterShowKw,
      "sanitized" : sanitizedKwMatcher,
   }
   privileged = True
   cliModel = ShowRunOutputModel.Mode

   @staticmethod
   def handler( mode, args ):
      return showRunningConfig( mode,
                                secureMonitor=( 'secure-monitor' in args ),
                                showSanitized=( 'sanitized' in args ) )

BasicCli.addShowCommandClass( ShowRunningConfig )

#####################################################################################
##     show [ secure-monitor ] running-config diffs
#####################################################################################

runningConfigDiffsKw = CliMatcher.KeywordMatcher(
   'diffs', helpdesc='Differences from startup-config' )

class ShowRunningConfigDiffs( ShowCommand.ShowCliCommandClass ):
   syntax = "show [ secure-monitor ] running-config diffs"
   data = {
      "secure-monitor" : secureMonitorKw,
      "running-config" : runningConfigAfterShowKw,
      "diffs" : runningConfigDiffsKw
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      return showRunningConfig( mode,
                                secureMonitor=( 'secure-monitor' in args ),
                                showDiffs=True )

BasicCli.addShowCommandClass( ShowRunningConfigDiffs )

#####################################################################################
##     show [ secure-monitor ] running-config digest
#####################################################################################

class HashFile:
   """Pseudo file object for hash"""
   def __init__( self, hashObj ):
      self.hashObj_ = hashObj

   def write( self, data ):
      self.hashObj_.update( data if isinstance( data, bytes ) else data.encode() )

   def writelines( self, data ):
      for d in data:
         self.hashObj_.update( d if isinstance( d, bytes ) else d.encode() )

   def close( self ):
      self.hashObj_ = None

class ShowRunningConfigDigest( ShowCommand.ShowCliCommandClass ):
   syntax = "show [ secure-monitor ] running-config digest"
   data = {
      "secure-monitor" : secureMonitorKw,
      "running-config" : runningConfigAfterShowKw,
      "digest" : runningConfigDigestKw
   }
   privileged = True
   cliModel = ShowRunModel.RunningConfigDigest

   @staticmethod
   def handler( mode, args ):
      secureMonitor = 'secure-monitor' in args
      sha1 = hashlib.sha1()
      CliSave.saveRunningConfig( mode.entityManager, HashFile( sha1 ),
                                 secureMonitor=secureMonitor,
                                 showHeader=False )
      model = ShowRunModel.RunningConfigDigest()
      model.digest = sha1.hexdigest()
      return model

BasicCli.addShowCommandClass( ShowRunningConfigDigest )

#####################################################################################
## Implementation of the show running-config all [detail] command
##              show running-config all [detail]
#####################################################################################

allKwMatcher = CliMatcher.KeywordMatcher( 'all',
      helpdesc='Configuration with defaults' )
detailKwMatcher = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='Detail configuration with defaults' )
class ShowRunningConfigAll( ShowCommand.ShowCliCommandClass ):
   syntax = "show running-config all [ detail ] [ sanitized ]"
   data = {
      "running-config" : runningConfigAfterShowKw,
      "sanitized" : sanitizedKwMatcher,
      "all" : allKwMatcher,
      "detail" : detailKwMatcher,
   }
   privileged = True
   cliModel = ShowRunOutputModel.Mode

   @staticmethod
   def handler( mode, args ):
      cmd = 'show running-config all'
      return showRunningConfigCommon( mode, cmd,
            saveAll=True,
            saveAllDetail='detail' in args,
            showSanitized='sanitized' in args,
            showJson=not mode.session_.shouldPrint() )

BasicCli.addShowCommandClass( ShowRunningConfigAll )

#####################################################################################
## Implementation of the show running-config section command
##              show running-config [all] section [all] <regexp>
#####################################################################################
sectionKwMatcher = CliMatcher.KeywordMatcher(
   'section',
   helpdesc='Display sections containing matching commands' )
sectionAllKwMatcher = CliMatcher.KeywordMatcher(
   'all',
   helpdesc='Display matching sections including '
            'same-level commands and their descendents' )
sectionPatternMatcher = CliMatcher.PatternMatcher(
   r'[^\|>].*',
   helpname='REGEX',
   helpdesc='Regular expression for matching commands' )

def showRunningConfigSection( mode, regexList, showAll=False, showSanitized=False,
                              showSectionAll=False ):
   # Pylint doesn't recognize the Url members
   # pylint: disable-msg=E1103
   if showAll:
      urlFunc = FileUrl.localRunningConfigAll
   else:
      urlFunc = FileUrl.localRunningConfig
   configSectionUrl = urlFunc( *Url.urlArgsFromMode( mode ), 
                                showSanitized=showSanitized )

   sectionFilterFn = sectionFilterAll if showSectionAll else sectionFilter
   try:
      with configSectionUrl.open() as configSectionUrlFd:
         sectionFilterFn( configSectionUrlFd, regexList )
   except OSError as e:
      # don't commit a double fault (can't catch them twice!?!)
      if e.errno != errno.EPIPE:
         mode.addError( "Error displaying {} ({})".format( configSectionUrl.url,
                                                       e.strerror ) )
   except re.error as e: # pylint: disable=unused-variable
      mode.addError ( "Invalid regular expression provided" )
   # pylint: enable-msg=E1103

class ShowRunningConfigSection( ShowCommand.ShowCliCommandClass ):
   syntax = """show running-config [ sanitized ]
               [ all ] section [ SECTION_ALL ] { PATTERN }"""
   data = {
      "running-config" : runningConfigAfterShowKw,
      "sanitized" : sanitizedKwMatcher,
      "all" : allKwMatcher,
      "section" : sectionKwMatcher,
      "SECTION_ALL": sectionAllKwMatcher,
      "PATTERN" : sectionPatternMatcher,
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      showRunningConfigSection( mode, args[ 'PATTERN' ],
                                showAll=( 'all' in args ),
                                showSanitized=( 'sanitized' in args ),
                                showSectionAll='SECTION_ALL' in args )

BasicCli.addShowCommandClass( ShowRunningConfigSection )

#####################################################################################
## Implementation of the show startup-config command
##     show [ secure-monitor ] startup-config [ section [ all ] PATTERN ]
#####################################################################################
startupConfigAfterShowKw = CliMatcher.KeywordMatcher(
   "startup-config", helpdesc='Configuration used at boot' )

def showStartupConfig( mode, secureMonitor=False ):
   # The Time portion of this show command denotes the current-time
   # and hence cannot be part of the startup-config file itself.
   # This info is therefore printed before displaying the contents
   # of the file. The 'try:...' makes sure that the file is present
   # in the flash file-system before attempting any print.
   # This will avoid printing the command/date info if this
   # file is not present.
   url = FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ),
                                     secureMonitor=secureMonitor )
   try:
      sz = url.size()
      if sz:
         showCommandAndTime( mode,
                             'show secure-monitor startup-config' if secureMonitor
                             else 'show startup-config' )

      FileCliUtil.showFile( mode, url )
   except OSError as e:
      mode.addError( f"Error displaying {url.url} ({e.strerror})" )

def showStartupConfigSection( mode, regexList, secureMonitor=False,
                              showSectionAll=False ):
   # Pylint doesn't recognize the Url members
   # pylint: disable-msg=E1103
   try:
      configSectionUrl = FileUrl.localStartupConfig(
         *Url.urlArgsFromMode( mode ), secureMonitor=secureMonitor )
      sectionFilterFn = sectionFilterAll if showSectionAll else sectionFilter
      sectionFilterFn( configSectionUrl.open(), regexList )
   except OSError as e:
      # don't commit a double fault (can't catch them twice!?!)
      if e.errno != errno.EPIPE:
         mode.addError( "Error displaying {} ({})".format( configSectionUrl.url,
                                                       e.strerror ) )
   except re.error as e: # pylint: disable=unused-variable
      mode.addError ( "Invalid regular expression provided" )
   # pylint: enable-msg=E1103

class ShowStartupConfig( ShowCommand.ShowCliCommandClass ):
   syntax = "show [ secure-monitor ] startup-config [ section [ all ] { PATTERN } ]"
   data = {
      "secure-monitor" : secureMonitorKw,
      "startup-config" : startupConfigAfterShowKw,
      "section" : sectionKwMatcher,
      "PATTERN" : sectionPatternMatcher,
      "all" : sectionAllKwMatcher,
      }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      secureMonitor = "secure-monitor" in args
      pattern = args.get( 'PATTERN' )
      if pattern:
         showStartupConfigSection( mode, pattern, secureMonitor=secureMonitor,
                                   showSectionAll='all' in args )
      else:
         showStartupConfig( mode, secureMonitor=secureMonitor )

BasicCli.addShowCommandClass( ShowStartupConfig )

#####################################################################################
## Implementation of the erase startup-config command
##     erase startup-config [ secure-monitor ]
#####################################################################################
class DeprecatedEraseCmd( CliCommand.CliCommandClass ):
   syntax = "erase | delete startup-config [ now ]"
   data = {
      'erase' : eraseDeprecatedKw,
      'delete' : deletekKw,
      'startup-config' : 'Erase contents of startup configuration',
      'now' : nowKwMatcher
      }
   @staticmethod
   def handler( mode, args ):
      now = 'now' in args
      eraseStartupConfig( mode, now=now )

em.addCommandClass( DeprecatedEraseCmd )

#####################################################################################
## [no] service running-config timestamp
#####################################################################################
runningConfigKwMatcher = CliMatcher.KeywordMatcher( 'running-config',
                            helpdesc="Configure the current running configuration" )

class ServiceRunningConfigTimestamp( CliCommand.CliCommandClass ):
   syntax = "service running-config timestamp"
   noOrDefaultSyntax = syntax
   data = {
      'service' : CliToken.Service.serviceKw,
      'running-config' : runningConfigKwMatcher,
      'timestamp' : "Show the current time in the running-config"
      }
   @staticmethod
   def handler( mode, args ):
      serviceConfig.timeInRunningConfig = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      serviceConfig.timeInRunningConfig = False

BasicCli.GlobalConfigMode.addCommandClass( ServiceRunningConfigTimestamp )

#####################################################################################
# [no] service running-config cache disabled
#####################################################################################
class ServiceRunningConfigCacheDisabled( CliCommand.CliCommandClass ):
   syntax = "service running-config cache disabled"
   noOrDefaultSyntax = syntax
   data = {
      'service': CliToken.Service.serviceKw,
      'running-config': runningConfigKwMatcher,
      'cache': 'Cache running-config output',
      'disabled': 'Disable caching running-config'
      }

   @staticmethod
   def handler( mode, args ):
      serviceConfig.runningConfigCacheDisabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      serviceConfig.runningConfigCacheDisabled = False

BasicCli.GlobalConfigMode.addCommandClass( ServiceRunningConfigCacheDisabled )

#####################################################################################
## [no] service running-config copy use-config-session
#####################################################################################
class ServiceRunningConfigCopyUseSession( CliCommand.CliCommandClass ):
   syntax = "service running-config copy use-config-session"
   noOrDefaultSyntax = syntax
   data = {
      'service' : CliToken.Service.serviceKw,
      'running-config' : runningConfigKwMatcher,
      'copy' : "Configure copy to running-config behavior",
      'use-config-session' : "Use config session in copying to running-config"
      }
   @staticmethod
   def handler( mode, args ):
      serviceConfig.useSessionInCopyRunningConfig = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      serviceConfig.useSessionInCopyRunningConfig = False

BasicCli.GlobalConfigMode.addCommandClass( ServiceRunningConfigCopyUseSession )

#####################################################################################
## [ no | default ] service configuration verification <feature>
#####################################################################################

# These tokens are used by other packages.
serviceKw = CliToken.Service.serviceKw
configKwAfterService = CliToken.Service.configKwAfterService
verificationKwAfterService = CliMatcher.KeywordMatcher( 'verification',
                                               helpdesc='Verify configuration' )

######################################################################
# Implement show active/show active all
######################################################################

def showActiveUrl( mode, showAll=False, showDetail=False ):
   if showAll:
      return FileUrl.localRunningConfigAll( *Url.urlArgsFromMode( mode ),
                                             showDetail=showDetail )
   else:
      return FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ) )

######################################################################
# CONFIG-MAN-MIB: copy command run, check if the event should be added
# to config history table
######################################################################

def maybeAddToHistoryTable( mode, surl, durl, commandSource ):
   configDest = durl.historyEntryName()
   if configDest is None:
      return

   # internally issued; doesn't need to be added
   if commandSource == "configure replace":
      return

   configDestUrl = "" # Since the source is set, no url
   configSource = surl.historyEntryName()
   configSourceUrl = ""

   if configSource is None:
      configSource = "url"
      configSourceUrl = "url"
      # Now, determine the url value
      try:
         configSourceUrl, _path = surl.url.split( ':' )
      except ValueError:
         # No specific extension was specified, for eg: copy foo startup
         # When no extension is specified, we assume it to be a flash url
         configSourceUrl = "flash"

   mode.session_.addToHistoryEventTable( commandSource,
                                         configSource, configDest, 
                                         configSourceUrl, configDestUrl )

def handleFileCopy( mode, surl, durl, commandSource ):
   maybeAddToHistoryTable( mode, surl, durl, commandSource )

   if durl == FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ) ):
      cliStatus.startupConfigLastWriteTime = Tac.now()
      info = UtmpDump.getUserInfo()
      Logging.log( SYS_CONFIG_STARTUP,
                   surl.url, info[ 'user' ], info[ 'tty' ], info[ 'ipAddr' ] )

def isSecureBootEnabled():
   return ( abootSb.status and abootSb.status.supported and
            abootSb.status.securebootDisabled == 0 )

def runningConfigCacheEnabled():
   return not serviceConfig.runningConfigCacheDisabled

def Plugin( entityManager ):
   global cliStatus
   global serviceConfig
   abootSb.status = LazyMount.mount( entityManager,
                                     Cell.path( "aboot/sb/status" ),
                                     "Aboot::Secureboot::Status", "r" )
   cliStatus = LazyMount.mount( entityManager, "cli/status",
                                "Cli::Status", "w" )
   serviceConfig = ConfigMount.mount( entityManager, 'sys/service/config',
                                      "System::ServiceConfig", "w" )
   CliSave.runningConfigCacheEnabledHook.addExtension( runningConfigCacheEnabled )

   BasicCliModes.showActiveUrlFactory = showActiveUrl
   # CONFIG-MAN-MIB: copy command run, check if the event should be added
   # to config history table
   FileCliUtil.copyFileNotifiers.addExtension( handleFileCopy )
   FlashUrl.registerSecurebootEnabledCallback( isSecureBootEnabled )

   # -------------------------------------------------------------
   # Register show file systems command into "show tech-support".
   # -------------------------------------------------------------
   # Timestamps are made up to maintain historical order within show tech-support
   CliPlugin.TechSupportCli.registerShowTechSupportCmd(
      '2010-01-01 00:18:00',
      cmds=[ 'show file systems' ],
      summaryCmds=[ 'show file systems' ] )
