# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
from CliMode.FileSystemMount import MountNfsMode
# pylint: disable-next=consider-using-from-import
import CliPlugin.ConfigMgmtMode as ConfigMgmtMode
import ConfigMount
import HostnameCli
import LazyMount
import ShowCommand
import Tac
import TableOutput
import Tracing
import Url
# Bogus import to generate dependency on the flash/file urls.
# pkgdeps: import UrlPlugin.FlashUrl

t0 = Tracing.trace0
__defaultTraceHandle__ = Tracing.Handle( "FileSystemMount" )

mountConfig = None
mountStatus = None

class MountConfigMode( ConfigMgmtMode.ConfigMgmtMode ):
   name = "Management File-Systems"

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__(self, parent, session, "file-systems")
      self.config_ = mountConfig

#--------------------------------------------------------------------------------
# [ no | default ] management file-systems
#--------------------------------------------------------------------------------
class ManagementFileSystemsCmd( CliCommand.CliCommandClass ):
   syntax = 'management file-systems'
   noOrDefaultSyntax = syntax
   data = {
      'management': ConfigMgmtMode.managementKwMatcher,
      'file-systems': 'Manage file systems',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( MountConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mountConfig.nfsConfig.clear()

BasicCliModes.GlobalConfigMode.addCommandClass( ManagementFileSystemsCmd )

# check whether folder path is allowed or not
def checkAllowedPath( path ):
   # if no restrictions
   # allowed paths on 'file:' : '/var/tmp', '/tmp' and 'var/log' 
   if path.fs.allowedPaths_ is None:
      return True

   fName = path.localFilename()
   return all( fName != x for x in path.fs.allowedPaths_ )

#----------------------------------------------------------------------
# Nfs mode
#----------------------------------------------------------------------
class NfsConfigMode( MountNfsMode, BasicCli.ConfigModeBase ):
   name = "NFS Configuration"

   def __init__( self, parent, session, localPath ):
      localFs = Url.filenameToUrl( localPath )
      MountNfsMode.__init__( self, localFs )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
   
      self.localPath = localPath

   def onExit( self ):
      nfsConfig = mountConfig.nfsConfig.get( self.localPath )
      if nfsConfig:
         # if remote config not specified then give warning
         if not nfsConfig.remoteAddr or not nfsConfig.remotePath:
            t0( 'no remote address and path configured' )
            self.addWarning( 'NFS mount would not be created until '
                              'remote address and path are configured' )
      
      BasicCli.ConfigModeBase.onExit( self )

   def addNfsMountPoint( self, localPath ):
      curNfsConfig = None
      # check if there is already a mount point
      curNfsConfig = mountConfig.nfsConfig.get( localPath )
      if curNfsConfig:
         # pylint: disable-next=consider-using-f-string
         self.addWarning( '%s is already a mount point' % localPath )
      else:
         curNfsConfig = mountConfig.newNfsConfig( localPath )

#--------------------------------------------------------------------------------
# [ no | default ] nfs LOCAL_PATH
#--------------------------------------------------------------------------------
class GotoNfsModeCmd( CliCommand.CliCommandClass ):
   syntax = 'nfs LOCAL_PATH'
   noOrDefaultSyntax = syntax
   data = {
            'nfs': 'Network File System',
            'LOCAL_PATH': Url.UrlMatcher(
               lambda fs: fs.fsType in ( 'flash', 'file' ), 'Directory Name' )
          }

   @staticmethod
   def handler( mode, args ):
      localPath = args[ 'LOCAL_PATH' ]
      if localPath.fs.scheme in [ 'flash:', 'file:' ]:
         # check whether the user used the root folder of allowed paths
         if not checkAllowedPath( localPath ):
            mode.addError( "Please use subdirs in " + localPath.localFilename() ) 
            return
         
         childMode = mode.childMode( NfsConfigMode, 
                              localPath=localPath.localFilename() )
         mode.session_.gotoChildMode( childMode )
         childMode.localPath = localPath.localFilename()
         
         childMode.addNfsMountPoint( localPath.localFilename() )
      else:
         mode.addError( "Invalid directory path" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      localPath = args[ 'LOCAL_PATH' ]
      localDir = localPath.localFilename()
      del mountConfig.nfsConfig[ localDir ]

MountConfigMode.addCommandClass( GotoNfsModeCmd )

#--------------------------------------------------------------------------------
# show management file-systems
#--------------------------------------------------------------------------------
class ShowManagementFileSystemsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management file-systems'
   data = {
      'management': ConfigMgmtMode.managementShowKwMatcher,
      'file-systems': 'File systems',
   }

   @staticmethod
   def handler( mode, args ):
      tableHeadings = ( "Protocol", "Remote Host", "Remote Path", "Local Path",
                        "State" )
      table = TableOutput.createTable( tableHeadings )
      f = TableOutput.Format( justify="left" )
      table.formatColumns( *( [ f ] * len( tableHeadings ) ) )

      for localPath in set( mountStatus.nfsStatus.keys() ).intersection( \
         mountConfig.nfsConfig.keys() ):
         nfsConfig = mountConfig.nfsConfig[ localPath ]
         nfsStatus = mountStatus.nfsStatus[ localPath ]
         active = "disabled"
         if nfsStatus.active:
            active = "active"
         
         table.newRow( 'nfs', nfsConfig.remoteAddr, nfsConfig.remotePath, 
                                                            localPath, active )

      print( table.output() )

BasicCli.addShowCommandClass( ShowManagementFileSystemsCmd )

#--------------------------------------------------------------------------------
# [ no | default ] access [ ro | rw ]
#--------------------------------------------------------------------------------
def changeConfigHandle( mode ):
   # change the gen id of the config
   nfsConfig = mountConfig.nfsConfig.get( mode.localPath )
   if nfsConfig:
      nfsConfig.genId = Tac.now()

class AccessCmd( CliCommand.CliCommandClass ):
   syntax = 'access [ ACCESS ]'
   noOrDefaultSyntax = 'access ...'
   data = {
      'access': 'Access permission of the mounted directory',
      'ACCESS': CliMatcher.EnumMatcher( { 'ro': 'Read-only',
                                          'rw': 'Read-Write' } )
   }

   @staticmethod
   def handler( mode, args ):
      nfsConfig = mountConfig.nfsConfig.get( mode.localPath )
      if not nfsConfig:
         return
      nfsConfig.access = args.get( 'ACCESS', nfsConfig.accessDefault )
      changeConfigHandle( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nfsConfig = mountConfig.nfsConfig.get( mode.localPath )
      if not nfsConfig:
         return
      nfsConfig.access = nfsConfig.accessDefault
      changeConfigHandle( mode )

NfsConfigMode.addCommandClass( AccessCmd )

#--------------------------------------------------------------------------------
# [ no | default ] options retransmit NUM
#--------------------------------------------------------------------------------
matcherOptions = CliMatcher.KeywordMatcher( 'options',
      helpdesc='Other options of file-system mounting' )

class OptionsRetransmitNumCmd( CliCommand.CliCommandClass ):
   syntax = 'options retransmit NUM'
   noOrDefaultSyntax = 'options retransmit ...'
   data = {
      'options': matcherOptions,
      'retransmit': 'Retransmit interval',
      'NUM': CliMatcher.IntegerMatcher( 1, 5, helpdesc='Value' ),
   }

   @staticmethod
   def handler( mode, args ):
      nfsConfig = mountConfig.nfsConfig.get( mode.localPath )
      if not nfsConfig:
         return
      nfsConfig.retransmit = args[ 'NUM' ]
      changeConfigHandle( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nfsConfig = mountConfig.nfsConfig.get( mode.localPath )
      if not nfsConfig:
         return
      nfsConfig.retransmit = nfsConfig.retransmitDefault
      changeConfigHandle( mode )

NfsConfigMode.addCommandClass( OptionsRetransmitNumCmd )

#--------------------------------------------------------------------------------
# [ no | default ] options timeout NUM
#--------------------------------------------------------------------------------
class OptionsTimeoutNumCmd( CliCommand.CliCommandClass ):
   syntax = 'options timeout NUM'
   noOrDefaultSyntax = 'options timeout ...'
   data = {
      'options': matcherOptions,
      'timeout': 'Timeout for receiving response from remote server',
      'NUM': CliMatcher.IntegerMatcher( 1, 10, helpdesc='Value' ),
   }

   @staticmethod
   def handler( mode, args ):
      nfsConfig = mountConfig.nfsConfig.get( mode.localPath )
      if not nfsConfig:
         return
      nfsConfig.timeout = args[ 'NUM' ]
      changeConfigHandle( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nfsConfig = mountConfig.nfsConfig.get( mode.localPath )
      if not nfsConfig:
         return
      nfsConfig.timeout = nfsConfig.timeoutDefault
      changeConfigHandle( mode )

NfsConfigMode.addCommandClass( OptionsTimeoutNumCmd )

#--------------------------------------------------------------------------------
# [ no | default ] remote REMOTE_ADDR PATH
#--------------------------------------------------------------------------------
class RemoteCmd( CliCommand.CliCommandClass ):
   syntax = 'remote REMOTE_ADDR PATH'
   noOrDefaultSyntax = 'remote ...'
   data = {
      'remote': 'Directory on remote server to be mounted',
      'REMOTE_ADDR': HostnameCli.IpAddrOrHostnameMatcher( helpdesc='Remote server',
         helpname='WORD' ),
      'PATH': CliMatcher.PatternMatcher( pattern='.*',
         helpdesc='Path to be mounted', helpname='PATH' ),
   }

   @staticmethod
   def handler( mode, args ):
      nfsConfig = mountConfig.nfsConfig.get( mode.localPath )
      if not nfsConfig:
         return 

      remoteAddr = args[ 'REMOTE_ADDR' ]
      remotePath = args[ 'PATH' ]
      nfsConfig.remoteAddr = remoteAddr
      nfsConfig.remotePath = remotePath
      changeConfigHandle( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nfsConfig = mountConfig.nfsConfig.get( mode.localPath )
      if not nfsConfig:
         return 
      nfsConfig.remoteAddr = ''
      nfsConfig.remotePath = ''
      changeConfigHandle( mode )

NfsConfigMode.addCommandClass( RemoteCmd )

def Plugin( entityManager ):
   global mountConfig
   global mountStatus
   mountConfig = ConfigMount.mount( entityManager, "mgmt/filesystems/config", 
                                                "Mgmt::FileSystems::Config", "w" )
   mountStatus = LazyMount.mount( entityManager, "mgmt/filesystems/status", 
                                                "Mgmt::FileSystems::Status", "r" )
