# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CliPlugin module for FlowCongestion configuration commands

import Tac
from Tracing import t0
import CliMatcher
import CliCommand
from CliPlugin.FlowTrackingCliLib import (
      CongestionMode,
)
from FlowTrackerConst import (
   congestionInterval,
   dampingTimeout,
)

# -------------------------
# Flow Congestion Context
# --------------------------

class CongestionContext:
   def __init__( self, trContext, congestionConfig=None ):
      self.trCtx_ = trContext
      self.congestionConfig_ = congestionConfig
      self.changed_ = False
      self.deleted_ = False
      self.interval = None
      self.dampingTimeout = None

      if not self.congestionConfig_:
         self.initFromDefaults()
      else:
         self.initFromConfig()

   def ftrType( self ):
      return self.trCtx_.ftrType()

   def trackerName( self ):
      return self.trCtx_.trackerName()

   def setChanged( self ):
      self.changed_ = True
      self.trCtx_.setChanged()

   def initFromDefaults( self ):
      self.interval = None
      self.dampingTimeout = None
      self.setChanged()

   def initFromConfig( self ):
      self.interval = self.congestionConfig_.interval
      self.dampingTimeout = self.congestionConfig_.dampingTimeout

   def commitCongestion( self, trConfig ):
      if not self.changed_:
         t0( 'Nothing changed for congestion' )
         return
      if self.deleted_:
         if self.congestionConfig_ is not None:
            t0( 'Deleting congestion config' )
            trConfig.congestionConfig = None
         else:
            t0( 'congConfig never created in flowTrackerConfig' )
         return

      if self.congestionConfig_ is None:
         t0( 'Add new congestion config' )
         trConfig.congestionConfig = ( 'congestion', )
         self.congestionConfig_ = trConfig.congestionConfig

      self.congestionConfig_.interval = self.interval
      self.congestionConfig_.dampingTimeout = self.dampingTimeout

   def deletedIs( self, deleted=False ):
      if self.deleted_ == deleted:
         return
      t0( 'congestionConfig deletedIs: ', deleted )
      self.deleted_ = deleted
      self.setChanged()
      # deleted and recreated in same session
      if not deleted:
         self.initFromDefaults()

   def deleted( self ):
      return self.deleted_

   def congestionConfig( self ):
      return self.congestionConfig_

   def dampingTimeoutIs( self, timeout ):
      # damping timeout is optional so leave unset if explicitly set to default
      if timeout is not None and timeout.value == dampingTimeout.defaultTimeout:
         timeout = None
      if timeout != self.dampingTimeout:
         self.dampingTimeout = timeout
         self.setChanged()

   def congestionIntervalIs( self, interval ):
      # congestion interval is optional so leave unset if explicitly set to default
      if interval is not None \
         and interval.before == congestionInterval.defaultBefore \
         and interval.after == congestionInterval.defaultAfter:
         interval = None
      if interval != self.interval:
         self.interval = interval
         self.setChanged()

# -------------------------------------------------------------------------------
# "[no|default] damping timeout <duration> seconds" command,
# in "config-ftr-tr-congestion" mode.
# -------------------------------------------------------------------------------
dampingHelpDesc = 'Minimum time between record exports'
dampingKw = CliMatcher.KeywordMatcher(
                  'damping', helpdesc=dampingHelpDesc )
dampingTimeoutKw = CliMatcher.KeywordMatcher(
                  'timeout', helpdesc=dampingHelpDesc )
durationMatcher = CliMatcher.IntegerMatcher(
                     dampingTimeout.minTimeout,
                     dampingTimeout.maxTimeout,
                     helpdesc=dampingHelpDesc )
secondsKw = CliMatcher.KeywordMatcher( 'seconds',
               helpdesc='Time unit in seconds' )

class DampingTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'damping timeout DURATION seconds'
   noOrDefaultSyntax = 'damping timeout ...'

   data = {
      'damping' : dampingKw,
      'timeout' : dampingTimeoutKw,
      'DURATION' : durationMatcher,
      'seconds' : secondsKw
   }

   @staticmethod
   def handler( mode, args ):
      timeout = args.get( 'DURATION', dampingTimeout.defaultTimeout )
      mode.context().dampingTimeoutIs( dampingTimeout( timeout ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().dampingTimeoutIs( None )

CongestionMode.addCommandClass( DampingTimeoutCmd )

# -------------------------------------------------------------------------------
# "[no|default] interval before <before> seconds after <after> seconds" command,
# in "config-ftr-tr-congestion" mode.
# -------------------------------------------------------------------------------
intervalHelpDesc = 'Time interval of data capture %s an event'
intervalKw = CliMatcher.KeywordMatcher(
                  'interval',
                  helpdesc=intervalHelpDesc % 'around' )
beforeKw = CliMatcher.KeywordMatcher(
                  'before', helpdesc=intervalHelpDesc % 'before' )
beforeIntervalDurationMatcher = \
      CliMatcher.IntegerMatcher(
            congestionInterval.minInterval,
            congestionInterval.maxInterval,
            helpdesc=intervalHelpDesc % 'before' )
afterKw = CliMatcher.KeywordMatcher(
                  'after', helpdesc=intervalHelpDesc % 'after' )
afterIntervalDurationMatcher = \
      CliMatcher.IntegerMatcher(
            congestionInterval.minInterval,
            congestionInterval.maxInterval,
            helpdesc=intervalHelpDesc % 'after' )

class CongestionIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'interval before BEFORE seconds1 after AFTER seconds2'
   noOrDefaultSyntax = 'interval ...'

   data = {
      'interval' : intervalKw,
      'before' : beforeKw,
      'BEFORE' : beforeIntervalDurationMatcher,
      'seconds1' : secondsKw,
      'after' : afterKw,
      'AFTER' : afterIntervalDurationMatcher,
      'seconds2' : secondsKw
   }

   @staticmethod
   def handler( mode, args ):
      mode.context() \
          .congestionIntervalIs(
             congestionInterval( args[ 'BEFORE' ], args[ 'AFTER' ] ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().congestionIntervalIs( None )

CongestionMode.addCommandClass( CongestionIntervalCmd )
