# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CliPlugin module for FlowExporter configuration commands

import Tac
import Tracing
import CliMatcher
import CliCommand
import HostnameCli
from CliPlugin import IntfCli
from CliPlugin.FlowTrackingCliLib import (
      ExporterMode,
      TrackerMode,
      defaultExportFormat,
      guardCollector,
      guardDropReportFormat,
      guardDscp,
      guardIpfixExport,
      guardLocalIntf,
      guardModSflowExport,
      guardPcapFormat,
      guardTemplate,
      guardUdpCollector,
)
from FlowTrackerConst import (
      constants,
      ipfixMtu,
      ipfixPort,
      ipfixVersion,
      templateInterval,
)
from TypeFuture import TacLazyType

traceHandle = Tracing.Handle( 'FlowExporterCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2
t3 = traceHandle.trace3

ExportFormat = TacLazyType( 'FlowTracking::ExportFormat' )
FtConst = TacLazyType( 'FlowTracking::Constants' )

#-------------------------
# Flow Exporter Context
#--------------------------

class ExporterContext:
   def __init__( self, trContext, expName, expConfig=None ):
      self.trCtx_ = trContext
      self.expName_ = expName
      self.expConfig_ = expConfig
      self.changed_ = False
      self.deleted_ = False
      self.collectorHostAndPort = None
      self.dscpValue = None
      self.ipfixMtu = None
      self.useSflowCollectorConfig = None
      self.exportFormat = None
      self.localIntfName = None
      self.templateInterval = None
      self.ipfixVersion = None
      t0( "trCtx:", trContext, "expName:", expName, "expConfig:", expConfig )

      if not expConfig:
         self.initFromDefaults()
      else:
         self.initFromConfig()

   def ftrType( self ):
      return self.trCtx_.ftrType()

   def trackerName( self ):
      return self.trCtx_.trackerName()

   def setChanged( self ):
      self.changed_ = True
      self.trCtx_.setChanged()

   def initFromDefaults( self ):
      self.collectorHostAndPort = {}
      self.dscpValue = constants.dscpValueDefault
      self.templateInterval = templateInterval.intervalDefault
      self.ipfixMtu = ipfixMtu.mtuDefault
      self.ipfixVersion = ipfixVersion.versionDefault
      self.localIntfName = constants.intfNameDefault
      self.useSflowCollectorConfig = False
      self.exportFormat = ExportFormat.formatDefault
      self.setChanged()

   def initFromConfig( self ):
      self.collectorHostAndPort = {}
      for hnp in self.expConfig_.collectorHostAndPort.values():
         self.collectorHostAndPort[ hnp.hostname ] = hnp
      self.dscpValue = self.expConfig_.dscpValue
      self.templateInterval = self.expConfig_.templateInterval
      self.ipfixMtu = self.expConfig_.ipfixMtu
      self.ipfixVersion = self.expConfig_.ipfixVersion
      self.localIntfName = self.expConfig_.localIntfName
      self.useSflowCollectorConfig = self.expConfig_.useSflowCollectorConfig
      self.exportFormat = self.expConfig_.exportFormat

   def addCollector( self, hostAndPort ):
      t1( 'addCollector', hostAndPort )
      try:
         hnp = self.collectorHostAndPort[ hostAndPort.hostname ]
      except KeyError:
         hnp = None

      if hnp is None or hnp.port != hostAndPort.port:
         t1( 'addCollector hnp:', hnp )
         self.collectorHostAndPort[ hostAndPort.hostname ] = hostAndPort
         self.setChanged()

   def removeCollector( self, hnp ):
      t1( 'removeCollector hnp:', hnp )
      if hnp.hostname in self.collectorHostAndPort:
         del self.collectorHostAndPort[ hnp.hostname ]
         self.setChanged()

   def setSflowCollector( self ):
      t1( 'setting sFlow as the collector' )
      self.useSflowCollectorConfig = True
      self.setChanged()

   def resetSflowCollector( self ):
      t1( 'reset sFlow as the collector' )
      self.useSflowCollectorConfig = False
      self.setChanged()

   def setExportFormat( self, exportFormat ):
      t1( 'setting ', exportFormat, ' as the export format' )
      if defaultExportFormat( self.ftrType() ) != exportFormat:
         self.exportFormat = exportFormat
         self.setChanged()
      else:
         self.resetExportFormat()

   def resetExportFormat( self ):
      t1( 'reset export format' )
      self.exportFormat = ExportFormat.formatDefault
      self.setChanged()

   def dscpIs( self, dscp ):
      t1( 'dscpIs:', dscp )
      if self.dscpValue != dscp:
         self.dscpValue = dscp
         self.setChanged()

   def templateIntervalIs( self, interval ):
      t1( 'templateIntervalIs:', interval )
      if self.templateInterval != interval:
         self.templateInterval = interval
         self.setChanged()

   def mtuIs( self, mtu ):
      if self.ipfixMtu != mtu:
         self.ipfixMtu = mtu
         self.setChanged()

   def ipfixVersionIs( self, _ipfixVersion ):
      if self.ipfixVersion != _ipfixVersion:
         self.ipfixVersion = _ipfixVersion
         self.setChanged()

   def localIntfNameIs( self, localIntfName ):
      if self.localIntfName != localIntfName:
         self.localIntfName = localIntfName
         self.setChanged()

   def commitExporter( self, trConfig ):
      if not self.changed_:
         t0( 'Nothing changed for', self.expName_ )
         return None
      if self.deleted_:
         if self.expConfig_ is not None:
            t0( 'Deleting exporter from flowTrackerConfig', self.expName_ )
            del trConfig.expConfig[ self.expName_ ]
         else:
            t0( 'Exporter never created in flowTrackerConfig', self.expName_ )
         return None

      if self.expConfig_ is None:
         t0( 'Add new exporter to flowTrackerConfig', self.expName_ )
         self.expConfig_ = \
            trConfig.expConfig.newMember( self.expName_ )
         t0( 'trConfig', trConfig, trConfig.expConfig[ self.expName_ ] )

      self.expConfig_.dscpValue = self.dscpValue
      self.expConfig_.templateInterval = self.templateInterval
      self.expConfig_.ipfixVersion = self.ipfixVersion
      self.expConfig_.ipfixMtu = self.ipfixMtu
      self.expConfig_.localIntfName = self.localIntfName
      self.expConfig_.useSflowCollectorConfig = self.useSflowCollectorConfig
      self.expConfig_.exportFormat = self.exportFormat

      # Collectors should be updated last as some syslogs emitted during their
      # processing may differ depending on the rest of the exporters' config.
      staleCollector = set( self.expConfig_.collectorHostAndPort )
      for hostname, hnp in self.collectorHostAndPort.items():
         t0( 'Adding / updating collector configuration for', hostname )
         self.expConfig_.collectorHostAndPort.addMember( hnp )
         staleCollector -= { hostname }
      for hostname in staleCollector:
         t0( 'Removing collector configuration for', hostname )
         del self.expConfig_.collectorHostAndPort[ hostname ]

      return self.expConfig_

   def deletedIs( self, deleted=False ):
      if self.deleted_ == deleted:
         return
      t0( 'exporter deletedIs: ', deleted )
      self.deleted_ = deleted
      self.setChanged()
      # deleted and recreated in same session
      if not deleted:
         self.initFromDefaults()

   def deleted( self ):
      return self.deleted_

   def expConfig( self ):
      return self.expConfig_


#-------------------------------------------------------------------------------
# "[no|default] exporter <exporter-name>" command, in "config-ftr-tr" mode.
#-------------------------------------------------------------------------------

def getExporterName( mode ):
   return list( mode.context().exporterCtx )

exporterNameMatcher = CliMatcher.DynamicNameMatcher( getExporterName,
                                                     "Exporter Name" )

def gotoExporterMode( mode, exporterName ):
   trContext = mode.context()
   trackerName = mode.context().trackerName()
   t1( 'gotoExporterMode of', trackerName, exporterName )

   if len( exporterName ) > FtConst.confNameMaxLen:
      mode.addError(
         f'Exporter name is too long (maximum {FtConst.confNameMaxLen})' )
      return

   exporterCtx = trContext.exporterContext( exporterName )

   if exporterCtx is None:
      t0( 'No exporterContext found, creating new exporter context' )
      exporterCtx = trContext.newExporterContext( exporterName )
   elif exporterCtx.deleted() or exporterCtx.expConfig():
      t0( 'Re-entering a deleted exporterContext' )
      exporterCtx.deletedIs( False )

   childMode = mode.childMode( ExporterMode,
                  context=exporterCtx, exporterName=exporterName )
   mode.session_.gotoChildMode( childMode )

def noExporterMode( mode, exporterName ):
   trContext = mode.context()
   trackerName = mode.context().trackerName()
   t1( 'noExporterMode of', trackerName, exporterName )

   exporterCtx = trContext.exporterContext( exporterName )

   if exporterCtx is None:
      return

   exporterCtx.deletedIs( True )
   return

class ExporterCmd( CliCommand.CliCommandClass ):
   syntax = '''exporter EXPORTER_NAME'''
   noOrDefaultSyntax = syntax

   data = {
      'exporter' : 'Configure flow exporter',
      'EXPORTER_NAME' : exporterNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      gotoExporterMode( mode, exporterName=args[ 'EXPORTER_NAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noExporterMode( mode, exporterName=args[ 'EXPORTER_NAME' ] )

TrackerMode.addCommandClass( ExporterCmd )

#-------------------------------------------------------------------------------
# "[no|default] collector ( ( <ipv4/6-addr> [ port <udp-port>] ) | sflow )" command
# in "config-ftr-tr-exp" mode
#-------------------------------------------------------------------------------

collectorMatcher = CliCommand.guardedKeyword(
      'collector',
      helpdesc='Configure flow collector',
      guard=guardCollector )
sflowMatcher = CliCommand.guardedKeyword(
      'sflow',
      helpdesc='Use sFlow collector configuration',
      guard=guardModSflowExport )

ipAddrOrHostnameMatcher = HostnameCli.IpAddrOrHostnameMatcher(
      ipv6=True,
      helpdesc='IPv4 address or IPv6 address or fully qualified domain name' )
ipAddrOrHostnameMatcherWithGuard = CliCommand.Node(
      matcher=ipAddrOrHostnameMatcher,
      guard=guardUdpCollector )

def getHostAndPort( mode, args ):
   hostname = args[ 'COLLECTOR' ]
   port = args.get( 'UDP_PORT', ipfixPort.ipfixPortDefault )
   hostnameStr = constants.ipAddrDefault if not hostname else str( hostname )
   trackerName = mode.context().trackerName()
   t1( 'getHostAndPort', trackerName, hostname, port, hostnameStr )
   return Tac.Value( 'Arnet::HostAndPort', hostname=hostnameStr, port=port )

class ExporterSflowCollectorCmd( CliCommand.CliCommandClass ):
   syntax = '''collector sflow'''
   noOrDefaultSyntax = syntax
   data = {
      'collector' : collectorMatcher,
      'sflow' : sflowMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().setSflowCollector()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().resetSflowCollector()

ExporterMode.addCommandClass( ExporterSflowCollectorCmd )

class ExporterCollectorCmd( CliCommand.CliCommandClass ):
   syntax = '''collector COLLECTOR [ port UDP_PORT ]'''
   noOrDefaultSyntax = syntax

   data = {
      'collector' : collectorMatcher,
      'COLLECTOR' : ipAddrOrHostnameMatcherWithGuard,
      'port' : 'Collector port',
      'UDP_PORT' : CliMatcher.IntegerMatcher(
                        ipfixPort.minPort,
                        ipfixPort.maxPort,
                        helpdesc='Collector port' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().addCollector( getHostAndPort( mode, args ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().removeCollector( getHostAndPort( mode, args ) )

ExporterMode.addCommandClass( ExporterCollectorCmd )

# -------------------------------------------------------------------------------
# "[no|default] dscp <dscp>" command in "config-ftr-tr-exp" mode
#-------------------------------------------------------------------------------
class ExporterDscpCmd( CliCommand.CliCommandClass ):
   syntax = '''dscp DSCP'''
   noOrDefaultSyntax = '''dscp ...'''

   data = {
      'dscp' : CliCommand.guardedKeyword( 'dscp',
                                          helpdesc='Set the DSCP value',
                                          guard=guardDscp ),
      'DSCP' : CliMatcher.IntegerMatcher( 0, 63,
                  helpdesc='DSCP value between 0 and 63' )
   }
   @staticmethod
   def handler( mode, args ):
      mode.context().dscpIs( args.get( 'DSCP', constants.dscpValueDefault ) )

   noOrDefaultHandler = handler

ExporterMode.addCommandClass( ExporterDscpCmd )

#-------------------------------------------------------------------------------
# "[no|default] format ( ( ipfix version VERSION [ max-packet-size MTU ] ) | sflow
#  | drop-report )"
# command in "config-ftr-tr-exp" mode
#-------------------------------------------------------------------------------

class ExporterFormat( CliCommand.CliCommandClass ):
   syntax = '''format (
               ( ipfix version VERSION [ max-packet-size MTU ] )
               | sflow
               | drop-report
               | pcap )'''
   noOrDefaultSyntax = \
   '''format [ ( ipfix version [ ... | VERSION max-packet-size ... ] ) | ... ]'''

   data = {
      'format' : 'Configure flow export format',
      'ipfix' : CliCommand.guardedKeyword(
                     'ipfix',
                     helpdesc='Configure flow export IPFIX format',
                     guard=guardIpfixExport ),
      'version' : 'Configure IPFIX version',
      'VERSION' : CliMatcher.IntegerMatcher(
                     ipfixVersion.minVersion,
                     ipfixVersion.maxVersion,
                     helpdesc='IPFIX version' ),
      'max-packet-size' : 'Configure IPFIX maximum packet size',
      'MTU' : CliMatcher.IntegerMatcher(
                     ipfixMtu.minMtu,
                     ipfixMtu.maxMtu,
                     helpdesc='IPFIX maximum packet size' ),
      'sflow' : CliCommand.guardedKeyword( 'sflow', helpdesc='Use sFlow '
                                           'for export',
                                           guard=guardModSflowExport ),
      'drop-report' : CliCommand.guardedKeyword( 'drop-report',
                                                 helpdesc='Use drop-report '
                                                 'for export',
                                                 guard=guardDropReportFormat ),
      'pcap' : CliCommand.guardedKeyword( 'pcap',
                                          helpdesc='Use pcap for export',
                                          guard=guardPcapFormat ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'sflow' in args:
         mode.context().setExportFormat( ExportFormat.formatSflow )
      elif 'drop-report' in args:
         mode.context().setExportFormat( ExportFormat.formatDropReport )
      elif 'pcap' in args:
         mode.context().setExportFormat( ExportFormat.formatPcap )
      else:
         version = args.get( 'VERSION', ipfixVersion.versionDefault )
         mtu = args.get( 'MTU', ipfixMtu.mtuDefault )
         t1( 'setFormatConfig version version', version, mtu )
         mode.context().ipfixVersionIs( version )
         mode.context().mtuIs( mtu )
         mode.context().setExportFormat( ExportFormat.formatIpfix )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      modeCtx = mode.context()
      modeCtx.mtuIs( ipfixMtu.mtuDefault )
      if args.get( 'VERSION' ):
         # If version is specified then user wants to remove MTU setting.
         return
      else:
         # if verison is not specified, user unconfiguring export format
         modeCtx.resetExportFormat()
         modeCtx.ipfixVersionIs( ipfixVersion.versionDefault )

ExporterMode.addCommandClass( ExporterFormat )

#-------------------------------------------------------------------------------
# "[no|default] local interface <intf>" command in "config-ftr-tr-exp" mode
#-------------------------------------------------------------------------------

localMatcher = CliCommand.guardedKeyword( 'local',
                                          helpdesc='Configure the local interface',
                                          guard=guardLocalIntf )

class ExporterLocalInterface( CliCommand.CliCommandClass ):
   syntax = '''local interface INTERFACE'''
   noOrDefaultSyntax = '''local interface ...'''

   data = {
      'local' : localMatcher,
      'interface' : 'Configure the local interface',
      'INTERFACE' : IntfCli.Intf.matcherWithIpSupport,
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().localIntfNameIs( args.get( 'INTERFACE' ).name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().localIntfNameIs( constants.intfNameDefault )

ExporterMode.addCommandClass( ExporterLocalInterface )

#-------------------------------------------------------------------------------
# "[no|default] template interval <interval>" command in "config-ftr-tr-exp" mode
#-------------------------------------------------------------------------------

templateMatcher = CliCommand.guardedKeyword(
                     'template',
                     helpdesc='Configure template parameters',
                     guard=guardTemplate )

class ExporterTemplateCommand( CliCommand.CliCommandClass ):
   syntax = '''template interval INTERVAL'''
   noOrDefaultSyntax = '''template interval ...'''

   data = {
      'template' : templateMatcher,
      'interval' : 'Configure template interval',
      'INTERVAL' : CliMatcher.IntegerMatcher(
                        templateInterval.minInterval,
                        templateInterval.maxInterval,
                        helpdesc='Template interval in milliseconds' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().templateIntervalIs(
               args.get( 'INTERVAL', templateInterval.intervalDefault ) )

   noOrDefaultHandler = handler

ExporterMode.addCommandClass( ExporterTemplateCommand )
