# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentDirectory
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.SubIntfCli as SubIntfCli
import CliPlugin.TunnelIntfCli as TunnelIntfCli
import CliPlugin.DpsIntfCli as DpsIntfCli
import Tracing
import SflowUtil
from if_ether_arista import (
   SFLOW_FORMAT_MULTIPLE_INTFS,
   SFLOW_FORMAT_PKT_DISCARD,
   SFLOW_INTERNAL_INTERFACE,
   SFLOW_OUTPUT_UNKNOWN,
)
from FlowTrackerCliUtil import (
   congestionSupportedForType,
   defaultExportFormatForType,
   dscpSupportedForType,
   exportMplsSupportedForType,
   ftrCapabilitiesPathPrefix,
   ftrCapabilitiesType,
   ftrConfigType,
   ftrMirrorOnDropConfigType,
   ftrTypes,
   ftrTypeCpuQueue,
   ftrTypeDfw,
   ftrTypeHardware,
   ftrTypeSampled,
   ftrTypeInbandTelemetry,
   ftrTypeMirrorOnDrop,
   ftrTypeByName,
   ftrTypeKwStr,
   getFtrTypeFromArgs,
   getFlowTrackingCliConfigPath,
   hwFtrConfigPathPrefix,
   hwFtrConfigType,
   hwFtrStatusPathPrefix,
   hwFtrStatusType,
   localIntfSupportedForType,
   supportedExportFormatsForType,
   templateSupportedForType,
   collectorSupportedForType,
)

from CliMode.FlowCongestion import CongestionModeBase
from CliMode.FlowTracking import FlowTrackingModeBase
from CliMode.FlowTracker import TrackerModeBase
from CliMode.FlowExporter import ExporterModeBase
from CliMode.FlowEncapFilter import EncapFilterModeBase
from TypeFuture import TacLazyType
import LazyMount
from Toggles.FlowTrackerToggleLib import (
   toggleSftGreExportEnabled,
   toggleSftFilteredSamplingEnabled,
)

traceHandle = Tracing.Handle( 'FlowTrackingCliLib' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

ftrCapabilities = {}
ftrSwCapabilities = {}
routingHwStatus = None
mplsHwCaps = None
bridgingHwCapabilities = None
sflowHwStatusDir = None
trackingConfig = {}
hwConfig = {}
hwStatus = {}

IP_GROUP = "IPv4"
EGRESS_IP_GROUP = "Egress-IPv4"
IP6_GROUP = "IPv6"
EGRESS_IP6_GROUP = "Egress-IPv6"

ExportFormat = TacLazyType( 'FlowTracking::ExportFormat' )
FtConsts = TacLazyType( 'FlowTracking::Constants' )
IntfDetails = TacLazyType( 'IntfSnmp::IntfDetails' )
IntfIndexMapType = TacLazyType( 'FlowTracking::IntfIndexMap' )
IpfixRecordType = TacLazyType( 'Ipfix::RecordType' )
SelectorAlgorithmType = TacLazyType( 'Ipfix::SelectorAlgorithm' )
ExpFormat = TacLazyType( 'FlowTracking::ExportFormat' )
SwCapabilities = TacLazyType( 'FlowTracking::Capabilities' )

def getMplsHwCaps():
   return mplsHwCaps

def getBridgingHwCaps():
   return bridgingHwCapabilities

def getFtrCaps( ftrType ):
   if ftrType not in ftrTypes:
      return None
   return ftrCapabilities[ ftrType ]

def getFtrSwCaps( ftrType ):
   if ftrType not in ftrTypes:
      return None
   return ftrSwCapabilities[ ftrType ]

def isSftAgentRunning( entityManager, activeAgentDir ):
   return ( 'sft' in activeAgentDir and
         AgentDirectory.agent( entityManager.sysname(), 'SftAgent' ) is not None )

def isHwAccelAgentRunning( entityManager, activeAgentDir ):
   return ( 'hwAccel' in activeAgentDir and
         AgentDirectory.agent( entityManager.sysname(), 'StftAgent' ) is not None )

def isSfeAgentRunning( entityManager, activeAgentDir ):
   return ( 'bess' in activeAgentDir and
            AgentDirectory.agent( entityManager.sysname(), 'Sfe' ) is not None )

def isInbandTelemetryAgentRunning( entityManager, activeAgentDir ):
   return ( 'inbandTelemetry' in activeAgentDir and
         AgentDirectory.agent( entityManager.sysname(),
                               'InbandTelemetry' ) is not None )

def isMirrorOnDropAgentRunning( entityManager, activeAgentDir ):
   return ( 'DropExport' in activeAgentDir and
            AgentDirectory.agent( entityManager.sysname(),
                                  'DropExport' ) is not None )

def isDfwOffloadClientAgentRunning( entityManager, activeAgentDir ):
   return ( 'dfw' in activeAgentDir and
            AgentDirectory.agent( entityManager.sysname(),
                                  'DfwOffloadClient' ) is not None )

def isDmaMonitorAgentRunning( entityManager, activeAgentDir ):
   return ( 'DmaMonitor' in activeAgentDir and
            AgentDirectory.agent( entityManager.sysname(),
                                  'DmaMonitor' ) is not None )

def flowTrackingAgentRunning( ftrType, entityManager, activeAgentDir ):
   if ftrType == ftrTypeSampled:
      return isSftAgentRunning( entityManager, activeAgentDir )
   if ftrType == ftrTypeHardware:
      return ( isHwAccelAgentRunning( entityManager, activeAgentDir ) or
               isSfeAgentRunning( entityManager, activeAgentDir ) )
   if ftrType == ftrTypeInbandTelemetry:
      return isInbandTelemetryAgentRunning( entityManager, activeAgentDir )
   if ftrType == ftrTypeMirrorOnDrop:
      return isMirrorOnDropAgentRunning( entityManager, activeAgentDir )
   if ftrType == ftrTypeDfw:
      return ( isDfwOffloadClientAgentRunning( entityManager, activeAgentDir ) or
               isSfeAgentRunning( entityManager, activeAgentDir ) )
   if ftrType == ftrTypeCpuQueue:
      return isDmaMonitorAgentRunning( entityManager, activeAgentDir )
   return False

def sflowHwStatus():
   return SflowUtil.sflowHwStatus( sflowHwStatusDir )

def sampleRateRangeFn( mode, context ):
   return ( sflowHwStatus().minSampleRate, sflowHwStatus().maxSampleRate )

def swFtrSupported():
   return ftrCapabilities[ ftrTypeSampled ].supported

def hwFtrSupported():
   return ftrCapabilities[ ftrTypeHardware ].supported

def intFtrSupported():
   return ftrCapabilities[ ftrTypeInbandTelemetry ].supported

def modFtrSupported():
   return ftrCapabilities[ ftrTypeMirrorOnDrop ].supported

def cpuQueueFtrSupported():
   return ftrCapabilities[ ftrTypeCpuQueue ].supported

def sampleLimitSupported( ftrType ):
   return ftrCapabilities[ ftrType ].sampleLimitSupported

def dfwFtrSupported():
   return ftrCapabilities[ ftrTypeDfw ].supported

def packetBufferModFtrSupported( ftrType ):
   return ftrCapabilities[ ftrType ].packetBufferDropsSupport

def packetBufferSamplingProbSupported( ftrType ):
   return ( packetBufferModFtrSupported( ftrType ) and
            ftrCapabilities[ ftrType ].packetBufferSamplingProbSupport )

def packetBufferShaperSupported( ftrType ):
   return ( packetBufferModFtrSupported( ftrType ) and
            ftrCapabilities[ ftrType ].packetBufferShaperSupport )

def defaultSampleLimitOnPlatform( mode ):
   capabilities = ftrCapabilities[ ftrTypeByName[ mode.ftrTypeStr ] ]
   # defaultSampleLimit is an optional attribute. If it's unset, assume disabled (0)
   return capabilities.defaultSampleLimit or 0

def defaultExportFormat( ftrType ):
   return defaultExportFormatForType( ftrCapabilities[ ftrType ],
                                      ftrSwCapabilities[ ftrType ] )

def modDefaultTrapSuppression():
   return ftrCapabilities[ ftrTypeMirrorOnDrop ].modDefaultTrapSuppression

def swExportSupported( ftrType ):
   return ftrCapabilities[ ftrType ].swExportSupported

def rewriteDscpSupported( ftrType ):
   return ( ftrType == ftrTypeSampled and
            routingHwStatus.sflowDscpRewriteSupported )

def hwOffloadSupported( ftrType ):
   return hwOffloadIpv4Supported( ftrType ) or hwOffloadIpv6Supported( ftrType )

def hwOffloadIpv4Supported( ftrType ):
   return ftrCapabilities[ ftrType ].hwOffloadIpv4

def hwOffloadIpv6Supported( ftrType ):
   return ftrCapabilities[ ftrType ].hwOffloadIpv6

def flowGroupConfigSupported( ftrType ):
   return ftrCapabilities[ ftrType ].flowGroupConfigSupported

def mirroringSupported( ftrType ):
   return ftrCapabilities[ ftrType ].mirroringSupported

def tunnelIntfFtrSupported( ftrType ):
   return ftrCapabilities[ ftrType ].tunnelIntfSupported

def dpsIntfFtrSupported( ftrType ):
   return ftrCapabilities[ ftrType ].dpsIntfSupported

def subIntfFtrSupported( ftrType ):
   return ftrCapabilities[ ftrType ].subIntfSupported

def trapSuppressionFtrSupported( ftrType ):
   return ftrCapabilities[ ftrType ].trapSuppressionSupported

def encapsulationFilterSupported( ftrType ):
   return ftrCapabilities[ ftrType ].encapFilterSupported

def exportFormatSupported( ftrType, exportFormat ):
   supportedExportFormats = supportedExportFormatsForType(
      ftrCapabilities[ ftrType ], ftrSwCapabilities[ ftrType ] )
   return exportFormat in supportedExportFormats

def greTunnelFilter( intfStatus ) -> bool:
   return intfStatus.mode == 'tunnelIntfModeGre'

def noFilter( intfStatus ) -> bool:
   return True

def noTunnelFilter( intfStatus ) -> bool:
   return False

def getIfIndexMap( ethIntfStatusDir, ftrCaps=None ):
   ifIndexMap = {}
   intfDetails = IntfDetails()
   filterFunc = noFilter
   if ftrCaps:
      tunnelSupported = ftrCaps.tunnelIntfSupported
      if tunnelSupported:
         filterFunc = noFilter
      elif toggleSftGreExportEnabled():
         filterFunc = greTunnelFilter
      else:
         filterFunc = noTunnelFilter
   for intf, intfStatus in ethIntfStatusDir.intfStatus.items():
      if filterFunc( intfStatus ):
         intfDetails.status = intfStatus
         # do not export interfaces that does not have proper snmpIdx
         if intfDetails.ifIndex > 0:
            ifIndexMap[ intfDetails.ifIndex ] = intf
   # also add mapping for 4 known intfs
   ifIndexMap[ SFLOW_FORMAT_PKT_DISCARD ] = IntfIndexMapType.discardIntfName
   ifIndexMap[ SFLOW_OUTPUT_UNKNOWN ] = IntfIndexMapType.unknownIntfName
   ifIndexMap[ SFLOW_INTERNAL_INTERFACE ] = IntfIndexMapType.cpuIntfName
   ifIndexMap[ SFLOW_FORMAT_MULTIPLE_INTFS ] = IntfIndexMapType.multicastIntfName
   return ifIndexMap

#-------------------------------------------------------------------------------
# Guard functions to prevent configuration on systems without flow tracker support
#-------------------------------------------------------------------------------
def guardFlowTracking( mode, token ):
   if ( swFtrSupported() or
        hwFtrSupported() or
        intFtrSupported() or
        modFtrSupported() or
        cpuQueueFtrSupported() or
        dfwFtrSupported() ):
      return None
   return CliParser.guardNotThisPlatform

def guardFlowTracker( mode, token ):
   return None

def guardIntfFtrSampled( mode, token ):
   if isinstance( mode.intf, TunnelIntfCli.TunnelIntf ):
      # tunnelIntfFtrSupported capability flag is ignored
      # as sampled ftr is not supported for Tunnel interface
      return CliParser.guardNotThisPlatform

   if isinstance( mode.intf, DpsIntfCli.DpsIntf ):
      # Ensure we explicitly return false for DpsIntf.
      # In case of breadth tests, swFtrSupported() will be true causing test to fail
      # in case if Dps1 saying the guard error was not raised.
      return CliParser.guardNotThisPlatform

   if swFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardIngressIntfFilteredSampling( mode, token ):
   if not toggleSftFilteredSamplingEnabled():
      return CliParser.guardNotThisPlatform

   if ( mode.intf.isSubIntf() and
        not sflowHwStatus().sflowSamplingOnSubInterfaceIngressSupported ):
      return CliParser.guardNotThisPlatform

   if not sflowHwStatus().ingressFilteredSamplingSupported:
      return CliParser.guardNotThisPlatform

   return None

def guardEgressIntfSampling( mode, token ):
   # as the egress keyword comes after sampled, the
   # guard for sampled is already applied. No need to recheck here
   if ftrCapabilities[ ftrTypeByName[ ftrTypeSampled ] ].\
      egressModifiedFlowTrackerIntfSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardIntfFtrHardware( mode, token ):
   if ( isinstance( mode.intf, TunnelIntfCli.TunnelIntf ) and
        not tunnelIntfFtrSupported( ftrTypeHardware ) ):
      return CliParser.guardNotThisPlatform

   if ( isinstance( mode.intf, SubIntfCli.SubIntf ) and
         not subIntfFtrSupported( ftrTypeHardware ) ):
      return CliParser.guardNotThisPlatform

   if ( isinstance( mode.intf, DpsIntfCli.DpsIntf ) and
        not dpsIntfFtrSupported( ftrTypeHardware ) ):
      return CliParser.guardNotThisPlatform

   if hwFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardIntfFtrDfw( mode, token ):
   if dfwFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardFtrTypeSampledMode( mode, token ):
   if mode.ftrTypeStr == ftrTypeKwStr[ ftrTypeSampled ]:
      return None
   return CliParser.guardNotThisPlatform

def guardFtrFlowKw( mode, token ):
   if ( mode.ftrTypeStr == ftrTypeKwStr[ ftrTypeSampled ] ) or \
      ( mode.ftrTypeStr == ftrTypeKwStr[ ftrTypeInbandTelemetry ] ):
      return None
   return CliParser.guardNotThisPlatform

guardSampleRate = guardFtrTypeSampledMode
guardFlowTableSize = guardFtrFlowKw
guardFlowKw = guardFtrFlowKw

def guardFtrEncapsulation( mode, token ):
   if ( mode.ftrTypeStr == ftrTypeKwStr[ ftrTypeSampled ] ) or \
      mode.ftrTypeStr == ftrTypeKwStr[ ftrTypeMirrorOnDrop ]:
      return None
   else:
      return CliParser.guardNotThisPlatform

def guardFtrEncapFilter( mode, token ):
   if encapsulationFilterSupported( ftrTypeByName[ mode.ftrTypeStr ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardFtrSampled( mode, token ):
   if swFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardFtrTelemetry( mode, token ):
   if intFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardFtrHardware( mode, token ):
   if hwFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardFtrMirrorOnDrop( mode, token ):
   if modFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardFtrCpuQueue( mode, token ):
   if cpuQueueFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardFtrDfw( mode, token ):
   if dfwFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardSampleLimit( mode, token ):
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   if sampleLimitSupported( ftrType ):
      return None
   return CliParser.guardNotThisPlatform

def guardPacketBufferMirrorOnDrop( mode, token ):
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   if packetBufferModFtrSupported( ftrType ):
      return None
   return CliParser.guardNotThisPlatform

def guardPacketBufferSamplingProb( mode, token ):
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   if packetBufferSamplingProbSupported( ftrType ):
      return None
   return CliParser.guardNotThisPlatform

def guardPacketBufferShaper( mode, token ):
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   if packetBufferShaperSupported( ftrType ):
      return None
   return CliParser.guardNotThisPlatform

def guardModSflowExport( mode, token ):
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   if exportFormatSupported( ftrType, ExportFormat.formatSflow ):
      return None
   return CliParser.guardNotThisPlatform

def guardMplsExport( mode, token ):
   if exportMplsSupportedForType( ftrTypeByName[ mode.ftrTypeStr ], mplsHwCaps ):
      return None
   return CliParser.guardNotThisPlatform

def guardCongestionExport( mode, token ):
   """
   'record export on congestion' is only supported by flow tracking cpu-queue
   """
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   if congestionSupportedForType( ftrSwCapabilities[ ftrType ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardInterval( mode, token ):
   """
   CLI guard that's used in the CLI command 'record export on interval'
   """
   # BUG857332 - Use FlowTracking::Capabilities instead
   if mode.ftrTypeStr != ftrTypeKwStr[ ftrTypeCpuQueue ]:
      return None
   return CliParser.guardNotThisPlatform

def guardActiveInterval( mode, token ):
   """
   An active interval of 0 is only supported by the SftAgent, configured via
   'flow tracking sampled' mode.
   """
   if mode.ftrTypeStr == ftrTypeKwStr[ ftrTypeSampled ]:
      return None
   return CliParser.guardNotThisPlatform

def guardInactive( mode, token ):
   """
   CLI guard that's used in the CLI command 'record export on inactive'
   """
   # BUG857332 - Use FlowTracking::Capabilities instead
   if mode.ftrTypeStr not in ( ftrTypeKwStr[ ftrTypeCpuQueue ],
                               ftrTypeKwStr[ ftrTypeDfw ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardMonitor( mode, token ):
   """
   Cli guard that's used in the CLI command 'monitor cpu-queue <cpuQueueName>'
   """
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   if ftrCapabilities[ ftrType ].monitoredCpuQueueSupported:
      return None
   return CliParser.guardNotThisPlatform

guardFtrMplsEncapsluation = guardMplsExport

def guardExportOnTcpState( mode, token ):
   t0( 'guardExportOnTcpState:', mode, mode.ftrTypeStr )
   if ftrCapabilities[ ftrTypeByName[ mode.ftrTypeStr ] ].exportOnTcpStateSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardSwExport( mode, token ):
   if swExportSupported( ftrTypeByName[ mode.ftrTypeStr ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardRewriteDscp( mode, token ):
   if rewriteDscpSupported( ftrTypeByName[ mode.ftrTypeStr ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardRecordUpdate( mode, token ):
   if swExportSupported( ftrTypeByName[ mode.ftrTypeStr ] ) or \
      rewriteDscpSupported( ftrTypeByName[ mode.ftrTypeStr ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardSampledHwOffload( mode, token ):
   if hwOffloadSupported( ftrTypeSampled ):
      return None
   return CliParser.guardNotThisPlatform

def guardHwOffload( mode, token ):
   if hwOffloadSupported( ftrTypeByName[ mode.ftrTypeStr ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardHwOffloadIpv4( mode, token ):
   if hwOffloadIpv4Supported( ftrTypeByName[ mode.ftrTypeStr ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardHwOffloadIpv6( mode, token ):
   if hwOffloadIpv6Supported( ftrTypeByName[ mode.ftrTypeStr ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardFlowGroupConfig( mode, token ):
   if flowGroupConfigSupported( ftrTypeByName[ mode.ftrTypeStr ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardShowFlowGroupConfig( mode, token ):
   # During show commands, the mode is enable mode by default and it does not
   # have access to ftrType
   if flowGroupConfigSupported( ftrTypeHardware ):
      return None
   return CliParser.guardNotThisPlatform

def guardMirroring( mode, token ):
   if mirroringSupported( ftrTypeByName[ mode.ftrTypeStr ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardFtrTrapSuppression( mode, token ):
   if trapSuppressionFtrSupported( ftrTypeByName[ mode.ftrTypeStr ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardDropReportFormat( mode, token ):
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   if exportFormatSupported( ftrType, ExpFormat.formatDropReport ):
      return None
   return CliParser.guardNotThisPlatform

def guardPcapFormat( mode, token ):
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   if exportFormatSupported( ftrType, ExpFormat.formatPcap ):
      return None
   return CliParser.guardNotThisPlatform

def guardShowMirrorOnDropFlowTable( mode, token ):
   ftrType = ftrTypeMirrorOnDrop
   if ( modFtrSupported() and
        exportFormatSupported( ftrType, ExpFormat.formatDropReport ) ):
      return None
   return CliParser.guardNotThisPlatform

guardShowMirrorOnDropTrackerCounters = guardShowMirrorOnDropFlowTable

def guardShowModReasonCounters( mode, token ):
   if modFtrSupported() and sampleLimitSupported( ftrTypeMirrorOnDrop ):
      return None
   return CliParser.guardNotThisPlatform

def guardIpfixExport( mode, token ):
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   if exportFormatSupported( ftrType, ExpFormat.formatIpfix ):
      return None
   return CliParser.guardNotThisPlatform

def guardDscp( mode, token ):
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   if dscpSupportedForType( ftrCapabilities[ ftrType ],
                            ftrSwCapabilities[ ftrType ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardLocalIntf( mode, toGuardError ):
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   if localIntfSupportedForType( ftrCapabilities[ ftrType ],
                                 ftrSwCapabilities[ ftrType ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardTemplate( mode, token ):
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   if templateSupportedForType( ftrCapabilities[ ftrType ],
                                ftrSwCapabilities[ ftrType ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardUdpCollector( mode, token ):
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   if collectorSupportedForType( ftrCapabilities[ ftrType ],
                                    ftrSwCapabilities[ ftrType ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardCollector( mode, token ):
   t0( "guardCollector" )
   # guard 'collector' if none of its subtokens are supported on this platform
   for guardFunc in [ guardModSflowExport, guardUdpCollector ]:
      if guardFunc( mode, token ) is None:
         return None
   return CliParser.guardNotThisPlatform

def getTrackerNamesAndIds( ftrType ):
   trackers = [ ( f.name, f.ftId ) for f in
                  hwConfig[ ftrType ].hwFtConfig.values() ]
   return trackers

def getFlowGroups( ftrType, trackerName ):
   hwFtConfig = hwConfig[ ftrType ].hwFtConfig.get( trackerName )
   if hwFtConfig is None:
      return []
   groups = [ g.fgName for g in hwFtConfig.hwFgConfig.values() ]
   return groups

def getFlowGroupsAndIds( ftrType, trackerName ):
   hwFtConfig = hwConfig[ ftrType ].hwFtConfig.get( trackerName )
   if hwFtConfig is None:
      return []
   groups = [ ( g.fgName, g.templateId ) for g in hwFtConfig.hwFgConfig.values() ]
   return groups

def getHwFlowGroupIds( ftrType, trackerName, groupName ):
   hwFtStatus = hwStatus[ ftrType ].hwFtStatus.get( trackerName )
   if hwFtStatus is None:
      return []

   hwFgStatus = hwFtStatus.hwFgStatus.get( groupName )
   if hwFgStatus is None:
      return []

   return hwFgStatus.hwFgId.members()

def getConfigToHwFgConfigMap( ftrType, trackerName ):
   hwFtConfig = hwConfig[ ftrType ].hwFtConfig.get( trackerName )
   if hwFtConfig is None:
      return []
   groups = [ ( g.fgName, g.seqno, g.hwFgNameAndSeqno ) for g in
              hwFtConfig.hwFgConfigMap.values() ]
   return groups

# Config Keywords
trackingConfigKw = CliCommand.guardedKeyword( 'tracking',
                        helpdesc='Configure flow tracking',
                        guard=guardFlowTracking )

sampledConfigKw = CliCommand.guardedKeyword(
                        'sampled',
                        helpdesc='Configure sampled flow tracking',
                        guard=guardFtrSampled )

telemetryConfigKw = CliCommand.guardedKeyword( 'telemetry',
                        helpdesc='Configure sampled telemetry',
                        guard=guardFtrTelemetry )

inbandConfigKw = CliCommand.guardedKeyword( 'inband',
                        helpdesc='Configure sampled inband telemetry',
                        guard=guardFtrTelemetry )

hardwareConfigKw = CliCommand.guardedKeyword(
                        'hardware',
                        helpdesc='Configure hardware flow tracking',
                        guard=guardFtrHardware )

mirrorOnDropConfigKw = CliCommand.guardedKeyword(
                           'mirror-on-drop',
                           helpdesc='Configure mirror on drop feature',
                           guard=guardFtrMirrorOnDrop )

cpuQueueConfigKw = CliCommand.guardedKeyword(
                           'cpu-queue',
                           helpdesc='Configure tracking of CPU queues',
                           guard=guardFtrCpuQueue )

firewallConfigKw = CliCommand.guardedKeyword(
                           'firewall',
                           helpdesc='Configure firewall flow tracking',
                           guard=guardFtrDfw )

distributedConfigKw = CliCommand.guardedKeyword(
                           'distributed',
                           helpdesc='Configure distributed firewall flow tracking',
                           guard=guardFtrDfw )

matcherRecord = CliMatcher.KeywordMatcher( 'record',
   helpdesc='Set characteristics of flow tracking record' )

packetBufferKw = CliCommand.guardedKeyword(
                           'packet-buffer',
                           helpdesc='Configure packet-buffer mirror on drop feature',
                           guard=guardPacketBufferMirrorOnDrop )

packetBufferDropKw = CliMatcher.KeywordMatcher( 'drop',
   helpdesc='Configure packet-buffer mirror on drop' )

packetBufferSampleProbKw25 = '0%-25%'
packetBufferSampleProbKw50 = '25%-50%'
packetBufferSampleProbKw75 = '50%-75%'
packetBufferSampleProbKw100 = '75%-100%'
packetBufferQueueUsageMatcher = CliMatcher.EnumMatcher( {
   packetBufferSampleProbKw25 :
   f'Configure sampling behavior when queue is {packetBufferSampleProbKw25} full',
   packetBufferSampleProbKw50 :
   f'Configure sampling behavior when queue is {packetBufferSampleProbKw50} full',
   packetBufferSampleProbKw75 :
   f'Configure sampling behavior when queue is {packetBufferSampleProbKw75} full',
   packetBufferSampleProbKw100 :
   f'Configure sampling behavior when queue is {packetBufferSampleProbKw100} full',
} )

# Show Keywords
trackingShowKw = CliCommand.guardedKeyword( 'tracking',
                     helpdesc='Show flow tracking',
                     guard=guardFlowTracking )

sampledShowKw = CliCommand.guardedKeyword( 'sampled',
                        helpdesc='Show sampled flow tracking information',
                        guard=guardFtrSampled,
                        storeSharedResult=True )

hardwareShowKw = CliCommand.guardedKeyword( 'hardware',
                        helpdesc='Show hardware flow tracking information',
                        guard=guardFtrHardware,
                        storeSharedResult=True )

telemetryShowKw = CliCommand.guardedKeyword( 'telemetry',
                        helpdesc='Show inband telemetry flow tracking information',
                        guard=guardFtrTelemetry,
                        storeSharedResult=True )

inbandShowKw = CliCommand.guardedKeyword( 'inband',
                        helpdesc='Show inband telemetry flow tracking information',
                        guard=guardFtrTelemetry,
                        storeSharedResult=True )

mirrorOnDropShowKw = CliCommand.guardedKeyword( 'mirror-on-drop',
                        helpdesc='Show mirror on drop flow tracking information',
                        guard=guardFtrMirrorOnDrop,
                        storeSharedResult=True )

cpuQueueShowKw = CliCommand.guardedKeyword( 'cpu-queue',
                        helpdesc='Show CPU queue tracking information',
                        guard=guardFtrCpuQueue,
                        storeSharedResult=True )

firewallShowKw = CliCommand.guardedKeyword( 'firewall',
                        helpdesc='Show firewall flow tracking information',
                        guard=guardFtrDfw,
                        storeSharedResult=True )

distributedShowKw = CliCommand.guardedKeyword( 'distributed',
                        helpdesc=( 'Show distributed firewall flow tracking '
                                   'information' ),
                        guard=guardFtrDfw,
                        storeSharedResult=True )

groupShowKw = CliCommand.guardedKeyword( 'group',
                        helpdesc='Show flow group',
                        guard=guardShowFlowGroupConfig )

trackerKw = CliMatcher.KeywordMatcher( 'tracker', helpdesc='Flow tracker' )

mirrorOnDropCountersShowKw = CliCommand.guardedKeyword( 'counters',
            helpdesc='Counters', guard=guardShowMirrorOnDropTrackerCounters )

mirrorOnDropFlowTableShowKw = CliCommand.guardedKeyword( 'flow-table',
            helpdesc='Flow table', guard=guardShowMirrorOnDropFlowTable )

groupKw = CliMatcher.KeywordMatcher( 'group', helpdesc='Flow group' )

exporterKw = CliMatcher.KeywordMatcher( 'exporter', helpdesc='Flow exporter' )

flowTableKw = CliMatcher.KeywordMatcher( 'flow-table', helpdesc='Flow table' )

countersKw = CliMatcher.KeywordMatcher( 'counters', helpdesc='Counters' )

interfaceKw = CliMatcher.KeywordMatcher( 'interface',
                                         helpdesc='Per interface flow count' )
dropReasonKw = CliCommand.guardedKeyword( 'drop-reason',
                  helpdesc='Drop reason counters', guard=guardShowModReasonCounters )

# Clear 
trackingClearKw = CliCommand.guardedKeyword( 'tracking',
                        helpdesc='Clear flow tracking',
                        guard=guardFlowTracking )

countersClearKw = CliMatcher.KeywordMatcher( 'counters',
                        helpdesc='Clear flow tracking counters' )

sampledClearKw = CliCommand.guardedKeyword( 'sampled',
                        helpdesc='Clear sampled flow tracking information',
                        guard=guardFtrSampled,
                        storeSharedResult=True )

hardwareClearKw = CliCommand.guardedKeyword( 'hardware',
                        helpdesc='Clear hardware flow tracking information',
                        guard=guardFtrHardware,
                        storeSharedResult=True )

firewallClearKw = CliCommand.guardedKeyword( 'firewall',
                        helpdesc='Clear firewall flow tracking information',
                        guard=guardFtrDfw,
                        storeSharedResult=True )

distributedClearKw = CliCommand.guardedKeyword( 'distributed',
                        helpdesc='Clear firewall flow tracking information',
                        guard=guardFtrDfw,
                        storeSharedResult=True )

mirrorOnDropClearKw = CliCommand.guardedKeyword( 'mirror-on-drop',
                        helpdesc='Clear mirror on drop flow tracking information',
                        guard=guardFtrMirrorOnDrop,
                        storeSharedResult=True )

cpuQueueClearKw = CliCommand.guardedKeyword( 'cpu-queue',
                        helpdesc='Clear CPU queue flow tracking information',
                        guard=guardFtrCpuQueue,
                        storeSharedResult=True )

telemetryClearKw = CliCommand.guardedKeyword( 'telemetry',
                        helpdesc='Clear telemetry inband flow tracking information',
                        guard=guardFtrTelemetry,
                        storeSharedResult=True )

inbandClearKw = CliCommand.guardedKeyword( 'inband',
                        helpdesc='Clear telemetry inband flow tracking information',
                        guard=guardFtrTelemetry,
                        storeSharedResult=True )

# Tracker Name
def getTrackerNamesFromContext( mode, context ):
   ftrType = getFtrTypeFromArgs( context.sharedResult )
   return trackingConfig[ ftrType ].flowTrackerConfig

trackerNameMatcher = CliCommand.Node( 
                        CliMatcher.DynamicNameMatcher( getTrackerNamesFromContext,
                                                       "Flow tracker name",
                                                       pattern=r'(?!egress$).+',
                                                       passContext=True ),
                         storeSharedResult=True, maxMatches=1 )

# Group
def getGroupNamesFromContext( mode, context ):
   ftrType = getFtrTypeFromArgs( context.sharedResult )
   trackerName = context.sharedResult[ 'TRACKER_NAME' ]
   if isinstance( trackerName, list ):
      trackerName = trackerName[ 0 ]
   tracker = trackingConfig[ ftrType ].flowTrackerConfig.get( trackerName )
   if tracker:
      return tracker.fgConfig
   else:
      return []

allGroupNameMatcher = CliCommand.Node( CliMatcher.EnumMatcher(
                                         { IP_GROUP : 'IPV4', IP6_GROUP : 'IPV6',
                                           EGRESS_IP_GROUP : 'Egress IPv4',
                                           EGRESS_IP6_GROUP : 'Egress IPv6' } ),
                         storeSharedResult=True, maxMatches=1 )

groupNameMatcher = CliCommand.Node( 
                        CliMatcher.DynamicNameMatcher( getGroupNamesFromContext,
                                                       "Flow group name",
                                                       passContext=True ),
                         storeSharedResult=True )

# Exporter
def getExporterNamesFromContext( mode, context ):
   ftrType = getFtrTypeFromArgs( context.sharedResult )
   trackerName = context.sharedResult[ 'TRACKER_NAME' ]
   tracker = trackingConfig[ ftrType ].flowTrackerConfig.get( trackerName )
   if tracker:
      return tracker.expConfig
   else:
      return []

exporterNameMatcher = CliMatcher.DynamicNameMatcher( getExporterNamesFromContext,
                                                   "Exporter name",
                                                   passContext=True )

class FlowTrackingMode( FlowTrackingModeBase, BasicCli.ConfigModeBase ):
   name = 'Flow tracking configuration'
   showActiveCmdRegistered_ = True

   def __init__( self, parent, session, context ):
      self.context_ = context
      self.session_ = session
      FlowTrackingModeBase.__init__( self, ftrTypeKwStr[ context.ftrType ] )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( 'FlowTrackingMode onExit ...' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def commitContext( self ):
      if self.context_ is None:
         t1( 'FlowTrackingMode.commitContext has no context' )
         return
      context = self.context_
      self.context_ = None
      context.commit()

   def abort( self ):
      t1( 'FlowTrackingMode.abort' )
      self.context_ = None
      self.session_.gotoParentMode()

   def context( self ):
      return self.context_

class TrackerMode( TrackerModeBase, BasicCli.ConfigModeBase ):
   name = 'Tracker configuration'
   showActiveCmdRegistered_ = True

   def __init__( self, parent, session, ftrCtx, trCtx ):
      self.ftrCtx_ = ftrCtx
      self.context_ = trCtx
      self.session_ = session
      TrackerModeBase.__init__( self, ( ftrTypeKwStr[ trCtx.ftrType() ],
                                        trCtx.trackerName() ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( 'TrackerMode onExit...' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      t1( 'FlowTracker.abort' )
      trName = self.trackerName()
      self.context_ = None
      self.session_.gotoParentMode()
      ftrCtx = self.ftrContext()
      del ftrCtx.trCtx[ trName ]
      self.ftrCtx_ = None

   def commitContext( self ):
      if self.context_ is None:
         t1( 'FlowTracker.commitContext has no context' )
         return
      context = self.context_
      self.context_ = None
      context.commit()
      self.ftrCtx_ = None

   def trConfig( self ):
      return self.context_.trConfig

   def ftrContext( self ):
      return self.ftrCtx_

   def context( self ):
      return self.context_

   def trackerName( self ):
      return self.context_.trackerName()

class ExporterMode( ExporterModeBase, BasicCli.ConfigModeBase ):
   name = 'Exporter configuration'

   def __init__( self, parent, session, context, exporterName ):
      self.session_ = session
      self.context_ = context
      ExporterModeBase.__init__( self,
                           ( ftrTypeKwStr[ context.ftrType() ],
                             context.trackerName(), exporterName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def context( self ):
      return self.context_

class EncapFilterMode( EncapFilterModeBase, BasicCli.ConfigModeBase ):
   name = 'Encapsulation filter configuration'

   def __init__( self, parent, session, context ):
      self.session_ = session
      self.context_ = context
      EncapFilterModeBase.__init__( self, ftrTypeKwStr[ context.ftrType ] )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def context( self ):
      return self.context_

class CongestionMode( CongestionModeBase, BasicCli.ConfigModeBase ):
   name = 'Congestion configuration'

   def __init__( self, parent, session, context ):
      self.session_ = session
      self.context_ = context
      CongestionModeBase.__init__( self,
                           ( ftrTypeKwStr[ context.ftrType() ],
                             context.trackerName() ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def context( self ):
      return self.context_

class SampledFlowTrackerCliModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      currentFtrType = ftrTypeByName[ mode.ftrTypeStr ]
      return currentFtrType == ftrTypeSampled

def Plugin( em ):
   global routingHwStatus
   global mplsHwCaps
   global sflowHwStatusDir
   global bridgingHwCapabilities

   sflowHwStatusDir = LazyMount.mount( em, 'sflow/hwstatus', 'Tac::Dir', 'ri' )
   routingHwStatus = LazyMount.mount( em, 'routing/hardware/status',
                                      'Routing::Hardware::Status', 'r' )
   mplsHwCaps = LazyMount.mount( em, 'routing/hardware/mpls/capability',
                                 'Mpls::Hardware::Capability', 'r' )
   bridgingHwCapabilities = LazyMount.mount( em, 'bridging/hwcapabilities',
                                 'Bridging::HwCapabilities', 'r' )

   for ftrType in ftrTypes:
      ftrCapabilities[ ftrType ] = LazyMount.mount( em,
                                    ftrCapabilitiesPathPrefix + ftrType,
                                    ftrCapabilitiesType, 'r' )
      ftrSwCapabilities[ ftrType ] = SwCapabilities( ftrType )
      hwConfig[ ftrType ] = LazyMount.mount( em, 
                                    hwFtrConfigPathPrefix + ftrType,
                                    hwFtrConfigType, 'r' )
      hwStatus[ ftrType ] = LazyMount.mount( em,
                                    hwFtrStatusPathPrefix + ftrType,
                                    hwFtrStatusType, 'r' )
      ftrEntityType = ( ftrMirrorOnDropConfigType if ftrType == ftrTypeMirrorOnDrop
                        else ftrConfigType )
      trackingConfig[ ftrType ] = LazyMount.mount( em,
                                    getFlowTrackingCliConfigPath( ftrType ),
                                    ftrEntityType, 'r' )
