#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Ark import (
   switchTimeToUtc
)
import Tac
import Tracing
from TypeFuture import TacLazyType
from CliPlugin.FlowTrackingCliLib import (
   FtConsts,
)
import CliPlugin.FlowTrackingCounterModel as ftCounterModel

traceHandle = Tracing.Handle( 'FlowTrackingCounterCliLib' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1


CollectorInfo = TacLazyType( 'Smash::Ipfix::CollectorInfo' )
CollectorStatisticsEntry = TacLazyType( 'Smash::Ipfix::CollectorStatisticsEntry' )
CollectorStatisticsKey = TacLazyType( 'Smash::Ipfix::CollectorStatisticsKey' )
FlowCounterKey = TacLazyType( 'Smash::FlowTracker::FlowCounterKey' )
FlowCounterEntry = TacLazyType( 'Smash::FlowTracker::FlowCounterEntry' )
FlowGroupCounterEntry = TacLazyType(
      'Smash::FlowTracker::FlowGroupCounterEntry' )
FlowGroupCounterKey = TacLazyType( 'Smash::FlowTracker::FlowGroupCounterKey' )
IpfixConst = TacLazyType( 'Arnet::IpfixConst' )
SmashCollectorInfo = TacLazyType( 'Smash::Ipfix::CollectorInfo' )
TemplateIdType = TacLazyType( 'Ipfix::TemplateId' )

def addCollectorSetsAndTemplates( collModel, stats, softwareFlowTable,
                                  counterModel ):
   templateSetIds = {
      IpfixConst.optionsTemplateSetId: 'optionsTemplate',
      IpfixConst.templateSetId: 'template',
   }
   for entry in stats.setEntryStorage.values():
      setId = entry.setId
      if setId == CollectorStatisticsEntry.setIdNotFound:
         continue
      if setId in templateSetIds:
         if not counterModel:
            templateModel = ftCounterModel.CollectorTemplateCounters()
         else:
            templateModel = counterModel.CollectorTemplateCounters()
         templateModel.templateType = templateSetIds[ setId ]
         templateModel.templates = entry.exportedTemplateCount
         collModel.templates.append( templateModel )
         t1( 'added template set, type:', templateModel.templateType )
      else:
         if not counterModel:
            setIdMin = FtConsts.reservedTemplateId( FtConsts.dataTemplateMin )
            setIdMax = FtConsts.reservedTemplateId( FtConsts.dataTemplateMax )
            if softwareFlowTable or not( setId >= setIdMin and setId <= setIdMax ):
               setModel = ftCounterModel.CollectorSetCounters()
               setModel.flowRecords = entry.exportedFlowRecordCount
               collModel.sets[ setId ] = setModel
         else:
            setModel = counterModel.CollectorSetCounters()
            setModel.flowRecords = entry.exportedFlowRecordCount
            collModel.sets[ setId ] = setModel
         t1( 'added data set, setId:', setId )   

def addCollector( expModel, hwExpConfig, trName, expName, collKey,
                  ipfixStats, softwareFlowTable, counterModel ):
   smashKey = CollectorStatisticsKey( trName, expName, collKey )
   t1( 'Process collector', smashKey.smashString() )
   stats = ipfixStats.stats.get( smashKey )
   if not stats:
      t0( 'No stats for', smashKey.smashString() )
      stats = CollectorStatisticsEntry( smashKey )
   if not counterModel:
      collModel = ftCounterModel.CollectorCounters()
      addrAndPort = ftCounterModel.IpGenericAddrAndPort()
   else:
      collModel = counterModel.CollectorCounters()
      addrAndPort = counterModel.IpGenericAddrAndPort()
   addrAndPort.ip = collKey.addr
   addrAndPort.port = collKey.port
   collModel.addrAndPort = addrAndPort
   if softwareFlowTable or counterModel:
      collModel.exportedMessageTotalCount = stats.exportedMessageTotalCount
      collModel.exportedFlowRecordTotalCount = stats.exportedFlowRecordTotalCount
      collModel.exportedOctetTotalCount = stats.exportedOctetTotalCount
   if not counterModel:
      lastUpdates = ftCounterModel.CollectorTimestamp()
   else:
      lastUpdates = counterModel.CollectorTimestamp()
   lastUpdates.message = switchTimeToUtc( stats.lastSentTime )
   lastUpdates.template = switchTimeToUtc( stats.lastTemplateSentTime )
   lastUpdates.dataRecord = switchTimeToUtc( stats.lastDataRecordSentTime )
   lastUpdates.optionsData = switchTimeToUtc( stats.lastOptionsDataSentTime )
   collModel.lastUpdates = lastUpdates
   addCollectorSetsAndTemplates( collModel, stats, softwareFlowTable,
                                 counterModel )
   expModel.collectors.append( collModel )

def addCollectors( expModel, hwExpConfig, trName, expName, ipfixStats,
                   softwareFlowTable, counterModel ):
   for collector in hwExpConfig.collectorIpAndPort.values():
      collKey = SmashCollectorInfo.fromIpAndPort( collector )
      addCollector( expModel, hwExpConfig, trName, expName, collKey,
                    ipfixStats, softwareFlowTable, counterModel )
   if not hwExpConfig.collectorIpAndPort:
      t1( 'No IPv4 collectors for tracker', trName, 'exporter', expName )

   for collector in hwExpConfig.collectorIp6AndPort.values():
      collKey = SmashCollectorInfo.fromIp6AndPort( collector )
      addCollector( expModel, hwExpConfig, trName, expName, collKey,
                    ipfixStats, softwareFlowTable, counterModel )
   if not hwExpConfig.collectorIp6AndPort:
      t1( 'No IPv6 collectors for tracker', trName, 'exporter', expName )

def addExporters( trModel, trName, expFilter, ftConfig, ipfixStats,
                  softwareFlowTable, counterModel=None ):
   hwFtConfig = ftConfig.hwFtConfig.get( trName )
   if not hwFtConfig:
      t0( 'WARNING: no hwFtConfig for tracker', trName )
      return
   for hwExpConfig in hwFtConfig.expConfig.values():
      expName = hwExpConfig.name
      if expFilter and expFilter != expName:
         t1( 'exporter', expName, 'did not match filter' )
         continue
      t1( 'process exporter', expName )
      if not counterModel:
         expModel = ftCounterModel.ExporterCounters()
      else:
         expModel = counterModel.ExporterCounters()
      expModel.exporterType = 'ipfix'
      clearTimeKey = CollectorStatisticsKey( trName, expName, CollectorInfo() )
      clearTime = ipfixStats.stats.get( clearTimeKey )
      if clearTime and clearTime.key == clearTimeKey:
         expModel.clearTime = switchTimeToUtc( clearTime.flowEntry.lastClearedTime )
      addCollectors( expModel, hwExpConfig, trName, expName, ipfixStats,
                     softwareFlowTable, counterModel )
      trModel.exporters[ expName ] = expModel
   if not hwFtConfig.expConfig:
      t0( 'No exporters for tracker', trName, 'hwFtConfig', hwFtConfig )

