#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Ark import timestampToStr
from ArnetModel import (
   IpGenericAddrAndPort,
)
from CliModel import (
   Bool,
   Dict,
   Enum,
   Float,
   Int,
   List,
   Model,
   Submodel,
)
from FlowTrackerCliUtil import (
   ftrTypeHardware,
   ftrTypeShowStr,
   renderCounter,
)
from TypeFuture import TacLazyType
import Tac

FtConsts = TacLazyType( 'FlowTracking::Constants' )

def pindent( indent, *args ):
   print( ' ' * ( 2 * indent - 1 ), *args )

def utcTimestampToStr( timestamp ):
   return timestampToStr( timestamp, now=Tac.utcNow() )

def lastSentStr( timestamp ):
   if timestamp is None:
      return ''
   else:
      return ', last sent ' + utcTimestampToStr( timestamp )

def collectorAddr( addrAndPort ):
   if addrAndPort.ip.af == 'ipv6':
      fmtStr = '[{}]:{}'
   else:
      fmtStr = '{}:{}'
   return fmtStr.format( addrAndPort.ip, addrAndPort.port )

def templateIdNum( staticTemplateIdEnum ):
   return FtConsts.reservedTemplateId( staticTemplateIdEnum )

# flow tracking counters
class CollectorSetCounters( Model ):
   flowRecords = Int(
      help="Number of flow records sent to the collector" )

class CollectorTemplateCounters( Model ):
   templateType = Enum(
      help="IPFIX template type",
      values=( "template", "optionsTemplate", ) )
   templates = Int(
      help="Number of templates sent to the collector" )

class CollectorTimestamp( Model ):
   message = Float(
      help="UTC time of last exported message",
      optional=True )
   template = Float(
      help="UTC time of last exported template",
      optional=True )
   dataRecord = Float(
      help="UTC time of last exported data record",
      optional=True )
   optionsData = Float(
      help="UTC time of last exported options data record",
      optional=True )

class CollectorCounters( Model ):
   addrAndPort = Submodel(
      help="Collector IP address and port number",
      valueType=IpGenericAddrAndPort )
   exportedMessageTotalCount = Int(
      help="The total number of messages sent to the collector",
      optional=True )
   exportedFlowRecordTotalCount = Int(
      help="The total number of flow records sent to the collector",
      optional=True )
   exportedOctetTotalCount = Int(
      help="The total number of bytes sent to the collector",
      optional=True )
   lastUpdates = Submodel(
      help="UTC time of last events",
      valueType=CollectorTimestamp )
   sets = Dict(
      help="A mapping of IPFIX template ID (set ID) to per-set-id counters",
      keyType=int,
      valueType=CollectorSetCounters )
   templates = List(
      help="A list of per-template counters",
      valueType=CollectorTemplateCounters )

class ExporterCounters( Model ):
   exporterType = Enum(
      help="Exporter type",
      values=( "ipfix", ) )
   clearTime = Float(
      help="UTC time when counters were last cleared",
      optional=True )
   collectors = List(
      help="A list of per-collector counters",
      valueType=CollectorCounters )

def exporterTypeStr( exporterType ):
   mapping = {
      'ipfix': 'IPFIX',
   }
   return mapping[ exporterType ]

class FlowGroupCounters( Model ):
   activeFlows = Int(
      help="Number of active flows" )
   expiredFlows = Int(
      help="Cumulative expired flow count" )
   flows = Int(
      help="Cumulative flow count" )
   packets = Int(
      help="Cumulative count of packets" )

class TrackerCounters( Model ):
   activeFlows = Int(
      help="Number of active flows", optional=True )
   expiredFlows = Int(
      help="Cumulative expired flow count", optional=True )
   flows = Int(
      help="Cumulative flow count", optional=True )
   packets = Int(
      help="Cumulative count of packets", optional=True )
   clearTime = Float(
      help="UTC time when counters were last cleared",
      optional=True )
   flowGroups = Dict(
      help="A mapping of flow group name to counters",
      keyType=str,
      valueType=FlowGroupCounters )
   exporters = Dict(
      help="A mapping of exporter name to counters",
      valueType=ExporterCounters )

class FtrCounters( Model ):
   running = Bool( help="Flow tracking agent is running" )
   softwareFlowTable = Bool( help="Software flow table is available" )
   trackers = Dict(
      help="A mapping of tracker name to counters",
      keyType=str,
      valueType=TrackerCounters )
   flows = Int( help="Cumulative flow count", optional=True )
   activeFlows = Int( help="Cumulative active flow count", optional=True )
   expiredFlows = Int( help="Cumulative expired flow count", optional=True )
   packets = Int( help="Cumulative count of packets", optional=True )
   clearTime = Float(
         help="UTC time when total counters were last cleared",
         optional=True )

   def renderFlowGroup( self, indent, groupInfo ):
      pindent( indent, '{} flows, {} RX packets'.format(
         renderCounter( groupInfo.activeFlows ),
         renderCounter( groupInfo.packets ) ) )

   def renderCollector( self, indent, collector ):
      flowRecords = 0
      optionsDataRecords = 0
      setIdMin = templateIdNum( FtConsts.dataTemplateMin )
      setIdMax = templateIdNum( FtConsts.dataTemplateMax )
      for setId, setCounter in collector.sets.items():
         if setId >= setIdMin and setId <= setIdMax:
            flowRecords += setCounter.flowRecords
         else:
            optionsDataRecords += setCounter.flowRecords
      if self.softwareFlowTable:
         pindent( indent, '{} messages{}'.format(
            renderCounter( collector.exportedMessageTotalCount ),
            lastSentStr( collector.lastUpdates.message ) ) )
         pindent( indent, '{} flow records{}'.format(
            renderCounter( flowRecords ),
            lastSentStr( collector.lastUpdates.dataRecord ) ) )
      pindent( indent, '{} options data records{}'.format(
         renderCounter( optionsDataRecords ),
         lastSentStr( collector.lastUpdates.optionsData ) ) )
      templates = sum( x.templates for x in collector.templates )
      pindent( indent, '{} templates{}'.format(
         renderCounter( templates ),
         lastSentStr( collector.lastUpdates.template ) ) )

   def renderExporter( self, indent, expInfo ):
      for collector in expInfo.collectors:
         pindent( indent, 'Collector:', collector.addrAndPort.ip, "port",
                  collector.addrAndPort.port )
         self.renderCollector( indent + 1, collector )

   def renderTrackerDetails( self, indent, tracker ):
      if tracker.clearTime:
         lastCleared = ' (Last cleared {})'.format(
               utcTimestampToStr( tracker.clearTime ) )
      else:
         lastCleared = ''
      if self.softwareFlowTable:
         pindent( indent, '{} flows, {} RX packets{}'.format(
            renderCounter( tracker.activeFlows ),
            renderCounter( tracker.packets ),
            lastCleared ) )
         pindent( indent,
                  'Flows created: {}, expired: {}'.format(
                  renderCounter( tracker.flows ),
                  renderCounter( tracker.expiredFlows ) ) )
         for fgName in sorted( tracker.flowGroups ):
            groupInfo = tracker.flowGroups[ fgName ]
            pindent( indent, 'Group:', fgName )
            self.renderFlowGroup( indent + 1, groupInfo )

      for expName in sorted( tracker.exporters ):
         expInfo = tracker.exporters[ expName ]
         expType = exporterTypeStr( expInfo.exporterType )
         lastCleared = ''
         if expInfo.clearTime:
            lastCleared = ' (Last cleared {})'.format(
                  utcTimestampToStr( expInfo.clearTime ) )
         pindent( indent, 'Exporter:', expName, '({}){}'.format( expType,
                                                                 lastCleared ) )
         self.renderExporter( indent + 1, expInfo )

   def renderTrackers( self, indent ):
      for trName in sorted( self.trackers ):
         tracker = self.trackers[ trName ]
         pindent( indent, 'Tracker:', trName )
         self.renderTrackerDetails( indent + 1, tracker )

   def render( self ):
      '''
      Render FtrCounters and all submodels.

      Sample output:
        Total active flows: 1.05K (1050), RX packets: 3.139T (3139123456789)
         (Last cleared 4 days, 19:55:12 ago)
        Total flows created: 4K (4000), expired: 3.05K (3050)
        Tracker: ftr1
          100 flows, 25.04K (25040) RX packets (Last cleared 4 days, 19:55:12 ago)
          Flows created: 2K (2000), expired: 1.5K (1500)
          Group: IPv4
            10 flows, 100 RX packets
          Group: IPv6
            2K (2000) flows, 2Q (2000000000000000) RX packets
          Group: VxlanIPv4
            300 flows, 2B (2000000000) RX packets
          Group: VxlanIPv6
            40 flows, 2M (2000000) RX packets
          Exporter: exp1 (IPFIX) (Last cleared 3 days, 18:55:12 ago)
            Collector: 10.0.1.6 port 2055
              1.5K (1500) messages, last sent 0:03:30 ago
              200 flow records, last sent 0:03:30 ago
              1K (1000) options data records, last sent 0:02:30 ago
              300 templates, last sent 0:05:00 ago
            Collector: b::6 port 1345
              1.5K (1500) messages, last sent 0:03:30 ago
              200 flow records, last sent 0:03:30 ago
              1K (1000) options data records, last sent 0:02:30 ago
              300 templates, last sent 0:05:00 ago
        Tracker: ftr2
          500 flows, 5.09K (5090) RX packets (Last cleared 3 days, 17:55:12 ago)
          Flows created: 2K (2000), expired: 1.55K (1550)
          ...
      '''
      if not self.running:
         print( "{} flow tracking is not active".format(
                ftrTypeShowStr[ ftrTypeHardware ] ) )
         return
      indent = 0
      if self.clearTime:
         lastCleared = ' (Last cleared {})'.format(
               utcTimestampToStr( self.clearTime ) )
      else:
         lastCleared = ''
      if self.softwareFlowTable:
         pindent( indent,
               'Total active flows: {}, RX packets: {}{}'.format(
                  renderCounter( self.activeFlows ),
                  renderCounter( self.packets ),
                  lastCleared ) )
         pindent( indent, 'Total flows created: {}, expired: {}'.format(
            renderCounter( self.flows ),
            renderCounter( self.expiredFlows ) ) )
      self.renderTrackers( indent )
