# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import TableOutput
import Tac
from CliModel import (
   Bool,
   Dict,
   Enum,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
from CliPlugin.FlowTrackingCliLib import (
   FtConsts,
   IpfixRecordType,
   SelectorAlgorithmType,
)
from FlowTrackerCliUtil import FtrTypeEnum
from natsort import natsorted

# flow tracking ipfix template
class TemplateEntryModel( Model ):
   fieldId = Enum( help="IPFIX template field ID",
                   values=IpfixRecordType.attributes )
   fieldLen = Int( help="IPFIX template field length (in bytes)" )
   fieldType = Str( help="IPFIX template field data type" )
   enterpriseIe = Bool( help="IPFIX template filed is enterprise IE", optional=True )
   entNum = Int( help="Ipfix Enterprise Number", optional=True )

class TemplateModel( Model ):
   name = Str( help="Template name" )
   templateId = Int( help="Template ID" )
   scopeFields = List( valueType=TemplateEntryModel,
                       help="Ordered list of template scope fields", optional=True )
   fields = List( valueType=TemplateEntryModel,
                  help="Ordered list of template fields" )
   groups = List( valueType=str, optional=True,
                  help="List of flow groups using the template" )

class TrackerTemplateModel( Model ):
   dataTemplates = Dict( keyType=int, valueType=TemplateModel,
                         help="A mapping between template ID and data template" )
   optionsTemplates = Dict( keyType=int, valueType=TemplateModel,
                      help="A mapping between template ID and options template" )

class FtTemplateModel( Model ):
   trackers = Dict( keyType=str, valueType=TrackerTemplateModel,
                    help="A mapping between tracker name and tracker templates" )

   def renderField( self, indent, field ):
      fieldLen = ( 'variable length' if field.fieldLen == FtConsts.variableLengthIe
                   else '%d bytes' % field.fieldLen )
      print( '%s%s%s (%d), %s' %
             ( indent, field.fieldType,
               '[E]' if field.enterpriseIe else '',
               Tac.enumValue( 'Ipfix::RecordType', field.fieldId ), fieldLen ) )

   def renderOptionsTemplate( self, indent, optionsTemplate ):
      print( '%sOptions Template: %s, Fields: %d, Template ID: %d' %
              ( indent, optionsTemplate.name, len( optionsTemplate.fields ),
                optionsTemplate.templateId ) )
      indent += '  '
      for field in optionsTemplate.scopeFields:
         self.renderField( indent, field )
      for field in optionsTemplate.fields:
         self.renderField( indent, field )
      print()

   def renderDataTemplate( self, indent, dataTemplate ):
      print( '%sData Template: %s, Fields: %d, Template ID: %d' %
              ( indent, dataTemplate.name, len( dataTemplate.fields ),
                dataTemplate.templateId ) )
      print( '%sGroups: %s' %
             ( indent, ", ".join( natsorted( dataTemplate.groups ) ) ) )
      indent += '  '
      for field in dataTemplate.fields:
         self.renderField( indent, field )
      print()

   def render( self ):
      '''
      Render data templates and options templates.

      Sample output:
      Tracker: tr1
        Data Template: IPv4, Fields: 26, Template ID: 261
        Groups: IPv4
          ingressVRFID (234), 4 bytes
          vlanId (58), 2 bytes
          sourceIPv4Address (8), 4 bytes
          ... ...

        Data Template: IPv6, Fields: 26, Template ID: 262
        Groups: IPv6
          ingressVRFID (234), 4 bytes
          vlanId (58), 2 bytes
          sourceIPv6Address (27), 16 bytes
          ... ...

        Options Template: VRF Mapping, Fields: 1, Template ID: 256
          ingressVRFID (234), 4 bytes
          VRFname (236), variable length

        Options Template: Interface Mapping, Fields: 1, Template ID: 257
          ingressInterface (10), 4 bytes
          interfaceName (82), variable length

        Options Template: Flow Key, Fields: 1, Template ID: 258
          templateId (145), 2 bytes
          flowKeyIndicator (173), 8 bytes

        Options Template: Tracker, Fields: 7, Template ID: 259
          observationDomainId (149), 4 bytes
          observationDomainName (300), variable length
          flowActiveTimeout (36), 2 bytes
          flowIdleTimeout (37), 2 bytes
          selectorAlgorithm (304), 2 bytes
          samplingSize (309), 4 bytes
          samplingPopulation (310), 4 bytes
          flowTrackingType[E] (1001), 2 bytes
      '''
      indent = '  '
      for trackerName, trackerModel in sorted( self.trackers.items() ):
         if trackerModel.dataTemplates or trackerModel.optionsTemplates:
            print( 'Tracker: %s' % trackerName )
         for _, dataTemplate in sorted( trackerModel.dataTemplates.items() ):
            self.renderDataTemplate( indent, dataTemplate )
         for _, optionsTemplate in sorted( trackerModel.optionsTemplates.items() ):
            self.renderOptionsTemplate( indent, optionsTemplate )

# flow tracking ipfix options-table
class OptTableModel( Model ):
   templateId = Int( help="Options template ID" )
   scopeFields = List( valueType=str, help="List of options table scope fields" )
   entries = Dict( keyType=int, valueType=str,
                   help="A mapping between table entry ID and entry value" )

class TrackerInfoModel( Model ):
   observationDomain = Str( help="Observation domain name" )
   observationDomainId = Int( help="Observation domain ID" )
   activeInterval = Int( help="Flow active interval in seconds" )
   inactiveTimeout = Int( help="Flow inactive timeout in seconds" )
   selectorAlgorithm = Enum( help="Flow selector algorithm",
                             values=SelectorAlgorithmType.attributes )
   sampleRate = Int( help="Flow tracking sample every x packets" )
   flowTrackingType = Enum( help="Flow tracking type",
                            values=FtrTypeEnum.attributes )

class TrackerOptTableModel( Model ):
   trackerInfo = Submodel( valueType=TrackerInfoModel,
                           help="Tracker info options table", optional=True )
   tables = Dict( keyType=str, valueType=OptTableModel,
                  help="A mapping between table name and options table",
                  optional=True )

class FtOptTableModel( Model ):
   trackers = Dict( keyType=str, valueType=TrackerOptTableModel,
                    help="A mapping between tracker name and tracker options table" )

   def renderTrackerInfo( self, indent, trackerInfo ):
      print( "%sObservation domain: %s, ID: %d" %
              ( indent, trackerInfo.observationDomain,
                trackerInfo.observationDomainId ) )
      print( "%sActive interval: %dsec" % ( indent, trackerInfo.activeInterval ) )
      print( "%sInactive timeout: %dsec" % ( indent, trackerInfo.inactiveTimeout ) )
      algoStr = ( "random" if trackerInfo.selectorAlgorithm ==
                  SelectorAlgorithmType.randomNofNSampling else "" )
      print( "{}Selector algorithm: {}({})".format( indent, algoStr, Tac.enumValue(
         'Ipfix::SelectorAlgorithm', trackerInfo.selectorAlgorithm ) ) )
      print( "%sSampling: 1/%d" % ( indent, trackerInfo.sampleRate ) )
      print( "{}Flow tracking type: {}({})".format( indent,
            trackerInfo.flowTrackingType,
            Tac.enumValue( 'FlowTracking::FlowTrackingType',
                           trackerInfo.flowTrackingType ) ) )
      print()

   def renderOptTable( self, indent, tableHeadings, tableName, tableModel ):
      headings = tableHeadings[ tableName ]
      table = TableOutput.createTable( headings, indent=len( indent ) )
      formatLeft = TableOutput.Format( justify="left" )
      table.formatColumns( formatLeft, formatLeft )
      print( "%s%s Table, Template ID: %d, Scope: %s" %
         ( indent, tableName, tableModel.templateId,
           ", ".join( tableModel.scopeFields ) ) )
      for key, value in sorted( tableModel.entries.items() ):
         if tableName == 'Interface':
            key = hex( key )
         table.newRow( key, value )
      print( table.output() )

   def render( self ):
      '''
      Render options tables.

      Sample output:
      Tracker: ftr1
         Observation domain: ftr1, ID: 1
         Active interval: 300sec
         Inactive timeout: 15sec
         Selector algorithm: random(3)
         Sampling: 1/1048576
         Flow tracking type: sampled(1)

         VRF Table, Template ID: 256, Scope: ingressVRFID
            VRF ID      VRF Name
         -------------- --------
            0           default
            2           red
            16777215

         Interface Table, Template ID: 257, Scope: ingressInterface
            Interface ID    Interface Name
         ------------------ --------------
            0x0             unknown
            0x1             Ethernet1
            0x2             Ethernet2
            0x3             Ethernet3
            0x3fffffff      CPU
            0x40000000      discard
            0x80000000      multicast

         Flow Keys Table, Template ID: 258, Scope: templateId
            Template ID    Flow Key Indicator
         ----------------- ------------------
            261            0x7f
            262            0x7f
      '''

      tableHeadings = {
         "VRF" : ( "VRF ID", "VRF Name" ),
         "Interface" : ( "Interface ID", "Interface Name" ),
         "Flow Keys" : ( "Template ID", "Flow Key Indicator" ),
      }
      for tracker, trackerModel in sorted( self.trackers.items() ):
         print( "Tracker: %s" % tracker )
         indent = "  "
         if trackerModel.trackerInfo:
            self.renderTrackerInfo( indent, trackerModel.trackerInfo )
         for tableName, tableModel in sorted( trackerModel.tables.items(),
                                              key=lambda x: x[ 1 ].templateId ):
            self.renderOptTable( indent, tableHeadings, tableName, tableModel )
