# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys
import typing

import Arnet
import BasicCli
import CliGlobal
import LazyMount
import Tracing
from CliToken.Flow import flowMatcherForShow
import CliExtensions
from Toggles.FlowTrackerToggleLib import toggleSftBgpExportCliEnabled
from FlowTrackerCliUtil import (
   ftrConfigType,
   ftrMirrorOnDropConfigType,
   ftrTypes,
   ftrTypeKwStr,
   ftrTypeMirrorOnDrop,
   ftrTypeCpuQueue,
   ftrTypeSampled,
   ftrTypeHardware,
   getFtrTypeFromArgs,
   hwFtrConfigPathPrefix,
   hwFtrStatusPathPrefix,
   ftrCapabilitiesPathPrefix,
   hwFtrConfigType,
   hwFtrStatusType,
   ftrCapabilitiesType,
   collectorInactiveReason,
   trackerInactiveReason,
   AddressFamily,
   ftrStatusPathPrefix,
   ftrStatusType,
   exporterInactiveReasonEnum,
   getExpReasonEnumList,
   getFlowTrackingCliConfigPath,
   getHardwareFlowTrackingFlowWatcherConfigPath,
   reservedGroupToSeqnoMap,
   dscpSupportedForType,
   exportFormatShowStr,
   showFtr,
   showActiveGroups,
   showCongestion,
   showExporter as _showExporter,
   showFlowTracking,
   showFlowTrackingTrailer,
   supportedExportFormatsForType,
   sampleModeAll,
   sampleModeFiltered,
   packetBufferStatusEnum,
)
from FlowTrackerConst import (
   activeInterval,
   inactiveTimeout,
   mirrorSelectorAlgo,
   constants,
   spaceConst,
)
from CliPlugin.FlowTrackingModel import ( CongestionIntervalModel )
from CliPlugin.FlowTrackingCliLib import (
   exporterNameMatcher,
   exporterKw,
   IP_GROUP,
   IP6_GROUP,
   hardwareShowKw,
   flowTrackingAgentRunning,
   trackingShowKw,
   cpuQueueShowKw,
   sampledShowKw,
   telemetryShowKw,
   inbandShowKw,
   firewallShowKw,
   distributedShowKw,
   mirrorOnDropShowKw,
   trackerKw,
   trackerNameMatcher,
   getFtrCaps,
   getFtrSwCaps,
   getMplsHwCaps,
   getBridgingHwCaps,
   groupShowKw,
   groupNameMatcher,
   packetBufferModFtrSupported,
   packetBufferSamplingProbSupported,
   packetBufferShaperSupported,
)
from CliPlugin import FlowTrackingModel
import ShowCommand
from TypeFuture import TacLazyType

traceHandle = Tracing.Handle( 'FlowTrackingShowCli' )

IntfId = TacLazyType( 'Arnet::IntfId' )
SubIntfId = TacLazyType( 'Arnet::SubIntfId' )
SwCapabilities = TacLazyType( 'FlowTracking::Capabilities' )
ExportFormatEnum = TacLazyType( 'FlowTracking::ExportFormat' )
SwCapabilities = TacLazyType( 'FlowTracking::Capabilities' )
PacketBufferStatus = TacLazyType( 'FlowTracking::PacketBuffer::Status' )

# Global variable holder.
gv = CliGlobal.CliGlobal(
   dict(
      activeAgentDir=None,
      entityManager=None,
      hwConfig={},
      hwTrackingStatus={},
      trackingConfig={},
      ftrCapabilities={},
      ftrSwCapabilities={},
      ftrStatus={},
      sampledHoStatus=None,
      flowWatcherTrackingConfig=None,
      dropExportHwStatus=None,
   )
)
isEMAvailable = CliExtensions.CliHook()

def _calcExporterActive( exporter, expStatus ) -> bool:
   """
      Args:
        exporter: None or instance of HwFlowTracking::ExporterConfig
        expStatus: None or instance of FlowTracking::ExporterStatus
      Returns:
        Is the exporter considered active?
   """
   if exporter is None:
      return False
   if expStatus:
      return expStatus.active
   return False

def _calcExportFormat( hwExporterConfig, expConfig ):
   """
      Args:
         hwExporterConfig: None or HwFlowTracking::ExporterConfig
         expConfig: instance of FlowTracking::ExporterConfig
      Returns:
         Instance of FlowTracking::ExporterConfig
   """
   if hwExporterConfig is None:
      return expConfig.exportFormat
   else:
      return hwExporterConfig.exportFormat

def _calcExporterFormatModel(
      hwExporterConfig,
      expConfig ) -> FlowTrackingModel.ExpFormatModel:
   """
      Args:
         hwExporterConfig: None or HwFlowTracking::ExporterConfig
         expConfig: instance of FlowTracking::ExporterConfig
   """
   expFormatModel = FlowTrackingModel.ExpFormatModel()
   exportFormat = _calcExportFormat( hwExporterConfig, expConfig )
   expFormatModel.name = exportFormatShowStr[ exportFormat ]
   if exportFormat == ExportFormatEnum.formatIpfix:
      if hwExporterConfig is None:
         expFormatModel.version = expConfig.ipfixVersion
         expFormatModel.mtu = expConfig.ipfixMtu
      else:
         expFormatModel.version = hwExporterConfig.ipfixVersion
         expFormatModel.mtu = hwExporterConfig.ipfixMtu
   return expFormatModel


# SHOW COMMANDS
#------------------------------------------------------------
# show flow tracking ( sampled | hardware | mirror-on-drop |
#                      ( telemetry inband ) | ( firewall distributed ) )
#         [ tracker < name > [ exporter < name > ] ]
#------------------------------------------------------------
def showExporter( ftrType, trName, trackerConfig, trackerModel, expName, tracker,
                  ftStatus ):
   # get exporter from hwConfig.
   exporter = tracker.expConfig.get( expName ) if tracker else None

   expConfig = None
   if trackerConfig:
      expConfig = trackerConfig.expConfig.get( expName )
   if exporter is None and expConfig is None:
      # exporter doesn't exist in both CLI config and hwConfig
      return

   expStatus = None
   if ftStatus:
      expStatus = ftStatus.expStatus.get( expName )

   expModel = FlowTrackingModel.ExpModel()
   expModel.active = _calcExporterActive( exporter, expStatus )
   trackerModel.exporters[ expName ] = expModel
   expReasonModel = FlowTrackingModel.ExpReasonModel()
   supportedExportFormats = supportedExportFormatsForType(
      gv.ftrCapabilities[ ftrType ], gv.ftrSwCapabilities[ ftrType ] )
   # Exporter configured but not yet present in hwConfig
   if exporter is None:
      reason = FlowTrackingModel.ExpReason()
      reason.expReason = exporterInactiveReasonEnum.exporterNotProcessed
      expReasonModel.expReasons.append( reason )
      expModel.inactiveReasons[ AddressFamily.ipunknown ] = expReasonModel
      # We need to provide some default values for mandatory arguments in
      # ExpModel
      if ftrType != ftrTypeCpuQueue:
         expModel.dscpValue = expConfig.dscpValue
      expModel.localIntf = expConfig.localIntfName
      expModel.srcIpAddr = constants.ipAddrDefault
      expModel.srcIp6Addr = constants.ip6AddrDefault

      if ExportFormatEnum.formatIpfix in supportedExportFormats:
         expModel.templateInterval = expConfig.templateInterval
      if ftrType == ftrTypeCpuQueue:
         expModel.vrfName = None
      else:
         expModel.vrfName = ''
      exportFormat = expConfig.exportFormat
      expModel.exportFormat = _calcExporterFormatModel( exporter, expConfig )
      return

   # Things should not crash if expStatus doesn't exist for any reason (though
   # this is unlikely)
   if not expStatus:
      reason = FlowTrackingModel.ExpReason()
      reason.expReason = exporterInactiveReasonEnum.exporterInactiveReasonUnknown
      expReasonModel.expReasons.append( reason )
      expModel.inactiveReasons[ AddressFamily.ipunknown ] = expReasonModel

   if expStatus and not expStatus.active:
      # It might be possible that expStatus exists but expStatus.inactiveReason is
      # not set because tacc reactors have not yet run.
      if not expStatus.inactiveReason:
         reason = FlowTrackingModel.ExpReason()
         reason.expReason = exporterInactiveReasonEnum.exporterInactiveReasonUnknown
         expReasonModel.expReasons.append( reason )
         expModel.inactiveReasons[ AddressFamily.ipunknown ] = expReasonModel
      else:
         for ipVersion in expStatus.inactiveReason:
            inactiveReason = expStatus.inactiveReason.get( ipVersion )
            reasonEnumList = getExpReasonEnumList( inactiveReason )
            expReasonModel = FlowTrackingModel.ExpReasonModel()
            expReasonModel.expReasons = []
            for reasonEnum in reasonEnumList:
               reason = FlowTrackingModel.ExpReason()
               reason.expReason = reasonEnum
               expReasonModel.expReasons.append( reason )
            expModel.inactiveReasons[ ipVersion ] = expReasonModel

   vrfName = exporter.vrfName
   dscpValue = exporter.dscpValue
   expConfig = None
   trackerConfig = gv.trackingConfig[ ftrType ].flowTrackerConfig.get( trName )
   localIntf = IntfId( "" )
   srcIpAddr = exporter.srcIpAddr
   srcIp6Addr = exporter.srcIp6Addr
   if trackerConfig:
      expConfig = trackerConfig.expConfig.get( expName )
      if expConfig:
         localIntf = IntfId( expConfig.localIntfName )
   exportFormat = exporter.exportFormat
   if exportFormat == ExportFormatEnum.formatSflow:
      dscpValue = None
      expModel.useSflowCollectorConfig = exporter.useSflowCollectorConfig
      if exporter.useSflowCollectorConfig:
         # Omit local/source info as we are using the sFlow configured source
         localIntf = IntfId( "" )
         srcIpAddr = Arnet.IpAddress( 0 )
         srcIp6Addr = Arnet.Ip6Addr( '::' )
         vrfName = None
   if ExportFormatEnum.formatIpfix in supportedExportFormats:
      expModel.templateInterval = exporter.templateInterval
   if vrfName is not None and ftrType != ftrTypeCpuQueue:
      expModel.vrfName = vrfName
   if dscpSupportedForType( gv.ftrCapabilities[ ftrType ],
                            gv.ftrSwCapabilities[ ftrType ] ):
      expModel.dscpValue = dscpValue
   expModel.collectors = {}
   expModel.localIntf = localIntf
   expModel.srcIpAddr = srcIpAddr
   expModel.srcIp6Addr = srcIp6Addr
   expModel.exportFormat = _calcExporterFormatModel( exporter, expConfig )
   if ftrType != ftrTypeCpuQueue:
      configCollector = []
      hwConfigCollector = []
      if expConfig:
         configCollector = list( expConfig.collectorHostAndPort )
      hwConfigCollector = list( set( exporter.collectorIpAndPort.keys() ).union(
                                set( exporter.collectorIp6AndPort.keys() ) ) )
      collectors = list( set( configCollector ).union(
                         set( hwConfigCollector ) ) )
      if expModel.useSflowCollectorConfig:
         # Omit collectors when we are using sFlow's configured collectors
         collectors = []

      for collectorName in collectors:
         collectorModel = FlowTrackingModel.CollectorModel()
         expModel.collectors[ collectorName ] = collectorModel
         collector = ( exporter.collectorIpAndPort.get( collectorName ) or
                       exporter.collectorIp6AndPort.get( collectorName ) )
         if collector:
            collectorModel.active = True
         else:
            collectorModel.active = False
            if( len( exporter.collectorIpAndPort ) +
                len( exporter.collectorIp6AndPort ) >=
                gv.ftrCapabilities[ ftrType ].maxCollectorsPerExporter ):
               collectorModel.inactiveReason = \
                     collectorInactiveReason.maxCollectorsLimit
            else:
               collectorModel.inactiveReason = \
                     collectorInactiveReason.collectorNotProcessed
            if expConfig:
               collector = expConfig.collectorHostAndPort.get( collectorName )
         if collector:
            collectorModel.port = collector.port

def showGroup( ftrType, trackerConfig, trackerModel, fgName, tracker, hwFtStatus ):
   assert ftrType != ftrTypeCpuQueue
   fgModel = FlowTrackingModel.FgModel()
   trackerModel.groups[ fgName ] = fgModel
   fgConfig = None
   if trackerConfig:
      fgConfig = trackerConfig.fgConfig.get( fgName )

   def getMirrorModel( mirrorConfig ):
      mirrorModel = FlowTrackingModel.MirrorModel()
      mirrorModel.sessionName = mirrorConfig.sessionName
      mirrorModel.initialCopy = mirrorConfig.initialCopy
      mirrorModel.mirrorSelectorAlgorithm = mirrorConfig.mirrorSelectorAlgo
      if mirrorConfig.mirrorSelectorAlgo == mirrorSelectorAlgo.fixed:
         mirrorModel.mirrorInterval = mirrorConfig.mirrorIntervalFixed
      elif mirrorConfig.mirrorSelectorAlgo == mirrorSelectorAlgo.random:
         mirrorModel.mirrorInterval = mirrorConfig.mirrorIntervalRandom
      else:
         mirrorModel.mirrorInterval = 0
      return mirrorModel

   if fgConfig is None:
      resFgSeqNo = reservedGroupToSeqnoMap.get( fgName )
      if resFgSeqNo:
         # For default reserved groups: use the info from hwFgConfig
         hwFgConfig = tracker.hwFgConfig.get( resFgSeqNo ) if tracker else None
         if hwFgConfig is None:
            # Shouldn't happen unless it is some transient corner case
            return

         fgModel.seqno = hwFgConfig.seqno
         # expNames
         for expName in hwFgConfig.expName:
            expModel = FlowTrackingModel.GroupExpModel()
            expModel.expName = expName
            fgModel.expNames[ expName ] = expModel

         # mirroring
         if gv.ftrCapabilities[ ftrType ].mirroringSupported:
            mirrorConfig = hwFgConfig.mirrorConfig
            mirrorModel = getMirrorModel( mirrorConfig )
            fgModel.mirroring = mirrorModel

         # hwGroups
         hwFgName = hwFgConfig.fgName
         hwFgModel = FlowTrackingModel.HwFgModel()
         hwFgModel.hwFgName = hwFgName
         hwFgModel.active = hwFgName in hwFtStatus.hwFgStatus
         fgModel.hwGroups[ int( hwFgConfig.seqno ) ] = hwFgModel
         return
      else:
         # If fgConfig got deleted while this command was being run but hwFgConfig
         # is not cleared yet, we cannot display this group in correct sequence
         # because we would have lost group seqno.
         # Even if we compute the seqno from hwFgConfig, there is no guarantee that
         # it would be in the correct sequence. We may get stale info in cases of
         # sequence number change due to resequencing or config change.
         # Futher, this would anyway be a transient case and things should be fine
         # from eventual consistency point of view.
         return

   # In regular case when a group exists in config, the attr value for each of the
   # corresponding hwGroups will be same. So we take it from config entity.
   # The status of things like exporter, mirroring etc will be taken later from
   # hwFgStatus (To Do Later)

   # seqno
   fgModel.seqno = fgConfig.seqno

   # Encapsulation and access lists are supported only for non-reserved groups
   if fgName not in reservedGroupToSeqnoMap:
      # encapTypes
      for encapType in fgConfig.encapType:
         encapTypeModel = FlowTrackingModel.EncapTypeModel()
         encapTypeModel.encapType = encapType
         fgModel.encapTypes.append( encapTypeModel )

      # ipAccessLists
      ipAccessList = fgConfig.ipAccessList
      if ipAccessList.aclName:
         ipAccessListModel = FlowTrackingModel.AccessListModel()
         ipAccessListModel.aclName = ipAccessList.aclName
         fgModel.ipAccessLists[ ipAccessList.seqno ] = ipAccessListModel

      # ip6AccessLists
      ip6AccessList = fgConfig.ip6AccessList
      if ip6AccessList.aclName:
         ip6AccessListModel = FlowTrackingModel.AccessListModel()
         ip6AccessListModel.aclName = ip6AccessList.aclName
         fgModel.ip6AccessLists[ ip6AccessList.seqno ] = ip6AccessListModel

   # expNames
   for expName in fgConfig.expName:
      expModel = FlowTrackingModel.GroupExpModel()
      expModel.expName = expName
      fgModel.expNames[ expName ] = expModel

   # mirroring
   if gv.ftrCapabilities[ ftrType ].mirroringSupported:
      mirrorConfig = fgConfig.mirrorConfig
      mirrorModel = getMirrorModel( mirrorConfig )
      fgModel.mirroring = mirrorModel

   # hwGroups
   hwFgConfigMap = tracker.hwFgConfigMap.get( fgName ) if tracker else None
   if hwFgConfigMap is not None:
      for hwFgSeqno, hwFgMap in hwFgConfigMap.hwFgNameAndSeqno.items():
         hwFgName = hwFgMap.fgName
         hwFgModel = FlowTrackingModel.HwFgModel()
         hwFgModel.hwFgName = hwFgName
         hwFgModel.active = hwFgName in hwFtStatus.hwFgStatus
         fgModel.hwGroups[ int( hwFgSeqno ) ] = hwFgModel

def getIntfsForTracker( ftrType, trName, tracker, egress=False ):
   """Return the active and inactive interfaces for a tracker.

   The return value is a 2-tuple containing the list of active interfaces, and the
   list of inactive interfaces for the tracker. The active interface lists consists
   of pairs of interface and sample mode, and the inactive interface list consists of
   pairs of interface and inactive reason.
   """
   if ftrType == ftrTypeCpuQueue:
      return None, None

   activeIntfs = []
   inactiveIntfs = []

   def getHwIntfFtConfig( ftrType, egress=False ):
      if egress:
         return gv.hwConfig[ ftrType ].hwEgressIntfFtConfig
      else:
         return gv.hwConfig[ ftrType ].hwIntfFtConfig

   def getFtIntfConfig( ftrType, egress=False ):
      if egress:
         return gv.trackingConfig[ ftrType ].flowTrackerEgressIntfConfig
      else:
         return gv.trackingConfig[ ftrType ].flowTrackerIntfConfig

   def getHwIntfFtStatus( ftrType, *, egress ):
      if egress:
         return gv.hwTrackingStatus[ ftrType ].egressHwIntfFtStatus
      else:
         return gv.hwTrackingStatus[ ftrType ].hwIntfFtStatus

   hifc = getHwIntfFtConfig( ftrType, egress )
   if tracker:
      hwIntfFtStatus = getHwIntfFtStatus( ftrType, egress=egress )
      # Look in hardware/flowtracking/config/...
      for intfName, hwIntfConfig in hifc.items():
         if hwIntfConfig.hwFtConfig == tracker:
            if SubIntfId.isSubIntfId( intfName ):
               # If the parent interface has the same flow tracker, and the same
               # sample mode configured do not show the sub-interface in
               # active / inactive interfaces.
               parentIntf = SubIntfId.parentIntfId( intfName )
               parentFtConfig = hifc.get( parentIntf )
               if ( parentFtConfig and parentFtConfig.hwFtConfig == tracker and
                    parentFtConfig.sampleMode == hwIntfConfig.sampleMode ):
                  continue
            activeIntf = True
            # Get inactive reason from HwFtStatus.
            intfFtStatus = hwIntfFtStatus.get( intfName )
            if intfFtStatus and not intfFtStatus.active:
               activeIntf = False
               inactiveReason = intfFtStatus.reason

            if activeIntf:
               activeIntfs.append( ( IntfId( intfName ), hwIntfConfig.sampleMode ) )
            else:
               inactiveIntfs.append( ( IntfId( intfName ), inactiveReason ) )

   fic = getFtIntfConfig( ftrType, egress )
   # Look in flowtracking/config/...
   for intfName, intfConfig in fic.items():
      if intfConfig.trackerName == trName:
         hwIntfFtConfig = hifc.get( intfName )
         if hwIntfFtConfig is None or hwIntfFtConfig.hwFtConfig is None:
            inactiveIntfs.append( ( IntfId( intfName ), None ) )
   return activeIntfs, inactiveIntfs

def showIntfsForTracker( trackerModel, ftrType, trName, tracker ):
   # Map the sampleMode values to the strings that should be used in the CAPI model.
   if ftrType == ftrTypeCpuQueue:
      trackerModel.activeIntfs = []
      trackerModel.inactiveIntfs = []
      trackerModel.activeEgressIntfs = []
      trackerModel.inactiveEgressIntfs = []
      return

   sampleModeMap = {
      sampleModeAll : "all",
      sampleModeFiltered : "filtered",
   }

   trackerModel.activeIntfs = []
   trackerModel.inactiveIntfs = []
   activeIntfs, inactiveIntfs = getIntfsForTracker( ftrType, trName, tracker )
   activeEgressIntfs, inactiveEgressIntfs = getIntfsForTracker( ftrType, trName,
                                                            tracker, egress=True )
   for intf, sampleMode in activeIntfs:
      intfModel = FlowTrackingModel.IntfModel(
         interface=intf, sampleMode=sampleModeMap[ sampleMode ] )
      trackerModel.activeIntfs.append( intfModel )
   for intf, inactiveReason in inactiveIntfs:
      intfModel = FlowTrackingModel.IntfModel( interface=intf,
                                               reason=inactiveReason )
      trackerModel.inactiveIntfs.append( intfModel )
   for intf, sampleMode in activeEgressIntfs:
      intfModel = FlowTrackingModel.IntfModel(
         interface=intf, sampleMode=sampleModeMap[ sampleMode ] )
      trackerModel.activeEgressIntfs.append( intfModel )
   for intf, inactiveReason in inactiveEgressIntfs:
      intfModel = FlowTrackingModel.IntfModel( interface=intf,
                                               reason=inactiveReason )
      trackerModel.inactiveEgressIntfs.append( intfModel )

def _calcSampleSize(
      ftrType,
      flowTrackingConfig,
      hwConfig ) -> typing.Optional[ int ]:
   """
      Args:
         ftrType: Type of tracker
         flowTrackingConfig: instance of FlowTracking::Config or None
         hwConfig: instance of HwFlowTracking::Config or None
      Returns:
        Number of bytes used in a sample, or None
   """
   if ftrType != ftrTypeCpuQueue:
      return None
   if hwConfig is not None:
      return hwConfig.sampleSize
   if flowTrackingConfig is not None:
      return flowTrackingConfig.sampleSize
   return None

def _calcSampleLimit(
      ftrType,
      flowTrackingConfig,
      *,
      defaultSampleLimit: typing.Optional[ int ] ) -> typing.Optional[ int ]:
   if ftrType not in [ ftrTypeCpuQueue, ftrTypeMirrorOnDrop ]:
      return None

   if flowTrackingConfig.sampleLimit is not None:
      return flowTrackingConfig.sampleLimit

   return defaultSampleLimit

def _calcExportTriggers(
      ftrType,
      trackerActive: bool,
      trackerConfig,
      hwTracker ) -> typing.Optional[ typing.List[ str ] ]:
   """
      Args:
         ftrType: Type of tracker
         trackerActive: Is the tracker active?
         trackerConfig: How the tracker is configurued.  May be None.
            Instance of FlowTracking::FlowTrackerConfig
         hwTracker: How the tracker is actually being used.  May be None.
            Instance of HwFlowTracking::FtConfig

      Returns:
         "congestion" in a list, 
         or empty list to indicate disabled,
         or None
   """
   if ftrType != ftrTypeCpuQueue:
      return None
   if not trackerActive:
      return None
   if trackerConfig and trackerConfig.congestionConfig:
      return [ "congestion" ]
   if hwTracker:
      if hwTracker.congestionConfig:
         return [ "congestion" ]
   return []

def _calcDampingTimeout(
      ftrType,
      trackerActive: bool,
      trackerConfig,
      hwTracker ) -> typing.Optional[ int ]:
   """
      Args:
         ftrType: Type of tracker
         trackerActive: Is the tracker active?
         trackerConfig: How the tracker is configurued.  May be None.
            Instance of FlowTracking::FlowTrackerConfig
         hwTracker: How the tracker is actually being used.  May be None.
            Instance of HwFlowTracking::FtConfig

      Returns:
         Number of seconds to wait between ending a record, and
         creating the next record.
   """
   if ftrType != ftrTypeCpuQueue:
      return None
   if not trackerActive:
      return None
   if hwTracker and hwTracker.congestionConfig:
      return hwTracker.congestionConfig.dampingTimeout
   if trackerConfig and trackerConfig.congestionConfig:
      return trackerConfig.congestionConfig.dampingTimeout
   return None

def _calcCaptureInterval(
      ftrType,
      trackerActive: bool,
      trackerConfig,
      hwTracker ) -> typing.Optional[ CongestionIntervalModel ]:
   """
      Args:
         ftrType: Type of tracker
         trackerActive: Is the tracker active?
         trackerConfig: How the tracker is configurued.  May be None.
            Instance of FlowTracking::FlowTrackerConfig
         hwTracker: How the tracker is actually being used.  May be None.
            Instance of HwFlowTracking::FtConfig

      Returns:
         Number of seconds to wait between ending a record, and
         creating the next record. Instance of FlowTracking::CongestionInterval.
   """
   if ftrType != ftrTypeCpuQueue:
      return None
   if not trackerActive:
      return None
   if hwTracker and hwTracker.congestionConfig:
      assert hwTracker.congestionConfig.interval
      result = CongestionIntervalModel()
      result.before = hwTracker.congestionConfig.interval.before
      result.after = hwTracker.congestionConfig.interval.after
      return result
   if ( trackerConfig and trackerConfig.congestionConfig and
      trackerConfig.congestionConfig.interval ):
      result = CongestionIntervalModel()
      result.before = trackerConfig.congestionConfig.interval.before
      result.after = trackerConfig.congestionConfig.interval.after
      return result
   return None

def _calcMonitoredCpuQueues(
      ftrType,
      trackerActive: bool,
      trackerConfig,
      hwTracker ) -> typing.Optional[ typing.List[ str ] ]:
   """
      Args:
         ftrType: Type of tracker
         trackerActive: Is the tracker active?
         trackerConfig: How the tracker is configurued.  May be None.
            Instance of FlowTracking::FlowTrackerConfig
         hwTracker: How the tracker is actually being used.  May be None.
            Instance of HwFlowTracking::FtConfig

      Returns:
        List of CPU queues that are being monitored, or None if the
        list is not applicable.
   """
   if ftrType != ftrTypeCpuQueue:
      return None
   if not trackerActive:
      return None
   useTrackerConfig = (
         trackerConfig and ( trackerConfig.monitoredCpuQueue is not None ) )
   useHwTracker = hwTracker and ( hwTracker.monitoredCpuQueue is not None )
   if not ( useTrackerConfig or useHwTracker ):
      return None
   names = set()
   if useTrackerConfig:
      names.update( trackerConfig.monitoredCpuQueue )
   if useHwTracker:
      names.update( hwTracker.monitoredCpuQueue )
   sortedNames = list( sorted( names ) )
   return sortedNames

def _showFlowTracking( mode, args ):
   ftrType = getFtrTypeFromArgs( args )
   trackerName = args.get( 'TRACKER_NAME' )
   exporterName = args.get( 'EXPORTER_NAME' )

   flowTrackingModel = FlowTrackingModel.FtrModel()
   flowTrackingModel.ftrType = ftrType
   # a hack to make sure the optional Dict attribute in the model to be a really
   # optional field and not show up in the json output when it is not set
   flowTrackingModel.ftrTrafficGroups = None
   flowTrackingModel.filterTypes = None

   if not flowTrackingAgentRunning( ftrType, gv.entityManager, gv.activeAgentDir ):
      flowTrackingModel.running = False
      return flowTrackingModel

   flowTrackingModel.running = True
   flowTrackingModel.enabledBy = "flow tracking %s" % ftrTypeKwStr[ ftrType ]
   flowTrackingConfig = gv.trackingConfig[ ftrType ]

   if toggleSftBgpExportCliEnabled():
      if ftrType == ftrTypeSampled:
         flowTrackingModel.bgpElementsExported = flowTrackingConfig.bgpExport

   flowTrackingModel.sampleSize = _calcSampleSize(
         ftrType, flowTrackingConfig, gv.hwConfig[ ftrType ] )
   flowTrackingModel.sampleLimit = _calcSampleLimit(
         ftrType, flowTrackingConfig,
         defaultSampleLimit=gv.ftrCapabilities[ ftrType ].defaultSampleLimit )

   if ftrType == ftrTypeMirrorOnDrop:
      modDefaultSampleRate = gv.ftrCapabilities[ ftrType ].defaultSampleLimit
      sampleLimit = 0
      if flowTrackingConfig.sampleLimit is None:
         sampleLimit = modDefaultSampleRate
      else:
         sampleLimit = flowTrackingConfig.sampleLimit
      flowTrackingModel.sampleLimit = sampleLimit
      trapSuppressionModel = FlowTrackingModel.TrapSuppressionModel()
      trapSuppressionModel.active = True
      modDefaultTrapSuppression = \
            gv.ftrCapabilities[ ftrType ].modDefaultTrapSuppression
      trapSuppressionSupported = gv.ftrCapabilities[ ftrType ]\
         .trapSuppressionSupported
      trapSuppression = None
      if not trapSuppressionSupported:
         trapSuppressionModel.active = False
         trapSuppressionModel.reason = 'not supported'
      elif trapSuppressionModel.active:
         for hook in isEMAvailable.extensions():
            if hook() is False:
               trapSuppressionModel.active = False
               trapSuppressionModel.reason = 'incomplete configuration'

      if flowTrackingConfig.trapSuppression is None:
         trapSuppression = modDefaultTrapSuppression
      else:
         trapSuppression = flowTrackingConfig.trapSuppression
      trapSuppressionModel.trapSuppression = trapSuppression

      if trapSuppressionModel.active:
         if trapSuppression is False:
            trapSuppressionModel.active = False
            trapSuppressionModel.reason = 'not configured'

      flowTrackingModel.trapSuppressionDetails = trapSuppressionModel

      for encapType in flowTrackingConfig.encap:
         encapTypeModel = FlowTrackingModel.EncapTypeModel()
         encapTypeModel.encapType = encapType
         flowTrackingModel.encapTypes.append( encapTypeModel )

      if flowTrackingConfig.filter:
         flowTrackingModel.filterTypes = []
         for filterType in flowTrackingConfig.filter:
            filterTypeModel = FlowTrackingModel.FilterTypeModel()
            filterTypeModel.filterType = filterType
            flowTrackingModel.filterTypes.append( filterTypeModel )

      if ( ftrType == ftrTypeMirrorOnDrop and
           packetBufferModFtrSupported( ftrType ) ):
         dropExportHwStatus = gv.dropExportHwStatus
         # The product supports packet buffer MOD so we will display some output.
         pbModel = FlowTrackingModel.PacketBufferModel()
         pbModel.status = packetBufferStatusEnum.disabled

         if dropExportHwStatus.pbStatus == PacketBufferStatus.enabled:
            pbModel.status = packetBufferStatusEnum.enabled

            if packetBufferSamplingProbSupported( ftrType ):
               if flowTrackingConfig.packetBufferSampleProb:
                  pbModel.prob25 = flowTrackingConfig.packetBufferSampleProb
                  pbModel.prob50 = flowTrackingConfig.packetBufferSampleProb
                  pbModel.prob75 = flowTrackingConfig.packetBufferSampleProb
                  pbModel.prob100 = flowTrackingConfig.packetBufferSampleProb

               if flowTrackingConfig.packetBufferSampleProb25:
                  pbModel.prob25 = flowTrackingConfig.packetBufferSampleProb25

               if flowTrackingConfig.packetBufferSampleProb50:
                  pbModel.prob50 = flowTrackingConfig.packetBufferSampleProb50

               if flowTrackingConfig.packetBufferSampleProb75:
                  pbModel.prob75 = flowTrackingConfig.packetBufferSampleProb75

               if flowTrackingConfig.packetBufferSampleProb100:
                  pbModel.prob100 = flowTrackingConfig.packetBufferSampleProb100
            elif packetBufferShaperSupported( ftrType ):
               pbModel.shaperRate = dropExportHwStatus.pbShaperRate
               pbModel.shaperBurst = dropExportHwStatus.pbShaperBurst
         else:
            # Handle error cases.
            if ( gv.dropExportHwStatus.pbStatus ==
                   PacketBufferStatus.insufficientBufferSpace ):
               pbModel.status = packetBufferStatusEnum.insufficientBufferSpace

         flowTrackingModel.packetBuffer = pbModel

   if ftrType == ftrTypeHardware:
      if gv.ftrCapabilities[ ftrType ].swExportSupported:
         flowTrackingModel.standardFormatForCounters = \
             gv.hwConfig[ ftrType ].swExport
         flowTrackingModel.standardFormatForTimeStamps = \
             gv.hwConfig[ ftrType ].swExport
      if not flowTrackingConfig.enabled and gv.flowWatcherTrackingConfig.enabled:
         # If hardware flow tracking is not enabled from CLI but enabled from
         # flowWatcher, use configuration from
         # flowtracking/input/config/hardware/flowwatcher
         flowTrackingConfig = gv.flowWatcherTrackingConfig
         flowTrackingModel.enabledBy = "monitor security awake"
   if ftrType == ftrTypeSampled:
      flowTrackingModel.ftrTrafficGroups = {}
      for addrFamily in [ IP_GROUP, IP6_GROUP ]:
         trafficTypeModel = FlowTrackingModel.FtrTrafficTypeModel()
         trafficTypeModel.sampleRate = gv.hwTrackingStatus[ ftrType ].sampleRate
         flowTrackingModel.ftrTrafficGroups[ addrFamily ] = trafficTypeModel
      # add encapsulation for show flow tracking sampled command.
      for encapType in flowTrackingConfig.encap:
         encapTypeModel = FlowTrackingModel.EncapTypeModel()
         encapTypeModel.encapType = encapType
         flowTrackingModel.encapTypes.append( encapTypeModel )

      def _handleFtrCapabilities( configHwOffload, hoStatusEnabled, disabledReason ):
         status = FlowTrackingModel.FtrHoStatusModel()
         status.status = "unconfigured"
         status.inactiveReasons = None
         if configHwOffload:
            if hoStatusEnabled:
               status.status = "active"
            else:
               status.status = "inactive"
               # add all the reasons, hardware offload is disabled currently
               status.inactiveReasons = []
               for reason in disabledReason:
                  hoReason = FlowTrackingModel.HoInactiveReasonModel()
                  hoReason.inactiveReason = reason
                  status.inactiveReasons.append( hoReason )
         return status

      # NOTE: HO config is direction agnostic
      if gv.ftrCapabilities[ ftrType ].hwOffloadIpv4:
         status = _handleFtrCapabilities( gv.hwConfig[ ftrType ].hwOffloadIpv4,
                                          gv.sampledHoStatus.v4enabled,
                                          gv.sampledHoStatus.v4disabledReason )
         flowTrackingModel.ftrTrafficGroups[ IP_GROUP ].hwOffloadStatus = status
      if gv.ftrCapabilities[ ftrType ].hwOffloadIpv6:
         status = _handleFtrCapabilities( gv.hwConfig[ ftrType ].hwOffloadIpv6,
                                          gv.sampledHoStatus.v6enabled,
                                          gv.sampledHoStatus.v6disabledReason )
         flowTrackingModel.ftrTrafficGroups[ IP6_GROUP ].hwOffloadStatus = status
      if gv.ftrCapabilities[ ftrType ].egressHwOffloadIpv4:
         status = _handleFtrCapabilities( gv.hwConfig[ ftrType ].hwOffloadIpv4,
                                          gv.sampledHoStatus.egressV4Enabled,
                                          gv.sampledHoStatus.egressV4DisabledReason )
         flowTrackingModel.ftrTrafficGroups[
               IP_GROUP ].egressHwOffloadStatus = status
      if gv.ftrCapabilities[ ftrType ].egressHwOffloadIpv6:
         status = _handleFtrCapabilities( gv.hwConfig[ ftrType ].hwOffloadIpv6,
                                          gv.sampledHoStatus.egressV6Enabled,
                                          gv.sampledHoStatus.egressV6DisabledReason )
         flowTrackingModel.ftrTrafficGroups[
               IP6_GROUP ].egressHwOffloadStatus = status

   flowTrackingModel.trackers = {}
   # Take union of config + hwConfig and iterate over them.
   # Reason being, if for any reason agent is not running or transient
   # (agent didn't react yet), there can be discrepancy due to that.
   # For example:
   #    - tracker deleted in config but exists in hwConfig,
   #    - tracker exists in config but not in hwConfig
   # we should display the current functional state.
   trackers = list( set( flowTrackingConfig.flowTrackerConfig.keys() ).union(
                    set( gv.hwConfig[ ftrType ].hwFtConfig.keys() ) ) )
   for trName in trackers:
      if trackerName and trName != trackerName:
         continue

      trackerModel = FlowTrackingModel.FtModel()
      if ftrType == ftrTypeCpuQueue:
         trackerModel.groups = {}
         trackerModel.activeIntfs = []
         trackerModel.inactiveIntfs = []
         trackerModel.activeEgressIntfs = []
         trackerModel.inactiveEgressIntfs = []
      else:
         trackerModel.monitoredCpuQueues = None
      ftStatus = gv.ftrStatus[ ftrType ].ftStatus.get( trName )
      # Need to set default values for testing purposes
      trackerModel.activeInterval = activeInterval.intervalDefault
      trackerModel.inactiveTimeout = inactiveTimeout.timeoutDefault
      flowTrackingModel.trackers[ trName ] = trackerModel
      hwTracker = gv.hwConfig[ ftrType ].hwFtConfig.get( trName )
      if hwTracker:
         trackerModel.active = ftStatus.active if ftStatus else False
      else:
         trackerModel.inactiveReason = (
            trackerInactiveReason.trackerInactiveReasonUnknown )
      # Even if tracker is present in hwFtConfig and ftStatus, it is possible
      # to have ftStatus.active as False because of race between CLI and
      # tacc agent code. So irrespective of whether tracker is present in hwFtConfig
      # or not, we need to get inactive reason when ftStatus.active flag is False.
      if ftStatus and not ftStatus.active:
         trackerModel.inactiveReason = ftStatus.inactiveReason
         trackerModel.active = ftStatus.active
         if trName not in gv.hwConfig[ ftrType ].hwFtConfig:
            trackerModel.monitoredCpuQueues = None
            trackerModel.exportTriggers = None
            continue
      elif not ftStatus:
         trackerModel.inactiveReason = \
               trackerInactiveReason.trackerNotProcessed
         trackerModel.active = False
         trackerModel.monitoredCpuQueues = None
         trackerModel.exportTriggers = None
         continue

      hwStatus = gv.hwTrackingStatus[ ftrType ].hwFtStatus.get( trName )
      if hwStatus is None:
         trackerModel.inactiveReason = \
               trackerInactiveReason.trackerNotProcessed
         trackerModel.active = False
         trackerModel.monitoredCpuQueues = None
         trackerModel.exportTriggers = None
         continue
      trackerModel.inactiveTimeout = hwStatus.inactiveTimeout

      trackerConfig = flowTrackingConfig.flowTrackerConfig.get( trName )
      trackerModel.congestionInterval = _calcCaptureInterval(
         ftrType, trackerModel.active, trackerConfig, hwTracker )
      trackerModel.exportTriggers = _calcExportTriggers(
              ftrType, trackerModel.active, trackerConfig, hwTracker )
      trackerModel.dampingTimeout = _calcDampingTimeout(
              ftrType, trackerModel.active, trackerConfig, hwTracker )
      trackerModel.monitoredCpuQueues = _calcMonitoredCpuQueues(
              ftrType, trackerModel.active, trackerConfig, hwTracker )

      trackerModel.activeInterval = hwStatus.activeInterval
      trackerModel.inactiveTimeout = hwStatus.inactiveTimeout

      configExporter = set( trackerConfig.expConfig ) if trackerConfig else set()
      hwExporter = set( hwTracker.expConfig ) if hwTracker else set()
      exporters = hwExporter.union( configExporter )

      for expName in exporters:
         if exporterName and expName != exporterName:
            continue
         showExporter( ftrType, trName, trackerConfig, trackerModel, expName,
                       hwTracker, ftStatus )
         if ftrType == ftrTypeMirrorOnDrop and trackerConfig:
            expConfig = trackerConfig.expConfig.get( expName )
            if expConfig:
               exportFormat = expConfig.exportFormat
               if exportFormat == ExportFormatEnum.formatSflow \
                  and trapSuppressionModel.active:
                  trapSuppressionModel.active = False
                  trapSuppressionModel.reason = 'sFlow configured'
                  flowTrackingModel.trapSuppressionDetails = trapSuppressionModel

      if gv.ftrCapabilities[ ftrType ].flowGroupConfigSupported:
         configFlowGroups = set( trackerConfig.fgConfig ) if trackerConfig else set()
         hwFgConfigMap = set( hwTracker.hwFgConfigMap ) if hwTracker else set()
         groups = hwFgConfigMap.union( configFlowGroups )
         groupName = args.get( 'GROUP_NAME' )
         for fgName in groups:
            if groupName and groupName != fgName:
               continue
            showGroup( ftrType, trackerConfig, trackerModel, fgName,
                       hwTracker, hwStatus )

         if hwTracker:
            # Reserved groups
            # Each flow tracking feature can have different reserved groups with
            # seqno >= constants.reservedFgSeqnoBase
            reservedFgSeqnoBase = constants.reservedFgSeqnoBase
            for hwFgseqno in sorted( hwTracker.hwFgConfig, reverse=True ):
               if hwFgseqno < reservedFgSeqnoBase:
                  break
               hwFgConfig = hwTracker.hwFgConfig.get( hwFgseqno )
               if hwFgConfig is None:
                  continue
               fgName = hwFgConfig.fgName
               if groupName and groupName != fgName or \
                  fgName in configFlowGroups:
                  continue
               showGroup( ftrType, trackerConfig, trackerModel, fgName,
                          hwTracker, hwStatus )
      elif hwTracker:
         if ftrType != ftrTypeCpuQueue:
            trackerModel.groups = {}
            for group in hwTracker.hwFgConfig.values():
               fgModel = FlowTrackingModel.FgModel()
               trackerModel.groups[ group.fgName ] = fgModel

      showIntfsForTracker( trackerModel, ftrType, trName, hwTracker )

   return flowTrackingModel

class ShowFlowTracking( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow tracking ( cpu-queue | sampled | hardware | mirror-on-drop '
   syntax += '| ( telemetry inband ) | ( firewall distributed )'
   syntax += ') [ tracker TRACKER_NAME [ exporter EXPORTER_NAME ]'
   syntax += ' [ group GROUP_NAME ]'
   syntax += ' ]'

   data = {
         'flow' : flowMatcherForShow,
         'tracking' : trackingShowKw,
         'cpu-queue' : cpuQueueShowKw,
         'sampled' : sampledShowKw,
         'hardware' : hardwareShowKw,
         'mirror-on-drop' : mirrorOnDropShowKw,
         'tracker' : trackerKw,
         'TRACKER_NAME' : trackerNameMatcher,
         'exporter' : exporterKw,
         'EXPORTER_NAME' : exporterNameMatcher,
   }

   data[ 'telemetry' ] = telemetryShowKw
   data[ 'inband' ] = inbandShowKw

   data[ 'firewall' ] = firewallShowKw
   data[ 'distributed' ] = distributedShowKw

   data[ 'group' ] = groupShowKw
   data[ 'GROUP_NAME' ] = groupNameMatcher

   cliModel = FlowTrackingModel.FtrModel
   handler = _showFlowTracking

BasicCli.addShowCommandClass( ShowFlowTracking )

def showActiveTracker( ftrType, trName, trConfig, output=None, space='',
                      detail=False ):
   # we cannot use FlowTrackerCli.showActiveTracker here because this will cause
   # inclusion issues when including ConfigCli into ShowCli library
   if trConfig is None:
      return
   if output is None:
      output = sys.stdout
   ftrCaps = getFtrCaps( ftrType )
   ftrSwCaps = getFtrSwCaps( ftrType )
   showFtr( output, trName, trConfig, ftrType, getMplsHwCaps(),
            ftrTypeCaps=ftrCaps, cliSave=False, space=space, detail=detail )
   showCongestion( output, trConfig, ftrSwCaps, cliSave=False,
                   detail=detail, space=space )
   for expName, expConfig in sorted( trConfig.expConfig.items() ):
      _showExporter( output, expName, expConfig, ftrTypeKwStr[ ftrType ],
                    ftrCaps, ftrSwCaps, cliSave=False, space=space,
                    saveAllDetail=detail )
   showActiveGroups( output, trConfig, space=space )

def showActive( config, ftrType, detail, output=None ):
   # we cannot use FlowTrackingCli._showActive here because this will cause
   # inclusion issues when including ConfigCli into ShowCli library
   ftrCaps = getFtrCaps( ftrType )
   if config is None:
      return
   if output is None:
      output = sys.stdout
   lines = showFlowTracking( config, ftrType,
                             ftrTypeCaps=ftrCaps, cliSave=False,
                             mplsHwCaps=getMplsHwCaps(),
                             bridgingHwCaps=getBridgingHwCaps(),
                             saveAllDetail=detail )
   for line in lines:
      output.write( line + '\n' )

   printHeader = not lines

   for trName, trConfig in sorted( config.flowTrackerConfig.items() ):
      if printHeader:
         output.write( 'flow tracking %s\n' % ftrTypeKwStr[ ftrType ] )
         printHeader = False
      showActiveTracker( ftrType, trName, trConfig, output,
                         space=spaceConst, detail=detail )

   lines = showFlowTrackingTrailer( config, ftrType, cliSave=False,
                                    saveAllDetail=detail )
   # Only Trailer is non-empty, so display flow tracking before trailer lines
   if printHeader and lines:
      output.write( 'flow tracking %s\n' % ftrTypeKwStr[ ftrType ] )
   for line in lines:
      output.write( line + '\n' )

def _showActiveConfig( mode, args ):
   ftrType = getFtrTypeFromArgs( args )
   flowTrackingConfig = gv.trackingConfig[ ftrType ]
   output = sys.stdout
   if not flowTrackingConfig.enabled and gv.flowWatcherTrackingConfig.enabled:
      # If hardware flow tracking is not enabled from CLI but enabled from
      # flowWatcher, use configuration from
      # flowtracking/input/config/hardware/flowwatcher
      flowTrackingConfig = gv.flowWatcherTrackingConfig
      output.write( "Configuration source: monitor security awake\n" )
   else:
      output.write( "Configuration source: CLI\n" )
   showActive( flowTrackingConfig, ftrType, 'all' in args, output=output )

class ShowFlowTrackingConfig( ShowCommand.ShowCliCommandClass ):
   syntax = '''show flow tracking ( hardware | ( firewall distributed ) )
               configuration active [ all ]'''

   data = {
         'flow' : flowMatcherForShow,
         'tracking' : trackingShowKw,
         'hardware' : hardwareShowKw,
         'firewall' : firewallShowKw,
         'distributed' : distributedShowKw,
         'configuration' : 'Flow tracking configuration',
         'active' : 'Active configuration',
         'all' : 'Active and default configuration',
   }

   handler = _showActiveConfig

BasicCli.addShowCommandClass( ShowFlowTrackingConfig )

#--------------------------
def Plugin( em ):
   gv.entityManager = em
   gv.activeAgentDir = LazyMount.mount( em, 'flowtracking/activeAgent',
                                        'Tac::Dir', 'ri' )
   gv.sampledHoStatus = LazyMount.mount( em, 'flowtracking/hoStatus/sampled',
                                         'FlowTracking::HoStatus', 'r' )
   for ftrType in ftrTypes:
      ftrEntityType = ( ftrMirrorOnDropConfigType if ftrType == ftrTypeMirrorOnDrop
                        else ftrConfigType )
      gv.trackingConfig[ ftrType ] = LazyMount.mount( em,
                                    getFlowTrackingCliConfigPath( ftrType ),
                                    ftrEntityType, 'r' )
      gv.hwConfig[ ftrType ] = LazyMount.mount( em,
                                    hwFtrConfigPathPrefix + ftrType,
                                    hwFtrConfigType, 'r' )
      gv.hwTrackingStatus[ ftrType ] = LazyMount.mount( em,
                                       hwFtrStatusPathPrefix + ftrType,
                                       hwFtrStatusType, 'r' )
      gv.ftrCapabilities[ ftrType ] = LazyMount.mount( em,
                                       ftrCapabilitiesPathPrefix + ftrType,
                                       ftrCapabilitiesType, 'r' )
      gv.ftrSwCapabilities[ ftrType ] = SwCapabilities( ftrType )
      gv.ftrStatus[ ftrType ] = LazyMount.mount( em,
                                       ftrStatusPathPrefix + ftrType,
                                       ftrStatusType, 'r' )
   gv.flowWatcherTrackingConfig = (
      LazyMount.mount( em, getHardwareFlowTrackingFlowWatcherConfigPath(),
                       ftrConfigType, 'r' ) )
   gv.dropExportHwStatus = LazyMount.mount(
         em, 'dropExport/hwStatus', 'HwFlowTracking::MirrorOnDropHwStatus', 'r' )
