# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

import CliMatcher
import CliParser
import CliParserCommon
from CliPlugin.IpGenAddrMatcher import IpGenPrefixMatcher

class RSMatcher( CliMatcher.Matcher ):
   '''Matcher for a flowspec nlri string.'''
   # The opVal components are rendered as an optional series of "<tag>:<value>;"
   # substrings.  These will be split out and matched against this regex.
   # Each component consists of a tag from a finite set of tags, and the value
   # can be an expression consisting of values and operators.  We won't attempt to
   # validate the grammar of this expression in the cli.
   opValComponentRe = re.compile( r'(IP|P|DP|SP|IT|IC|TCP|LEN|DSCP|FRAG|FL):\S+$' )
   # In case of IPv4/v6 Flowspec, a valid rule string must start with a hex digit
   # or *, since the first part is the dest prefix. In case of IPv4/v6 Flowspec
   # VPN, a valid rule string must start with "RD", signifying a route
   # distinguisher. If the user has started entering something else, it is no
   # longer a possible rule string.
   ruleStringRe = re.compile( r'(RD|[A-Fa-f0-9\*]+).*' )

   def __init__( self, helpdesc, **kargs ):
      super().__init__( helpdesc=helpdesc,
                        **kargs )
      self.ipGenPrefixMatcher_ = IpGenPrefixMatcher( helpdesc )

   def match( self, mode, context, token ):
      # Split out the prefix components so we can validate them appropriately.
      # We'll strip any trailing delimiter first so we don't have to deal wth
      # an empty string at the end of the list.
      components = token.rstrip( ';' ).split( ';' )
      if components[ 0 ].startswith( 'RD' ):
         # There must be at least three components (RD, dest & src prefixes).
         minNumComps = 3
         destPrefixCompIndex = 1
      else:
         # There must be at least two components (dest & src prefixes).
         minNumComps = 2
         destPrefixCompIndex = 0
      if len( components ) < minNumComps:
         return CliParserCommon.noMatch
      # The dest & src prefix fields are either '*' or a valid IpGenPrefix
      for prefix in components[ destPrefixCompIndex : destPrefixCompIndex + 2 ]:
         if prefix == '*':
            continue
         if prefix == '':
            return CliParserCommon.noMatch
         if( self.ipGenPrefixMatcher_.match( mode, context, prefix ) is
               CliParserCommon.noMatch ):
            return CliParserCommon.noMatch
      # The remaining tokens should match the opval component regex
      for opValComponent in components[ destPrefixCompIndex + 2 : ]:
         m = self.opValComponentRe.match( opValComponent )
         if m is None:
            return CliParserCommon.noMatch
      # Success!
      return CliParserCommon.MatchResult( token, token )

   def completions( self, mode, context, token ):
      # A valid rule string must start with a hex digit or *, since the first
      # part is the dest prefix.  If the user has started entering something
      # else, it is no longer a possible rule string.
      if not token:
         return [ CliParser.Completion( 'STRING', 'BGP Flowspec rule string',
                                         False ) ]
      m = re.match( self.ruleStringRe, token )
      if m is None:
         return []
      else:
         return [ CliParser.Completion( 'STRING', 'BGP Flowspec rule string',
                                         False ) ]

   def __str__( self ):
      return 'BGP Flowspec rule string'

ruleStringMatcher = RSMatcher( 'BGP NLRI string' )
