# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
from CliMode.Generator import (
   GeneratorsModeBase,
)
from GeneratorCliUtils import (
   generatorTypeKwLinkQualification,
   generatorTypeKwRFC2544,
   featureEnabledEnumToShowKwMap,
)

# Assigning globals
satHwCapabilities = None
satProfilesConfig = None
rfc2544HwCapabilities = None
rfc2544ProfileConfigDir = None
featureConfig = None
featureStatus = None
hwFeatureStatus = None

def satLinkQualificationSupportedGuard( mode, token ):
   if satHwCapabilities and satHwCapabilities.hwCapable:
      return None
   return CliParser.guardNotThisPlatform

def guardRfc2544InitiatorSupported( mode, token ):
   if rfc2544HwCapabilities and rfc2544HwCapabilities.rfc2544InitiatorSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardLinkQualificationCommand( mode, token ):
   if mode.genType == generatorTypeKwLinkQualification:
      return None
   return CliParser.guardNotThisPlatform

def guardRfc2544Command( mode, token ):
   if mode.genType == generatorTypeKwRFC2544:
      return None
   return CliParser.guardNotThisPlatform

def getProfileNames( mode ):
   profileNames = None
   if mode.genType == generatorTypeKwLinkQualification:
      profileNames = satProfilesConfig.txConfig
   elif mode.genType == generatorTypeKwRFC2544:
      profileNames = rfc2544ProfileConfigDir.profileConfig
   return profileNames

def isFeatureEnabled( feature ):
   error = ''
   featureEnabled = False
   if featureConfig:
      featureEnabled = featureConfig.featureEnabled.get( feature )
   if not featureEnabled:
      error = featureEnabledEnumToShowKwMap[ feature ] + \
              ' generator config is not enabled'
   elif not featureStatus or not hwFeatureStatus or \
      featureStatus.featureEnabled != feature or \
      hwFeatureStatus.featureEnabled != feature:
      error = featureEnabledEnumToShowKwMap[ feature ] + \
              ' generator is not running'
   return error

rfc2544ConfigKwMatcher = CliCommand.guardedKeyword(
   'rfc2544',
   helpdesc='RFC2544 benchmark testing config',
   guard=guardRfc2544InitiatorSupported )
profileMatcher = CliMatcher.DynamicNameMatcher(
   getProfileNames,
   'Name of the profile',
   pattern='[a-zA-Z0-9_-]+',
   helpname='WORD' )
linkQualProfileMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: satProfilesConfig.txConfig,
   'Name of the profile',
   pattern='[a-zA-Z0-9_-]+',
   helpname='WORD' )
generatorsNode = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher(
      'generators',
      helpdesc='Generator config' ) )
interfaceNode = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher(
      'interface',
      helpdesc='Interface to be tested' ) )

class GeneratorsMode( GeneratorsModeBase, BasicCli.ConfigModeBase ):
   name = 'Monitor generators configuration'

   def __init__( self, parent, session, genType ):
      self.genType = genType
      GeneratorsModeBase.__init__( self, genType )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------
# Plugin Func
#-------------------------------------------------------------------
def Plugin( entityManager ):
   global satHwCapabilities
   global satProfilesConfig
   global rfc2544HwCapabilities
   global rfc2544ProfileConfigDir
   global featureConfig
   global featureStatus
   global hwFeatureStatus

   satHwCapabilities = LazyMount.mount(
      entityManager, 'sat/hardware/capabilities', 'Sat::SatHwCaps', 'r' )
   satProfilesConfig = ConfigMount.mount(
      entityManager, 'sat/profile/cliConfig',
      'Sat::SatProfilesConfig', 'w' )
   rfc2544HwCapabilities = LazyMount.mount(
      entityManager, 'rfc2544Initiator/hardware/capabilities',
      'Rfc2544Initiator::HwCapabilities', 'r' )
   rfc2544ProfileConfigDir = ConfigMount.mount(
      entityManager, 'rfc2544Initiator/profileConfigDir',
     'Rfc2544Initiator::ProfileConfigDir', 'w' )
   featureConfig = LazyMount.mount(
      entityManager, 'generator/featureConfig',
      'FlowGenerator::FeatureConfig', 'r' )
   featureStatus = LazyMount.mount(
      entityManager, 'generator/featureStatus',
      'FlowGenerator::FeatureStatus', 'r' )
   hwFeatureStatus = LazyMount.mount(
      entityManager, 'generator/hwFeatureStatus',
      'FlowGenerator::FeatureStatus', 'r' )
