# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Arnet
from CliModel import (
   Model,
   Float,
   Int,
   Dict,
   Enum,
   Bool,
)
import IntfModels
import Tac, Ark
from EoamTypes import enumToRateUnitMap
from TypeFuture import TacLazyType

testStatus = TacLazyType( 'Sat::TestStatus' )
rateUnit = TacLazyType( 'FlowGenerator::FlowRateUnit' )

class LinkQualificationProfileModel( Model ):
   pktSize = Int( help='Packet size in bytes' )
   duration = Int( help='Duration of the link qualification test in seconds' )
   rate = Int( help='Traffic rate value for the link qualification test' )
   rateUnit = Enum( values=rateUnit.attributes,
                    help='Traffic rate unit for the link qualification test' )

   def render( self ):
      print( 'Packet size:', self.pktSize, 'bytes' )
      print( 'Link qualification duration:', self.duration, 'seconds' )
      unit = enumToRateUnitMap.get( self.rateUnit, 'bps' )
      print( 'Traffic rate:', self.rate, unit, end='\n' )

class LinkQualificationAllProfilesModel( Model ):
   profiles = Dict( keyType=str, valueType=LinkQualificationProfileModel,
                 help='Summary of each link qualification profile config, \
                       keyed by profile name' )
   enabled = Bool( help='Link Qualification feature is enabled' )

   def render( self ):
      if not self.profiles:
         return
      profiles = sorted( self.profiles )
      lastProfile = profiles[ -1 ]
      for profile in profiles:
         print( 'Link Qualification Profile:', profile )
         self.profiles[ profile ].render()
         if profile != lastProfile:
            print()

class LinkQualificationGeneratorIntfModel( Model ):
   errorCode = Int( help='Error code to report different failures', optional=True )
   state = Enum( values=testStatus.attributes, help='Link qualification test state' )
   startTime = Float( help='Time (in seconds) when test was started, Unix time' )
   endTime = Float( help='Time (in seconds) when test ended, Unix time' )
   pktSize = Int( help='Packet size in bytes' )
   duration = Int( help='Duration of the test in seconds' )
   rate = Int( help='Traffic rate value for the test' )
   rateUnit = Enum( values=rateUnit.attributes,
                    help='Traffic rate unit for the test' )
   pktSent = Int( help='Number of packets sent' )
   pktRcvd = Int( help='Number of packets received' )
   pktDropped = Int( help='Number of packets lost' )
   txErrors = Int( help='Number of transmit errors' )
   rxErrors = Int( help='Number of receive errors' )
   txOctetCount = Int( help='Number of octets sent' )
   rxOctetCount = Int( help='Number of octets received' )

   def renderTime( self, _time, startTime=True ):
      if not startTime and self.state == testStatus.running:
         print( 'End time: -' )
         return

      assert _time < Tac.now(), \
            'test %s time is off' % ( 'start' if startTime else 'end' )

      if startTime:
         print( 'Start time: %s' % Ark.timestampToStr( _time, relative=True ) )
      else:
         print( 'End time: %s' % Ark.timestampToStr( _time, relative=True ) )

   def renderState( self ):
      # BUG613334: once the different possible failures have been identified we 
      # we will revisit rendering test state.
      if self.state == 'none':
         print( 'State: not started' )
      elif self.state == 'running':
         print( 'State: running' )
      elif self.state == 'complete':
         print( 'State: complete (successful)' )
      elif self.state == 'completeWithFailure':
         print( 'State: complete (failed)' )
      elif self.state == 'interfaceFailed':
         print( 'State: interface status changed (failed)' )
      elif self.state == 'cancelled':
         print( 'State: cancelled (stopped)' )
      elif self.state == 'failed':
         print( 'State: programming error' )
      else:
         assert False, 'Bad state: %s' % self.state

   def render( self ):
      self.renderState()
      if self.state != 'none':
         self.renderTime( self.startTime, startTime=True )
         self.renderTime( self.endTime, startTime=False )
         print( 'Packet size:', self.pktSize )
         print( 'Link qualification duration:', self.duration )
         unit = enumToRateUnitMap.get( self.rateUnit, 'bps' )
         print( 'Traffic rate:', self.rate, unit )
         
         if self.state != 'running':
            print( 'Packets sent:', self.pktSent )
            print( 'Packets received:', self.pktRcvd )
            print( 'Packets dropped:', self.pktDropped )
            print( 'Packet transmit errors:', self.txErrors )
            print( 'Packet receive errors:', self.rxErrors )
            print( 'Octets sent:', self.txOctetCount )
            print( 'Octets received:', self.rxOctetCount )

class LinkQualificationGeneratorAllIntfsModel( Model ):
   interfaces = Dict( 
      keyType=IntfModels.Interface, valueType=LinkQualificationGeneratorIntfModel,
      help='Summary of each generator interface config and the test result, '
           'keyed by generator interface' )
   enabled = Bool( help='Link Qualification feature is enabled' )

   def render( self ):
      if not self.interfaces:
         return
      intfs = Arnet.sortIntf( self.interfaces )
      lastIntf = intfs[ -1 ]
      for intf in intfs:
         print( 'Interface:', intf )
         self.interfaces[ intf ].render()
         if intf != lastIntf:
            print()
