# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import BasicCli
import CliCommand
import CliMatcher
import CliSession
import CliToken.Configure
import CommonGuards
import GnmiSetCliSession
import GnmiCommonLib
import LazyMount
import Tac
import Tracing
from CliPlugin import SessionCli
from CliPlugin import CliFragmentCli as CFC
import os
import threading

t0 = Tracing.Handle( "GnmiSet" ).trace0
t9 = Tracing.Handle( "GnmiSet" ).trace9
fragmentConfig = None

cmvStateLock = threading.Lock()
cmvState = None

relativePathMatcher = CliMatcher.PatternMatcher(
      r'[-_a-zA-Z0-9/]+',
      helpname='WORD',
      helpdesc='Relative path for gNMI operations' )

#-----------------------------------------------------------------------------------
# configure session type gnmi
#-----------------------------------------------------------------------------------

class ConfigSessionType( CliCommand.CliCommandClass ):
   syntax = 'configure session type gnmi'
   data = {
            'configure': CliToken.Configure.configureParseNode,
            'session': SessionCli.sessionKwForConfig,
            'type': CliCommand.hiddenKeyword( 'type' ),
            'gnmi': CliCommand.hiddenKeyword( 'gnmi' )
          }

   @staticmethod
   @Ark.synchronized( CliSession.sessionLock )
   def handler( mode, args ):
      # Current action of type gnmi is to setup commit anchor that disables/enables
      # all ToExternalSm during EntityCopy. This is done by setting the anchor
      # logic path in session's pendingChange config.
      sessionName = CliSession.currentSession( mode.entityManager )
      if sessionName is None:
         mode.addError( "No current session." )
         return
      # Set the session state required for gNMI SET
      GnmiSetCliSession.resetGnmiSetState()
      state = Tac.singleton( "AirStream::GnmiSetSessionState" )
      state.sessionName = sessionName
      state.sessionPath = ( f"/{mode.entityManager.sysname()}/Sysdb/session/"
                            f"sessionDir/{sessionName}/" )

BasicCli.EnableMode.addCommandClass( ConfigSessionType )

#-----------------------------------------------------------------------------------
# session-root add group airstream-cmv
#-----------------------------------------------------------------------------------

# Once BUG925966 is fixed, we can remove this
class SessionRootAddGroup( CliCommand.CliCommandClass ):
   syntax = 'session-root add group airstream-cmv'
   data = {
      'session-root': CliCommand.hiddenKeyword( 'session-root' ),
      'add': 'Add group paths to session',
      'group': 'Config group',
      'airstream-cmv': 'The airstream-cmv group'
   }

   @staticmethod
   def handler( mode, args ):
      em = mode.session.entityManager
      h = CliSession.handlerDir( em )
      for path in h.airstreamConfigGroup:
         CliSession.addSessionRoot( em, path )

SessionCli.ConfigSessionMode.addCommandClass( SessionRootAddGroup )

# -----------------------------------------------------------------------------------
#  gnmi add path {PATH}
# -----------------------------------------------------------------------------------

def gnmiAddPathHelper( em, path, sessionPath=None ):
   def getParentPathAndChildName( path ):
      tokens = path.rsplit( "/", 1 )
      return tokens[ 0 ], tokens[ 1 ]

   # Called when the requested CMV entity's parent path is not created yet in
   # ConfigAgent
   def createParentPath( prefix, parentPath ):
      tokens = parentPath.split( '/' )
      currPath = prefix
      parent = None
      for token in tokens:
         currPath = os.path.join( currPath, token )
         try:
            parent = Tac.root.entity[ currPath ]
         except KeyError:
            parent = Tac.root.entity[ currPath.rsplit( '/', 1 )[ 0 ] ].mkdir( token )
      return parent

   cmvProfile = cmvState.cmvProfile
   cmvPath = cmvProfile.cmvPath

   if path not in cmvPath:
      error = f'{path} is not a CMV path, ignoring'
      return error

   prefix = sessionPath or f'/{em.sysname()}/ConfigAgent/'
   info = cmvPath[ path ]
   parentPath, entName = getParentPathAndChildName( path )
   t9( f'gnmiAddPathHelper: {parentPath}, {entName}' )
   try:
      parent = Tac.root.entity[ prefix + parentPath ]
   except KeyError:
      t0( f'Parent path "{parentPath}" of "{entName}" does not exist, creating' )
      parent = createParentPath( prefix, parentPath )

   # Instantiate CMV
   cmv = None
   if sessionPath:
      rootEnt = parent.newEntity( 'AirStream::CmvRootContainer', 'CmvRoot' )
      rootEnt.root = Tac.newInstance( 'Tac::Dir', 'root' )
      cmv = rootEnt.root.createEntity( info.cmvType, entName )
   else:
      cmv = parent.newEntity( info.cmvType, entName )
   # Set all relevant EOS entities in CMV
   prefix = sessionPath or f'/{em.sysname()}/Sysdb/'
   for attr, eosPath in info.eosAttr.items():
      t9( f'{attr=}, {eosPath=}' )
      try:
         cmv.__setattr__( attr, Tac.root.entity[ prefix + eosPath ] )
      except KeyError:
         t0( f'Cannot set {attr} to {prefix + eosPath}, path does not exist' )
         parent.deleteEntity( entName )
   return None

@Ark.synchronized( cmvStateLock )
def cmvStateInit( prefix ):
   global cmvState
   if not cmvState:
      cmvState = Tac.root.entity[ prefix ].newEntity(
         'AirStream::CmvState', "cmvState" )
      # Load CMV profiles
      cmvState.cmvProfile = Tac.newInstance( 'AirStream::CmvProfile' )
      GnmiCommonLib.loadCmvProfiles( cmvState )
   return cmvState

def gnmiAddPath( mode, args ):
   em = mode.entityManager
   gnmiConf = em.lookup( 'mgmt/gnmi/config' )
   octaConf = em.lookup( 'mgmt/octa/config' )
   if not gnmiConf and not octaConf:
      error = "no gNMI transports enabled yet"
      mode.addError( error )
      return error
   # Get CmvState object
   prefix = '/' + em.sysname() + '/ConfigAgent/'
   # Instantiate CmvState if required
   cmvStateInit( prefix )
   paths = args.get( 'PATH' )
   for path in paths:
      error = gnmiAddPathHelper( em, path )
      if error is not None:
         mode.addError( error )
         return error

   return ''

class GnmiAddPath( CliCommand.CliCommandClass ):
   syntax = 'gnmi cmv add path {PATH}'
   data = {
      'gnmi': 'gNMI operations',
      'cmv': 'CMV operations',
      'add': 'Add the specified path',
      'path': 'Add the specified path',
      'PATH': CliCommand.Node( matcher=relativePathMatcher,
         guard=CommonGuards.ssoStandbyGuard ),
   }
   hidden = True
   handler = gnmiAddPath

BasicCli.EnableMode.addCommandClass( GnmiAddPath )

# -----------------------------------------------------------------------------------
# session gnmi add path {PATH}
# -----------------------------------------------------------------------------------

def sessionGnmiAddPath( mode, args, sessionName=None ):
   paths = args.get( 'PATH' )
   for path in paths:
      if path.startswith( '/' ):
         error = f'{path} has root as the prefix'
         mode.addError( error )
         return error
   state = Tac.singleton( "AirStream::GnmiSetState" )
   em = mode.session.entityManager
   if not sessionName:
      sessionName = CliSession.currentSession( em )

   if state.sessionName != sessionName:
      t0( f"session name changed from {state.sessionName} to {sessionName}" )
      GnmiSetCliSession.resetGnmiSetState()
      state.sessionName = sessionName

   # Add session root, update the state.path, start SM if needed
   for path in paths:
      t9( f'add path {path}' )
      # If we are dealing with a CMV path, we only entity copy the native entities to
      # the session root, and we create the CMV entity manually inside the session by
      # calling gnmiAddPathHelper with extra args.
      prefix = '/' + em.sysname() + '/ConfigAgent/'
      cmvStateInit( prefix )
      if path in cmvState.cmvProfile.cmvPath:
         cmvPath = cmvState.cmvProfile.cmvPath[ path ]
         sessionPath = f'/{em.sysname()}/Sysdb/session/sessionDir/{sessionName}/'
         for eosPath in cmvPath.eosAttr.values():
            CliSession.addSessionRoot( em, eosPath, sessionName )
         error = gnmiAddPathHelper( em, path, sessionPath=sessionPath )
         if error is not None:
            mode.addError( error )
            return error
      else:
         if error := CliSession.addSessionRoot( em, path, sessionName ):
            mode.addError( error )
            return error
      state.path[ path ] = True
      GnmiSetCliSession.SmRunner.runSm( em, sessionName, path )
   return ''

class SessionGnmiAddPath( CliCommand.CliCommandClass ):
   syntax = 'session gnmi add path {PATH}'
   data = {
      'session': 'Commands for session',
      'gnmi': 'gNMI operations',
      'add': 'Add the specified path',
      'path': 'Add the specified path',
      'PATH': CliCommand.Node( matcher=relativePathMatcher,
         guard=CommonGuards.ssoStandbyGuard ),
   }
   hidden = True
   handler = sessionGnmiAddPath

SessionCli.ConfigSessionMode.addCommandClass( SessionGnmiAddPath )

# -----------------------------------------------------------------------------------
# session gnmi sync
# -----------------------------------------------------------------------------------
def sessionGnmiSync( mode, args, sessionName=None ):
   state = Tac.singleton( "AirStream::GnmiSetState" )
   em = mode.session.entityManager
   # If the sessionName is none, this is 'session gnmi sync' and must be called
   # inside the session.
   # Otherwise, it is 'config fragment gnmi ... sync' for config fragment and
   # must be call outside session.
   if sessionName is None:
      sessionName = CliSession.currentSession( em )
      if sessionName is None:
         error = 'The command must be called inside the session'
         mode.addError( error )
         GnmiSetCliSession.resetGnmiSetState()
         return error
   elif CliSession.currentSession( em ) is not None:
      error = ( f'The sync for fragment session {sessionName} must be '
                f'called outside session {CliSession.currentSession( em )}' )
      mode.addError( error )
      GnmiSetCliSession.resetGnmiSetState()
      return error

   if state.sessionName != sessionName:
      error = ( f'Sync without prior "add path" command. '
                f'state.sessionName {state.sessionName} sessionName {sessionName}' )
      mode.addError( error )
      GnmiSetCliSession.resetGnmiSetState()
      return error

   try:
      t0( 'sessionGnmiSync: runToNativeSynchers' )
      GnmiSetCliSession.runToNativeSyncers( em, sessionName, state )
   except CliSession.PreCommitHandlerError as e:
      t0( f'sessionGnmiSync: {e.debugMsg_}' )
      error = e.recordAndResponse()
      mode.addError( error )
      return error

   return ''

class SessionGnmiSync( CliCommand.CliCommandClass ):
   syntax = 'session gnmi sync'
   data = {
      'session': 'Commands for session',
      'gnmi': 'gNMI operations',
      'sync': 'sync external entities to native entities',
   }
   hidden = True
   handler = sessionGnmiSync

SessionCli.ConfigSessionMode.addCommandClass( SessionGnmiSync )

#-----------------------------------------------------------------------------------
# configure fragment gnmi <fragmentName> add path {PATH}
#-----------------------------------------------------------------------------------
def fragmentGnmiAddPath( mode, args ):
   fragmentName = args.get( '<fragmentName>' )
   t0( f"add paths for fragment {fragmentName}" )
   if error:= CFC.verifyPendingSessions( 'enableGnmi', mode, [ fragmentName ] ):
      return error

   sessionName = \
         fragmentConfig.cliFragmentDir.config.get( fragmentName ).pendingSessionName
   if error := sessionGnmiAddPath( mode, args, sessionName=sessionName ):
      return error
   state = Tac.singleton( "AirStream::GnmiSetState" )
   state.fragmentName = fragmentName
   return ''

class FragmentGnmiAddPath( CliCommand.CliCommandClass ):
   # The gNMI operations can be applied on only one session at a time,
   # so this is a command for one specific fragment.
   syntax = 'configure fragment gnmi <fragmentName> add path {PATH}'
   data = {
      'configure': CliToken.Configure.configureParseNode,
      'fragment': CFC.fragmentKwForConfig,
      'gnmi': 'gNMI operations for the specified fragment',
      '<fragmentName>': CliCommand.Node(
         matcher=CFC.fragmentNameMatcher,
         guard=CommonGuards.ssoStandbyGuard ),
      'add': 'Add the specified path',
      'path': 'Add the specified path',
      'PATH': CliCommand.Node( matcher=relativePathMatcher,
         guard=CommonGuards.ssoStandbyGuard ),
   }
   hidden = True
   handler = fragmentGnmiAddPath

BasicCli.EnableMode.addCommandClass( FragmentGnmiAddPath )

#-----------------------------------------------------------------------------------
# configure fragment gnmi <fragmentName> sync
#-----------------------------------------------------------------------------------
def fragmentGnmiSync( mode, args ):
   fragmentName = args.get( '<fragmentName>' )
   state = Tac.singleton( "AirStream::GnmiSetState" )
   if state.fragmentName != fragmentName:
      error = ( "Fragment name doesn't match previous 'add path' command: "
         f"cli {fragmentName} state {state.fragmentName}" )
      mode.addError( error )
      return error

   if error:= CFC.verifyPendingSessions( 'closeGnmi', mode, [ fragmentName ] ):
      return error

   sessionName = \
         fragmentConfig.cliFragmentDir.config.get( fragmentName ).pendingSessionName
   if error := sessionGnmiSync( mode, args, sessionName ):
      return error

   return ''

class FragmentGnmiSync( CliCommand.CliCommandClass ):
   # Sync the current gNMI fragment.
   syntax = 'configure fragment gnmi <fragmentName> sync'
   data = {
      'configure': CliToken.Configure.configureParseNode,
      'fragment': CFC.fragmentKwForConfig,
      'gnmi': 'gNMI operations for the specified fragment',
      '<fragmentName>': CliCommand.Node(
         matcher=CFC.fragmentNameMatcher,
         guard=CommonGuards.ssoStandbyGuard ),
      'sync': 'sync external entities to native entities',
   }
   hidden = True
   handler = fragmentGnmiSync

BasicCli.EnableMode.addCommandClass( FragmentGnmiSync )

def Plugin( entityManager ):
   global fragmentConfig
   fragmentConfig = LazyMount.mount( entityManager,
                                     "cli/fragment/config",
                                     "Cli::Session::FragmentConfig",
                                     "r" )
