#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from operator import attrgetter

from Arnet import IpGenPrefix
from CliPlugin.IpAddrMatcher import ipPrefixMatcher
from CliPlugin.Ip6AddrMatcher import ip6PrefixMatcher
from CliPlugin.GribiModel import (
   AcknowledgementFecVersionId,
   AcknowledgementRouteVersionId,
   GribiMplsAftType,
   GribiMplsAftDetail,
   GribiFecDetail,
   GribiFecVia,
   GribiIpAft,
   GribiIpAftVersionType,
   GribiPendingIpAft,
   GribiIpAftEntry,
   GribiMplsAftTable,
   GribiNhgNhAftType,
   GribiNhgAftVersionType,
   GribiPendingNhgAftTable,
   GribiNhgAftType,
   GribiNhgAftTable,
   GribiNhAftType,
   GribiIpInIpEncap,
   GribiNhAftDetail,
   GribiNhAftVersionType,
   GribiPendingNhAftTable,
   GribiNhAftTable,
   GribiAclStatus,
   GribiEndpointStatus,
   GribiElectionId,
   GribiModifyRpcClientIpPortStatus,
   GribiModifyRpcClientIpStatus,
   GribiModifyRpcStreamStatus,
   GribiRpcStatus,
)
from CliPlugin.SrTePolicyLibCli import getSegmentListVias
from CliPlugin.TunnelCli import (
   TunnelTableIdentifier,
   readMountTunnelTable,
)
from CliPlugin.VrfCli import (
   ALL_VRF_NAME,
   vrfExists,
   getVrfNames,
   generateVrfCliModel,
   VrfExecCmdDec,
   VrfExprFactory,
)
from CliPlugin import AclCli
from CliPlugin import ConfigMgmtMode
import CliCommand
from IpLibConsts import DEFAULT_VRF
from SrTePolicyLib import MplsLabel
from TypeFuture import TacLazyType

import AuthnUserPriorityCli
import BasicCli
import CliMatcher
import LazyMount
import SharedMem
import ShowCommand
import Smash
import Tac
import Toggles.gribiToggleLib

DynTunnelIntfId = TacLazyType( "Arnet::DynamicTunnelIntfId" )
TunnelId = TacLazyType( "Tunnel::TunnelTable::TunnelId" )
VersionId = TacLazyType( "Routing::Fib::VersionId" )

aclCpConfig = None
aclStatus = None
aclCheckpoint = None
smashEm = None
readerInfo = None
gribiAftEntries = None
gribiAcknowledgedAftEntries = None
gribiPendingAckEntries = None
gribiConfig = None
gribiStatus = None
mgmtGribiStatus = None
gribiSessionParams = None
gribiNhToTunnelMap = None
gribiNhgIdFecIdMap = None
slTunnelTable = None
srteForwardingStatus = None
programmedIpv4RouteStatusAllVrf = {} # per vrf programmedRouteStatus
programmedIpv6RouteStatusAllVrf = {} # per vrf programmedRouteStatus
programmedFecStatus = None
gribiModifyRpcStreamStatus = None

ethAddrZero = Tac.Type( "Arnet::EthAddr" ).ethAddrZero

matcherTrafficEngineering = CliMatcher.KeywordMatcher( 'traffic-engineering',
      helpdesc='Traffic Engineering related information' )
matcherManagement = ConfigMgmtMode.managementShowKwMatcher
matcherGribi = CliMatcher.KeywordMatcher( 'gribi',
      helpdesc='Show gRIBI AFT entries' )
matcherGribiMgmtApi = CliMatcher.KeywordMatcher( 'gribi',
      helpdesc='Show status of gRIBI endpoints' )
matcherNexthopGroupAft = CliMatcher.KeywordMatcher( 'nexthop-group-aft',
      helpdesc='Show gRIBI Nexthop Group AFT entries' )
matcherNhgId = CliMatcher.KeywordMatcher( 'id',
      helpdesc='Nexthop Group AFT ID' )
matcherMplsAft = CliMatcher.KeywordMatcher( 'mpls-aft',
      helpdesc='Show gRIBI MPLS AFT entries' )
matcherIpv4Aft = CliMatcher.KeywordMatcher( 'ipv4-aft',
      helpdesc='Show gRIBI IPv4 AFT entries' )
matcherIpv6Aft = CliMatcher.KeywordMatcher( 'ipv6-aft',
   helpdesc='Show gRIBI IPv6 AFT entries' )
matcherVrfName = VrfExprFactory(
      helpdesc='Show AFT entries for a VRF',
      inclAllVrf=True, inclDefaultVrf=True )
matcherNexthopAft = CliMatcher.KeywordMatcher( 'nexthop-aft',
      helpdesc='Show gRIBI Nexthop AFT entries' )
matcherLabel = CliMatcher.KeywordMatcher( 'label',
      helpdesc='Show gRIBI MPLS AFT entry for a label' )
matcherAcknowledged = CliMatcher.KeywordMatcher( 'acknowledged',
      helpdesc='latest acknowledged entries' )
matcherUnacknowledged = CliMatcher.KeywordMatcher( 'unacknowledged',
      helpdesc='unacknowledged entries' )
matcherMismatched = CliMatcher.KeywordMatcher( 'mismatched-ack',
      helpdesc='mismatched-ack entries' )
matcherFailed = CliMatcher.KeywordMatcher( 'failed',
      helpdesc='Entries that failed hardware programming' )
matcherNhId = CliMatcher.KeywordMatcher( 'id',
      helpdesc='Nexthop AFT ID' )
matcherDetail = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='Show AFT detail' )
matcherAccessList = AclCli.accessListKwMatcherForServiceAcl
matcherApiForShow = CliMatcher.KeywordMatcher( 'api',
      helpdesc='Show management APIs' )
matcherDynamic = CliMatcher.KeywordMatcher( 'dynamic',
      helpdesc='Dynamic (non-persistent) access-list' )
matcherSummary = CliMatcher.KeywordMatcher( 'summary',
      helpdesc='Access list summary' )
nodeDetail = CliCommand.guardedKeyword( 'detail',
      helpdesc='Access list detail',
      guard=AclCli.countersPerChipEnabledGuard )


IpAftVrfModel = generateVrfCliModel( GribiIpAft,
                                     desc="IP AFT entries per VRF" )

PendingIpAftVrfModel = generateVrfCliModel( GribiPendingIpAft,
                                     desc="Unacknowledged AFT entries per VRF" )

def fibAckEnabled():
   return ( gribiSessionParams is not None and
            gribiSessionParams.ackType == 'ribAndFibAck' )

def nhgIdToFecId( nhgId ):
   return gribiNhgIdFecIdMap.nhgIdToFecId.get( nhgId )

def isFailedNhgAftEntry( key ):
   fecId = nhgIdToFecId( key )
   if fecId is not None:
      fecVersion = programmedFecStatus.fecVersion.get( fecId )
      if fecVersion is not None:
         return fecVersion.getRawAttribute( 'versionId' ).nack
   return False

def isMismatchedNhgAftEntry( key, entry ):
   fecId = nhgIdToFecId( key )
   if fecId is not None:
      fecVersion = programmedFecStatus.fecVersion.get( fecId )
      if fecVersion is not None:
         fecVersion = fecVersion.versionId & VersionId.versionMask
      else:
         fecVersion = 0
      entryVersion = entry.versionId & VersionId.versionMask
      return fecVersion != entryVersion
   return False

def getOneNhgAft( key, nhgAftEntry, aftTable, detail=False ):
   aft = GribiNhgAftType()
   aft.nhgId = nhgAftEntry.nhgAftEntryId
   if nhgAftEntry.backupNhgId:
      aft.backupNhgId = nhgAftEntry.backupNhgId
   fecId = nhgIdToFecId( key )
   if fibAckEnabled():
      reqFibAck = AcknowledgementFecVersionId()
      reqFibAck.versionId = nhgAftEntry.versionId & VersionId.versionMask
      reqFibAck.requestedDelete = bool(
         nhgAftEntry.versionId & VersionId.tombstoneBit )
      # we never request for a failed (nack) update, so reqFibAck.failed is always
      # false
      reqFibAck.failed = False
      aft.fibAckRequested = reqFibAck
      ackFibAck = AcknowledgementFecVersionId()
      fecVersion = None
      if fecId is not None:
         fecVersion = programmedFecStatus.fecVersion.get( fecId )
      if fecVersion is not None:
         ackFibAck.versionId = fecVersion.versionId & VersionId.versionMask
         ackFibAck.requestedDelete = fecVersion.getRawAttribute(
            'versionId' ).tombstone
         ackFibAck.failed = fecVersion.getRawAttribute( 'versionId' ).nack
         aft.fibAckAcknowledged = ackFibAck
   if detail:
      # For NHGs that have IP next hops GribiRoute writes the NHGID->FECID
      # mapping in nhgIdFecMap
      # gribiStatus.nhgToFecEntry is written by SrTePolicy agent for the MPLS
      # use case.
      aft.fecId = gribiStatus.nhgToFecEntry.get( aft.nhgId )
      if not aft.fecId:
         if fecId:
            aft.fecId = fecId

   for index in sorted( nhgAftEntry.nhIdColl ):
      nhgNhIdCollEntry = nhgAftEntry.nhIdColl[ index ]
      nhId = nhgNhIdCollEntry.nhAftEntryId
      weight = nhgNhIdCollEntry.weight

      nhgNhEntry = GribiNhgNhAftType()
      nhgNhEntry.nhId = nhId
      nhgNhEntry.weight = weight
      # default value of a 'List' is "[]" json output will print "[]"
      # which could cause confusion in non-detailed version of the
      # command so explicitly set it to None value so that it will
      # not be printed in json output.
      nhgNhEntry.mplsVias = None

      if detail:
         nhEntry = aftTable.nhAft.entry.get( nhId )
         if nhEntry is not None:
            if nhEntry.ipAddr and not nhEntry.ipAddr.isAddrZero:
               nhgNhEntry.ipAddress = nhEntry.ipAddr
            if nhEntry.intf:
               nhgNhEntry.interface = nhEntry.intf
            if nhEntry.macAddr and nhEntry.macAddr != ethAddrZero:
               nhgNhEntry.macAddress = nhEntry.macAddr
         if nhEntry is not None:
            if nhEntry.decap != 'none':
               nhgNhEntry.decapActionType = nhEntry.decap
            if nhEntry.encap != 'none':
               nhgNhEntry.encapActionType = nhEntry.encap
            if nhEntry.fallbackVrf != "":
               nhgNhEntry.fallbackVrf = nhEntry.fallbackVrf
            if nhEntry.ipInIpEncap \
               and not ( nhEntry.ipInIpEncap.srcIp.isAddrZero
                         and nhEntry.ipInIpEncap.dstIp.isAddrZero ):
               nhgNhEntry.ipInIpEncap = GribiIpInIpEncap()
               nhgNhEntry.ipInIpEncap.srcIp = nhEntry.ipInIpEncap.srcIp
               nhgNhEntry.ipInIpEncap.dstIp = nhEntry.ipInIpEncap.dstIp
         tunnelId = gribiNhToTunnelMap.nhToTunnelMap.get( nhId )
         if tunnelId:
            tunnelEntry = slTunnelTable.entry.get( tunnelId, None )
            if tunnelEntry:
               nhgNhEntry.mplsVias, _ = getSegmentListVias( tunnelEntry )

      aft.nhgNhs.append( nhgNhEntry )
   return aft

def populatePendingNhgAft( nhgId, detail=False ):
   # get keys
   keys = []
   if nhgId is not None:
      keys.append( nhgId )
   else:
      keys = gribiPendingAckEntries.pendingNhgAcks
   # populate nexthop-group AFT entries
   nhgAftTable = GribiPendingNhgAftTable()
   nhgAftTable.unacknowledgedNhgs = {}
   nhgAftTable.details_ = detail
   for key in sorted( keys ):
      nhgColl = gribiPendingAckEntries.pendingNhgAcks.get( key )
      if nhgColl:
         nhgAftVersionTable = GribiNhgAftVersionType()
         nhgAftTable.unacknowledgedNhgs[ key ] = nhgAftVersionTable
         versions = nhgColl.versionColl
         for version in versions:
            nhgAftEntry = versions.get( version )
            if nhgAftEntry is None:
               continue
            aft = getOneNhgAft( key, nhgAftEntry, gribiAftEntries, detail )
            nhgAftVersionTable.nhgVersionAft[ version & VersionId.versionMask ] = aft
   return nhgAftTable

def populateNhgAft( nhgId, detail=False, acknowledged=False,
                    onlyFailedEntries=False,
                    onlyMismatchedEntries=False ):
   # get keys
   keys = []
   if not acknowledged:
      aftTable = gribiAftEntries
   else:
      aftTable = gribiAcknowledgedAftEntries

   if nhgId is not None:
      keys.append( nhgId )
   else:
      keys = aftTable.nhgAft.entry

   # populate nexthop-group AFT entries
   nhgAftTable = GribiNhgAftTable()
   nhgAftTable.nhgAft = {}
   nhgAftTable.details_ = detail
   for key in sorted( keys ):
      nhgAftEntry = aftTable.nhgAft.entry.get( key )
      if nhgAftEntry is None:
         continue
      if ( onlyFailedEntries and not isFailedNhgAftEntry( key ) ):
         continue
      if ( onlyMismatchedEntries
           and not isMismatchedNhgAftEntry( key, nhgAftEntry ) ):
         continue
      aft = getOneNhgAft( key, nhgAftEntry, aftTable, detail )
      nhgAftTable.nhgAft[ int( nhgAftEntry.nhgAftEntryId ) ] = aft
   return nhgAftTable

def populateMplsAft( label, detail=False ):
   # get keys
   keys = []
   if label is not None:
      keys.append( label )
   else:
      keys = gribiAftEntries.mplsAft.entry

   # populate mpls-aft
   mplsAftTable = GribiMplsAftTable()
   mplsAftTable.mplsAft = {}
   for key in sorted( keys ):
      mplsAft = gribiAftEntries.mplsAft.entry.get( key )
      if mplsAft is None:
         continue
      aft = GribiMplsAftType()
      aft.label = mplsAft.label
      aft.nhgId = mplsAft.nhgAftEntryId

      if detail:
         # populate mpls-aft detail
         aft.mplsAftDetail = GribiMplsAftDetail()
         fecId = gribiStatus.nhgToFecEntry.get( aft.nhgId )
         if fecId is None:
            mplsAftTable.mplsAft[ int( mplsAft.label ) ] = aft
            continue
         fecDetail = GribiFecDetail()
         fecDetail.fecId = fecId
         fec = srteForwardingStatus.fec.get( fecId )
         if fec:
            for via in fec.via.values():
               if not DynTunnelIntfId.isDynamicTunnelIntfId( via.intfId ):
                  continue
               fecVia = GribiFecVia()
               fecVia.tunnelIdx = \
                     TunnelId( DynTunnelIntfId.tunnelId( via.intfId )
                             ).tunnelIndex()
               fecVia.weight = via.weight
               # get tunnel info
               tunnelId = TunnelId( DynTunnelIntfId.tunnelId( via.intfId ) )
               tunnelEntry = slTunnelTable.entry.get( tunnelId, None )
               if tunnelEntry:
                  fecVia.mplsVias, _ = getSegmentListVias( tunnelEntry )
               fecDetail.fecVias.append( fecVia )
            fecDetail.fecVias = sorted( fecDetail.fecVias,
                                         key=attrgetter( 'tunnelIdx' ) )
         aft.mplsAftDetail.fecDetail = fecDetail
      mplsAftTable.mplsAft[ int( mplsAft.label ) ] = aft
   return mplsAftTable

def pfxVrfFromIpAftEntryKeyFn( prefixVrfname ):
   return prefixVrfname.split( '|', 1 )

def isFailedIpAftEntry( programmedEntry ):
   if not ( programmedEntry and fibAckEnabled() ):
      return False
   return programmedEntry.getRawAttribute( 'versionId' ).nack

def isMismatchedIpAftEntry( programmedEntry, entry ):
   if programmedEntry and fibAckEnabled():
      programmedVersionId = programmedEntry.versionId & VersionId.versionMask
   else:
      programmedVersionId = 0
   return ( ( entry.versionId & VersionId.versionMask ) !=
             programmedVersionId )

def getOnePendingIpAft( pfx, entry, version ):
   modelEntry = GribiIpAftEntry()
   modelEntry.prefix = pfx
   modelEntry.nhgId = entry.nhgAftEntryId
   if entry.metadata:
      modelEntry.metadata = entry.metadata
   if fibAckEnabled():
      reqFibAck = AcknowledgementRouteVersionId()
      reqFibAck.versionId = version & VersionId.versionMask
      reqFibAck.requestedDelete = bool( version & VersionId.tombstoneBit )
      reqFibAck.failed = bool( version & VersionId.nackBit )
      modelEntry.fibAckRequested = reqFibAck
   return modelEntry

def getOneIpAft( pfx, entry, programmedEntry ):
   modelEntry = GribiIpAftEntry()
   modelEntry.prefix = pfx
   modelEntry.nhgId = entry.nhgAftEntryId
   if entry.metadata:
      modelEntry.metadata = entry.metadata
   if fibAckEnabled():
      reqFibAck = AcknowledgementRouteVersionId()
      reqFibAck.versionId = entry.versionId & VersionId.versionMask
      reqFibAck.requestedDelete = bool( entry.versionId & VersionId.tombstoneBit )
      # We can never request for a failed ack aka nack, so failed is always false
      reqFibAck.failed = False
      modelEntry.fibAckRequested = reqFibAck
      ackFibAck = AcknowledgementRouteVersionId()
      if programmedEntry is not None:
         ackFibAck.versionId = programmedEntry.versionId & VersionId.versionMask
         ackFibAck.requestedDelete = programmedEntry.getRawAttribute(
            'versionId' ).tombstone
         ackFibAck.failed = programmedEntry.getRawAttribute( 'versionId' ).nack
      modelEntry.fibAckAcknowledged = ackFibAck
   return modelEntry

def populatePendingIpAftEntry( model, keys, pendingRouteAcks, v4 ):
   for pfxVrfname in sorted( keys ):
      ( pfx, _ ) = pfxVrfFromIpAftEntryKeyFn( pfxVrfname )
      v4pfx = ':' not in pfx
      if v4 != v4pfx:
         continue
      routeVersion = pendingRouteAcks.get( pfxVrfname )
      if routeVersion is None:
         continue
      # populate ipv4 AFT entries
      ipAftVersionTable = GribiIpAftVersionType()
      ipAftVersionTable.ipVersionAft = {}
      versions = routeVersion.versionColl
      aft = {}
      for version in sorted( versions ):
         entry = versions.get( version )
         if entry is None:
            continue
         aft = getOnePendingIpAft( pfxVrfname, entry, version )
         ipAftVersionTable.ipVersionAft[ version & VersionId.versionMask ] = aft
      yield pfx, ipAftVersionTable

def populateIpAftEntry( model, pfxs, vrfAft, programmedEntries, v4=True,
                        onlyFailedEntries=False,
                        onlyMismatchedEntries=False ):
   for pfx in pfxs:
      entry = vrfAft.entry.get( pfx )
      if entry is None:
         continue
      programmedEntry = programmedEntries.routeVersion.get( pfx.v4Prefix if v4
                                                            else pfx.v6Prefix )
      if ( onlyFailedEntries and not isFailedIpAftEntry( programmedEntry ) ):
         continue
      if ( onlyMismatchedEntries
           and not isMismatchedIpAftEntry( programmedEntry, entry ) ):
         continue
      aft = getOneIpAft( pfx, entry, programmedEntry )
      yield pfx.stringValue, aft

def getProgrammedRouteStatus( af, vrfName ):
   '''
   Return the ProgrammedRouteStatus collection for `af` address family for `vrfName`
   VRF.
   Also mounts the smash table if not mounted.
   '''
   if af == 'ipv4':
      if vrfName not in programmedIpv4RouteStatusAllVrf:
         path = 'routing/vrf/routeVersion/' + vrfName
         if vrfName == DEFAULT_VRF:
            path = 'routing/routeVersion'
         programmedIpv4RouteStatusAllVrf[ vrfName ] = smashEm.doMount(
            path, 'Smash::Fib::ProgrammedRouteStatus', readerInfo )
      returnProgrammedRouteStatus = programmedIpv4RouteStatusAllVrf[ vrfName ]
   else:
      if vrfName not in programmedIpv6RouteStatusAllVrf:
         path = 'routing6/vrf/routeVersion/' + vrfName
         if vrfName == DEFAULT_VRF:
            path = 'routing6/routeVersion'
         programmedIpv6RouteStatusAllVrf[ vrfName ] = smashEm.doMount(
            path, 'Smash::Fib6::ProgrammedRouteStatus', readerInfo )
      returnProgrammedRouteStatus = programmedIpv6RouteStatusAllVrf[ vrfName ]
   return returnProgrammedRouteStatus

def showPendingIpAft( mode, vrfName, prefix, v4=True ):
   if not ( vrfName is None or vrfExists( vrfName ) ):
      if vrfName != ALL_VRF_NAME:
         mode.addError( "Ip routing table %s does not exist." % vrfName )
         return None

   model = GribiPendingIpAft()
   model._tableFmt = v4 # pylint: disable=protected-access
   model._vrf = vrfName # pylint: disable=protected-access
   pendingRouteAcks = gribiPendingAckEntries.pendingRouteAcks
   pfxVrfs = []
   if prefix and vrfName:
      pfxVrfs.append( prefix + '|' + vrfName )
   elif not ( prefix or vrfName ):
      pfxVrfs = pendingRouteAcks
   elif prefix:
      for pfxVrfname in pendingRouteAcks:
         ( pfx, vrf ) = pfxVrfFromIpAftEntryKeyFn( pfxVrfname )
         if pfx == prefix:
            pfxVrfs.append( pfxVrfname )
   elif vrfName:
      for pfxVrfname in pendingRouteAcks:
         ( pfx, vrf ) = pfxVrfFromIpAftEntryKeyFn( pfxVrfname )
         if vrf == vrfName:
            pfxVrfs.append( pfxVrfname )

   model.unacknowledgedPrefixes = populatePendingIpAftEntry(
      model, pfxVrfs, pendingRouteAcks, v4 )
   return model

@VrfExecCmdDec( getVrfsFunc=getVrfNames,
                cliModel=PendingIpAftVrfModel )
def showPendingIpv4Aft( mode, vrfName, prefix ):
   return showPendingIpAft( mode, vrfName, prefix, v4=True )

@VrfExecCmdDec( getVrfsFunc=getVrfNames,
                cliModel=PendingIpAftVrfModel )
def showPendingIpv6Aft( mode, vrfName, prefix ):
   return showPendingIpAft( mode, vrfName, prefix, v4=False )

def showIpAft( mode, vrfName, prefix, acknowledged=False,
               v4=True,
               onlyFailedEntries=False,
               onlyMismatchedEntries=False ):
   if not ( vrfName is None or vrfExists( vrfName ) ):
      if vrfName != ALL_VRF_NAME:
         mode.addError( "Ip routing table %s does not exist." % vrfName )
         return None
   if not acknowledged:
      aftTable = gribiAftEntries
   else:
      aftTable = gribiAcknowledgedAftEntries
   model = GribiIpAft()
   model._tableFmt = v4 # pylint: disable=protected-access
   model._vrf = vrfName # pylint: disable=protected-access
   if v4:
      vrfAft = aftTable.ipv4Aft.aft.get( vrfName )
      programmedRouteEntries = getProgrammedRouteStatus( 'ipv4', vrfName )
   else:
      vrfAft = aftTable.ipv6Aft.aft.get( vrfName )
      programmedRouteEntries = getProgrammedRouteStatus( 'ipv6', vrfName )

   if vrfAft is None:
      # This does not imply that the VRF does not exist. Just that there are no
      # IPv4(or IPv6) AFT entries in the VRF. So we do not
      # return a "vrf doesn't exist" error
      return model

   pfxs = []
   if prefix is not None:
      pfxs = [ IpGenPrefix( prefix ) ]
   else:
      pfxs = sorted( vrfAft.entry )
   model.prefixes = populateIpAftEntry( model, pfxs, vrfAft,
                                        programmedRouteEntries, v4,
                                        onlyFailedEntries,
                                        onlyMismatchedEntries )
   return model

@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=IpAftVrfModel )
def showFailedIpv4Aft( mode, vrfName, prefix ):
   return showIpAft( mode, vrfName, prefix, v4=True, onlyFailedEntries=True )

@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=IpAftVrfModel )
def showFailedIpv6Aft( mode, vrfName, prefix ):
   return showIpAft( mode, vrfName, prefix, v4=False, onlyFailedEntries=True )

@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=IpAftVrfModel )
def showIpv4Aft( mode, vrfName, prefix, acknowledged=False, mismatched=False ):
   return showIpAft( mode, vrfName, prefix, acknowledged, v4=True,
                     onlyMismatchedEntries=mismatched )

@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=IpAftVrfModel )
def showIpv6Aft( mode, vrfName, prefix, acknowledged=False, mismatched=False ):
   return showIpAft( mode, vrfName, prefix, acknowledged, v4=False,
                    onlyMismatchedEntries=mismatched )

def populatePendingNhAft( nhId ):
   # get keys
   keys = []
   if nhId is not None:
      keys.append( nhId )
   else:
      keys = gribiPendingAckEntries.pendingNhAcks
   # populate nexthop-aft entries
   nhAftTable = GribiPendingNhAftTable()
   nhAftTable.unacknowledgedNhs = {}
   for key in sorted( keys ):
      nhAft = gribiAftEntries.nhAft.entry.get( key )
      if nhAft is None:
         continue
      aft = GribiNhAftType()
      aft.nhId = nhAft.nhAftEntryId
      nhAftVersionTable = GribiNhAftVersionType()
      nhAftTable.unacknowledgedNhs[ key ] = nhAftVersionTable
      nhAftVersionTable.nhVersionAft[ 1 ] = aft
   return nhAftTable

def populateNhAft( nhId, detail=False ):
   # get keys
   keys = []
   if nhId is not None:
      keys.append( nhId )
   else:
      keys = gribiAftEntries.nhAft.entry

   # populate nexthop-aft
   nhAftTable = GribiNhAftTable()
   nhAftTable.nhAft = {}
   for key in sorted( keys ):
      nhAft = gribiAftEntries.nhAft.entry.get( key )
      if nhAft is None:
         continue
      aft = GribiNhAftType()
      aft.nhId = nhAft.nhAftEntryId
      aft.pushMplsLabels = []
      for i in range( nhAft.pushMplsLabelStack.stackSize ):
         aft.pushMplsLabels.append( nhAft.pushMplsLabelStack.labelStack( i ) )
      if nhAft.ipAddr and not nhAft.ipAddr.isAddrZero:
         aft.ipAddress = nhAft.ipAddr
      if nhAft.intf:
         aft.interface = nhAft.intf
      if nhAft.macAddr and nhAft.macAddr != ethAddrZero:
         aft.macAddress = nhAft.macAddr
      if nhAft.decap != 'none':
         aft.decapActionType = nhAft.decap
      if nhAft.encap != 'none':
         aft.encapActionType = nhAft.encap
      if nhAft.ipInIpEncap \
            and not ( nhAft.ipInIpEncap.srcIp.isAddrZero
                   and nhAft.ipInIpEncap.dstIp.isAddrZero ):
         aft.ipInIpEncap = GribiIpInIpEncap()
         aft.ipInIpEncap.srcIp = nhAft.ipInIpEncap.srcIp
         aft.ipInIpEncap.dstIp = nhAft.ipInIpEncap.dstIp
      if nhAft.fallbackVrf != "":
         aft.vrf = nhAft.fallbackVrf
      if detail:
         aft.nhAftDetail = GribiNhAftDetail()
         tunnelIdx = gribiNhToTunnelMap.nhToTunnelMap.get( aft.nhId )
         if tunnelIdx is None:
            nhAftTable.nhAft[ int( nhAft.nhAftEntryId ) ] = aft
            continue
         # populate nexthop-aft detail
         fecVia = GribiFecVia()
         aft.nhAftDetail.tunnelDetail = fecVia
         tunnelId = TunnelId( tunnelIdx ).tunnelIndex()
         fecVia.tunnelIdx = tunnelId
         tunnelEntry = slTunnelTable.entry.get( tunnelIdx, None )
         if tunnelEntry:
            fecVia.mplsVias, _ = getSegmentListVias( tunnelEntry )
      nhAftTable.nhAft[ int( nhAft.nhAftEntryId ) ] = aft
   return nhAftTable

class ShowGribiPendingNhgAftCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering gribi nexthop-group-aft unacknowledged ' \
      '[ id ID ] [ detail ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'gribi': matcherGribi,
      'nexthop-group-aft': matcherNexthopGroupAft,
      'unacknowledged': matcherUnacknowledged,
      'id': matcherNhgId,
      'ID': CliMatcher.IntegerMatcher( 0, 0xFFFFFFFFFFFFFFFF,
                                       helpdesc='Nexthop group AFT ID' ),
      'detail': matcherDetail,
   }
   cliModel = GribiPendingNhgAftTable

   @staticmethod
   def handler( mode, args ):
      detail = 'detail' in args
      return populatePendingNhgAft( args.get( 'ID' ), detail )

class ShowGribiFailedNhgAftCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering gribi nexthop-group-aft failed ' \
      '[ id ID ] [ detail ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'gribi': matcherGribi,
      'nexthop-group-aft': matcherNexthopGroupAft,
      'failed': matcherFailed,
      'id': matcherNhgId,
      'ID': CliMatcher.IntegerMatcher( 0, 0xFFFFFFFFFFFFFFFF,
                                       helpdesc='Nexthop group AFT ID' ),
      'detail': matcherDetail,
   }
   cliModel = GribiNhgAftTable

   @staticmethod
   def handler( mode, args ):
      detail = 'detail' in args
      return populateNhgAft( args.get( 'ID' ), detail, onlyFailedEntries=True )

class ShowGribiNhgAftCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering gribi nexthop-group-aft '\
      '[ acknowledged | mismatched-ack ] [ id ID ] [ detail ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'gribi': matcherGribi,
      'nexthop-group-aft': matcherNexthopGroupAft,
      'acknowledged': matcherAcknowledged,
      'mismatched-ack': matcherMismatched,
      'id': matcherNhgId,
      'ID': CliMatcher.IntegerMatcher( 0, 0xFFFFFFFFFFFFFFFF,
                                       helpdesc='Nexthop group AFT ID' ),
      'detail': matcherDetail,
   }
   cliModel = GribiNhgAftTable

   @staticmethod
   def handler( mode, args ):
      detail = 'detail' in args
      acknowledged = 'acknowledged' in args
      mismatched = 'mismatched-ack' in args
      return populateNhgAft( args.get( 'ID' ), detail, acknowledged,
                             onlyMismatchedEntries=mismatched )

class ShowGribiMplsAftCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering gribi mpls-aft [ label LABEL ] [ detail ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'gribi': matcherGribi,
      'mpls-aft': matcherMplsAft,
      'label': matcherLabel,
      'LABEL': CliMatcher.IntegerMatcher( MplsLabel.min,
                                          MplsLabel.max,
                                          helpdesc='MPLS label' ),
      'detail': matcherDetail,
   }
   cliModel = GribiMplsAftTable

   @staticmethod
   def handler( mode, args ):
      detail = 'detail' in args
      return populateMplsAft( args.get( 'LABEL' ), detail )

class ShowGribiPendingIpv4AftCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering gribi ipv4-aft unacknowledged '\
      '[ VRF ] [ PREFIX ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'gribi': matcherGribi,
      'ipv4-aft': matcherIpv4Aft,
      'unacknowledged': matcherUnacknowledged,
      'VRF': matcherVrfName,
      'PREFIX': ipPrefixMatcher,
   }

   cliModel = PendingIpAftVrfModel

   @staticmethod
   def handler( mode, args ):
      return showPendingIpv4Aft( mode, vrfName=args.get( 'VRF' ),
                                 prefix=args.get( 'PREFIX' ) )

class ShowGribiPendingIpv6AftCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering gribi ipv6-aft unacknowledged '\
      '[ VRF ] [ PREFIX6 ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'gribi': matcherGribi,
      'ipv6-aft': matcherIpv6Aft,
      'unacknowledged': matcherUnacknowledged,
      'VRF': matcherVrfName,
      'PREFIX6': ip6PrefixMatcher
   }

   cliModel = PendingIpAftVrfModel

   @staticmethod
   def handler( mode, args ):
      prefix = args.get( 'PREFIX6' )
      if prefix is not None:
         prefix = prefix.stringValue
      return showPendingIpv6Aft( mode, vrfName=args.get( 'VRF' ),
                                 prefix=prefix )

class ShowGribiFailedIpv4AftCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering gribi ipv4-aft failed '\
      '[ VRF ] [ PREFIX ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'gribi': matcherGribi,
      'ipv4-aft': matcherIpv4Aft,
      'failed': matcherFailed,
      'VRF': matcherVrfName,
      'PREFIX': ipPrefixMatcher,
   }

   cliModel = IpAftVrfModel

   @staticmethod
   def handler( mode, args ):
      return showFailedIpv4Aft( mode, vrfName=args.get( 'VRF' ),
                                prefix=args.get( 'PREFIX' ) )

class ShowGribiFailedIpv6AftCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering gribi ipv6-aft failed '\
      '[ VRF ] [ PREFIX6 ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'gribi': matcherGribi,
      'ipv6-aft': matcherIpv6Aft,
      'failed': matcherFailed,
      'VRF': matcherVrfName,
      'PREFIX6': ip6PrefixMatcher
   }

   cliModel = IpAftVrfModel

   @staticmethod
   def handler( mode, args ):
      prefix = args.get( 'PREFIX6' )
      if prefix is not None:
         prefix = prefix.stringValue
      return showFailedIpv6Aft( mode, vrfName=args.get( 'VRF' ),
                                prefix=prefix )

class ShowGribiIpv4AftCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering gribi ipv4-aft '\
      '[ acknowledged | mismatched-ack ] [ VRF ] [ PREFIX ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'gribi': matcherGribi,
      'ipv4-aft': matcherIpv4Aft,
      'acknowledged': matcherAcknowledged,
      'mismatched-ack': matcherMismatched,
      'VRF': matcherVrfName,
      'PREFIX': ipPrefixMatcher,
   }
   cliModel = IpAftVrfModel

   @staticmethod
   def handler( mode, args ):
      acknowledged = 'acknowledged' in args
      mismatched = 'mismatched-ack' in args
      return showIpv4Aft( mode, vrfName=args.get( 'VRF' ),
                          prefix=args.get( 'PREFIX' ),
                          acknowledged=acknowledged,
                          mismatched=mismatched )

class ShowGribiIpv6AftCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering gribi ipv6-aft '\
      '[ acknowledged | mismatched-ack ] [ VRF ] [ PREFIX6 ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'gribi': matcherGribi,
      'ipv6-aft': matcherIpv6Aft,
      'acknowledged': matcherAcknowledged,
      'mismatched-ack': matcherMismatched,
      'VRF': matcherVrfName,
      'PREFIX6': ip6PrefixMatcher,
   }
   cliModel = IpAftVrfModel

   @staticmethod
   def handler( mode, args ):
      acknowledged = 'acknowledged' in args
      mismatched = 'mismatched-ack' in args
      prefix = args.get( 'PREFIX6' )
      if prefix is not None:
         prefix = prefix.stringValue
      return showIpv6Aft( mode, vrfName=args.get( 'VRF' ),
                          prefix=prefix, acknowledged=acknowledged,
                          mismatched=mismatched )

class ShowGribiPendingNhAftCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering gribi nexthop-aft unacknowledged '\
      '[ id ID ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'gribi': matcherGribi,
      'nexthop-aft': matcherNexthopAft,
      'unacknowledged': matcherUnacknowledged,
      'id': matcherNhId,
      'ID': CliMatcher.IntegerMatcher( 0, 0xFFFFFFFFFFFFFFFF,
                                       helpdesc='Nexthop AFT ID' ),
   }
   cliModel = GribiPendingNhAftTable

   @staticmethod
   def handler( mode, args ):
      return populatePendingNhAft( args.get( 'ID' ) )

class ShowGribiNhAftCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering gribi nexthop-aft '\
      '[ id ID ] [ detail ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'gribi': matcherGribi,
      'nexthop-aft': matcherNexthopAft,
      'id': matcherNhId,
      'ID': CliMatcher.IntegerMatcher( 0, 0xFFFFFFFFFFFFFFFF,
                                       helpdesc='Nexthop AFT ID' ),
      'detail': matcherDetail,
   }
   cliModel = GribiNhAftTable

   @staticmethod
   def handler( mode, args ):
      detail = 'detail' in args
      return populateNhAft( args.get( 'ID' ), detail )

BasicCli.addShowCommandClass( ShowGribiPendingNhgAftCmd )
BasicCli.addShowCommandClass( ShowGribiFailedNhgAftCmd )
BasicCli.addShowCommandClass( ShowGribiNhgAftCmd )
BasicCli.addShowCommandClass( ShowGribiMplsAftCmd )
BasicCli.addShowCommandClass( ShowGribiPendingNhAftCmd )
BasicCli.addShowCommandClass( ShowGribiNhAftCmd )
BasicCli.addShowCommandClass( ShowGribiPendingIpv4AftCmd )
BasicCli.addShowCommandClass( ShowGribiFailedIpv4AftCmd )
BasicCli.addShowCommandClass( ShowGribiIpv4AftCmd )
BasicCli.addShowCommandClass( ShowGribiPendingIpv6AftCmd )
BasicCli.addShowCommandClass( ShowGribiFailedIpv6AftCmd )
BasicCli.addShowCommandClass( ShowGribiIpv6AftCmd )
# -------------------------------------------------------------------------------
# The "show management api gribi access-list" command
# -------------------------------------------------------------------------------

def showAclStatusGribi( mode, args ):

   aclName = args.get( 'ACL' )
   params = [ aclName, 'summary' in args ]
   model = GribiAclStatus()
   for name, endpoint in gribiConfig.endpoints.items():
      if ( not endpoint.enabled
           or ( endpoint.serviceAcl == '' and endpoint.serviceAclV6 == '' )
           or ( aclName and aclName not in
                [ endpoint.serviceAcl, endpoint.serviceAclV6 ] ) ):
         # Skip if an ACL is not configured for the enabled endpoint or when the
         # ACL name argument (if provided) does not correspond with an endpoint
         # ACL name.
         continue
      if not aclName:
         aclType = None
      elif aclName == endpoint.serviceAcl:
         aclType = 'ip'
      else:
         aclType = 'ipv6'
      model.transports[ name ] = AclCli.showServiceAcl( mode,
         aclCpConfig,
         aclStatus,
         aclCheckpoint,
         aclType,
         params,
         serviceName='gribi' )
   return model

# --------------------------------------------------------------------------------
# show management api gribi access-list [ ACL ] [ summary ] [ dynamic ] [ detail ]
# --------------------------------------------------------------------------------
class ManagementApiGribiAccessListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show management api gribi access-list
               [ ACL ] [ summary ] [ dynamic ] [ detail ]
            '''
   data = {
      'management': matcherManagement,
      'api': matcherApiForShow,
      'gribi': matcherGribiMgmtApi,
      'access-list': CliCommand.Node( matcher=matcherAccessList,
                                      guard=AclCli.serviceAclGuard ),
      'summary': matcherSummary,
      'dynamic': matcherDynamic,
      'detail': nodeDetail,
      'ACL': AclCli.ipAclNameMatcher,
   }
   handler = showAclStatusGribi
   cliModel = GribiAclStatus

BasicCli.addShowCommandClass( ManagementApiGribiAccessListCmd )

def populateGribiRpcStatus():
   if mgmtGribiStatus.totalClients == 0:
      return None

   rpcStatus = GribiRpcStatus()
   rpcStatus.totalClients = mgmtGribiStatus.totalClients
   rpcStatus.primaryClientId = mgmtGribiStatus.primaryClientId
   rpcStatus.primaryClientIp = mgmtGribiStatus.primaryClientIp
   rpcStatus.primaryClientPort = mgmtGribiStatus.primaryClientPort
   rpcStatus.highestElectionId = GribiElectionId()
   rpcStatus.highestElectionId.high = mgmtGribiStatus.highestElectionId.high
   rpcStatus.highestElectionId.low = mgmtGribiStatus.highestElectionId.low
   rpcStatus.highestElectionIdTimeStamp = \
         float( mgmtGribiStatus.highestElectionIdTimeStamp )

   for key, entry in gribiModifyRpcStreamStatus.entry.items():
      modifyRpcStreamStatus = GribiModifyRpcStreamStatus()
      modifyRpcStreamStatus.connectionTimeStamp = float( entry.connectionTimeStamp )
      modifyRpcStreamStatus.electionId = GribiElectionId()
      modifyRpcStreamStatus.electionId.high = entry.electionId.high
      modifyRpcStreamStatus.electionId.low = entry.electionId.low
      modifyRpcStreamStatus.electionIdTimeStamp = float( entry.electionIdTimeStamp )
      if entry.sessionParamsTimeStamp != 0:
         modifyRpcStreamStatus.clientRedundancy = gribiSessionParams.redundancy
         modifyRpcStreamStatus.aftPersistence = gribiSessionParams.persistence
         # We always apply the ACK type requested from the time we added support
         # for FIB-ACK, so populate both requested and applied w/ the same thing.
         modifyRpcStreamStatus.aftOpAckTypeRequested = gribiSessionParams.ackType
         modifyRpcStreamStatus.aftOpAckTypeApplied = gribiSessionParams.ackType
      modifyRpcStreamStatus.sessionParamsTimeStamp = \
            float( entry.sessionParamsTimeStamp )
      modifyRpcStreamStatus.lastAftOpTimeStamp = float( entry.lastAftOpTimeStamp )

      if entry.ipAddress not in rpcStatus.modifyRpcClients:
         # create Modify RPC client status
         rpcStatus.modifyRpcClients[ entry.ipAddress ] = \
               GribiModifyRpcClientIpStatus()
      clientIpStatus = rpcStatus.modifyRpcClients[ entry.ipAddress ]
      if entry.port not in clientIpStatus.ports:
         # create port status
         clientIpStatus.ports[ entry.port ] = GribiModifyRpcClientIpPortStatus()
      clientIpPortStatus = clientIpStatus.ports[ entry.port ]
      # add the stream status under the corresponding client, ip
      clientIpPortStatus.streams[ int( key ) ] = modifyRpcStreamStatus

   return rpcStatus

def populateGribiEndpointStatus():
   endpointStatus = GribiEndpointStatus()
   endpointStatus.enabled = mgmtGribiStatus.enabled
   endpointStatus.started = mgmtGribiStatus.started
   endpointStatus.port = mgmtGribiStatus.port
   endpointStatus.error = mgmtGribiStatus.error
   endpointStatus.sslProfile = mgmtGribiStatus.sslProfile
   endpointStatus.mTls = mgmtGribiStatus.mTls
   endpointStatus.authzEnabled = mgmtGribiStatus.authzEnabled
   if Toggles.gribiToggleLib.toggleGribiAUPEnabled():
      endpointStatus.authnUsernamePriority = [
            AuthnUserPriorityCli.authnUsernamePriorityToCLI[ v.source ]
               for v in mgmtGribiStatus.authnUsernamePriority.values() ]
   else:
      endpointStatus.certUsernameAuthn = mgmtGribiStatus.certUsernameAuthn
   if Toggles.gribiToggleLib.toggleGribiAccountingRequestsEnabled():
      endpointStatus.accountingRequests = mgmtGribiStatus.accountingRequests
   endpointStatus.vrfName = mgmtGribiStatus.vrfName
   endpointStatus.dscp = mgmtGribiStatus.dscp
   endpointStatus.lastServiceStartTimeStamp = \
         float( mgmtGribiStatus.lastServiceStartTimeStamp )
   endpointStatus.rpcStatus = populateGribiRpcStatus()
   for endpoint in gribiConfig.endpoints.values():
      endpointStatus.serviceAcl = endpoint.serviceAcl
      endpointStatus.serviceAclV6 = endpoint.serviceAclV6
   return endpointStatus

# --------------------------------------------------------------------------------
# show management api gribi
# --------------------------------------------------------------------------------
class ManagementApiGribiCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api gribi'
   data = {
            'management': matcherManagement,
            'api': matcherApiForShow,
            'gribi': matcherGribiMgmtApi,
         }
   cliModel = GribiEndpointStatus

   @staticmethod
   def handler( mode, args ):
      return populateGribiEndpointStatus()

BasicCli.addShowCommandClass( ManagementApiGribiCmd )


def Plugin( entityManager ):
   global gribiAftEntries
   global gribiAcknowledgedAftEntries
   global gribiPendingAckEntries
   global gribiConfig, gribiStatus
   global mgmtGribiStatus
   global gribiSessionParams
   global gribiNhToTunnelMap
   global gribiNhgIdFecIdMap
   global slTunnelTable
   global srteForwardingStatus
   global programmedFecStatus
   global aclCpConfig, aclStatus, aclCheckpoint
   global gribiModifyRpcStreamStatus
   global smashEm
   global readerInfo

   gribiAftEntries = LazyMount.mount( entityManager,
                                      "routing/gribi/afts",
                                      "Gribi::GribiAfts", "r" )
   gribiAcknowledgedAftEntries = LazyMount.mount( entityManager,
                                      "routing/gribi/ackedafts",
                                      "Gribi::GribiAfts", "r" )
   gribiPendingAckEntries = LazyMount.mount( entityManager,
                                      "mgmt/gribi/pendingAcks",
                                      "Gribi::PendingAcks", "r" )
   gribiConfig = LazyMount.mount( entityManager, "mgmt/gribi/config",
                                    "Gribi::Config", "r" )
   gribiStatus = LazyMount.mount( entityManager,
                                  "routing/gribi/status",
                                  "Gribi::GribiStatus", "r" )
   mgmtGribiStatus = LazyMount.mount( entityManager,
                                  "mgmt/gribi/status",
                                  "Gribi::Status", "r" )
   gribiSessionParams = LazyMount.mount( entityManager,
                                         "mgmt/gribi/sessionParams",
                                         "Gribi::SessionParams", "r" )
   gribiNhToTunnelMap = LazyMount.mount( entityManager,
                                         "routing/gribi/nhtotunnmap",
                                         "Gribi::NhToTunnelMap", "r" )
   gribiNhgIdFecIdMap = LazyMount.mount( entityManager,
                                    "routing/gribi/nhgidfecidmap",
                                    "Gribi::NhgIdFecIdMap", "r" )
   gribiModifyRpcStreamStatus = LazyMount.mount( entityManager,
                                    "mgmt/gribi/modifyRpcStreamStatus",
                                    "Gribi::ModifyRpcStreamStatus", "r" )
   aclCpConfig = LazyMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "r" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                    "Acl::CheckpointStatus", "r" )

   slTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.srTeSegmentListTunnelTable, entityManager )

   smashEm = SharedMem.entityManager( sysdbEm=entityManager )
   readerInfo = Smash.mountInfo( 'reader' )
   # pkgdeps: library FibLib
   srteForwardingStatus = smashEm.doMount(
      "forwarding/srte/status", "Smash::Fib::ForwardingStatus", readerInfo )
   programmedFecStatus = smashEm.doMount(
      "forwarding/fecVersion", "Smash::Fib::ProgrammedFecStatus", readerInfo )
