# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliToken.Hardware import hardwareMatcherForShow
from CliPlugin import HardwareModels, TechSupportCli
import ShowCommand
import LazyMount
import BasicCli
import Tracing

import re

__defaultTraceHandle__ = Tracing.Handle( 'HardwareCli' )

inventory = None

def populateComponent( component, entity, isDetail=False, secondaryPrefdl=False ):
   def cull( val ):
      # Used to determine if variable value is a byte string. These are
      # visually uninterpretable so we only show their variable names.
      cullHeuristic = 100

      cull = isinstance( val, str ) and len( val ) > cullHeuristic
      return '<truncated>' if cull and not isDetail else val


   def entityToDict( entity ):
      return { key : cull( val.val ) for key, val in entity.items() }

   def isPrototypeHw( asy ):
      if not asy:
         return False
      return asy[ -2 : ].isnumeric()

   prefdl = entity.prefdl
   if secondaryPrefdl:
      prefdl = entity.prefdl2

   component.dataFormat = prefdl.dataFormat
   component.pca = prefdl.pca
   component.kvn = prefdl.kvn
   component.serialNumber = prefdl.serialNumber
   component.sku = prefdl.sku
   component.hwApi = prefdl.hwApi
   component.sid = prefdl.sid
   component.hwRev = prefdl.hwRev
   component.deviation = list( prefdl.deviation )
   component.fdlVariables = entityToDict( prefdl.fdlVariable )
   component.mfgTime = prefdl.mfgTime
   component.mfgTime2 = prefdl.mfgTime2
   component.asy = prefdl.asy
   component.crc = prefdl.crc
   component.cpuPca = prefdl.cpuPca
   component.cpuSerialNumber = prefdl.cpuSerialNumber
   component.cpuKvn = prefdl.cpuKvn
   component.cpuSku = prefdl.cpuSku
   component.cpuSid = prefdl.cpuSid
   component.cpuHwApi = prefdl.cpuHwApi
   component.cpuFdlVariables = entityToDict( prefdl.cpuFdlVariable )
   component.cpuMfgTime = prefdl.cpuMfgTime
   component.ztpToken = cull( prefdl.ztpToken )
   component.certificate = cull( prefdl.certificate )
   component.signature = cull( prefdl.signature )
   component.signatureList = list( prefdl.signatureList )
   component.other = list( prefdl.other )

   # macAddrBase is of type ArnetModel.MacAddress which can't be a string
   component.macAddrBase = prefdl.macAddrBase or None

   # Prefdl field that already existed in Inventory::FixedSystem.
   component.hwEpoch = entity.hwEpoch

   # Last two chars of ASY ( revision ) indicate production vs prototype
   component.prototype = isPrototypeHw( component.asy )

class ShowPrefdl( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hardware eeprom [ detail ]'
   data = {
      'hardware': hardwareMatcherForShow,
      'eeprom': 'Show EEPROM contents',
      'detail': 'Show values untruncated',
   }

   cliModel = HardwareModels.Hardware

   @staticmethod
   def handler( mode, args ):
      def formatName( sliceId ):
         # Card slice ids don't have spaces (e.g. Linecard8).
         # Here we add the space (e.g. Linecard 8)
         return re.sub( r'([0-9]+)', r' \1', sliceId )

      def formatNameOffset( sliceId, offset ):
         slot = int( re.findall( r'([0-9]+)', sliceId )[ 0 ] ) + offset
         return re.sub( r'([0-9]+)', " " + str( slot ), sliceId )

      isDetail = 'detail' in args
      hardware = HardwareModels.Hardware()
      fixedSystem = inventory.get( 'fixedSystem', None )

      # Populate fixed tac model prefdl
      if fixedSystem:
         fs = HardwareModels.Component()
         populateComponent( fs, fixedSystem, isDetail=isDetail )
         hardware.components[ 'System' ] = fs

      # Populate modular tac model prefdl
      else:
         modularSystem = inventory.get( 'modularSystem', None )
         if not modularSystem:
            return hardware

         # Standby supervisors may not have chassis populated
         if modularSystem.chassis:
            chassis = HardwareModels.Component()
            populateComponent( chassis, modularSystem.chassis, isDetail=isDetail )
            hardware.components[ 'Chassis' ] = chassis

         for cardEntity in modularSystem.card.values():
            if cardEntity.prefdl:
               card = HardwareModels.Component()
               populateComponent( card, cardEntity, isDetail=isDetail )
               if cardEntity.prefdl2IsPrimary and cardEntity.prefdl2:
                  name = formatNameOffset( cardEntity.sliceId,
                     cardEntity.numPrimaryCards )
               else:
                  name = formatName( cardEntity.sliceId )
               hardware.components[ name ] = card

            if cardEntity.prefdl2:
               card = HardwareModels.Component()
               populateComponent( card, cardEntity, isDetail=isDetail,
                                  secondaryPrefdl=True )
               if cardEntity.prefdl2IsPrimary:
                  name = formatName( cardEntity.sliceId )
               else:
                  name = formatNameOffset( cardEntity.sliceId,
                     cardEntity.numPrimaryCards )
               hardware.components[ name ] = card

      return hardware

TechSupportCli.registerShowTechSupportCmd(
   '2024-07-24 16:35:00',
   cmds=[ 'show hardware eeprom detail' ] )

def Plugin( entityManager ):
   global inventory
   # Need immediate mount so we can check whether modularSystem dir or
   # fixedSystem dir exists
   inventory = LazyMount.mount( entityManager, 'hardware/inventory',
                                'Tac::Dir', 'ri' )

BasicCli.addShowCommandClass( ShowPrefdl )

