# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import natsort

import ArnetModel
import CliModel

dataFormatHelp = ( 'First four bytes of the PreFDL. This self-describes the PreFDL '
                   'format' )
pcaHelp = 'Printed circuit assembly number'
serialNumberHelp = 'Board serial number'
kvnHelp = ( 'The KVN verification field is no longer used. Because it is a '
            'required field in formats 0000, 0001, and 0002, the default value '
            '111 is stored as a placeholder' )
skuHelp = 'Software-display SKU' 
hwApiHelp = 'Hardware API'
sidHelp = ( 'System ID. Used by run-time software (along with the HwAPI) to '
            'identify the system and choose a FDL' )
macAddrBaseHelp = 'The first MAC address allocated to the board'
hwRevHelp = ( 'Hardware revision. This field is used to display the hardware '
              'revision to the user' )
hwEpochHelp = 'Hardware epoch'
deviationHelp = ( 'Defines an action that was taken that was not part of the'
                  'original build instructions' )
fdlVariablesHelp = ( 'Python code (a dictionary) allowing for arbitrary variables '
                     'to be set' )
mfgTimeHelp = 'When the PreFDL was generated'
mfgTime2Help = "The system clock when the PreFDL was generated, deprecates MfgTime"
asyHelp = 'Assembly number'
crcHelp = 'Last four bytes of the PreFDL. This is a CRC-32 checksum'
cpuPcaHelp = 'PCA for CPU card'
cpuSerialNumberHelp = 'Serial Number for CPU card'
cpuKvnHelp = 'KVN for CPU card'
cpuSkuHelp = 'SKU for CPU card'
cpuSidHelp = 'SID for CPU card'
cpuHwApiHelp = 'HwApi for CPU card'
cpuFdlVariablesHelp = 'FDL variables for CPU card'
cpuMfgTimeHelp = 'MfgTime for CPU card'
ztpTokenHelp = 'JWT token to validate the switch in SecureZTP'
certificateHelp = 'ASCII readable PEM format (Base64 encoded) certificate'
signatureHelp = 'ASCII readable PEM format (Base64 encoded) signature'
signatureListHelp = ( 'PreFDL values that should be used in validating the '
                      'signature' )
prototypeHelp = 'This is prototype hardware'
otherHelp = 'Present fields which are uncommon'
componentsHelp = 'Mapping of component names to their PreFDL fields'
printOrderHelp = 'Order in which components will be printed'

class Component( CliModel.Model ):
   dataFormat = CliModel.Str( help=dataFormatHelp, optional=True)
   pca = CliModel.Str( help=pcaHelp, optional=True )
   serialNumber = CliModel.Str( help=serialNumberHelp, optional=True )
   kvn = CliModel.Str( help=kvnHelp, optional=True )
   sku = CliModel.Str( help=skuHelp, optional=True )
   hwApi = CliModel.Str( help=hwApiHelp, optional=True )
   sid = CliModel.Str( help=sidHelp, optional=True )
   macAddrBase = ArnetModel.MacAddress( help=macAddrBaseHelp, optional=True )
   hwRev = CliModel.Str( help=hwRevHelp, optional=True )
   hwEpoch = CliModel.Str( help=hwEpochHelp, optional=True )
   deviation = CliModel.List( valueType=str, help=deviationHelp, optional=True )
   fdlVariables = CliModel.Dict( valueType=str, help=fdlVariablesHelp, 
                                 optional=True )
   mfgTime = CliModel.Str( help=mfgTimeHelp, optional=True )
   mfgTime2 = CliModel.Str( help=mfgTime2Help, optional=True )
   asy = CliModel.Str( help=asyHelp, optional=True )
   crc = CliModel.Str( help=crcHelp, optional=True )
   cpuPca = CliModel.Str( help=cpuPcaHelp, optional=True )
   cpuSerialNumber = CliModel.Str( help=cpuSerialNumberHelp, optional=True )
   cpuKvn = CliModel.Str( help=cpuKvnHelp, optional=True )
   cpuSku = CliModel.Str( help=cpuSkuHelp, optional=True )
   cpuSid = CliModel.Str( help=cpuSidHelp, optional=True )
   cpuHwApi = CliModel.Str( help=cpuHwApiHelp, optional=True )
   cpuFdlVariables = CliModel.Dict( valueType=str, help=cpuFdlVariablesHelp,
                                    optional=True )
   cpuMfgTime = CliModel.Str( help=cpuMfgTimeHelp, optional=True )
   ztpToken = CliModel.Str( help=ztpTokenHelp, optional=True )
   certificate = CliModel.Str( help=certificateHelp, optional=True )
   signature = CliModel.Str( help=signatureHelp, optional=True )
   signatureList = CliModel.List( valueType=str, help=signatureListHelp,
                                  optional=True )
   prototype = CliModel.Bool( help=prototypeHelp, optional=True )
   other = CliModel.List( valueType=str, help=otherHelp, optional=True )

class Hardware( CliModel.Model ):

   components = CliModel.Dict( valueType=Component, help=componentsHelp )

   def render( self ):
      componentOutput = ( '%s\n\n'
                          'DataFormat: %s\n'
                          'PCA: %s\n'
                          'SerialNumber: %s\n'
                          'KVN: %s\n'
                          'SKU: %s\n'
                          'HwApi: %s\n'
                          'SID: %s\n'
                          'MacAddrBase: %s\n'
                          'HwRev: %s\n'
                          'HwEpoch: %s\n'
                          'Deviation: %s\n'
                          'FdlVariables: \n%s'
                          'MfgTime: %s\n'
                          'MfgTime2: %s\n'
                          'ASY: %s\n'
                          'Crc: %s\n'
                          '%s'
                          'ZtpToken: %s\n'
                          'Certificate: %s\n'
                          'Signature: %s\n'
                          'SignatureList: %s\n'
                          'Prototype: %s\n'
                          'Other: %s\n')

      # Cards don't have cpu boards so we can omit these fields for them
      componentCpuOutput = ( 'CpuPCA: %s\n'
                             'CpuSerialNumber: %s\n'
                             'CpuKVN: %s\n'
                             'CpuSKU: %s\n'
                             'CpuSID: %s\n'
                             'CpuHwApi: %s\n'
                             'CpuFdlVariables: \n%s'
                             'CpuMfgTime: %s\n' )

      numLeadingSpaces = 8
      colonSpacePadding = 2
      leadingSpaces = ' ' * numLeadingSpaces

      def variablesToStr( variables ):
         resStr = []
         otherVariables = []

         # Consistent variable ordering, and makes this easier to test
         keyVals = sorted( variables.items() )

         for key, val in keyVals:
            padding =  leadingSpaces + ( ' ' * ( len( key ) + colonSpacePadding ) )
            lines = val.strip().split( '\n' )
            lines = [ lines[ 0 ] ] + [ padding + line for line in lines[ 1: ] ]
            val = '\n'.join( lines )
            resStr.append( leadingSpaces + f'{key}: {val}\n' )
         resStr += otherVariables
         return ''.join( resStr )

      def formatToken( name, token ):
         paddingLen = len( name ) + colonSpacePadding
         charOffset = 80 - paddingLen
         lines = [ token[ i:i + charOffset ]
                   for i in range( 0, len( token ), charOffset ) ]
         if lines:
            lines = ( [ lines[0] ] +
                      [ ' ' * paddingLen + line for line in lines[ 1: ] ] )
         return '\n'.join( lines )

      def sortKey( component ):
         return ( component == 'Chassis', component == 'System',
                  component[ 0 ] )
      # Expected order: Chassis, System, Supervisor 1, Supervisor 2, Supervisor 10,
      # Linecard1, Linecard2, Linecard10, Fabric1, Fabric2, Fabric10
      printOrder = sorted( natsort.natsorted( list( self.components ) ), key=sortKey,
                           reverse=True )

      output = []
      for name in printOrder:
         component = self.components[ name ]

         # Format dictionaries and lists 
         fdlVariableStr = variablesToStr( component.fdlVariables )
         cpuFdlVariableStr = variablesToStr( component.cpuFdlVariables )
         other = ''.join( 
               f'\n{leadingSpaces}{field}' 
               for field in sorted( component.other ) )
         deviation = ', '.join( sorted( component.deviation ) )
         signatureList = ', '.join( sorted( component.signatureList ) )

         ztpToken = formatToken( "ZtpToken", component.ztpToken )
         certificate = formatToken( "Certificate", component.certificate )
         signature = formatToken( "Signature", component.signature )

         # If times are 0 then default to empty string
         mfgTime = str( component.mfgTime ) if component.mfgTime else ''
         mfgTime2 = str( component.mfgTime2 ) if component.mfgTime2 else ''
         cpuMfgTime = str( component.cpuMfgTime ) if component.cpuMfgTime else ''

         # True/False values must be lowercase on CLI
         prototype = str( component.prototype ).lower()

         # In case macAddrBase doesn't exist and is therefore None
         macAddrBase = ( component.macAddrBase.displayString 
                         if component.macAddrBase else '' )

         isNotCard = ( name in ( 'Chassis', 'System' ) 
                       or name.startswith( 'Supervisor' ) )

         cpuOutput = componentCpuOutput if isNotCard else ''
         outputVals = ( name, component.dataFormat, component.pca, 
                        component.serialNumber, component.kvn, component.sku,
                        component.hwApi, component.sid, macAddrBase,
                        component.hwRev, component.hwEpoch, deviation,
                        fdlVariableStr, mfgTime, mfgTime2, component.asy,
                        component.crc, cpuOutput, ztpToken, certificate, signature,
                        signatureList, prototype, other )
         outputStr = componentOutput % outputVals

         # If not card, add cpu fields to output
         if isNotCard:
            cpuVals = ( component.cpuPca, component.cpuSerialNumber,
                        component.cpuKvn, component.cpuSku, component.cpuSid,
                        component.cpuHwApi, cpuFdlVariableStr, cpuMfgTime )
            outputStr %= cpuVals
         output.append( outputStr )

      print( '\n'.join( output ) )

