# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=consider-using-from-import
import Tac
import ConfigMount
import BasicCli
import AgentCommandRequest 
import AgentDirectory
import json
from CliToken.Ip import ipMatcherForShow, ipv4MatcherForShow, ipMatcherForConfigIf
from CliToken.Ipv6 import ipv6MatcherForShow, ipv6MatcherForConfigIf
from CliToken.McastCommon import multicastMatcherForShow, sourceMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.VlanCli as VlanCli
import CliPlugin.VirtualIntfRule as VirtualIntfRule
from CliPlugin import IntfCli
from CliPlugin.HostInjectCliModels import AttachedHostModel
from CliPlugin.IraIpCli import vrfKwMatcher, vrfNameMatcher
from CliPlugin.IraCommonCli import preferenceRangeMatcher
from CliPlugin.VlanIntfCli import VlanIntfModelet as modelet
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.LagIntfCli as LagIntfCli
import HostInjectAgent
import CliCommand, ShowCommand
from CliMatcher import KeywordMatcher, IntegerMatcher
from io import StringIO
import Toggles.HostInjectToggleLib

enableHostInjectPortChannelandEthernet = \
      Toggles.HostInjectToggleLib.toggleEthernetAndPortchannelHostInjectEnabled()

hostConfigDir = None
host6ConfigDir = None

defaultPref = Tac.Value( "HostInject::Constants" ).defaultPref
defaultV4Prefix = '0.0.0.0/0'
defaultV6Prefix = '::/0'
defaultV4Length = 32
defaultV6Length = 128

#common tokens
attachedHostKwMatcher = KeywordMatcher( 'attached-host',
                                         helpdesc='ARP generated host routes')
routeKwMatcher = KeywordMatcher( 'route', helpdesc='Export routes') 
exportKwMatcher = KeywordMatcher( 'export', helpdesc='Export' )
vlanKwMatcher = KeywordMatcher( "vlan", helpdesc="On VLAN" )
matchKwMatcher = KeywordMatcher( 'match', helpdesc='Match neighbor entry' )
prefixKwMatcher = KeywordMatcher( 'prefix', helpdesc="Match to prefix" )
prefixLengthKwMatcher = KeywordMatcher( 'prefix-length',
                                        helpdesc="Length of generated host routes" )
prefixLengthMatcher = IntegerMatcher( 0, 128,
      helpdesc="Length of the prefix in bits" )
interfaceMatcher = VirtualIntfRule.IntfMatcher()

# Please Note that EthPhyIntf.ethMatcher has a prior problem ,
# when you try to do the autocomplete in namespace Duts it says
# unrecognized command as shown below .
# This is a known issue and isnt observed in phyical Duts hence not a priority
# This issue is observed everywhere ethMatcher is used
# Example - ( Namespace Duts only issue )
# rtr7#show ip attached-host route export Ethernet ?
# % Unrecognized command
interfaceMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
interfaceMatcher |= LagIntfCli.EthLagIntf.matcher

def deleteAttachedHost( intfName, prefix, ipv4=False ):
   assert ( intfName and prefix )
   hostConfig = hostConfigDir if ipv4 else host6ConfigDir
   prefix = Tac.newInstance( "Arnet::IpGenPrefix", prefix )
   configKey = Tac.newInstance( "HostInject::ConfigKey", intfName, prefix )
   if configKey in hostConfig.config:
      del hostConfig.config[ configKey ]

def setAttachedHost( mode, hostConfig, prefix, pref, length ):
   assert ( prefix and pref and length )
   intfName =  mode.intf.name
   prefix = Tac.newInstance( "Arnet::IpGenPrefix", prefix )
   configKey = Tac.newInstance( "HostInject::ConfigKey", intfName, prefix )

   if prefix.len > length:
      mode.addError( "Configured prefix-length %d can not be lesser than the length"
                     " of configured match prefix %s" %
                     ( length, prefix.stringValue ) )
      return False
   if configKey not in hostConfig.config:
      for cKey in hostConfig.config:
         if cKey.intfId == intfName:
            if cKey.prefix.isDefaultRoutePrefix or prefix.isDefaultRoutePrefix:
               continue
            if cKey.prefix.overlaps( prefix ):
               mode.addError( "Match prefix %s can not be configured in presence of"
                              " an overlapping match prefix %s" %
                              ( prefix.stringValue, cKey.prefix.stringValue ) )
               return False
      hostConfig.config.newMember( configKey, length )
   else:
      hostConfig.config[ configKey ].length = length
   # We should change the preference attribute to be a part of the constructor
   # args in host inject config tac model.
   hostConfig.config[ configKey ].pref = pref
   return True

class ipAttachedHost( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      intf = self.intf_.name
      # Delete all the configs under this interface
      for cKey in hostConfigDir.config:
         if cKey.intfId == intf:
            del hostConfigDir.config[ cKey ]
      for cKey in host6ConfigDir.config:
         if cKey.intfId == intf:
            del host6ConfigDir.config[ cKey ]

#-------------------------------------------------------------------------------
# The '[no|default] ip attached-host route export <pref>'
#-------------------------------------------------------------------------------
class V4AttachedHostCmd( CliCommand.CliCommandClass ):
   syntax = "ip attached-host route export [ PREF ]"
   noOrDefaultSyntax = "ip attached-host route export ..."
   data = {
       "ip": ipMatcherForConfigIf,
       "attached-host": attachedHostKwMatcher,
       "route": routeKwMatcher,
       "export": exportKwMatcher,
       "PREF": preferenceRangeMatcher
   }

   @staticmethod
   def handler( mode, args ):
      pref = args.get( "PREF", defaultPref )
      setAttachedHost( mode, hostConfigDir, prefix=defaultV4Prefix, pref=pref,
                       length=defaultV4Length )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfName = mode.intf.name
      deleteAttachedHost( intfName, prefix=defaultV4Prefix, ipv4=True )

modelet.addCommandClass( V4AttachedHostCmd )

if enableHostInjectPortChannelandEthernet:
   EthIntfCli.EthIntfModelet.addCommandClass( V4AttachedHostCmd )
   LagIntfCli.LagIntfConfigModelet.addCommandClass( V4AttachedHostCmd )

#-------------------------------------------------------------------------------
# The '[no|default] ipv6 attached-host route [ match prefix < prefix >]
#                        export [ < pref > ] [ prefix-length < len > ]'
#-------------------------------------------------------------------------------
class V6AttachedHostCmd( CliCommand.CliCommandClass ):
   syntax = """ipv6 attached-host route [ match prefix V6PREFIX ] export 
               [ PREF ] [ prefix-length PREFIXLEN ]"""
   noOrDefaultSyntax = """ipv6 attached-host route [ match prefix V6PREFIX ]
                          export ..."""
   data = {
       "ipv6": ipv6MatcherForConfigIf,
       "attached-host": attachedHostKwMatcher,
       "route": routeKwMatcher,
       "match": matchKwMatcher,
       "prefix": prefixKwMatcher,
       "V6PREFIX": Ip6AddrMatcher.Ip6PrefixMatcher(
           "IPv6 address prefix", overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
       "export": exportKwMatcher,
       "PREF": preferenceRangeMatcher,
       "prefix-length": prefixLengthKwMatcher,
       "PREFIXLEN": prefixLengthMatcher
   }

   @staticmethod
   def handler( mode, args ):
      prefix = str( args.get( "V6PREFIX", defaultV6Prefix ) )
      prefixLen = args.get( "PREFIXLEN", defaultV6Length )
      pref = args.get( "PREF", defaultPref )
      setAttachedHost( mode, host6ConfigDir, prefix=prefix, pref=pref,
                       length=prefixLen )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfName = mode.intf.name
      prefix = str( args.get( "V6PREFIX", defaultV6Prefix ) )
      deleteAttachedHost( intfName, prefix, ipv4=False )

modelet.addCommandClass( V6AttachedHostCmd )

if enableHostInjectPortChannelandEthernet:
   EthIntfCli.EthIntfModelet.addCommandClass( V6AttachedHostCmd )
   LagIntfCli.LagIntfConfigModelet.addCommandClass( V6AttachedHostCmd )


def doShowAttachedHosts( mode, vlan=None, ipv4=True, vrf=None, multicast=False ):
   output = []
   if AgentDirectory.agent( mode.sysname, "HostInject" ):
      buff = StringIO()
      command = "ATHS"
      if ipv4:
         command += "#v4"
      else:
         command += "#v6"
      if multicast:
         command += "#multicast"
      if vlan is not None:
         command += "#vlan#%s" % vlan
      if vrf is not None:
         command += "#vrf#%s" % vrf
      AgentCommandRequest.runSocketCommand( mode.entityManager,
                                            HostInjectAgent.name, "hostInject",
                                            command, stringBuff=buff )
      result = buff.getvalue()
      buff.close()
      result = json.loads( result )
      output = result[ 'output' ]
   return output 

def doShowAttachedHostsForInterface( mode,
      intfId=None, ipv4=True, vrf=None, multicast=False ):
   output = []
   if AgentDirectory.agent( mode.sysname, "HostInject" ):
      buff = StringIO()
      command = "ATHS"
      if ipv4:
         command += "#v4"
      else:
         command += "#v6"
      if multicast:
         command += "#multicast"
      if intfId is not None:
         command += "#interface#%s" % intfId
      if vrf is not None:
         command += "#vrf#%s" % vrf
      AgentCommandRequest.runSocketCommand( mode.entityManager,
                                            HostInjectAgent.name, "hostInject",
                                            command, stringBuff=buff )
      result = buff.getvalue()
      buff.close()
      result = json.loads( result )
      output = result[ 'output' ]
   return output

#-----------------------------------------------------------------------------------
#  "show ip[v6] attached-hosts route export [ vrf < vrfName > ] [ vlan < vlanId > ]"
#-----------------------------------------------------------------------------------
class ShowUnicastAttachedHosts( ShowCommand.ShowCliCommandClass ):
   syntax = """show ( ip | ipv6 ) attached-hosts route export [ vrf VRF ]
               [ vlan VLAN ]"""
   data = {
       "ip": ipMatcherForShow,
       "ipv6": ipv6MatcherForShow,
       "attached-hosts": attachedHostKwMatcher,
       "route": routeKwMatcher,
       "export": exportKwMatcher,
       "vrf": vrfKwMatcher,
       "VRF": vrfNameMatcher,
       "vlan": vlanKwMatcher,
       "VLAN": VlanCli.vlanIdMatcher
   }
   cliModel = AttachedHostModel

   @staticmethod
   def handler( mode, args ):
      vlanId = args.get( "VLAN" ).id if "VLAN" in args else None
      output = doShowAttachedHosts( mode, vlan=vlanId, ipv4="ip" in args,
                                    vrf=args.get( "VRF" ) )
      model = AttachedHostModel()
      model.setAttachedHosts( output )
      return model

# -----------------------------------------------------------------------------------
#  "show ip[v6] attached-hosts route export [ vrf < vrfName > ] [ (vlan < vlanId > )
#   | (  INTF ) ]"
# -----------------------------------------------------------------------------------
class ShowUnicastAttachedHostsForInterface( ShowCommand.ShowCliCommandClass ):
   syntax = """show ( ip | ipv6 ) attached-hosts route export [ vrf VRF ]
               [ ( vlan VLAN ) | ( INTF ) ]"""
   data = {
       "ip": ipMatcherForShow,
       "ipv6": ipv6MatcherForShow,
       "attached-hosts": attachedHostKwMatcher,
       "route": routeKwMatcher,
       "export": exportKwMatcher,
       "vrf": vrfKwMatcher,
       "VRF": vrfNameMatcher,
       "vlan": vlanKwMatcher,
       "VLAN": VlanCli.vlanIdMatcher,
       'INTF': interfaceMatcher
   }
   cliModel = AttachedHostModel

   @staticmethod
   def handler( mode, args ):
      if "VLAN" in args:
         vlanId = args.get( "VLAN" ).id
         intfId = "Vlan" + str( vlanId )
      else:
         intfId = args.get( "INTF" )
      output = doShowAttachedHostsForInterface( mode, intfId=intfId,
            ipv4="ip" in args, vrf=args.get( "VRF" ) )
      model = AttachedHostModel()
      model.setAttachedHosts( output )
      return model

if enableHostInjectPortChannelandEthernet:
   BasicCli.addShowCommandClass( ShowUnicastAttachedHostsForInterface )
else:
   BasicCli.addShowCommandClass( ShowUnicastAttachedHosts )
#-----------------------------------------------------------------------------------
#  show multicast [ipv4|ipv6] [vrf <vrfName>] source route export [ vlan <vlanId> ]
#-----------------------------------------------------------------------------------
class ShowMulticastAttachedHosts( ShowCommand.ShowCliCommandClass ):
   syntax = """show multicast ( ipv4 | ipv6 ) [ vrf VRF ] source route export 
   [ vlan VLAN ]"""
   data = {
       "multicast": multicastMatcherForShow,
       "ipv4": ipv4MatcherForShow,
       "ipv6": ipv6MatcherForShow,
       "vrf": vrfKwMatcher,
       "VRF": vrfNameMatcher,
       "source": sourceMatcher,
       "route": routeKwMatcher,
       "export": exportKwMatcher,
       "vlan": vlanKwMatcher,
       "VLAN": VlanCli.vlanIdMatcher
   }

   cliModel = AttachedHostModel

   @staticmethod
   def handler( mode, args ):
      vlanId = args.get( "VLAN" ).id if "VLAN" in args else None
      output = doShowAttachedHosts( mode, vlan=vlanId, ipv4="ipv4" in args,
                                    vrf=args.get( "VRF" ), multicast=True )
      model = AttachedHostModel()
      model.setAttachedHosts( output )
      return model

BasicCli.addShowCommandClass( ShowMulticastAttachedHosts )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global hostConfigDir
   global host6ConfigDir

   hostConfigDir = ConfigMount.mount( entityManager, 
                                      "routing/attached-host/config", 
                                      "HostInject::HostInjectConfig", "w" )
   host6ConfigDir = ConfigMount.mount( entityManager,
                                       "routing6/attached-host/config",
                                       "HostInject::HostInjectConfig", "w" )
   IntfCli.Intf.registerDependentClass( ipAttachedHost )
