# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Intf
import Arnet, Tac
from CliModel import Dict, List, Model, Int, Str
from ArnetModel import Ip4Address
from ArnetModel import IpGenericAddress
from IntfModels import Interface
from datetime import timedelta
# pylint: disable-next=ungrouped-imports
from CliModel import Bool, Enum, Float, Submodel, DeferredModel

class AclInfo( Model ):
   sequence = Int( help="The sequence number for an ACL on an interface", 
         optional=True )
   sourceGroupInfo = Str( help="source and group addresses for this "
         "ACL or an error message" )

class IgmpAclInfo( Model ):
   errorMessage = Str( help="error message for improperly configured ACL",
         optional=True )
   aclDetails = List( valueType=AclInfo, 
         help="list of objects containing detailed information about ACLs" )
  

   def renderAclInfo( self, aclName ):
      if self.errorMessage:
         print( "\t" + self.errorMessage )

      if self.aclDetails:     
         print( "\tacl %s: %d pieces of info" % ( aclName,
      len( self.aclDetails ) ) )
         print( "\tacl %s" % aclName )
         for line in self.getFormat():
            print( '\t' + line )

      
   def getFormat( self ):
      ret = []
      for aclInfo in self.aclDetails:
         if aclInfo.sequence != None: # pylint: disable=singleton-comparison
            ret.append( "\tSeq no %d: %s" % 
                  ( aclInfo.sequence, aclInfo.sourceGroupInfo ) )
         else:
            ret.append( '\t' + aclInfo.sourceGroupInfo )
      return ret

class InterfaceAddrs( Model ):
   sourceAddr = Ip4Address( help="source address for group interface" )
   groupAddr = Ip4Address( help="group address for group interface" )

   def render( self ):
      print( '\t  ( %s, %s ) ' % ( self.sourceAddr, 
                                  self.groupAddr ) )

class IgmpStaticGroupsInterface( Model ):
   groupAddrsList = List( valueType=InterfaceAddrs,
                        help="list of source-group addr pairs for this interface" )
   aclInfo = Dict( keyType=str, valueType=IgmpAclInfo, 
                   help="map from ACL names to ACL info objects", optional=True )

   def render( self ):
      if self.aclInfo:
         for aclName in sorted( self.aclInfo ):
            self.aclInfo[ aclName ].renderAclInfo( aclName )
      print( '\tManually configured groups:' )
      for addrs in self.groupAddrsList:
         addrs.render()

class IgmpStaticGroups( Model ):
   intfAddrs = Dict( keyType=Interface, valueType=IgmpStaticGroupsInterface,
                      help="Map interface name to list of source-group addrs pairs" )

   def render( self ):
      # loop through interfaces in dictionary
      for intf in Arnet.sortIntf( self.intfAddrs ):
         print( "Interface %s:" % intf )
         self.intfAddrs[ intf ].render()

class IgmpInterfacesList( Model ):
   interfaces = List( valueType=Interface, help="List of interface names")

class IgmpSourceGroups ( Model ):
   sourceGroups = Dict( keyType=Ip4Address, valueType=IgmpInterfacesList,
                        help="Map from sourceGroup addr to interface names" )
   def render( self ):
      for groupName in sorted( self.sourceGroups ):
         print( "Source Group: %s" % groupName )
         intfList = self.sourceGroups[ groupName ].interfaces
         printIntfs = Intf.IntfRange.intfListToCanonical( intfList, strLimit=60 )
         for intfName in printIntfs:
            print( "\tInterfaces: %s" % intfName )

# Various Text Wdith for Rendering
groupAddressWidth = 17
intfWidth = 25
uptimeWidth = 10
expiresWidth = 10
lastReporterWidth = 20
includeSrcWidth = 20
excludeSrcWidth = 20

class IgmpGroupSource( Model ):
   ''' CAPI model for Source in IgmpGroup '''
   uptime = Float( 
         help="The timestamp in UTC at which this source last became active." )
   expires = Float( 
         help="The timestamp in UTC after which this source expires "
         "if there is no new activity.", optional=True )
   def render( self ):
      print( '    Uptime: %s' % str( timedelta( 
         seconds=int( Tac.utcNow() - self.uptime ) ) ) )
      print( '    Expires: %s' % ( str( timedelta( 
         seconds=int( self.expires - Tac.utcNow() ) ) )
         if self.expires else "never" ) )

class IgmpGroups( DeferredModel ):
   ''' CAPI model for show ip igmp groups '''
   _detail = Bool( help='Include detail attributes', default=False )
   class IgmpGroup( Model ):
      ''' CAPI model for IgmpGroup '''
      interfaceName = Interface( help='Interface on which this group is joined' )
      groupAddress = Ip4Address( help='Group Address of interest' )
      interfaceAddress = Ip4Address( help='Interface Address', optional=True )
      uptime = Float( 
            help="The timestamp in UTC at which this group last became active." )
      expires = Float( help="The timestamp in UTC after which this group expires "
                            "if there is no new activity.", optional=True )
      lastReporter = Ip4Address( 
            help="The IP Address of the latest host who joined this group " )
      filterMode = Enum( help="Group Mode for IGMP",
               values=( "filterModeInclude", "filterModeExclude" ) )

      class IgmpGroupDetail( Model ):
         ''' CAPI model for IgmpGroupDetail '''
         # We only need the attributes that we don't have in IgmpGroup
         interfaceAddress = Ip4Address( help="Interface IP Address" )
         sources = Dict( keyType=Ip4Address, valueType=IgmpGroupSource,
               help="Map of Source Addresses for this group" )

      groupDetail = Submodel( valueType=IgmpGroupDetail, 
            help="Optional details about this IgmpGroup", optional=True )

      def render( self ): 
         # pylint: disable-next=superfluous-parens
         if ( self.filterMode == "filterModeExclude" ):
            filterMode = 'exclude'
         else: 
            filterMode = 'include'

         if self.groupDetail:
            print( 'Interface: %s (%s)' % ( self.interfaceName.stringValue, 
                  self.groupDetail.interfaceAddress ) )
            print( '  Group: %s' % self.groupAddress )
            print( '  Uptime: %s' % str( timedelta( 
               seconds=int( Tac.utcNow()  - self.uptime ) ) ) )
            print( '  Expires: %s' % ( str( timedelta( 
               seconds=int( self.expires - Tac.utcNow() ) ) ) 
               if self.expires else "never" ) )
            print( '  Group mode: %s' % filterMode )
            print( '  Last reporter: %s'  % self.lastReporter )
            for ( srcAddress, src ) in sorted( self.groupDetail.sources.items() ):
               print( '    Source: %s' % srcAddress )
               src.render()
         else:
            print(  ( '%s' % self.groupAddress ).ljust( groupAddressWidth ) + 
                  ( '%s' % self.interfaceName.stringValue ).ljust( intfWidth )  + 
                  ( '%s' % str( timedelta( seconds=int( 
                     Tac.utcNow() - self.uptime ) ) ) ).ljust( uptimeWidth ) + 
                  ( '%s' % ( str( timedelta( 
                     seconds=int( self.expires - Tac.utcNow() ) ) ) 
                     if self.expires else "never" ) ).ljust( expiresWidth ) + 
                  ( '%s' % self.lastReporter ).ljust( lastReporterWidth )  )
 
   groupList = List( valueType=IgmpGroup, 
         help="List of IGMP groups joined on different interfaces" )
 
   def render( self ):
      if not self._detail:
         print( 'NOTE: static-group information not shown below.  Use the' )
         print( '      \'show ip igmp static-groups\' command.' )
         print( 'IGMP Connected Group Membership' )
         print(  'Group Address'.ljust( groupAddressWidth ) + 
                'Interface'.ljust( intfWidth )  + 
                'Uptime'.ljust( uptimeWidth ) + 
                'Expires'.ljust( expiresWidth ) + 
                'Last Reporter'.ljust( lastReporterWidth )  )
      for grp in sorted( self.groupList, 
            key=lambda x: x.groupAddress ):
         grp.render()

class IgmpInterfaces( DeferredModel ):
   ''' CAPI model for show ip igmp interface <> '''

   class IgmpInterface( Model ):
      ''' CAPI model for Igmp Interface '''
      interfaceAddress = Ip4Address( help='Ip Address of this interface' )
      igmpEnabled = Bool( help='Denotes whether IGMP is enabled on this interface' )
      multicastRoutingEnabled =  Bool( 
            help='Denotes whether multicast routing is enabled' )
      multicastTTL = Int( help='Multicast TTL Threshold', default=1 )
      routerVersion = Enum( help='IGMP router version on this interface', 
                  values=( "-", "1", "2", "3" ) )
      igmpQueryInterval = Int( help='IGMP Query interval in seconds' )
      maxQueryResponse = Float( 
            help='IGMP max query response interval in seconds' )
      lastMemberQueryResponseInterval = Float( 
            help='IGMP last member query response interval in seconds' )
      lastMemberQueryResponseCount = Int( 
            help='IGMP last member query response count' )
      igmpQuerier = Ip4Address( 
            help='Ip Address of the IGMP Querier on this interface' )
      robustness = Int( help='IGMP Robustness on this interface' )
      routerAlert = Enum( help='IGMP Router Alert requirement on this interface',
            values= ( "routerAlertMandatory", "routerAlertOptional", 
               "routerAlertOptionalConnected" ) )
      startupQueryInterval = Float( 
            help='IGMP startup query interval in seconds' )
      startupQueryCount = Int( help='IGMP startup query count' )
      generalQueryExpiryTime = Float( 
            help='IGMP general query timer expiry in seconds', optional=True )
      otherQuerierExpiryTime = Float( 
            help='Other querier present timer expiry in seconds', optional=True )
      multicastGroups = List( valueType=Ip4Address, 
            help='The list of multicast groups joined on this interface' )

   IgmpInterfaces = Dict( keyType=Interface, valueType=IgmpInterface, 
         help="Map of IGMP enabled interfaces" )

class IgmpGroupsCount( Model ):
   count = Int( help="Number of Igmp Groups" )
   interfaceName = Interface( help='Interface on which to count groups.',
                              optional=True )

   def render( self ):
      if self.interfaceName:
         # pylint: disable-next=bad-string-format-type
         print(  "Number of groups joined on %s: %d" % 
                ( self.interfaceName.stringValue, self.count )  )
      else:
         print(  "Number of total groups joined across all IGMP interfaces: %d" % 
                ( self.count )  )

class IgmpStatistics( Model ):
   ''' CAPI model for show ip igmp statistics <> '''

   class IgmpStatisticsInterface( Model ):
      ''' CAPI model for Igmp Statistics '''
      v1QueriesSent = Int( help='IGMP v1 queries sent' )
      v2QueriesSent = Int( help='IGMP v2 queries sent' )
      v3QueriesSent = Int( help='IGMP v3 queries sent' )
      totalGeneralQueriesSent = Int( help='IGMP total general queries sent' )
      v3GSQueriesSent = Int( help='IGMP v3 group-specific queries sent' )
      v3GSSQueriesSent = Int( help='IGMP v3 group-source-specific queries sent' )
      v1QueriesReceived = Int( help='IGMP v1 queries received' )
      v2QueriesReceived = Int( help='IGMP v2 queries received' )
      v3QueriesReceived = Int( help='IGMP v3 queries received' )
      v1ReportReceived = Int( help='IGMP v1 reports received' )
      v2ReportReceived = Int( help='IGMP v2 reports received' )
      v2LeaveReceived = Int( help='IGMP v2 leaves received' )
      v3ReportReceived = Int( help='IGMP v3 reports received' )
      errorPacketReceived = Int( help='error packets received' )
      otherPacketReceived = Int( help='other packets received' )
 
   IgmpStatistics = Dict( keyType=Interface, valueType=IgmpStatisticsInterface, 
         help="Map of statistics for IGMP interfaces" )
       
   def render( self ):
      for ( intfId, igmpIntf ) in sorted( self.IgmpStatistics.items() ):
         print( 'IGMP counters for %s:' % intfId )
         print( '  V1 queries sent: %s' % igmpIntf.v1QueriesSent )
         print( '  V2 queries sent: %s' % igmpIntf.v2QueriesSent )
         print( '  V3 queries sent: %s' % igmpIntf.v3QueriesSent )
         print( '  Total general queries sent: %s' %
                igmpIntf.totalGeneralQueriesSent )
         print( '  V3 group specific queries sent: %s' % igmpIntf.v3GSQueriesSent )
         print(  '  V3 group-source specific queries sent: %s' % 
                igmpIntf.v3GSSQueriesSent  )
         print( '  V1 queries received: %s' % igmpIntf.v1QueriesReceived )
         print( '  V2 queries received: %s' % igmpIntf.v2QueriesReceived )
         print( '  V3 queries received: %s' % igmpIntf.v3QueriesReceived )
         print( '  V1 reports received: %s' % igmpIntf.v1ReportReceived )
         print( '  V2 reports received: %s' % igmpIntf.v2ReportReceived )
         print( '  V3 reports received: %s' % igmpIntf.v3ReportReceived )
         print( '  V2 leaves received: %s' % igmpIntf.v2LeaveReceived )
         print( '  Error Packets received: %s' % igmpIntf.errorPacketReceived )
         print( '  Other Packets received: %s' % igmpIntf.otherPacketReceived )

class IgmpMemGrp( Model ):
   ''' CAPI model for IgmpMemGrp '''
   interfaceName = Interface( help="Interface on which this group is joined" )
   groupAddress = IpGenericAddress( help="Group Address of interest" )
   includeSrcs = List( valueType=IpGenericAddress,
            help="List of Include Source Addresses for this group" )
   excludeSrcs = List( valueType=IpGenericAddress,
            help="List of Exclude Source Addresses for this group" )

class IgmpMemGrpList( Model ):
   groupList = List( valueType=IgmpMemGrp, help="List of IGMP membership groups" )

class IgmpMembership( DeferredModel ):
   ''' CAPI model for show ip igmp membership '''

   members = Dict( keyType=Interface, valueType=IgmpMemGrpList,
         help="Map interface to a list of IGMP membership status" )
