#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
from CliPlugin.AclCliModel import AllAclList
from CliPlugin.AclCli import countersPerChipEnabledGuard, ipAclNameMatcher, \
      userIpAclNameMatcher
import CliPlugin.IgmpCli as IgmpCli # pylint: disable=consider-using-from-import
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.IntfCli import Intf
from CliPlugin.IgmpCliModel import IgmpGroups, IgmpGroupsCount, IgmpInterfaces, \
      IgmpMembership, IgmpStatistics, IgmpStaticGroups, IgmpSourceGroups
from CliPlugin.McastCommonCli import mcastRoutingSupportedGuard
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from CliToken.Ip import ipMatcherForShow
from CliToken.Lanz import matcherDynamic, matcherSummary, matcherDetail, \
      matcherAccessList
import ShowCommand

matcherIgmp = CliMatcher.KeywordMatcher( 'igmp',
      helpdesc='IGMP related status and configuration' )
nodeIgmp = CliCommand.Node( matcher=matcherIgmp, guard=mcastRoutingSupportedGuard )
nodeDetail = CliCommand.Node( matcher=matcherDetail,
      guard=countersPerChipEnabledGuard )
matcherStaticGroups = CliMatcher.KeywordMatcher( 'static-groups',
      helpdesc='Show IGMP static groups' )
matcherGroups = CliMatcher.KeywordMatcher( 'groups',
      helpdesc='Show IGMP group membership information' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Show IGMP interface information' )
matcherCount = CliMatcher.KeywordMatcher( 'count',
      helpdesc='Show IGMP groups total count' )
vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='Configure VRFs' )

#--------------------------------------------------------------------------------
# show ip igmp access-list [ ACLNAME ] [ summary ] [ dynamic ] [ detail ]
#--------------------------------------------------------------------------------
class IpIgmpAccessListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp access-list [ ACLNAME ] [ summary ] [ dynamic ] [ detail ]'
   data = {
      'ip': ipMatcherForShow,
      'igmp': nodeIgmp,
      'access-list': matcherAccessList,
      'ACLNAME': ipAclNameMatcher,
      'summary': matcherSummary,
      'dynamic': matcherDynamic,
      'detail': nodeDetail,
   }
   handler = IgmpCli.showIpAcl
   cliModel = AllAclList

BasicCli.addShowCommandClass( IpIgmpAccessListCmd )

#--------------------------------------------------------------------------------
# show ip igmp [ vrf VRF ] groups [ detail ]
#--------------------------------------------------------------------------------
class IpIgmpGroupsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp [ VRF ] groups [ detail ]'
   data = {
      'ip': ipMatcherForShow,
      'igmp': nodeIgmp,
      'VRF': vrfExprFactory,
      'groups': matcherGroups,
      'detail': matcherDetail,
   }
   handler = IgmpCli.cmdShowIpIgmpGroups
   cliModel = IgmpGroups

BasicCli.addShowCommandClass( IpIgmpGroupsCmd )

#--------------------------------------------------------------------------------
# show ip igmp [ vrf VRF ] groups interface INTF [ detail ]
#--------------------------------------------------------------------------------
class IpIgmpGroupsInterfaceIntfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp [ VRF ] groups interface INTF [ detail ]'
   data = {
      'ip': ipMatcherForShow,
      'igmp': nodeIgmp,
      'VRF': vrfExprFactory,
      'groups': matcherGroups,
      'interface': matcherInterface,
      'INTF': Intf.matcher,
      'detail': matcherDetail,
   }
   handler = IgmpCli.cmdShowIpIgmpGroupsInterface
   cliModel = IgmpGroups

BasicCli.addShowCommandClass( IpIgmpGroupsInterfaceIntfCmd )

#--------------------------------------------------------------------------------
# show ip igmp [ vrf VRF ] groups count
#--------------------------------------------------------------------------------
class IpIgmpGroupsCountCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp [ VRF ] groups count'
   data = {
      'ip': ipMatcherForShow,
      'igmp': nodeIgmp,
      'VRF': vrfExprFactory,
      'groups': matcherGroups,
      'count': matcherCount,
   }
   handler = IgmpCli.cmdShowIpIgmpGroupsCount
   cliModel = IgmpGroupsCount

BasicCli.addShowCommandClass( IpIgmpGroupsCountCmd )

#--------------------------------------------------------------------------------
# show ip igmp [ vrf VRF ] groups GROUPADDR [ detail ]
#--------------------------------------------------------------------------------
class IpIgmpGroupsGroupaddrCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp [ VRF ] groups GROUPADDR [ detail ]'
   data = {
      'ip': ipMatcherForShow,
      'igmp': nodeIgmp,
      'VRF': vrfExprFactory,
      'groups': matcherGroups,
      'GROUPADDR': IpAddrMatcher( helpdesc='Multicast group address' ),
      'detail': matcherDetail,
   }
   handler = IgmpCli.cmdShowIpIgmpGroupsAddress
   cliModel = IgmpGroups

BasicCli.addShowCommandClass( IpIgmpGroupsGroupaddrCmd )

#--------------------------------------------------------------------------------
# show ip igmp [ vrf VRF ] groups interface INTF count
#--------------------------------------------------------------------------------
class IpIgmpGroupsInterfaceIntfCountCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp [ VRF ] groups interface INTF count'
   data = {
      'ip': ipMatcherForShow,
      'igmp': nodeIgmp,
      'VRF': vrfExprFactory,
      'groups': matcherGroups,
      'interface': matcherInterface,
      'INTF': Intf.matcher,
      'count': matcherCount,
   }
   handler = IgmpCli.cmdShowIpIgmpGroupsCountIntf
   cliModel = IgmpGroupsCount

BasicCli.addShowCommandClass( IpIgmpGroupsInterfaceIntfCountCmd )

#--------------------------------------------------------------------------------
# show ip igmp [ vrf VRF ] interface [ INTF ]
#--------------------------------------------------------------------------------
class IpIgmpInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp [ VRF ] interface [ INTF ]'
   data = {
      'ip': ipMatcherForShow,
      'igmp': nodeIgmp,
      'VRF': vrfExprFactory,
      'interface': matcherInterface,
      'INTF': Intf.matcher,
   }
   handler = IgmpCli.cmdShowIpIgmpInterface
   cliModel = IgmpInterfaces

BasicCli.addShowCommandClass( IpIgmpInterfaceCmd )

#--------------------------------------------------------------------------------
# show ip igmp [ vrf VRF ] membership [ interface INTF ] [ group GROUPADDR ]
#--------------------------------------------------------------------------------
class IpIgmpMembershipCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip igmp [ VRF ] membership '
              '[ interface INTF ] [ group GROUPADDR ]' )
   data = {
      'ip': ipMatcherForShow,
      'igmp': nodeIgmp,
      'VRF': vrfExprFactory,
      'membership': ( 'Show the effective membership state from '
                      'dynamic & static membership' ),
      'interface': matcherInterface,
      'INTF': Intf.matcher,
      'group': 'Option to see status of a designated group',
      'GROUPADDR': IpAddrMatcher( helpdesc='Multicast group address' ),
   }
   handler = IgmpCli.cmdShowIpIgmpMembership
   cliModel = IgmpMembership

BasicCli.addShowCommandClass( IpIgmpMembershipCmd )

#--------------------------------------------------------------------------------
# show ip igmp [ vrf VRF ] statistics [ interface INTF ]
#--------------------------------------------------------------------------------
class IpIgmpStatisticsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp [ VRF ] statistics [ interface INTF ]'
   data = {
      'ip': ipMatcherForShow,
      'igmp': nodeIgmp,
      'VRF': vrfExprFactory,
      'statistics': 'Show IGMP counters',
      'interface': 'Display IGMP counters for a specific interface',
      'INTF': Intf.matcher,
   }
   handler = IgmpCli.cmdShowIpIgmpStatistics
   cliModel = IgmpStatistics

BasicCli.addShowCommandClass( IpIgmpStatisticsCmd )

#--------------------------------------------------------------------------------
# show ip igmp [ vrf VRF ] static-groups group [ GROUPADDR ]
#--------------------------------------------------------------------------------
class IpIgmpStaticGroupsGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp [ VRF ] static-groups group [ GROUPADDR ]'
   data = {
      'ip': ipMatcherForShow,
      'igmp': nodeIgmp,
      'VRF': vrfExprFactory,
      'static-groups': matcherStaticGroups,
      'group': 'Display IGMP static groups, organized by group',
      'GROUPADDR': IpAddrMatcher( helpdesc='Multicast group address' ),
   }
   handler = IgmpCli.cmdShowIpIgmpStaticGroupsGroup
   cliModel = IgmpSourceGroups

BasicCli.addShowCommandClass( IpIgmpStaticGroupsGroupCmd )

#--------------------------------------------------------------------------------
# show ip igmp [ vrf VRF ] static-groups [ detail ] [ interface INTF ]
#--------------------------------------------------------------------------------
class IpIgmpStaticGroupsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip igmp [ VRF ] static-groups '
              '[ detail ] [ interface INTF ]' )
   data = {
      'ip': ipMatcherForShow,
      'igmp': nodeIgmp,
      'VRF': vrfExprFactory,
      'static-groups': matcherStaticGroups,
      'detail': matcherDetail,
      'interface': 'Display IGMP static groups for a specific interface',
      'INTF': Intf.matcher,
   }
   handler = IgmpCli.cmdShowIpIgmpStaticGroups
   cliModel = IgmpStaticGroups

BasicCli.addShowCommandClass( IpIgmpStaticGroupsCmd )

#--------------------------------------------------------------------------------
# show ip igmp [ vrf VRF ] static-groups acl ACLNAME
#--------------------------------------------------------------------------------
class IpIgmpStaticGroupsAclNameCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp [ VRF ] static-groups acl ACLNAME'
   data = {
      'ip': ipMatcherForShow,
      'igmp': nodeIgmp,
      'VRF': vrfExprFactory,
      'static-groups': matcherStaticGroups,
      'acl': 'IP ACL for use as static group list',
      'ACLNAME': userIpAclNameMatcher,
   }
   handler = IgmpCli.cmdShowIpIgmpStaticGroupsAcl

BasicCli.addShowCommandClass( IpIgmpStaticGroupsAclNameCmd )
